# $id$
#
# Contributed by Horace Vallas
#===========================================================
#
# updates:  9-16-96  hav
#
# NEW COMMANDS:
#
# delete_nc ?cid   - deletes a named counter from the NCRegistry file
#
# reset_nc ?cid <?new_value> <?comments>  -deletes an existing nc and recreates it with the
#                                          ?new_vallue as the initial counts
#
# rename_nc ?cid ?new_name  - renames a named counter
#
#
#  1. Web Server Administrators can now set the maximum number of 
#     named counters that can be registered by a single user by
#     setting the value of the NeoWebServerConf(max_nc) parameter.
#     if this parametrer is not set, then the max allowed will be 200
#
#  2. Make_nc_registry nolonger prints an error on the page if the
#     registry already exists.
#
#  3. make_nc_registry now returns 1 for success and 0 for error
#
#  4. use of register_nc will create the NCRegistry file if it does not
#     yet exist - the NCRegistry will be created for max number of
#     named counters allowed.
#
#  5. changed the default number of namced counters allowed in a NCRegistry
#     from 20 to 200
#
#  6. corrected errors that were generated in incr_nc by non-browser visitors 
#     (i.e. telnet, etc.)
#
#  7. delete_nc  NEW proc that deletes a named-counter
#
#  8. reset_nc   NEW proc that resets the initial counts of a named counter and
#                deletes all existing daily count records for that counter
#
#  9. rename_nc  NEW proc to allow user to rename an existing named counter
#
# 10. all spaces in a nc name in register_nc  will be converted to _
#



#===========================================================
# make_nc_registry
#
# This proc is used to create a Named Counter Registry.  Each UID may
# have one such NCRegistry file accessible from NeoWebScript.  Each NCRegistry
# can hold a minimum of 20 Named Counters.  The owner will specify the
# maximum number of Named Counters allowed as an argument to this proc.
# No more than 200 Named Counters per NCRegistry will  be allowed.
#
# Each NCRegistry has one registry record which maintains a list of the
# Named Counters in the registry.  
# fields: ncmax, nccount, date (quoted), comments (quoted)
#           also one field for each registered Named Counter with the
#           field name is the counter's name and a value which is the 
#           date on which the counter was registered.
#
# This proc must be called BEFORE any register_nc calls will be honored.
#
#   Calling Sequence:
#
#        make_nc_registry {{ncmax 0} {comments ""}}
#
#        ncmax      - maximum number of Named Counters that can be registered
#                       in this NCRegistry
#        comments - user comments
#
#  Returns:  1 if Success
#            0 if error 
# 

proc make_nc_registry {{ncmax 200} {comments ""}} {
    global NeoWebServerConf
    if {[dbexists NCRegistry]} {
        return 0
    } else {
        if {[info exists NeoWebServerConf(nc_max)]} {
            set marray(ncmax) $NeoWebServerConf(nc_max)
        } else {
            scan $ncmax %d marray(ncmax) 
            if {$marray(ncmax) <= 20} {set marray(ncmax) 200}
            if {$marray(ncmax) > 200} {set marray(ncmax) 200}
        }
        set marray(nccount) 0
        set marray(date) [quote_string [clock format [clock seconds]]]
        set marray(comments)  $comments
        dbstore NCRegistry registry marray
        html "<i>The NCRegistry file has been created</i>"

    }
    return 1
}
  
#===========================================================
# 
# register_nc
#
#  This proc  is used to allow users to register NAMED COUNTERs.
#  It controls the creation of new Named Counters within the user's
#  NCRegistry.
#
# NOTE: If the registry file does not exist, then it will be created
#
#  A Named Counter exists within the NCRegistry as a collection
#  of keyed records.  There are three record types for each Named
#  Counter in an NCRegistry:
#
#   1. init-$cid - holds the initial count values for the Named Counter
#                     with the name $cid
#                     1 record per Named Counter
#                     fields: date (quoted), comments (quoted), mozilla, mosaic,
#                               lynx, mie, other 
#
#   2. totals-$cid - holds the current lifetime total counts for the 
#                         Named Counter
#                         1 record per Named Counter
#                         fields: total (all browsers),  mozilla, mosaic,
#                                  lynx, mie, other 
#
#  3. $cid-yyyymmdd - holds the courrent counts for the day yyyymmdd
#                              for the Named Counter with name $cid
#                              1 record for every day since the counter was registered
#                              fields: mozilla, mosaic, lynx, mie, other 
#
#  Calling Sequence:
#
#          register_nc {cid {initcounts 0} {comments ""}}
#
#   cid        - the name of the counter (must be unique within the NCRegistry) 
#   initcounts - the initial counts assigned to the counter (optional; default=0)
#   comments   - a creation comment {optional; default="")
#

proc register_nc {cid  {initcounts 0} {comments ""}} {

    if {![info exists cid]} {
        html "<i>Error: Named Counter not specified</i>"
        return 0
    } else {
        if {![dbexists NCRegistry]} {
            make_nc_registry 200
        }
        dbfetch NCRegistry registry rarray

        if {[dbfetch NCRegistry init-$cid carray]} {
            html  "<i>Error: Named Counter ($cid) already exists</i>"
            return 0
        } elseif {$rarray(nccount) >= $rarray(ncmax)} {
            # only a certain number of NCs are allowed for a given user
            html  "<i>Error: Too many Named Counters in this registry</i>"
            return 0
        }
        regsub " " $cid _ cid1
        set cid $cid1
        # all ok, so start by updating registry record info
        incr rarray(nccount)
        set rarray($cid) [quote_string "[clock format [clock seconds]]"]
        # now setup the specific counter init and totals records
        scan $initcounts %d ccount
        set tarray(total) $ccount
        # total values for init count
        scan [expr $ccount * .85] %d tarray(mozilla)
        scan [expr $ccount * .07] %d tarray(other) 
        scan [expr $ccount * .05] %d tarray(mosaic)
        scan [expr $ccount * .02] %d tarray(lynx) 
        scan [expr $ccount * .01] %d tarray(mie)
 
        set carray(date) rarray($cid)
        set carray(comments) [quote_string $comments]
        set carray(mozilla) $tarray(mozilla)
        set carray(other) $tarray(other) 
        set carray(mosaic) $tarray(mosaic)
        set carray(lynx) $tarray(lynx) 
        set carray(mie) $tarray(mie)
          
        dbstore NCRegistry registry rarray
        dbstore NCRegistry init-$cid carray
        dbstore NCRegistry totals-$cid tarray 
    }
    return 1
}

#===========================================================
#
#  delete_nc
#
# deletes a named counter

proc delete_nc {cid} {

    if {![info exists cid]} {
        html "<i>Error: Named Counter not specified</i>"
        return 0
    } elseif {![dbfetch NCRegistry registry rarray]} {
        html  "<i>Error: Missing registry record</i>"
        return 0
    } elseif {![dbfetch NCRegistry init-$cid carray]} {
        html  "<i>Error: Named Counter ($cid) does not exist</i>"
        return 0
    }
    # ok so delete all records pertaining to the counter
    dbdelkey NCRegistry init-$cid
    dbdelkey NCRegistry totals-$cid
    foreach key [dbkeys NCRegistry $cid"-"*] {
        dbdelkey NCRegistry $key
    }
    dbfetch NCRegistry registry darray
    unset darray($cid)
    incr darray(nccount) -1
    dbstore NCRegistry registry darray
    return 1
}


#===========================================================
#
# rename_nc
#
# rename a counter

proc rename_nc {cid to} {

    if {![info exists cid]} {
        html "<i>Error: Named Counter not specified</i>"
        return 0
    } elseif {![info exists to]} {
        html "<i>Error: New Name not specified</i>"
        return 0
    } elseif {![dbfetch NCRegistry registry rarray]} {
        html  "<i>Error: Missing registry record</i>"
        return 0
    } elseif {![dbfetch NCRegistry init-$cid carray]} {
        html  "<i>Error: Named Counter ($cid) does not exist</i>"
        return 0
    } elseif {[dbfetch NCRegistry init-$to carray]} {
        html  "<i>Error: Named Counter ($to) already exists</i>"
        return 0
    }
    # ok so rewrite all appropriate records
    dbfetch NCRegistry init-$cid darray
    dbstore NCRegistry init-$to darray

    dbfetch NCRegistry totals-$cid tarray
    dbstore NCRegistry totals-$to tarray

    foreach key [dbkeys NCRegistry $cid"-"*] {
        dbfetch NCRegistry $key darray
        regsub $cid $key $to newkey
        dbstore NCRegistry $newkey darray
        catch {unset darray}
    }
    set rarray($to) $rarray($cid)
    unset rarray($cid)
    dbstore NCRegistry registry rarray
    
    # ok, now that all updates have been done, delete all old records
    # that still ref the old name
    dbdelkey NCRegistry init-$cid
    dbdelkey NCRegistry totals-$cid
    foreach key [dbkeys NCRegistry $cid"-"*] {
        dbdelkey NCRegistry $key
    }
    return 1
}


#===========================================================
#
#  reset_nc
#
#  resets the initial counts and delete all daily count records
#  for the specified named counter


proc reset_nc {cid {initcounts 0} {comments ""}} {

    if {[delete_nc $cid]} {
        return [register_nc $cid $initcounts $comments]
    } else {
        return 0
    }
}




#===========================================================
#
#  incr_nc
#
# increments the Named Counter for the current day (appropriate browser) 
# and the counter's total (overall and for the appropriate browser).
#
# If this is the first visit of the day, then a new database record is created.
# Returns, by default the total visits (all browsers all days) since the counter 
# was registered.   The caller may request the daily total instead.  The caller
# may also request only a specific browser's count be returned.
#
# Calling Sequence:
#
#   incr_nc {cid {show show} {browser all} {rtntype lifetime}}
#
#  cid        - is the name of the counter to be incremented
#  show     - is an indicator to return or not resurn a count
#                    default is show (any other value means noshow)
#  browser - is the browser count to be returned  - default all
#                   values: all this mozilla mosaic lynx mie other
#  rtntype  - is the count to be returned - default lifetime
#                   values lifetime (any other value means today only)

proc incr_nc {cid {show show} {browser all} {rtntype lifetime}} {
    global webenv 
    if {![info exists cid]} {
        html "<i>Error: Named Counter not specified</i>"
        return 0
        # first verify that there is such a named counter registered
        # and if not, then return an error
    } elseif {![dbfetch NCRegistry totals-$cid tarray]} {
        html  "<i>Error: Named Counter ($cid) is not registered</i>"
        return 0
    } else {
        # if first count today, then create a new daily record with all values 0
        set dayindex [clock format [clock seconds] -format "%Y%m%d"]
        if {![dbfetch NCRegistry ${cid}-${dayindex} carray]} {
            set carray(mozilla) 0
            set carray(mosaic) 0
            set carray(lynx) 0
            set carray(mie) 0
            set carray(other) 0
        }
        if {![info exists webenv(HTTP_USER_AGENT)]} {
            set user_agent "unknown"
        } else {
            set user_agent $webenv(HTTP_USER_AGENT)
        }
        # increment the appropriate daily and total counts
        if {[string last "Mozilla" $user_agent] != -1} {
            if {[string last "MSIE" $user_agent] != -1} {
                incr carray(mie)
                incr tarray(mie)
            } else {
                incr carray(mozilla)
                incr tarray(mozilla)
            }
        } elseif {[string last "Mosaic" $user_agent] != -1} {
            incr carray(mosaic)
            incr tarray(mosaic)
        } elseif {[string last "Lynx" $user_agent] != -1} {
            incr carray(lynx)
            incr tarray(lynx)
        } elseif {[string last "Microsoft" $user_agent] != -1} {
            incr carray(mie)
            incr tarray(mie)
        } else {
            incr carray(other)
            incr tarray(other)
        }
        incr tarray(total)

        # save the updated daily and totals records after incrementing the full
        # total count
       dbstore NCRegistry ${cid}-${dayindex} carray
       dbstore NCRegistry totals-$cid tarray

       # get and return the appropriate value
       #   NOTE: code from other proc is duplicated here to 
       #              avoid the overhead of a call and additional dbfetch-ing
       set rtnval 0
       if {$show == "show"} {
           if {$rtntype == "lifetime"} {
               if {$browser == "all"} {set rtnval  $tarray(total)
               } elseif {[info exists tarray($browser)]} {set rtnval $tarray($browser)
               }
           } else {
               if {$browser == "all"} {
                  set rtnval [expr  $carray(mozilla) + $carray(mosaic) + $carray(lynx) + $carray(mie) + $carray(other)]
               } elseif {[info exists carray($thevar)]} {set rtnval $carray($browser)
               }
           }
           return $rtnval
       }
   }
}

#===========================================================
# 
#  get_nc
#
#    returns the visit count since the named counter has been in use
#    returns -1 if the named counter does not exist
#    returns the total visits by default or the mizilla 
#    or other count by request  
#
#  Calling Sequence:
#
#   get_nc {cid {browser all} {rtntype lifetime}}
#
#  cid        - is the name of the counter to be returned
#  browser - is the browser count to be returned  - default all
#                   values: all this mozilla mosaic lynx mie other
#  rtntype  - is the count to be returned - default lifetime
#                   values: lifetime today yyymmdd (a date with leading 0 in m & d)

proc get_nc {cid {browser all} {rtntype lifetime}} {
    if {![info exists cid]} {return -1}
    set rtnval -1
    if {$rtntype == "lifetime"} {
        if {[dbfetch NCRegistry totals-$cid tarray]} {
            if {$browser == "all"} {set rtnval $tarray(total)
            } elseif {[info exists tarray($browser)]} {set rtnval $tarray($browser)
            }
        }
    } else {
        if {$rtntype == "today"} {
            set dayindex [clock format [clock seconds] -format "%Y%m%d"]
        } else {
            set dayindex $rtntype
        }
        if {[dbfetch NCRegistry $cid"-"$dayindex carray]} {
            if {$browser == "all"} {set rtnval [expr  $carray(mozilla) + $carray(mosaic) + $carray(lynx) + $carray(mie) + $carray(other)]
            } elseif {[info exists carray($browser)]} {set rtnval $carray($browser)
            }
        }
    }
    return $rtnval
}



#===========================================================
# 
#   get_init_nc
#
#    returns the initial counts assigned to the counter
#    returns -1 if the named counter does not exist
#    returns the total visits by default or the specific
#      browser by request
#
#  Calling Sequence:
#
#          get_init_nc {cid {browser all}}
#
#  cid        - is the name of the counter to be returned
#  browser - is the browser count to be returned  - default all
#                   values: all this mozilla mosaic lynx mie other


proc get_init_nc {cid {browser all}} {
    if {![info exists cid]} {return -1}
    set rtnval -1
    if {[dbfetch NCRegistry init-$cid carray]} {
        if {$browser == "all"} {
             set rtnval [expr  $carray(mozilla) + $carray(mosaic) + $carray(lynx) + $carray(mie) + $carray(other)]
        } else {
              if {[info exists carray($browser)]} {
                  set rtnval $carray($browser)
              }
        }
    }
    return $rtnval
}



#===========================================================
# 
#   get_real_nc
#
#    returns the "real" counts since the counter was registered - i.e. (total - init_count)
#    returns -1 if the named counter does not exist
#    returns the total visits by default or the specific
#      browser by request
#
#  Calling Sequence:
#
#          get_real_nc {cid {browser all}}
#
#  cid        - is the name of the counter to be returned
#  browser - is the browser count to be returned  - default all
#                   values: all this mozilla mosaic lynx mie other

proc get_real_nc {cid {browser all}} {
    if {![info exists cid]} {return -1}
    set rtnval -1
    if {[dbfetch NCRegistry init-$cid carray]} {
        if {[set this_tot [get_nc $cid $browser]] >= 0} {
            if {[set this_init [get_init_nc $cid $browser]] >= 0} {
               set rtnval [expr $this_tot - $this_init]
            }
        }
    }
    return $rtnval
}


#===========================================================
#
#   get_avg_nc
#
#       returns the "real" visit count since the named counter has been in use
#       AVERAGED by the number of days the counter has been in use
#       (i.e. a daily average)
#
#  NOTE: the averaged value DOES NOT include initial counts
#
#       returns -1 if the named counter does not exist
#    returns the total visits by default or the specific
#      browser by request
#
#  Calling Sequence:
#
#          get_avg_nc {cid {browser all}}
#
#  cid        - is the name of the counter to be returned
#  browser - is the browser count to be returned  - default all
#                   values: all this mozilla mosaic lynx mie other

proc get_avg_nc {cid {browser all}} {
    if {![info exists cid]} {return -1}
    set rtnval -1
    if {[dbfetch NCRegistry totals-$cid tarray]} {
        if {[set numer [get_real_nc $cid $browser]] >= 0} {
            dbfetch  NCRegistry registry iniarray
            set rdate  [clock scan [unquote_string $iniarray($cid)]]
            set timedif [expr [clock seconds] - $rdate]
            scan [expr $timedif / 86400] %d denom   ; #  86400 = sec/day
            scan [expr $numer / $denom] %d rtnval
        }
    }
    return $rtnval
}
