/*
 * Decode MIME parts.
 */
/* (C) Copyright 1993,1994 by Carnegie Mellon University
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation, and that the name of Carnegie
 * Mellon University not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission.  Carnegie Mellon University makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * $Id: decode.c,v 1.1.1.1 1999/03/31 20:54:36 damon Exp $
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "xmalloc.h"
#include "common.h"
#include "part.h"

extern FILE *os_newtypedfile();

/* The possible content transfer encodings */
enum encoding { enc_none, enc_qp, enc_base64 };

char *ParseHeaders();
params ParseContent();
char *getParam();
char *getDispositionFilename();


/*
 * Read and handle an RFC 822 message from the body-part 'inpart'.
 */
handleMessage(inpart, defaultContentType)
struct part *inpart;
char *defaultContentType;
{
    char *headers, *contentType, *contentDisposition;
    params contentParams;

    /* Parse the headers, getting the ones we're interested in */
    headers = ParseHeaders(inpart, &contentType, &contentDisposition);
    if (!headers) return 1;

    /* If no content type, or a non-MIME content type, use the default */
    if (!contentType || !strchr(contentType, '/')) {
	contentType = defaultContentType;
    }
    contentParams = ParseContent(&contentType);

    if (!strncasecmp(contentType, "multipart/", 10)) {
	return handleMultipart(inpart, contentType, contentParams);
    }
    else {
	/* Some sort of attachment, extract it */
	return saveToFile(inpart, contentType, contentParams,
			  contentDisposition);
    }
}

/*
 * Skip whitespace and RFC-822 comments.
 */
SkipWhitespace(s)
char **s;
{
    char *p = *s;
    int commentlevel = 0;

    while (*p && (isspace(*p) || *p == '(')) {
	if (*p == '\n') {
	    p++;
	    if (*p != ' ' && *p != '\t') {
		*s = 0;
		return;
	    }
	}
	else if (*p == '(') {
	    p++;
	    commentlevel++;
	    while (commentlevel) {
		switch (*p) {
		case '\r':
		    p++;
		    /* FALL THROUGH */
		case '\n':
		    p++;
		    if (*p == ' ' || *p == '\t') break;
		    /* FALL THROUGH */
		case '\0':
		    *s = 0;
		    return;
		    
		case '\\':
		    p++;
		    break;

		case '(':
		    commentlevel++;
		    break;

		case ')':
		    commentlevel--;
		    break;
		}
		p++;
	    }
	}
	else p++;
    }
    if (*p == 0) {
	*s = 0;
    }
    else {
	*s = p;
    }
}

/*
 * Read and parse the headers of an RFC 822 message, returning them in
 * a pointer to a static buffer.  The headers are read from 'inpart'.
 * A pointer to the value of any Content-Type: or
 * Content-Disposition: header is stored in the space
 * pointed to by 'contentTypep', and 'contentDispositionp', 
 * respectively.  
 */
#define HEADGROWSIZE 1000
char *ParseHeaders(inpart, contentTypep, contentDispositionp)
struct part *inpart;
char **contentTypep;
char **contentDispositionp;
{
    static int alloced = 0;
    static char *headers;
    int left, len, i;
    char *next, *val;

    /* Read headers into buffer pointed to by "headers" */
    if (!alloced) {
	headers = xmalloc(alloced = HEADGROWSIZE);
    }
    next = headers;
    *next++ = '\n';		/* Leading newline to make matching header names easier */
    left = alloced - 2;		/* Allow room for terminating null */

    while (part_gets(next, left, inpart) && (*next != '\n' || next[-1] != '\n')) {
	len = strlen(next);

	if (next[-1] == '\n') {
	    /* Check for valid header-ness of "next" */
	    for (i = 0; i < len; i++) {
		if (next[i] == ':' ||
		    next[i] <= ' ' || next[i] >= '\177') break;
	    }
	    if (i == 0 || next[i] != ':') {
		/* Check for header continuation line */
		if (next == headers+1 || (next[0] != ' ' && next[0] != '\t')) {
		    /*
		     * Not a valid header, push back on input stream
		     * and stop reading input.
		     */
		    part_ungets(next, inpart);
		    break;
		}
	    }
	}

	left -= len;
	next += len;

	if (left < 100) {
	    len = next - headers;
	    alloced += HEADGROWSIZE;
	    left += HEADGROWSIZE;
	    headers = xrealloc(headers, alloced);
	    next = headers + len;
	}
    }

    *next = '\0';

    /* Look for the headers we find particularly interesting */
    *contentTypep = *contentDispositionp = 0;
    for (next = headers; *next; next++) {
	if (*next == '\n') {
	    switch(next[1]) {
	    case 'c':
	    case 'C':
		if (!strncasecmp(next+2, "ontent-type:", 12)) {
		    val = next+14;
		    SkipWhitespace(&val);
		    if (val) *contentTypep = val;
		}
		else if (!strncasecmp(next+2, "ontent-disposition:", 19)) {
		    val = next+21;
		    SkipWhitespace(&val);
		    if (val) *contentDispositionp = val;
		}
	    }
	}
    }
    return headers;
}

/*
 * Parse the value of a Content-Type: header.
 * 'headerp' points to a pointer to the input string.
 * The pointer pointed to by 'headerp' is changed to point to
 * a static buffer containing the content type stripped of whitespace
 * and parameters.  The parameters are converted to a type suitable for
 * getParm() and returned.
 */
#define PARAMGROWSIZE 10
params ParseContent(headerp)
char **headerp;
{
    char *header;
    static int palloced = 0;
    static char **param;
    static int calloced = 0;
    static char *cbuf;
    char *p;
    int nparam;

    p = header = *headerp;

    /* Find end of header, including continuation lines */
    do {
	p = strchr(p+1, '\n');
    } while (p && isspace(p[1]));
    if (!p) {
	p = header + strlen(header);
    }

    /* If necessary, allocate/grow cbuf to hold header. */
    if (p - header >= calloced) {
	calloced = p - header + 1;
	if (calloced < 200) calloced = 200;
	cbuf = xrealloc(cbuf, calloced);
    }

    /* Copy header to cbuf */
    strncpy(cbuf, header, p - header);
    cbuf[p - header] = 0;
    header = *headerp = cbuf;
    
    nparam = 0;

    /* Strip whitespace from content type */
    /* ParseHeaders() stripped leading whitespace */
    p = header;
    while (header && *header && *header != ';') {
	while (*header && !isspace(*header) && *header != '(' &&
	       *header != ';') {
	    *p++ = *header++;
	}
	SkipWhitespace(&header);
    }
    if (!header || !*header) return 0;
    header++;
    *p = '\0';
    
    /* Parse the parameters */
    while (*header) {
	SkipWhitespace(&header);
	if (!header) break;

	if (nparam+1 >= palloced) {
	    palloced += PARAMGROWSIZE;
	    param = (char **) xrealloc((char *)param, palloced * sizeof(char *));
	}
	param[nparam++] = header;

	/* Find any separating semicolon.  Pay attention to quoted-strings */
	while (*header && *header != ';') {
	    if (*header == '\"') {
		++header;
		while (*header && *header != '\"') {
		    if (*header == '\\') {
			++header;
			if (!*header) break;
		    }
		    ++header;
		}
		if (!*header) break;
	    }
	    else if (*header == '(') {
		/* Convert comments to spaces */
		p = header;
		SkipWhitespace(&p);
		if (!p) {
		    break;
		}
		while (header < p) *header++ = ' ';
		header--;
	    }
	    header++;
	}
	if (*header) *header++ = '\0';
    }
    param[nparam] = 0;
    return param;
}

/*
 * Get the value of the parameter named 'key' from the content-type
 * parameters 'cParams'.  Returns a pointer to a static bufer which
 * contains the value, or null if no such parameter was found.
 */
#define VALUEGROWSIZE 100
char *getParam(cParams, key)
params cParams;
char *key;
{
    static char *value;
    static int alloced = 0;
    int left;
    int keylen = strlen(key);
    char *from, *to;

    if (!cParams) return 0;

    if (!alloced) {
	value = xmalloc(alloced = VALUEGROWSIZE);
    }

    /* Find the named parameter */
    while (*cParams) {
	if (!strncasecmp(key, *cParams, keylen) &&
	    ((*cParams)[keylen] == '=' || isspace((*cParams)[keylen]))) break;
	cParams++;
    }
    if (!*cParams) return 0;

    /* Skip over the "=" and any surrounding whitespace */
    from = *cParams + keylen;
    while (*from && isspace(*from)) from++;
    if (*from++ != '=') return 0;
    while (*from && isspace(*from)) from++;
    if (!*from) return 0;

    /* Copy value into buffer */
    to = value;
    left = alloced - 1;
    if (*from == '\"') {
	/* Quoted-string */
	from++;
	while (*from && *from != '\"') {
	    if (!--left) {
		alloced += VALUEGROWSIZE;
		value = xrealloc(value, alloced);
		to = value + alloced - left - 2;
	    }
	    if (*from == '\\') {
		from++;
		if (!*from) return 0;
	    }
	    *to++ = *from++;
	}
	if (!*from) return 0;
    }
    else {
	/* Just a token */
	while (*from && !isspace(*from)) {
	    if (!--left) {
		alloced += VALUEGROWSIZE;
		value = xrealloc(value, alloced);
		to = value + alloced - left - 2;
	    }
	    *to++ = *from++;
	}
    }
    *to = '\0';
    return value;
}

/*
 * Skip over a message object from the file 'inpart'.
 */
ignoreMessage(inpart)
struct part *inpart;
{
    while (part_getc(inpart) != EOF);
    return 0;
}

/*
 * Read and handle a multipart object from 'inpart'.
 */
handleMultipart(inpart, contentType, contentParams)
struct part *inpart;
char *contentType;
params contentParams;
{
    char *id;
    char *defaultContentType = "text/plain";
    int isAppleDouble = 0;

    if (!(id = getParam(contentParams, "boundary"))) {
	warn("multipart message has no boundary parameter");
	id="";
    }

    /* Add the new boundary id */
    part_addboundary(inpart, id);

    /*
     * Skip over preamble.
     * HACK: The initial boundary doesn't have to start with a newline,
     * so we deal with this by stuffing an initial newline into the input
     * stream
     */
    part_ungetc('\n', inpart);
    ignoreMessage(inpart);

    /* Handle the component messages */
    while (!part_readboundary(inpart)) {
	handleMessage(inpart, defaultContentType);
    }

    /* Skip over postamble */
    ignoreMessage(inpart);

    /* Remove any lingering unused description file */
    (void) remove(TEMPFILENAME);

    return 0;
}

/*
 * Handle a text message object from 'inpart' by saving it to
 * the temporary description file.
 */
int handleText(inpart)
struct part *inpart;
{
    FILE *descfile;

    descfile = fopen(TEMPFILENAME, "w");
    if (!descfile) {
	perror(TEMPFILENAME);
	ignoreMessage(inpart);
	return 1;
    }

    fromnone(inpart, descfile);

    fclose(descfile);
    return 0;
}

void
emit(char *fname, char *contentType, char *uploadedFilename, char *contentDisposition)
{
    char *dispNL = strchr(contentDisposition, '\n');
    char *fieldType;

    if (*uploadedFilename == '\0') {
	fieldType = "field";
	uploadedFilename = "-";
    } else {
	fieldType = "file";
    }
    *dispNL = '\0';
    printf("%s %s %s {%s} %s\n", fname, contentType, fieldType, uploadedFilename, contentDisposition);
    *dispNL = '\n';
}

/*
 * Read a message object from 'inpart' and save it to a file.
 */
saveToFile(inpart, contentType, contentParams, contentDisposition)
struct part *inpart;
char *contentType;
params contentParams;
char *contentDisposition;
{
    FILE *outfile = 0;
    int suppressCR = 0;
    char *fname = (char *)malloc(16);
    static int partNumber = 0;
    char *uploadedFilename;

    if (!strncasecmp(contentType, "text/", 5)) {
	suppressCR = 1;
    }

    uploadedFilename = getDispositionFilename(contentDisposition);
    if (uploadedFilename == (char *)NULL) uploadedFilename = "";

    /* Find an appropriate filename and create the output file */
    sprintf(fname, "x%d", ++partNumber);
    outfile = fopen(fname, "w");

    /* only one type of content encoding -- straight through */
    fromnone(inpart, outfile);

    fclose(outfile);

    emit(fname, contentType, uploadedFilename, contentDisposition);

    free(fname);
    return 0;
}

fromnone(inpart, outfile)
struct part *inpart;
FILE *outfile;
{
    int c, save_c = -5;

    while ((c = part_getc(inpart)) != EOF) {
	if (save_c != -5) {
	    putc(save_c, outfile);
	}
	save_c = c;
    }
}

/*
 * Get the value of the "filename" parameter in a Content-Disposition:
 * header.  Returns a pointer to a static buffer containing the value, or
 * a null pointer if there was no such parameter.
 */
char *
getDispositionFilename(disposition)
char *disposition;
{
    static char *value;
    static int alloced = 0;
    int left;
    char *to;

    if (!disposition) return 0;

    /* Skip until we find ";" "filename" "=" tokens. */
    for (;;) {
	/* Skip until we find ";" */
	while (*disposition != ';') {
	    if (!*disposition) return 0;
	    else if (*disposition == '\"') {
		++disposition;
		while (*disposition && *disposition != '\"') {
		    if (*disposition == '\\') {
			++disposition;
			if (!*disposition) return 0;
		    }
		    ++disposition;
		}
		if (!*disposition) return 0;
	    }
	    else if (*disposition == '(') {
		SkipWhitespace(&disposition);
		if (!disposition) return 0;
		disposition--;
	    }
	    disposition++;
	}

	/* Skip over ";" and trailing whitespace */
	disposition++;
	SkipWhitespace(&disposition);
	if (!disposition) return 0;

	/*
	 * If we're not looking at a "filename" token, go back
	 * and look for another ";".  Otherwise skip it and
	 * trailing whitespace.
	 */
	if (strncasecmp(disposition, "filename", 8) != 0) continue;
	disposition += 8;
	if (!isspace(*disposition) && *disposition != '=' &&
	    *disposition != '(') {
	    continue;
	}
	SkipWhitespace(&disposition);
	if (!disposition) return 0;

	/* If we're looking at a ";", we found what we're looking for */
	if (*disposition++ == '=') break;
    }

    SkipWhitespace(&disposition);
    if (!disposition) return 0;
      
    if (!alloced) {
	value = xmalloc(alloced = VALUEGROWSIZE);
    }

    /* Copy value into buffer */
    to = value;
    left = alloced - 1;
    if (*disposition == '\"') {
	/* Quoted-string */
	disposition++;
	while (*disposition && *disposition != '\"') {
	    if (!--left) {
		alloced += VALUEGROWSIZE;
		value = xrealloc(value, alloced);
		to = value + alloced - left - 2;
	    }
#if 0
	    if (*disposition == '\\') {
		disposition++;
		if (!*disposition) return 0;
	    }
#endif
	    *to++ = *disposition++;
	}
	if (!*disposition) return 0;
    }
    else {
	/* Just a token */
	while (*disposition && !isspace(*disposition) &&
	       *disposition != '(') {
	    if (!--left) {
		alloced += VALUEGROWSIZE;
		value = xrealloc(value, alloced);
		to = value + alloced - left - 2;
	    }
	    *to++ = *disposition++;
	}
    }
    *to = '\0';
    return value;
}    

