/*
 * tclXcomma.c --
 *
 * Tcl commands to parse comma-separated ascii exchange formats.
 *---------------------------------------------------------------------------
 * Copyright 1994 Karl Lehenbauer.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tclXcomma.c,v 2.0 1994/12/23 20:25:10 karl Exp $
 *-----------------------------------------------------------------------------
 */

#include "tclExtdInt.h"


/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_CommaSplitCmd --
 *
 * Implements the `comma_split' Tcl command:
 *    comma_split $line
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *-----------------------------------------------------------------------------
 */
int
Tcl_CommaSplitCmd (notUsed, interp, argc, argv)
    ClientData   notUsed;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    char *first, *next;
    char c;

    if (argc != 2) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
                          " string", (char *) NULL);
        return TCL_ERROR;
    }

    /* handle the trivial case */
    if (STREQU (argv[1], "")) return TCL_OK;

    first = next = argv[1];

    while (1) {
        c = *next;

	/* if the first character of this part is a quote, scan to
	 * the closing quote, make that a field, and update */

	if (c == '"') {
	    next = ++first;
	    while (1) {
	        c = *next;
	        if (c == '\0') goto format_error;

	        if (c == '"') {
		    *next = '\0';
		    Tcl_AppendElement (interp, first);

		    /* skip the closing quote that we overwrote, and the
		     * following comma if there is one.
		     */

		    ++next;
		    c = *next;
		    if (c == '\0')
			return TCL_OK;
		    if (c != ',') {
		      format_error:
			Tcl_ResetResult (interp);
			Tcl_AppendResult (interp, "format error in string: ",
					  argv[0], " ", (char *) NULL);
			return TCL_ERROR;
		    }
		    first = ++next;
		    break;
		}
		next++;
	    }
	    continue;
	}

	/* if we get here, we're at the start of a field */
	next = first;
	while (1) {
	    c = *next;

	    if (c == '\0') {
		Tcl_AppendElement (interp, first);
		    return TCL_OK;
	    }

	    if (c == ',') {
		*next = '\0';
		Tcl_AppendElement (interp, first);
		first = ++next;
		break;
	    }
	    next++;
	}
    }
    Tcl_AppendElement (interp, first);
    return TCL_OK;
}


/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_CommaJoinCmd --
 *
 * Implements the `comma_join' Tcl command:
 *    comma_join $list
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *-----------------------------------------------------------------------------
 */
int
Tcl_CommaJoinCmd (notUsed, interp, argc, argv)
    ClientData   notUsed;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int listArgc;
    char **listArgv;
    char *s;
    int listIdx, didField;

    if (argc != 2) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
                          " list", (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_SplitList (interp, argv[1], &listArgc, &listArgv) != TCL_OK) {
	return TCL_ERROR;
    }

    didField = 0;
    for (listIdx = 0; listIdx < listArgc; listIdx++) {
	for (s = listArgv[listIdx]; *s != '\0'; s++) {
	    if (*s == ',') {
		if (didField) {
		    Tcl_AppendResult (interp, ",", (char *)NULL);
		}
		didField = 1;
		Tcl_AppendResult (interp, "\"", listArgv[listIdx], 
		    "\"", (char *)NULL);
		goto reloop;
	    }
	}
	if (didField) {
	    Tcl_AppendResult (interp, ",", (char *)NULL);
	}
	didField = 1;
	Tcl_AppendResult (interp, listArgv[listIdx], (char *)NULL);
      reloop: ;
    }
    return TCL_OK;
}


/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_LassignArrayCmd --
 *     Implements the TCL lassign_array command:
 *         lassign_array list arrayname elementname ?elementname...?
 *
 * Results:
 *      Standard TCL results.
 *
 *-----------------------------------------------------------------------------
 */
int
Tcl_LassignArrayCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int        listArgc, listIdx, idx, remaining;
    char     **listArgv;
    char      *varValue;

    if (argc < 4) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv [0], 
                          " list arrayname elementname ?elementname..?", (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_SplitList (interp, argv[1], &listArgc, &listArgv) == TCL_ERROR)
        return TCL_ERROR;

    for (idx = 3, listIdx = 0; idx < argc; idx++, listIdx++) {
        varValue = (listIdx < listArgc) ? listArgv[listIdx] : "" ;
        if (Tcl_SetVar2 (interp, argv[2], argv[idx], varValue,
	    TCL_LEAVE_ERR_MSG) == NULL) {
            goto error_exit;
        }
    }
    remaining = listArgc - argc + 3;
    if (remaining > 0) {
        Tcl_SetResult (interp,
                       Tcl_Merge (remaining, listArgv + argc - 3),
                       TCL_DYNAMIC);
    }
    ckfree((char *) listArgv);
    return TCL_OK;

  error_exit:
    ckfree((char *) listArgv);
    return TCL_ERROR;
}


/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_LassignFieldsCmd --
 *     Implements the TCL lassign_fields command:
 *         lassign_fields list fieldpositionarray arrayname fieldname ?fieldname...?
 *
 * Results:
 *      Standard TCL results.
 *
 *-----------------------------------------------------------------------------
 */
int
Tcl_LassignFieldsCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int        listArgc, listIdx, idx, remaining;
    char     **listArgv;
    char      *varValue;
    char      *fieldPositionText;

    if (argc < 5) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv [0], 
                          " list fieldpositionarray arrayname fieldname ?fieldname..?", (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_SplitList (interp, argv[1], &listArgc, &listArgv) == TCL_ERROR)
        return TCL_ERROR;

    for (idx = 4; idx < argc; idx++, listIdx++) {
	if ((fieldPositionText = Tcl_GetVar2 (interp, argv[2], argv[idx], 
	        TCL_LEAVE_ERR_MSG)) == NULL)
	    goto error_exit;

        if (Tcl_GetInt (interp, fieldPositionText, &listIdx) != TCL_OK) goto error_exit;

        varValue = (listIdx < listArgc) ? listArgv[listIdx] : "" ;

	/* if the array name field is empty, assign fields to
	 * variables, else assign fields to elements of the named
	 * array. */
	if (*argv[3] == '\0') {
	    if (Tcl_SetVar (interp, argv[idx], varValue,
		    TCL_LEAVE_ERR_MSG) == NULL)
		goto error_exit;
	} else {
	    if (Tcl_SetVar2 (interp, argv[3], argv[idx], varValue,
		    TCL_LEAVE_ERR_MSG) == NULL)
		goto error_exit;
	}
    }
    ckfree((char *) listArgv);
    return TCL_OK;

  error_exit:
    ckfree((char *) listArgv);
    return TCL_ERROR;
}
