/* 
 * tclXinet.c --
 *
 *  Internetworking Extensions.
 *-----------------------------------------------------------------------------
 * Copyright 1994 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tclXinet.c,v 2.0 1994/12/23 20:25:35 karl Exp $
 *-----------------------------------------------------------------------------
 */

#include "tclExtdInt.h"

#ifdef HAVE_GETHOSTBYNAME

#include <sys/types.h>
#ifndef NO_SYS_SOCKET_H
#    include <sys/socket.h>
#endif
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#ifndef HAVE_BCOPY
#    define bcopy(from, to, length)    memmove((to), (from), (length))
#endif
#ifndef HAVE_BZERO
#    define bzero(to, length)          memset((to), '\0', (length))
#endif

extern int h_errno;


/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_InetCmd --
 *     Implements the TCL inet command:
 *         inet hostbyname hostname officialNameVar aliasesVar addressesVar
 *
 * Results:
 *      Standard TCL results.
 *
 *-----------------------------------------------------------------------------
 */
int
Tcl_InetCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (argc < 3) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv [0], 
                          " option string ?string?...", (char *) NULL);
        return TCL_ERROR;
    }

    if (STREQU (argv[1], "hostbyname")) {
        struct hostent *hostEntry;
	struct in_addr address;
	int aliasIndex;
	char **addresses;

    if (argc != 6) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv [0], 
		" hostbyname hostname officialNameVar aliasesVar addressesVar",
		(char *) NULL);
        return TCL_ERROR;
    }
        hostEntry = gethostbyname (argv[2]);

        if (hostEntry == NULL) {
            char  *errorMsg;
            char  *errorCode;

            switch (h_errno) {
              case HOST_NOT_FOUND:
                errorMsg = "host not found";
		errorCode = "HOST_NOT_FOUND";
                break;
              case TRY_AGAIN:
                errorMsg = "try again";
		errorCode = "TRY_AGAIN";
                break;
              case NO_RECOVERY:
                errorMsg = "unrecordable server error";
		errorCode = "NO_RECOVERY";
                break;
              case NO_DATA:
                errorMsg = "no data";
		errorCode = "NO_DATA";
                break;
            }
	    Tcl_SetErrorCode (interp, "INET", errorCode, errorMsg,
	                     (char *)NULL);
            Tcl_AppendResult (interp, "host name lookup failure: ",
                              argv[2], " (", errorMsg, ")",
                              (char *) NULL);
            return TCL_ERROR;
        }

        if (Tcl_SetVar (interp, argv[3], hostEntry->h_name,
	    TCL_LEAVE_ERR_MSG) == (char *)NULL) {
		return TCL_ERROR;
	}

        Tcl_SetVar (interp, argv[4], "", 0);
	for (aliasIndex = 0; hostEntry->h_aliases[aliasIndex] != (char *)NULL; aliasIndex++) {
	    if (Tcl_SetVar (interp, argv[4], hostEntry->h_aliases[aliasIndex], 
		TCL_LEAVE_ERR_MSG | 
		TCL_APPEND_VALUE | TCL_LIST_ELEMENT) == (char *)NULL) {
		    return TCL_ERROR;
	    }
	}

        Tcl_SetVar (interp, argv[5], "", 0);
        addresses = hostEntry->h_addr_list;
	while (*addresses != (char *)NULL) {
             bcopy (*addresses, &address, hostEntry->h_length);

	    if (Tcl_SetVar (interp, argv[5], inet_ntoa (address), 
		TCL_LEAVE_ERR_MSG | TCL_LEAVE_ERR_MSG |
		TCL_APPEND_VALUE | TCL_LIST_ELEMENT) == (char *)NULL) {
		    return TCL_ERROR;
	    }
	    addresses++;
	}
	return TCL_OK;
    }
    Tcl_AppendResult (interp, tclXWrongArgs, argv [0], 
	    ": bad option: hostbyname",
	    (char *) NULL);
    return TCL_ERROR;
}

#endif
