/*
 * tclXserver.c --
 *
 * High level commands for connecting to TCP/IP based servers.
 *---------------------------------------------------------------------------
 * Copyright 1991-1994 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#include "tclExtdInt.h"

#ifdef HAVE_GETHOSTBYNAME

#include <sys/types.h>
#ifndef NO_SYS_SOCKET_H
#    include <sys/socket.h>
#endif
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#ifndef INADDR_NONE
#    define INADDR_NONE  ((long) -1)
#endif

#ifndef HAVE_BCOPY
#    define bcopy(from, to, length)    memmove((to), (from), (length))
#endif
#ifndef HAVE_BZERO
#    define bzero(to, length)          memset((to), '\0', (length))
#endif

extern int h_errno;

/*
 * Prototypes of internal functions.
 */
static int
ReturnGetHostError _ANSI_ARGS_((Tcl_Interp *interp,
                                char       *host));

static struct hostent *
InfoGetHostByName _ANSI_ARGS_((Tcl_Interp *interp,
                               int         argc,
                               char      **argv));

/*
 *-----------------------------------------------------------------------------
 *
 * ReturnGetHostError --
 *
 *   Return an error message when gethostbyname or gethostbyaddr fails.
 *
 * Parameters:
 *   o interp (O) - The error message is returned in the result.
 *   o host (I) - Host name or address that got the error.
 * Globals:
 *   o h_errno (I) - The list of file handles to parse, may be empty.
 * Returns:
 *   Always returns TCL_ERROR.
 *-----------------------------------------------------------------------------
 */
static int
ReturnGetHostError (interp, host)
    Tcl_Interp *interp;
    char       *host;
{
    char  *errorMsg;
    char  *errorCode;

    switch (h_errno) {
      case HOST_NOT_FOUND:
	errorCode = "HOST_NOT_FOUND";
        errorMsg = "host not found";
        break;
      case TRY_AGAIN:
	errorCode = "TRY_AGAIN";
        errorMsg = "try again";
        break;
      case NO_RECOVERY:
	errorCode = "NO_RECOVERY";
        errorMsg = "unrecordable server error";
        break;
      case NO_DATA:
	errorCode = "NO_DATA";
        errorMsg = "no data";
        break;
    }
    Tcl_SetErrorCode (interp, "INET", errorCode, errorMsg, (char *)NULL);
    Tcl_AppendResult (interp, "host lookup failure: ",
                      host, " (", errorMsg, ")",
                      (char *) NULL);
    return TCL_ERROR;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerOpenCmd --
 *     Implements the TCL server_open command:
 *
 *        server_open ?option? host service
 *
 *  Opens a stream socket to the specified host (host name or IP address),
 *  service name or port number.  Options maybe -buf or -nobuf, and
 *  -hostip hostname.
 *
 * Results:
 *   If successful, a pair Tcl fileids are returned for -buf or a single fileid
 * is returned for -nobuf.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerOpenCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char               *host, *service;
    int                 socketFD = -1, socketFD2 = -1, nextArg, buffered;
    struct hostent     *hostEntry;
    struct sockaddr_in  server, local;
    FILE               *filePtr;
    int                 specifiedLocalIp = 0;
    int			myPort;

    /*
     * Parse arguments.
     */
    if ((argc < 3) || (argc > 6)) {
	goto tclArgError;
    }

    bzero (&local, sizeof (local));
    nextArg = 1;
    buffered = TRUE;
    argc--;
    while (argc > 2) {
        if (STREQU ("-buf", argv [nextArg])) {
            buffered = TRUE;
	    nextArg++;
	    argc--;
        } else if (STREQU ("-nobuf", argv [nextArg])) {
            buffered = FALSE;
	    nextArg++;
	    argc--;
        } else if (STREQU ("-myip", argv [nextArg])) {
	    local.sin_addr.s_addr = inet_addr (argv [++nextArg]);
	    if (local.sin_addr.s_addr == INADDR_NONE) {
		Tcl_AppendResult (interp, "malformed address: \"",
		    argv[nextArg], "\"", (char *) NULL);
		return TCL_ERROR;
	    }
	    local.sin_family = AF_INET;
	    nextArg++;
	    argc -= 2;
	    specifiedLocalIp = 1;
        } else if (STREQU ("-myport", argv [nextArg])) {
	    if (Tcl_GetInt (interp, argv [++nextArg], &myPort) != TCL_OK)
		return TCL_ERROR;
	    local.sin_port = htons (myPort);
	    nextArg++;
	    argc -= 2;
        } else {
            Tcl_AppendResult (interp, "expected one of \"-buf\", \"-nobuf\"",
			      " \"-myip\" or \"-myport\"",
                              ", got \"", argv [nextArg], "\"", (char *) NULL);
            return TCL_ERROR;
        }
    }

    if (argc != 2) {
      tclArgError:
        Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
                          " ?option? host service|port", (char *) NULL);
        return TCL_ERROR;
    }

    host = argv [nextArg];
    service = argv [nextArg + 1];

    /*
     * Convert service number or lookup the service name.
     */
    bzero (&server, sizeof (server));

    if (ISDIGIT (service [0])) {
        int  port;
        
        if (Tcl_GetInt (interp, service, &port) != TCL_OK)
            return TCL_ERROR;
        server.sin_port = htons (port);
    } else {
        struct servent *servEntry;

        servEntry = getservbyname (service, NULL);
        if (servEntry == NULL) {
            Tcl_AppendResult (interp, "unknown service: ", service,
                              (char *) NULL);
	    Tcl_SetErrorCode (interp, "INET", "UNKNOWN_SERVICE",
				"unknown service", (char *)NULL);
            return TCL_ERROR;
        }
        server.sin_port = servEntry->s_port;
    }

    /*
     * Convert IP address or lookup host name.
     */
    server.sin_addr.s_addr = inet_addr (host);
    if (server.sin_addr.s_addr != INADDR_NONE) {
        server.sin_family = AF_INET;
        hostEntry = NULL;
    } else {
        hostEntry = gethostbyname (host);
        if (hostEntry == NULL)
            return ReturnGetHostError (interp, host);

        server.sin_family = hostEntry->h_addrtype;
        bcopy (hostEntry->h_addr_list [0], &server.sin_addr,
               hostEntry->h_length);
        hostEntry->h_addr_list++;
    }

    /*
     * Open a socket and connect to the server.  If the connect fails and
     * other addresses are available, try them.
     */
    socketFD = socket (server.sin_family, SOCK_STREAM, 0);
    if (socketFD < 0)
        goto unixError;

    if (specifiedLocalIp) {
	if (bind (socketFD, (struct sockaddr *) &local, sizeof (local)) < 0) {
	    goto unixError;
	}
    }
    while (TRUE) {
        if (connect (socketFD, (struct sockaddr *) &server,
                     sizeof (server)) >= 0)
            break;  /* Got it */

        if ((hostEntry == NULL) || (hostEntry->h_addr_list [0] == NULL))
            goto unixError;

        /*
         * Try next address.
         */
        bcopy (hostEntry->h_addr_list [0], &server.sin_addr,
               hostEntry->h_length);
        hostEntry->h_addr_list++;
    }

    /*
     * Set up stdio FILE structures.  If buffered, a pair (read/write) is 
     * returned.  If not buffered, a single one is returned.
     */
    if (!buffered) {
        filePtr = Tcl_SetupFileEntry (interp, socketFD,
                                      TCL_FILE_READABLE | TCL_FILE_WRITABLE);
        if (filePtr == NULL)
            goto errorExit;

        setbuf (filePtr, NULL);
        sprintf (interp->result, "file%d", socketFD);
        return TCL_OK;
    }

    if (Tcl_SetupFileEntry (interp, socketFD, TCL_FILE_READABLE) == NULL)
        goto errorExit;

    socketFD2 = dup (socketFD);
    if (socketFD2 < 0)
        goto unixError;

    if (Tcl_SetupFileEntry (interp, socketFD2, TCL_FILE_WRITABLE) == NULL)
        goto errorExit;

    sprintf (interp->result, "file%d file%d", socketFD, socketFD2);
    return TCL_OK;

    /*
     * Exit points for errors.
     */
  unixError:
    interp->result = Tcl_PosixError (interp);

  errorExit:
    if (socketFD >= 0)
        Tcl_CloseForError (interp, socketFD);
    if (socketFD2 >= 0)
        Tcl_CloseForError (interp, socketFD2);
    return TCL_ERROR;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerCreateCmd --
 *     Implements the TCL server_create command:
 *
 *        server_create ?options?
 *
 *  Creates a socket, binds the address and port on the local machine 
 * (optionally specified by the caller), and starts the port listening 
 * for connections by calling listen (2).
 *
 *  Options may be "-myip ip_address", "-myport port_number",
 *   and "-backlog backlog"
 *
 * Results:
 *   If successful, a Tcl fileid is returned.
 *
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerCreateCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char               *host, *service;
    int                 socketFD = -1, buffered, nextArg;
    struct hostent     *hostEntry;
    struct sockaddr_in  local;
    FILE               *filePtr;
    int			myPort;
    int			backlog = 5;

    /*
     * Parse arguments.
     */
    if ((argc & 1) != 1) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
                          " ?options?", (char *) NULL);
        return TCL_ERROR;
    }

    bzero (&local, sizeof (local));
    local.sin_family = AF_INET;
    local.sin_addr.s_addr = INADDR_ANY;
    nextArg = 1;
    buffered = TRUE;
    argc--;

    while (argc > 0) {
        if (STREQU ("-myip", argv [nextArg])) {
	    local.sin_addr.s_addr = inet_addr (argv [++nextArg]);
	    if (local.sin_addr.s_addr == INADDR_NONE) {
		Tcl_AppendResult (interp, "malformed address: \"",
		    argv[nextArg], "\"", (char *) NULL);
		return TCL_ERROR;
	    }
	    nextArg++;
	    argc -= 2;
        } else if (STREQU ("-myport", argv [nextArg])) {
	    if (Tcl_GetInt (interp, argv [++nextArg], &myPort) != TCL_OK)
		return TCL_ERROR;
	    local.sin_port = htons (myPort);
	    nextArg++;
	    argc -= 2;
        } else if (STREQU ("-backlog", argv [nextArg])) {
	    if (Tcl_GetInt (interp, argv [++nextArg], &backlog) != TCL_OK)
		return TCL_ERROR;
	    nextArg++;
	    argc -= 2;
        } else {
            Tcl_AppendResult (interp, "expected  ",
			      " \"-myip\", \"-myport\" or \"-backlog\"",
                              ", got \"", argv [nextArg], "\"", (char *) NULL);
            return TCL_ERROR;
        }
    }

    /*
     * Open a socket and bind an address and port to it.
     */
    socketFD = socket (local.sin_family, SOCK_STREAM, 0);
    if (socketFD < 0)
        goto unixError;

    if (bind (socketFD, (struct sockaddr *) &local, sizeof (local)) < 0) {
	goto unixError;
    }

    if (listen (socketFD, backlog) < 0) goto unixError;

    if (Tcl_SetupFileEntry (interp, socketFD, TCL_FILE_READABLE) == NULL)
        goto errorExit;

    sprintf (interp->result, "file%d", socketFD);
    return TCL_OK;

    /*
     * Exit points for errors.
     */
  unixError:
    interp->result = Tcl_PosixError (interp);

  errorExit:
    if (socketFD >= 0)
        Tcl_CloseForError (interp, socketFD);
    return TCL_ERROR;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerAcceptCmd --
 *     Implements the TCL server_accept command:
 *
 *        server_accept ?option?
 *
 *  Accepts an IP connection request to a socket created by server_create.
 *  Option maybe -buf or -nobuf.
 *
 * Results:
 *   If successful, a pair Tcl fileids are returned for -buf or a single fileid
 * is returned for -nobuf.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerAcceptCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char               *host, *service;
    int                 acceptSocketFD, addrLen;
    int                 socketFD = -1, socketFD2 = -1, nextArg, buffered;
    struct hostent     *hostEntry;
    struct sockaddr_in  connectSocket;
    FILE	       *acceptFilePtr;
    int			myPort;

    /*
     * Parse arguments.
     */
    if (argc > 3) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv[0], " file",
                          " ?option?", (char *) NULL);
        return TCL_ERROR;
    }

    bzero (&connectSocket, sizeof (connectSocket));
    buffered = TRUE;

    if (argc == 3) {
        if (STREQU ("-buf", argv [2])) {
            buffered = TRUE;
        } else if (STREQU ("-nobuf", argv [2])) {
            buffered = FALSE;
        } else {
            Tcl_AppendResult (interp, "expected \"-buf\" or \"-nobuf\"",
                              ", got \"", argv [nextArg], "\"", (char *) NULL);
            return TCL_ERROR;
        }
    }

    /*
     * Accept a socket connection on the socket created by server_create.
     */

    if (Tcl_GetOpenFile (interp, argv[1], 0, 1, &acceptFilePtr) == TCL_ERROR) {
	goto unixError;
    }
    acceptSocketFD = fileno (acceptFilePtr);
    addrLen = sizeof (connectSocket);
    socketFD = accept (acceptSocketFD, 
		       (struct sockaddr *)&connectSocket, 
		       &addrLen);
    if (socketFD < 0)
	goto unixError;

    /*
     * Set up stdio FILE structures.  If buffered, a pair (read/write) is 
     * returned.  If not buffered, a single one is returned.
     */
    if (!buffered) {
        acceptFilePtr = Tcl_SetupFileEntry (interp, socketFD,
                                      TCL_FILE_READABLE | TCL_FILE_WRITABLE);
        if (acceptFilePtr == NULL)
            goto errorExit;

        setbuf (acceptFilePtr, NULL);
        sprintf (interp->result, "file%d", socketFD);
        return TCL_OK;
    }

    if (Tcl_SetupFileEntry (interp, socketFD, TCL_FILE_READABLE) == NULL)
        goto errorExit;

    socketFD2 = dup (socketFD);
    if (socketFD2 < 0)
        goto unixError;

    if (Tcl_SetupFileEntry (interp, socketFD2, TCL_FILE_WRITABLE) == NULL)
        goto errorExit;

    sprintf (interp->result, "file%d file%d", socketFD, socketFD2);
    return TCL_OK;

    /*
     * Exit points for errors.
     */
  unixError:
    interp->result = Tcl_PosixError (interp);

  errorExit:
    if (socketFD >= 0)
        Tcl_CloseForError (interp, socketFD);
    if (socketFD2 >= 0)
        Tcl_CloseForError (interp, socketFD2);
    return TCL_ERROR;
}

/*
 *-----------------------------------------------------------------------------
 *
 * InfoGetHostByName --
 *
 *   Validate arguments and call gethostbyname for the server_info options
 * that return info about a host name.
 *
 * Parameters:
 *   o interp (O) - The error message is returned in the result.
 *   o argc, argv (I) - Command argments.
 * Returns:
 *   Pointer to the host entry or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static struct hostent *
InfoGetHostByName (interp, argc, argv)
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    struct hostent *hostEntry;

    if (argc != 3) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv [0], " ", argv [1],
                          " hostname", (char *) NULL);
        return NULL;
    }

    hostEntry = gethostbyname (argv [2]);
    if (hostEntry == NULL) {
        ReturnGetHostError (interp, argv [2]);
        return NULL;
    }
    return hostEntry;
}

/*
 *-----------------------------------------------------------------------------
 *
 * InfoGetHostByAddr --
 *
 *   Validate arguments and call gethostbyaddr for the server_info options
 * that return info about a host address.
 *
 * Parameters:
 *   o interp (O) - The error message is returned in the result.
 *   o argc, argv (I) - Command argments.
 * Returns:
 *   Pointer to the host entry or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static struct hostent *
InfoGetHostByAddr (interp, argc, argv)
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    struct hostent *hostEntry;
    struct in_addr address;

    if (argc != 3) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv [0], " ", argv [1],
                          " hostaddr", (char *) NULL);
        return NULL;
    }

    address.s_addr = inet_addr (argv[2]);
    if (address.s_addr == INADDR_NONE) {
        Tcl_AppendResult (interp, "malformed address: ", argv [0], " ", 
	    argv [1], " ", argv[2], (char *) NULL);
        return NULL;
    }

    hostEntry = gethostbyaddr ((const char *)&address, sizeof (address), AF_INET);
    if (hostEntry == NULL) {
        ReturnGetHostError (interp, argv [2]);
        return NULL;
    }
    return hostEntry;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerInfoCmd --
 *     Implements the TCL server_info command:
 *
 *        server_info addresses hostname
 *        server_info addresses hostname
 *
 * Results:
 *   For hostname, a list of address associated with the host.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerInfoCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    struct hostent *hostEntry;
    struct in_addr  inAddr;
    int             idx;

    if (argc < 2) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
                          " option ...", (char *) NULL);
        return TCL_ERROR;
    }

    if (STREQU (argv [1], "addresses")) {
        hostEntry = InfoGetHostByName (interp, argc, argv);
        if (hostEntry == NULL)
            return TCL_ERROR;

        for (idx = 0; hostEntry->h_addr_list [idx] != NULL; idx++) {
            bcopy (hostEntry->h_addr_list [idx], &inAddr,
                   hostEntry->h_length);
            Tcl_AppendElement (interp, inet_ntoa (inAddr));
        }
        return TCL_OK;
    }

    if (STREQU (argv [1], "address_name")) {
        hostEntry = InfoGetHostByAddr (interp, argc, argv);
        if (hostEntry == NULL)
            return TCL_ERROR;

        for (idx = 0; hostEntry->h_addr_list [idx] != NULL; idx++) {
            bcopy (hostEntry->h_addr_list [idx], &inAddr,
                   hostEntry->h_length);
            Tcl_AppendElement (interp, hostEntry->h_name);
        }
        return TCL_OK;
    }

    if (STREQU (argv [1], "official_name")) {
        hostEntry = InfoGetHostByName (interp, argc, argv);
        if (hostEntry == NULL)
            return TCL_ERROR;

        Tcl_SetResult (interp, hostEntry->h_name, TCL_STATIC);
        return TCL_OK;
    }

    if (STREQU (argv [1], "aliases")) {
        hostEntry = InfoGetHostByName (interp, argc, argv);
        if (hostEntry == NULL)
            return TCL_ERROR;

        for (idx = 0; hostEntry->h_aliases [idx] != NULL; idx++) {
            Tcl_AppendElement (interp, hostEntry->h_aliases [idx]);
        }
        return TCL_OK;
    }

    Tcl_AppendResult (interp, "invalid option \"", argv [1],
                      "\", expected one of \"addresses\", \"official_name\",",
                      "\"address_name\"or \"aliases\"", (char *) NULL);
    return TCL_ERROR;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerSendCmd --
 *     Implements the TCL server_send command:
 *
 *        server_send fileid message [nonewline out_of_band dont_route]
 *
 * Results:
 *   Nothing.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerSendCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    FILE	*filePtr;
    int          bytesAttempted;
    int		 bytesSent;
    int		 flags = 0;
    int		 nonewline = 0;
    int		 idx;

    if ((argc < 3) || (argc > 5)) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
                          " fileid message [options...]", (char *) NULL);
        return TCL_ERROR;
    }

    for (idx = 3; idx < argc; idx++) {
	if (STREQU (argv[idx], "nonewline")) {
	    nonewline = 1;
	} else if (STREQU (argv[idx], "out_of_band")) {
	    flags |= MSG_OOB;
	} else if (STREQU (argv[idx], "dont_route")) {
	    flags |= MSG_DONTROUTE;
	} else {
	    Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
			      " bad option \"",
			      argv[idx], 
			      "\", should be one of \"nonewline\",",
			      " \"out_of_band\",",
			      " or \"dont_route\"", (char *) NULL);
	    return TCL_ERROR;
	}
    }
    if (Tcl_GetOpenFile (interp, argv[1], FALSE, FALSE, /* No checking */
	&filePtr) != TCL_OK)
	    return TCL_ERROR;

    bytesAttempted = strlen (argv[2]);
    if (!nonewline)
	argv[2][bytesAttempted++] = '\n';

    if ((bytesSent = send (fileno (filePtr), (const void *)argv[2], 
	      bytesAttempted, flags)) < 0) {
		  interp->result = Tcl_PosixError (interp);
		  return TCL_ERROR;
    }

    if (bytesSent != bytesAttempted) {
	    Tcl_AppendResult (interp, argv[0],
			      " sent fewer bytes than attempted",
			      (char *)NULL);
    }
    return TCL_OK;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerReceiveCmd --
 *     Implements the TCL server_receive command:
 *
 *        server_receive fileid length [out_of_band peek wait_all]
 *
 * Results:
 *   The data received.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerReceiveCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    FILE	*filePtr;
    int          bytesAttempted;
    int		 bytesReceived;
    int		 flags = 0;
    int		 nonewline = 0;
    int		 idx;
    char        *dataBuffer;
    int          keepNewline = 0;

    if (argc < 3) {
        Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
                          " fileid length [options...]", (char *) NULL);
        return TCL_ERROR;
    }

    for (idx = 3; idx < argc; idx++) {
	if (STREQU (argv[idx], "out_of_band")) {
	    flags |= MSG_OOB;
	} else if (STREQU (argv[idx], "peek")) {
	    flags |= MSG_DONTROUTE;
	} else if (STREQU (argv[idx], "wait_all")) {
#ifdef MSG_WAITALL
	    flags |= MSG_WAITALL;
#else
	    Tcl_AppendResult (interp, 
                "socket option not available on this architecture ",
	        (char *) NULL);

            return TCL_ERROR;
#endif
	} else if (STREQU (argv[idx], "keepnewline")) {
	    keepNewline = 1;
	} else {
	    Tcl_AppendResult (interp, tclXWrongArgs, argv[0],
			      " bad option \"",
			      argv[idx], 
			      "\", should be one of \"out_of_band\",",
			      " \"peek\",",
			      " or \"wait_all\"", (char *) NULL);
	    return TCL_ERROR;
	}
    }
    if (Tcl_GetOpenFile (interp, argv[1], FALSE, FALSE, /* No checking */
	&filePtr) != TCL_OK)
	    return TCL_ERROR;

    if (Tcl_GetInt (interp, argv [2], &bytesAttempted) != TCL_OK)
	return TCL_ERROR;

    dataBuffer = ckalloc (bytesAttempted + 1);
    bytesReceived = recvfrom (fileno (filePtr), 
			      dataBuffer,
			      bytesAttempted,
			      flags,
			      (struct sockaddr *)NULL,
			      (int *)NULL);

    if (bytesReceived < 0) {
      interp->result = Tcl_PosixError (interp);
	return TCL_ERROR;
    }

    dataBuffer[bytesReceived] = '\0';
    if (!keepNewline && dataBuffer[bytesReceived - 1] == '\n') {
        dataBuffer[bytesReceived - 1] = '\0';
    }

    Tcl_SetResult (interp, dataBuffer, TCL_DYNAMIC);
    return TCL_OK;
}
#else

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerOpenCmd --
 *     Version of this command that return an error on systems that don't have
 * sockets.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerOpenCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Tcl_AppendResult (interp, argv [0], " not available on this system",
                      (char *) NULL);
    return TCL_ERROR;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerCreateCmd --
 *     Version of this command that return an error on systems that don't have
 * sockets.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerCreateCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Tcl_AppendResult (interp, argv [0], " not available on this system",
                      (char *) NULL);
    return TCL_ERROR;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerAcceptCmd --
 *     Version of this command that return an error on systems that don't have
 * sockets.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerAcceptCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Tcl_AppendResult (interp, argv [0], " not available on this system",
                      (char *) NULL);
    return TCL_ERROR;
}

/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerInfoCmd --
 *     Version of this command that return an error on systems that don't have
 * sockets.
 *-----------------------------------------------------------------------------
 */
int
Tcl_ServerInfoCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return Tcl_ServerOpenCmd (clientData, interp, argc, argv);
}
/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerSendCmd --
 *     Version of this command that return an error on systems that don't have
 * sockets.
 *-----------------------------------------------------------------------------
 */

int
Tcl_ServerSendCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return Tcl_ServerOpenCmd (clientData, interp, argc, argv);
}
/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_ServerReceiveCmd --
 *     Version of this command that return an error on systems that don't have
 * sockets.
 *-----------------------------------------------------------------------------
 */

int
Tcl_ServerReceiveCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    return Tcl_ServerOpenCmd (clientData, interp, argc, argv);
}
#endif /* HAVE_GETHOSTBYNAME */
