#ifdef notyet
/*
 * tclXsyslog.c --
 *
 * Tcl commands to access syslog.
 *---------------------------------------------------------------------------
 * Copyright 1994 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tclXsyslog.c,v 2.0 1994/12/23 20:26:01 karl Exp $
 *-----------------------------------------------------------------------------
 */

#include "tclExtdInt.h"


/*
 *-----------------------------------------------------------------------------
 *
 * Tcl_SyslogCmd --
 *     Implements the TCL syslog command:
 *
 *        syslog user ?name?
 *        id convert user <name>
 *
 *        id userid ?uid?
 *        id convert userid <uid>
 *
 *        id group ?name?
 *        id convert group <name>
 *
 *        id groupid ?gid?
 *        id convert groupid <gid>
 *
 *        id process
 *        id process parent
 *        id process group
 *        id process group set
 *
 *        id effective user
 *        id effective userid
 *
 *        id effective group
 *        id effective groupid
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *-----------------------------------------------------------------------------
 */

typedef struct syslog_code {
	char	*c_name;
	int	c_val;
} SYSLOG_CODE;

static SYSLOG_CODE tcl_syslog_priorities[] = {
#ifdef LOG_ALERT
	"alert",	LOG_ALERT,
#endif
#ifdef LOG_CRIT
	"crit",		LOG_CRIT,
#endif
#ifdef LOG_DEBUG
	"debug",	LOG_DEBUG,
#endif
#ifdef LOG_EMERG
	"emerg",	LOG_EMERG,
#endif
#ifdef LOG_ERR
	"err",		LOG_ERR,
	"error",	LOG_ERR,		/* DEPRECATED */
#endif
#ifdef LOG_INFO
	"info",		LOG_INFO,
#endif
#ifdef LOG_NOTICE
	"notice",	LOG_NOTICE,
#endif
#ifdef LOG_EMERG
	"panic", 	LOG_EMERG,		/* DEPRECATED */
#endif
#ifdef LOG_WARNING
	"warn",		LOG_WARNING,		/* DEPRECATED */
	"warning",	LOG_WARNING,
#endif
	NULL,		-1,
};

static SYSLOG_CODE syslog_facilitynames[] = {
#ifdef LOG_AUTH
	"auth",		LOG_AUTH,
#endif
#ifdef LOG_AUTHPRIV
	"authpriv",	LOG_AUTHPRIV,
#endif
#ifdef LOG_CRON
	"cron", 	LOG_CRON,
#endif
#ifdef LOG_DAEMON
	"daemon",	LOG_DAEMON,
#endif
#ifdef LOG_KERN
	"kern",		LOG_KERN,
#endif
#ifdef LOG_LPR
	"lpr",		LOG_LPR,
#endif
#ifdef LOG_MAIL
	"mail",		LOG_MAIL,
#endif
#ifdef LOG_NEWS
	"news",		LOG_NEWS,
#endif
#ifdef LOG_AUTHPRIV
	"security",	LOG_AUTH,		/* DEPRECATED */
#endif
#ifdef LOG_AUTHPRIV
	"syslog",	LOG_SYSLOG,
#endif
#ifdef LOG_USER
	"user",		LOG_USER,
#endif
#ifdef LOG_UUCP
	"uucp",		LOG_UUCP,
#endif
#ifdef LOG_LOCAL0
	"local0",	LOG_LOCAL0,
#endif
#ifdef LOG_LOCAL1
	"local1",	LOG_LOCAL1,
#endif
#ifdef LOG_LOCAL2
	"local2",	LOG_LOCAL2,
#endif
#ifdef LOG_LOCAL3
	"local3",	LOG_LOCAL3,
#endif
#ifdef LOG_LOCAL4
	"local4",	LOG_LOCAL4,
#endif
#ifdef LOG_LOCAL5
	"local5",	LOG_LOCAL5,
#endif
#ifdef LOG_LOCAL6
	"local6",	LOG_LOCAL6,
#endif
#ifdef LOG_LOCAL7
	"local7",	LOG_LOCAL7,
#endif
	NULL,		-1,
};

static SYSLOG_CODE syslog_optionnames[] = {
#ifdef LOG_PID
	"pid",		LOG_PID,
#endif
#ifdef LOG_CONS
	"console",	LOG_CONS,
#endif
#ifdef LOG_PERROR
	"perror",	LOG_PERROR,
#endif
	NULL,		-1,
};

static int
syslog_table_name_to_number(interp, idName, tablePtr, tableName, resultPtr)
Tcl_Interp *interp;
char *idName;
SYSLOG_CODE *tablePtr;
char *tableName;
int *resultPtr;
{
    SYSLOG_CODE *tabPtr;

    for (tabPtr = tablePtr; tabPtr->c_val != -1; tabPtr++) {
	if (STREQU (idName, tabPtr->c_name)) {
	    *resultPtr = tabPtr->c_val;
	    return TCL_OK;
	}
    }
    if (Tcl_StrToInt (idName, 0, resultPtr))
	return TCL_OK;


    Tcl_AppendResult (interp, tableName, " table element ", idName, 
		      " unknown", (char *) NULL);
    return TCL_ERROR;
}

Tcl_SyslogCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (argc < 2) 
        goto bad_args;

    /*
     * If the first argument is "log", log the message.
     */
    if (STREQU (argv[1], "log")) {
	int priority;

        if (argc != 4) {
            Tcl_AppendResult (interp, tclXWrongArgs, argv [0], 
                              " log priority message", (char *) NULL);

	    if (syslog_table_name_to_number(interp,
					    argv[2],
					    tcl_syslog_priorities,
					    "priority",
					    &priority))
            return TCL_ERROR;
        }
	syslog (priority, argv[3]);
    }

    /*
     * If the first argument is "close", close the log.
     */
    if (STREQU (argv[1], "close")) {
        if (argc != 2) {
            Tcl_AppendResult (interp, tclXWrongArgs, argv [0], 
                              " close", (char *) NULL);
            return TCL_ERROR;
        }
	closelog();
    }

    if (STREQU (argv[1], "mask")) {
	setlogmask(int);
    }

    /*
     * If the first argument is "open", open the log.
     */
    if (STREQU (argv[1], "open")) {
	int facility;

        if (argc != 5) {
            Tcl_AppendResult (interp, tclXWrongArgs, argv [0], 
                              " open ident log_options facility", (char *) NULL);
            return TCL_ERROR;
        }

	if (syslog_table_name_to_number(interp,
					argv[4],
					tcl_syslog_facilities,
					"facility",
					&facility))
	return TCL_ERROR;
	openlog(argv[2],,facility);
	return TCL_OK;
    }

    if (STREQU (argv[1], "mask")) {
	setlogmask(int);
    }

    Tcl_AppendResult (interp, tclXWrongArgs, argv [0], " arg ?arg..?",
                      (char *) NULL);
    return TCL_ERROR;

}

/*
 * arguments to setlogmask.
 */
#define	LOG_MASK(pri)	(1 << (pri))		/* mask for one priority */
#define	LOG_UPTO(pri)	((1 << ((pri)+1)) - 1)	/* all priorities through pri */

#endif
