/*
 * dpmulticast.c -- C Code for MLbRPC client commands.
 *
 * Copyright (c) 1994 Loughborough University of Technology
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  Loughborough University of Technology
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Author: Jon Knight <J.P.Knight@lut.ac.uk>
 *
 * $Log: dpmulticast.c,v $
 * Revision 1.1.1.1  1995/03/30  07:42:50  karl
 * Initial checkin of multicast-capable tcl-dp.
 *
 * Revision 1.7  1994/08/18  13:34:26  jon
 * Fixed bug concerning failure to respond to more than one response in every
 * retry period.  Also added a hash table to prevent the same (cached) reply
 * for a single host being accepted as a valid response more than once in
 * every transaction.
 *
 * Revision 1.6  1994/08/17  13:52:24  jon
 * Fixed it so that a zero timeout value to mlbrpc_clientexec doesn't result
 * in an immediate return with no result.  Now it sends the multicast request
 * and then sits there with an infinite timeout waiting for the requisit
 * number of replies (so the number of retries is ignored as long as it is
 * greater than zero).
 *
 * Revision 1.5  1994/08/17  13:17:41  jon
 * Replaced icky signals with select call which should work on all
 * architectures.  Also ensured that memory is allocated for the result
 * string to be passed by to the calling interpreter.
 *
 * Revision 1.4  1994/08/17  09:29:30  jon
 * Fixed positioning of _BSD_COMPAT definition to allow compilation on SGIs
 *
 * Revision 1.3  1994/08/17  08:45:21  jon
 * Now uses signals to allow retransmissions.
 *
 * Revision 1.2  1994/08/16  16:12:44  jon
 * First working version.  This interoperates with the prototype TCL scripts
 * but doesn't include any timeout and retries.
 *
 * Revision 1.1  1994/08/15  13:37:50  jon
 * Initial revision
 *
 */
#ifdef MULTICAST	/* Only compile this file if we're using multicast */
#ifdef sgi
#define _BSD_COMPAT
#endif /*IRIX*/
#include <stdio.h>
#include <assert.h>
#include <ctype.h>
#include "tk.h"
#include "dpInt.h"
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <signal.h>
#define FALSE 0
#define TRUE !FALSE

static char timedout;	/* Whether a request has timedout or not */

extern int errno;
extern char *sys_errlist[];

/*
 * Function: Mlbrpc_MakeTID
 *
 * Parameters: None
 *
 * Results: returns a character string containing a unique hex string
 *
 * Side Effects: The MLbRPC_ReqNum variable is incremented for every TID
 *	request generated.  This is required to prevent a single process from
 *	generating two or more identical TIDs within quick succession.
 *
 * Purpose: The function generates a TID string for an MLbRPC
 * transaction.  The TID so generated is formed from the unique host ID
 * of the machine on which this process is running, the time the call is
 * made, the process ID of this application and a constantly incremented
 * request counter.  Uniqueness is guaranteed as long as the host on
 * which this function is running is incapable of processing this
 * function more times than can be held in a long before wrapping round
 * in under one second.
 */

char *Mlbrpc_MakeTID()
{
	int hostid;
/*	long when;*/
	int pid;
	static long MLbRPC_ReqNum = 0;
	static char tid[256];

	hostid=gethostid();
/*	when=(long)time(NULL);*/
	pid=getpid();
/*	sprintf(tid,"%08x%08x%04x%08x",hostid,when,pid,MLbRPC_ReqNum++);*/
	sprintf(tid,"%08x%04x%08x",hostid,pid,MLbRPC_ReqNum++);
	return(tid);
}

/*
 * Function: Mlbrpc_ClientExec
 *
 * Parameters: string containing the code to be executed, the multicast
 *	IP group, port and ttl, the timeout in seconds, the number of
 *	retries to allow and the maximum number of replies desired.
 *
 * Results: returns a character string containing a list of the returned
 *	results.
 *
 * Side Effects: none
 *
 * Purpose: Client command to send off an MLbRPC request and await the replies
 */
char *Mlbrpc_ClientExec(code, group, port, ttl, timeout, retries, replies)
char *code;
char *group;
int port;
int ttl;
int timeout;
int retries;
int replies;
{
	extern int Tdp_inet_connect	_ANSI_ARGS_((char *host, int port,
                                             int server,
                                             int udp,
                                             int reuseAddr,
                                             int lingerTime));
	extern int Tdp_multicast_connect	_ANSI_ARGS_((char *group,
                                                     int port,
                                                     int ttl));
	extern void Mlbrpc_ExportTimeout	_ANSI_ARGS_(());
	char *tid;		/* Space for the TID */
	char *request;		/* String containing the outgoing request */
	char hostname[16];	/* String for our hostname/IP address */
	Tcl_DString resultlist;	/* Dynamic string for the list of results */
	char *alldone;		/* Pointer to final results */
	char returned[2048];	/* String for a single returned result */
	int attempts;		/* Number of attempts made */
	int answers;		/* Number of answers retrieved */
	int outsocket;		/* File descriptor for the outgoing code */
	int insocket;		/* File descriptor for the incoming results */
        struct sockaddr_in sockaddr;	/* Struct for local unicast socket */
        int res, len;		/* Temps for finding unicast port number */
	unsigned long ipaddr;	/* Our raw IP address */
	fd_set readmask;	/* Mask for the select reading insocket */
	struct timeval expire;	/* Timeout structure for select */
	Tcl_HashTable hostlist;	/* Table of hosts who've replied already */

	/* Generate a unique TID for this transaction */
	tid = Mlbrpc_MakeTID();

	/* Create a hash table for replying hosts' IP addresses */
	Tcl_InitHashTable(&hostlist, TCL_STRING_KEYS);

	/* Open a connection to the multicast group */
	outsocket=Tdp_multicast_connect(group,port,ttl);

	/* Open a unicast connection for the results */
	insocket = Tdp_inet_connect(NULL,0,0,1,1,0);

        /* Find the local port we're using for the connection. */
        len = sizeof (sockaddr);
        res = getsockname (insocket, (struct sockaddr *) &sockaddr, &len);

	ipaddr=ntohl(sockaddr.sin_addr.s_addr);
	sprintf(hostname,"%d.%d.%d.%d",	(ipaddr&0xFF000000)>>24,
					(ipaddr&0x00FF0000)>>16,
					(ipaddr&0x0000FF00)>>8,
					(ipaddr&0x000000FF));
        if (res < 0) {
            return NULL;
        } else  {
		request=(char *)malloc((56+strlen(code))*sizeof(char));
        	sprintf(request, "%s %d %s {%s}\n",hostname,
			ntohs(sockaddr.sin_port),tid,code);
        }

	/* Send the code to the multicast group */
	attempts = 0;
	answers = 0;
	timedout=TRUE;
	expire.tv_sec=timeout;
	expire.tv_usec=0;
	Tcl_DStringInit(&resultlist);
	do {
		int len = strlen(request);
		struct sockaddr_in addr;
		int addrLen,count,numBytes,status;
		int flags = 0;

		/* Send the request out */
/*		Mlbrpc_Export(outsocket,code); */
		if(timedout==TRUE) {
		        addrLen = sizeof(addr);
			memset((char *) &addr, 0, addrLen);
	        	getsockname (outsocket, (struct sockaddr *) &addr, &addrLen);
			addr.sin_addr.s_addr=inet_addr(group);
			if((status =sendto(outsocket,request, len, 0, 
				(struct sockaddr *)&addr,
				addrLen))==-1) {
				return NULL;
			}
			timedout=FALSE;
		}

		/* Get an answer back */
		FD_ZERO(&readmask);
		FD_SET(insocket,&readmask);
		addrLen = sizeof(addr);
		memset((char *) &addr, 0, addrLen);
		numBytes = 2048;
		if(timeout!=0) {
			if ((status = select (FD_SETSIZE, &readmask, 0, 0, &expire)) == -1) {
				return NULL;
			} else if (status == 0) {
				timedout=TRUE;
				attempts++;
			}
		}
		if(timedout==FALSE) {
			count = recvfrom(insocket, returned, numBytes, flags,
				(struct sockaddr *)&addr, &addrLen);
			/* If there was an error in recvfrom return a null
			 * string to the caller */
			if (count == -1) {
				return NULL;
			}
			ipaddr=ntohl(addr.sin_addr.s_addr);
			sprintf(hostname,"%d.%d.%d.%d",	(ipaddr&0xFF000000)>>24,
							(ipaddr&0x00FF0000)>>16,
							(ipaddr&0x0000FF00)>>8,
							(ipaddr&0x000000FF));
			if(Tcl_FindHashEntry(&hostlist,hostname)==NULL){
				int new;
				returned[count] = 0;
				answers++;
				Tcl_DStringAppendElement(&resultlist,returned);
				Tcl_SetHashValue(Tcl_CreateHashEntry(
					&hostlist,hostname, &new),
					(ClientData)hostname);
			}
		}
	} while(((attempts<retries)||(timedout==FALSE))&&(answers<replies));

	/* Shutdown sockets */
	close(insocket);
	close(outsocket);

	/* Free space allocated to the request */
	free(request);

	/* Delete hash table used to detect multiple response from one host */
	Tcl_DeleteHashTable(&hostlist);

	/* Return the results to the caller */
	alldone=(char *)malloc((Tcl_DStringLength(&resultlist)+1)*sizeof(char));
	strcpy(alldone,Tcl_DStringValue(&resultlist));
	Tcl_DStringFree(&resultlist);
	return alldone;
}

/*
 * Function: Mlbrpc_MakeTIDCmd
 *
 * Parameters: A (currently unused) set of Client Data, a pointer to the
 *	TCL interpreter in which the command is being executed, a count of
 *	the number of arguments being passed and the arguments themselves.
 *
 * Results: returns a TCL_OK result code
 *
 * Side Effects: Sets the Tcl Results data structure to a string containing
 *	the unique TID.
 *
 * Purpose: This function is an interface between the TCL interpreter and the
 *	Mlbrpc_MakeTID function.
 */
int
Mlbrpc_MakeTIDCmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
	Tcl_SetResult(interp,Mlbrpc_MakeTID(),TCL_VOLATILE);
	return TCL_OK;
}

/*
 * Function: Mlbrpc_ClientExecCmd
 *
 * Parameters: A (currently unused) set of Client Data, a pointer to the
 *	TCL interpreter in which the command is being executed, a count of
 *	the number of arguments being passed and the arguments themselves.
 *
 * Results: returns a TCL_OK result code
 *
 * Side Effects: Sets the Tcl Results data structure to a list
 *	containing the results from the exportation.
 *
 * Purpose: This function is an interface between the TCL interpreter and the
 *	Mlbrpc_ClientExec function.
 */
int
Mlbrpc_ClientExecCmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
	char *code;	/* Code/data to be exported */
	char *group;	/* Group to export it to */
	int port;	/* Port number of remote server */
	int temp;	/* Temp required for finding TTL */
	unsigned char ttl;	/* Time To Live for Multicast packets */
	int timeout;	/* How long to wait before timing out a request */
	int retries;	/* Number of retries to allow */
	int replies;	/* Number of replies to expect */
	char *answer;	/* The collection of replies */

	if(argc != 8) {
		interp->result = "wrong # args";
		return TCL_ERROR;
	}
	code = argv[1];
	group = argv[2];
	if(Tcl_GetInt(interp,argv[3],&port) != TCL_OK) {
		return TCL_ERROR;
	}
	if(Tcl_GetInt(interp,argv[4],&temp) != TCL_OK) {
		return TCL_ERROR;
	}
	ttl = (unsigned char)temp;
	if(Tcl_GetInt(interp,argv[5],&timeout) != TCL_OK) {
		return TCL_ERROR;
	}
	if(Tcl_GetInt(interp,argv[6],&retries) != TCL_OK) {
		return TCL_ERROR;
	}
	if(Tcl_GetInt(interp,argv[7],&replies) != TCL_OK) {
		return TCL_ERROR;
	}
	
	answer = Mlbrpc_ClientExec(code,group,port,temp,timeout,
		retries,replies);
	if (answer==NULL) {
		interp->result = sys_errlist[errno];
		return TCL_ERROR;
	} else {
		Tcl_SetResult(interp,answer,TCL_VOLATILE);
		return TCL_OK;
	}
}

/*
 * Function: Mlbrpc_Init
 *
 * Parameters: pointer to TCL interpreter
 *
 * Results: none
 *
 * Side Effects: Registers all commands in the TCL interpreter's data
 *	structure.
 *
 * Purpose: Initialise all the MLbRPC C support commands in the TCL
 *	interpreter
 */
Tcl_Interp *Mlbrpc_Init(interp)
    Tcl_Interp *interp;         /* Tcl interpreter */
{
    Tcl_CreateCommand(interp, "mlbrpc_maketid",
        (Tcl_CmdProc *)Mlbrpc_MakeTIDCmd,
        (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "mlbrpc_clientexec",
        (Tcl_CmdProc *)Mlbrpc_ClientExecCmd,
        (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    return interp;
}

#endif /* MULTICAST */

