#
## 		init.tcl:  Procedures to extend the safe interpreter
###		Written by George Porter <gporter@neosoft.com>
##		NeoWebScript-SA.  Copyright 1997 NeoSoft, Inc.  All rights reserved.
# 		$Id: init.tcl,v 1.14 1997/06/26 20:41:32 gporter Exp $

proc setup_safe_interpreter {sock} {
	set safeInterp interp_web_$sock

	$safeInterp alias dbstore SAFE_dbstore $safeInterp
	$safeInterp alias dbfetch SAFE_dbfetch $safeInterp
	$safeInterp alias dbkeys SAFE_dbkeys
	$safeInterp alias dbdelkey SAFE_dbdelkey
	$safeInterp alias dbexists SAFE_dbexists
	$safeInterp alias dbfiles SAFE_dbfiles
	$safeInterp alias delete_db_file SAFE_delete_db_file

	$safeInterp alias access_data_file SAFE_access_data_file $sock
	$safeInterp alias delete_data_file SAFE_delete_data_file $sock
	$safeInterp alias log_message SAFE_log_message $sock

	$safeInterp alias neo_clock_to_rfc850_gmt http_date
	$safeInterp alias html html
	$safeInterp alias neo_make_cookie SAFE_neo_make_cookie $sock
	$safeInterp alias remote_hostname SAFE_remote_hostname $sock
	$safeInterp alias random SAFE_random $sock
	$safeInterp alias estimate_hits_per_hour SAFE_estimate_hits_per_hour
	$safeInterp alias set_header SAFE_set_header

	$safeInterp alias load_virtual SAFE_load_virtual $sock
	$safeInterp alias load_file SAFE_load_file $sock
	$safeInterp alias load_cookies SAFE_load_cookies $sock
}

proc old_html {text {extra ""}} {
	global web_conf

	set sock $web_conf(currentSock)

	if {$extra == ""} {
		interp_web_$sock eval "global _tclReturn ; [list append _tclReturn $text]"
	} else {
		interp_web_$sock eval "global _tclReturn ; [list append _tclReturn [format <%s>%s</%s> $extra $text $extra]]"
	}
}

proc html {text {extra ""}} {
	global web_conf
	set sock $web_conf(currentSock)
	upvar #0 web_$sock array

	if {$extra == ""} {
		append array(_tclReturn) "$text"
	} else {
		append array(_tclReturn) "[format <%s>%s</%s> $extra $text $extra]"
	}
}

##########
# Procedures to handle the Neoscript directives
##########

proc handle_server_side_eval {ssiCode} {
	global web_conf errorInfo
	upvar #0 web_$web_conf(currentSock) array

	set safeInterp interp_web_$web_conf(currentSock)
	set array(_tclReturn) ""

	set tmp ""
	if {[catch {$safeInterp eval $ssiCode} result] == 1} {
		append tmp "\{NEOSCRIPT ERROR: $result\}"
		if {$web_conf(debugging)} {
			append tmp "\n<pre>$errorInfo</pre>"	
		}
		gui_handle_document_error $web_conf(currentSock) $tmp
		return $tmp
	}

	set text $array(_tclReturn)
	catch {unset array(_tclReturn)}
	return $text
}

proc handle_server_side_return {ssiCode} {
	global web_conf errorInfo

	set safeInterp interp_web_$web_conf(currentSock)

	set tmp ""
	if {[catch {$safeInterp eval $ssiCode} result] == 1} {
		append tmp "\{NEOSCRIPT ERROR: $result\}"
		if {$web_conf(debugging)} {
			append tmp "\n<pre>$errorInfo</pre>"	
		}
		gui_handle_document_error $web_conf(currentSock) $tmp
		return $tmp
	} else {
		if {$result != ""} {
			return $result
		}
		return
	}

}

proc handle_server_side_variable {ssiCode} {
	global web_conf errorInfo

	set safeInterp interp_web_$web_conf(currentSock)

	set tmp ""
	if {[catch {$safeInterp eval set $ssiCode} result] == 1} {
		append tmp "\{NEOSCRIPT ERROR: $result\}"
		if {$web_conf(debugging)} {
			append tmp "\n<pre>$errorInfo</pre>"	
		}
		gui_handle_document_error $web_conf(currentSock) $tmp
		return $tmp
	} else {
		if {$result != ""} {
			return $result
		}
		return
	}

}

proc handle_server_side_expr {ssiCode} {
	global web_conf errorInfo

	set safeInterp interp_web_$web_conf(currentSock)

	set tmp ""
	if {[catch {$safeInterp eval expr $ssiCode} result] == 1} {
		append tmp "\{NEOSCRIPT ERROR: $result\}"
		if {$web_conf(debugging)} {
			append tmp "\n<pre>$errorInfo</pre>"	
		}
		gui_handle_document_error $web_conf(currentSock) $result
		return $tmp
	} else {
		if {$result != ""} {
			return $result
		}
		return
	}

}

##########
# End procedures to handle Neoscript directives
##########



##########
# Begin procedures for db support
##########
proc check_db_name {dbName} {
	if ![regexp {^[a-zA-Z0-9_]*$} $dbName] {
		return 0
	}
	return 1
}

proc SAFE_dbstore {safeInterp dbName key arrayName} {
	set arrayList [$safeInterp eval "array get $arrayName"]

	if {[check_db_name $dbName]} {
		set result [dbstore $dbName $key [concat $arrayList]]
	} else {
		html "Error:  Illegal database name: $dbName"
	}
}

proc SAFE_dbfetch {safeInterp dbName key arrayName} {
	if {[check_db_name $dbName]} {
		set arrayList [list [dbfetch $dbName $key]]
		$safeInterp eval "array set $arrayName $arrayList"
	} else {
		html "Error:  Illegal database name: $dbName"
	}
}

proc SAFE_dbkeys {dbName {pattern ""}} {
	if {[check_db_name $dbName]} {
		return [dbkeys $dbName $pattern]
	} else {
		html "Error:  Illegal database name: $dbName"	
	}
}

proc SAFE_dbdelkey {dbName key} {
	if {[check_db_name $dbName]} {
		dbdelkey $dbName $key
	} else {
		html "Error:  Illegal database name: $dbName"	
	}
}

proc SAFE_dbexists {dbName} {
	if {[check_db_name $dbName]} {
		return [dbexists $dbName]
	} else {
		html "Error:  Illegal database name: $dbName"	
	}
}

proc SAFE_dbfiles {} {
	return [dbfiles]
}

proc SAFE_delete_db_file {dbName} {
	if {[check_db_name $dbName]} {
		return [delete_data_file $dbName]
	} else {
		html "Error:  Illegal database name: $dbName"	
	}
}

##########
# end procedures for db support
##########


proc import_keyvalue_pairs {arrayName string} {
    upvar $arrayName array

    set len [llength $string]
    if {$len % 2 != 0} {
        error "unmatched key-value pair"
    }

    for {set i 0} {$i < $len} {incr i 2} {
        set key [lindex $string $i]
        if {[string index $key 0] != "-"} {
            error "key $key of key-value pairs doesn't start with a dash"
        }
        set array([string range $key 1 end]) [lindex $string [expr $i + 1]]
    }
}

#
# Create a cookie (send a browser a little message that you'll get
# back when they retrieve pages.)
#
# neo_make_cookie cookieName cookieValue [-days expireInDays]
#    [-hours expireInHours] [-minutes expireInMinutes]
#    [-path uriPathCookieAppliesTo]
#    [-secure 1|0]
#
proc SAFE_neo_make_cookie {sock name value args} {
	upvar #0 out_$sock out_headers
    import_keyvalue_pairs params $args

	set cookie "$name = $value"

    set expiresIn 0
    if [info exists params(days)] {
	incr expiresIn [expr $params(days) * 86400]
    }

    if [info exists params(hours)] {
	incr expiresIn [expr $params(hours) * 3600]
    }

    if [info exists params(minutes)] {
	incr expiresIn [expr $params(minutes) * 60]
    }

    if {$expiresIn != 0} {
	append cookie "; expires=[http_date [expr [clock seconds] + $expiresIn]]"
    }

   if [info exists params(path)] {
       append cookie "; path=$params(path)"
   }

   if [info exists params(domain)] {
       append cookie "; domain=$params(domain)"
   }

   if {[info exists params(secure)] && $params(secure) == 1} {
       append cookie "; secure"
   }
   set out_headers(Set-Cookie) $cookie
}

#
# load_virtual
proc SAFE_load_virtual {sock upath} {
	global web_conf

	set safePath [eval file join [lrange [file split $upath] 1 end]]
	if {![file exists [file join $web_conf(DocumentRoot) $safePath]]} {
		error "Error:  Path not found:  $upath"
	}
	if {[catch {open [file join $web_conf(DocumentRoot) $safePath] r} result] == 1} {
		error "Error:  Opening file:  $upath"	
	} else {
		set text [read $result]
		close $result
		interp_web_$sock eval $text
	}
}

#
# load_file
proc SAFE_load_file {sock upath} {
	global web_conf
	upvar #0 web_$sock array

	set safeRoot [file dirname $array(filename)]
	set fileList [file split $upath]

	if {[lindex $fileList 0] == "/"} {
		set fileList [lrange $fileList 1 end]
	}

	set safePath [eval file join $safeRoot $fileList]

	if {![file exists $safePath]} {
		error "Error:  Path not found:  $upath"
	}
	if {[catch {open $safePath r} result] == 1} {
		error "Error:  Opening file:  $upath"	
	} else {
		set text [read $result]
		close $result
		interp_web_$sock eval $text
	}
}

#
# Load cookies
proc SAFE_load_cookies {sock {cookieVar "cookies"}} {
	upvar #0 headers_$sock headers_array

	if ![info exists headers_array(Cookie)] return

	foreach pair [split $headers_array(Cookie) ";"] {
		set pair [split [string trim $pair] "="]
		set key [lindex $pair 0]
		set value [lindex $pair 1]
		interp_web_$sock eval set ${cookieVar}($key) [list $value]
	}
}

#
# remote_hostname
proc SAFE_remote_hostname {sock {ip ""}} {
	if {$ip == ""} {
		set _name $ip
	} else {
		set _name [lindex [fconfigure $sock -peername] 1]
	}
	return $_name
}

# random:  acts like the tclx random command
proc SAFE_random {sock limit {seedval ""}} {
	upvar #0 web_$sock array

	if {$limit == "seed"} {
		if {$seedval != ""} {
			expr srand($seedval)
		} else {
			expr srand([clock seconds])
		}
	} else {
		set tmprand [expr rand()]
		set bigmod [expr round([expr $tmprand * 100000000])]
		return [expr $bigmod % $limit]
	}
}

# access_data_file
proc SAFE_access_data_file {sock dataFileName} {
	global errorCode errorInfo
	set userName [get_username]

	if {![check_db_name $dataFileName]} {
		error "Illegal database name:  $dataFileName"
	}

	set file [file join [user2path $userName] data $dataFileName.data]
	if {[catch {set fp [open $file "CREAT RDWR" 0660]}] == 1} {
		if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
			error "$errorCode"
		}
		create_user_dir $userName data
		set fp [open $file "CREAT RDWR" 0660]
	}

	set safeInterp "interp_web_$sock"
	interp transfer "" $fp $safeInterp
	return $fp
}

# delete_data_file
proc SAFE_delete_data_file {sock dataFileName} {
	global errorCode errorInfo
	set userName [get_username]

	if {![check_db_name $dataFileName]} {
		error "Illegal database name:  $dataFileName"
	}

	set file [file join [user2path $userName] data $dataFileName.data]
	file delete $file
}

# log_message
proc SAFE_log_message {sock logFileName string} {
	upvar #0 web_$sock array
	global errorCode errorInfo
	set userName [get_username]

	if {![check_db_name $logFileName]} {
		error "Illegal database name:  $logFileName"
	}

	set file [file join [user2path $userName] log $logFileName.log]
	if {[catch {set fp [open $file "CREAT WRONLY APPEND" 0660]}] == 1} {
		if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
			error "$logFileName: $errorCode"
		}
		create_user_dir $userName log
		set fp [open $file "CREAT WRONLY APPEND" 0660]
	}

	puts $fp [list [clock seconds] $array(hostName) $string]
	close $fp
	return
}

# estimate_hits_per_hour
proc SAFE_estimate_hits_per_hour {} {
	global web_conf

	set secs_up [expr [clock seconds] - $web_conf(softwareStart)]
	set hits_per_sec [expr $web_conf(numhits).0 / $secs_up.0]
	return [expr round($hits_per_sec * 3600)]
}

# set_header
proc SAFE_set_header {sock key value} {
	upvar #0 out_$sock headers_out

	if {[llength $value] != 1} {
		set value [list $value]
	}
	if {[llength $key] != 1} {
		set key [list $key]
	}

	set headers_out($key) $value
}
