#
##		web.tcl:  NeoWebScript-sa core
###		Written by George Porter <gporter@neosoft.com>
##		Copyright 1997, NeoSoft, Inc.  All rights reserved.
#		$Id: web.tcl,v 1.36 1997/06/30 16:41:09 gporter Exp $

#
##
###
###
## Misc functions used throughout the module
#

proc debug {string} {
	global web_conf

	if {$web_conf(debugging)} {
		puts stdout $string
		flush stdout
	}
}

proc bgerror {string} {
	global errorCode errorInfo

	gui_handle_bgerror $string $errorCode $errorInfo
}

proc validate_uri {uri} {
    set uri_list [split $uri "/"]
    set new_uri_list {}
    foreach segment $uri_list {
        if {$segment == "." || $segment == {}} {
            continue
        } elseif {$segment == ".."} {
            if {[llength $new_uri_list] > 0} {
                set endshort [expr [llength $new_uri_list] - 2]
                set new_uri_list [lrange $new_uri_list 0 $endshort]
            } else {
		break
            }
        } else {
            lappend new_uri_list $segment
        }
    }
    if {[llength $new_uri_list] == 0} {
        set new_uri "."
    } else {
        set new_uri [join $new_uri_list "/"]
    }
    return $new_uri
}

proc register_webhandler {procname} {
	global _web_handlers

	if {[llength $_web_handlers] == 0} {
		lappend _web_handlers ""
	}
	lappend _web_handlers $procname
}

proc get_username {} {
	return "default"
	# return "gporter"
}

# Given a username, returns the name of their parallel directory
proc user2path {name} {
	global web_conf

	set firstChar [string index $name 0]
	set path [file join $web_conf(ServerRoot) neoscript-data users $firstChar $name]
	return $path
}

proc create_user_dir {userName type} {
	global web_conf errorCode errorInfo

	set path [user2path $userName]

	file mkdir [file join $path $type]
}

proc http_date {args} {
	if {[llength $args] > 0} {
		set time [lindex $args 0]
	} else {
		set time [clock seconds]
	}

	return [clock format $time -format "%a, %d %b %Y %T GMT" -gmt 1]
}

proc bytes2k {bytes} {
	set k [expr $bytes / 1024]
	if {$k < 1} {
		set k 1
	}
	return $k
}

proc log {message} {
	global web_conf

	puts $web_conf(logFP) [list [http_date] $message]
	flush $web_conf(logFP)
}

proc log_request {sock} {
	global web_conf
	upvar #0 web_$sock array
	upvar #0 headers_$sock headers_array

	if {![info exists headers_array(User-Agent)]} {
		set headers_array(User-Agent) "-"	
	}
	if {![info exists array(uri)]} {
		set array(uri) "TIMEOUT"
	}
	puts $web_conf(logFP) [list [http_date] $array(status) $array(addr) $array(uri) $headers_array(User-Agent)]
	flush $web_conf(logFP)
}

proc error_document {sock code {message ""}} {
	global web_conf errorCode errorInfo
	upvar #0 web_$sock array
	upvar #0 out_$sock out_headers

	set array(status) $code
	array set codes {
		200	"OK"
		301 "Moved Permanently"
		302 "Moved Temporarily"
		304 "Not Modified"
		400 "Bad Request"
		401 "Unauthorized"
		403 "Forbidden"
		404 "Not Found"
		500 "Internal Server Error"
		501 "Not Implemented"
		503 "Service Unavailable"
	}

	puts $sock "HTTP/1.0 $code $codes($code)"
	puts $sock "Date: [http_date]"
	puts $sock "Server: $web_conf(Server)"
	puts $sock "Content-type: text/html"

	foreach line [array names out_headers] {
		puts $sock "$line: $out_headers($line)"
	}

	puts $sock ""
	puts $sock "<head><title>$codes($code)</title></head>"
	puts $sock "<body><h1>$code $codes($code)</h1>"
	puts $sock "while processing $array(uri).<br>"
	puts $sock "$message"
	puts $sock "</body>"

	finish_session $sock
	error $errorCode $errorInfo

}

# A procedure to store info about a filename.  We can know if there
# is neowebscript before we even open it.
# return 0 if regular html, 1 if neowebscript (or if we don't know)
proc check_lookaside_cache {fn} {
	return 1	
}

proc set_lookaside_cache {fn mode} {
	return 0
}


#
##
###
###
## Begin the actual procedures that handle the request
#

proc process_headers {sock} {
	global web_conf
	upvar #0 web_$sock array
	upvar #0 headers_$sock headers_array

	set array(method) [lindex $array(line) 0]
	set array(uri) [lindex $array(line) 1]
	set array(protocol) [lindex $array(line) 2]

	set new_uri [validate_uri $array(uri)]
	if {$new_uri  == ""} {
		error_document $sock 400 "Bad Request"
	}
	set array(uri) $new_uri
}

proc map_uri {sock} {
	global web_conf tcl_platform
	upvar #0 web_$sock array
	upvar #0 out_$sock out_headers

	set uri $array(uri)

	# Eventually we can put in things like redirection here, possibly
	# cgi-scripts, etc.  For now, we will return the filename, after
	# stripping off any QUERY info

	regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $uri {[format %c 0x\1]} uri
	set uri [subst $uri]

	set qIndex [string first "?" $uri]
	if {$qIndex != -1} {
		set array(query_object) [string range $uri [expr $qIndex + 1] end]
		set uri [string range $uri 0 [expr $qIndex - 1]]

		foreach pair [split $array(query_object) "&"] {
			set key [lindex [split $pair "="] 0]
			set value [lindex [split $pair "="] 1]

			regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $key {[format %c 0x\1]} key
			regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $value {[format %c 0x\1]} value
			set key [subst $key]
			set value [subst $value]

			regsub -all {\+} $value " " value
			regsub -all {([][$\\])} $value {\\\1} value
			set value [subst $value]

			set query($key) $value
		}

		set array(query_data) [array get query]
	}

	set machine_path [eval file join $web_conf(DocumentRoot) $uri]

	###

	if {[file isdirectory $machine_path]} {
		if {!$web_conf(ShowDirectoryIndex)} {
			error_document $sock 404
		}
		if {[file exists [file join $machine_path $web_conf(DirectoryIndex)]]} {
			set array(filename) [eval file join $machine_path $web_conf(DirectoryIndex)]
			set array(type) file
		} else {
			set array(filename) $machine_path
			set array(type) directory
		}
	} else {
		set array(filename) $machine_path
		set array(type) file
	}
}


proc access_check {sock} {
	upvar #0 web_$sock array

	# We can add code here to look at array(filename) to see if it applies
	# to certain patterns, and get more authentication that way.  In the
	# mean time, we'll just check to make sure we can read the file.

	if {[file exists $array(filename)] != 1} {
		error_document $sock 404 "The file \"$array(uri)\" could not be found."
	}

	if {[file readable $array(filename)] != 1} {
		error_document $sock 403 "The file you requested in not readable by you."
	}
	set array(status) 200
}

# given a filename (not handle), we return back the MIME type from mime.tcl.
proc determine_mime_type {fn} {
	global mime_type

	if {[lsearch [array names mime_type] [file extension $fn]] == -1} {
		return "text/plain"
	} else {
		return $mime_type([file extension $fn])
	}
}

proc send_response {sock} {
	upvar #0 web_$sock array
	upvar #0 headers_$sock headers_array

	if {[regexp -nocase (GET) $array(method)] == 1} {
		emit_file $sock
		return
	} elseif {[regexp -nocase (HEAD) $array(method)] == 1} {
		emit_file $sock 0
		return
	} elseif {[regexp -nocase (POST) $array(method)] == 1} {
		foreach pair [split $headers_array(body) "&"] {
			set key [lindex [split $pair "="] 0]
			set value [lindex [split $pair "="] 1]

			regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $key {[format %c 0x\1]} key
			regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $value {[format %c 0x\1]} value
			set key [subst $key]
			set value [subst $value]

			regsub -all {\+} $value " " value
			regsub -all {([][$\\])} $value {\\\1} value
			set value [subst $value]

			set post($key) $value
		}

		set array(post_data) [array get post]
		emit_file $sock
		return

	} else {
		error_document $sock 501 "Sorry, unsupported method $array(method)."
	}
}

#
##
###
##  Procedures to actually handle text parsing/generation
#
proc handle_subst_data {sock text} {
	upvar #0 web_$sock array
	set tmp ""

	while {1} {
		if {[regexp -indices "<nwsubst>" $text spos] == 1} {
			if {[regexp -indices "</nwsubst>" $text epos] != 1} {
				append tmp "<h1>OOps!  The closing NWS tag is missing.</h1>"
				return $tmp
			} elseif {[lindex $spos 0] > [lindex $epos 0]} {
				append tmp "<h1>OOps!  You must put <nwsubst> tags before </nwsubst> ones."
				return $tmp
			}
		
		set m [expr [lindex $spos 0] - 1]
		set n [expr [lindex $spos 1] + 1]
		set x [expr [lindex $epos 0] - 1]
		set y [expr [lindex $epos 1] + 1]

		append tmp [string range $text 0 $m]
		set tclbody [string range $text $n $x]

		# here we subst the body
		interp_web_$sock eval "set tclbody [list $tclbody]"
		if {[interp_web_$sock eval {catch {subst $tclbody} result}] == 1} {
			set errorstr [interp_web_$sock eval {set result}]
			append tmp "\n<h2>OOps!</h2>\n"
			append tmp "<b>An error occured in your document:</b>  $errorstr"
			gui_handle_document_error $sock $errorstr
		} else {
			append tmp [interp_web_$sock eval {set result}]
		}
		interp_web_$sock eval {catch {unset result}}

		###

		set text [string range $text $y end]
				
		} else {
			set_lookaside_cache $array(filename) 0
			return [append tmp $text]
		}
	}
}

proc handle_directory_index {sock} {
	upvar #0 web_$sock array
	global web_conf
	global mime

	puts $sock "HTTP/1.0 200 OK"
	puts $sock "Date: [http_date]"
	puts $sock "Server: $web_conf(Server)"
	puts $sock "Content-type: text/html"
	puts $sock ""

	puts $sock "<HEAD><TITLE>Index of $array(uri)</TITLE></HEAD><BODY>"
	puts $sock "<H1>Index of $array(uri)</H1>"
	puts $sock "<PRE>"
	puts $sock "Name                      Last modified                   Size      Description"
	puts $sock "<HR>"

	set parentDir [file dirname $array(uri)]
	file stat [file dirname $array(filename)] pInfo
	puts $sock "   <a href=\"$parentDir\">../</a>                    [clock format $pInfo(mtime)]    [bytes2k $pInfo(size)]Kb       Directory<p>"

	# go through all the files and add them.
	foreach line [lsort [glob -nocomplain -- $array(filename)/*]] {
		file stat $line ftmp

		if {[string length [file tail $line]] > 24} {
			set tmpfname "[string range [file tail $line] 0 23]&gt"
		} else {
			set rem [expr 24 - [string length [file tail $line]]]
			set newuri [file join $array(uri) [file tail $line]]
			set tmpfname "<A HREF=\"$newuri\">[file tail $line]</A>"
			for {set i 0} {$i < $rem} {incr i} {
				append tmpfname " "
			}
		}

		if {$ftmp(type) != "directory"} {
			set mtype [determine_mime_type $line]
		} else {
			set mtype "Directory"
		}

		set textstr [format "%s  %-30s  %-8s  %-25s" $tmpfname [clock format $ftmp(mtime)] [bytes2k $ftmp(size)]Kb $mtype]
		puts $sock "$textstr<p>"
	}
	puts $sock "</PRE></BODY>"	
}

proc extend_interps {sock} {
	global web_conf
	upvar #0 web_$sock array
	upvar #0 headers_$sock headers_array

	set name $array(filename)

	# setup the environment for the sub-slave interp
	set webenv(DOCUMENT_NAME) $name
	set webenv(DOCUMENT_ROOT) $web_conf(DocumentRoot)
	set webenv(DOCUMENT_URI) $array(uri)
	if {[info exists headers_array(Host)]} {
		set webenv(HTTP_HOST) $headers_array(Host)
	}
	if [info exists headers_array(Host)] {
		set webenv(HTTP_HOST) $headers_array(Host)
	}
	set webenv(NEOSCRIPT_VERSION) $web_conf(Server)
	set webenv(NEO_LAST_MODIFIED) [file mtime $name]
	set webenv(NEO_SOFTWARE_START) [clock format $web_conf(softwareStart)]
	set webenv(NEO_TIME_FORMAT) "%A, %d-%b-%y %T %Z"
	set webenv(REMOTE_ADDR) $array(addr)
	set webenv(REMOTE_HOST) $array(hostName)
	set webenv(REQUEST_METHOD) $array(method)
	set webenv(SERVER_ADMIN) $web_conf(ServerAdmin)
	set webenv(SERVER_NAME) [info hostname]
	set webenv(SERVER_PORT) $web_conf(Port)
	set webenv(SERVER_PROTOCOL) $array(protocol)
	set webenv(SERVER_ROOT) $web_conf(ServerRoot)
	set webenv(SERVER_SOFTWARE) $web_conf(Server)

	if {[info exists headers_array(User-Agent)]} {
		set webenv(HTTP_USER_AGENT) $headers_array(User-Agent)
	}
	if {[info exists headers_array(Connection)]} {
		set webenv(HTTP_CONNECTION) $headers_array(Connection)
	}
	if {[info exists headers_array(Accept)]} {
		set webenv(HTTP_ACCEPT) $headers_array(Accept)
	} else {
		set webenv(HTTP_ACCEPT) "*/*"
	}
	if {[info exists headers_array(Referer)]} {
		set webenv(HTTP_REFERER) $headers_array(Referer)
	}
	if {[info exists array(query_object)]} {
		set webenv(QUERY_STRING) $array(query_object)
	}
	if {[info exists array(post_data)]} {
		interp_web_$sock eval "array set response [list $array(post_data)]"
		set webenv(NEO_POST_DATA) $headers_array(body)
	}
	if {[info exists array(query_data)]} {
		interp_web_$sock eval "array set response [list $array(query_data)]"
	}

	interp_web_$sock eval "array set webenv [list [array get webenv]]"

	# Load conf/client.tcl into the safe interp.
	set clientName [file join $web_conf(ServerRoot) conf client.tcl]
	if {[file exists $clientName]} {
		set clientFP [open $clientName r]
		set clientStr [read $clientFP]
		close $clientFP

		interp_web_$sock eval "$clientStr"
	}

	set initName [file join $web_conf(ServerRoot) conf init.tcl]
	if {[file exists $initName]} {
		uplevel #0 source $initName
		setup_safe_interpreter $sock
	}

	set develName [file join $web_conf(ServerRoot) conf devel.tcl]
	if {[file exists $develName]} {
		if {$web_conf(debugging)} {uplevel #0 source $develName}
		devel_setup $sock
	}

}

proc handle_neoscript_segment {segment} {
	regsub -- {(^.*neoscript *)} $segment {} newseg	
	regsub -- {=} $newseg "\001" newseg

	set _ssi_code [string trim [lindex [split $newseg "\001"] 1]]
	set ssi_startchar [string index $_ssi_code 0]

	set ssi_directive [string trim [lindex [split $newseg "\001"] 0]]
	set ssi_code [string trim $_ssi_code $ssi_startchar]

	switch $ssi_directive {
		return {
			return [handle_server_side_return $ssi_code]
		}
		code {
			return [handle_server_side_eval $ssi_code]
		}
		eval {
			return [handle_server_side_eval $ssi_code]
		}
		var {
			return [handle_server_side_variable $ssi_code]
		}
		expr {
			return [handle_server_side_expr $ssi_code]
		}
		default {
			return "<h2>NEOSCRIPT Error:  unknown directive $ssi_directive</h2>"
		}
	}
}

proc emit_file {sock {k 1}} {
	global web_conf errorInfo errorCode
	upvar #0 web_$sock array
	upvar #0 headers_$sock headers_array
	upvar #0 out_$sock out_headers

	set name $array(filename)

	gui_sendout_request $sock $array(hostName) $array(uri)

	if {$array(type) == "directory"} {
		handle_directory_index $sock
		finish_session $sock
		error $errorCode $errorInfo
	}

	if {$array(mime_type) == "text/html"} {
		if {[check_lookaside_cache $name] == 1} {

			set fp [open $name]

			# setup the interps here
			interp create interp_web_$sock -safe

			extend_interps $sock

			## Actually handle the text

			set text [read $fp]

			regsub -all {<nws>} $text {<!--#neoscript code='} text
			regsub -all {</nws>} $text {'-->} text

			if {[regexp <nwsubst> $text] == 1} {
				set text [handle_subst_data $sock $text]
			}

			if {[regexp {<!--#} $text] == 1} {
				regsub -all {<!--#([^ ]*)} $text "\001\002 \\1" newcode
				regsub -all -- {-->} $newcode "\001" newcode
				set pairs [split $newcode "\001"]

				set tmp "\n"

				foreach segment $pairs {
					# set segment [string trim $segment]
					if {[string index $segment 0] != "\002"} {
						if {[string range $segment 0 1] == "<!"} {
							append tmp $segment
							append tmp " -->"
						} else {
							append tmp $segment
						}
					} else {
						append tmp [handle_neoscript_segment $segment]
					}
				}
				set text $tmp
			}
		}


		puts $sock "HTTP/1.0 200 OK"
		puts $sock "Date: [http_date]"
		puts $sock "Server: $web_conf(Server)"
		puts $sock "Content-type: $array(mime_type)"
		puts $sock "Content-length: [string length $text]"
		puts $sock "Last-modified: [http_date [file mtime $name]]"

		foreach var [array names out_headers] {
			puts $sock "$var: $out_headers($var)"
		}

		puts $sock ""

		set array(sent) 1

		if {$k} {
			puts $sock $text
		}

		#delete the interp
		interp delete interp_web_$sock
		if [info exists webenv] {
			unset webenv
		}

		close $fp
		return
	}

	# The actions below are for non-html documents.

	puts $sock "HTTP/1.0 200 OK"
	puts $sock "Date: [http_date]"
	puts $sock "Server: $web_conf(Server)"
	puts $sock "Content-type: $array(mime_type)"
	puts $sock "Content-length: [file size $name]"
	puts $sock "Last-modified: [http_date [file mtime $name]]"

	foreach var [array names out_headers] {
		puts $sock "$var: $out_headers($var)"
	}

	puts $sock ""

	set array(sent) 1

	if {$k} {
		fconfigure $sock -blocking 0 -translation binary -buffering none

		set inFP [open $name]
		fconfigure $inFP -blocking 0 -translation binary -buffering none

		# fcopy is commented out because it will only send out
		# 20240 bytes.  Doing a read, puts will output the entire file,
		# though.  Besides, puts can handle binary data.
		# fcopy $inFP $sock
		set oh_no [read $inFP]
		puts -nonewline $sock $oh_no

		close $inFP
	}
	return
}

proc timeout_request {sock} {
	global errorCode errorInfo

	finish_session $sock
	error $errorCode $errorInfo
}

proc finish_session {sock} {
	global errorInfo errorCode
	upvar #0 web_$sock array
	upvar #0 headers_$sock headers_array

	if {[info exists array]} {
		set status $array(status)

		flush $sock	
		close $sock

		after cancel $array(timeout)

		log_request $sock

		gui_finish_session $sock

		global web_$sock headers_$sock out_$sock

		if [info exists web_$sock] {
			unset web_$sock
		}
		if [info exists headers_$sock] {
			unset headers_$sock
		}
		if [info exists out_$sock] {
			unset out_$sock
		}

		if {[interp exists interp_web_$sock]} {
			interp delete interp_web_$sock
		}

		# error $status $errorInfo
		# error $errorCode $errorInfo
	}
}

proc process_data {sock} {
	global errorCode errorInfo _web_handlers
	upvar #0 web_$sock array

	# Used to do any header processing necessary
	process_headers $sock

	# Map the URI to a path-to-filename.  Store the uri in array(uri), and
	# get back the filename in array(filename).
	map_uri $sock

	# Check for authentication.  This involves:
	# 1) is the object they are requesting secured?
	# 2) do they have the neccessary credentials?
	# 3) if not, issue a 403 error.
	access_check $sock

	# Determine MIME type of the object they request
	set array(mime_type) [determine_mime_type $array(filename)]

	# A method for Webmasters to write handler functions that do stuff
	# before a request is given.  This could be other types of auth,
	# Disk caching routines, gathering statistics, etc.
	# Syntax for the procedures is:
	#   proc web_handler_ProcName {sock} { body }

	if {[info exists _web_handlers]} {
		foreach line $_web_handlers {
			eval $line $sock
		}
	}

	# Send the request out, possibly putting it through emit_file
	# to handle embedded tcl code.
	send_response $sock

	# Flush, close, and log the session
	finish_session $sock
	error $errorCode $errorInfo
}


#
##
###
###	Procedures that handle setting up the socket for use, grabbing the
##	request, and getting the application ready to use.
#

proc read_in_configuration {} {
	global web_conf tcl_platform

	if {[catch {source [file join conf httpd.conf]}] == 1} {
		puts "Error:  Unable to open config file httpd.conf!"
		puts "You must run web.tcl in the directory before conf."
		exit
	}

	set web_conf(softwareStart) [clock seconds]
	set web_conf(numhits) 0

	if {$tcl_platform(platform) == "windows"} {
		if {[string index $web_conf(DocumentRoot) 1] != ":"} {
			message .lab -text "You must specify the drive in your DocumentRoot." ; pack .lab
			button .exit -text "Exit" -command exit -background red ; pack .exit		
		}
		if {[string index $web_conf(ServerRoot) 1] != ":"} {
			message .lab -text "You must specify the drive in your ServerRoot." ; pack .lab
			button .exit -text "Exit" -command exit -background red ; pack .exit		
		}

		if {[catch {set web_conf(DocumentRoot) [file attributes $web_conf(DocumentRoot) -shortname]} result] == 1} {
			puts stdout "Error:  Your DocumentRoot is setup incorrectly.  You must specify the drive in the path."
			exit
		}
		if {[catch {set web_conf(ServerRoot) [file attributes $web_conf(ServerRoot) -shortname]} result] == 1} {
			puts stdout "Error:  Your ServerRoot is setup incorrectly.  You must specify the drive in the path."
			exit
		}

	}
}

proc get_data {sock} {
	global web_conf errorCode errorInfo
	upvar #0 web_$sock array
	upvar #0 headers_$sock headers_array

	if {[eof $sock]} {
		close $sock
		global web_$sock headers_$sock
		if [info exists web_$sock] {
			unset web_$sock
		}
		if [info exists headers_$sock] {
			unset headers_$sock
		}
		return
	}

	if {$array(init) == 0} {
		gets $sock array(line)
		set array(init) 1
	}

	gets $sock line

	set headers_array([lindex [split $line ":"] 0]) [string trim [lindex [split $line ":"] 1]]

	if {$line == ""} {
		if [info exists headers_array(Content-length)] {
			set headers_array(body) [read $sock $headers_array(Content-length)]
		}

		fileevent $sock readable ""

		if {[catch {process_data $sock} result] != 0} {
			if {![regexp (NONE) $result]} {
				log "OOps!  An error occured:  $result."
				gui_toplevel_error $sock $result
				error_document $sock 500 "There was an internal server error.  Please contact the server administrator."
				finish_session $sock
			}
		}
	}
}

proc accept_http_connection {sock addr port} {
	global web_conf
	global web_$sock
	if [info exists web_$sock] {
		unset web_$sock
	}
	upvar #0 web_$sock array

	set array(addr) $addr
	set array(hostName) [lindex [fconfigure $sock -peername] 1]
	set array(port) $port
	set array(sock) $sock
	set array(init) 0
	set array(sent) 0
	set array(status) 000
	set array(timeout) [after 900000 timeout_request $sock]
	set web_conf(currentSock) $sock

	fconfigure $sock -blocking 0 -translation {auto lf} -buffering line
	fileevent $sock readable "get_data $sock"
}

proc setup_http_server {port} {
    set socketID [socket -server accept_http_connection $port]
}

proc doit {{argv ""}} {
	global web_conf mime_type tcl_platform

	if {$tcl_platform(platform) == "unix"} {
		catch {signal trap INT exit}
	}

	read_in_configuration

	set initName [file join $web_conf(ServerRoot) conf init.tcl]
	if {[file exists $initName]} {
		uplevel #0 source $initName
	}

	set mimeName [file join $web_conf(ServerRoot) mime.tcl]
	uplevel #0 source $mimeName

	set guiName [file join $web_conf(ServerRoot) gui.tcl]
	uplevel #0 source $guiName

	set dbName [file join $web_conf(ServerRoot) conf db.tcl]
	if {[file exists $dbName]} {
		uplevel #0 source $dbName
	}

	set web_conf(logFP) [open $web_conf(LogFile) a]

	set pidFP [open $web_conf(PidFile) w]
	puts $pidFP [pid]
	close $pidFP

	log "Starting $web_conf(Server), running as process [pid]."

	setup_http_server $web_conf(Port)

	gui_init_interface

	vwait endProgram
}

doit
