#
# NeoWebScript - Server Side Programming based on Safe Tcl
#
# Copyright (C) NeoSoft, All Rights Reserved.  See NeoWebScript LICENSE
# files with this distribution for conditions on use and redistribution.
#
# $Id: common.tcl,v 1.1.1.1 1999/03/31 20:53:26 damon Exp $
#

#
# PROPERTY LIST STORAGE AND RETRIEVAL
#

proc setup_data_access {} {
    global parallelDir parallelUserBase webpageOwnerName webenv

    if [info exists parallelDir] return

    if [info exists NeoWebDirConf(WebPageOwnerName)] {
	set webpageOwnerName $NeoWebDirConf(WebPageOwnerName)
    } else {
	set webpageOwnerName [id convert userid $webenv(NEO_DOCUMENT_UID)]
    }
    set parallelDir $parallelUserBase/[cindex $webpageOwnerName 0]/$webpageOwnerName/
}

proc create_user_dir {plusSubDir} {
    global parallelDir errorCode

    setup_data_access

    if {[catch {mkdir -path $parallelDir$plusSubDir} result] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX EEXIST"} {
	    error "$errorCode"
	}
    }
}

proc ul_name_check {dbname dbtype fileNameVar} {
    if ![regexp {^[-a-zA-Z0-9_=+.]*$} $dbname] {
	error "illegal database name: $dbname, lowercase, uppercase, 0-9, _, -, +, =, and .  only"
    }
    upvar $fileNameVar dbFileName
    global parallelDir

    set dbFileName ${parallelDir}$dbtype/$dbname.$dbtype
}

proc db_name_check {dbname dbtype fileNameVar {projectName {}}} {
    global errorCode
    if ![regexp {^[a-zA-Z0-9_]*$} $dbname] {
	return -code error "illegal database name: $dbname, upper/lowercase and 0-9 only"
    }
    upvar $fileNameVar dbFileName
    global parallelDir

    set dbFileName ${parallelDir}$dbtype
    if {$projectName != {}} {
	if ![regexp {^[a-zA-Z0-9_]*$} $projectName] {
	    return -code error "illegal project name: $projectName, upper/lowercase and 0-9 only"
	}
	append dbFileName "/$projectName"
	if {[catch {mkdir -path $dbFileName} result] == 1} {
	    if {[lrange $errorCode 0 1] != "POSIX EEXIST"} {
		return -code error "$errorCode"
	    }
	}
    }
    append dbFileName "/$dbname.$dbtype"
}

proc proj_name_check {type projName fileNameVar} {
    global parallelDir errorCode
    if ![regexp {^[a-zA-Z0-9_]*$} $projName] {
	return -code error \
	    "illegal project name: $projName, upper/lowercase and 0-9 only"
    }
    upvar $fileNameVar projFileName

    set projFileName ${parallelDir}$type/$projName
}

proc split_project {string projVar fileVar} {
    upvar 1 $projVar project
    upvar 1 $fileVar file

    lassign [split $string :] project file
    if [lempty $file] {set file $project; set project {}}
}

SAFE_proc_and_alias filerm {safeInterp type database args} {
    setup_data_access

    set data(project) {}
    split_project $database data(project) database
    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    db_name_check $database $type dbFileName $data(project)
    file delete -force -- $dbFileName

    if ![lempty $data(project)] {
	set dir [file dirname $dbFileName]
	if [lempty [glob -nocomplain [file join $dir *]]] {
	    file delete -force -- $dir
	}
    }
}

SAFE_proc_and_alias filemv {safeInterp type old_database database args} {
    setup_data_access

    set data(project1) {}
    set data(project2) {}
    split_project $old_database data(project1) old_database
    split_project $database data(project2) database

    set data(project) {}
    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![lempty $data(project)] {
	set data(project1) $data(project)
	set data(project2) $data(project)
    }
    db_name_check $old_database $type old_dbFileName $data(project1)
    db_name_check $database $type dbFileName $data(project2)

    if ![file exists $old_dbFileName] {
	return -code error "attempt to move non-existent $type file $old_database"
    }
    file copy -force -- $old_dbFileName $dbFileName
    file delete -force -- $old_dbFileName

    if ![lempty $data(project)] {
	set dir [file dirname $old_dbFileName]
	if [lempty [glob -nocomplain [file join $dir *]]] {
	    file delete -force -- $dir
	}
    }
}

SAFE_proc_and_alias filecp {safeInterp type old_database database args} {
    setup_data_access

    set data(project1) {}
    set data(project2) {}
    split_project $old_database data(project1) old_database
    split_project $database data(project2) database

    set data(project) {}
    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![lempty $data(project)] {
	set data(project1) $data(project)
	set data(project2) $data(project)
    }
    db_name_check $old_database $type old_dbFileName $data(project1)
    db_name_check $database $type dbFileName $data(project2)

    if ![file exists $old_dbFileName] {
	error "attempt to copy non-existent $type file $old_database"
    }
    file copy -force -- $old_dbFileName $dbFileName
}

SAFE_proc_and_alias list_files {safeInterp type {pattern *}} {
    setup_data_access
    global parallelDir

    split_project $pattern project file

    set db "${parallelDir}$type"
    if ![lempty $project] {
	append db "/$project"
    }
    append db "/$file.$type"

    set result {}
    foreach file [glob -nocomplain $db] {
	lappend result [file root [file tail $file]]
    }
    return $result
}

SAFE_proc_and_alias file_exists {safeInterp type file} {
    setup_data_access
    global parallelDir

    split_project $file project file
    db_name_check $file $type dbFileName $project
    return [file exists $db]
}

SAFE_proc_and_alias project_exists {safeInterp type project} {
    setup_data_access
    global parallelDir

    proj_name_check $type $project projName
    if {[file exists $projName] && [file isdirectory $projName]} { return 1 }
    return 0
}

SAFE_proc_and_alias list_projects {safeInterp type {pattern *}} {
    setup_data_access
    global parallelDir
    set path $parallelDir$type

    set projects {}
    foreach file [glob -nocomplain $path/$pattern] {
	if ![file isdirectory $file] continue
	lappend projects [lindex [file split $file] end]
    }
    return $projects
}

SAFE_proc_and_alias projectrm {safeInterp type projectName} {
    setup_data_access
    global parallelDir
    proj_name_check $type $projectName projPath
    file delete -force -- $projPath
}

SAFE_proc_and_alias projectcp {safeInterp type old_proj new_proj} {
    setup_data_access
    global parallelDir
    proj_name_check $type $old_proj oldProjPath
    proj_name_check $type $new_proj newProjPath
    file copy -force -- $oldProjPath $newProjPath
}

SAFE_proc_and_alias projectmv {safeInterp type old_proj new_proj} {
    setup_data_access
    global parallelDir
    proj_name_check $type $old_proj oldProjPath
    proj_name_check $type $new_proj newProjPath
    file copy -force -- $oldProjPath $newProjPath
    if [file exists $newProjPath] {
	file delete -force -- $oldProjPath
    }
}

# proc to import key value pairs of the form "-color blue"
# into an array in the caller's context.
#
# and dialog box thingie that uses it
#
# i am really missing incr tcl

safe_and_trusted_proc import_keyvalue_pairs {arrayName string} {
    upvar $arrayName array

    set len [llength $string]
    if {$len % 2 != 0} {
        error "unmatched key-value pair"
    }

    for {set i 0} {$i < $len} {incr i 2} {
        set key [lindex $string $i]
        if {[string index $key 0] != "-"} {
            error "key $key of key-value pairs doesn't start with a dash"
        }
        set array([string range $key 1 end]) [lindex $string [expr $i + 1]]
    }
}

#
# set_array_defaults arrayName -key value -key value -key value
#
proc set_array_defaults {arrayName args} {
    upvar $arrayName array
    import_keyvalue_pairs array $args
}

#
# Access a data file with all the file stuff present in safe interpreters.
#
SAFE_proc_and_alias access_data_file {safeInterp datafile args} {
    global parallelDir errorCode
    setup_data_access

    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $datafile data dataFileName $data(project)

    if {[catch {set fp [open $dataFileName "CREAT RDWR" 0660]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$logFileName: $errorCode"
	}
	create_user_dir data
	set fp [open $dataFileName "CREAT RDWR" 0660]
    }

    interp transfer "" $fp $safeInterp
    return $fp
}

SAFE_proc_and_alias delete_data_file {safeInterp datafile args} {
    global parallelDir errorCode
    setup_data_access

    if {[cindex $args 0] == "-"} {
	import_keyvalue_pairs data $args
    }

    if ![info exists data(project)] {set data(project) {}}
    db_name_check $datafile data dataFileName $data(project)

    if {[catch {unlink $dataFileName} result] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$result"
	}
    }
    return
}

#
# List the data files in the server-maintained data directory
# for the user.
#
#
SAFE_proc_and_alias list_data_files {safeInterp args} {
    global parallelDir errorCode
    setup_data_access

    set data(pattern) *
    if {[cindex $args 0] == "-"} { import_keyvalue_pairs data $args }

    set db "${parallelDir}data"
    if [info exists data(project)] {
	    append db "/$data(project)"
    }
    append db "/$data(pattern).data"

    set result {}
    foreach file [glob -nocomplain $db] {
	lappend result [file root [file tail $file]]
    }
    return $result
}

#
# Return 1 if a datafile exists in the server-maintained data directory.
#
SAFE_proc_and_alias data_file_exists {safeInterp datafile args} {
    global parallelDir errorCode
    setup_data_access

    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args }

    if ![info exists data(project)] {set data(project) {}}
    db_name_check $datafile data dataFileName $data(project)

    return [file exists $dataFileName]
}

#
# check_password tests sent_pw against real_pw and returns a value
# suitable to be returned to the C auth code in apache.  real_pw
# is already encrypted, but sent_pw is not.  the first two bytes of
# real_pw is used as the salt.
#
proc check_password {sent_pw real_pw} {
    return [cequal [neo_crypt $sent_pw $real_pw] $real_pw]
}

# Authorization routine for DB Auth.
proc tcl_db_auth {dbOwner dbName username sent_pw} {
    global parallelUserBase
    upvar _${dbOwner}_${dbName}_auth_cache authCache
    if [info exists authCache($username)] {
        lassign $authCache($username) password expire
        if {$expire > [clock seconds] && [check_password $sent_pw $password]} {
            return
        }
    }
    set password *
    set dbFileName $parallelUserBase/[cindex $dbOwner 0]/$dbOwner/db/$dbName.db
    if {[catch {set db [db open $dbFileName hash r]}] == 1} {
        return "user $username: unable to access $dbFileName"
    }
    set found [db get $db $username password]
    db close $db
    if $found {
        set authCache($username) [list $password [expr [clock seconds] + 60]]
        if [check_password $sent_pw $password] return
    }
    return "user $username: invalid password"
}

proc tcl_db_access {user type} {
        if {$type == "valid-user"} { return OK }
        return AUTH_REQUIRED
}

# Authorization routine for /etc/passwd access.  Requires the 'getpass'
# program to be built.

proc tcl_passwd_auth {username sent_pw} {
    global server binPath
    if [catch {exec $binPath/getpass $username} password] {
	    return "user $username not found"
    }
    if [check_password $sent_pw $password] return
    return "$pwError (/etc/passwd)"
}

proc tcl_passwd_access {user type} {
    if {$type == "valid-user"} { return OK }
    return AUTH_REQUIRED
}

# Routines for the test authentication based on Tcl
proc test_tcl_auth {args} {
    return [neo_crypt test ab]
}

proc test_tcl_access {args} {
    return OK
}

#
# detect switches from a string (e.g. "-all -regexp -- foo bar args") and
# extracts them into an array
#
safe_and_trusted_proc import_switch_args {arrayName argsList {switchList {}}} {
    upvar $arrayName array
    set array(args) {}
    set array(switches) {}
    set argsIndex 0
    if {[llength $switchList] > 0} {
	set proofSwitches 1
    } else {
	set proofSwitches 0
    }
    foreach arg $argsList {
        if {![string match "-\[a-zA-Z0-9\]*" $arg]} {
            set array(args) [lrange $argsList $argsIndex end]
            break
        } elseif {$arg == "--"} {
	    set array(args) [lrange $argsList [expr $argsIndex+1] end]
	    break
	}
        set switch [crange $arg 1 end]
	if {!$proofSwitches || [lsearch -exact $switchList $switch] >= 0} {
            set array($switch) $argsIndex
	    lappend array(switches) $switch
	}
        incr argsIndex
    }
}

#
# determine whether a Tcl string converts to a Tcl number
#
safe_and_trusted_proc is_num {number} {
    if {[catch {expr $number+1} errorMsg] == 0} {
        return 1
    }
    return 0
}

#
# escapes reserved SGML characters so that they show up properly in an
# HTML document
#
safe_and_trusted_proc escape_attribute {args} {
    set quoteChar ""
    set text ""
    import_switch_args mode $args
    foreach arg $mode(switches) {
        switch -exact -- $arg {
            "-singlequotes"	{set quoteChar {'}}
            "-doublequotes"	{set quoteChar \"}
            "-noquotes"		{set quoteChar ""}
        }
    }
    set text [lindex $mode(args) 0]
    return $quoteChar[escape_sgml_chars $text]$quoteChar
}

safe_and_trusted_proc import_args_with_pairs {arrayName argsList} {
    upvar 1 $arrayName data

    if {[cindex $argsList 0] != "-"} {
	set data(args) $argsList
	return
    }

    set data(args) {}
    set looking 0
    set endit 0
    foreach arg $argsList {
	if $endit {
	    lappend data(args) $arg
	    continue
	}
	if $looking {
	    set data($varName) $arg
	    set looking 0
	    continue
	}
	if {[cindex $arg 0] == "-"} {
	    if {$arg == "--"} {
		set endit 1
		continue
	    }
	    if {$arg == "-args"} {
		return -code error "Args is a reserved value."
	    }
	    set varName [crange $arg 1 end]
	    set looking 1
	    continue
	}
	lappend data(args) $arg
    }
}
