#
# NeoWebScript - Server Side Programming based on Safe Tcl
#
# Copyright (C) 1997 NeoSoft, All Rights Reserved.  See NeoWebScript LICENSE
# files with this distribution for conditions on use and redistribution.
#
# Helpful routines for interfacing with Postgres.
#
# $Id: postgres.tcl,v 1.2 1999/07/09 20:49:08 damon Exp $
#

#
# Configuration directives for this version of postgres.tcl (intended for
# Posgresql6.0, but might work for earlier versions.
# 

# The following directives go in httpd.conf:
# NeoWebServerConf HavePostgres 1
#	There really is a postgres around somewhere
# NeoWebServerConf PostgresLib /usr/local/pgsql/lib/libpgtcl.so.1
#	Or wherever it is.  If staticly bound, and Tcl_StaticPackage
#	is called correctly, you can use "" instead of the path.
# NeoWebServerConf PostgresPkg Pgtcl
#	The name of the package to be loaded.  The old version was "Pg",
#	and the new version is "Pgtcl".
# NeoWebServerConf PostgresHost localhost
#	Optional: Use this directive if you want to specify a default host for
#	connections.  Localhost is the default.
# NeoWebServerConf PostgresPort 5432
#	Optional: Use this directive if you want to specify a default port for
#	connections.  5432 is the default.
# 

# To specify a directory which is to automatically have the pg_* commands
# in access.conf:
# NeoWebDirConf Postgres [Any | Yes]
#	"Any" means that any database name may be specified.  Otherwise
#	pg_connect is replaced with an alias that forces the user's database
#	names to always be prefixed by the web page owner name.  For example,
#	if user 'sam' owns the .html file executing "pg_connect mydb", the 
#	database name is forced to become "sam__mydb".

set sqllog [open $server(SERVER_ROOT)/logs/sql_log "WRONLY APPEND CREAT"]

#load $NeoWebServerConf(PostgresLib) $NeoWebServerConf(PostgresPkg)

proc postgres_setup {safeInterp} {
    global NeoWebDirConf NeoWebServerConf PostgresVersion
    if ![info exists NeoWebDirConf(Postgres)] return
    $safeInterp alias pg_connect SAFE_pg_connect $safeInterp
    $safeInterp alias pg_createdb SAFE_pg_createdb $safeInterp
    $safeInterp alias pg_version pg_version $safeInterp
    global PostgresLoaded
    set PostgresLoaded 0
    if [info exists NeoWebDirConf(PostgresVersion)] {
	set PostgresVersion $NeoWebDirConf(PostgresVersion)
    } else {
	set PostgresVersion ""
    }
}

proc pg_version {safeInterp version args} {
    global PostgresVersion NeoWebServerConf PostgresLoaded
    if {$PostgresLoaded} {
	return -code error "A PostgreSQL library is already loaded and cannot be changed"
    }
    if ![info exists NeoWebServerConf(PostgresLib$version)] {
	return -code error "$version is not a supported Postgresql library version"
    }
    set PostgresVersion $version
    postgres_load $safeInterp
}

proc postgres_load {safeInterp} {
    global PostgresLoaded PostgresVersion NeoWebServerConf NeoWebDirConf
    global env
    if $PostgresLoaded return
    set PostgresLoaded 1

    if [info exists NeoWebServerConf(PostgresPort$PostgresVersion)] {
	set env(PGPORT) $NeoWebServerConf(PostgresPort$PostgresVersion)
    }
    if [info exists NeoWebServerConf(PostgresHost$PostgresVersion)] {
	set env(PGHOST) $NeoWebServerConf(PostgresHost$PostgresVersion)
    }
#    if [info exists env(LD_LIBRARY_PATH)] {
#	append env(LD_LIBRARY_PATH) :
#    }
#    append env(LD_LIBRARY_PATH) [file dirname $NeoWebServerConf(PostgresLib$PostgresVersion)]
    httpd child_terminate
    load $NeoWebServerConf(PostgresLib$PostgresVersion) $NeoWebServerConf(PostgresPkg$PostgresVersion) $safeInterp
    load $NeoWebServerConf(PostgresLib$PostgresVersion) $NeoWebServerConf(PostgresPkg$PostgresVersion)
    if [info exists NeoWebDirConf(PostgresOptions)] {
	eval pg_option $NeoWebDirConf(PostgresOptions)
    }
    if {$NeoWebDirConf(Postgres) != "Any"} {
	$safeInterp alias pg_connect SAFE_pg_connect $safeInterp
	$safeInterp alias pg_createdb SAFE_pg_createdb $safeInterp
	$safeInterp alias pg_log pg_log
	$safeInterp eval {
	    rename pg_exec __pg_exec
	    proc pg_exec args {
		set start [clock seconds]
		set err [catch {eval __pg_exec $args} result]
		set duration [expr [clock seconds] - $start]
		if $err {
		    pg_log "pg_exec $args: $result"
		    global errorInfo
		    return -code error -errorinfo $errorInfo $result
		}
		catch {pg_log "$duration $args"}
		return $result
	    }
	}
    }
}

proc pg_log params {
    global sqllog webenv
    regsub {^/usr/ftp/(\./)?pub/users/[a-z]/} $webenv(SCRIPT_FILENAME) ~ path
    puts $sqllog "[clock format [clock seconds] -format "%D %T"] \
	$webenv(REMOTE_ADDR) $params $path"
    flush $sqllog
}

proc SAFE_pg_connect {safeInterp {databaseName {}} args} {
    global webpageOwnerName NeoWebServerConf NeoWebDirConf
    global PostgresVersion

    postgres_load $safeInterp
    if {$NeoWebDirConf(Postgres) == "Any"} {
	return [$safeInterp eval pg_connect $databaseName $args]
    }
    setup_data_access
    if {[lempty $databaseName] || $databaseName == $webpageOwnerName} {
      set databaseName $webpageOwnerName
    } else {
      set databaseName ${webpageOwnerName}__${databaseName}
    }
    if [info exists NeoWebDirConf(PostgresArgs)] {
	array set options $NeoWebDirConf(PostgresArgs)
    }
    set options(-host) $NeoWebServerConf(PostgresHost$PostgresVersion)
    set options(-port) $NeoWebServerConf(PostgresPort$PostgresVersion)
    array set options $args
    if {$options(-host) == "localhost"} {
	set options(-host) $NeoWebServerConf(PostgresHost$PostgresVersion)
	set options(-port) $NeoWebServerConf(PostgresPort$PostgresVersion)
    }
    set code [catch {eval pg_connect $databaseName [array get options]} handle]
    pg_log "pg_connect $databaseName [array get options]: $handle"
    if $code {return -code $code $handle}
    if {$NeoWebServerConf(PostgresPkg$PostgresVersion) == "Pgtcl"} {
	interp transfer {} $handle $safeInterp
	return $handle
    } else {
	return [pg_handle $safeInterp $handle]
    }
}

proc SAFE_pg_createdb {safeInterp {databaseName {}}} {
    global webpageOwnerName NeoWebServerConf PostgresVersion

    postgres_load $safeInterp
    setup_data_access
    if {[lempty $databaseName] || $databaseName == $webpageOwnerName} {
	set databaseName $webpageOwnerName
    } else {
	set databaseName ${webpageOwnerName}__${databaseName}
    }
    
    set sql_command "create database $databaseName;"
    set postgresFP [pg_connect template1]
    set result [pg_exec $postgresFP $sql_command]
    pg_result $result -clear
    pg_disconnect $postgresFP
    return
}

# Authorization routine for PostGres95.  Maintains a db cache locally
# which expires in 15 minutes.
# Use the following in .htaccess to reference this authentication
# routine:
#
# TclAuthBasic dbname table userCol passwordCol password
#
proc postgres_auth {database table userCol passwordCol username sent_pw} {
    global parallelSystemBase
    set password *
    set reference ($database.$table.$userCol/$passwordCol)
    set dbFileName $parallelSystemBase/pg95cache.db
    if {[catch {set db [db open $dbFileName hash rwL 0644]}] == 1} {
	set db [db open $dbFileName hash ctL 0664]
    }
    if [db get $db $database/$table/$username list] {
	lassign $list password expire
	if {[clock seconds] < $expire} {
	    if [check_password $sent_pw $password] {
		db close $db
		return
	    }
	}
    }
    # If Postgres is down, use the use cached password, even if
    # it is old.
    if [catch {set conn [pg_connect $database]}] {
	db close $db
	if [check_password $sent_pw $password] { return }
	return "user $username: database connect problem ($database)"
    }
    set result [pg_exec $conn \
	"select $passwordCol from $table where $userCol = '$username'"]
    set numTuples [pg_result $result -numTuples]
    if {$numTuples == 1} {
	set password [lindex [pg_result $result -getTuple 0] 0]
	db put $db $database/$table/$username [list $password [expr [clock seconds]+900]]
    }
    pg_result $result -clear
    pg_disconnect $conn
    db close $db
    if {$numTuples != 1} {
	return "user $username not found $reference"
    }
    if [check_password $sent_pw $password] return
    return $pwError
}

proc postgres_access {user type} {
    if {$type == "valid-user"} { return OK }
    return AUTH_REQUIRED
}

