# Carnet : a simple addressbook written in tcl/tk
# Copyright (C) 1996 Nocera Luciano
# Revision: $Id: browse.tcl,v 1.3 1997/03/14 18:09:19 lnocera Exp lnocera $
# State: Exp
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

#
# Configurable variables
#
# $BROWSE(NAME) : window and icon title
#
# $BROWSE(TITLE) : window title
#
# $BROWSE(DIR)  : directory where the browser is started
#
# $BROWSE(FILTER) : Filter used to browse files. Default is "*"
#
# $BROWSE(LISTDOTFILES) : set to 1 to list ".*" or 0 otherwise
#

#
# Thanks to Levison Libby.
#

# First define some usefull variables:
# filter used for ".*" dot files
#

set BROWSE(DOTFILTER) {[\.]*}
    
# Default name to be used if the title and the name are not 
# provided
#

set BROWSE(DEFAULT_NAME) "Browser"

#
# Proc follows
#

# Procedure to initialize the browser
#
proc BrowseInit {} {
    global BROWSE
    global browseval lfile
    
    if ![info exists BROWSE(TITLE)] { 
	set BROWSE(TITLE) $BROWSE(DEFAULT_NAME) 
    }
    if ![info exists BROWSE(NAME)]  { 
	set BROWSE(NAME)  $BROWSE(DEFAULT_NAME)
    }
    if ![info exists BROWSE(DIR)]   { 
	set BROWSE(DIR) [glob -nocomplain "./"] 
    }
    if ![info exists BROWSE(DIR)] { 
	set BROWSE(DIR) [glob -nocomplain "./"] 
    }
    if ![info exists BROWSE(LISTDOTFILES)] {
	set BROWSE(LISTDOTFILES) 0
    }
    # BROWSE(FILTER) don't need to be initialized
    
    if [winfo exists .seldb] {
	focus .seldb
    } else {
	toplevel .seldb
	wm title .seldb "$BROWSE(NAME)"
	wm iconname .seldb "$BROWSE(NAME)"
	# wm resizable .seldb 1 0
	wm minsize .seldb 20 10
	
	# debug
	bind .seldb <Control-g> { dbg:make }

	# the title
	set f [frame .seldb.title -relief raised -bd 2 -height 2]
	label $f.label -text $BROWSE(TITLE)
	pack $f.label -fill x -pady 5 -padx 5 
	pack $f -fill x

	# display files and dir
	set f [frame .seldb.show -relief raised -bd 2]
	scrollbar $f.scroll -command "$f.list yview"
	pack $f.scroll -side right -fill y
	set BROWSE(LIST) [listbox $f.list -yscroll "$f.scroll set" \
		      -relief sunken -setgrid yes]
	set BROWSE(MESSAGE) ""
	pack $f.list -side left -fill both -expand yes
	pack $f -fill both -expand true

	# message
	set f [frame .seldb.message -relief raised -bd 2]
	label $f.lab -textvariable BROWSE(MESSAGE)
	pack $f.lab -pady 5 -padx 5
	pack $f -fill x 
	
	# filter window
	set f [frame .seldb.filter -relief raised -bd 2]
	checkbutton $f.dot -variable BROWSE(LISTDOTFILES) \
	    -command "BrowseComplete"
	label $f.lab -text "Filter :"
	label $f.lab2 -text "Show .* files"
	set BROWSE(FILTER_ENTRY) [entry $f.filter \
				      -textvariable BROWSE(FILTER) \
				      -width 22]
	pack $f.lab $f.filter $f.lab2 $f.dot -pady 5 -padx 5 -side left
	pack $f -fill x
	
	# file selector
	set f [frame .seldb.sel -relief raised -bd 2]
	label $f.lab -text "File   :"
	set BROWSE(ENTRY) [entry $f.sel -textvariable lfile -width 40]
	
	pack $f.lab $f.sel -pady 5 -padx 5 -side left
	pack $f -fill x

	# a button frame
	set f [frame .seldb.buttons -relief raised -bd 2]
	set f2 [frame $f.ok -relief sunken -bd 2]
	button $f2.ok -text "Ok" -command {
	    if [file isfile $lfile] { set browseval 1 }}
	pack $f2.ok -padx 5 -pady 5
	button $f.abort -text "Abort" -command {set browseval 0}

	pack $f2 $f.abort -padx 5 -pady 5 -side left -expand true
	pack $f -fill x 
	focus $f2.ok

	# the rest of the stuff
	# list current dir
	BrowseBrowse
	
	# Set up bindings for the browser.
	bind $BROWSE(LIST) <Double-Button-1> {BrowseBrowse [selection get]}
	bind $BROWSE(ENTRY) <Tab> "BrowseComplete; break"
	bind $BROWSE(ENTRY) <Return> "BrowseComplete; break"
	
	bind $BROWSE(FILTER_ENTRY) <Tab> "BrowseComplete; break"
	bind $BROWSE(FILTER_ENTRY) <Return> "BrowseComplete; break"
	
	focus $BROWSE(ENTRY)
	tkwait variable browseval
	
	grab release .seldb
	
	if {$browseval == 1} {
	    wm withdraw .seldb
	    update
	    destroy .seldb
	    return $lfile
	} else {
	    wm withdraw .seldb
	    update
	    destroy .seldb
	    return
	}
    }
}

proc BrowseComplete {} {
    global BROWSE

    set BROWSE(MESSAGE) " "
    set file [$BROWSE(ENTRY) get]
    set dir [file dirname $file]
    set file [file tail $file]

    set completions {}
    set status [catch { exec /usr/bin/ls -ap $dir } matches]

    if { !$status } {
	foreach i $matches {
	    if [string match $file* $i] {
		lappend completions $i
	    }
	}
    } else {
	set BROWSE(MESSAGE) "\[ Can't open file $BROWSE(DIR) \] "
	return
    }

    set howmuch [llength $completions]
    switch -- $howmuch {
	0 {
	    $BROWSE(LIST) delete 0 end
	    set BROWSE(MESSAGE) "\[ No match \] "
	}
	1 {
	    # BrowseBrowse $dir
	    set BROWSE(MESSAGE) ""
	    set BROWSE(DIR) $dir/$completions
	    $BROWSE(ENTRY) delete 0 end
	    $BROWSE(ENTRY) insert insert $BROWSE(DIR)
	    if [file isdirectory [glob -nocomplain $BROWSE(DIR)]] { 
		BrowseComplete
	    } else {
		$BROWSE(LIST) delete 0 end
		# this guy whants the file and it is 
		# the $completions: show it in the list
		$BROWSE(LIST) insert end $completions
		set BROWSE(MESSAGE) " "
	    }
	}
	default {
	    # patch
	    set root $file
	    set rootLength [string length $root]

	    # scan completions to see if the new root is still a root

	    set cont 0
	    set stopLength [string length [lindex $completions 0]]
	    while {$cont < $stopLength } {
		# add a char to file
		set nroot [string range \
			       [lindex $completions 0] \
			       0 [expr $rootLength - 1]]
		set ok 1
		foreach i $completions {
		    if ![string match "$nroot*" $i] { set ok 0 }
		}

		if {$ok} {
		    set root $nroot
		    incr rootLength 1
		} else {
		    set cont $stopLength
		}
		incr cont 1
	    }

	    # now update the entry widget
	    if {$dir == "/"} { set dir ""}
	    $BROWSE(LIST) delete 0 end
	    $BROWSE(ENTRY) delete 0 end
	    $BROWSE(ENTRY) insert insert $dir/$root
	    set BROWSE(DIR) $dir/$root
	    set BROWSE(MESSAGE) ""
	    
	    if { $BROWSE(FILTER) == ""} { set BROWSE(FILTER) "*" }
	    set cont 0
	    foreach i $completions {
		if [file isfile $BROWSE(DIR)/$i] {
		    # use the filter to insert files
		    if [string match $BROWSE(FILTER) $i] {
			if { !$BROWSE(LISTDOTFILES) } {
			    if ![string match $BROWSE(DOTFILTER) $i] {
				$BROWSE(LIST) insert end $i
				incr cont 1
			    }
			} else {
			    $BROWSE(LIST) insert end $i
			    incr cont 1
			}
		    }
		} else {
		    if { !$BROWSE(LISTDOTFILES) } {
			if { [string match "./" $i] || \
				 [string match "../" $i] } {
			    $BROWSE(LIST) insert end $i
			    incr cont 1
			} else {
			    if ![string match $BROWSE(DOTFILTER) $i] {
				# do not filter directories
				$BROWSE(LIST) insert end $i
				incr cont 1
			    }
			}
		    } else {
			$BROWSE(LIST) insert end $i
			incr cont 1
		    }
		}
	    }
	    # set error message
	    if { $cont == 0 } { set BROWSE(MESSAGE) "\[ No match \] "}
	}
    }
}
    
# The procedure below is invoked to open a browser on a given file;  if the
# file is a directory then another instance of this program is invoked; if
# the file is a regular file then the Mx editor is invoked to display
# the file.
proc BrowseBrowse {{file ""}} {
    global env BROWSE
    set oldDir $BROWSE(DIR)

    set BROWSE(MESSAGE) ""
    if {$BROWSE(DIR) == ""} { set BROWSE(DIR) [glob "~/"] }
    if {$file == "./"} { set file ""}
    
    if [file isdirectory $BROWSE(DIR)/$file] {
	
	if { $file == "../" } {
	    
	    # remove .. in the path
	    regsub {/[^/]+/*$} $BROWSE(DIR) "" BROWSE(DIR) 
	    if {$BROWSE(DIR) == ""} {set BROWSE(DIR) "/"}
	    set BROWSE(DIR) [glob -nocomplain $BROWSE(DIR)/]
	} else {
	    if { $file != "." } {
		if {$BROWSE(DIR) != "/"} {
		    set BROWSE(DIR) [glob -nocomplain $BROWSE(DIR)/$file]
		} else {
		    set BROWSE(DIR) [glob -nocomplain $BROWSE(DIR)$file]
		}
	    }
	}
	# treat some special cases ...

	if {$BROWSE(DIR) == "//"} { set BROWSE(DIR) "/"}

	set status [catch { exec /usr/bin/ls -ap $BROWSE(DIR) } files]

	if { !$status } {

	    $BROWSE(LIST) delete 0 end
	    $BROWSE(ENTRY) delete 0 end
	    $BROWSE(ENTRY) insert insert $BROWSE(DIR)
	    set BROWSE(MESSAGE) ""
	    if { $BROWSE(FILTER) == ""} { set BROWSE(FILTER) "*" }
	    set cont 0
	    foreach i $files {
		if [file isfile $BROWSE(DIR)/$i] {
		    # use the filter to insert files
		    if [string match $BROWSE(FILTER) $i] {
			if { !$BROWSE(LISTDOTFILES) } { ;# for .* files
			    if ![string match $BROWSE(DOTFILTER) $i] {
				$BROWSE(LIST) insert end $i
				incr cont 1
			    }
			} else {
			    $BROWSE(LIST) insert end $i
			    incr cont 1
			}
		    }
		} else {
		    if { !$BROWSE(LISTDOTFILES) } { ;# coucou
			if { [string match "./" $i] || \
				 [string match "../" $i] } {
			    $BROWSE(LIST) insert end $i
			    incr cont 1
			} else {
			    if ![string match $BROWSE(DOTFILTER) $i] {
				# do not filter directories
				$BROWSE(LIST) insert end $i
				incr cont 1
			    }
			}
		    } else {
			$BROWSE(LIST) insert end $i
			incr cont 1
		    }
		}
	    }
	    
	    # set error message
	    if { $cont == 0 } { set BROWSE(MESSAGE) "\[ No match \] "}

	} else {
	    
	    set BROWSE(MESSAGE) "\[ Can't open file $BROWSE(DIR) \] "
	    set BROWSE(DIR) $oldDir
	    return
	    
	}

    } elseif [file isfile $BROWSE(DIR)/$file] {
	# for files
	set BROWSE(MESSAGE) ""
	$BROWSE(ENTRY) delete 0 end
	regsub {/[^/]+$} "$BROWSE(DIR)$file" "/$file" name
	set BROWSE(DIR) 
	$BROWSE(ENTRY) insert insert $name
	
    } else {
	# clean-up the BROWSE(DIR) in case a vile user has inserted some 
	# characters after the slash
	regsub {/[^/]+$} "$BROWSE(DIR)" "/$file" BROWSE(DIR)
	$BROWSE(LIST) delete 0 end
	BrowseBrowse
    }
}