# Carnet : a simple addressbook written in tcl/tk
# Copyright (C) 1996 Nocera Luciano
# Revision: $Id: proc.tcl,v 1.3 1997/03/14 18:09:19 lnocera Exp lnocera $
# State: Exp
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

#######################################################################
# procs 
#
#######################################################################

set helpTopics(command_line) "$INTERNAL(NAME) is an address-book written in tcl/tk.

$INTERNAL(NAME) \[options\] \[args\]

Search options:
$INTERNAL(NAME) \[-regexp\|-global\|-exact\] \[pattern\]

Help options (display this help):
$INTERNAL(NAME) -?

Load external file:
$INTERNAL(NAME) -f file

Interface:
$INTERNAL(NAME)"

proc TestInstall {} {
    global INTERNAL
    
    if ![file isdirectory $INTERNAL(LOCAL_DIR_NAME)] { 
	if [tk_dialog .delete {Load failed} \
		 "$INTERNAL(NAME): No $INTERNAL(LOCAL_DIR_NAME) directory found. Press Initialise to set-up your local database." \
		info 0 "Initialise" "Abort"] { exit }
	if [catch { exec "mkdir" $INTERNAL(LOCAL_DIR_NAME) } err ] {
	    tk_dialog .delete {Create failed} \
		"$INTERNAL(NAME) : $err, exiting." error 0  "Ok"
	    exit
	}
	Save 1
	# chmod
	set file $INTERNAL(DBPRIVATE_NAME)
	if [catch { exec "chmod" "600" $file }] {
	    tk_dialog .delete {Create file failed} \
		"$INTERNAL(NAME) : Couldn't execute chmod 600 on $file. Check permissions on $file." \
		error 0  "Ok"
	}
	# chmod
	set file $INTERNAL(DBPUBLIC_NAME)
	if [catch { exec "chmod" "644" $file }] {
	    tk_dialog .delete {Create file failed} \
		"$INTERNAL(NAME) : Couldn't execute chmod 600 on $file. Check permissions on $file." \
		error 0  "Ok"
	}
	SpyNewUser ;# used the first time carnet is started 
    }
}

proc CheckTouch {} {
    # return 1 if record is modified 0 otherwise. If record is modified ask 
    # for saving
    global INTERNAL DATABASE RECORD
    # current record
    set new_record [GetRecord]
    # last edited record
    set old_key $INTERNAL(CURRENT_KEY)
    set new_key [GetCurrentKey]

    if { $old_key != {} } { ;# for the first time
	
	set old_record $DATABASE($old_key)
	set old_priv [lindex $old_record 6]
	set touch 0
	for {set i 0} { $i < 7 } {incr i 1} {
	    if { [lindex $new_record $i] != [lindex $old_record $i] } {
		set touch 1
		# puts $i
		# puts [lindex $new_record $i]
		# puts [lindex $old_record $i]
	    }
	}
	if {$new_key != ""} {
	    if {$touch} {
		set rep [tk_dialog .delete {Confirm Action} \
			     {Current record is modified: save changes?} \
			     questhead 0  "Yes"  "No" "Abort"]
		switch -- $rep {
		    {0} { return [Update]}
		    {1} { return 0 }
		    {2} { return 1 }
		}
	    } else {
		return 0
	    }
	} else { return [CheckInsertedText] }
    } else { return [CheckInsertedText] }
    return 0
}

proc CheckInsertedText {} {
    set record [GetRecord]
    set touch 0
    for {set i 0} {$i < 6} {incr i 1} { ;# do not test for public checkbutton
	if {[lindex $record $i] != {}} {set touch 1}
    }
    if {$touch == 1} {
	set rep [tk_dialog .delete {Confirm Action} \
		     {Remove inserted text?} \
		     questhead 0  "Yes"  "No" "Abort"]
	switch -- $rep {
	    {0} { return 0 }
	    {1} { return 1 }
	    {2} { return 1 }
	}
    } else {
	return 0
    }
}

proc CheckSave {} {
    global INTERNAL
    
    if {$INTERNAL(SAVE)} {
	set rep [tk_dialog .delete {Confirm Action} \
		     {The database is modified: save changes?} \
		     questhead 0  "Yes"  "No" "Abort"]
	switch -- $rep {
	    {0}  { Save 1
		return 1
	    }
	    {1} { set INTERNAL(SAVE) 0
		return 1
	    }
	    {2} { return 0 }
	}
    }
    return 1    
}

proc ClearAndSearch {} {
    global SEARCH
    
    SearchRecord
    focus $SEARCH(ENTRY)
}

proc GetStringToSearch {} {
    global SEARCH
    set pattern [string trim [$SEARCH(ENTRY) get 1.0 end]]
    return $pattern
}

proc ParseImportFile {} {
    # parse INTERNAL(IMPORTFILE_NAME) for email and names to 
    # insert in the database
    global INTERNAL

    set file $INTERNAL(IMPORT_FILE_NAME)
    
    if [file exists $file] {
	if [catch {open $file r} dbf] {
	    set mess
	    "$INTERNAL(NAME): Couldn't open $file."
	} else {
	    set db [read $dbf]
	    set db [string trim $db]
	    set all [split $db "\n"]
	    set phone ""
	    set company ""
	    set address ""
	    set notes ""
	    set priv "public"
	    foreach r $all {
		regexp -nocase {[a-z.0-9_\-]+@[a-z.0-9_\-]+} $r email
		regsub $email $r "" name
		if { $name == "" } {
		    # in case we have only the mail
		    # use the login as the name
		    regexp -nocase {[a-z.0-9_\-]+} $r name
		} else {
		    # remove <> or ()
		    regsub -all {\(} $name "" name
		    regsub -all {\)} $name "" name
		    regsub -all {\<} $name "" name
		    regsub -all {\>} $name "" name
		}
		# try to insert the new record
		# InsertImport check also for the correcteness of
		# the record
		InsertImport $name $company $email \
		    $phone $address $notes $priv
	    }
	    exec {/bin/rm} $file
	    Save 1
	}
    }
}

proc SearchRecord {{how {}} {pattern ""}} {
    # search using a given pattern
    
    global DATABASE INTERNAL SEARCH CONFIG
    ParseImportFile ;# 
    if [ CheckTouch ] { return }
    if {$pattern == ""} { set pattern [GetStringToSearch] }
    if {$how == {}} { set how $CONFIG(SEARCH)}
    regsub -all "{" $pattern {\{} pattern
    regsub -all "}" $pattern {\}} pattern
    Message $pattern
    set rmatch [SearchMatch $how $pattern]
    set howmuch [llength $rmatch]
    ClearSearchList
    Message "$howmuch records matching \"$pattern\""
    switch -- $howmuch {
	0 { Message "No match" }
	1 { set match [lindex $rmatch 0]
	    ShowRecord $rmatch
	    set label [GetLabelFromKey $rmatch]
	    $SEARCH(LIST) insert end $label
	}
	default {
	    foreach key [lsort $rmatch] {
		set label [GetLabelFromKey $key]
		$SEARCH(LIST) insert end $label
	    }
	}
    }
}

proc ClearRecord { {check 1} } {

    global RECORD SEARCH INTERNAL
    if {$check} {if [ CheckTouch ] { return }}
    set INTERNAL(CURRENT_KEY) ""
    foreach i {NAME COMPANY EMAIL PHONE ADDRESS NOTES} {$RECORD($i) delete 1.0 end}
    set RECORD(PRIVATE) 0
    $RECORD(EXTERNAL) config -text ""
    UpdateLog "" ""
}

proc ResetRecord {} {
    global RECORD
    
    ClearRecord
    focus $RECORD(NAME)
}

proc ClearSearchList {} {
    global RECORD SEARCH

    $SEARCH(LIST) delete 0 end
}

proc DeleteRecord {} {
    global RECORD DATABASE INTERNAL
    
    set key [GetCurrentKey]
    if { $key != "" } {
	if [info exist DATABASE($key)] {
	    set rep [tk_dialog .delete {Confirm Action} \
			 {Really delete this record ?} \
			 questhead 0 Yes No Abort ]
	    if {$rep == 0} {
		unset DATABASE($key)
		set INTERNAL(CURRENT_KEY) ""
		UpdateLog "" ""
		ClearRecord 0 ;# do not test for modified record
		SearchRecord
		set INTERNAL(SAVE) 1
		set INTERNAL(MODIFIED) 1
		Save
	    }
	} else { Message "Sorry: this record is not yet in the database" }
    } else {
	tk_dialog .delete {Confirm Action} \
	    "Can't delete: select a record first and retry" \
	    error 0  "Ok"
    }
}

proc Reload {} {
    global  DATABASE RECORD SEARCH CONFIG INTERNAL

    if [ CheckTouch ] { return }
    if [tk_dialog .delete {Confirm Action} \
	    "Reloading drops external database records: continue?" \
	    questhead 0  "No, abort" "Yes reload"] {
    
	if {$INTERNAL(SAVE)} {
	    set rep [tk_dialog .delete {Confirm Action} \
			 "The database is modified: discard changes ?" \
			 questhead 0  "No, save and reload" "Yes, discard and reload" "Abort"]
	    switch -- $rep {
		{0} { Save 1 }
		{2} { return }
	    }
	}

	# clean internal variables
	foreach key [array names DATABASE] { unset DATABASE($key) }
	set INTERNAL(CURRENT_KEY) ""
	UpdateLog "" ""
	set INTERNAL(SAVE) 0
	set INTERNAL(MODIFIED) 0
	ClearRecord 0 ;# do not check for any displayed record
	Load public $INTERNAL(DBPUBLIC_NAME)
	Load private $INTERNAL(DBPRIVATE_NAME)
    }
}

proc Load {what file} {
    # load a database
    global DATABASE RECORD SEARCH CONFIG INTERNAL

    set mess [ LoadDb $what $file]
    if {$mess != ""} { HelpMessage "Load report" $mess }
    SearchRecord
    Message "Database loaded"
}

proc SaveIt {what file} {
    global DATABASE RECORD SEARCH CONFIG INTERNAL
    
    if [catch {open $file w} dbf] {
	tk_dialog .delete {write failed} "$INTERNAL(NAME) $dbf" \
	    error 0 Ok
    } else {
	# header first
	puts -nonewline $dbf $INTERNAL(HEADER)$INTERNAL(KEND)
	if [array exist DATABASE] { ;# initialization
	    # records follow
	    foreach k [lsort [ array names DATABASE ]] {
		if {[lindex $DATABASE($k) 6] == $what } {
		    puts -nonewline $dbf \
			[join [lrange $DATABASE($k) 0 5] $INTERNAL(KSEP)]$INTERNAL(KEND)
		}
	    }
	}
    }
    close $dbf
}

proc Save {{force 0}} {
    global DATABASE RECORD SEARCH CONFIG INTERNAL
    
    if {$CONFIG(AUTOSAVE) || $force} {
	SaveIt "private" $INTERNAL(DBPRIVATE_NAME)
	SaveIt "public" $INTERNAL(DBPUBLIC_NAME)
	set INTERNAL(SAVE) 0
	Message "The database have been saved"
    }
}

proc ShowRecord {key} {
    # show the current record
   
    global RECORD DATABASE INTERNAL SEARCH

    if [ CheckTouch ] { return }
    set INTERNAL(CURRENT_KEY) ""
    foreach i {NAME COMPANY EMAIL PHONE ADDRESS NOTES} {$RECORD($i) delete 1.0 end}
    $RECORD(EXTERNAL) config -text ""
    Message ""
    set INTERNAL(CURRENT_KEY) $key
    set record $DATABASE($key)
    $RECORD(NAME) insert insert [lindex $record 0]
    $RECORD(COMPANY) insert insert [lindex $record 1]
    $RECORD(EMAIL) insert insert [lindex $record 2]
    $RECORD(PHONE) insert insert [lindex $record 3]
    $RECORD(ADDRESS) insert insert [lindex $record 4]
    $RECORD(NOTES) insert insert [lindex $record 5]
    # hypertext
    foreach item { EMAIL PHONE ADDRESS NOTES } { HyperParse $RECORD($item) }
    # search hilight
    foreach item { NAME COMPANY EMAIL PHONE ADDRESS NOTES } { 
	ParseSearch $RECORD($item) [GetStringToSearch]
    }
    $RECORD(EXTERNAL) config -text ""
    switch -- [lindex $record 6] {
	"private" { set RECORD(PRIVATE) 1 }
	"public" { set RECORD(PRIVATE) 0 }
	"external" {
	    set RECORD(PRIVATE) 0
	    $RECORD(EXTERNAL) config -text "External"
	}
    }
    UpdateLog "[lindex $record 0] - [lindex $record 1]" "[lindex $record 6] record"
}

proc ShowRecordUsingLabel {} {
    # show the current record from the label
    global RECORD DATABASE SEARCH
    set sel [$SEARCH(LIST) index active]
    if {$sel != "" } {
	set label [$SEARCH(LIST) get $sel]
	set key [GetKeyFromLabel $label]
	if {$key != {}} { ShowRecord $key }
    }
}

#
# proc deifning the key for a record
#

proc GetKey {name company} {
    global INTERNAL
    # return the key from the name and the company
    # if no name is given assume it is a company and return a key
    # defined as the name of the company in lower case
    # if the name is given define the key as lastnamefirstname 
    # in lower case (this is the usual definition used in
    # the from line of the mail.

    if {$name != ""} {
	# set it to the name in lower case
	# set key [string tolower $name]

	# use lastname first
	set length [expr [llength $name] - 1]
 	set lastname [string tolower [lindex $name $length]]
 	regsub -nocase $lastname $name "" firstname
 	set firstname [string tolower $firstname]
 	set key "$lastname$firstname"
	# puts "$name  -> $lastname $firstname $key"
    } else {
	# this is for the list of names that needs a marker defining
	# the company.  We make sure the name do not contain this
	# marker: nuke the marker if exist.
	regsub -all $INTERNAL(COMPANY_MARK) $company "" company
	# set it to the company name in lower case
	set key [string tolower $company]
	# puts ">>> $company  ->  $key"
    }
    regsub -all " " $key "" key ;# remove blanks if any
    return $key
}
proc GetCurrentKey {} {
    global RECORD INTERNAL
    # remove newlines just in the name and company (bug fixed)
    regsub -all "\n" [string trim [$RECORD(NAME) get 1.0 end]] "" name
    regsub -all "\n" [string trim [$RECORD(COMPANY) get 1.0 end]] "" company
    if ![ string compare $name "" ] {
	# this is for the list of names that needs a marker defining
	# the company.  We make sure the name do not contain this
	# marker: nuke the marker if exist.
	regsub -all $INTERNAL(COMPANY_MARK) $company "" company
	# change also in the display
	$RECORD(COMPANY) delete 1.0 end
	$RECORD(COMPANY) insert insert $company
    }
    set key [GetKey $name $company]
    return $key
}
proc GetKeyFromLabel {label} {
    # compute the key from the label used in the search list
    # this has to be compatible with the definition of GetLabelFromKey
    # the reverse function and GetKey
    global INTERNAL
    # keyes are lower case
    set label [string tolower $label]

    # find if it is a record using the special control charachter
    if [regsub -all $INTERNAL(COMPANY_MARK) $label "" guess] {
	# it is a company: remove marks
	set key $guess
    } else {
	# it is a name: construct the key using GetKey
	set key [GetKey $label ""]
    }
    regsub -all " " $key "" key
    return $key
}
proc GetLabelFromKey {key} {
    # find the label displayed in the list from a given record key
    global DATABASE INTERNAL
    set name [lindex $DATABASE($key) 0]
    set company [lindex $DATABASE($key) 1]
    if {$name != ""} {
	set label $name
    } else {
	# add some marker used to track companies see the reverse function
	# GetKeyFromLabel
	set label [concat \
		       $company \
		       $INTERNAL(COMPANY_MARK)]
    }
    return $label
}
proc GetRecordLabel {} {
    global RECORD

    set name    [string trim [$RECORD(NAME)    get 1.0 end]]
    set company [string trim [$RECORD(COMPANY) get 1.0 end]]
    if {$name != ""} { set label $name
    } else { set label $company }
    return $label
}

#
#
#

proc GetRecord {} {
    global RECORD INTERNAL

    set name    [string trim [$RECORD(NAME)    get 1.0 end]]
    set company [string trim [$RECORD(COMPANY) get 1.0 end]]
    set email   [string trim [$RECORD(EMAIL)   get 1.0 end]]
    set phone   [string trim [$RECORD(PHONE)   get 1.0 end]]
    set address [string trim [$RECORD(ADDRESS) get 1.0 end]]
    set notes   [string trim [$RECORD(NOTES)   get 1.0 end]]

    if ![ string compare $name "" ] {
	# this is for the list of names that needs a marker defining
	# the company.  We make sure the name do not contain this
	# marker: nuke the marker if exist.
	regsub -all $INTERNAL(COMPANY_MARK) $company "" company
    }
    
    if {[$RECORD(EXTERNAL) cget -text] == "External"} {
	set priv "external"
    } else {
	if {$RECORD(PRIVATE)} { set priv "private" } else { set priv "public" }
    }
    set record [list $name $company $email $phone $address $notes $priv]
    return $record
}

proc InsertImport {name company email phone address notes priv} {
    global SEARCH DATABASE INTERNAL
    
    set name    [string trim $name    ]
    set company [string trim $company ]
    set email   [string trim $email   ]
    set phone   [string trim $phone   ]
    set address [string trim $address ]
    set notes   [string trim $notes   ]

    if ![ string compare $name "" ] {
	# this is for the list of names that needs a marker defining
	# the company.  We make sure the name do not contain this
	# marker: nuke the marker if exist.
	regsub -all $INTERNAL(COMPANY_MARK) $company "" company
    }
    
    set key [ GetKey $name $company ]
    set record [list $name $company $email $phone $address $notes $priv]
    
    if { $key == "" } {
	return 1
    }
    
    # warn for inserted records ealready in the database
    if [info exists DATABASE($key)] {
	# check if the email add is the same as the previously recorded
	set oemail [lindex $DATABASE($key) 2]
	set ophone [lindex $DATABASE($key) 3]
	set oaddress [lindex $DATABASE($key) 4]
	set onotes [lindex $DATABASE($key) 5]
	if ![regexp -nocase $email $oemail match] {
	    set email  [string trim "$email, $oemail"]
	}
	if ![regexp -nocase $phone $ophone match] {
	    set phone [string trim "$phone\n$ophone"]
	} else {
	    set phone $ophone
	}
	if ![regexp -nocase $address $oaddress match] {
	    set address [string trim "$address\n$oaddress"]
	} else {
	    set address $oaddress
	}
	if ![regexp -nocase $notes $onotes match] {
	    set notes [string trim "$notes\n$onotes"]
	} else {
	    set notes $onotes
	}
	set record [list $name $company $email $phone $address \
			$notes $priv]
	
	if ![string compare $INTERNAL(CURRENT_KEY) $key] {
	    # prevent the modif message to appear
	    ClearRecord 0;# do not check for the displayed record
	    set DATABASE($key) $record
	    ShowRecord $key
	    return 0
	}
    } else {
	# else insert the record as new
	set DATABASE($key) $record
	# database is modified and need to be saved
	# set INTERNAL(SAVE) 1
	# set INTERNAL(MODIFIED) 1
	return 0
    }
}

proc Update {} {
    global RECORD SEARCH DATABASE INTERNAL
    
    set key [GetCurrentKey]
    set old_key $INTERNAL(CURRENT_KEY)
    
    if {$key == ""} {
	tk_dialog .delete {Confirm Action} \
	    "You must define at least a name or a company to insert a record" \
	    error 0  "Ok"
	return 0
    }
    
    # warn for inserted records ealready in the database
    if [info exists DATABASE($key)] {
	if { $old_key != $key } {
	    set rep [tk_dialog .delete {Confirm Action} \
			 "Record ealready exists: override it?" \
			 questhead 0  "Abort" "Yes override"]
	    if {$rep != 1} { return 1 }
	}
    }
	    
    if {$old_key != ""} {
	# warn for external records
	if [info exists DATABASE($old_key)] {
	    if {[lindex $DATABASE($old_key) 6] == "external"} {
		if {![tk_dialog .delete {Confirm Action} \
			 "Record is external: insert it as yours ?" \
			  questhead 0  "Abort" "Yes register as mine"]} {
		    return 1 
		} else {
		    $RECORD(EXTERNAL) config -text ""
		}
	    }
	}

	if {$old_key != $key} {
	    # ask if the current record should be replaced or not
	    set name [GetLabelFromKey $old_key]
	    set newname [GetRecordLabel]
	    set rep [tk_dialog .delete {Confirm Action} \
			 "Replace \"$name\" with \"$newname\" ?" \
			 questhead 0  "Yes replace" "No, create a new record" "Abort"]
	    switch -- $rep {
		{0} { unset DATABASE($old_key) }
		{2} { return 1 }
	    }
	}
    }
    # database is modified and need to be saved
    set DATABASE($key) [GetRecord]
    set INTERNAL(SAVE) 1
    set INTERNAL(MODIFIED) 1
    set INTERNAL(CURRENT_KEY) $key
    # display the record
    SearchRecord
    ShowRecord $key
    Save
    return 0
}

proc Quit {} {
    global INTERNAL
    SaveOptions
    if [CheckTouch] { return }
    if {$INTERNAL(MODIFIED)} { Backup }
    if [CheckSave]  { exit   }
}

proc Exit {} {
    global INTERNAL
    if {$INTERNAL(MODIFIED)} {
	if [tk_dialog .delete {Confirm Action} \
		"Really exit $INTERNAL(NAME) without saving changes?" \
		error 0  "Abort" "Yes exit without saving"] { exit }
    } else { exit }
    return
}

proc UpdateLog {title message} {
    Message $message
    TitleLog $title
}

proc TitleLog {title} {
    global INTERNAL CONFIG
    set w [expr ($CONFIG(ENTRY_WIDTH) + $CONFIG(LIST_WIDTH))]
    set title [string range $title 0 $w]
    set INTERNAL(TITLE) $title
}

proc SaveOptions {} {
    # save the current options in the config file
    
    global INTERNAL CONFIG
    
    set file $INTERNAL(CONFIG_FILE)
    if {$CONFIG(REMEMBER_GEOMETRY)} {
	set CONFIG(GEOMETRY) [wm geometry .]
    } else {
	set CONFIG(GEOMETRY) ""
    }
    if [catch {open $file w} dbf] {
	tk_dialog .delete {Confirm Action} \
	    "Couldn't open $file: check $file and retry" \
	    error 0  "Ok"
	return
    } else {
	puts $dbf "
#----------------------------------------------------------------------#
# Do not edit :
# Option settings for $INTERNAL(NAME) automatically generated
#----------------------------------------------------------------------#

#----------------------------------------------------------------------#
# Appearance settings
#----------------------------------------------------------------------#

set CONFIG(TOOLBAR_ON)         		\{$CONFIG(TOOLBAR_ON)\}
set CONFIG(LOGBAR_ON)          		\{$CONFIG(LOGBAR_ON)\}
set CONFIG(ALPHABAR_ON)        		\{$CONFIG(ALPHABAR_ON)\}
set CONFIG(XSCROLLBAR_ON)      		\{$CONFIG(XSCROLLBAR_ON)\}
set CONFIG(YSCROLLBAR_ON)      		\{$CONFIG(YSCROLLBAR_ON)\}

#----------------------------------------------------------------------# 
# Print options
#----------------------------------------------------------------------#

set CONFIG(PRINT_EMAIL)        		\{$CONFIG(PRINT_EMAIL)\}
set CONFIG(PRINT_PHONE)        		\{$CONFIG(PRINT_PHONE)\}
set CONFIG(PRINT_ADDRESS)      		\{$CONFIG(PRINT_ADDRESS)\}
set CONFIG(PRINT_NOTES)        		\{$CONFIG(PRINT_NOTES)\}
set CONFIG(PRINT_FORMAT)       		\{$CONFIG(PRINT_FORMAT)\}
set CONFIG(PRINT_FILE)         		\{$CONFIG(PRINT_FILE)\}
set CONFIG(PRINT_WHAT)         		\{$CONFIG(PRINT_WHAT)\}

#----------------------------------------------------------------------# 
# preferences
#----------------------------------------------------------------------#

set CONFIG(BACKUP_LEVEL)                \{$CONFIG(BACKUP_LEVEL)\}
set CONFIG(USE_ACCENTS)        		\{$CONFIG(USE_ACCENTS)\}
set CONFIG(AUTOSAVE)           		\{$CONFIG(AUTOSAVE)\}
set CONFIG(REMEMBER_GEOMETRY)        	\{$CONFIG(REMEMBER_GEOMETRY)\}
set CONFIG(GEOMETRY)        		\{$CONFIG(GEOMETRY)\}
set CONFIG(ENTRY_WIDTH)        		\{$CONFIG(ENTRY_WIDTH)\}
set CONFIG(ENTRY_HEIGHT)       		\{$CONFIG(ENTRY_HEIGHT)\}
set CONFIG(LIST_WIDTH)         		\{$CONFIG(LIST_WIDTH)\}
set CONFIG(EDITOR_COMMAND) 		\{$CONFIG(EDITOR_COMMAND)\}
set CONFIG(BROWSER_COMMAND) 		\{$CONFIG(BROWSER_COMMAND)\}
set CONFIG(FTP_COMMAND)      		\{$CONFIG(FTP_COMMAND)\}
set CONFIG(MAIL_COMMAND)       		\{$CONFIG(MAIL_COMMAND)\}
set CONFIG(PHONE_COMMAND)      		\{$CONFIG(PHONE_COMMAND)\}
set CONFIG(BROWSER_HIGHLIGHT_COLOR) 	\{$CONFIG(BROWSER_HIGHLIGHT_COLOR)\}
set CONFIG(ANNIVERSARY_HIGHLIGHT_COLOR) \{$CONFIG(ANNIVERSARY_HIGHLIGHT_COLOR)\}
set CONFIG(FTP_HIGHLIGHT_COLOR) 	\{$CONFIG(FTP_HIGHLIGHT_COLOR)\}
set CONFIG(FILE_HIGHLIGHT_COLOR) 	\{$CONFIG(FILE_HIGHLIGHT_COLOR)\}
set CONFIG(PHONE_HIGHLIGHT_COLOR) 	\{$CONFIG(PHONE_HIGHLIGHT_COLOR)\}
set CONFIG(MAIL_HIGHLIGHT_COLOR)	\{$CONFIG(MAIL_HIGHLIGHT_COLOR)\}
set CONFIG(SEARCH)             		\{$CONFIG(SEARCH)\}
set CONFIG(SEARCH_CASE)        		\{$CONFIG(SEARCH_CASE)\}
set CONFIG(SEARCH_INTERACTIVE) 		\{$CONFIG(SEARCH_INTERACTIVE)\}
set CONFIG(DATE_FORMAT) 		\{$CONFIG(DATE_FORMAT)\}
set CONFIG(ANNIVERSARY_WARNINGS_DELAY)  \{$CONFIG(ANNIVERSARY_WARNINGS_DELAY)\}

#----------------------------------------------------------------------#
# End of file
#----------------------------------------------------------------------#
"
    }
    close $dbf
    Message "Options saved"
}

proc HyperSetTag {widget tagprefix regexp action color message
		  {extrabinding {}}} {

    global CONFIG
    
    set index 1.0
    set count 0
    while { [set index [$widget search -nocase -regexp -count long -- \
			    $regexp $index end]] != "" } {
	set tag $tagprefix$count
	set end "$index + $long chars"
	set match [$widget get $index $end]
	$widget tag add $tag $index $end
	$widget tag configure $tag -borderwidth 2 -background $color -relief raised
	# $widget tag configure $tag -borderwidth 2 -relief raised
	if {[winfo depth .] > 1} {
	    set normal "-background $color -relief raised -borderwidth 1"
  	    set bold "-background snow"
	} else {
	    set normal "-foreground white -background black"
	    set bold "-foreground {} -background {}"
	}
	
	$widget tag bind $tag <Button-1> "$action [list $match]"
	$widget tag bind $tag <Enter>    "%W tag configure $tag $bold
	    Message {$message}"
	$widget tag bind $tag <Leave>    "%W tag configure $tag $normal
	    Message {}"
	if { $extrabinding != {} } {
	    $widget tag bind $tag <Button-3> \
		"$extrabinding [list $match]; break"
	}
	incr count 1
	set index [$widget index $tag.last]
    }
}

proc InvokeFtp {ftp} {

    global CONFIG INTERNAL

    if { $ftp == "" } { return }

    switch -- $CONFIG(FTP_COMMAND) {
	"netscape" {
	    if [catch {eval {exec netscape} \
			   {-remote openURL($ftp)} >& /dev/null } tmp ] {
		tk_dialog .delete {Error Massage} \
		    "$INTERNAL(NAME): Starting netscape. Please Wait. " \
		    info 0  "Ok"
		if [catch {
                    eval {exec netscape} { $ftp & }
                    Message "Starting netscape"
                } err] {
                    tk_dialog .delete {Error Massage} \
			"netscape: \n\n$err" \
		    info 0  "Ok"
                }
	    }
	}
	default {
	    # rm last / and the ftp keyword if exists
	    regsub -all "/" $ftp "" ftp
	    regsub -all "ftp:" $ftp "" ftp
	    # (find-file "/anonymous@ftp.x.org:
	    if [regsub -all {FTP} $CONFIG(FTP_COMMAND) "$ftp" cmd] {
		if [catch { eval {exec /usr/bin/sh} {-c} {$cmd &} } tmp] {
		    tk_dialog .delete {Error Massage} \
			"$INTERNAL(NAME): $cmd" \
			info 0  "Ok"
		}
	    } else {
		tk_dialog .delete {Error Massage} \
		    "$INTERNAL(NAME): command not found $cmd" \
		    info 0  "Ok"
	    }
	}
    }
}

proc InvokeBrowser {url} {
    
    global CONFIG INTERNAL
    
    if { $url == "" } { return }
    switch -- $CONFIG(BROWSER_COMMAND) {
	"netscape" {
	    if [catch {eval {exec netscape} \
			   {-remote openURL($url)} >& /dev/null } tmp ] {
		tk_dialog .delete {Error Massage} \
		    "$INTERNAL(NAME): Starting netscape. Please Wait. " \
		    info 0  "Ok"
		if [catch {
                    eval {exec netscape} { $url & }
                    Message "Starting netscape"
                } err] {
		    tk_dialog .delete {Error Massage} \
			"netscape: \n\n$err" \
			info 0  "Ok"
                }
	    }
	}
	default {
	    if [regsub -all {URL} $CONFIG(BROWSER_COMMAND) "$url" cmd] {
	
		if [catch { eval {exec /usr/bin/sh} {-c} {$cmd &} } tmp] {
		    tk_dialog .delete {Error Massage} \
			"$INTERNAL(NAME): $cmd" \
			info 0  "Ok"
		}
	    } else {
		tk_dialog .delete {Error Massage} \
		    "$INTERNAL(NAME): command not found $cmd" \
		    info 0  "Ok"
	    }
	}
    }
}
proc InvokeEditor {path} {
    
    global CONFIG INTERNAL
    
    if { $path == "" } { return }

    regsub -all "file:" $path "" file
    set file [glob -nocomplain $file] ;# expands ~/...

    set path "file:$file"
    switch -- $CONFIG(EDITOR_COMMAND) {
	"netscape" {
	    if [catch {eval {exec netscape} \
			   {-remote openURL($path)} >& /dev/null } tmp ] {
		tk_dialog .delete {Error Massage} \
		    "$INTERNAL(NAME): Starting netscape. Please Wait. " \
		    info 0  "Ok"
		if [catch {
		    eval {exec netscape} { $path & }
		    Message "Starting netscape"
		} err] {
		    tk_dialog .delete {Error Massage} \
			"netscape: \n\n$err" \
			info 0  "Ok"
		}
	    }
	}
	default {
	    # xterm -e less -i
	    if [regsub -all {FILE} $CONFIG(EDITOR_COMMAND) "$file" cmd] {
		if [catch { eval {exec /usr/bin/sh} {-c} {$cmd &} } tmp] {
		    tk_dialog .delete {Error Massage} \
			"$INTERNAL(NAME): $cmd" \
			info 0  "Ok"
		}
	    } else {
		tk_dialog .delete {Error Massage} \
		    "$INTERNAL(NAME): command not found $cmd" \
		    info 0  "Ok"
	    }
	}
    }
}

proc InvokePhone {number} {
    
    global CONFIG INTERNAL
    
    if { $number == "" } { return }

    # remove white spaces in the phone number
    set number [join $number ""]
    regsub -all {\.} $number "" number ;# remove dots in 92.92.92.92
    if [regsub -all {PHONE} $CONFIG(PHONE_COMMAND) "$number" cmd] {
	if [catch { eval {exec /usr/bin/sh} {-c} {$cmd &} } tmp] {
	    tk_dialog .delete {Error Massage} \
		"$INTERNAL(NAME): $cmd" \
		info 0  "Ok"
	}
    } else {
	tk_dialog .delete {Error Massage} \
	    "$INTERNAL(NAME): command not found $cmd" \
	    info 0  "Ok"
    }
}

proc InvokeFinger {email} {
    
    global CONFIG INTERNAL

    if { $email == "" } { return }

    set result [catch {eval exec $INTERNAL(FINGERCOMMAND) {$email}} err]
    regsub -all "\r" $err "" err ;# remove \r
    HelpMessage "Finger at $email" $err {Finger result}
}

proc InvokeMail {email} {
    
    global CONFIG INTERNAL
	
    switch -- $CONFIG(MAIL_COMMAND) {
	"netscape" {
	    if [catch {eval {exec netscape} \
			   {-remote mailto($email)} >& /dev/null } tmp] {
		tk_dialog .delete {Confirm Action} \
		    "$INTERNAL(NAME): Starting netscape. Retry when netscape is running." \
		    info 0  "Ok"
		if [catch {eval {exec netscape} {&} 
		    Message "Starting netscape"
		    } err] {
		    tk_dialog .delete {Error Massage} \
			"netscape:: $err" \
			error 0  "Ok"
		}
	    }
	}
	"SimpleMail" {
	    SimpleMail $email
	}
	"" { ;# default to simplemail
	     SimpleMail $email
	}
	default {
	    # check if EMAIL is on the command string for 
	    # gnudoit compatibility
	    if [regsub -all {EMAIL} $CONFIG(MAIL_COMMAND) "$email" cmd] {
		if [catch { eval {exec /usr/bin/sh} {-c} {$cmd &} } tmp] {
		    tk_dialog .delete {Error Massage} \
			"$INTERNAL(NAME): $cmd" \
			info 0  "Ok"
		}
	    } else {
		tk_dialog .delete {Error Massage} \
		    "$INTERNAL(NAME): command not found $cmd" \
		    info 0  "Ok"
	    } 
	}
    }
}
    
proc ParseSearch {w string} {

    $w tag remove search 0.0 end
    if {$string == ""} {
	return
    }
    set cur 1.0
    while 1 {
	set cur [$w search -nocase -count length $string $cur end]
	if {$cur == ""} { break }
	$w tag add search $cur "$cur + $length char"
	set cur [$w index "$cur + $length char"]
    }
    $w tag configure search -background red -foreground black
}
proc WarnAnniversary {date} {
    global INTERNAL
    
    set this_day [exec date +%d]
    set this_month [exec date +%m]

    regexp {([0-9][0-9])/([0-9][0-9])/.*} $date match day month
    
    regsub -all $INTERNAL(COMPANY_MARK) \
	[GetLabelFromKey $INTERNAL(CURRENT_KEY)] "" name
    
    if { $day == $this_day && $month == $this_month } {
	tk_dialog .delete {} \
		"Yes, today it's the anniversary date of $name!." \
	    info 0 "Ok"
    }
}

proc WarnTodayAnniversaries {} {
    global CONFIG INTERNAL DATABASE
    
    set date [exec date $CONFIG(DATE_FORMAT)]
    # nuke the year
    regsub {[0-9][0-9]$} $date "" date
    # process each matching key
    set match [SearchMatch "global" $date]
    set howmuch [llength $match]
    
    if { $howmuch != 0 } {
	foreach key $match {
	    set record $DATABASE($key)
	    set name [lindex $DATABASE($key) 0]
	    # warn only for people not for companies!
	    if { $name != "" } {
		tk_dialog .delete \
		    "$INTERNAL(NAME): $name Anniversary!!!!" \
		    "Today is $name's anniversary!" \
		    info 0 Ok
	    }
	}
    }
    # reschedule the next warnings
    after $CONFIG(ANNIVERSARY_WARNINGS_DELAY) WarnTodayAnniversaries
}

proc HyperParse {widget} {
    global CONFIG INTERNAL
    
    # dates
    HyperSetTag $widget "anniversary" $INTERNAL(ANNIVERSARY_REGEXP) \
	WarnAnniversary $CONFIG(ANNIVERSARY_HIGHLIGHT_COLOR) "Anniversary"

    # browser
    if { $CONFIG(BROWSER_COMMAND) != "" } {
	# http addresses
	HyperSetTag $widget "http"  $INTERNAL(HTTP_REGEXP) InvokeBrowser \
	    $CONFIG(BROWSER_HIGHLIGHT_COLOR) \
	    "Mouse Button-1 or Button-3 browse URL" \
	    InvokeBrowser
    }

    # ftp addresses
    if { $CONFIG(FTP_COMMAND) != "" } {
	HyperSetTag $widget "ftp" $INTERNAL(FTP_REGEXP) InvokeFtp \
	    $CONFIG(FTP_HIGHLIGHT_COLOR) \
	    "Mouse Button-1 or Button-3 browse FTP" \
	    InvokeFtp
    }

    # files
    if { $CONFIG(EDITOR_COMMAND) != "" } {
	HyperSetTag \
	    $widget "file" $INTERNAL(FILE_REGEXP) InvokeEditor \
	    $CONFIG(FILE_HIGHLIGHT_COLOR) \
	    "Mouse Button-1 or Button-3 edit FILE" \
	    InvokeEditor
    }
    
    # phone
    if {$CONFIG(PHONE_COMMAND) != "" } {
	HyperSetTag $widget "phone1" \
	    $INTERNAL(PHONE_REGEXP1) \
	    InvokePhone $CONFIG(PHONE_HIGHLIGHT_COLOR) \
	    "Mouse Button-1 or Button-3 dial NUMBER" \
	    InvokePhone
	HyperSetTag $widget "phone2" \
	    $INTERNAL(PHONE_REGEXP2) \
	    InvokePhone $CONFIG(PHONE_HIGHLIGHT_COLOR) \
	    "Mouse Button-1 or Button-3 dial NUMBER" \
	    InvokePhone
	HyperSetTag $widget "phone3" \
	    $INTERNAL(PHONE_REGEXP3) \
	    InvokePhone $CONFIG(PHONE_HIGHLIGHT_COLOR) \
	    "Mouse Button-1 or Button-3 dial NUMBER" \
	    InvokePhone
    }
    
    # EMAIL
    # add a binding for finger
    if {$CONFIG(MAIL_COMMAND) != "" } {
	HyperSetTag $widget "email" $INTERNAL(EMAIL_REGEXP) \
	    InvokeMail $CONFIG(MAIL_HIGHLIGHT_COLOR) \
	    "Mouse Button-1 compose MAIL / Button-3 finger ADDRESS" \
	    InvokeFinger
    }
}
    
proc Backup {} {
    global INTERNAL

    BackupFile $INTERNAL(DBPRIVATE_NAME)
    BackupFile $INTERNAL(DBPUBLIC_NAME)
    Message "Backup done"
}

proc BackupFile {file} {
    global CONFIG

    set file [glob -nocomplain $file]
    set level [expr $CONFIG(BACKUP_LEVEL) - 1]

    for { set i 1} { $i <= $level } {incr i 1} {
	set i1 [expr $i - 1]
	if [file exist $file.bck_$i] {
	    if [catch {eval {exec cp} $file.bck_$i $file.bck_$i1} err] {
		tk_dialog .delete {Warning} "$err"  error 0  "Ok"
	    }
	}
    }
    if [file exist $file] {
	if [catch {eval {exec cp} $file $file.bck_$level} err] {
	    tk_dialog .delete {Warning} "$err"  error 0  "Ok"
	}
    }
}

proc SpyNewUser {} {

    global INTERNAL env

    if {$INTERNAL(SPYNEWUSER)} {

	# the following code is to know who use carnet
	# it is assumed is the first time carnet is used
	set user ""
	if [info exists env(USER)] {
	    set user $env(USER)
	} elseif [info exists env(LOGNAME)] {
	    set user $env(LOGNAME)
	}
	if { $user != "" } {
	    if [catch {eval {exec echo "To: $INTERNAL(MAINTAINER_MAIL)
Subject: carnet : new user $user
----------------
new user $user
." | $INTERNAL(MAILCOMMAND) $INTERNAL(MAINTAINER_MAIL)}} err] {
		tk_dialog .delete {Warning} \
		    "The mail command appear not to work: 

$err

Please ask to your local system administrator how to fix this, or you will be
enable to mail from $INTERNAL(NAME).
" \
		    error 0  "Ok"
	    }
	}
    }
}


proc ShowDemo {} {
    global RECORD
    
    if [tk_dialog .delete {demo} \
	    {This is a simple demo: it shows how to insert and delete a record, with hypertext bindings.} \
	    info 0  "Continue"  "Abort"] { return }
    ClearSearchList
    ClearRecord
    if [tk_dialog .delete {demo} \
	    {To insert a record you have to set at least the name and/or the company of the person.} \
	    info 0  "Continue"  "Abort"] { return }
    $RECORD(NAME) insert insert "Albert Einstein"
    $RECORD(COMPANY) insert insert "A demo & co."
    set key [GetCurrentKey]
    $RECORD(EMAIL) insert insert "a valid email ae@exemple.fr"
    $RECORD(PHONE) insert insert "
Valid phone numbers:

<> ONE digit numbers: +33 16 1 123456

<> TWO digits numbers: (at least 6 numbers)
12 12 12 or 12.12.12 or 12-12-12
or
12 12 123 or 12.12.123 or 12-12-123
or
12 12 1234 or 12.12.1234 or 12-12-1234
and so on.

<> THREE digits numbers: (at least 6 numbers)
123 123 or 123.123 or 123-123
or 
123 123 123 or 123.123.123 or 123-123-123
or
123 123 1234 or 123.123.1234 or 123-123-1234
and so on.

Any of the above forms should be prefixed by a valid prefix of the form:

(country prefix) (region prefix) number

as in the above examples:
+33 16 1 123 123 or 16 1 12-12-12 or 369 333-4444
"
    $RECORD(ADDRESS) insert insert "the address
1, Champs Elysees
75 001 Paris
France"
    $RECORD(NOTES) insert insert "http://www.inria.fr 
ftp://ftp.inria.fr prefixed by ftp://
file:~/.Xdefaults prefixed by file:"
    tk_dialog .delete {demo} \
	{Use "update" in the Edit menu to update the database.} \
	info 0  "Ok"
    Update
    tk_dialog .delete {demo} \
	{Now that the record is inserted in the database, hypertext have been added for email ftp and http addresses, and for phone numbers. Using the mouse button-1 you can now send a mail or browse http and ftp addresses and files, or call a phone number. Actions are defined in the "Option Preferences menu".} \
	info 0  "Continue"
    tk_dialog .delete {demo} \
	{To delete this record use "delete" in the Edit menu. The demo is finished.} \
	info 0  "Ok"
}

proc LoadDb {what file} {
    global INTERNAL DATABASE

    if [file exists $file] {
	if [catch {open $file r} dbf] {
	    Message "$INTERNAL(NAME): Couldn't open $file."
	    exit
	} else {
	    set db [read $dbf]
	    set all [split $db $INTERNAL(KEND)]
	    set title [lindex $all 0]
	    # check for valid database file
	    if {$title == $INTERNAL(HEADER)} {
		set records [lrange $all 1 end] 
		set mess ""
		foreach r $records {
		    set  v [split $r $INTERNAL(KSEP)]
		    set key [GetKey [lindex $v 0] [lindex $v 1]]
		    if [info exist DATABASE($key)] {
			append mess \
			    "[lindex $v 0] -- [lindex $v 1] already exist not used\n"
		    } else {
			# do not override existing records
			set DATABASE($key) "$v $what"
			# add here code to print in another format
			# with eventually an exit.
			# puts "$DATABASE($key) \n"
		    }
		}
		close $dbf
		catch {unset DATABASE()} ;# forget the last empty
	    } else {
		set mess "$INTERNAL(NAME): $file is not a valid database."
	    }
	}
    } else {
	set mess "$INTERNAL(NAME): No $file found."
    }
    return $mess
}
proc SearchMatch {how pattern} {
    global DATABASE CONFIG INTERNAL

    switch -- $how {
	regexp {
	    # regexp search
	    regsub -all {\[} $pattern {\\\[} pattern
	    regsub -all {\]} $pattern {\\\]} pattern
	    set pattern "*[join $pattern *]*"
	    set pattern [string tolower $pattern]
	    set match [array names DATABASE $pattern]
	}
	exact {
	    # exact search
	    regsub -all {\[} $pattern {\\\[} pattern
	    regsub -all {\]} $pattern {\\\]} pattern
	    set pattern "[join $pattern ""]*"
	    set pattern [string tolower $pattern]
	    set match [array names DATABASE $pattern]
	}
	global {
	    # global search
	    if {$CONFIG(SEARCH_CASE)} { 
		set command {regexp "$pattern" $text}
	    } else { set command {regexp -nocase "$pattern" $text} }
	    set match {}
	    foreach key [array names DATABASE] {
		set record $DATABASE($key)
		set i 0
		while { $i <= 6 } { ;# six fields in a record (to match external, public...)
		    set text [lindex $record $i]
		    catch {eval $command} res
		    if {$res == 1} { 
			lappend match $key
			set i 6
		    } elseif {$res != 0} { 
			Message $res
			return {}
		    }
		    incr i 1
		}
	    }
	}
    }
    # filter match to display a requested database
    if { $INTERNAL(DATABASE) != "all"} {
	set rmatch {}
	foreach key $match {
	    if {[lindex $DATABASE($key) 6] == $INTERNAL(DATABASE)} {
		lappend rmatch $key
	    }
	}
    } else {
	set rmatch $match
    }
    return $rmatch
}

proc Message {message} {
    global INTERNAL CONFIG
    if [info exist INTERNAL(MESSAGE)] {
	set w [expr ($CONFIG(ENTRY_WIDTH) + $CONFIG(LIST_WIDTH))]
	set message [string range $message 0 $w]
	set INTERNAL(MESSAGE) $message
    } else {
	puts $message
    }
}



proc TermDoIt {} {
    global argc argv helpTopics INTERNAL

    if {$argc > 0} {
	set argv0 [lindex $argv 0]
	set argv1 ""
	if {$argc > 1} { set argv1 [lindex $argv 1] }
	switch -- $argv0 {
	    "-f" {
		LoadDb external $argv1
		eval {proc Save {{force 0}} {
		    global INTERNAL
		    set INTERNAL(SAVE) 0
		    tk_dialog .delete {Warning} \
			"You couldn't save when $INTERNAL(NAME) is invoked with the -f option." \
			error 0  "Ok"
		}}
	    }
	    "-Help" { puts $helpTopics(command_line) 
		exit
	    }
	    "-global" { TermSearch "global" $argv1 }
	    "-exact" { TermSearch "exact" $argv1 }
	    "-regexp" { TermSearch "regexp" $argv1 }
	    default { TermSearch "global" $argv0 }
	}
    }
    LoadDb public $INTERNAL(DBPUBLIC_NAME)
    LoadDb private $INTERNAL(DBPRIVATE_NAME)
}

proc TermSearch {how pattern} {
    global INTERNAL DATABASE

    set mess [LoadDb public $INTERNAL(DBPUBLIC_NAME)]
    if {$mess != ""} { puts $mess }
    set mess [LoadDb private $INTERNAL(DBPRIVATE_NAME)]
    if {$mess != ""} { puts $mess }
    set rmatch [SearchMatch $how $pattern]
    set howmuch [llength $rmatch]
    switch -- $howmuch {
	0 { Message "No match" }
	1 { set match [lindex $rmatch 0]
	    puts "----------------------------------"
	    for {set i 0} {$i < 6} {incr i 1} { puts [lindex $DATABASE($rmatch) $i] }
	    puts "----------------------------------"
	}
	default { 
	    puts "----------------------------------"
	    foreach key [lsort $rmatch] { 
		for {set i 0} {$i < 6} {incr i 1} { puts [lindex $DATABASE($key) $i] }
		puts "----------------------------------"
	    }
	}
    }
    exit
}
