# --- bracket.tcl --- A small text editor that keeps track of nesting levels

# Copyright(C) I.B.Findleton, 1998. All rights reserved.

package require dde 2.0

global Dirty types

set Dirty 0

# A bunch of file extensions that I personnally use a lot...

set types {
			{{Command Files}	{.cmd}	}
			{{Tcl/Tk Files}	{.tcl}	}
			{{C++ Source Files} {.cpp}	}
			{{C Source Files}	{.c}	  	}
			{{C/C++ Headers}	{.h}		}
			{{Module Defs}		{.def} 	}
			{{Resource Files}	{.rc}		}
			{{Resource Headers}	{.rh}	}
			{{Text Files}		{.txt}	}
			{{All Files}		{*.*}		}
			}

# Wrapper for the scrollbar commands

proc YView { w op val { unit "" }} {

	if { $op == "moveto" } {
		eval { $w.text yview $op $val }
		eval { $w.status yview $op $val }
	} else {
		eval { $w.text yview $op $val $unit }
		eval { $w.status yview $op $val $unit }
		}

	}

# Get the root of the editor windows

proc GetRoot { w { branch {}}} {

	set root [split $w .]
	set len [expr [llength $root] - 1]

	if { $branch == "" } {
		return [join [lreplace $root $len $len] .]
	} else {
		return [join [lreplace $root $len $len $branch] .]
		}
	}

# Set cursor position

proc SetCursorPosition { w pos } {

	set status [GetRoot $w status]

	$w mark set insert $pos
	$w see insert
	$w tag remove sel 1.0 end

	$status mark set insert [LineIndex [GetLineNumber $w insert]]
	$status see insert
	}

# Make a line index

proc MakeLineIndex { r c } {

	return [format "%d.%d" $r $c]
	}

# Get the current number of display lines for the editor

proc EditorLines { w } {

	return [lindex [$w config -height] 4]
	}

# Get the line number for the start of a page

proc PageStart { w { page 0 }} {

	return [MakeLineIndex [expr ([EditorLines $w] * $page) + 1] \
					[GetColumnNumber $w insert]]
	}

# Get the index of the bottom of the page

proc PageBottom { w page } {

	return "[PageStart $w $page] + 19 lines"
	}

# Get the current page number ( 0 to n )

proc PageNumber { w } {

	return [expr [GetLineNumber $w insert] / [EditorLines $w]]
	}

# Get help information about the current selection

proc DoHelp { w } {

	puts stdout "Help"
	if { ![catch {set topic [$w get sel.first sel.last]}] } {
		puts stdout $topic
		help $topic
		}
	}

# Setup the bindings for the editor window

proc SetupEditorBindings { w } {

	bind $w <KeyPress> { if [EditText %W %A %K] { break } }

	bind $w <Left> {
		if [%W compare insert == 1.0 ] { break }
		SetCursorPosition %W insert-1c
		break
		}

	bind $w <Right> {
		if [%W compare insert == end] { break }
		SetCursorPosition %W insert+1c
		break
		}

	bind $w <Up> {
		if { [GetLineNumber %W insert] } {
			SetCursorPosition %W [MakeLineIndex [expr [GetLineNumber %W insert] - 1] \
						[GetColumnNumber %W insert]]
			break
		} else { break }
		}

	bind $w <Down> {
		if { [GetLineNumber %W insert] < [GetLineNumber %W end] } {
			SetCursorPosition %W [MakeLineIndex [expr [GetLineNumber %W insert] + 1] \
						 [GetColumnNumber %W insert]]
			break
		} else { break }
		}

	bind $w <Prior> {
		if { [PageNumber %W] == 0 } {
			SetCursorPosition %W [MakeLineIndex 0 [GetColumnNumber %W insert]]
			break
		} else {
			SetCursorPosition %W [PageStart %W [expr [PageNumber %W] - 1]]
			break
			}
		}

	bind $w <Next> {
		if { [expr [GetLineNumber %W end] - [GetLineNumber %W insert]] < \
					[EditorLines %W] } {
			SetCursorPosition %W end
			break
		} else {
			SetCursorPosition %W [PageBottom %W [expr [PageNumber %W] + 1]]
			break
			}

		bind $w <F1> {
			DoHelp %W
			break
			}
		}

# Set up some tags to make the output look pretty

	$w tag config Green -foreground green
	$w tag config Yellow -foreground orange
	$w tag configure Blue -foreground blue
	}

# Add a new line to a the status text

proc AddLine { w after } {

	set w $w.status

	$w mark set insert [format "%d.end" $after]
	$w config -state normal
	$w insert insert "\n"
	$w config -state disabled
	}

# Delete a line from the status window

proc DeleteLine { w at } {

	set w $w.status

	puts stdout "DeleteLine $at"

	$w mark set insert [format "%d.0" $at]
	$w config -state normal
	$w delete insert "insert lineend"
   $w delete insert
	$w config -state disabled

	}

# Display some non printing characters

proc DisplayChars { c } {

	set len [string length $c]
	set idx 0
	set s "EMPTY"

	while { $idx < $len } {
		set n [string index $c $idx]
		switch -exact -- $n {
		"\r"	{ set s "CR" }
		"\n"	{ set s "LF" }
		"\t"	{ set s "TAB" }
		"\b"	{ set s "BS" }
		default { set s $n }
			}
		incr idx
		}

	puts -nonewline stdout "$s "
	}

# Handle any changes to the text

proc EditText { w c k } {

	global Dirty

	set result 0
	switch -- $c {

		"\b" {  	set nc [$w get "insert" "insert+1c"]
					DisplayChars $nc
					if { $nc == "\n" } {
						DeleteLine [GetRoot $w] [GetLineNumber $w "insert" ]
						}
					$w delete "insert-1c" "insert"
					UpdateBracketStatus $w [GetLineNumber $w insert] 0
					set result 1
					}
		"\n" { 	UpdateBracketStatus $w [GetLineNumber $w insert] 1 }
		"\r" { 	UpdateBracketStatus $w [GetLineNumber $w "insert -1 line"] 1
					}
		"\t" { }
		"{"	-
		"\["	-
		"("	-
		"<"	-
		"}"	-
		"]"	-
		")"	-
		">"  { 	$w insert insert $c Yellow
					UpdateBracketStatus $w [GetLineNumber $w insert] 1
					set result 1
					}

	default {	$w insert insert $c Yellow
					set result 1
					}
		}

	if { $Dirty == 0 } {
		set Dirty 1
		SetButtonState
		}
	return $result
	}

# Count brackets

proc CountBrackets { cur line } {

	global OpenBrackets CloseBrackets

	set len [string length $line]
	set idx 0

	while { $idx < $len } {
		switch -exact -- [string index $line $idx] {

			"(" 	-
			"{"   -
			"<"   -
			"\[" 	{ incr cur }
			"}"   -
			"\]"   -
			")"   -
			">"	{ set cur [expr $cur - 1] }
				}
		incr idx
		}

	return $cur
	}

# Make the index of the start of a line

proc LineIndex { line } {

	return [format "%d.0" $line]
	}

# Get the line index from an index

proc GetLineNumber { w loc } {

	return [lindex [split [[GetRoot $w text] index $loc] .] 0]
	}

# Get the column number for an index

proc GetColumnNumber { w loc } {

	return [lindex [split [[GetRoot $w text] index $loc] .] 1]
	}

# Mark the status of a line based on the nesting level

proc MarkLineStatus { w nesting loc { lf 1 }} {

	set w $w.status

	$w config -state normal

	if { $nesting } {
		if { $nesting < 0 } {
			$w insert $loc "$nesting" Red
		} else {
			$w insert $loc "$nesting"
			}
	} else {
		$w insert $loc "  "
		}
	if { $lf } {
		$w insert insert "\n"
		}
	$w config -state disabled
	}

# Get the nesting level of a line

proc GetNestingLevel { w line } {

	set loc [LineIndex $line]
	set cur [$w.status get $loc "$loc lineend"]

	puts stdout "Loc $loc Cur \"$cur\""

	set level 0
	scan $cur "%d" level

	return $level
}

# Get a line of data

proc GetLine { w line } {

	set cur [LineIndex $line]
	return [$w.text get $cur "$cur lineend"]
	}

# Clear the current status

proc ClearLineStatus { w loc } {

	global Editor

	$w.status config -state normal
	$w.status delete $loc "$loc lineend"
	$w.status config -state disabled

	}

# Update the bracket count

proc UpdateBracketStatus { w line lf } {

	set loc [LineIndex $line]
	if { $line } {
		set cur [GetNestingLevel [GetRoot $w] [expr $line - 1]]
	} else {
		set cur 0
		}
	set data [GetLine [GetRoot $w] $line]
	set cur [CountBrackets $cur $data]

	puts stdout "Line \"$data\" Cur $cur"

	ClearLineStatus [GetRoot $w] $loc
	MarkLineStatus [GetRoot $w] $cur $loc $lf
	}

# Clean up a line

proc CleanLine { line } {

	set len [string length $line]
	set idx 0
	set result ""

	while { $idx < $len } {
			set next [string index $line $idx]
				switch -- $next {

		"\t" {	set cur [string length $result]
					set num [expr $cur % 5]
					if { $num == 0 } {
						set num 5
						}
					while { $num } {
						append result " "
						set num [expr $num - 1]
						}
					}
		default { append result $next }
				}
			incr idx
			}
		return $result
	}

# Create the editor windows

proc BracketEditor { w args } {

	set Editor [NewFrame $w]

# The bracket count window

	text $Editor.status -height 20 -width 2 -bg gray80 -font "Fixedsys 12" -state disabled

	$Editor.status tag config Red -foreground red
	$Editor.status tag config Green -foreground green
	$Editor.status tag config Blue -foreground blue

# The text display window

	eval { text $Editor.text -height 20 -width 80 -yscrollcommand "$Editor.sy set" -bg black -fg white \
			-font "Fixedsys 12" -insertbackground white } $args

	SetupEditorBindings $Editor.text

# A vertical scrollbar to handle both windows

	scrollbar $Editor.sy -orient vertical -command "YView $Editor"
	pack $Editor.sy -side right -fill y
	pack $Editor.status -side left -fill y
	pack $Editor.text -side left -fill both -expand true

	return $Editor
	}

# Clear the editor text

proc ClearEditor { w } {

	$w.text delete 1.0 end
	$w.status config -state normal
	$w.status delete 1.0 end
	$w.status config -state disabled
	}

# Display text in the editor

proc DisplayInEditor { w line nesting } {

	$w.status config -state normal

	set len [string length $line]
	if { [string index $line 0] != "#" } {
		if { $len > 0 } {
			set loc [lindex [split [$w.text index end] .] 0]
			$w.text insert end [format "%s\n" $line]
			set nesting [CountBrackets $nesting $line]
			MarkLineStatus $w $nesting end
		} else {
			$w.text insert end "\n"
			$w.status insert end " \n"
			}
	} else {
		$w.text insert end [format "%s\n" $line] Green
		$w.status insert end " \n"
		}

	$w.status config -state disabled
	return $nesting
	}

# Get a line from the editor. Gets the line with the end of line character!

proc GetALine { w cur } {

	return [[GetRoot $w text] get [LineIndex $cur] [LineIndex [expr $cur + 1]]]
	}

# Create a frame

proc NewFrame { w { width 2 } { relief flat }} {

	return [frame $w -relief $relief -borderwidth $width]
	}

# View the file in the variable FileName

proc ViewFile {w name } {

	global Dirty

	if { $name == "" } {
		return 0
		}

	ClearEditor $w

	set fd [open $name "r"]

	set nesting 0
	while { [eof $fd] == 0 } {
		gets $fd line
		set nesting [DisplayInEditor $w [CleanLine [string trimright $line]] $nesting]
		}

	close $fd

	set Dirty 0
	}

# Get a file name

proc GetOpenFileName { t { ext .cmd } } {

	global types

	return [tk_getOpenFile -filetypes $types -defaultextension $ext \
									-title $t]
	}

# Get a save file name

proc GetSaveFileName { t { ext .cmd } } {

	global types

	return [tk_getSaveFile -filetypes $types -defaultextension $ext \
									-title $t ]
	}


