# GUI for testing the features of the DDE extension

load dde

set version [package require dde 2.0]
if  { $version != "2.0" } {
	puts stdout "This script requires version 2.0 of the DDE package"
	return -code error
	}

# Establish a base window for the GUI interface

catch [destroy $f0 $f1 $f2 $f3 $f4 $f5 .gui]

set base [toplevel .gui]
global base ResultWindow Status Help

# Set the default characteristics of a conversation

set conv(service) Tcl
set conv(topic) System
set conv(id) -1
set conv(item) ""
set conv(command) ""

set Status ""

# Initialize vars

set NoConvCount 0
set ConvCount 0

# Get the current list of servers

set ServiceList [dde services]
set TopicList ""
set ItemList ""

global conv NeedsConv NeedsNoConv NoConvCount ConvCount
global ServiceList TopicList ItemList

# Make a frame with some default characteristics

proc NewFrame { w } {

	return [frame $w -relief ridge -borderwidth 2]

	}

# Create a button that works when a conversation is available

proc ConvFunc { w args } {

	global ConvCount NeedsConv

	set NeedsConv($ConvCount) [eval { button $w } $args]
	incr ConvCount
	return $w
	}

# Create a button the is active when there is no available conversation

proc NoConvFunc { w args } {

	global NeedsNoConv NoConvCount

	set NeedsNoConv($NoConvCount) [eval { button $w } $args]
	incr NoConvCount
	return $w
	}

# Select from a list

proc ListSelect { w item } {

	global conv

	set conv($item) [$w get active]

	}

# Choose a new service

proc NewService {} {

	global ServiceList conv base

	Disconnect

	set ServiceList [dde services]

	catch [destroy $base.prompt]
	set f [toplevel $base.prompt]

	wm title $f "Choose a Service"

	set f0 [NewFrame $f.choice]
	listbox $f0.list -height [llength $ServiceList] -bg white

	set f1 [NewFrame $f.actions]
	button $f1.ok -text Ok -width 8 -command [list destroy $f]
	button $f1.cancel -text Cancel -width 8 -command [list destroy $f]
	pack $f1.ok $f1.cancel -side left

	pack $f0.list -fill both
	pack $f0 $f1 -side top

	bind $f0.list <ButtonPress-1> \
		{ ListSelect %W service }
	bind $f0.list <Double-ButtonPress-1> \
		"ListSelect %W service ; destroy $f"
	eval { $f0.list insert end } $ServiceList

	tkwait variable conv(service)
	}

# Choose a new topic for the current service

proc NewTopic {} {

	global TopicList conv base

	Disconnect

	set TopicList [dde topics $conv(service)]

	catch [destroy $base.prompt]
	set f [toplevel $base.prompt]

	wm title $f "Choose a Topic"
	listbox $f.list -height [llength $TopicList] -bg white -font "Fixedsys 12"
	bind $f.list <ButtonPress-1> \
		{ ListSelect %W topic }
	bind $f.list <Double-ButtonPress-1> \
		"ListSelect %W topic ; destroy $f"
	eval { $f.list insert end } $TopicList
	set f1 [frame $f.actions -relief ridge -borderwidth 2]
	button $f1.ok -text Ok -width 8 -command "destroy $f"
	button $f1.cancel -text Cancel -width 8 -command "destroy $f"

	pack $f1.ok $f1.cancel -side left

	pack $f.list
	pack $f1

	tkwait variable conv(topic)
	}

# Choose a new item for the current service and topic

proc NewItem {} {

	global ItemList conv base

	set ItemList [dde request $conv(id) "topicitemlist"]

# Because the Tcl service is a broadcast service, the result is a list of sublists,
# but since we only want one of the sub-lists, get the first one...

	if { $conv(service) == "Tcl" || $conv(service) == "tcl" } {
		set ItemList [lindex $ItemList 0]
		}

	catch [destroy $base.prompt]
	set f [toplevel $base.prompt]

	wm title $f "Choose an item"

	set len [llength $ItemList]
	if  { $len > 10 } { set len 10 }
	set f0 [frame $f.lb]
	listbox $f0.list -height $len -yscrollcommand "$f0.sy set" -bg white \
			-font "Fixedsys 12"
	scrollbar $f0.sy -orient vert -command "$f0.list yview"
	eval { $f0.list insert end } $ItemList
	bind $f0.list <ButtonPress-1> \
		{ ListSelect %W item }
	bind $f0.list <Double-ButtonPress-1> \
		 "ListSelect %W item ; destroy $f "

	pack $f0.sy -side right -fill y
	pack $f0.list -side left -fill y

	set f1 [frame $f.actions -relief ridge -borderwidth 2]
	button $f1.ok -text Ok -width 8 -command "destroy $f"
	button $f1.cancel -text Cancel -width 8 -command "destroy $f"
	pack $f1.ok $f1.cancel -side left

	pack $f0
	pack $f1

	tkwait variable conv(item)
	}

# Set the state of the buttons

proc SetButtonState {} {

	global conv NeedsConv NeedsNoConv ConvCount NoConvCount

	if { $conv(id) != -1 } {
		set state1 normal
		set state2 disabled
	} else {
		set state1 disabled
		set state2 normal
		}

	set count 0
	while { $count < $ConvCount } {
		$NeedsConv($count) config -state $state1
		incr count
		}
	set count 0
	while { $count < $NoConvCount } {
		$NeedsNoConv($count) config -state $state2
		incr count
		}
	}

# Display a result in the text window

proc DisplayResult { result } {

	global ResultWindow

	$ResultWindow config -state normal
	$ResultWindow insert end $result
	$ResultWindow insert end "\n"
	$ResultWindow see end
	$ResultWindow config -state disabled

	}

# Conncet to the current service and topic

proc Connect {} {

	global conv

	set conv(id) [dde connect $conv(service) $conv(topic)]

	if { $conv(id) != -1 } {
		SetButtonState
	} else {
		DisplayResult "Connection to \"$conv(service)\" about \"$conv(topic)\" refused!"
		}
	}

# Disconnect from the current service and topic

proc Disconnect {} {

	global conv

	if { $conv(id) != -1 } {
		set conv(id) [dde disconnect $conv(id)]
		SetButtonState
		set conv(item) ""
		}
	}

# Request the current item o the current conversation

proc Request {} {

	global conv

	if { $conv(id) != -1 } {
		DisplayResult [dde request $conv(id) $conv(item)]
		}
	}

# Get the service partner and topic for the current conversation

proc Information {} {

	global conv

	if { $conv(id) != -1 } {
		set result [dde information $conv(id)]
	} else {
		set result [dde information]
		}
	if { $result == "" } {
		DisplayResult "No connections active"
	} else {
		DisplayResult $result
		}
	}

# Execute a command over the current conversation

proc Execute {} {

	global conv base

	catch [destroy $base.prompt]
	set f [toplevel $base.prompt]

	set done 0
	entry $f.entry -width 60 -font "Fixedsys 12" -textvariable conv(command)
	bind $f.entry <Return> "set done 1"
	pack $f.entry

	wm title $f "Enter a command,press Return..."

	focus $f

	tkwait variable done

	DisplayResult [dde exec $conv(id) $conv(command)]

	destroy $f
	}

# Display the list of services available

proc Services {} {

	global ServiceList

	DisplayResult [set ServiceList [dde services]]

	}

# Display the list of topics available for the current service

proc Topics {} {

	global conv TopicList

	DisplayResult [set TopicList [dde topics $conv(service)]]
	}

# A procedure to test the workings...

proc TestDDE { w { m "Hello, world!" } } {
	send $w { destroy .t }
	send $w { text .t -height 1 -width 30 -fg green -bg black -font "Crayon 20" }
	send $w { pack .t }
	set c [list .t insert end $m]
	send $w $c
	}

# Create a labeled entry

proc LabeledEntry { w var name text { state normal } } {

	global $var

	set f [frame $w -relief ridge -borderwidth 2]
	entry $f.entry -width 20 -justify right -textvariable conv($name) -state $state \
				-font "Fixedsys 12"
	label $f.label -text $text
	pack $f.label -side left
	pack $f.entry -side right

	return $f
	}

# Create the block of labeled entries

proc ConnectData { w } {

	set f [NewFrame $w]
	LabeledEntry $f.service conv service "Service Name"
	LabeledEntry $f.topic conv topic "Topic Name"
	LabeledEntry $f.item conv item "Item Name"
	LabeledEntry $f.connection conv id "Connection Identifier"

	pack $f.service -fill x
	pack $f.topic -fill x
	pack $f.item -fill x
	pack $f.connection -fill x

	return $w
	}

# Create a text box to hold the results of dde commands

proc ResultData { w } {

	global ResultWindow

	set f [NewFrame $w]
	set ResultWindow [text $f.text -height 5 -width 60 -yscrollcommand "$f.sb set" \
						-font "Fixedsys 12"]
	scrollbar $f.sb -orient vert -command "$f.text yview"
	pack $f.sb -side right -fill y
	pack $f.text -side left -fill both

	return $w
	}

# Exit procedure

proc Quit {} {

	exit

	}

# Create the list of action buttons

proc Actions { w } {

	set f [NewFrame $w]
	NoConvFunc $f.connect -text Connect -width 10 -command Connect
	ConvFunc $f.disconnect -text Disconnect -width 10 -state disabled -command Disconnect
	ConvFunc $f.request -text Request -width 10 -state disabled -command Request
	button $f.info -text Information -width 10 -state normal -command Information
	button $f.help -text Help -width 10 -state normal -command "help \"The DDE Explorer\""
	NoConvFunc $f.exit -text Exit -width 10 -command "Quit"

	pack $f.connect $f.disconnect $f.request $f.info $f.help $f.exit -side left

	return $w
	}

# Create the block of choice buttons

proc Choices { w } {

	set f [NewFrame $w]
	set f1 [NewFrame $f.left]
	set f2 [NewFrame $f.right]

	ConvFunc $f1.exec -text "Execute Command" -width 15 -command Execute -state disabled
	button $f1.services -text "List Services" -width 15 -command Services
	button $f1.topics -text "List Topics" -width 15 -command Topics

	button $f2.service -text "New Service" -width 15 -command NewService
	button $f2.topic -text "New Topic" -width 15 -command NewTopic
	ConvFunc $f2.item -text "New Item" -width 15 -command NewItem

	pack $f1.exec $f1.services $f1.topics -side top
	pack $f2.service $f2.topic $f2.item -side top

	pack $f.left $f.right -side left

	return $w
	}

# A status bar

proc StatusBar { w } {

	upvar #0 Status Status

	set f [NewFrame $w]
	entry $f.e -font "Fixedsys 12" -bg gray80 -textvariable Status -width 60
	pack $f.e -anchor w -fill x

	return $f
	}

# Display button help

set Help(New_Service) "Select a new service name"
set Help(New_Topic) "Select a new topic name"
set Help(List_Services) "Display all currently available services"
set Help(List_Topics) "Diisplay all topics for the current service"
set Help(Connect) "Connect to the current service and topic"
set Help(Disconnect) "Disconnect from the current service and topic"
set Help(Information) "Display information about the current connection"
set Help(Request) "Request the current item from the current service"
set Help(Execute_Command) "Execute a command on the current service"
set Help(Exit) "Terminate the application"
set Help(New_Item) "Select a new item for the current topic and service"
set Help(Help) "Display help information on this application"
set Help(Current) ""

# Generate an index for the help messages

proc MakeIdx { str } {

	return [join [split $str] _]
	}

proc ButtonHelp { w } {

	global Status Help

	if { $w != $Help(Current) } {
		set Help(Current) $w
		if { [$w cget -state] == "normal" } {
			set idx [MakeIdx [$w cget -text]]

			if [info exists Help($idx)] {
				set Status $Help($idx)
				}
			}
		}
	}

set f0 [NewFrame $base.top]
set f1 [ConnectData $f0.data]
set f2 [ResultData $base.results]
set f3 [Actions $base.actions]
set f4 [Choices $f0.choices]
set f5 [StatusBar $base.status]

pack $f1 $f4 -side left
pack $f0 $f2 $f3 $f5 -side top -fill x

SetButtonState

bind Button <Motion> "ButtonHelp %W"
bind Button <Leave> "set Status \"\""

wm title $base "DDE Explorer"
