# --- secret1.tcl --- Demonstration of a service using private topics

# Copyright(C) I.B.Findleton, 1998. All Rights Reserved

# This script sets up a server that will return records from the small database
# in response to requests from a client in the form...

#			dde request $id {item}

# where $id is a connection identifier for a conversation with the Secret service
# and item is an item in the database for the associated topic. See the secret2.tcl
# script for implementation details.

load dde

package require dde 2.0

package provide secret1 1.0

catch [destroy .all]

global Database TopicList TopicItemList Connections Serial outwin

set Connections 0
set Serial 0

# Load the database and some manipulation routines...

source $tcl_library/dde/database.tcl

# List the private topics for the Secret service...

set TopicList { Database Monica Bill Saddam Hillary }

# Items available in the database

set TopicItemList { PhoneNumber BestFriend FondestDesire MentalState }

# Delete the standard Tcl service name and register the Secret name. This
# actually removes the server from the Tcl service, so it will not be seen
# by other interps doing a 'winfo interps'. That is why it is the 'Secret'
# service. It will, however, look like a normal dde server application. When the
# new service is registered, add a set of private topics.

dde unregister
eval { dde register Secret } $TopicList

# Process the WildConnect event to help make the service invisible. Unless this is
# a query directed at the service itself, a response is blocked. The point here is that
# a WildConnect could have an empty service name, which is a query for all service
# names and possibly all topics that a server supports. Such a query will be rejected
# bu the following event handler code. This procedure could also
# be modified so that only specific users on specific machines can see the Secret
# service.

proc DdeWildConnectEvent { service topic user password machine application } {

	set service [string tolower $service]

	if { $service != "secret" } {
		return false
	} else {
		return true
		}
	}

# Get the external name of an item

proc ItemName { item } {

	global TopicItemList

	set len [llength $TopicItemList]

	set item [string tolower $item]

	set idx 0
	while { $idx < $len } {
		set name [string tolower [lindex $TopicItemList $idx]]
		if { $name == $item } {
			return [lindex $TopicItemList $idx]
			}
		incr idx
		}

	return ""
	}

# Install some procedures to handle requests. This is how a private topic is
# implemented.

proc DdeRequestEvent { service topic item format user } {

	global Database TopicList TopicItemList

# Handle requests that are not for the database by returning nothing, unless
# they are requests for information about the database...

	set topic [string tolower $topic]
	set item [string tolower $item]

# See if this is a request for information about the database

	if { $topic == "database" } {
		if { $item == "topicitemlist" } {
			return $TopicItemList
		} elseif { $item == "sysitems" } {
			return $TopicList
		} elseif {$item == "greeting" } {
			return [format "Welcome to the \"Secret\" service, %s" [Capitalize $user]]
			}
		return "Item $item is not supported"
		}

# Generate a key and return a record if it exists...

	set record [Key $topic $item]

	if [info exists Database($record)] {
		UpdateDatabase $user $topic $item
		set result $Database($record)
		return $result
	} else {
		return "Record \"$item\" about \"$topic\" is not available!"
		}
	}

# Capitalize a name

proc Capitalize { name } {

	set first [string index $name 0]
	set rest [string range $name 1 [expr [string length $name] - 1]]
	set first [string toupper $first]

	return [format "%s%s" $first $rest]
	}

# Implementation of a command language. Only the phone number can be
# changed, and only by some users for some people...

proc DdeExecuteEvent { service topic format user command } {

	global Database

	set keylist { phone Phone phonenumber Phonenumber PhoneNumber }

	set user [string tolower $user]

# The format of a commantd is....

#		set item value

#	where item is a key in the database and value is the new value.

	set command [split [string trim $command]]
	if { [lindex $command 0] == "set" } {
		set item [string tolower [lindex $command 1]]
		set value [lindex $command 2]
		if { [lsearch $keylist $item] != -1} {
			switch -exact -- $user {

	bill { 		SetItem bill phonenumber $value
					SetItem saddam mentalstate Elated
					SetItem hillary mentalstate Frustration
					SetItem monica mentalstate Euphoric
					SetItem nill fondestdesire "Friendship with Saddam"
					return "[Capitalize $user]'s new phone number is $value"
				}
	monica { 	SetItem monica phonenumber $value
					SetItem monica mentalstate Distraught
					SetItem hillary mentalstate Elated
					SetItem bill mentalstate Dispondent
					SetItem saddam mentalstate Frustration
					SetItem bill fondestdesire "Reconcilliation with Monica"
					return "[Capitalize $user]'s new phone number is $value"
				}
	hillary {	SetItem hillary phonenumber $value
					SetItem bill mentalstate Desperation
					SetItem monica mentalstate Elated
					SetItem saddam mentalstate Euphoric
					SetItem Hillary mentalstate Vindication
					SetItem bill fondestdesire "Reconciliation with Hillary"
					return "[Capitalize $user]'s new phone number is $value"
				}
	saddam { 	SetItem saddam phonenumber $value
					SetItem bill mentalstate Worried
					SetItem hillary mentalstate Smug
					SetItem monica mentalstate Annoyed
					SetItem saddam fondestdesire "Love and affection from anyone"
					return "[Capitalize $user]'s new phone number is $value"
				}
	ken { 		SetItem monica phonenumber $value
					SetItem hillary mentalstate Elated
					SetItem bill mentalstate Relieved
					SetItem saddam mentalstate Disappointed
					return "Monica's new phone number is $value"
				}
	linda { 		SetItem bill phonenumber $value
					SetItem hillary mentalstate Enraged
					SetItem saddam mentalstate Euphoric
					SetItem monica mentalstate Devastated
					return "Bill's new phone number is $value"
				}

	default { return "Sorry, [Capitalize $user], you do not have the permission required to modify the database!" }
				}
			} else {
				return "Invalid database key \"$item\". Valid keys are $keylist"
				}
		} else {
			return "Command [lindex $command 0] not supported!"
			}
	}

# Implementation of a private topic advise loop. In this example, the code is
# entirely superfluous because the Database is a set of Tcl variables, which could
# be monitored directly. Here, however, is how it would look if you have to do it
# the hard way....

proc DdeAdviseStartEvent { service topic item format user } {

	global Database

	set record [Key $topic $item]

	if [info exists Database($record)] {
		trace variable Database($record) w NotifyClient
		return true
	} else { return false; }
	}

# How to cancel a private topic advise loop

proc DdeAdviseStopEvent { service topic item format user } {

	global Database

	set record [Key $topic $item]
	if [info exists Database($record)] {
		trace vdelete Database($record) w NotifyClient
		return true
	} else { return false }
	}

# Provide data for an advise request event

proc DdeAdviseRequestEvent { service topic item format user } {

	global Database

# There could be other AdviseRequestevents for topics and items that are
# not part of the database. In this case, the System topic is being used
# to monitor the status of the server, so here we just allow that one to
# proceed by returning true.

	if { $topic == "System" } { return true }

# Assume that this is for the database.

	set record [Key $topic $item]
	if [info exists Database($record)] {
		return $Database($record)
	} else {
		return "Record \"$item\" or subject \"$topic\" does not exist!"
		}
	}

# Notify all clients of a change to a monitored private topic item. note that the
# rather convenient design of the database keys makes this somewhat simpler than
# itmight be in real life...

proc NotifyClient { name1 name2 op } {

	set list [split $name2 ,]
	set topic [Capitalize [lindex $list 0]]
	set item [ItemName [lindex $list 1]]

# Because the server will block when not ready, then we delay a short time
# before sending the notify on the change. A better way would be to get the
# server status and act when it is ready!

	after 250 "dde notify $topic $item"
	}

# Handle the quit button

proc Quit {} {

	destroy .all

	wm title . [tk appname]

        exit
	}

# Handle a connection request

set BadPasswords { "mary" "bob" "carol" "ted" "alice" "spot" "lassie" "timmy" "michele" \
						 "elmo" "ariel" "dumbo" "daddy" "mommy" "pocahantis" "mickey" \
						 "lucy" "jackie" "donald" "george" "fritz" "goofy" "secret" \
						 "barney" "kermit" "me" "melissa" "john" "tristan" "whitney" \
						 "poobear" "piglet" "eeyore" "tiger" "julien" "buffy" }

proc DdeConnectEvent { service topic user pw machine app } {

	global Connections Serial BadPasswords

	incr Serial

# Just for demonstration purposes, lets reject some users that make use
# of the names of girlfriends, boyfriends, pets and such...

	if { [lsearch $BadPasswords $pw] != -1 } {
		Display [format "%02d:User \"%s\" from \"%s\" refused connection. Bad Password \"%s\"" \
						$Serial $user $appname $pw]
		return false
		}

	incr Connections

	Display [format "%02d:User \"%s\" connected from \"%s\" about \"%s\"" \
								$Serial $user $app $topic]
	return true
	}

# Handle a disconnection

proc DdeDisconnectEvent { service topic user pw machine app } {

	global Connections Serial

	set Connections [expr $Connections - 1]
	incr Serial

	Display [format "%02d:User \"%s\" from \"%s\" disconnected from topic \"%s\"" \
				 $Serial $user $app $topic]
	}

# Display a message

proc Display { msg } {

	global outwin

	$outwin config -state normal
	$outwin insert end $msg
	$outwin insert end "\n"
	$outwin see end
	$outwin config -state disabled
	}

# Clear the log area

proc Clear {} {

	global outwin
	$outwin config -state normal
	$outwin delete 1.0 end
	$outwin config -state disabled

	}

wm title . "The \"Secret\" service"

set f0 [frame .all]
set f1 [frame $f0.status -relief ridge -borderwidth 2]

label $f1.l1 -text Connections
entry $f1.e1 -textvariable Connections -state disabled -justify right

pack $f1.l1 $f1.e1 -side left

set f2 [frame $f0.info -relief ridge -borderwidth 2]

set outwin [text $f2.text -width 80 -height 10 -fg black -bg gray80 \
			-font "Fixedsys 12" -yscrollcommand "$f2.sy set"]
scrollbar $f2.sy -orient vert -command "$f2.text yview"

pack $f2.sy -side right -fill y
pack $f2.text -side left -fill both -expand true

set f3 [frame $f0.actions -relief ridge -borderwidth 2]

button $f3.quit -text "Quit" -command "Quit" -width 10
button $f3.clear -text "Clear" -command "Clear" -width 10
button $f3.help -text "Help" -width 10 -command { help "The Secret Service" }

pack $f3.clear $f3.help $f3.quit -side left

pack $f1 $f2 $f3 -side top -fill x

pack $f0



