# --- secret2.tcl --- Demonstration of the use of private topics

# Copyright(c) I.B.Findleton, 1998 All Rights Reserved

load dde

package require dde 2.0

# This script defines procedures that will get information from a small database
# maintained by the "Secret" service. See the script secret1.tcl for the implementation
# details of the server.

global MonitorData MonitorCount Subjects Names UserName Password ConnectionId 

set Names ""
set Subjects ""
set UserName ""
set Password ""
set ConnectionId -1
set SubjectId(database) -1

set MonitorCount 0

# Get the connection identifier for a subject

proc GetConversation { topic } {

	global SubjectId

	set topic [string tolower $topic]

	if { [info exists SubjectId($topic)] } {
		return $SubjectId($topic)
	} else { return -1 }
	}

# Ask for an item from the server and display it in a colorful fashion...

proc AskFor { name item } {

	set id [GetConversation $name]
	if { $id != -1 } {
		set result [lindex [dde request $id $item] 0]

		DisplayMessage [format "%s,s" [Capitalize $name]] blue
		DisplayMessage [format " %s is " $item]
		DisplayMessage "$result\n" blue
	} else {
		DisplayMessage "Unable to find the conversation for $name!\n" red
		}
	}

# Compute the ordinal of the next entry in a list

proc NextEntry { cur len } {

	set cur [expr ($cur + 1) % $len]
	if { $cur < 1 } { incr cur }
	return $cur
}

# Check if a name is the current user

proc IsCurrentUser { cur } {

	global Names UserName

	set name [string tolower [lindex $Names $cur]]
	set user [string tolower $UserName]

	if { $name == $user } { return 1 } else { return 0 }
	}

# Change the subject

proc NextSubject {} {

	global who Names UserName Password

# Get a new subject

	set len [llength $Names]
	if { $who != "" } {
		set cur [lsearch $Names $who]
		set cur [NextEntry $cur $len]
		if  [IsCurrentUser $cur] {
				set cur [NextEntry $cur $len]
				}
		set who [lindex $Names $cur]
	} else {
		set cur [expr rand() * $len]
		set cur [expr int($cur)]
		if { $cur < 1 } { set cur 1 }
		if	{ $cur >= $len } {
			set cur [expr $len - 1]
		} else {
			set cur 1
			}
		if [IsCurrentUser $cur] { set cur [NextEntry $cur $len] }
		set who [lindex $Names $cur]
		}
	}

# Clear the display

proc Clear {} {

	global outwin

	$outwin config -state normal
	$outwin delete 1.0 end
	$outwin config -state disabled
	}

# Set up the list of buttons that will handle the available information items

proc SetupButtons { f1 width } {

	global Subjects

	set len [llength $Subjects]
	set idx 0

	set f1 [frame $f1 -relief ridge -borderwidth 2]

	while { $idx < $len } {
		button $f1.b$idx -text [lindex $Subjects $idx] -width $width \
				 -command "GetItem $idx" -background orange
		bind $f1.b$idx <Enter> { set help "Click to query this topic" }
		pack $f1.b$idx -side left -fill x
		incr idx
		}

	return $f1
	}

# Take action when a button is pressed...

proc GetItem { idx } {

	global Subjects who

	AskFor $who [lindex $Subjects $idx]
	}

# Display the status of the server

proc DisplayServerStatus { state } {

	global status

	set status [lindex $state 1]
	}

# Build a labeled entry

proc LabeledEntry { w text var width } {

	global $var

	set f [frame $w -relief ridge -borderwidth 2]
	label $f.label -text $text -width 20
	entry $f.entry -width $width -textvariable $var -font "Fixedsys 12" -justify right
	pack $f.entry -side right -anchor w
	pack $f.label -side left -anchor e -fill x

	return $f
	}

# Get the user name and password for a connection

proc GetUserId {} {

	global UserName Password

	set UserName ""
	set Password ""

	set p [toplevel .prompt]
	set f0 [frame $p.all -relief ridge -borderwidth 2]
	set f1 [LabeledEntry $f0.user "User Name" UserName 10]
	bind $f1.entry <Return> { set Ok 1 }
	set f2 [LabeledEntry $f0.password "Password" Password 10]
	$f2.entry config -show false
	bind $f2.entry <Return> { set Ok 1 }

	set f3 [frame $f0.actions -relief ridge -borderwidth 2]
	button $f3.login -width 10 -text "Login" -command "destroy $p"
	button $f3.cancel -width 10 -text "Cancel" -command "destroy $p"
	pack $f3.login $f3.cancel -side left

	pack $f1 $f2 $f3 -side top -fill x
	pack $f0

	wm title $p "Please Login"

	focus $f1.entry
	tkwait variable Ok
	focus $f2.entry
	tkwait variable Ok

	destroy $p
}

# Open conversations to on all of the acceptable topics

proc OpenSubjectConversations { list user password } {

	global SubjectId

	set len [llength $list]
	set idx 0

	while { $idx < $len } {
		set topic [lindex $list $idx]
		set SubjectId([string tolower $topic]) [dde connect Secret $topic $user $password]
		incr idx
		}
	}

# Close all conversations on topics

proc CloseSubjectConversations { list } {

	global SubjectId

	set len [llength $list]
	set idx 0

	while { $idx < $len } {
		set topic [string tolower [lindex $list $idx]]
		set id $SubjectId($topic)
		set SubjectId($topic) [dde disconnect $id]
		incr idx
		}
	}

# Enable and disable a button

proc EnableButton { w { state normal } } {

	if { $state == "normal" } {
		$w config -state normal -background green
	} else {
		$w config -state disabled  -background SystemButtonFace
		}
	}

# Logon to the service

proc Logon {} {

	global help status logon logoff execute monitor monitorx Names Subjects subject clear quit
	global ConnectionId UserName Password MonitorCount

# Get the user name and password for access to the server...

	GetUserId

# If the user name and password are invalid, the connection attempt will fail!

	set ConnectionId [dde connect Secret Database $UserName $Password]
	if { $ConnectionId == -1 } {
		set help "Logon to server failed!"
		return
		}

# The user name and password are valid...

	Clear
	wm title . [lindex [dde request $ConnectionId greeting] 0]
	set status "On Line"

# Get the topics that the Secret service will talk about A list element is returned for
# each server responding to the request. Here we assume there is only 1 valued respondant

	set Subjects [lindex [dde request $ConnectionId TopicItemList] 0]

# Get the topics that the Secret service will talk about

	set Names [lindex [dde request $ConnectionId SysItems] 0]

	BuildButtonList
	dde disconnect $ConnectionId

	OpenSubjectConversations $Names $UserName $Password

	set MonitorCount 0

# Set up an advise loop with the server for its Status string so we can watch
# how the load on the database is affecting performance.....

	set ConnectionId [dde connect Secret System $UserName $Password]
	if { $ConnectionId != -1 } {
		dde advise $ConnectionId Status { DisplayServerStatus {%VALUE} }

		EnableButton $logoff
		EnableButton $logon disabled
		EnableButton $clear
		EnableButton $execute

		if {[llength $Names] > 1 } {
			EnableButton $subject
			EnableButton $monitor
			}

		EnableButton $quit disabled

		NextSubject
		}
	}

# Capitalize a name

proc Capitalize { name } {

	set first [string index $name 0]
	set rest [string range $name 1 [expr [string length $name] - 1]]
	set first [string toupper $first]

	return [format "%s%s" $first $rest]
	}

# Modify the database

proc Modify {} {

	global command

	catch [destroy .prompt]
	set f [toplevel .prompt]

	wm title .prompt "Modify your personal information..."
	set command "set Phone "
	set done 0

	entry $f.entry -width 40 -textvariable command -font "Fixedsys 12"
	bind $f.entry <Return> "set done 1"

	pack $f.entry

	focus $f.entry

	tkwait variable done

	destroy .prompt

	set id [GetConversation Database]

	if { $id != -1 } {
		DisplayMessage "[lindex [dde exec $id $command] 0]\n" maroon
	} else {
		DisplayMessage "Can not find connection to the database!\n" red
		}
	}

# Log off from the service

proc Logoff {} {

	global who monitor monitorx status logon logoff quit subject execute clear Names Subjects
	global ConnectionId MonitorCount

	Clear
	set who ""
	wm title . [tk appname]

# Close down the advise loop for the status variable

	dde advise $ConnectionId Status

	if { $MonitorCount } {
		StopMonitorLoops
		set MonitorCount 0
		}

# Close all open connections

	set ConnectionId -1

	CloseSubjectConversations $Names

	dde disconnect

	set Names ""
	set Subjects ""

	BuildButtonList

	EnableButton $logon
	EnableButton $logoff disabled
	EnableButton $quit
	EnableButton $subject disabled
	EnableButton $clear disabled
	EnableButton $execute disabled
	EnableButton $monitor disabled
	EnableButton $monitorx disabled

	set status "Off Line"
	}

# Choose an item from a listbox

proc ChooseItem { TopicList title item } {

	global MonitorData MonitorCount

	if { [llength $TopicList] == 0 } {
		return ""
		}

	catch [destroy .prompt]
	set f [toplevel .prompt]

	wm title $f $title
	listbox $f.list -height [llength $TopicList] -bg white -font "Fixedsys 12"
	bind $f.list <Double-ButtonPress-1> \
		 "ListSelect %W $item"
	eval { $f.list insert end } $TopicList
	set f1 [frame $f.actions -relief ridge -borderwidth 2]
	button $f1.cancel -text Cancel -width 8 -command "destroy $f"

	pack $f1.cancel -side left

	pack $f.list
	pack $f1

	tkwait variable MonitorData($MonitorCount,$item)

	destroy .prompt

	return $MonitorData($MonitorCount,$item)
	}

# Get the list selection

proc ListSelect {w item } {

	global MonitorData MonitorCount

	set MonitorData($MonitorCount,$item) [$w get active]
	}

# Get the list of persons from the topicitemlist

proc PersonNames { list } {

	return [lreplace $list 0 0]
	}

# Monitor a database variable

proc Monitor {} {

	global Subjects Names MonitorData MonitorCount UserName Password
	global monitorx monitor

	set topic [ChooseItem [PersonNames $Names] "Which person?" Topic]
	set item [ChooseItem $Subjects "Which item?" Item]

	set id [GetConversation $topic]
	if { $id != -1 } {
		dde advise $id $item { DisplayNewValue %t %i %v }
		set MonitorData($MonitorCount,Id) $id
		incr MonitorCount
		EnableButton $monitorx
		DisplayMessage "Monitoring of $item for $topic activated!\n" blue
	} else {
		DisplayMessage "Unable to establish monitoring on $item for $topic!\n" red
		}

	}

# Find monitor loop

proc FindMonitorLoop { topic item } {

	global MonitorCount MonitorData

	set idx 0
	while { $idx < $MonitorCount } {
		if { $MonitorData($idx,Id) != -1 } {
			if { $MonitorData($idx,Topic) == $topic && $MonitorData($idx,Item) == $item } {
				return $idx
				}
			}
		incr idx
		}
	return -1
	}

# Make a list of the currently monitored things

proc MonitoredList {} {

	global MonitorData MonitorCount

	set list ""
	set idx 0
	while { $idx < $MonitorCount } {
		if { $MonitorData($idx,Id) != -1 } {
			lappend list [format "%s:%s" $MonitorData($idx,Topic) $MonitorData($idx,Item)]
			}
		incr idx
		}

	return $list
}

# Stop monitor loops

proc StopMonitorLoops {} {

	global MonitorCount MonitorData

	set idx 0
	while { $idx < $MonitorCount } {
		if { $MonitorData($idx,Id) != -1 } {
			dde advise $MonitorData($idx,Id) $MonitorData($idx,Item)
			}
		incr idx
		}
	return -1
	}

# Cancel a monitor on a variable

proc CancelMonitor {} {

	global Subjects Names MonitorData MonitorCount monitorx monitor

	set loop [ChooseItem [MonitoredList] "Which Item" Loop]

	if { $loop == "" } {
		DisplayMessage "No monitoring active!\n" red
		return
		}

	set loop [split $loop :]
	set topic [lindex $loop 0]
	set item [lindex $loop 1]

	set id [FindMonitorLoop $topic $item]

	if { $id != -1 } {
		dde advise $MonitorData($id,Id) $item
		set MonitorData($id,Id) -1
		DisplayMessage "Monitoring of $item for $topic terminated!\n" blue
		if { [MonitoredList] == "" } {
			EnableButton $monitorx disabled
			}
	} else {
		DisplayMessage "Monitoring of $item for $topic is not avtive!\n" red
		}
	}

# Display a message in the client text window

proc DisplayMessage { msg { color black } } {

	global outwin

	$outwin config -state normal
	$outwin insert end $msg $color
	$outwin see end
	$outwin config -state disabled

	}

# Display a changed value for a database item

proc DisplayNewValue { who what value } {

	DisplayMessage [format "%s's %s has changed to %s\n" $who $what $value] red
	}

# Handle an exit request

proc Quit {} {

	destroy .all

	exit
	}

# Build the list of buttons for actions on the database

proc BuildButtonList {} {

	global logon logoff quit clear helpbtn subject execute monitor monitorx Subjects Names outwin
	global who status help

	destroy .all

	set f [frame .all]

	set f1 [frame $f.actions -relief ridge -borderwidth 2]
	set buttonwidth 15

	bind Button <Leave> { set help "" }

	set logon [button $f1.logon -text "Logon" -width $buttonwidth -command "Logon"]
	bind $logon <Enter> { set help "Click to log on to the server" }
	set logoff [button $f1.logoff -width $buttonwidth -text "Logoff" \
								-command "Logoff" -state disabled]
	bind $logoff <Enter> { set help "Click to logoff of the server" }
	set subject [button $f1.subject -text "Subject" -width $buttonwidth \
								-command "NextSubject" -state disabled]
	bind $subject <Enter> { set help "Click to change the subject of your query" }
	set clear [button $f1.clear -text "Clear" -width $buttonwidth \
								-command "Clear" -state disabled]
	set execute [button $f1.execute -text "Modify" -width $buttonwidth \
								-command "Modify" -state disabled ]
	bind $execute <Enter> { set help "Click to modify the database." }
	bind $clear <Enter> { set help "Click to erase the query window" }
	set helpbtn [button $f1.help -text "Help" -state normal \
					-background green -command { help "The Secret Service" }]
	bind $helpbtn <Enter> { set help "Click to see the help topic for this script" }
	set quit [button $f1.quit -text "Quit" -width $buttonwidth -command "Quit"]
	bind $quit <Enter> { set help "Click to quit" }
	set monitor [button $f1.monitor -text "Monitor" -width $buttonwidth \
								-command "Monitor" -state disabled ]
	bind $monitor <Enter> { set help "Click to monitor a database record" }
	set monitorx [button $f1.monitorx -text "Cancel Monitor" -width $buttonwidth \
								-command "CancelMonitor" -state disabled]
	bind $monitorx <Enter> { set help "Click to cancel monitoring a database record" }

	pack $f1.logon $f1.logoff -side top -fill x

	pack $f1.subject $f1.clear $f1.execute $f1.monitor $f1.monitorx $f1.help $f1.quit -side top -fill x

# Build up the status area

	set f0 [frame $f.app]

	bind Entry <Leave> { set help "" }

	set f2 [frame $f0.person -relief ridge -borderwidth 2]
	label $f2.label1 -text "Subject Name"
	label $f2.label2 -text "Server Status"
	entry $f2.entry -width 10 -textvariable who -font "Fixedsys 12" \
				-justify center -state disabled
	bind $f2.entry <Enter> { set help "The subject of your queries" }
	entry $f2.status -width 10 -textvariable status -font "Fixedsys 12" \
				-bg gray80 -justify center -state disabled
	bind $f2.status <Enter> { set help "The current status of the server" }
	pack $f2.status -side right -anchor e
	pack $f2.label1 $f2.entry $f2.label2 -side left -fill x -ipadx 4

# A text window to handle the responses from the server

	set f3 [frame $f0.results -relief ridge -borderwidth 2]
	set outwin [text $f3.text -width 50 -yscrollcommand "$f3.sy set" \
		-fg black -bg gray80 -font "Fixedsys 12" \
		-height 13 -state disabled]

	set colorlist { red blue green orange maroon black violet cyan magenta purple }

	set colors [llength $colorlist]
	while { $colors } {
			set colors [expr $colors - 1]
			$outwin tag config [lindex $colorlist $colors] -foreground [lindex $colorlist $colors]
			}

	bind $outwin <Enter> { set help "These are results of your queries" }
	bind $outwin <Leave> { set help "" }

	scrollbar $f3.sy -orient vert -command "$f3.text yview"
	pack $f3.sy -side right -fill y
	pack $f3.text -side left -fill x -expand true

# build the buttons for queries

	set f5 [SetupButtons $f0.queries $buttonwidth]

# A status line for help information

	set f4 [frame $f0.helpmes -relief ridge -borderwidth 2]

	entry $f4.help -fg blue -bg gray80 \
				 -textvariable help -state disabled -width 70
	pack $f4.help -side left -fill x

	pack $f2 $f3 $f5 $f4 -side top -fill x

# Put it all together

	pack $f1 -side right -anchor n
	pack $f0 -side left -fill both

	pack $f -fill both
	}

global outwin who status help logon quit helpbtn

set who ""
set outwin ""
set status "Off Line"
set help ""

# Build up the buttons

BuildButtonList
EnableButton $logon
EnableButton $quit
EnableButton $helpbtn


