# ---------------------------------------------------------------------
#  $Id: connect.tcl,v 1.21 1997/06/14 22:24:42 adabas Exp $
# ---------------------------------------------------------------------
# Copyright (c) 1996-1997 Christian Krone. All rights reserved.
# This program is free software; you can redistribute it and/or
# modify it under the same terms as Tcl itself.
# See also licence.terms
# ----------------------------------------------------------------------------

#-----------------------------------------------------------------------
# usage: If an invalid argument is passed to this script,
#        it will terminate with the following instructive message.
#-----------------------------------------------------------------------
proc usage {} {
    global argv0
    puts -nonewline stderr "Usage: [file tail $argv0] \[-n <dbnode>] "
    puts -nonewline stderr "\[-d <dbname>] \[-u <user,passwd>] "
    puts -nonewline stderr "\[-U <xuserKey>] \[-S <sqlmode>"
    puts stderr "        -n   database node (default=local);"
    puts stderr "        -d   database name (default=SERVERDB);"
    puts stderr "        -u   name and password of user;"
    puts stderr "        -U   name of xuser key;"
    puts stderr "        -S   sqlmode: adabas, ansi, db2 or oracle."
    exit 1
}

# ---------------------------------------------------------------------
# getDbnameUserPassword: get the database name, user name and the
#                        corresponding password from anywhere you can
#                        get it: the environment, the command line or
#                        interactivly from the user...
# ---------------------------------------------------------------------
proc getDbnameUserPassword {{title "Adabas D Connect"} {cmd tryConnect}} {
    global connect env

    # Just to be shure...
    foreach param {cachelimit clearpw isolation serverdb
                   servernode sqlmode timeout user userkey} {
	catch {unset connect($param)}
    }

    # First check, if the command line contains valid parameters...
    validateCmdLine

    # ... then call adabas to fiddle with the options.
    if {[catch {array set connect [adabas xuser args]}]} usage

    # If not given by command line or xuser, take serverdb from environment.
    if ![string length $connect(serverdb)] {
	if [info exists env(SERVERDB)] {
	    set connect(serverdb) $env(SERVERDB)
	} elseif [info exists env(DBNAME)] {
	    set connect(serverdb) $env(DBNAME)
	}
    }

    # If, for some reason, serverdb still contains the servernode,
    # split it now into its two parts.
    if {[set colon [string first : $connect(serverdb)]] >= 0} {
	set connect(servernode) \
		[string range $connect(serverdb) 0 [expr $colon-1]]
	set connect(serverdb) \
		[string range $connect(serverdb) [expr $colon+1] end]
    }

    # ...and finally, if not all is already given, ask the user.
    if {![string length $connect(serverdb)] \
	    || ![connectCallCmd $cmd]} {
	getConnectDataFromUser $title $cmd
    }
}

proc validateCmdLine {} {
    global argv

    # On some systems (notably HP and DEC Alpha) by the restart of wish
    # with exec wish4.1 "$0" -- "$@" an empty list is given as argument,
    # if the original commandline consists only of the name of the command.
    # So here we ignore it, even if that mean, that now a call of e.g.
    # tkquery "" is considered valid.
    if {[llength $argv] <= 1 && ![string length [lindex $argv 0]]} return

    foreach arg $argv {
	if {[info exists optionFound]} {
	    switch $optionFound {
		u {if {![string match *,* $arg]} usage}
	    }
	    unset optionFound
	} else {
	    switch -glob -- $arg {
		-d  {set optionFound d}
		-n  {set optionFound n}
		-U  {set optionFound U}
		-u  {set optionFound u}
		-S  {set optionFound S}
		-d* - -U* - -n* - -S* {}
		-u* {if {![string match *,* $arg]} usage}
		default {usage}
	    }
	}
    }
    if {[info exists optionFound]} {usage}
}

proc getConnectDataFromUser {title cmd} {
    global connect bitmap tkCompat

    # Just to look nicer...
    option add *connect*Entry.foreground   black widgetDefault
    option add *connect*Entry.background   white widgetDefault
    option add *connect*highlightThickness 0     widgetDefault

    toplevel .connect
    
    image create bitmap logo -background "" -foreground blue \
	    -data $bitmap(sag-logo)
    image create bitmap sag  -background "" -foreground black \
	    -data $bitmap(sag)

    frame .connect.sag
    label .connect.sag.logo -image logo
    label .connect.sag.sag  -image sag
    pack  .connect.sag -fill x -expand 1 -padx 30 -pady 10
    pack  .connect.sag.logo -side left -fill x -expand 1 -anchor e
    pack  .connect.sag.sag  -side left -fill x -expand 1 -anchor w

    message .connect.msg -text $title -aspect 1000
    if $tkCompat(fontCmd) {
	.connect.msg configure -font "Times 24 bold"
    } else {
	catch {.connect.msg configure -font -*-Times-Bold-R-Normal-*-24-*-*-*-*-*}
    }
    pack    .connect.msg -fill x -pady 10 -padx 10

    frame .connect.up -relief sunken -border 2
    frame .connect.db -relief sunken -border 2
    pack  .connect.up -expand 1 -fill x -padx 20 -ipadx 5 -ipady 4
    pack  .connect.db -expand 1 -fill x -padx 20 -ipadx 5 -ipady 4

    frame .connect.user
    label .connect.user.text -text "Username:"
    entry .connect.user.entr -textvariable connect(user) \
	    -relief sunken -width 18
    pack  .connect.user.text -side left
    pack  .connect.user.entr -side left -expand 1 -pady 3 -anchor e
    pack  .connect.user      -expand 1 -fill x -in .connect.up \
	    -padx 7 -anchor s

    frame .connect.clearpw
    label .connect.clearpw.text -text "Password:"
    entry .connect.clearpw.entr -textvariable connect(clearpw) -show * \
	    -relief sunken -width 18
    pack  .connect.clearpw.text -side left
    pack  .connect.clearpw.entr -side left -expand 1 -pady 3 -anchor e
    pack  .connect.clearpw      -expand 1 -fill x -in .connect.up \
	    -padx 7 -anchor n

    frame .connect.serverdb
    label .connect.serverdb.text -text "Serverdb:"
    entry .connect.serverdb.entr -textvariable connect(serverdb) \
	    -relief sunken -width 18
    pack  .connect.serverdb.text -side left
    pack  .connect.serverdb.entr -side left -expand 1 -pady 3 -anchor se
    pack  .connect.serverdb      -expand 1 -fill x -in .connect.db \
	    -padx 7 -anchor s

    frame .connect.servernode
    label .connect.servernode.text -text "Servernode:"
    entry .connect.servernode.entr -textvariable connect(servernode) \
	    -relief sunken -width 18
    pack  .connect.servernode.text -side left
    pack  .connect.servernode.entr -side left -expand 1 -pady 3 -anchor ne
    pack  .connect.servernode      -expand 1 -fill x -in .connect.db \
	    -padx 7 -anchor n

    message .connect.err -textvariable connect(error) -aspect 10000
    pack    .connect.err -fill x  

    frame  .connect.but
    button .connect.but.ok  -text Connect \
	    -command "connectCallCmd $cmd .connect"
    button .connect.but.end -text "End (F3)" -command "exit"
    pack .connect.but.ok  -side left -fill x -expand 1
    pack .connect.but.end -side left -fill x -expand 1
    pack .connect.but     -fill x -expand 1

    focus .connect.user.entr

    bind .connect.user.entr   <Return> {focus [tk_focusNext %W]}
    foreach entry {clearpw serverdb servernode} {
	bind .connect.$entry.entr <Return> \
		".connect.$entry.entr icursor 0; .connect.but.ok invoke"
    }
    bind .connect <F3> ".connect.but.end invoke"

    wm title .connect "Adabas D Connect"

    tkwait window .connect
    
    image delete logo
    image delete sag
}

# ---------------------------------------------------------------------
# connectCallCmd: This procedure will be called, when the user pushes
#                 the OK-Button. If all needed information is gathered
#                 together (dbname, user and password), it calls the
#                 given boolean procedure, which could connect to the
#                 database, check the correctness of the mentioned
#                 information or check nothing at all...
# ---------------------------------------------------------------------
proc connectCallCmd {cmd {window ""}} {
    global connect

    if {[string length $connect(servernode)] \
	    && [string first : $connect(serverdb)] < 0} {
	set orgServerdb       $connect(serverdb)
	set connect(serverdb) $connect(servernode):$connect(serverdb)
    }

    if {[info exists connect(userkey)] && ![info exists connect(clearpw)]} {
        if [$cmd] {
	    return 1
        }
        unset connect(userkey)
	set   connect(clearpw) ""
    } else {
        foreach entry {user clearpw serverdb} {
	    if ![string length $connect($entry)] {
	        lappend missing $entry
	    }
	}
        if [info exists missing] {
	    set connect(error) "Please enter [join $missing " and "]"
	    return 0
        }

        set connect(error)   ""
    }

    if [string length $window] {
	foreach entry {user clearpw} {
	    if ![regexp "^\[ \t]*\".*\"\[ \t]*$" $connect($entry)] {
		set connect($entry) [string toupper $connect($entry)]
	    }
	}
	update idletask
    }

    if [$cmd] {
	if [string length $window] {
	    if [regexp "^\[ \t]*\"(.*)\"\[ \t]*$" $connect(user) match user] {
		set connect(user) $user
	    }
	    destroy $window
	}
	return 1
    }

    set connect(clearpw) ""
    if [info exists orgServerdb] {
	set connect(serverdb) $orgServerdb
    }
    return 0
}

# ---------------------------------------------------------------------
# Two procedures, that can be given as second parameter of
# getDbnameUserPassword: One tries to connect to the database and
# stores the created connection in connect(adabas), the other does
# nothing at all...
# ---------------------------------------------------------------------
proc tryConnect {} {
    global connect adamsg

    set connect(cmd) adalogon

    if [info exists connect(clearpw)] {
	lappend connect(cmd) $connect(user),$connect(clearpw)
    } elseif [info exists connect(userkey)] {
        lappend connect(cmd) ,$connect(userkey)
    } else {
        lappend connect(cmd) ,
    }

    lappend connect(cmd) -serverdb $connect(serverdb)

    switch  [string tolower $connect(sqlmode)] {
	ansi - db2 - oracle {
	    lappend connect(cmd) -sqlmode [string tolower $connect(sqlmode)]
	}
    }

    if [catch $connect(cmd) rc] {
	if [info exists adamsg(rc)]&&[info exists adamsg(errortxt)] {
	    set connect(error) "$adamsg(rc): $adamsg(errortxt)"
	} else {
	    switch $rc {
		"xuser of default user failed: cannot find xuser file" {}
		"xuser of default user failed: the USERKEY is unknown" {}
		default {set connect(error) $rc}
	    }
	}
	return 0
    }

    set connect(db) $rc
    return 1
}

proc noCheck {} {
    return 1
}

set bitmap(sag) {
#define sag_width 255
#define sag_height 45
static unsigned char sag_bits[] = {
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0xff,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xff,
0x1f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0xc0,0xff,0x1f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0xff,0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0xff,
0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0xff,0x3f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0x07,0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0x07,
0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0xe0,0xff,0x3f,0x00,0x00,0x00,0xe0,0x07,0x3f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xf8,0xff,0xff,0xe0,
0xff,0xff,0xe1,0xff,0xff,0xff,0x7f,0xc0,0xc7,0x1f,0xf8,0xff,
0x3f,0xc0,0xff,0x7f,0x80,0xff,0xff,0x3f,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0xfc,0xff,0xff,0xe1,0xff,0xff,0xe7,0xff,
0xff,0xff,0x7f,0xc0,0xc7,0x7f,0xfc,0xff,0xff,0xf0,0xff,0xff,
0xc1,0xff,0xff,0x3f,0x00,0xf8,0xff,0xff,0xc0,0xff,0xff,0x81,
0xfc,0xff,0xff,0xf1,0xff,0xff,0xe7,0xff,0xff,0xff,0x7f,0xc0,
0xc7,0x7f,0xfc,0xff,0xff,0xf0,0xff,0xff,0xc3,0xff,0xff,0x3f,
0x00,0xfc,0xff,0xff,0xe0,0xff,0xff,0x83,0xfc,0xff,0xff,0xf3,
0xff,0xff,0xef,0xff,0xff,0xff,0x7f,0xc0,0xc7,0xff,0xfc,0xff,
0xff,0xf8,0xff,0xff,0xe7,0xff,0xff,0x3f,0x00,0xfc,0xff,0xff,
0xf1,0xff,0xff,0x87,0xfc,0xff,0xff,0xf3,0xff,0xff,0xef,0xff,
0xff,0xff,0x7f,0xc0,0xc7,0xff,0xfe,0xff,0xff,0xf9,0xff,0xff,
0xe7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xf1,0xff,0xff,0x8f,
0xfc,0xff,0xff,0xfb,0xff,0xff,0xef,0xff,0xff,0xff,0x7f,0xc0,
0xc7,0xff,0xfe,0xff,0xff,0xf9,0xff,0xff,0xf7,0xff,0xff,0x3f,
0x00,0xfe,0xff,0xff,0xf9,0xff,0xff,0x8f,0xfc,0x00,0xe0,0xfb,
0xff,0xff,0xef,0xff,0xff,0xff,0x7e,0xc0,0xc7,0xff,0xfe,0x00,
0xf8,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,
0xfb,0xff,0xff,0x8f,0xfc,0x00,0xe0,0xfb,0x01,0xc0,0xef,0x07,
0x7f,0x00,0x7e,0xc0,0x07,0xfc,0x7e,0x00,0xf8,0xfd,0x01,0xe0,
0xf7,0x07,0x00,0x00,0x00,0xfe,0x00,0xf8,0xfb,0xff,0xff,0x8f,
0xfc,0xff,0x7f,0xf8,0x01,0xc0,0xef,0x03,0x3f,0x00,0x7e,0xc0,
0x07,0xf8,0x7e,0x00,0xf0,0xfd,0x00,0xe0,0xf7,0x03,0x00,0x00,
0x00,0x7e,0x00,0xf0,0xfb,0x01,0xc0,0x8f,0xfc,0xff,0xff,0xf8,
0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,
0xff,0xfd,0xff,0xff,0xf7,0x03,0x00,0x00,0x00,0x7e,0x00,0xf0,
0xfb,0x01,0x00,0x80,0xfc,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,
0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,
0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xfb,0x81,0xff,0x8f,
0xfc,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,
0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,
0x00,0xfe,0xff,0xff,0xfb,0xc1,0xff,0x8f,0xf8,0xff,0xff,0xfb,
0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,
0xff,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,
0xfb,0xc1,0xff,0x8f,0xf0,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,
0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,
0xf3,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xfb,0xe1,0xff,0x8f,
0x00,0x00,0xf8,0xfb,0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,
0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,0xf1,0xff,0xff,0x1f,
0x00,0xfe,0xff,0xff,0xfb,0xc1,0xff,0x8f,0xfc,0x00,0xf0,0xfb,
0x01,0x80,0xef,0x03,0x3e,0xf0,0x7f,0xc0,0x07,0xf8,0x7e,0x00,
0xf8,0xfd,0x01,0xf8,0xf1,0x03,0x00,0x00,0x00,0xfe,0xff,0xff,
0xfb,0x81,0xff,0x8f,0xfc,0x00,0xf0,0xfb,0x01,0xc0,0xef,0x03,
0x3e,0xf0,0x7f,0xe0,0x07,0xfc,0x7e,0x00,0xf0,0xfd,0x00,0xf8,
0xf1,0x03,0x00,0x00,0x00,0x7e,0x00,0xf8,0xfb,0x01,0xc0,0x8f,
0xfc,0xff,0xff,0xfb,0xff,0xff,0xef,0x03,0x7e,0xf8,0xff,0xf0,
0x0f,0xfe,0x7e,0x00,0xf0,0xfd,0x00,0xf8,0xf1,0x03,0x00,0x00,
0x00,0x7e,0x00,0xf0,0xfb,0x01,0xc0,0x8f,0xfc,0xff,0xff,0xf3,
0xff,0xff,0xef,0x03,0xfe,0xff,0xff,0xff,0xff,0xff,0x7e,0x00,
0xf0,0xfd,0x00,0xf0,0xf3,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xfb,0x03,0xe0,0x8f,0xfc,0xff,0xff,0xf3,0xff,0xff,0xef,0x03,
0xfe,0xff,0xff,0xff,0xff,0xff,0x7e,0x00,0xf0,0xfd,0x00,0xf0,
0xf3,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,0xfb,0xff,0xff,0x8f,
0xf8,0xff,0xff,0xf3,0xff,0xff,0xef,0x03,0xfe,0xff,0xfd,0xff,
0xff,0xff,0x7e,0x00,0xf0,0xfd,0x00,0xf0,0xe3,0xff,0xff,0x1f,
0x00,0x7e,0x00,0xf0,0xfb,0xff,0xff,0x8f,0xf8,0xff,0xff,0xf1,
0xff,0xff,0xe7,0x03,0xfe,0xff,0xfc,0xff,0xff,0x7f,0x7e,0x00,
0xf0,0xfd,0x00,0xe0,0xe7,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xf3,0xff,0xff,0x8f,0xf0,0xff,0xff,0xe1,0xff,0xff,0xe7,0x03,
0xfc,0xff,0xf8,0xff,0xff,0x7f,0x7e,0x00,0xf0,0xfd,0x00,0xe0,
0xe7,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,0xf3,0xff,0xff,0x87,
0xe0,0xff,0xff,0xc0,0xff,0xff,0xe3,0x03,0xf8,0x7f,0xf8,0xff,
0xfe,0x3f,0x7e,0x00,0xf0,0xfd,0x00,0xe0,0xc7,0xff,0xff,0x1f,
0x00,0x7e,0x00,0xf0,0xe3,0xff,0xff,0x87,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0x03,0xe0,0x1f,0xc0,0x3f,0xf8,0x0f,0x3e,0x00,
0xf0,0xf9,0x00,0xc0,0x87,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xc3,0xff,0xff,0x83,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x7e,0x00,0xf0,0x83,0xff,0xff,0x81,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80};
}

set bitmap(sag-logo) {
#define sag-logo_width 37
#define sag-logo_height 61
static unsigned char sag-logo_bits[] = {
0x00,0x00,0x00,0x00,0xe0,0x00,0x00,0x00,0x00,0xe0,0x00,0x00,
0x00,0x00,0xe0,0x00,0x00,0x00,0x00,0xe0,0x00,0x80,0xff,0xff,
0xe3,0x00,0xf0,0xff,0xff,0xe3,0x00,0xf8,0xff,0xff,0xe3,0x00,
0xfc,0xff,0xff,0xe3,0x00,0xfe,0xff,0xff,0xe3,0x00,0xff,0xff,
0xff,0xe3,0x80,0xff,0xff,0xff,0xe3,0xc0,0xff,0xff,0xff,0xe3,
0xc0,0xff,0xff,0xff,0xe3,0xe0,0xff,0xff,0xff,0xe3,0xe0,0xff,
0xff,0xff,0xe3,0xf0,0xff,0xff,0xff,0xe3,0xf0,0xff,0xff,0xff,
0xe3,0xf0,0xff,0xff,0xff,0xe3,0xf8,0xff,0x07,0x00,0xe0,0xf8,
0xff,0x03,0x00,0xe0,0xf8,0xff,0x01,0x00,0xe0,0xf8,0xff,0x00,
0x00,0xe0,0xf8,0xff,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,
0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,
0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,
0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,
0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,
0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,
0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,
0xe0,0xff,0xe3,0xfc,0x7f,0xf0,0xff,0xe3,0x00,0x00,0xf0,0xff,
0xe1,0x00,0x00,0xf8,0xff,0xe1,0x00,0x00,0xfe,0xff,0xe1,0xfc,
0xff,0xff,0xff,0xe1,0xfc,0xff,0xff,0xff,0xe1,0xfc,0xff,0xff,
0xff,0xe1,0xfc,0xff,0xff,0xff,0xe0,0xfc,0xff,0xff,0xff,0xe0,
0xfc,0xff,0xff,0xff,0xe0,0xfc,0xff,0xff,0x7f,0xe0,0xfc,0xff,
0xff,0x7f,0xe0,0xfc,0xff,0xff,0x3f,0xe0,0xfc,0xff,0xff,0x3f,
0xe0,0xfc,0xff,0xff,0x1f,0xe0,0xfc,0xff,0xff,0x0f,0xe0,0xfc,
0xff,0xff,0x03,0xe0,0xfc,0xff,0xff,0x00,0xe0,0x00,0x00,0x00,
0x00,0xe0,0x00,0x00,0x00,0x00,0xe0,0x00,0x00,0x00,0x00,0xe0,
0x00,0x00,0x00,0x00,0xe0};
}
