# ---------------------------------------------------------------------
#  $Id: fotos.tcl,v 1.34 1997/06/14 22:24:35 adabas Exp $
# ---------------------------------------------------------------------

# ---------------------------------------------------------------------
# The procedure, that creates all the elements in the toplevel, does the
# key binding, calls the select statement, and reads the first record.
# ---------------------------------------------------------------------
proc createMain {} {
    global logonHandle cursor diaZeit adamsg adabastcl_library tkCompat

    initOptions

    # Now open the cursor and fetch the first row...
    if [catch openCursor] {
	append err "This application is only functional, if the connected "  \
		"user has the select privilege to a table fotos.\n\n"        \
		"Quit now by pressing Cancel and try to connect as another " \
		"user, who has this privilege granted, or give your Okay "   \
		"to create the fotos table and load a first picture."

	switch [tk_messageBox -title "Fatal error" -message $err \
		-icon question -type okcancel -default cancel] {
	    cancel {exit 2}
	}

	set createCmd "CREATE TABLE fotos (name CHAR(20) KEY, foto LONG BYTE)"
	if [catch {adasql $cursor $createCmd} msg] {
	    tk_messageBox -title "Fatal error" -icon error \
		    -message "$msg: $adamsg(errortxt)"
	    exit 3
	}

	# Here we don't ask the returncode no longer,
	# since there is no legal cause to fail here...
	adasql $cursor "INSERT INTO fotos (name) VALUES ('Adabas')"
	adawritelong $cursor -table fotos -column foto -where name='Adabas' \
		-filename [file join $adabastcl_library panel.gif]
	adacommit $logonHandle

	openCursor
    }

    fetchFoto first

    # Just to get a nice looking window...
    wm title    . "Adabas D: Fotos"
    wm iconname . "Fotos"

    # Create two labels (for name and foto) and a frame (for the buttons).
    # Note: Create the button frame first, since then it stays in the
    #       toplevel, even if the user resizes it to a very small window...
    pack [frame .b -relief sunken -borderwidth 2] -fill x -side bottom
    pack [label .name -textvariable name]
    pack [label .foto -image foto -relief ridge -borderwidth 2] -expand 1
    if $tkCompat(fontCmd) {
	.name configure -font "Times 17 bold"
    } else {
	if [catch {.name configure -font -*-times-bold-r-*-*-17-*-*-*-*-*-*-*}] {
	    catch {.name configure -font -*-times-bold-r-*-*-14-*-*-*-*-*-*-*}
	}
    }

    # And here are the buttons in the bottom row...
    pack [button .b.first -text <<   -command "fetchFoto first"] -side left
    pack [button .b.prev  -text <    -command "fetchFoto prev"]  -side left
    pack [button .b.last  -text >>   -command "fetchFoto last"]  -side right
    pack [button .b.next  -text >    -command "fetchFoto next"]  -side right
    pack [button .b.quit  -text Quit -command exit] -fill x -expand 1

    # All commands can be called by keystrokes, too...
    bind .foto <Prior> "call .b.prev"
    bind .foto <Next>  "call .b.next"
    bind .foto <Home>  "call .b.first"
    bind .foto <End>   "call .b.last"
    bind .foto <F3>    "call .b.quit"
    bind .foto <F1>    "callHtmlHelp fotos"

    # A popup will offer all this choises plus some additional ones...
    menu .m
    .m add cascade -label Help -menu .m.help
    .m add separator
    .m add command -label Select  -command selectFoto
    .m add command -label Import  -command importFoto
    .m add command -label Export  -command exportFoto
    .m add command -label Delete  -command deleteFoto
    .m add separator
    .m add check   -label Diashow -command diaShow -variable dia
    .m add cascade -label Refresh -menu .m.refr
    .m add separator
    .m add command -label First   -command "fetchFoto first"
    .m add command -label Prev    -command "fetchFoto prev"
    .m add command -label Next    -command "fetchFoto next"
    .m add command -label Last    -command "fetchFoto last"
    .m add separator
    .m add command -label Quit    -command exit -acc "F3"

    menu .m.help
    .m.help add command -label "Help..."      -underline 0 -acc F1 \
	    -command "callHtmlHelp fotos"
    .m.help add command -label "About..."     -underline 0 \
	    -command "aboutHelp Fotos"
    .m.help add check   -label "Use Netscape" -underline 0 \
	    -variable options(val:netscapeHelp)

    menu .m.refr
    .m.refr add radio -label " 5 sec" -variable diaZeit -value  5000
    .m.refr add radio -label "10 sec" -variable diaZeit -value 10000
    .m.refr add radio -label "20 sec" -variable diaZeit -value 20000
    .m.refr add radio -label "30 sec" -variable diaZeit -value 30000
    set diaZeit 10000

    bind .foto <3> "tk_popup .m %X %Y"
    bind . <Enter> "focus .foto"

    if [string length [info globals _noFotoFound]] {
	confMenueEntry .m Export -state disabled
	confMenueEntry .m Delete -state disabled
    }

    # Open the toplevel window (it may be unmapped by the connect procedure)...
    wm deiconify .
}

proc initOptions {} {
    global options tkCompat

    frame .fotos ;# Just for the options...
    option add *fotos.netscapeHelp False widgetDefault
    catch {option readfile ~/.Xdefaults userDefault}
    set options(val:netscapeHelp) [string match {[Tt]*} \
	    [option get .fotos netscapeHelp NetscapeHelp]]

    if !$tkCompat(imageViaData) {
	# If we have an interpreter, which didn't read images from data,
	# we have to determine the temp directory.
	global tmpDir env tcl_platform

	foreach var {TMPDIR TMP TEMP} {
	    if [info exists env($var)] {
		set tmpDir $env($var)
		break
	    }
	}
	if ![info exists tmpDir] {
	    switch $tcl_platform(platform) {
		windows {set tmpDir c:/tmp}
		default {set tmpDir /tmp}
	    }
	}
	if {![file isdirectory $tmpDir] || ![file writable $tmpDir]} {
	    tk_messageBox -title "Fatal error" -icon error \
		    -message "Fatal error:
Temp directory \"$tmpDir\" doesn't exist or isn't writable.
Cannot continue."
	    exit 3
	}
    }
}

proc selectFoto {{sName ""}} {
    global logonHandle name _selectIndex tkCompat

    if ![string length $sName] {
	catch {destroy .sel}
	toplevel .sel
	wm title .sel "Fotos: Select"

	listbox   .sel.l           -yscrollcommand ".sel.s set"
	scrollbar .sel.s -orient vertical -command ".sel.l yview"
	frame     .sel.b
	button    .sel.b.dis -text Dismiss -command {destroy .sel}
	button    .sel.b.ok  -text Okay    -command {
	    set _selectIndex [expr {[.sel.l index active]+1}]
	    destroy .sel
	}

	grid .sel.l -row 0 -column 0 -sticky nsew
	grid .sel.s -row 0 -column 1 -sticky ns
	grid .sel.b -row 1 -column 0 -sticky ew -columnspan 2

	grid    rowconfigure .sel 0 -weight $tkCompat(gridWeightMax)
	grid columnconfigure .sel 0 -weight $tkCompat(gridWeightMax)

	grid .sel.b.dis -row 0 -column 0 -sticky ew
	grid .sel.b.ok  -row 0 -column 1 -sticky ew
	grid columnconfigure .sel.b 0 -weight $tkCompat(gridWeightMax)
	grid columnconfigure .sel.b 1 -weight $tkCompat(gridWeightMax)

	bind .sel.l <F3>       ".sel.b.dis invoke; break"
	bind .sel.l <Return>   ".sel.b.ok  invoke; break"
	bind .sel.l <Double-1> ".sel.b.ok  invoke; break"

	set selectCmd "SELECT name FROM fotos ORDER BY name"
	set searchCmd ".sel.l insert end @1"
    } else {
	set selectCmd "SELECT name, ROWNO FROM fotos ORDER BY name"
	set searchCmd [list setIndexIfName $sName @1 @2]
    }

    set selectCursor [adaopen $logonHandle]
    if [catch {adasql $selectCursor $selectCmd -resulttable fotoSearch}] {
	adaclose $selectCursor
	destroy .sel
	tk_messageBox -title Error -icon error -message "No foto found"
	return
    }
    adafetch $selectCursor -command $searchCmd
    adaclose $selectCursor

    if ![string length $sName] {
	focus .sel.l
	tkwait window .sel
    }

    # If the user pressed Dismiss or destroyed the window, better jump off.
    if ![info exists _selectIndex] {return ""}

    fetchFoto $_selectIndex

    # Make sure, the _selectIndex isn't defined after leaving the proc.
    unset _selectIndex
}

# ---------------------------------------------------------------------
# setIndexIfName: Help function to use as -command option of adafetch.
#                 We don't write the if statement directly as option,
#                 since Tcl8.0 chokes about the if expression...
# ---------------------------------------------------------------------
proc setIndexIfName {sName name rowno} {
    global _selectIndex

    if ![string compare $name $sName] {
	set _selectIndex $rowno
    }
}

proc importFoto {} {
    global logonHandle

    set fileName [tk_getOpenFile -title "Foto: Import" \
	    -filetypes {{"Gif files" *.gif} {"All Files" *}}]
    if ![string length $fileName] return

    if [file isfile $fileName] {
	set name [file tail [file rootname $fileName]]
	set insertCmd "INSERT INTO fotos (name) VALUES ('$name')"

	set insertCursor  [adaopen $logonHandle]
	if [catch {adasql $insertCursor $insertCmd} retCode] {
	    if {$retCode == 200} {
		tk_messageBox -title "Import" -icon error \
			-message "$name already exists"
		return
	    } else {
		return -code error $retCode
	    }
	}
	adawritelong $insertCursor -filename $fileName \
		-table fotos -column foto -where "name='$name'"
	adaclose $insertCursor
	adacommit $logonHandle

	openCursor
	selectFoto $name

	confMenueEntry .m Export -state normal
	confMenueEntry .m Delete -state normal
    } else {
	tk_messageBox -title "Import" -icon error \
		-message "File '$fileName' not found"
    }
}

proc exportFoto {} {
    global cursor name

    set fileName [tk_getSaveFile -title "Fotos: Export" \
	    -filetypes {{"Gif files" *.gif} {"All Files" *}}]
    if ![string length $fileName] return

    if [catch {adareadlong $cursor -filename $fileName \
	    -table fotos -column foto -where "name='$name'"} errMsg] {
	tk_messageBox -title "Export" -icon error -message $errMsg
    }
}

proc deleteFoto {} {
    global logonHandle name

    switch [tk_messageBox -title "Delete" \
	    -message "Really delete foto $name?" \
	    -icon question -type yesno -default no] {
	no {return}
    }

    set deleteCmd "DELETE FROM fotos WHERE name = '$name'"

    set deleteCursor [adaopen $logonHandle]
    adasql $deleteCursor $deleteCmd
    adaclose $deleteCursor
    adacommit $logonHandle

    if {![fetchFoto next] && ![fetchFoto prev]} {
	openCursor
	fetchFoto first
    }
}

proc diaShow {} {
    global dia diaZeit

    while {$dia} {
	if ![fetchFoto next] {fetchFoto first}

	after $diaZeit set pause 1
	vwait pause
    }
}

# ---------------------------------------------------------------------
# confMenueEntry: Since the menuentries in tearoffs are not updated
#                 together with their original, you should use this
#                 procedure instead of entryconfigure directly.
# ---------------------------------------------------------------------
proc confMenueEntry {m index args} {
    set tearOffs [info commands .tearoff*]
    if [llength $tearOffs] {
        set m "$m $tearOffs"
    }
    foreach menu $m {
        catch {eval $menu entryconfigure [list $index] $args}
    }
}

# ---------------------------------------------------------------------
# This Procedure calls the given button. It does a lot of work, so that
# the button is pressed while calling the procedure and depressed thereafter.
# ---------------------------------------------------------------------
proc call {b} {
    set orgRelief [$b cget -relief]
    set orgState  [$b cget -state]
    $b configure -state active -relief sunken
    update idletask
    $b invoke
    $b configure -relief $orgRelief -state $orgState
}

proc openCursor {} {
    global cursor adamsg _noFotoFound

    # Now open the cursor with the sql statement
    set selectCmd "SELECT name, foto FROM fotos ORDER BY name"

    if [catch {adasql $cursor $selectCmd} rc] {
	if {$adamsg(rc) != 100} {return -code error $rc}

	set selectCmd "SELECT 'No foto stored', NULL FROM dual"
	adasql $cursor $selectCmd

	# We do this within a catch, since the
	# menu isn't guaranteed to exist yet...
	catch {confMenueEntry .m Export -state disabled}
	catch {confMenueEntry .m Delete -state disabled}
	set _noFotoFound 1
    }
}

# ---------------------------------------------------------------------
# The procedure fetchFoto does all the real work: It fetches the row
# in the given direction (first, last, next, prev or pos) and puts it
# via external file into the image foto...
# ---------------------------------------------------------------------
proc fetchFoto {dir} {
    global cursor name adamsg bitmap tkCompat tcl_platform tmpDir

    set row [adafetch $cursor -position $dir]
    if [string compare $row ""] {
	set name  [lindex $row 0]
	set descr [lindex $row 1]
	
	if [string compare $descr $adamsg(nullvalue)] {
	    if $tkCompat(imageViaData) {
		image create photo foto -data [adareadlong $cursor \
			-descriptor [lindex $row 1] -encoding base64]
	    } else {
		set imageFileName [file join $tmpDir foto_[pid]]
		adareadlong $cursor -descriptor [lindex $row 1] \
			-filename $imageFileName
		image create photo foto -file $imageFileName
		switch $tcl_platform(platform) {
		    windows {exec del $imageFileName}
		    default {exec rm  $imageFileName}
		}
	    }
	} else {
	    image create bitmap foto -foreground black -background "" \
		    -data $bitmap(sag)
	}
	return 1
    } else {
	return 0
    }
}

# ---------------------------------------------------------------------
# 'adaConnect' will be called, when the user presses the Connect button
# in the connect dialog. It tries to connect with the given user/password
# combination to the database, and sets - if succesful - the global
# variables 'logonHandle' and 'cursor'.
# ---------------------------------------------------------------------
proc adaConnect {} {
    global connect logonHandle cursor

    if ![tryConnect] {return 0}

    set logonHandle $connect(db)
    set cursor [adaopen $logonHandle]

    return 1
}


set bitmap(sag) {
#define sag_width 255
#define sag_height 45
static unsigned char sag_bits[] = {
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0xff,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xff,
0x1f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0xc0,0xff,0x1f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0xff,0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0xff,
0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0xff,0x3f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0x07,0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0x07,
0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0xe0,0xff,0x3f,0x00,0x00,0x00,0xe0,0x07,0x3f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xf8,0xff,0xff,0xe0,
0xff,0xff,0xe1,0xff,0xff,0xff,0x7f,0xc0,0xc7,0x1f,0xf8,0xff,
0x3f,0xc0,0xff,0x7f,0x80,0xff,0xff,0x3f,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0xfc,0xff,0xff,0xe1,0xff,0xff,0xe7,0xff,
0xff,0xff,0x7f,0xc0,0xc7,0x7f,0xfc,0xff,0xff,0xf0,0xff,0xff,
0xc1,0xff,0xff,0x3f,0x00,0xf8,0xff,0xff,0xc0,0xff,0xff,0x81,
0xfc,0xff,0xff,0xf1,0xff,0xff,0xe7,0xff,0xff,0xff,0x7f,0xc0,
0xc7,0x7f,0xfc,0xff,0xff,0xf0,0xff,0xff,0xc3,0xff,0xff,0x3f,
0x00,0xfc,0xff,0xff,0xe0,0xff,0xff,0x83,0xfc,0xff,0xff,0xf3,
0xff,0xff,0xef,0xff,0xff,0xff,0x7f,0xc0,0xc7,0xff,0xfc,0xff,
0xff,0xf8,0xff,0xff,0xe7,0xff,0xff,0x3f,0x00,0xfc,0xff,0xff,
0xf1,0xff,0xff,0x87,0xfc,0xff,0xff,0xf3,0xff,0xff,0xef,0xff,
0xff,0xff,0x7f,0xc0,0xc7,0xff,0xfe,0xff,0xff,0xf9,0xff,0xff,
0xe7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xf1,0xff,0xff,0x8f,
0xfc,0xff,0xff,0xfb,0xff,0xff,0xef,0xff,0xff,0xff,0x7f,0xc0,
0xc7,0xff,0xfe,0xff,0xff,0xf9,0xff,0xff,0xf7,0xff,0xff,0x3f,
0x00,0xfe,0xff,0xff,0xf9,0xff,0xff,0x8f,0xfc,0x00,0xe0,0xfb,
0xff,0xff,0xef,0xff,0xff,0xff,0x7e,0xc0,0xc7,0xff,0xfe,0x00,
0xf8,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,
0xfb,0xff,0xff,0x8f,0xfc,0x00,0xe0,0xfb,0x01,0xc0,0xef,0x07,
0x7f,0x00,0x7e,0xc0,0x07,0xfc,0x7e,0x00,0xf8,0xfd,0x01,0xe0,
0xf7,0x07,0x00,0x00,0x00,0xfe,0x00,0xf8,0xfb,0xff,0xff,0x8f,
0xfc,0xff,0x7f,0xf8,0x01,0xc0,0xef,0x03,0x3f,0x00,0x7e,0xc0,
0x07,0xf8,0x7e,0x00,0xf0,0xfd,0x00,0xe0,0xf7,0x03,0x00,0x00,
0x00,0x7e,0x00,0xf0,0xfb,0x01,0xc0,0x8f,0xfc,0xff,0xff,0xf8,
0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,
0xff,0xfd,0xff,0xff,0xf7,0x03,0x00,0x00,0x00,0x7e,0x00,0xf0,
0xfb,0x01,0x00,0x80,0xfc,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,
0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,
0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xfb,0x81,0xff,0x8f,
0xfc,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,
0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,
0x00,0xfe,0xff,0xff,0xfb,0xc1,0xff,0x8f,0xf8,0xff,0xff,0xfb,
0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,
0xff,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,
0xfb,0xc1,0xff,0x8f,0xf0,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,
0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,
0xf3,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xfb,0xe1,0xff,0x8f,
0x00,0x00,0xf8,0xfb,0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,
0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,0xf1,0xff,0xff,0x1f,
0x00,0xfe,0xff,0xff,0xfb,0xc1,0xff,0x8f,0xfc,0x00,0xf0,0xfb,
0x01,0x80,0xef,0x03,0x3e,0xf0,0x7f,0xc0,0x07,0xf8,0x7e,0x00,
0xf8,0xfd,0x01,0xf8,0xf1,0x03,0x00,0x00,0x00,0xfe,0xff,0xff,
0xfb,0x81,0xff,0x8f,0xfc,0x00,0xf0,0xfb,0x01,0xc0,0xef,0x03,
0x3e,0xf0,0x7f,0xe0,0x07,0xfc,0x7e,0x00,0xf0,0xfd,0x00,0xf8,
0xf1,0x03,0x00,0x00,0x00,0x7e,0x00,0xf8,0xfb,0x01,0xc0,0x8f,
0xfc,0xff,0xff,0xfb,0xff,0xff,0xef,0x03,0x7e,0xf8,0xff,0xf0,
0x0f,0xfe,0x7e,0x00,0xf0,0xfd,0x00,0xf8,0xf1,0x03,0x00,0x00,
0x00,0x7e,0x00,0xf0,0xfb,0x01,0xc0,0x8f,0xfc,0xff,0xff,0xf3,
0xff,0xff,0xef,0x03,0xfe,0xff,0xff,0xff,0xff,0xff,0x7e,0x00,
0xf0,0xfd,0x00,0xf0,0xf3,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xfb,0x03,0xe0,0x8f,0xfc,0xff,0xff,0xf3,0xff,0xff,0xef,0x03,
0xfe,0xff,0xff,0xff,0xff,0xff,0x7e,0x00,0xf0,0xfd,0x00,0xf0,
0xf3,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,0xfb,0xff,0xff,0x8f,
0xf8,0xff,0xff,0xf3,0xff,0xff,0xef,0x03,0xfe,0xff,0xfd,0xff,
0xff,0xff,0x7e,0x00,0xf0,0xfd,0x00,0xf0,0xe3,0xff,0xff,0x1f,
0x00,0x7e,0x00,0xf0,0xfb,0xff,0xff,0x8f,0xf8,0xff,0xff,0xf1,
0xff,0xff,0xe7,0x03,0xfe,0xff,0xfc,0xff,0xff,0x7f,0x7e,0x00,
0xf0,0xfd,0x00,0xe0,0xe7,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xf3,0xff,0xff,0x8f,0xf0,0xff,0xff,0xe1,0xff,0xff,0xe7,0x03,
0xfc,0xff,0xf8,0xff,0xff,0x7f,0x7e,0x00,0xf0,0xfd,0x00,0xe0,
0xe7,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,0xf3,0xff,0xff,0x87,
0xe0,0xff,0xff,0xc0,0xff,0xff,0xe3,0x03,0xf8,0x7f,0xf8,0xff,
0xfe,0x3f,0x7e,0x00,0xf0,0xfd,0x00,0xe0,0xc7,0xff,0xff,0x1f,
0x00,0x7e,0x00,0xf0,0xe3,0xff,0xff,0x87,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0x03,0xe0,0x1f,0xc0,0x3f,0xf8,0x0f,0x3e,0x00,
0xf0,0xf9,0x00,0xc0,0x87,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xc3,0xff,0xff,0x83,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x7e,0x00,0xf0,0x83,0xff,0xff,0x81,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80};
}

# ---------------------------------------------------------------------
# Mainprogram
# ---------------------------------------------------------------------
wm withdraw .
loadAdabastcl
getDbnameUserPassword "Adabas D: Fotos" adaConnect
createMain
