# Simple HTML display library by Stephen Uhler (stephen.uhler@sun.com)
# Copyright (c) 1995 by Sun Microsystems
# Version 0.3 Fri Sep  1 10:47:17 PDT 1995
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# Copyright (c) 1995 by CRC for Advanced Computational Systems
# $Id: htmllib.tcl,v 1.11 1996/06/26 04:32:02 steve Exp $
# See http://surfit.anu.edu.au/SurfIt/COPYRIGHT for information on usage
# and redistribution.
#
# To use this package, see http://surfit.anu.edu.au/SurfIt/eng/

package provide html_library [lindex {$Revision: 1.11 $} 1]

proc HMlibrary_init {} {

    # There is a bug in the tcl library focus routines that prevents focus
    # from every reaching an un-viewable window.  Use our *own*
    # version of the library routine, until the bug is fixed, make sure we
    # over-ride the library version, and not the otherway around

    auto_load tkFocusOK
    proc tkFocusOK w {
	set code [catch {$w cget -takefocus} value]
	if {($code == 0) && ($value != "")} {
	    if {$value == 0} {
		return 0
	    } elseif {$value == 1} {
		return 1
	    } else {
		set value [uplevel #0 $value $w]
		if {$value != ""} {
		    return $value
		}
	    }
	}
	set code [catch {$w cget -state} value]
	if {($code == 0) && ($value == "disabled")} {
	    return 0
	}
	regexp Key|Focus "[bind $w] [bind [winfo class $w]]"
    }

    HMoptimize
}

##################################################################
############################################
# mapping of html tags to text tag properties
# properties beginning with "T" map directly to text tags

# These are Defined in HTML 3.0

# HMtag_map contains mapping that apply to the hyperwindow

array set HMtag_map {
	address	{style i}
	b      {weight bold}
	blockquote	{style i}
	bq		{style i}
	cite   {style i}
	code   {family courier}
	dfn    {style i}	
	em     {style i}
	h1     {size 24 weight bold}
	h2     {size 22}		
	h3     {size 20}	
	h4     {size 18}
	h5     {size 16}
	h6     {style i}
	i      {style i}
	kbd    {family courier weight bold}
	pre    {family courier}
	samp   {family courier}		
	strong {weight bold}		
	tt     {family courier}
	u	 {Tunderline underline}
	var    {style i}	
}

# These are defined in html3.2

array set HMtag_map {
	center {Tcenter center}
	strike {Tstrike strike}
	big    {size 18}
	small  {size 8}
	sub    {size 8 Tsubscript subscript}
	sup    {size 8 Tsuperscript superscript}
}

# HMwintag_map contains values that are only set for a window

array set HMwintag_map {
	blockquote	{Trindent rindent indent 1}
	bq		{Trindent rindent indent 1}
	dir	{indent 1}
	dl	{indent 1}
	dd	{indent 1}
	menu	{indent 1}
	ol	{indent 1}
	pre	{fill 0 Tnowrap nowrap pre 1}
	ul     {indent 1}
}

# initial values

set HMtag_map(hmstart) {
	family times   weight medium   style r   size 14
	Tcenter ""   Tlink ""   Tunderline ""   list list
	counter 0 adjust 0
}
set HMwintag_map(hmstart) {
	Tnowrap "" fill 1 pre 0
}

# Left alignment is the default, so no need to explicitly set a tag
# (kludge to fix proper stacking of alignment)

array set HMalign_map {
	left {}
	center {Tcenter center}
	right {Tright right}
}

# Map HTML standard colours that are not X standard colours

array set HMcolour_map {
	aqua aquamarine  fuchsia hotpink lime limegreen  olive olivedrab  silver #d9d9d9 teal lightseagreen
}
proc HMmap_colour {colour} {
	global HMcolour_map
	if {[regexp {^#.*} $colour]} {return $colour}
	catch {set colour $HMcolour_map($colour)}
	return $colour
}

# html tags that insert white space
# A better algorithm to insert white space is to only add
# the first newline if the insertion point is not at the
# beginning of the line.  Normalise the array HMinsert_map
# so that every entry is assumed to cause a newline if the
# above condition holds.

array set HMinsert_map {
	address	"\n"
	blockquote "\n" /blockquote ""
	br	""
	dd	"" /dd	""
		   /dl	"\n"
	dt	"\n"
	form ""	/form ""
	h1	"\n"	/h1	"\n"
	h2	"\n"	/h2	"\n"
	h3	"\n"	/h3	""
	h4	""	/h4	""
	h5	""	/h5	""
	h6	""	/h6	""
	li   ""
	/dir ""
	/ul ""
	/ol ""
	/menu ""
	p	"\n"	/p	""
	center	""	/center	""
	pre ""	/pre ""
}

# tags that are list elements, that support "compact" rendering

array set HMlist_elements {
	ol 1   ul 1   menu 1   dl 1   dir 1
}
############################################
# initialize the window and stack state

proc HMinit_hyperwin {win} {
	HMinit_state $win
	HMpush_win HM$win $win
}

############################################
# initialize a window and push it onto the widget stack

proc HMpush_win {state win} {
	upvar #0 $state var
	upvar #0 HN$win winvar

	# Push the window
	set var(current) $win
	set var(stack) [linsert $var(stack) 0 $win]

	# Initialise its state
	HMinit_win_state $win
	$win tag configure underline -underline 1
	$win tag configure center -justify center
	$win tag configure left -justify left
	$win tag configure right -justify right
	$win tag configure nowrap -wrap none
	$win tag configure rindent -rmargin $var(S_tab)c
	$win tag configure strike -overstrike 1
	$win tag configure subscript -offset -4
	$win tag configure superscript -offset 4
	$win tag configure mark -foreground [HMmap_colour red]		;# list markers
	$win tag configure list -spacing1 3p -spacing3 3p		;# regular lists
	$win tag configure compact -spacing1 0p		;# compact lists
	if {[catch {$win tag configure bullet -font [HMx_font zapfdingbats 12 medium r]}]} {
		# Default to same as hmstart
		$win tag configure bullet -font [HMx_font times 14 medium r]
		set var(S_symbols) O*=+-o\xd7\xb0>:\xb7
	}
	$win tag configure link -borderwidth 2 -foreground [HMmap_colour $var(linkfg)]	;# hypertext links
	# Handle B&W displays
	if {[winfo depth $win] == 1} {
		$win tag configure link -underline 1
	}
	HMset_indent $win $var(S_tab)
	$win configure -wrap word
	catch {$win configure -background [HMmap_colour $var(bgcolor)]}
	catch {$win configure -foreground [HMmap_colour $var(fgcolor)]}

	# configure the text insertion point
	$win mark set $var(S_insert) 1.0

	# for horizontal rules
	if {[catch {$win tag configure thin -font [HMx_font times 2 medium r]}]} {
		# Default to standard size
		$win tag configure thin -font [HMx_font times 8 medium r]
	}
	$win tag configure hr -relief sunken -borderwidth 2 -wrap none  -tabs [winfo width $win]
	bind $win <Configure> {
		%W tag configure hr -tabs %w
		%W tag configure last -spacing3 %h
	}

	# generic link enter callback

	$win tag bind link <ButtonRelease-1> "HMlink_hit $state $win %x %y"
}

proc HMpop_win {state} {
	upvar #0 $state var
	set var(stack) [lreplace $var(stack) 0 0]
	set var(current) [lindex $var(stack) 0]
}

# set the indent spacing (in cm) for lists
# TK uses a "weird" tabbing model that causes \t to insert a single
# space if the current line position is past the tab setting

proc HMset_indent {win cm} {
	set tabs [expr $cm / 2.0]
	$win configure -tabs ${tabs}c
	foreach i {1 2 3 4 5 6 7 8 9} {
		set tab [expr $i * $cm]
		$win tag configure indent$i -lmargin1 ${tab}c -lmargin2 ${tab}c  -tabs "[expr $tab + $tabs]c [expr $tab + 2*$tabs]c"
	}
}

# reset the state of window - get ready for the next page
# remove all but the font tags for the main window, 
# and remove all form state

proc HMreset_win {win} {
	upvar #0 HM$win var
	regsub -all { +[^L ][^ ]*} " [$win tag names] " {} tags
	catch "$win tag delete $tags"
	eval $win mark unset [$win mark names]
	$win delete 0.0 end
	$win tag configure hr -tabs [winfo width $win]

	# configure the text insertion point
	$win mark set $var(S_insert) 1.0

	# remove form state.  If any check/radio buttons still exists, 
	# their variables will be magically re-created, and never get
	# cleaned up.
	catch {eval unset [info globals HM$win.form*]}

	# Remove previous window state variables
	foreach w [info globals HN$win*] {
		upvar #0 $w winvar
		unset winvar
	}

	HMinit_state $win
	HMpush_win HM$win $win
	return HM$win
}

# initialize the window's state array
# two (or more) variables are used to store the state.
# HM$win (upvar'd to var) stores state associated with the document as a whole
# $var(current) is the window currently being rendered
# $var(stack) is a list of the current stack of nested windows
# ([lindex $var(stack) 0] == $var(currwin))
# HN$var(current) (upvar'd to winvar) stores state associated 
# with rendering into that window
# (NB. the use of the "HN" prefix rather than "HM" to avoid clashes)

# Parameters beginning with S_ are NOT reset
#  adjust_size:		global font size adjuster
#  unknown:		character to use for unknown entities
#  tab:			tab stop (in cm)
#  symbols:		Symbols to use on un-ordered lists
#  current:		Text widget to render
#  stack:		the widget stack

proc HMinit_state {win} {
	upvar #0 HM$win var
	array set tmp [array get var S_*]
	catch {unset var}
	array set var {
		S_adjust_size 0
		S_tab 1.0
		S_unknown \xb7
		S_symbols O*=+-o\xd7\xb0>:\xb7
		S_insert Insert
		current {}
		stack {}
		tables {}
		tags 0
		linkfg blue
	}
	set var(basewin) $win
	set var(S_bgcolor) [$win cget -background]
	set var(S_fgcolor) [$win cget -foreground]
	array set var [array get tmp]
	set var(bgcolor) $var(S_bgcolor)
	set var(fgcolor) $var(S_fgcolor)
}

proc HMinit_win_state {win} {
	upvar #0 HN$win winvar
	catch {unset winvar}
	array set winvar {
		fill 1
		list list
		indent ""
	}
}

# alter the parameters of the text state
# this allows an application to over-ride the default settings
# it is called as: HMset_state -param value -param value ...

array set HMparam_map {
	-tab S_tab
	-unknown S_unknown
	-size S_adjust_size
	-symbols S_symbols
    -insert S_insert
    -foreground S_fgcolor
    -background S_bgcolor
}

proc HMset_state {win args} {
	upvar #0 HM$win var
	global HMparam_map
	set bad 0
	if {[catch {array set params $args}]} {return 0}
	foreach i [array names params] {
		incr bad [catch {set var($HMparam_map($i)) $params($i)}]
	}
	return [expr $bad == 0]
}

############################################
# manage the display of html

# HMrender gets called for every html tag
#   cookie:   The name of the main text widget to render into
#   tag:   The html tag (in arbitrary case)
#   not:   a "/" or the empty string
#   param: The un-interpreted parameter list
#   text:  The plain text until the next html tag

proc HMrender {cookie tag not param text} {
	upvar #0 HM$cookie var
	if {![catch {$var(noProcessCmd) HM$cookie $var(current) tag not param text} result] && !$result} return;
	upvar #0 HN$var(current) winvar

	global HMtag_map HMwintag_map HMinsert_map HMlist_elements
	set tag [string tolower $tag]
	set text [HMmap_esc $text]

	# manage compact rendering of lists
	if {[info exists HMlist_elements($tag)]} {
		set list "list [expr {[HMextract_param $param compact] ? "compact" : "list"}]"
	} else {
		set list ""
	}

	# adjust (push or pop) tag state
	catch {HMstack HM$cookie $not $HMtag_map($tag)}
	catch {HMstack HN$var(current) $not "$HMwintag_map($tag) $list"}

	# insert white space (with current font)
	# adding white space can get a bit tricky.  This isn't quite right
	if {[info exists HMinsert_map($not$tag)]} {
		if {[lindex [split [$var(current) index $var(S_insert)] .] 1] != 0} {
			$var(current) insert $var(S_insert) \n "space $winvar(font)"
		}
		$var(current) insert $var(S_insert) $HMinsert_map($not$tag) "space $winvar(font)"
		if {[lindex $winvar(fill) end]} {
			set text [string trimleft $text]
		}
	}

	# to fill or not to fill
	if {[lindex $winvar(fill) end]} {
		set text [HMzap_white $text]
	} else {
		set text [HMtranslate_cr $text]
	}

	# generic mark hook
	catch {HMmark $not$tag $win $param text}

	# do any special tag processing
	catch {HMtag_$not$tag HM$cookie $var(current) $param text}
	#puts "HMtag_$not$tag returned \"$err\""
	# NB. current window may have changed, so winvar may now be invalid

	# add the text with proper tags
	set tags [HMcurrent_tags HM$cookie $var(current)]
	$var(current) insert $var(S_insert) $text $tags
}

proc HMgetUniqueNum {state} {
	upvar #0 $state var
	return [incr var(tags)]
}

# Empty html tags - all tags are defined here to improve performance,
# but may be redefined later

proc HMmark {args} {}
foreach tag {html /html head /head body /body  title /title isindex base style script meta link banner /banner  h1 h2 h3 h4 h5 h6 /h1 /h2 /h3 /h4 /h5 /h6  address /address map /map  p /p pre /pre ul /ul ol /ol li dl /dl dt dd div /div  blockquote /blockquote center /center hr br  form /form input select /select textarea /textarea option  table /table caption /caption tr td th  b /b i /i u /u big /big small /small sub /sub sup /sup font /font tt /tt strike /strike  em /em strong /strong dfn /dfn code /code samp /samp kbd /kbd var /var cite /cite  a /a img applet /applet param} {
    proc HMtag_$tag {args} {}
}

# html tags requiring special processing
# Procs of the form HMtag_<tag> or HMtag_</tag> get called just before
# the text for this tag is displayed.  These procs are called inside a 
# "catch" so it is OK to fail.
#   state: The name of the hyperdocument state array
#   win:   The name of the text widget to render into
#   param: The un-interpreted parameter list
#   text:  A pass-by-reference name of the plain text until the next html tag
#          Tag commands may change this to affect what text will be inserted
#          next.

# A pair of pseudo tags are added automatically as the 1st and last html
# tags in the document.  The default is <HMstart> and </HMstart>.
# Append enough blank space at the end of the text widget while
# rendering so HMgoto can place the target near the top of the page,
# then remove the extra space when done rendering.

proc HMtag_hmstart {state win param text} {
	upvar #0 $state var
	$win mark gravity $var(S_insert) left
	$win insert end "\n " last
	$win mark gravity $var(S_insert) right
}

proc HMtag_/hmstart {state win param text} {
	$win delete last.first end
}

# Comment processing.  The HTML 2.0 spec seems to imply that tags cannot be
# nested within comments, but it is common practice - even W3C pages!
# This breaks the HTML parser (which is extremely simplistic), so we have to
# do alot of munging to fix things up.

proc HMtag_!-- {state win param text} {
	debug_puts [list HTML start comment]
	upvar #0 $state var
	upvar $text data
	if {![regexp -- {--$} $param]} {
		# Nested tag(s)
		set data {}
		set var(noProcessCmd) HMcomment
	}
}

# Consume tags until we find '-->' (the end of the comment)

proc HMcomment {state win tag not param text} {
	upvar #0 $state var
	upvar $param elparam
	upvar $text data
	set remains $data
	if {[regexp -- {--$} $elparam] || [regexp -- {-->(.*)} $data all remains]} {
		upvar $tag eltag
		upvar $not elnot
		set data $remains
		set elparam {}
		set eltag {}
		set elnot {}
		unset var(noProcessCmd)
		return 1
	} else {return 0}
}

# put the document title in the window banner, and remove the title text
# from the document

proc HMtag_title {state win param text} {
	upvar $text data
	wm title [winfo toplevel $win] $data
	wm iconname [winfo toplevel $win] $data
	set data ""
}

# put enclosed text in the hyperdocument banner - a nonscrolled
# region.  There can only be one banner.

proc HMtag_banner {state win param text} {
	upvar #0 $state var
	# preserve the Insertion point
	set index [[set banner [HMget_banner $win]] index $var(S_insert)]
	# push banner onto the state
	HMpush_win $state $banner
	$banner mark set $var(S_insert) $index
}

proc HMtag_/banner {state win param text} {
	# Make the banner fit its text
	HMpop_win $state
}

# Links may be used to include an icon in the document "toolbar".
# In this case we use the hyperdocument banner as the toolbar.
# The HREF attribute is mandatory.

# draft-ietf-html-relrev-00 proposes these relationships:
#	MADE/AUTHOR COPYRIGHT DISCLAIMER META PUBLISHER TRADEMARK
#	TOC/CONTENTS INDEX NAVIGATOR 
#	CHILD PARENT SIBLING TOP/ORIGIN
#	NEXT PREV/PREVIOUS BEGIN/FIRST END/LAST
#	BIBLIOENTRY BIBLIOGRAPHY CITATION DEFINITION GLOSSARY FOOTNOTE 
# these are reserved
#	BACK FORWARD HOME

proc HMtag_link {state win param text} {
	upvar #0 $state var
	if {([HMextract_param $param rev link] || [HMextract_param $param rel link]) &&
	     [HMextract_param $param href]} {
		if {[regexp [set link [string tolower $link]] {back forward home}]} return;	# Illegal values
		set banner [HMget_banner $win]
		set item [button $banner.[HMgetUniqueNum $state]  -text [string toupper [string index $link 0]][string range $link 1 end]  -foreground orange  -command "HMlink_callback $state $banner $href"]
		HMlink_window_setup $item $href
		HMset_icon $state $item $link
		$banner window create $var(S_insert) -window $item -align center
	}
}

# Application must override this

proc HMget_banner {win} {
	return $win
}

# HTML 3.2 introduces various attributes for the BODY tag.
# BACKGROUND (Tk 4.1 cannot support this), BGCOLOR, TEXT, LINK, VLINK, ALINK

proc HMtag_body {state win param text} {
	upvar #0 $state var
	# Only valid at the toplevel
	if {$var(basewin) != $win} return;
	if {[HMextract_param $param bgcolor]} {
		set bgcolor [string tolower $bgcolor]
		if {![catch {$win configure -background [HMmap_colour $bgcolor]}]} {
			set var(bgcolor) [HMmap_colour $bgcolor]
		}
	}
	# GOTCHA: HMextract_param clobbers text argument
	if {[HMextract_param $param text]} {
		set text [string tolower $text]
		if {![catch {$win configure -foreground [HMmap_colour $text]}]} {
			set var(fgcolor) [HMmap_colour $text]
		}
	}
	if {[HMextract_param $param link]} {
		set link [string tolower $link]
		if {![catch {$win tag configure link -foreground [HMmap_colour $link]}]} {
			set var(linkfg) [HMmap_colour $link]
		}
	}
}

proc HMtag_hr {state win param text} {
	upvar #0 $state var
	$win insert $var(S_insert) "\n" space "\n" thin "\t" "thin hr" "\n" thin
}

proc HMtag_font {state win param text} {
	upvar #0 $state var
	if {[HMextract_param $param size]} {
		HMstack $state "" "size [expr [lindex $var(size) end] + $size]"
	} else {
		HMstack $state "" [list size [lindex $var(size) end]]
	}
	if {[HMextract_param $param color]} {
		set color [string tolower $color]
		$win tag configure colour$color -foreground [HMmap_colour $color]
		HMstack $state "" "Tcolour colour$color"
	} else {
		HMstack $state "" [list Tcolour {}]
	}
}

# Pop font size & colour stacks and pop tag stack as appropriate

proc HMtag_/font {state win param text} {
	upvar #0 $state var
	HMstack $state "/" [list size {}]
	HMstack $state "/" [list Tcolour {}]
}

proc HMtag_address {state win param text} {
	if {[HMextract_param $param nowrap]} {
		HMstack HN$win "" "Tnowrap nowrap"
	}
}

proc HMtag_/address {state win param text} {
	upvar #0 HN$win winvar
	if {[info exists winvar(Tnowrap)]} {
		unset winvar(Tnowrap)
		HMstack HN$win / "Tnowrap nowrap"
	}
}

proc HMtag_p {state win param text} {
	global HMalign_map
	upvar #0 HN$win winvar

	# Paragraphs acts as line breaks inside preformatted text
	if {$winvar(pre)} {
		HMrender $win br "" "" ""
	}

	# Implicitly end the previous alignment
	if {[info exists winvar(curralign)]} {HMtag_/p $state $win {} {}}

	set align [string tolower [HMget_alignment $param]]
	HMstack $state "" "$HMalign_map($align)"
	set winvar(curralign) $align

	if {[HMextract_param $param nowrap]} {
		HMstack HN$win "" "Tnowrap nowrap"
	}
}

proc HMtag_/p {state win param text} {
	upvar #0 $state var
	upvar #0 HN$win winvar
	global HMalign_map
	if {$winvar(pre)} return
	catch {HMstack $state / "$HMalign_map($winvar(curralign))"}
	catch {unset winvar(curralign)}
	if {[info exists winvar(Tnowrap)]} {
		unset winvar(Tnowrap)
		HMstack HN$win / "Tnowrap nowrap"
	}
}

# Process attributes common to all headings

proc HMheading {state win param} {
	global HMalign_map
	upvar #0 HN$win winvar

	if {[HMextract_param $param src]} {
		HMtag_img $state $win "src=$src" ""
	}

	# Positioning the SRC image before the DINGBAT image
	# is an arbitrary choice
	if {[HMextract_param $param dingbat]} {
		HMload_icon $state $win [string tolower $dingbat]
	}

	set align [string tolower [HMget_alignment $param]]
	HMstack $state "" "$HMalign_map($align)"
	set winvar(curralign) $align

	if {[HMextract_param $param nowrap]} {
		HMstack HN$win "" "Tnowrap nowrap"
	}
}

# Remove any alignment and nowrap attributes.
# Ending paragraphs does the same, so use that code.

proc HMendHeading {state win param} { 
	HMtag_/p $state $win $param {}
}

proc HMtag_h1 {state win param text} {
	HMheading $state $win $param
}
proc HMtag_h2 {state win param text} {
	HMheading $state $win $param
}
proc HMtag_h3 {state win param text} {
	HMheading $state $win $param
}
proc HMtag_h4 {state win param text} {
	HMheading $state $win $param
}
proc HMtag_h5 {state win param text} {
	HMheading $state $win $param
}
proc HMtag_h6 {state win param text} {
	HMheading $state $win $param
}

proc HMtag_/h1 {state win param text} {
	HMendHeading $state $win $param
}
proc HMtag_/h2 {state win param text} {
	HMendHeading $state $win $param
}
proc HMtag_/h3 {state win param text} {
	HMendHeading $state $win $param
}
proc HMtag_/h4 {state win param text} {
	HMendHeading $state $win $param
}
proc HMtag_/h5 {state win param text} {
	HMendHeading $state $win $param
}
proc HMtag_/h6 {state win param text} {
	HMendHeading $state $win $param
}

# Standard HTML v3.0 icon entity names are:

# ftp gopher telnet archive filing.cabinet folder fixed.disk disk.drive document unknown.document
# text.document binary.document binhex.document audio film image map form mail parent
# next previous home toc glossary index summary calculator caution clock compressed.data
# diskette display fax mail.in mail.out mouse printer tn3270 trash uuencoded.document

# Returns the widget pathname which has the icon image loaded

# Application should override these
proc HMload_icon {state win icon {href {}}} {
	puts "==> load icon \"$icon\""
}

proc HMset_icon {state win icon} {
	puts "==> set icon \"$icon\""
}

# list element tags

proc HMtag_ol {state win param text} {
	upvar #0 HN$win winvar
	set winvar(count$winvar(level)) 0
}

proc HMtag_ul {state win param text} {
	upvar #0 HN$win winvar
	catch {unset winvar(count$winvar(level))}
}

proc HMtag_menu {state win param text} {
	upvar #0 HN$win winvar
	set winvar(menu) ->
	set winvar(compact) 1
}

proc HMtag_/menu {state win param text} {
	upvar #0 HN$win winvar
	catch {unset winvar(menu)}
	catch {unset winvar(compact)}
}

# Definition lists are managed by dl & dd pushing the indent
# stack and dt popping it.

proc HMtag_dt {state win param text} {
	upvar #0 $state var
	if {![info exists var(defnTerm)]} {
		HMstack HN$win "/" "indent \"\""
		set var(defnTerm) 1
	}
}

# Sometimes people don't balance their <dt>'s and <dd>'s
# If we get a <dd> without a <dt> then pop the stack

proc HMtag_dd {state win param text} {
	upvar #0 $state var
	if {![info exists var(defnTerm)]} {
		HMstack HN$win "/" "indent \"\""
	}
	catch {unset var(defnTerm)}
}

proc HMtag_/dl {state win param text} {
	upvar #0 $state var
	if {[info exists var(defnTerm)]} {
		catch {HMstack HN$win "/" "indent \"\""}
	}
	catch {unset var(defnTerm)} err
}

proc HMtag_li {state win param text} {
	upvar #0 $state var
	upvar #0 HN$win winvar
	set level $winvar(level)
	incr level -1

	if {[HMextract_param $param dingbat]} {
		$win insert $var(S_insert) "\t" "mark [lindex $winvar(list) end] indent$level"
		HMload_icon $state $win [string tolower $dingbat]
		$win insert $var(S_insert) "\t"
	} else {
		set x [string index $var(S_symbols)+-+-+-+- $level]
		set b bullet
		catch {set x [incr winvar(count$level)]; set b ""}
		catch {set x $winvar(menu); set b ""}
		$win insert $var(S_insert) \t$x\t "mark $b [lindex $winvar(list) end] indent$level $winvar(font)"
	}
}

# Manage hypertext "anchor" links.  A link can be either a source (href)
# a destination (name) or both.  If its a source, register it via a callback,
# and set its default behavior.  If its a destination, check to see if we need
# to go there now, as a result of a previous HMgoto request.  If so, schedule
# it to happen with the closing </a> tag, so we can highlight the text up to
# the </a>.  We also need to remember which window the destination occurred in.

proc HMtag_a {state win param text} {
	upvar #0 $state var
	upvar #0 HN$win winvar

	# Check for missing end tag on previous anchor
	if {[info exists var(Tref)] || [info exists var(Tname)]} {
		# Synthesize the end tag
		HMtag_/a $state $win {} {}
	}

	# a source

	if {[HMextract_param $param href]} {
		# check the relation
		set rel {}	;# default is hypertext link
		HMextract_param $param rel
		if {[string tolower $rel] == "embed"} {
			# assume it's an applet
			HMload_applet $state $win $href hyperpage
		} else {
			set var(Tref) [list L:$href]
			HMstack $state "" "Tlink link"
			HMlink_setup $state $win $href
		}
	}

	# a destination

	if {[HMextract_param $param name]} {
		set var(Tname) [list N:$name]
		set var(N:$name) $win
		HMstack $state "" "Tanchor anchor"
		$win mark set N:$name "$var(S_insert) - 1 chars"
		$win mark gravity N:$name left
		if {[info exists var(goto)] && $var(goto) == $name} {
			unset var(goto)
			set var(going) $name
		}
	}
}

# The application should call here with the fragment name
# to cause the display to go to this spot.
# If the target exists, go there (and do the callback),
# otherwise schedule the goto to happen when we see the reference.
# In this case win is the main text window

proc HMgoto {state where {callback HMwent_to}} {
	upvar #0 $state var
	if {[info exists var(N:$where)]} {
		$var(N:$where) see N:$where
		update
		eval $callback $var(N:$where) [list $where]
		return 1
	} else {
		set var(goto) $where
		return 0
	}
}

# We actually got to the spot, so highlight it!
# This should/could be replaced by the application
# We'll flash it orange a couple of times.
# win is the window in which the element occurred, but
# this implementation has problems where the element
# is split over multiple windows.

proc HMwent_to {win where {count 0} {color orange}} {
	if {$count > 5} return
	catch {$win tag configure N:$where -foreground $color}
	update
	after 200 [list HMwent_to $win $where [incr count]  [expr {$color=="orange" ? "" : "orange"}]]
}

proc HMtag_/a {state win param text} {
	upvar #0 $state var
	if {[info exists var(Tref)]} {
		unset var(Tref)
		HMstack $state / "Tlink link"
	}

	# goto this link, then invoke the call-back.

	if {[info exists var(going)]} {
		$win yview N:$var(going)
		update
		HMwent_to $win $var(going)
		unset var(going)
	}

	if {[info exists var(Tname)]} {
		unset var(Tname)
		HMstack $state / "Tanchor anchor"
	}
}

#           Inline Images
# This interface is subject to change
# Most of the work is getting around a limitation of TK that prevents
# setting the size of a label to a widthxheight in pixels
#
# Images have the following parameters:
#    align:  top,middle,bottom
#    alt:    alternate text
#    ismap:  A clickable image map
#    src:    The URL link
# Netscape supports (and so do we)
#    width:  A width hint (in pixels)
#    height:  A height hint (in pixels)
#    border: The size of the window border

proc HMtag_img {state win param text} {
	global HMinsert_map HMalign_map
	upvar #0 $state var
	upvar #0 HN$win winvar

	set align [HMget_alignment $param bottom]		;# The spec isn't clear what the default should be
	set imgalign {}
	if {$align == "left"} {
		# Force a line break, text flows to right of graphic
		$win insert $var(S_insert) $HMinsert_map(br) "space $winvar(font)"
		set align bottom
		set imgalign left
	} elseif {$align == "right"} {
		# Right-align the graphic, text flows around to the left (not currently possible)
		HMstack $state "" "$HMalign_map(right)"
		set align bottom
		set imgalign right
	}

	# get alternate text
	set alt "<image>"
	HMextract_param $param alt
	set alt [HMmap_esc $alt]

	# get the border width
	set border 1
	HMextract_param $param border

	set tags [HMcurrent_tags $state $win]
	set link {}
	if {[set link [lindex $tags [lsearch -glob $tags L:*]]] != {}} {
		regsub L: $link {} link
		# This image is within an anchor
		set c button
		set sourceanchor 1
	} else {
		set c label
		set sourceanchor 0
	}

	# see if we have an image size hint
	# If so, make a frame the "hint" size to put the label in
	# otherwise just make the label
	set item $win.[HMgetUniqueNum $state]
	# catch {destroy $item}
	if {[HMextract_param $param width] && [HMextract_param $param height]} {
		frame $item -width $width -height $height -bd 0
		pack propagate $item 0
		set label $item.label
		$c $label
		pack $label -expand 1 -fill both
	} else {
		set label $item
		$c $label 
	}

	$label configure -fg orange -text $alt -bg $var(bgcolor) -highlightthickness 0
	catch {$label configure -bd $border} err
	$win window create $var(S_insert) -align $align -window $item -pady 2 -padx 2

	# add in all the current tags (this is overkill)
	foreach tag $tags {
		$win tag add $tag $item
	}

	# Pop the right-align tag if necessary
	if {$imgalign == "right"} {
		catch {HMstack $state / "$HMalign_map(right)"}
	}

	# set imagemap callbacks
	if {[HMextract_param $param ismap]} {
		global HMevents
		regsub -all {%} $link {%%} link2
		foreach i [array names HMevents] {
			bind $label <$i> "catch \{%W configure $HMevents($i)\}"
		}
		bind $label <ButtonRelease-1> "+HMlink_callback $state $win $link2?%x,%y"
	} elseif {$sourceanchor} {
		# The image will be in a button
		$label configure -command "HMlink_callback $state $win $link"
		# Callback to the application
		HMlink_window_setup $label $link
	}

	# now callback to the application
	set src ""
	HMextract_param $param src
	HMset_image $state $win $label $src
	return $label	;# used by the forms package for input_image types
}

proc HMget_alignment {param {default left}} {
	array set align_map {left left	top top    middle center    bottom baseline}
	set align $default
	HMextract_param $param align
	catch {set align $align_map([string tolower $align])}
	return $align
}

# The app needs to supply one of these
proc HMset_image {state win handle src} {
	HMgot_image $handle "can't get\n$src"
}
# and one of these
proc HMlink_window_setup {b link} {
	puts "link \"$link\" for button $b"
}

# When the image is available, the application should call back here.
# If we have the image, put it in the label, otherwise display the error
# message.  If we don't get a callback, the "alt" text remains.
# if we have a clickable image, arrange for a callback

proc HMgot_image {win image_error} {
	# if we're in a frame turn on geometry propogation
	if {[winfo name $win] == "label"} {
		pack propagate [winfo parent $win] 1
	}
	if {[catch {$win configure -image $image_error}]} {
		$win configure -image {}
		$win configure -text $image_error
	}
}

# Sample hypertext link callback routine - should be replaced by app
# This proc is called once for each <A> tag.
# Applications can overwrite this procedure, as required, or
# replace the HMevents array
#   win:   The name of the text widget to render into
#   href:  The HREF link for this <a> tag.

array set HMevents {
	Enter	{-borderwidth 2 -relief raised }
	Leave	{-borderwidth 2 -relief flat }
	1		{-borderwidth 2 -relief sunken}
	ButtonRelease-1	{-borderwidth 2 -relief raised}
}

# We need to escape any %'s in the href tag name so the bind command
# doesn't try to substitute them.

proc HMlink_setup {state win href} {
	global HMevents
	regsub -all {%} $href {%%} href2
	foreach i [array names HMevents] {
		eval {$win tag bind  L:$href <$i>}  \{$win tag configure \{L:$href2\} $HMevents($i)\}

#		eval $win tag bind L:$href <$i>  #			\{$win tag configure \{L:$href2\} $HMevents($i)\}
#		$win tag bind L:$href <$i>  #			"$win tag configure \{L:$href2\} $HMevents($i)"
	}
}

# generic link-hit callback
# This gets called upon button hits on hypertext links
# Applications are expected to supply ther own HMlink_callback routine
#   state: The global array for the hyperwindow
#   win:   The name of the text widget to render into
#   x,y:   The cursor position at the "click"

proc HMlink_hit {state win x y} {
	set tags [$win tag names @$x,$y]
	set link [lindex $tags [lsearch -glob $tags L:*]]
	# regsub -all {[^L]*L:([^ ]*).*}  $tags {\1} link
	regsub L: $link {} link
	if {$link == {}} return;
	HMlink_callback $state $win $link
}

# replace this!
#   win:   The name of the text widget to render into
#   href:  The HREF link for this <a> tag.

proc HMlink_callback {state win href} {
	puts "Got hit on $win, link $href"
}

#
# Table handling
#
# Each table cell is rendered in its own Text widget which has its own,
# separate state.  A stack is maintained of nested tables (the window stack
# is not sufficient).  The Text widget cannot size itself properly, so
# once the table has been finished the cells are resized.

# Simple port to the grid geometry manager from blt_table

proc HMtag_table {state win param text} {
	debug_puts [list HMtag_table $state $win $param $text]
	catch {HMrealtag_table $state $win $param $text} err
	debug_puts [list HMrealtag_table returned $err]
}

proc HMrealtag_table {state win param text} {
	upvar #0 $state var
	upvar #0 HN$win winvar

	if {[HMextract_param $param border]} {
		if {[info exists border] && $border == "0"} {
			set bdopt {}
		} else {set bdopt {-borderwidth 2 -relief groove}}
	} else {set bdopt {}}

	# Create a frame for sizing the table
	set tableframe $win.table[HMgetUniqueNum $state]
	catch {destroy $tableframe}
	eval frame $tableframe $bdopt
	HMwin_install $state $win $tableframe

	# Create frame for containing the table and push onto table stack
	set table [frame $tableframe.f -borderwidth 0 -background $var(bgcolor)]
	set parent [lindex $var(tables) end]
	lappend var(tables) $table
	upvar #0 HN$table tblvar
	pack $table
	pack propagate $tableframe 0

	if {[HMextract_param $param cols]} {
		# Use fixed layout algorithm.  Columns get equal width by default
		array set tblvar [list columns $cols layout fixed]
		for {set c 0} {$c < $cols} {incr c} {
			grid columnconfigure $table $c -weight 1.0
		}
	} else {set tblvar(layout) auto}

	# Initialise row and column counters, and other variables
	array set tblvar [list parent $parent  rowcntr 1 colcntr 0 colspec 0  in_table_cell 0 bdopt $bdopt]

	# Set width as specified
	set width 100%
	HMextract_param $param width
	set tblvar(tablewidth) $width
	if {[regexp {(.+)%} $width all w]} {
		# BUG: doesn't take right margin into account
		set winwidth [expr [winfo width $win] - 2 * [$win cget -borderwidth] - 4 - [winfo pixels $win $winvar(level)c]]
		debug_puts [list width specified as percentage - $w]
		debug_puts [list avail width $winwidth, set $tableframe width to [expr $winwidth * $w / 100] pixels]
		$tableframe configure -width [expr $winwidth * $w / 100]
	} else {
		debug_puts [list width specified as $width]
		catch {$tableframe configure -width [HMtable_stdSize $width]}
	}
	pack configure $table -fill both -expand true
}

# Column groups not yet implemented
proc HMtag_colgroup {state win param text} {}

proc HMtag_col {state win param text} {
	debug_puts [list HMtag_col $state $win $param $text]
	upvar #0 $state var
	upvar #0 HN[set table [lindex $var(tables) end]] tblvar

	set span 1
	HMextract_param $param span

	if {[HMextract_param $param width]} {
		if {[regexp {(.*)\*$} $width all val]} {
			set opt -weight
			set cmd2 HMnoop; set opt2 {}
		} else {
			set opt -minsize 
			set val [HMtable_stdSize $val]
			set cmd2 {grid columnconfigure}
			set opt2 [list -weight 0]
		}
		for {} {$span} {incr tblvar(colspec); incr span -1} {
			eval $cmd2 $table $tblvar(colspec) $opt2
			grid columnconfigure $table $tblvar(colspec) $opt $val
		}
	} else {incr tblvar(colspec) $span}
}

# Convert HTML standard sizes to Tk screen measurements
# NB. Tk does not provide a convenient way to get the default height
# of a font, so we assume 12 pixels

proc HMtable_stdSize {size} {
	regexp {([0-9]+)([^0-9]*)} $size all measure units
	switch $units {
		pt {return [expr $measure / 72.27]i}
		pi {return [expr $measure / (72.27 * 12)]i}
		in {return ${measure}i}
		cm {return ${measure}c}
		mm {return ${measure}m}
		em {return [expr $measure * 12]}
		px -
		default {return $measure}
	}
}

proc HMtag_/table {state win param text} {
	debug_puts [list HMtag_/table $state $win $param $text]
	upvar #0 $state var
	upvar #0 HN[set table [lindex $var(tables) end]] tblvar

	# If the previous cell is still "open" then close it
	if {$tblvar(in_table_cell)} {
		HMtag_/td $state $win $param $text
	}

	# Make the caption span all columns
	lassign [grid size $table] cols rows
	if {[winfo exists $table.caption]} {
		grid configure $table.caption -columnspan $cols
	}

	# Pop table stack
	set var(tables) [lrange $var(tables) 0 [expr [llength $var(tables)] - 2]]

	# When the entire table has been rendered, fix up the cell geometries
	if {$tblvar(layout) == "auto"} {
		# Initialise columns to equal weighting
		for {set i $tblvar(colspec)} {$i < $cols} {incr i} {
			grid columnconfigure $table $i -weight 1.0
		}
	}
	if {$var(tables) == {}} {HMtable_setup_geom $state $table}

	# Can't remove this yet, since resizing code uses it
	#unset tblvar
}

proc HMtable_setup_geom {state table} {
	debug_puts [list HMtable_setup_geom $state $table]
	foreach cell [winfo children $table] {
		HMcell_setupGeom $state $cell.c $table
		regsub -all "($cell\\.c\\.\[^t\]\[^ \]*)" [winfo children $cell.c] {} tables
		set tables [string trim $tables]
		foreach leaf $tables {HMtable_setup_nested $state $leaf.f}
	}
}

# Set table width according to available space for nested tables

proc HMtable_setup_nested {state table} {
	debug_puts [list HMtable_setup_nested $state $table]
	catch {HMrealtable_setup_nested $state $table} err
	debug_puts [list HMrealtable_setup_nested returned $err]
}

proc HMrealtable_setup_nested {state table} {
	upvar #0 HN$table tblvar
	update	;# Necessary to make 'winfo width' accurate
	set tableframe [winfo parent $table]
	if {[regexp {(.+)%} $tblvar(tablewidth) all w]} {
		# BUG: doesn't take margins into account
		set win [winfo parent [winfo parent $table]]
		set winwidth [expr [winfo width $win] - 2 * [$win cget -borderwidth] - 4]
		debug_puts [list width specified as percentage - $w]
		debug_puts [list $win avail width $winwidth, set $tableframe width to [expr $winwidth * $w / 100] pixels]
		$tableframe configure -width [expr $winwidth * $w / 100]
	} else {
		debug_puts [list width specified as $width]
		catch {$tableframe configure -width [HMtable_stdSize $width]}
	}

	HMtable_setup_geom $state $table
}

proc HMcell_setupGeom {state win table} {
	debug_puts [list HMcell_setupGeom $state $win $table]
	catch {HMrealcell_setupGeom $state $win $table} err
	debug_puts [list HMrealcell_setupGeom returned $err]
}

proc HMrealcell_setupGeom {state win table} {
	upvar #0 $state var
	upvar #0 HN$win winvar
	upvar #0 HN$table tblvar

	if {$tblvar(layout) == "fixed"} {
		debug_puts [list fixed layout for table $table]
	} else {
		debug_puts [list auto layout for table $table]
		# Work out min/max values
		# Assume that a character is 8 pixels wide
		set min 0
		set max 0; set line 0
		foreach {key value} [$win dump -text -window 1.0 end] {
			switch $key {
				text {
					if {$value == "\n"} {
						set max [max $max $line]
						set line 0
					} else {
						regsub -all \" $value X value
						regsub -all \{ $value X value
						regsub -all \} $value X value
						regsub -all \\\\ $value X value
						incr line [expr [string length $value] * 8]
						foreach word $value {
							set min [max $min [expr [string length $word] * 8]]
						}
					}
				}
				window {
					# Ignore nested tables in calc.
					if {![regexp {table.*} [winfo name [winfo parent $value]]]} {
						set min [max $min [winfo reqwidth $value]]
						incr line [winfo reqwidth $value]
					}
				}
			}
		}
		set max [max $max $line]	;# catch the last line
		debug_puts [list min is $min, max is $max]
		regexp {cell_([0-9]+)_([0-9]+)} [winfo name [winfo parent $win]] all row col
		debug_puts [list window $win is in column $col]
		if {$min > [grid columnconfigure $table $col -minsize]} {
			debug_puts [list setting column $col minsize to $min]
			grid columnconfigure $table $col -minsize $min
		}
		if {$min > 0 && [expr double($max) / double($min)] > [grid columnconfigure $table $col -weight]} {
			debug_puts [list setting column $col weighting to [expr double($max) / double($min)]]
			grid columnconfigure $table $col -weight [expr double($max) / double($min)]
		}
	}
	HMtable_setGeom $state $table
	UTIL_resetFitText $win
	UTIL_fitTextVertical $win 1 [list HMtable_setGeom $state $table]
}

proc HMtable_setGeom {state table} {
	debug_puts [list HMtable_setGeom $state $table]
	upvar #0 HN$table tblvar

	# Set the height appropriately for this cell
	[winfo parent $table] configure -height [winfo reqheight $table]
	debug_puts [list [winfo parent $table] configure -height [winfo reqheight $table]]
}

# The caption is rendered into its own text widget, which is then considered
# to be on its own row

proc HMtag_caption {state win param text} {
	debug_puts [list HMtag_caption $state $win $param $text]
	upvar #0 $state var
	upvar #0 HN[set table [lindex $var(tables) end]] tblvar

	set caption $table.caption
	catch {destroy $caption}
	eval text $caption -takefocus 0 -width 1 -height 1 $tblvar(bdopt) -highlightthickness 0

	# Enter into the table
	grid $caption -row 0 -sticky news

	# push caption onto the state
	HMpush_win $state $caption
}

proc HMtag_/caption {state win param text} {
	debug_puts [list HMtag_/caption $state $win $param $text]
	$win configure -wrap none
	UTIL_resetFitText $win
	UTIL_fitTextVertical $win 1 {}

	# Pop state
	HMpop_win $state
}

proc HMtag_tr {state win param text} {
	upvar #0 $state var
	upvar #0 HN[lindex $var(tables) end] tblvar

	# If the previous cell is still "open" then close it
	if {$tblvar(in_table_cell)} {
		HMtag_/td $state $win $param $text
	}

	# Increment the row counter
	incr tblvar(rowcntr)
	# Reset the column counter
	set tblvar(colcntr) 0
}

proc HMtag_th {state win param text} {
	global HMtag_map
	HMtable_cell $state $win $param $text
	catch {HMstack $state $cell {} "$HMtag_map(center)"}
}

proc HMtag_td {state win param text} {
	HMtable_cell $state $win $param $text
}

proc HMtable_cell {state win param text} {
	upvar #0 $state var
	upvar #0 HN[set table [lindex $var(tables) end]] tblvar
	debug_puts [list HMtable_cell $state $win $param $text in table $table]

	# If the previous cell is still "open" then close it
	if {$tblvar(in_table_cell)} {
		HMtag_/td $state $win $param $text
	}
	set tblvar(in_table_cell) 1

	# Determine various parameters
	set rowspan 1
	HMextract_param $param rowspan
	set colspan 1
	HMextract_param $param colspan

	# Put the cell in a frame to control size pixel-wise
	set cell $table.cell_$tblvar(rowcntr)_$tblvar(colcntr)
	catch {destroy $cell}
	eval frame $cell $tblvar(bdopt)
	eval text $cell.c -takefocus 0 -width 1 -height 1 -borderwidth 0 -highlightthickness 0
	pack $cell.c -fill both -expand 1

	# Enter into the table
	grid $cell -row $tblvar(rowcntr) -column $tblvar(colcntr)  -sticky news -rowspan $rowspan -columnspan $colspan

	# Advance the column counter
	incr tblvar(colcntr)

	# push cell onto the state
	HMpush_win $state $cell.c
	# Initialise it
	HMrender $var(basewin) hmstart {} {} {}
}

# there isn't really a </TD> tag!

# The cell's width is fixed after its height has been
# determined.

proc HMtag_/td {state win param text} {
	debug_puts [list HMtag_/td $state $win $param $text]
	upvar #0 $state var
	upvar #0 HN[set table [lindex $var(tables) end]] tblvar

	HMrender $var(basewin) hmstart / {} {}

	# Pop window stack
	HMpop_win $state
	# Popping window stack does not pop table stack
	set tblvar(in_table_cell) 0
}

# extract a value from parameter list (this needs a re-do)
# returns "1" if the keyword is found, "0" otherwise
#   param:  A parameter list.  It should alredy have been processed to
#           remove any entity references
#   key:    The parameter name
#   val:    The variable to put the value into (use key as default)

proc HMextract_param {param key {val ""}} {

	if {$val == ""} {
		upvar $key result
	} else {
		upvar $val result
	}
    set ws "    \n\r"
 
    # look for name=value combinations.  Either (') or (") are valid delimeters
    if {
      [regsub -nocase [format {.*%s[%s]*=[%s]*"([^"]*).*} $key $ws $ws] $param {\1} value] ||
      [regsub -nocase [format {.*%s[%s]*=[%s]*'([^']*).*} $key $ws $ws] $param {\1} value] ||
      [regsub -nocase [format {.*%s[%s]*=[%s]*([^%s]+).*} $key $ws $ws $ws] $param {\1} value] } {
        set result $value
        return 1
    }

	# now look for valueless names
	# I should strip out name=value pairs, so we don't end up with "name"
	# inside the "value" part of some other key word - some day
	
	set bad \[^a-zA-Z\]+
	if {[regexp -nocase  "$bad$key$bad" -$param-]} {
		return 1
	} else {
		return 0
	}
}

# These next three routines manage the display state of the page.

# Push or pop tags to/from stack.
# Each orthogonal text property has its own stack, stored as a list.
# The current (most recent) tag is the last item on the list.
# Push is {} for pushing and {/} for popping
# Properties may belong to either the hyperwindow or to a window.

proc HMstack {state push list} {
	upvar #0 $state var
	array set tags $list
	if {$push == ""} {
		foreach tag [array names tags] {
			lappend var($tag) $tags($tag)
		}
	} else {
		foreach tag [array names tags] {
			# set cnt [regsub { *[^ ]+$} $var($tag) {} var($tag)]
			set var($tag) [lreplace $var($tag) end end]
		}
	}
}

# extract set of current text tags
# tags starting with T map directly to text tags, all others are
# handled specially.  There is an application callback, HMset_font
# to allow the application to do font error handling
# This is complicated by some tags being managed in the hyperwindow
# state, and some in the window state

proc HMcurrent_tags {state win} {
	upvar #0 $state var
	upvar #0 HN$win winvar

	set font font
	foreach i {family size weight style} {
		set $i [lindex $var($i) end]
		append font :[set $i]
	}
	set xfont [HMx_font $family $size $weight $style $var(S_adjust_size)]
	HMset_font $win $font $xfont
	set indent [llength $winvar(indent)]
	#incr indent -1
	lappend tags $font indent$indent
	foreach tag [array names var T*] {
		lappend tags [lindex $var($tag) end]	;# test
	}
	foreach tag [array names winvar T*] {
		lappend tags [lindex $winvar($tag) end]
	}
	set winvar(font) $font
	set winvar(xfont) [$win tag cget $font -font]
	set winvar(level) $indent
	return $tags
}

# allow the application to do do better font management
# by overriding this procedure

proc HMset_font {win tag font} {
	catch {$win tag configure $tag -font $font; $win tag lower $tag mark}
}

# generate an X font name
proc HMx_font {family size weight style {adjust_size 0}} {
	catch {incr size $adjust_size}
	return "-*-$family-$weight-$style-normal-*-*-${size}0-*-*-*-*-*-*"
}

# Optimize HMrender (hee hee)
# This is experimental

proc HMoptimize {} {
	regsub -all "\n\[ 	\]*#\[^\n\]*" [info body HMrender] {} body
	regsub -all ";\[ 	\]*#\[^\n]*" $body {} body
	regsub -all "\n\n+" $body \n body
	proc HMrender {cookie tag not param text} $body
}
############################################
# Turn HTML into TCL commands
#   html    A string containing an html document
#   incomplete	A variable to store leftover HTML
#   cmd		A command to run for each html tag found
#   start	The name of the dummy html start/stop tags

# The routine populates the given array with the HTML elements, indexed by a 
# number.  Retrieving elements from an array will be faster than a list.
# A sentinel value, "\win\" is included as a placeholder for application-
# specific arguments.
# The routine returns the last index used.

proc HMparse_html {html parsedArray startElement numTags incomplete {cmd HMtest_parse} {start hmstart}} {
	upvar #0 $parsedArray var
	upvar $incomplete incvar

	regsub -all \{ $html {\&ob;} html
	regsub -all \} $html {\&cb;} html
	regsub -all \\\\ $html {\&bsl;} html
	set w " \t\r\n"	;# white space
	proc HMcl x {return "\[$x\]"}
	set exp <(/?)([HMcl ^$w>]+)[HMcl $w]*([HMcl ^>]*)>
	set sub "\}\n[list $cmd] {\\2} {\\1} {\\3} \{"
	regsub -all $exp $html $sub html
	# Check for incomplete final tag
	set incvar {}
	if {[regexp {[^<]*(<.*)} [lindex "\{$html\}" end] all incvar]} {
		set html [string range $html 0 [expr [string last "<" $html] - 1]]
	}

	for {set i 0; set group {}} {$i < $numTags} {incr i} {
		append group "a$i c$i d$i e$i f$i\n"
		regsub -all {(a[0-9]+)} $group {$\1 \\\\win\\\\} subgroup
		regsub -all {([b-z]+[0-9]+)} $subgroup {{$\1}} subgroup
	}
	foreach $group "[list $cmd] {$start} {} {} \{ $html \}" {
		set var([incr startElement]) [subst $subgroup]
	}
	return $startElement
}

# Just consume arguments and do nothing
proc HMnoop {args} {}

proc HMtest_parse {command tag slash text_after_tag} {
	puts "==> $command $tag $slash $text_after_tag"
}

# Convert multiple white space into a single space

proc HMzap_white {data} {
	regsub -all "\[ \t\r\n\]+" $data " " data
	return $data
}

proc HMtranslate_cr {data} {
	regsub -all "\r+" $data "\n" data
	return $data
}

# find HTML escape characters of the form &xxx;

proc HMmap_esc {text} {
	if {![regexp & $text]} {return $text}
	regsub -all {([][$\\])} $text {\\\1} new
	regsub -all {&#([0-9][0-9]?[0-9]?);?}  $new {[format %c [scan \1 %d tmp;set tmp]]} new
	regsub -all {&([a-zA-Z]+);?} $new {[HMdo_map \1]} new
	return [subst $new]
}

# convert an HTML escape sequence into character

proc HMdo_map {text {unknown ?}} {
	global HMesc_map
	set result $unknown
	catch {set result $HMesc_map($text)}
	return $result
}

# table of escape characters (ISO latin-1 esc's are in a different table)

array set HMesc_map {
   lt <   gt >   amp &   quot \"   copy \xa9
   reg \xae   ob \x7b   cb \x7d   nbsp \xa0
   bsl \\
}
#############################################################
# ISO Latin-1 escape codes

array set HMesc_map {
	nbsp \xa0 iexcl \xa1 cent \xa2 pound \xa3 curren \xa4
	yen \xa5 brvbar \xa6 sect \xa7 uml \xa8 copy \xa9
	ordf \xaa laquo \xab not \xac shy \xad reg \xae
	hibar \xaf deg \xb0 plusmn \xb1 sup2 \xb2 sup3 \xb3
	acute \xb4 micro \xb5 para \xb6 middot \xb7 cedil \xb8
	sup1 \xb9 ordm \xba raquo \xbb frac14 \xbc frac12 \xbd
	frac34 \xbe iquest \xbf Agrave \xc0 Aacute \xc1 Acirc \xc2
	Atilde \xc3 Auml \xc4 Aring \xc5 AElig \xc6 Ccedil \xc7
	Egrave \xc8 Eacute \xc9 Ecirc \xca Euml \xcb Igrave \xcc
	Iacute \xcd Icirc \xce Iuml \xcf ETH \xd0 Ntilde \xd1
	Ograve \xd2 Oacute \xd3 Ocirc \xd4 Otilde \xd5 Ouml \xd6
	times \xd7 Oslash \xd8 Ugrave \xd9 Uacute \xda Ucirc \xdb
	Uuml \xdc Yacute \xdd THORN \xde szlig \xdf agrave \xe0
	aacute \xe1 acirc \xe2 atilde \xe3 auml \xe4 aring \xe5
	aelig \xe6 ccedil \xe7 egrave \xe8 eacute \xe9 ecirc \xea
	euml \xeb igrave \xec iacute \xed icirc \xee iuml \xef
	eth \xf0 ntilde \xf1 ograve \xf2 oacute \xf3 ocirc \xf4
	otilde \xf5 ouml \xf6 divide \xf7 oslash \xf8 ugrave \xf9
	uacute \xfa ucirc \xfb uuml \xfc yacute \xfd thorn \xfe
	yuml \xff
}

##########################################################
# html forms management commands

# As each form element is located, it is created and rendered.  Additional
# state is stored in a form specific global variable to be processed at
# the end of the form, including the "reset" and "submit" options.
# Remember, there can be multiple forms existing on multiple pages.  When
# HTML tables are added, a single form could be spread out over multiple
# text widgets, which makes it impractical to hang the form state off the
# HN$win structure.  We don't need to check for the existance of required
# parameters, we just "fail" and get caught in HMrender

# An applet may be attached to the form.  The applet is given the starting
# index of the form.  The form_id is stored for the applet (but not visible
# to the applet).  The applet is given the widget pathname of each element
# in the form, and is informed when the </form> is seen.  There are callins
# after the form is reset and just before it is submitted.

# This causes line breaks to be preserved in the inital values
# of text areas
array set HMwintag_map {
	textarea    {fill 0}
}

##########################################################
# html isindex tag.  Although not strictly forms, they're close enough
# to be in this file

# is-index forms
# make a frame with a label, entry, and submit button

proc HMtag_isindex {state win param text} {
	upvar #0 $state var
	upvar #0 HN$win winvar

	set item $win.[HMgetUniqueNum $state]
	if {[winfo exists $item]} {
		destroy $item
	}
	frame $item -relief ridge -bd 3 -background $var(bgcolor)
	set prompt "Enter search keywords here"
	HMextract_param $param prompt
	label $item.label -text [HMmap_esc $prompt] -font $winvar(xfont)  -background $var(bgcolor) -foreground $var(fgcolor)
	entry $item.entry -background $var(bgcolor) -foreground $var(fgcolor)
	bind $item.entry <Return> "$item.submit invoke"
	button $item.submit -text search -font $winvar(xfont) -command  [format {HMsubmit_index %s %s {%s} [HMmap_reply [%s get]]}  $state $win $param $item.entry]  -background $var(bgcolor) -foreground $var(fgcolor)
	pack $item.label -side top
	pack $item.entry $item.submit -side left

	# insert window into text widget

	$win insert $var(S_insert) \n isindex
	HMwin_install $state $win $item
	$win insert $var(S_insert) \n isindex
	bind $item <Visibility> {focus %W.entry}
}

# This is called when the isindex form is submitted.
# The default version calls HMlink_callback.  Isindex tags should either
# be deprecated, or fully supported (e.g. they need an href parameter)

proc HMsubmit_index {state win param text} {
	HMlink_callback $state $win ?$text
}

# initialize form state.  All of the state for this form is kept
# in a global array whose name is stored in the form_id field of
# the main window array.
# Parameters: ACTION, METHOD, ENCTYPE

proc HMtag_form {state win param text} {
	upvar #0 $state var

	# create a global array for the form
	set id HM$win.form[HMgetUniqueNum $state]
	upvar #0 $id form

	# missing /form tag, simulate it
	if {[info exists var(form_id)]} {
		# puts "Missing end-form tag !!!! $var(form_id)"
		HMtag_/form $state $win {} {}
	}
	catch {unset form}
	set var(form_id) $id

	set form(param) $param		;# form initial parameter list
	set form(reset) ""			;# command to reset the form
	set form(reset_button) ""	;# list of all reset buttons
	set form(submit) ""			;# command to submit the form
	set form(submit_button) ""	;# list of all submit buttons

	# If the SCRIPT attribute is specified, load the applet and attach
	# it to the form

	if {[HMextract_param $param script]} {
		HMload_applet $state $win $script form $id
	}
}

# Application should override this procedure
proc HMload_applet {state win script level {formID {}}} {
	puts "==> HMload_applet $state $win $script $level $formID"
}

# Where we're done try to get all of the state into the widgets so
# we can free up the form structure here.  Unfortunately, we can't!

proc HMtag_/form {state win param text} {
	upvar #0 $state var
	upvar #0 $var(form_id) form

	# make submit button entries for all radio buttons
	foreach name [array names form radio_*] {
		regsub radio_ $name {} name
		lappend form(submit) [list $name \$form(radio_$name)]
	}

	# process the reset button(s)

	append form(reset) "; HMreset_form $state $win $var(form_id)"	;# Allow applet callins
	foreach item $form(reset_button) {
		$item configure -command $form(reset)
	}

	# no submit button - add one
	if {$form(submit_button) == ""} {
		HMinput_submit $state $win {}
	}

	# process the "submit" command(s)
	# each submit button could have its own name,value pair

	foreach item $form(submit_button) {
		set submit $form(submit)
		catch {lappend submit $form(submit_$item)}
		$item configure -command   [list HMsubmit_button $state $win $var(form_id) $form(param)  $submit]
	}

	# Callin to the applet to let it know we're done.
	HMapplet_/form $state $win $var(form_id)

	# unset all unused fields here
	unset form(reset) form(submit) form(reset_button) form(submit_button)
	unset var(form_id)
}

# Application should override this
proc HMapplet_/form {state win form_id} {
	puts "==> end of form, state $state, window $win, form $form_id,  at index [$win index {end - 1 char}]"
}

###################################################################
# handle form input items
# each item type is handled in a separate procedure
# Each "type" procedure needs to:
# - create the window
# - initialize it
# - add the "submit" and "reset" commands onto the proper Q's
#   "submit" is subst'd
#   "reset" is eval'd
# - return the name of the new window

proc HMtag_input {state win param text} {
	upvar #0 $state var

	set type text	;# the default
	HMextract_param $param type
	set type [string tolower $type]
	set name {}; HMextract_param $param name
	set value {}; HMextract_param $param value
	# Pass name & value to proc to avoid need to reparse attributes
	if {[catch {HMinput_$type $state $win $param} err]} {
		#puts stderr $err
	} else {
		# Inform the applet of the new window
		HMapplet_item $state $win $var(form_id) $type $name $value $err
	}
}

# input type=text
# parameters NAME (reqd), MAXLENGTH, SIZE, VALUE

proc HMinput_text {state win param {show {}}} {
	upvar #0 $state var
	upvar #0 $var(form_id) form

	# make the entry
	HMextract_param $param name		;# required
	set item $win.input_text,[HMgetUniqueNum $state]
	set size 20; HMextract_param $param size
	set maxlength 0; HMextract_param $param maxlength
	entry $item -width $size -show $show -background $var(bgcolor) -foreground $var(fgcolor)

	# set the initial value
	set value ""; HMextract_param $param value
	$item insert 0 $value
		
	# insert the entry
	HMwin_install $state $win $item

	# set the "reset" and "submit" commands
	append form(reset) ";$item delete 0 end;$item insert 0 [list $value]"
	lappend form(submit) [list $name "\[$item get]"]

	# handle the maximum length (broken - no way to cleanup bindtags state)
	if {$maxlength} {
		bindtags $item "[bindtags $item] max$maxlength"
		bind max$maxlength <KeyPress> "%W delete $maxlength end"
	}

	return $item
}

# password fields - same as text, only don't show data
# parameters NAME (reqd), MAXLENGTH, SIZE, VALUE

proc HMinput_password {state win param} {
	HMinput_text $state $win $param *
}

# checkbuttons are missing a "get" option, so we must use a global
# variable to store the value.
# Parameters NAME, VALUE, (reqd), CHECKED

proc HMinput_checkbox {state win param} {
	upvar #0 $state var
	upvar #0 $var(form_id) form

	HMextract_param $param name
	HMextract_param $param value

	# Set the global variable, don't use the "form" alias as it is not
	# defined in the global scope of the button
	set uniqnum [HMgetUniqueNum $state]
	set variable $var(form_id)(check_$uniqnum)	
	set item $win.input_checkbutton,$uniqnum
	checkbutton $item -variable $variable -off {} -on $value -text "  "  -background $var(bgcolor) -foreground $var(fgcolor)
	if {[HMextract_param $param checked]} {
		$item select
		append form(reset) ";$item select"
	} else {
		append form(reset) ";$item deselect"
	}

	HMwin_install $state $win $item
	lappend form(submit) [list $name \$form(check_$uniqnum)]
	return $item
}

# radio buttons.  These are like check buttons, but only one can be selected

proc HMinput_radio {state win param} {
	upvar #0 $state var
	upvar #0 $var(form_id) form

	HMextract_param $param name
	HMextract_param $param value

	set first [expr ![info exists form(radio_$name)]]
	set variable $var(form_id)(radio_$name)
	set variable $var(form_id)(radio_$name)
	set item $win.input_radiobutton,[HMgetUniqueNum $state]
	radiobutton $item -variable $variable -value $value -text " " -background $var(bgcolor) -foreground $var(fgcolor)

	HMwin_install $state $win $item

	if {$first || [HMextract_param $param checked]} {
		$item select
		append form(reset) ";$item select"
	} else {
		append form(reset) ";$item deselect"
	}

	# do the "submit" actions in /form so we only end up with 1 per button grouping
	# contributing to the submission

	return $item
}

# hidden fields, just append to the "submit" data
# params: NAME, VALUE (reqd)

proc HMinput_hidden {state win param} {
	upvar #0 $state var
	upvar #0 $var(form_id) form
	HMextract_param $param name
	set value ""; HMextract_param $param value
	lappend form(submit) [list $name $value]
	return {}
}

# handle input images.  The spec isn't very clear on these, so I'm not
# sure its quite right
# Use std image tag, only set up our own callbacks
#  (e.g. make sure ismap isn't set)
# params: NAME, SRC (reqd) ALIGN

proc HMinput_image {state win param} {
	upvar #0 $state var
	upvar #0 $var(form_id) form
	HMextract_param $param name
	set name		;# barf if no name is specified
	set item [HMtag_img $state $win $param {}]
	$item configure -relief raised -bd 2 -bg blue

	# make a dummy "submit" button, and invoke it to send the form.
	# We have to get the %x,%y in the value somehow, so calculate it during
	# binding, and save it in the form array for later processing

	set submit $win.dummy_submit,[HMgetUniqueNum $state]
	if {[winfo exists $submit]} {
		destroy $submit
	}
	button $submit	-takefocus 0;# this never gets mapped!
	lappend form(submit_button) $submit
	set form(submit_$submit) [list $name $name.\$form(X).\$form(Y)]
	
	$item configure -takefocus 1
	bind $item <FocusIn> "catch \{$win see $item\}"
	bind $item <1> "$item configure -relief sunken"
	bind $item <Return> "
		set $var(form_id)(X) 0
		set $var(form_id)(Y) 0
		$submit invoke	
	"
	bind $item <ButtonRelease-1> "
		set $var(form_id)(X) %x
		set $var(form_id)(Y) %y
		$item configure -relief raised
		$submit invoke	
	"
	return $item
}

# Set up the reset button.  Wait for the /form to attach
# the -command option.  There could be more that 1 reset button
# params VALUE

proc HMinput_reset {state win param} {
	upvar #0 $state var
	upvar #0 $var(form_id) form

	set value reset
	HMextract_param $param value

	set item $win.input_reset,[HMgetUniqueNum $state]
	button $item -text [HMmap_esc $value]  -background $var(bgcolor) -foreground $var(fgcolor)
	HMwin_install $state $win $item
	lappend form(reset_button) $item
	return $item
}

# Set up the submit button.  Wait for the /form to attach
# the -command option.  There could be more that 1 submit button
# params: NAME, VALUE

proc HMinput_submit {state win param} {
	upvar #0 $state var
	upvar #0 $var(form_id) form

	HMextract_param $param name
	set value submit
	HMextract_param $param value
	set item $win.input_submit,[HMgetUniqueNum $state]
	button $item -text [HMmap_esc $value]  -background $var(bgcolor) -foreground $var(linkfg)
	HMwin_install $state $win $item
	lappend form(submit_button) $item
	# need to tie the "name=value" to this button
	# save the pair and do it when we finish the submit button
	catch {set form(submit_$item) [list $name $value]}
	return $item
}

#########################################################################
# selection items
# They all go into a list box.  We don't what to do with the listbox until
# we know how many items end up in it.  Gather up the data for the "options"
# and finish up in the /select tag
# params: NAME (reqd), MULTIPLE, SIZE 

proc HMtag_select {state win param text} {
	upvar #0 $state var
	upvar #0 $var(form_id) form

	HMextract_param $param name
	set size 5;  HMextract_param $param size
	set form(select_size) $size
	set form(select_name) $name
	set form(select_values) ""		;# list of values to submit
	if {[HMextract_param $param multiple]} {
		set mode multiple
	} else {
		set mode single
	}
	set item $win.select,[HMgetUniqueNum $state]
    frame $item -background $var(bgcolor)
    set form(select_frame) $item
	listbox $item.list -selectmode $mode -width 0 -exportselection 0  -background $var(bgcolor) -foreground $var(linkfg)
	HMwin_install $state $win $item
}

# select options
# The values returned in the query may be different from those
# displayed in the listbox, so we need to keep a separate list of
# query values.
#  form(select_default) - contains the default query value
#  form(select_frame) - name of the listbox's containing frame
#  form(select_values)  - list of query values
# params: VALUE, SELECTED

proc HMtag_option {state win param text} {
	upvar #0 $state var
	upvar #0 $var(form_id) form
	upvar $text data
	set frame $form(select_frame)

	# set default option (or options)
	if {[HMextract_param $param selected]} {
        lappend form(select_default) [$form(select_frame).list size]
    }
    set value [string trimright $data " \n"]
    $frame.list insert end $value
	HMextract_param $param value
	lappend form(select_values) $value
	set data ""
}
 
# do most of the work here!
# if SIZE>1, make the listbox.  Otherwise make a "drop-down"
# listbox with a label in it
# If the # of items > size, add a scroll bar
# This should probably be broken up into callbacks to make it
# easier to override the "look".

proc HMtag_/select {state win param text} {
	upvar #0 $state var
	upvar #0 $var(form_id) form
	set frame $form(select_frame)
	set size $form(select_size)
	set items [$frame.list size]

	# set the defaults and reset button
	append form(reset) ";$frame.list selection clear 0  $items"
	if {[info exists form(select_default)]} {
		foreach i $form(select_default) {
			$frame.list selection set $i
			append form(reset) ";$frame.list selection set $i"
		}
	} else {
		$frame.list selection set 0
		append form(reset) ";$frame.list selection set 0"
	}

	# set up the submit button. This is the general case.  For single
	# selections we could be smarter

	for {set i 0} {$i < $size} {incr i} {
		set value [format {[expr {[%s selection includes %s] ? {%s} : {}}]}  $frame.list $i [lindex $form(select_values) $i]]
		lappend form(submit) [list $form(select_name) $value]
	}
	
	# show the listbox - no scroll bar

	if {$size > 1 && $items <= $size} {
		$frame.list configure -height $items
		pack $frame.list

	# Listbox with scrollbar

	} elseif {$size > 1} {
		scrollbar $frame.scroll -command "$frame.list yview"   -orient v -takefocus 0
		$frame.list configure -height $size  -yscrollcommand "$frame.scroll set"
		pack $frame.list $frame.scroll -side right -fill y

	# This is a joke!

	} else {
		scrollbar $frame.scroll -command "$frame.list yview"   -orient h -takefocus 0
		$frame.list configure -height 1  -yscrollcommand "$frame.scroll set"
		pack $frame.list $frame.scroll -side top -fill x
	}

	# Inform the applet
	HMapplet_item $state $win $var(form_id) select $form(select_name) {} $frame

	# cleanup

	foreach i [array names form select_*] {
		unset form($i)
	}
}

# do a text area (multi-line text)
# params: COLS, NAME, ROWS (all reqd, but default rows and cols anyway)

proc HMtag_textarea {state win param text} {
	upvar #0 $state var
	upvar #0 $var(form_id) form
	upvar $text data

	set rows 5; HMextract_param $param rows
	set cols 30; HMextract_param $param cols
	HMextract_param $param name
	set item $win.textarea,[HMgetUniqueNum $state]
	frame $item -background $var(bgcolor)
	text $item.text -width $cols -height $rows -wrap none  -yscrollcommand "$item.scroll set" -padx 3 -pady 3  -background $var(bgcolor) -foreground $var(linkfg)
	scrollbar $item.scroll -command "$item.text yview"  -orient v
	$item.text insert 1.0 $data
	HMwin_install $state $win $item
	pack $item.text $item.scroll -side right -fill y
	lappend form(submit) [list $name "\[$item.text get 0.0 end]"]
	append form(reset) ";$item.text delete 1.0 end;  $item.text insert 1.0 [list $data]"
	set data ""

	# Inform the applet
	HMapplet_item $state $win $var(form_id) textarea $name {} $item
}

# procedure to install windows into the text widget
# - state: name of hyperwindow state array
# - win:  name of the text widget
# - item: name of widget to install

proc HMwin_install {state win item} {
	upvar #0 $state var
	upvar #0 HN$win winvar
	$win window create $var(S_insert) -window $item -align bottom
	$win tag add indent$winvar(level) $item
	set focus [expr {[winfo class $item] != "Frame"}]
	$item configure -takefocus $focus
	bind $item <FocusIn> "$win see $item"
}

#####################################################################
# Assemble and submit the query
# each list element in "stuff" is a name/value pair
# - The names are the NAME parameters of the various fields
# - The values get run through "subst" to extract the values
# - We do the user callback with the list of name value pairs

proc HMsubmit_button {state win form_id param stuff} {
	upvar #0 $state var
	upvar #0 $form_id form
	set query ""
	foreach pair $stuff {
		set value [subst [lindex $pair 1]]
		if {$value != ""} {
			set item [lindex $pair 0]
			lappend query $item $value
		}
	}
	# this is the user callback.
	HMsubmit_form $state $win $form_id $param $query
}

# sample user callback for form submission
# should be replaced by the application
# Sample version generates a string suitable for http
# The application may wish to have the applet process
# the query first

proc HMsubmit_form {state win form_id param query} {
	set result ""
	set sep ""
	foreach i $query {
		append result  $sep [HMmap_reply $i]
		if {$sep != "="} {set sep =} {set sep &}
	}
	puts $result
}

# sample user callback for form reset
# should be replaced by the application
# The application should call the form's applet

proc HMreset_form {state win form_id} {
	puts "==> reset form $state $win $form_id"
}
# do x-www-urlencoded character mapping
# The spec says: "non-alphanumeric characters are replaced by '%HH'"
 
set HMalphanumeric	a-zA-Z0-9	;# definition of alphanumeric character class
for {set i 1} {$i <= 256} {incr i} {
    set c [format %c $i]
    if {![string match \[$HMalphanumeric\] $c]} {
        set HMform_map($c) %[format %.2x $i]
    }
}

# These are handled specially
array set HMform_map {
    " " +   \n %0d%0a
}

# 1 leave alphanumerics characters alone
# 2 Convert every other character to an array lookup
# 3 Escape constructs that are "special" to the tcl parser
# 4 "subst" the result, doing all the array substitutions
 
proc HMmap_reply {string} {
    global HMform_map HMalphanumeric
    regsub -all \[^$HMalphanumeric\] $string {$HMform_map(&)} string
    regsub -all \n $string {\\n} string
    regsub -all \t $string {\\t} string
    regsub -all {[][{})\\]\)} $string {\\&} string
    return [subst $string]
}

# convert a x-www-urlencoded string int a a list of name/value pairs

# 1  convert a=b&c=d... to {a} {b} {c} {d}...
# 2, convert + to  " "
# 3, convert %xx to char equiv

proc HMcgiDecode {data} {
	set data [split $data "&="]
	foreach i $data {
		lappend result [cgiMap $i]
	}
	return $result
}

proc HMcgiMap {data} {
	regsub -all {\+} $data " " data
	
	if {[regexp % $data]} {
		regsub -all {([][$\\])} $data {\\\1} data
		regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $data  {[format %c 0x\1]} data
		return [subst $data]
	} else {
		return $data
	}
}




