# ---------------------------------------------------------------------
#  $Id: panel.tcl,v 1.51 1997/07/06 15:30:17 adabas Exp $
# ---------------------------------------------------------------------

# ---------------------------------------------------------------------
# createPanel: Creates a panel canvas with a lot of objects on it.
#              Note the traffic light near the middle.
# ---------------------------------------------------------------------
proc createPanel {} {
    global adabastcl_library options bitmap tkCompat

    option add *panel.edition      "Database Server" widgetDefault
    option add *panel.netscapeHelp False             widgetDefault

    image create photo panel -file [file join $adabastcl_library panel.gif]
    canvas .panel -highlightthickness 0 \
	    -width [image width panel] -height [image height panel]
    .panel create image 0 0 -image panel -anchor nw  -tag bg
    set width  [image width  panel]
    set height [image height panel]

    .panel create text  [expr {$width-80}] 12 \
	    -text "Adabas D" -fill black  -anchor n -tag adabasD
    .panel create text  [expr {$width-82}] 10 \
	    -text "Adabas D" -fill red -anchor n -tag adabasD
    .panel create text  [expr {$width-80}] 50 \
	    -text [option get .panel edition Edition] -fill red -tag edition

    if $tkCompat(fontCmd) {
	.panel itemconfigure adabasD -font "Times 34 bold"
	.panel itemconfigure edition -font "Times 14 {bold italic}"	
    } else {
	if [catch {.panel itemconfigure adabasD -font \
		-Adobe-Times-Bold-R-Normal-*-34-*-*-*-*-*-*-*}] {
	    catch {.panel itemconfigure adabasD -font \
		    -Adobe-Times-Bold-R-Normal-*-24-*-*-*-*-*-*-*}
	}
	catch {.panel itemconfigure edition -font \
		-Adobe-Times-Bold-I-Normal-*-14-*-*-*-*-*-*-*}
    }

    pack .panel
    wm title     . "Adabas D"
    wm resizable . 0 0
    update

    .panel create text 10 60                -fill orange -anchor w -tag msg
    .panel create text  7 [expr $height-10] -fill blue   -anchor w -tag dbState
    foreach {label color above} {
	Info blue  120
	Help green  85
	Quit red    50
    } {
	.panel create text [expr {$width-30}] [expr {$height-$above}] \
		-fill $color  -text $label -tag "$label $label-text"
	.panel create rect [expr {$width-60}] [expr {$height-$above-10}] \
		$width [expr {$height-$above+10}] \
		-fill "" -outline "" -tag $label
    }

    set lightX [expr {$width/2+30}]
    set lightY [expr {$height-83-30}]

    image create bitmap light -data $bitmap(light) -foreground MidnightBlue
    .panel create image $lightX $lightY -image light

    foreach {onColor offColor diffY} {
	red    firebrick -70
	yellow goldenrod -38
	green  DarkGreen  -6
    } {
	image create bitmap shade:$onColor -data $bitmap(shade) -foreground blue
	.panel create image $lightX [expr {$lightY+$diffY}] \
		-image shade:$onColor

	image create bitmap ring:$onColor -data $bitmap(ring) -foreground gray60
	.panel create image $lightX [expr {$lightY+$diffY+13}] \
		-image ring:$onColor

	image create bitmap bulb:$onColor -data $bitmap(bulb) -foreground $offColor
	.panel create image $lightX [expr {$lightY+$diffY+13}] \
		-image bulb:$onColor

	image create bitmap shine:$onColor -data $bitmap(shine) -foreground white
	.panel create image $lightX [expr {$lightY+$diffY+13}] \
		-image shine:$onColor -tag shine:$onColor
	.panel lower shine:$onColor

	.panel create oval [expr {$lightX-13}] [expr {$lightY+$diffY}]  \
		[expr {$lightX+13}] [expr {$lightY+$diffY+26}] \
		-fill "" -outline "" -tag $onColor
    }

    # On every system there should be Helvetica-bold the font to display texts.
    if $tkCompat(fontCmd) {
	foreach text {msg dbState Quit-text Info-text Help-text} {
	    .panel itemconfigure $text -font "Helvetica 14 bold"
	}
    }

    # Bind the commands to the corresponding items (texts ans traffic lights)
    # on the canvas. Also bind some mouse clicks.
    foreach comm {red yellow green Quit Info Help} {
	.panel bind $comm <Enter> {.panel config -cursor hand2}
	.panel bind $comm <Leave> {.panel config -cursor top_left_arrow}
    }
    .panel bind red    <1>   {changeStateTo offline; break}
    .panel bind yellow <1>   {changeStateTo cold;    break}
    .panel bind green  <1>   {changeStateTo warm;    break}
    .panel bind Quit   <1>   {quitPanel 1;           break}
    .panel bind Info   <1>   {infoWindow;            break}
    .panel bind Help   <1>   {callHtmlHelp panel;    break}
    bind  .panel <Control-c> {quitPanel 1;           break}
    bind  .panel <F1>        {callHtmlHelp panel;    break}
    bind  .panel <F2>        {editKnlDiag;           break}
    bind  .panel <F3>        {quitPanel 1;           break}
    focus .panel

    .panel lower light
    .panel lower shine

    set options(val:netscapeHelp) [string match {[Tt]*} \
	    [option get .panel netscapeHelp NetscapeHelp]]

    update
}

# ---------------------------------------------------------------------
# panelShowMsg: Displays the given message on the picture. Without
#               parameters it removes any message from the canvas.
# ---------------------------------------------------------------------
proc panelShowMsg {{msg ""}} {
    .panel itemconfigure msg -text $msg
    if [string match "*see*for errors*" $msg] {
	.panel itemconfigure Info-text -text "Diag"
	.panel bind Info <1> "editKnlDiag; break"
	diagBlink
    } else {
	.panel itemconfigure Info-text -text "Info"
	.panel bind Info <1> "infoWindow; break"
	diagBlink cancel
    }
    update
}

proc diagBlink {{colorIndex 0}} {
    global blinkId

    set colors {
	turquoise1 turquoise2 turquoise3 turquoise4
	blue1      blue2      blue3      blue4
    }

    if [catch [list lindex $colors $colorIndex] color] {
	catch {after cancel $blinkId}
	set color blue
    } else {
	if {[incr colorIndex] >= [llength $colors]} {set colorIndex 0}
	set blinkId [after 500 diagBlink $colorIndex]
    }
    .panel itemconfigure Info-text -fill $color
}

# ---------------------------------------------------------------------
# panelShowDBState: The procedure to switch the traffic light, so that
#                   it corresponds to the state of the databas server.
#                   It switches the correct light on, and activates the
#                   corresponding (dimmed) lights.
#                   It also displays the name and state of the database.
# ---------------------------------------------------------------------
proc panelShowDBState {} {
    global connect currVal

    foreach {onColor offColor} {
	red    firebrick
	yellow goldenrod
	green  DarkGreen
    } {
	ring:$onColor configure -foreground gray60
	bulb:$onColor configure -foreground $offColor
	.panel lower shine:$onColor
    }
    foreach entry {Offline Cold Warm} {
	confMenueEntry .m $entry -state normal
    }

    set currentState [dbState]

    switch $currentState {
	offline {set color red;    set entry  Offline}
	cold    {set color yellow; set entry Cold}
	warm    {set color green;  set entry Warm}
	default {
	    .panel itemconfigure dbState -text "$currentState"
	    error $currentState
	}
    }

    .panel itemconfigure dbState -text \
	    "Database $connect(serverdb) is $currentState"

    ring:$color configure -foreground white
    bulb:$color configure -foreground $color
    .panel raise shine:$color
    confMenueEntry .m $entry -state disabled
    
    set currVal(state) [string toupper $currentState]
    return $currentState
}

# ---------------------------------------------------------------------
# changeToState: The callback procedure for all colors of the panel
#                traffic light. The actual color will be given coded
#                as (warm|cold|offline) in parameter targetState.
# ---------------------------------------------------------------------
proc changeStateTo {targetState} {
    global connect extProgOut

    if {[catch panelShowDBState currentState]} return

    if {![string compare $currentState $targetState]} return

    switch $currentState {
	offline {
	    if [catch {externalProg "x_start $connect(serverdb)" \
		    "Starting '$connect(serverdb)'" cold} currentState] {
		if {[string first "already started" $extProgOut] < 0} return

		set    text "Database $connect(serverdb) already started or "
		append text "not stopped correctly.\nTo clean up after a "
		append text "crash before try to start again press Retry"
		switch [tk_messageBox -title "Clear?" -message $text \
			-icon question -type retrycancel -default cancel] {
		    retry {
			if [catch {externalProg "x_clear $connect(serverdb)" \
				"Clearing '$connect(serverdb)'" offline} \
				currentState] return
			if [catch {externalProg "x_start $connect(serverdb)" \
				"Starting '$connect(serverdb)'" cold} \
				currentState] return
		    }
		    cancel {
			return
		    }
		}
	    }
	}
	warm {
	    if ![string compare [autosaveShow] ENABLED] {
		catch {utilityCmd "AUTOSAVE END" "Disabling autosave"}
	    }
	    if [catch {utilityCmd shutdown \
				  "Shutdown '$connect(serverdb)'"} \
			currentState] return
	}
    }

    if ![string compare $currentState $targetState] return

    switch $targetState {
	warm {
	    catch {utilityCmd restart "Restarting '$connect(serverdb)'"}
	}
	offline {
	    catch {externalProg "x_stop $connect(serverdb)" \
				"Stopping '$connect(serverdb)'" offline}
	}
    }
}

# ---------------------------------------------------------------------
# externalProg: Execute the given external program while displaying an
#               info message.
#               Return the resulting database state (offline|cold).
# ---------------------------------------------------------------------
proc externalProg {prog msg targetState} {
    global extProgOut tcl_platform

    panelShowMsg "$msg,\nplease wait"
    set extProgOut ""

    switch $tcl_platform(platform) {
	unix    {set interp ""}
	windows {set interp "cmd.exe /c start /min"}
    }
    set inpipe     [open "| $interp $prog"]

    fileevent  $inpipe readable [list protExtProg $inpipe]
    tkwait variable progReady

    if {[catch panelShowDBState currentState]} {
	return
    }
    if [string compare $currentState $targetState] {
	if {[string first "already started" $extProgOut] < 0} {
	    panelShowMsg $extProgOut
	} else {
	    panelShowMsg
	}
	error ""
    }
    panelShowMsg
    return $currentState
}

# ------------------------------------------------------------------------
# protExtProg: Close the command pipe; write the final line into protocol
# ------------------------------------------------------------------------
proc protExtProg {inpipe} {
    global extProgOut progReady

    if [eof $inpipe] {
	catch "close $inpipe" extProgError
	set progReady 1
    } else {
	set line [gets $inpipe]
	if {[string length $line] \
		&& [string first "starting server" $line] < 0} {
	    append extProgOut "$line\n"
	}
    }
}

# ------------------------------------------------------------------------
# knlDiagPath: Evaluate the path name of the knldiag file and store it
#              into global array connect(knlDiagPath)
# ------------------------------------------------------------------------
proc knlDiagPath {} {
    global connect env tcl_platform

    set getParam "[file join $env(DBROOT) pgm getparam] $connect(serverdb)"

    if [catch {open "|$getParam RUNDIRECTORY"} inpipe] return
    set runDir [gets $inpipe]
    if [catch {close $inpipe}] return

    # The xparam key for the knldiag file name differs between 6.1 and 6.2.
    # And if this is not enough, on windows it got also another name.
    # So here we try all of them one after the other...
    # Note, that we return immediately, if the open fails, since in this
    # case getParam didn't start at all. If the key is unknown, the error
    # is reported, not before we close the command pipe.

    foreach paramKey {_KERNELDIAGFILE DIAGFILE OPMSG3} {
	if [catch {open "|$getParam $paramKey"} inpipe] return
	set diagFile [gets $inpipe]
	if ![catch {close $inpipe}] {
	    if ![string compare $tcl_platform(platform) windows] {
		regsub %DBROOT% $runDir $env(DBROOT) runDir
	    }
	    set connect(knlDiagPath) [file join $runDir $diagFile]
	    confMenueEntry .m "Kernel diagnose" -state normal
	    return
	}
    }
}

# ---------------------------------------------------------------------
# utilityCmd: Execute the given utility command while displaying an
#             info message.
#             Return the resulting database state (cold|warm).
# ---------------------------------------------------------------------
proc utilityCmd {cmd msg} {
    panelShowMsg "$msg,\nplease wait"
    set error [sendKernel $cmd]
    if {$error} {
	panelShowMsg "Error $error while $cmd"
	return ""
    }
    panelShowMsg
    if {[catch panelShowDBState currentState]} {return ""}
    return $currentState
}

# ---------------------------------------------------------------------
# quitPanel: quits this program, not after asking, whether the
#            database should be stopped, too...
# ---------------------------------------------------------------------
proc quitPanel {askShutdown} {
    global connect

    if {$askShutdown} {
	switch [dbState] cold - warm {
	    set    message "Should database '$connect(serverdb)' be stopped,"
	    append message " before quitting Panel?"

	    switch [tk_messageBox -title "Stop" -icon question \
		    -message $message -type yesnocancel -default no ] {
		cancel {return}
		yes    {update; changeStateTo offline}
	    }
	}
    }
    exit 0
}

# ---------------------------------------------------------------------
# createPopup: Create a popup menu.
# ---------------------------------------------------------------------
proc createPopup {} {
    global currVal

    menu .m

    .m add cascade -label Help -menu .m.help
    menu .m.help
    .m.help add command -label "Help..."      -underline 0 -acc F1 \
	    -command "callHtmlHelp panel"
    .m.help add command -label "About..."     -underline 0 \
	    -command "aboutHelp Panel"
    .m.help add check   -label "Use Netscape" -underline 0 \
	    -variable options(val:netscapeHelp)
    .m add separator

    .m add command -label "Change State of DB" -state disabled
    foreach state {Offline Cold Warm} {
	.m add command -label $state -und 0 \
		-command "changeStateTo [string tolower $state]"
    }

    .m add separator
    .m add command -label "Info Sheet" -und 0 -command "infoWindow"
    .m add cascade -label "Refresh"    -menu .m.refr
    .m add command -label "Kernel diagnose" -und 0 -state disabled \
            -command "editKnlDiag" -acc F2

    .m add separator
    .m add command -label "Quit Panel" -und 0 -command "quitPanel 0"

    bind .panel <3> "tk_popup .m %X %Y"

    menu .m.refr
    .m.refr add radio -label "10 sec" -variable currVal(refresh) -value  10000
    .m.refr add radio -label "30 sec" -variable currVal(refresh) -value  30000
    .m.refr add radio -label " 2 min" -variable currVal(refresh) -value 120000
    .m.refr add radio -label "10 min" -variable currVal(refresh) -value 600000
    set currVal(refresh) 10000
}

# ---------------------------------------------------------------------
# infoWindow: Creates a toplevel window of its own with a control like
#             outfit. Since most of this info sheet is in infoSheet.tcl,
#             here we are creating only the frame, and call a procedure
#             to fill the values.
# ---------------------------------------------------------------------
proc infoWindow {} {
    global connect

    if {[winfo exists .info]} {
	if {![string compare [wm state .info] iconic]} {wm deiconify .info}
	raise .info
	return
    }

    toplevel .info
    wm title    .info "Adabas D: Info Sheet for $connect(serverdb)"
    wm iconname .info "$connect(serverdb) Info Sheet"

    button .info.b -text Dismiss -command "dismissInfo"
    pack   .info.b -side bottom -expand 1 -fill x

    
    if {[catch {mkInfoSheet .info.i} msg]} {
	destroy .info
	tk_messageBox -title "No Info Sheet" -icon error \
		-message "No info sheet due to the following error:\n$msg"
	return
    }

    setCurrVal
}

proc editKnlDiag {} {
    global connect

    if ![info exists connect(knlDiagPath)] return

    if [catch "open $connect(knlDiagPath)" f] {
	tk_messageBox -title "No Kernel Diagnose" -icon error \
		-message "Couldn't open file $connect(knlDiagPath): $f"
	return
    }

    set t     .knldiag
    set title "Kernel Diagnose"

    if [winfo exists $t] {
	$t.t configure -state normal
	$t.t delete 1.0 end
    } else {
	toplevel $t
	wm title    $t $title
	wm iconname $t $title

	scrollbar $t.s -orient vertical -command "$t.t yview" \
		-highlightthickness 0 -takefocus 1
	pack $t.s -side right -fill y -padx 1
	text $t.t -yscrollcommand "$t.s set" -wrap word -width 85 -height 30 \
		-setgrid 1 -highlightthickness 0 \
		-padx 4 -pady 2 -takefocus 0
	pack $t.t -expand y -fill both -padx 1
	button $t.b -text Dismiss -command "destroy $t"
	pack $t.b -side bottom -fill x

	bind  $t.t <Prior> "$t.t yview scroll -1 pages; break"
	bind  $t.t <Next>  "$t.t yview scroll  1 pages; break"
	bind  $t.t <Home>  "$t.t yview moveto  0;       break"
	bind  $t.t <End>   "$t.t yview moveto  1;       break"
	bind  $t.t <F3>    "destroy $t;                 break"
	bind  $t.t <Enter> "focus $t.t;                 break"
    }

    # Now read in the file. Insert lines only, if they are written.
    # On Unix, we can be shure, that the first line containing only
    # blanks is the end marker. On windows systems, the second line
    # is a blank one, so we have to scan a little further...

    set empty 0
    while {![eof $f] && $empty < 3} {
	set line [gets $f]
        if [regexp {^ *$} $line] {
            incr empty
        } else {
            set empty 0
        }
	$t.t insert end "$line\n"
    }
    close $f
    $t.t configure -state disabled
}
# ---------------------------------------------------------------------
# setCurrVal: A procedure to fill the values of the Info Sheet. First
#             it determines the state of the database server
#             (offline,cold,warm). If the database is warm, it sends
#             additional selects, to adjust the labels and bars.
#             At last it reschedules itself to be called again, when
#             the refresh timespan is elapsed (by means of 'after').
#             Note the SELECT INTOS(!) at the core of this procedure.
# ---------------------------------------------------------------------
proc setCurrVal {} {
    global currVal connect adamsg

    set currVal(state) WARM
    if {![controlConnect]} {
        if {[info exists adamsg(errortxt)]} {
	    if {$adamsg(rc) == -8000} {
		set currVal(state) COLD
	    } else {
		if {$adamsg(rc) == -4008} {
		    set connect(error) "No valid Control user given"
		} else {
		    set connect(error) "$adamsg(rc): $adamsg(errortxt)"
		}
		getDbnameUserPassword "Adabas D Panel Connect" controlConnect
		if {![info exists connect(db)]} {
		    error "Invalid Control User"
		    return
		}
	    }
	} else {
	    set currVal(state) OFFLINE
	    return
        }
    }

    if {![string compare $currVal(state) WARM]} {
	set l $connect(db)
	set c $connect(cursor)

	regexp {\(for Adabas D (.*)\)} $adamsg(version) match version
	set vers62 [expr {$version >= 6.2}]

	# Switch on monitoring, if not already turned on.
	if {$vers62} {
	    adasql $c {SELECT value INTO :monitorOnOff
                         FROM sysdd.server_db_state
                         WHERE description = 'MONITORING'}
        } else {
	    adasql $c {SELECT value INTO :monitorOnOff
                         FROM sysdd.serverdbstate
                         WHERE description = 'MONITORING'}
        }
	if {[string compare $monitorOnOff ON]} {
	    adasql $c {MONITOR ON}
	}

	if {$vers62} {
	    adasql $c {SELECT used_temp_pages, pct_temp_used,
                              used_perm_pages, pct_perm_used,
                              log_not_saved,   pct_log_not_saved,
                              used_log_pages,  pct_logused
                         INTO :currVal(temp), :currVal(temp_pct),
                              :currVal(perm), :currVal(perm_pct),
                              :currVal(lns),  :currVal(lns_pct),
                              :currVal(log),  :currVal(log_pct)
                         FROM sysdd.serverdb_stats}
	    adasql $c {SELECT SUM(numeric_value) INTO :currVal(data_sum)
                         FROM sysdd.configuration
                         WHERE description LIKE 'DATA DEVSPACE *'}
	    adasql $c {SELECT SUM(numeric_value) INTO :currVal(log_sum)
                         FROM sysdd.configuration
                         WHERE description LIKE 'LOG SIZE *'}
	    adasql $c {SELECT value INTO :currVal(redoSize)
                         FROM sysdd.xparameters
                         WHERE description='RESERVED_REDO_SIZE'}
	    set currVal(data)     [expr {$currVal(perm) + $currVal(data_sum)/6}]
	    set currVal(data_pct) [expr {$currVal(data)*100 / $currVal(data_sum)}]
	    adasql $c {SELECT numeric_value INTO :currVal(badIndexes)
                         FROM sysdd.server_db_state
                         WHERE description='NO OF BAD INDEXES'}
	    if $currVal(badIndexes) {
		set currVal(warning) "$currVal(badIndexes) BAD INDEXES"
	    }

        } else {
	    adasql $c {SELECT used_perm_pages, pct_used_perm,
                              used_log_pages,  pct_used_log,
                              completed_log_segm
                         INTO :currVal(data), :currVal(data_pct),
                              :currVal(log),  :currVal(log_pct),
                              :currVal(logSegments)
                         FROM sysdd.serverdbstatistics}
	    if {![string compare $currVal(logSegments) 0]} {
		set currVal(logSegments) "NOT COMPLETED"
	    } else {
		set currVal(logSegments) "COMPLETED"
	    }
        }

	if [catch {adasql $c "EXISTS TABLE domain.tables"}] {
	    set currVal(sysTabsLoaded) "WARNING - Systables not loaded"
	}

	adasql $c {SELECT value INTO :maxUser FROM sysdd.xparameters
	  	     WHERE description = 'MAXUSERTASKS'}

        adasql $c {SELECT COUNT (session) INTO :currVal(sessions)
                     FROM sysdd.transactions}
        set currVal(sessions_pct) [expr {$currVal(sessions)*100/$maxUser}]

        adasql $c {SELECT value INTO :currVal(reads) FROM sysdd.monitor
                     WHERE type = 'PAGES'
                       AND description = 'VIRTUAL  READS'}

        adasql $c {SELECT value INTO :currVal(writes) FROM sysdd.monitor
                     WHERE type = 'PAGES'
                       AND description = 'VIRTUAL  WRITES'}

        adasql $c {SELECT value INTO :currVal(sqlCommands) FROM sysdd.monitor
                     WHERE type = 'TRANS'
                       AND description = 'SQL COMMANDS'}

        adasql $c {SELECT value INTO :currVal(hitrate) FROM sysdd.monitor
                     WHERE type = 'CACHES'
                       AND description = 'DATA CACHE HIT RATE (%)'}

        adasql $c {SELECT value INTO :currVal(converter) FROM sysdd.monitor
                     WHERE type = 'CACHES'
                       AND description = 'CONVERTER CACHE HIT RATE (%)'}

        set currVal(stateAt) [clock format [clock seconds] \
		-format "%Y-%m-%d %H:%M:%S"]
		  
        adaclose  $c
        adalogoff $l

	set currVal(autosaveLog) [autosaveShow]

	set serverPort [expr {$vers62 ? "sql6" : "sql30"}]
	if [catch {socket [info hostname] $serverPort} msg] {
	    set currVal(remoteSQL) DISABLED
	    if {[string first "expected integer" $msg] >= 0} {
		# No x_server available.
	    } else {
		# x_server not running.
	    }
	} else {
	    set currVal(remoteSQL) ENABLED
	    close $msg
	}
    }
    set currVal(afterID) [after $currVal(refresh) setCurrVal]
}

# ---------------------------------------------------------------------
# dismissInfo: Removes the Info Sheet and cancels any outstanding refresh.
# ---------------------------------------------------------------------
proc dismissInfo {} {
    global currVal

    catch {after cancel $currVal(afterID)}
    destroy .info
}

proc autosaveShow {} {
    global connect

    set ret DISABLED
    if {[utilConnect]} {
	if {![catch {adautil $connect(db) "AUTOSAVE SHOW"}]} {
	    set ret ENABLED
	}
	adalogoff $connect(db)
    }
    return $ret
}

# ---------------------------------------------------------------------
# dbState: The procedure, that communicates with the database to
#          evaluate its state and return it coded as (warm|cold|offline).
#          It expects some valuable information in dbname and user/passwd.
# ---------------------------------------------------------------------
proc dbState {} {
    global connect adamsg argv

    if {[catch "adalogon $connect(param) -service utility \
	    				 -serverdb $connect(serverdb)" log]} {
        if {[info exists adamsg(errortxt)]} {
	    if {$adamsg(rc) == -4008} {
		set connect(error) "No valid Control user given"
	    } else {
		set connect(error) "$adamsg(rc): $adamsg(errortxt)"
	    }
	    getDbnameUserPassword "Adabas D Panel Connect" utilConnect
	    if {[info exists connect(db)]} {
	        set log $connect(db)
	    } else {
	        error "Invalid Control User"
	    }
	} else {
	    return offline
	}
    }
    set stateOut [adautil $log "STATE"]
    catch {adautil $log "COMMIT WORK RELEASE"}
    adalogoff $log
    return [lindex $stateOut 0]
}

# ---------------------------------------------------------------------
# sendKernel: send the given utility command to the database kernel.
# ---------------------------------------------------------------------
proc sendKernel {command} {
    global connect

    if {[catch "adalogon $connect(param) -service utility \
	    				 -serverdb $connect(serverdb)" db]} {
	return "permission denied"
    }

    catch {adautil $db $command} commandOut
    catch {adalogoff $db}
    return 0
}

# ---------------------------------------------------------------------
# utilConnect: Try to connect with the given user key or user,password
#              combination as utility service.
# ---------------------------------------------------------------------
proc utilConnect {} {
    global connect adamsg

    if {[info exists connect(userkey)] && ![info exists connect(clearpw)]} {
        set connect(param) ",$connect(userkey)"
    } else {
	set connect(param) "$connect(user),$connect(clearpw)"
    }

    if {[catch "adalogon $connect(param) -service utility \
	    				 -serverdb $connect(serverdb)" db]} {
	if {[info exists adamsg(rc)]} {
	    if {$adamsg(rc) == -4008} {
		set connect(error) "No valid Control user given"
	    } else {
		set connect(error) "$adamsg(rc): $db"
	    }
	} else {
	    set connect(error) $db
	}
	return 0
    }

    set connect(db) $db
    return 1
}

# ---------------------------------------------------------------------
# controlConnect: Try to connect with the given user key or user,password
#                 combination as control user.
# ---------------------------------------------------------------------
proc controlConnect {} {
    global connect adamsg

    if {[info exists connect(userkey)] && ![info exists connect(clearpw)]} {
        set connect(param) ",$connect(userkey)"
    } else {
	set connect(param) "$connect(user),$connect(clearpw)"
    }
    catch {unset connect(servernode)}

    if {[catch "adalogon $connect(param) -service control \
	    				 -serverdb $connect(serverdb)" db]} {
	if {[info exists adamsg(rc)]} {
	    if {$adamsg(rc) == -4008} {
		set connect(error) "No valid Control user given"
	    } else {
		set connect(error) "$adamsg(rc): $db"
	    }
	} else {
	    set connect(error) $db
	}
	return 0
    }

    set c [adaopen $db]
    if {[catch {adasql $c {SELECT value FROM sysdd.xparameters
                             WHERE description='CONTROLUSERID'
			       AND value=USER}}]} {
        adaclose  $c
	adalogoff $db
	set adamsg(rc)       -4008
	set adamsg(errortxt) "UNKNOWN USER/PASSWORD COMBINATION"
	set connect(error)   "No valid Control user given"
	return 0
    }

    set connect(db)     $db
    set connect(cursor) $c
    return 1
}

# ---------------------------------------------------------------------
# checkLocal: Try to determine, if the given connect data really stays
#             on this host. This procedure is very dumb(!)
# ---------------------------------------------------------------------
proc checkLocal {} {
    global connect

    # But make shure, that at least a user key or user name was given...
    if {(![info exists connect(userkey)]
         || ![string length $connect(userkey)])
        && ![string length $connect(user)]} {
	set connect(error) "Enter user name and password"
	return 0
    }

    if {[string length $connect(servernode)]} {
	set connect(error) "No remote panel possible"
	return 0
    }

    if {[info exists connect(userkey)] && ![info exists connect(clearpw)]} {
        set connect(param) ",$connect(userkey)"
    } else {
	set connect(param) "$connect(user),$connect(clearpw)"
    }

    return 1
}

# ---------------------------------------------------------------------
# confMenueEntry: Since the menuentries in tearoffs are not updated
#                 together with their original, you should use this
#                 procedure instead of entryconfigure directly.
# ---------------------------------------------------------------------
proc confMenueEntry {m index args} {
    set tearOffs [info commands .tearoff*]
    if {[llength $tearOffs]} {
        set m "$m $tearOffs"
    }
    foreach menu $m {
        catch {eval $menu entryconfigure [list $index] $args}
    }
}

# ---------------------------------------------------------------------
# Bitmaps
# ---------------------------------------------------------------------
set bitmap(bulb) {
#define bulb_width 24
#define bulb_height 24
static unsigned char bulb_bits[] = {
0x00,0x7e,0x00,0x80,0xff,0x01,0xe0,0xff,0x07,0xf0,0xff,0x0f,
0xf8,0xff,0x1f,0xfc,0xff,0x3f,0xfc,0xff,0x3f,0xfe,0xff,0x7f,
0xfe,0xff,0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xe7,0xfe,0xff,0x67,
0xfe,0xff,0x73,0xfc,0xff,0x33,0xfc,0xff,0x38,0xf8,0x3f,0x1c,
0xf0,0x3f,0x0f,0xe0,0xff,0x07,0x80,0xff,0x01,0x00,0x7e,0x00};
}
set bitmap(light) {
#define light_width 28
#define light_height 146
static unsigned char light_bits[] = {
0x00,0xff,0x0f,0x00,0xe0,0xff,0x7f,0x00,0xfc,0xff,0xff,0x03,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,0x0f,
0xfe,0xff,0xff,0x07,0xfe,0xff,0xff,0x07,0xf8,0xff,0xff,0x01,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,0x00,0xf0,0x00,0x00,
0x00,0xf0,0x00,0x00,0x00,0x60,0x00,0x00};
}
set bitmap(ring) {
#define ring_width 26
#define ring_height 26
static unsigned char ring_bits[] = {
0x00,0xfc,0x01,0x00,0x00,0x03,0x03,0x00,0xc0,0x00,0x0c,0x00,
0x30,0x00,0x10,0x00,0x18,0x00,0x20,0x00,0x08,0x00,0x40,0x00,
0x04,0x00,0x80,0x00,0x04,0x00,0x80,0x00,0x02,0x00,0x00,0x01,
0x02,0x00,0x00,0x01,0x01,0x00,0x00,0x02,0x01,0x00,0x00,0x02,
0x01,0x00,0x00,0x02,0x01,0x00,0x00,0x02,0x01,0x00,0x00,0x02,
0x01,0x00,0x30,0x02,0x02,0x00,0x30,0x01,0x02,0x00,0x18,0x01,
0x04,0x00,0x98,0x00,0x04,0x00,0x8e,0x00,0x08,0x80,0x47,0x00,
0x10,0x80,0x21,0x00,0x20,0x00,0x10,0x00,0xc0,0x00,0x0c,0x00,
0x00,0x03,0x03,0x00,0x00,0xfe,0x00,0x00};
}
set bitmap(shade) {
#define shade_width 32
#define shade_height 7
static unsigned char shade_bits[] = {
0x00,0xfc,0x3f,0x00,0x80,0xff,0xff,0x01,0xf0,0xff,0xff,0x0f,
0xfc,0x01,0x80,0x3f,0x7f,0x00,0x00,0xfe,0x0f,0x00,0x00,0xf0,
0x03,0x00,0x00,0xc0};
}
set bitmap(shine) {
#define shine_width 32
#define shine_height 32
static unsigned char shine_bits[] = {
0x00,0x20,0x04,0x00,0x00,0x20,0x04,0x00,0x00,0x21,0x84,0x00,
0x00,0x02,0x40,0x00,0x10,0x02,0x40,0x08,0x20,0x00,0x00,0x04,
0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x20,
0x18,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x07,0x00,0x00,0xe0,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x07,0x00,0x00,0xe0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x18,0x00,0x00,0x18,0x04,0x00,0x00,0x20,
0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x02,0x20,0x00,0x00,0x04,
0x10,0x02,0x40,0x08,0x00,0x02,0x40,0x00,0x00,0x21,0x84,0x00,
0x00,0x20,0x04,0x00,0x00,0x20,0x04,0x00};
}

# ---------------------------------------------------------------------
# Mainprogram
# ---------------------------------------------------------------------
wm withdraw .
loadAdabastcl
createPanel
createPopup
getDbnameUserPassword "Adabas D Panel Connect" checkLocal
knlDiagPath
wm deiconify .
update
if {![catch panelShowDBState currentState]} {
    after 100 {changeStateTo warm}
}
