
# ---------------------------------------------------------------------
#  $Id: tkquery.tcl,v 1.95 1997/07/06 15:36:30 adabas Exp $
# ---------------------------------------------------------------------

# set some global defines...
set ROW_NOT_FOUND 100
set MASSCNT       20

# set what the Execute button should first do
set execCmd doSql

# Set default values for the option switches.
array set options {
    dom:fontFamily   {helvetica times courier}
    def:fontFamily   helvetica
    dom:fontSize     {8 10 12 14 18 24}
    def:fontSize     14
    dom:fontStyle    {roman bold italic bolditalic}
    def:fontStyle    bold
    dom:autoCommit   boolean
    def:autoCommit   True
    dom:dateFormat   {ansi eur internal iso jis oracle usa}
    def:dateFormat   internal
    dom:outputStyle  {plain html latex}
    def:outputStyle  plain
    dom:withProt     boolean
    def:withProt     False
    dom:history      integer
    def:history      10
    dom:nullValue    string
    def:nullValue    ""
    dom:specialNull  string
    def:specialNull  "***"
    dom:withToolBar  boolean
    def:withToolBar  True
    dom:netscapeHelp boolean
    def:netscapeHelp False
    dom:balloonHelp  boolean
    def:balloonHelp  True
    dom:errorPopup   boolean
    def:errorPopup   False
}
# this corresponds to the following excerpt from ~/.Xdefaults...
# *tkquery.fontFamily:	 helvetica
# *tkquery.fontSize:	 14
# *tkquery.fontStyle:	 bold
# *tkquery.autoCommit:	 True
# *tkquery.dateFormat:	 internal
# *tkquery.outputStyle:	 plain
# *tkquery.withProt:	 False
# *tkquery.history:	 10
# *tkquery.nullValue:	
# *tkquery.specialNull:	 ***
# *tkquery.withToolBar:  True
# *tkquery.netscapeHelp: False
# *tkquery.balloonHelp:  True
# *tkquery.errorPopup:   False

frame .tkquery ;# Just for the options...

foreach optInd [array names options def:*] {
    set    optName  [string range $optInd 4 end]
    set    optClass [string toupper [string index $optName 0]]
    append optClass [string range $optName 1 end]

    option add *tkquery.$optName $options(def:$optName) widgetDefault
    set options(val:$optName) [option get .tkquery $optName $optClass]
    switch $options(dom:$optName) {
	boolean {
	    set options(val:$optName) [string match {[Tt]*} $options(val:$optName)]
	}
	integer {
	    if [catch {int($options(val:$optName))}] {
		set options(val:$optName) $options(def:$optName)
	    }
	}
	string {
	}
	default {
	    if {[lsearch -exact $options(dom:$optName) $options(val:$optName)]<0} {
		set options(val:$optName) $options(def:$optName)
	    }
	}
    }
}

set cur {}
set lda {}

set current(file)     ""
set current(query)    ""
trace variable current w setHeader

set fontCorr 1.0

set cmdIdx  0
set cmdLast 0
for {set i 0} {$i < $options(val:history)} {incr i} {
  set cmdRing($i) ""
}

array set format {
    filSta,plain ""
    filSta,html  "<HTML><HEAD><TITLE>TkQuery Protocol</TITLE></HEAD>
                  <BODY><H1>TkQuery Protocol</H1>"
    filSta,latex "
    \\documentclass\[german,a4paper]{article}
    \\usepackage{babel}
    \\pagestyle{empty}
    \\begin{document}
    \\small"
    filEnd,plain ""
    filEnd,html  "</BODY>"
    filEnd,latex "\\end{document}"
    header,plain "headerPlain"
    header,html  "headerHtml"
    header,latex "headerLatex"
    vrbSta,plain ""
    vrbSta,html  "<PRE>"
    vrbSta,latex "\\begin{verbatim}"
    vrbEnd,plain ""
    vrbEnd,html  "</PRE>"
    vrbEnd,latex "\\end{verbatim}"
    hdrSta,plain ""
    hdrSta,html  "<TR><TH>"
    hdrSta,latex "\\bf "
    hdrSep,plain " "
    hdrSep,html  "</TH><TH>"
    hdrSep,latex "&\\bf "
    hdrEnd,plain ""
    hdrEnd,html  "</TH></TR><TR><TD>"
    hdrEnd,latex "\\\\"
    colSep,plain " "
    colSep,html  "</TD><TD>"
    colSep,latex "&"
    rowSep,plain ""
    rowSep,html  "</TD></TR><TR><TD>"
    rowSep,latex "\\\\"
    filter,plain "format"
    filter,html  "filterHtml"
    filter,latex "filterLatex"
    null,html    "nullHtml"
    tailer,plain "\n"
    tailer,html  "</TD></TR></TABLE>"
    tailer,latex "\\end{tabular}"
}

proc headerPlain {fmt cnt} {
    return ""
}

proc headerHtml {format cnt} {
    upvar $format fmt

    set result "<TABLE BORDER COLSPEC=\""
    for {set i 0} {$i < $cnt} {incr i} {
	append result [expr {[string match {%[0-9]*} $fmt($i)] ? "r " : "L "}]
    }
    append result "\">"
    return $result
}

proc headerLatex {format cnt} {
    upvar $format fmt

    set result "\\begin\{tabular\}\{"
    for {set i 0} {$i < $cnt} {incr i} {
	append result [expr {[string match {%[0-9]*} $fmt($i)] ? "r" : "l"}]
    }
    append result "\}"
    return $result
}

proc filterHtml {fmt data} {
    regsub -all {&} $data {\&amp;}   data
    regsub -all {} $data {\&auml;}  data
    regsub -all {} $data {\&ouml;}  data
    regsub -all {} $data {\&uuml;}  data
    regsub -all {} $data {\&Auml;}  data
    regsub -all {} $data {\&Ouml;}  data
    regsub -all {} $data {\&Uuml;}  data
    regsub -all {} $data {\&szlig;} data
    regsub -all {<} $data {\&lt;}    data
    regsub -all {>} $data {\&gt;}    data
    return $data
}

proc filterLatex {fmt data} {
    regsub -all {} $data "\"a"  data
    regsub -all {} $data "\"o"  data
    regsub -all {} $data "\"u"  data
    regsub -all {} $data "\"A"  data
    regsub -all {} $data "\"O"  data
    regsub -all {} $data "\"U"  data
    regsub -all {} $data "\"s"  data
    regsub -all {&} $data {\\\&} data
    return $data
}

proc nullHtml {value} {
    return "&nbsp;"
}

proc pannedListboxes {w {focus ""}} {
    global oldCols tkCompat

    pack [frame $w] -expand 1 -fill both
    set c [canvas $w.c -relief sunken -xscrollcommand "$w.horz set"]
    $c create window 1 1 -window [frame $c.f] -anchor nw -tag outframe
    grid rowconfigure $c.f 1 -weight $tkCompat(gridWeightMax)
    frame $c.f.pad
    set oldCols($c) 0
    bind $c <Configure> "canvasConf $c"

    label $w.pad -text ""
    if $tkCompat(fontCmd) {
	$w.pad configure -font outFont
    }

    scrollbar $w.vert -relief sunken -command "bindOutYView $c" \
	    -orient vertical -highlightthickness 0
    scrollbar $w.horz -relief sunken -command "$c xview" \
	    -orient horizontal -highlightthickness 0
    $w.vert set 0 1

    grid $c      -row 0 -column 0 -sticky nsew -rowspan 2
    grid $w.pad  -row 0 -column 1
    grid $w.vert -row 1 -column 1 -sticky ns
    grid $w.horz -row 2 -column 0 -sticky ew
    grid columnconfigure $w 0 -weight $tkCompat(gridWeightMax)
    grid rowconfigure    $w 1 -weight $tkCompat(gridWeightMax)

    if ![string length $focus] {
	set focus $w
    }
    bind $focus <F10>     "scrollBox $w left;  break"
    bind $focus <F11>     "scrollBox $w right; break"
    bind $focus <Prior>   "scrollBox $w up;    break"
    bind $focus <Next>    "scrollBox $w down;  break"
    bind $focus <Home>    "scrollBox $w first; break"
    bind $focus <End>     "scrollBox $w last;  break"
    bind $w     <FocusIn> "focus     $focus;   break"
}

proc confListbox {w colNo action {colName ""}} {
    global oldCols tkCompat boxConf fontCorr

    set c $w.c
    switch -- $action {
	<start> {
	    $c.f configure -width 10000
	}
	<stop> {
	    # forget old lists
	    for {set col $colNo} {$col < $oldCols($c)} {incr col} {
		foreach s [grid slaves $c.f -column $col] {
		    grid forget $s; destroy $s
		}
		grid columnconfigure $c.f $col -weight $tkCompat(gridWeightMin)
	    }

	    grid $c.f.pad -row 1 -column $colNo
	    grid columnconfigure $c.f $colNo -weight $tkCompat(gridWeightMax)

	    set oldCols($c) $colNo
	}
	<normal> - <disabled> {
	    for {set ix 0} {$ix < $oldCols($c)} {incr ix} {
		$c.f.l$ix configure -state [string trim $action <>]
	    }
	}
	default {
	    if {$colNo < $oldCols($c)} {
		$c.f.h$colNo configure $action $colName
		$c.f.l$colNo delete 1.0 end
	    } else {
		catch {destroy $c.f.h$colNo $c.f.l$colNo}
		set label [label $c.f.h$colNo $action $colName]
		
		set width [expr int(ceil($boxConf(width,$w,$colNo)*$fontCorr))]
		set list  [text  $c.f.l$colNo -width $width \
			-highlightthickness 0 -wrap none -state disabled]
		if !$colNo {
		    $list configure -yscrollcommand "$w.vert set"
		}
		if ![string compare $boxConf(type,$w,$colNo) numeric] {
		    $list tag configure just -justify right
		}
		if $tkCompat(fontCmd) {
		    $label configure -font outFont
		    $list  configure -font outFont
		}
		grid $label -row 0 -column $colNo -ipadx 2 -padx 1 -sticky w
		grid $list  -row 1 -column $colNo -ipadx 2 -padx 1 -sticky nsew
	    }
	    grid columnconfigure $c.f $colNo -weight $tkCompat(gridWeightMin)
	    bind $c.f.h$colNo <3> "switchPanel $w $colNo visible 0"
	}
    }
}

proc setBoxConf {w colNo col} {
    global adamsg boxConf fmt

    set c       $w.c
    set headLen [string length $col]

    set boxConf(visible,$w,$colNo) 1
    set boxConf(title,$w,$colNo)   $col
    set boxConf(width,$w,$colNo)   $headLen

    switch -glob [lindex $adamsg(coltypes) $colNo] {
	long_byte* {
	    set fmt($colNo,head) %-${headLen}s
	    set fmt($colNo)      %-${headLen}s
	    set boxConf(type,$w,$colNo) data
	}
	long_* {
	    set fmt($colNo,head) %-50s
	    set fmt($colNo)      %-50.50s
	    set boxConf(type,$w,$colNo) long
	}
	fixed* - float* {
	    set fmtLen  [max [lindex $adamsg(collengths) $colNo] $headLen]
	    set fmtPrec [lindex $adamsg(colscales) $colNo]
	    if $fmtPrec {
		set fmt($colNo,number) %${fmtLen}.${fmtPrec}f
	    } else {
		set fmt($colNo,number) %${fmtLen}d
	    }
	    set fmt($colNo)      %${fmtLen}s
	    set fmt($colNo,head) %${fmtLen}s
	    set boxConf(type,$w,$colNo) numeric
	}
	default {
	    set fmtLen [max [lindex $adamsg(collengths) $colNo] $headLen]
	    set fmt($colNo)      %-${fmtLen}s
	    set fmt($colNo,head) %-${fmtLen}s
	    set boxConf(type,$w,$colNo) text
	}
    }
}

proc scrollBox {w dir} {
    switch $dir {
	left  {$w.c xview        scroll -1 pages}
	right {$w.c xview        scroll  1 pages}
	up    {bindOutYView $w.c scroll -1 pages}
	down  {bindOutYView $w.c scroll  1 pages}
	first {bindOutYView $w.c moveto  0}
	last  {bindOutYView $w.c moveto  1}
    }
}

proc insertListbox {w colNo {colValue ""}} {
    global boxConf fontCorr

    set c $w.c
    set colWidth [string length $colValue]
    if {$colWidth > $boxConf(width,$w,$colNo)} {
	set boxConf(width,$w,$colNo) $colWidth
	set width [expr int(ceil($colWidth*$fontCorr))]
	$c.f.l$colNo configure -width $width
    }
    $c.f.l$colNo insert end "$colValue\n" just
}

proc mkPanel {panel w} {
    global oldCols boxConf tkCompat

    if [catch {toplevel $panel}] {
	eval destroy [grid slaves $panel]
    }
    wm title $panel "Column Configure"
    wm group $panel .

    grid [label $panel.l -text "Visible columns"] -row 0 -sticky ew

    set c $w.c
    for {set colNo 0} {$colNo < $oldCols($c)} {incr colNo} {
	set control [checkbutton $panel.l$colNo   \
		-textvariable boxConf(title,$w,$colNo)  \
		-variable     boxConf(visible,$w,$colNo) \
		-command  "switchPanel $w $colNo visible"]
	grid $control -row [expr {$colNo+1}] -column 0 -sticky w
    }

    grid [button $panel.b -text "Dismiss" -command "destroy $panel"] \
	    -row [expr {$colNo+2}] -column 0 -sticky ew
    grid columnconfigure $panel 0 -weight $tkCompat(gridWeightMax)

    update idletask
}

proc switchPanel {w colNo kind {value ""}} {
    global boxConf oldCols

    set c $w.c
    if [string length $value] {
	set boxConf($kind,$w,$colNo) $value
    }

    switch $kind {
	visible {
	    if $boxConf($kind,$w,$colNo) {
		grid $c.f.h$colNo -row 0 -column $colNo -ipadx 2 -padx 1 -sticky w
		grid $c.f.l$colNo -row 1 -column $colNo -ipadx 2 -padx 1 -sticky nsew
	    } else {
		grid forget $c.f.h$colNo $c.f.l$colNo
	    }
	}
	default {
	    return
	}
    }
    updateListboxes $c $oldCols($c)
}

proc updateListboxes {c colNo} {
    set bbox0 [grid bbox $c.f 0      1]
    set bboxN [grid bbox $c.f $colNo 1]
    set boxWidth [expr {[lindex $bboxN 0] - [lindex $bbox0 0]}]
    set infinity [$c.f cget -width]

    $c.f configure -width $boxWidth
    $c   configure -scrollregion [$c bbox outframe]
    $c.f configure -width $infinity
}

#
# createMain
#
#   create the main window
#

proc createMain {} {
    global lda cur adamsg connect options tcl_platform menu options tkCompat

    wm geom     . 600x500
    wm minsize  . 400 370
    wm title    . "Adabas D: TkQuery ($connect(serverdb)/$connect(user))"
    wm iconname . "TkQuery"
    wm protocol . WM_DELETE_WINDOW confirmExit

    # create a menu bar with some menu buttons
    if $tkCompat(menubars) {
	. configure -menu [menu .mb -tearoff 0]

	set menu(sql)  .mb.sql
	set menu(exec) .mb.exec
	set menu(out)  .mb.out
	set menu(op)   .mb.op
	set menu(ob)   .mb.ob
	set menu(help) .mb.help

	.mb add cascade -label "SQL"         -menu $menu(sql)  -und 0
	.mb add cascade -label "Execute"     -menu $menu(exec) -und 0
	.mb add cascade -label "Results"     -menu $menu(out)  -und 0
	.mb add cascade -label "Options"     -menu $menu(op)   -und 0
	.mb add cascade -label "DB Objects"  -menu $menu(ob)   -und 0
	.mb add cascade -label "Help"        -menu $menu(help) -und 0
    } else {
	frame .mb -relief raised -borderwidth 2
	pack  .mb  -side top -fill x

	set menu(sql)  .mb.sql.m
	set menu(exec) .mb.exec.m
	set menu(out)  .mb.out.m
	set menu(op)   .mb.op.m
	set menu(ob)   .mb.ob.m
	set menu(help) .mb.help.m

	menubutton .mb.sql  -text "SQL"        -menu $menu(sql)  -und 0
	menubutton .mb.exec -text "Execute"    -menu $menu(exec) -und 0
	menubutton .mb.out  -text "Results"    -menu $menu(out)  -und 0
	menubutton .mb.op   -text "Options"    -menu $menu(op)   -und 0
	menubutton .mb.ob   -text "DB Objects" -menu $menu(ob)   -und 0
	menubutton .mb.help -text "Help"       -menu $menu(help) -und 0

	pack .mb.sql .mb.exec .mb.out .mb.op .mb.ob -side left
	if ![string compare $tcl_platform(platform) unix] {
	    pack .mb.help -side right
	} else {
	    pack .mb.help -side left
	}
    }

    menu $menu(sql)
    $menu(sql) add command -label "New"  -command doNew        -und 2 -acc F2
    $menu(sql) add command -label "Prev" -command "prevSql -1" -und 0 -acc F9
    $menu(sql) add command -label "Next" -command "prevSql  1" -und 0 -acc F6
    $menu(sql) add separator
    $menu(sql) add command -label "Stored commands" -state disabled
    $menu(sql) add command -label "Get..."     -und 0 -command getQuery
    $menu(sql) add command -label "Replace"    -und 0 -command putQuery
    $menu(sql) add command -label "Insert..."  -und 0 -command putQueryAs
    $menu(sql) add command -label "Delete..."  -und 0 -command deleteQuery
    $menu(sql) add separator
    $menu(sql) add command -label "Open..."    -und 0 -command openSql
    $menu(sql) add command -label "Save"       -und 0 -command saveSql 
    $menu(sql) add command -label "Save As..." -und 5 -command saveSqlAs
    $menu(sql) add separator
    $menu(sql) add command -label "Exit" -command confirmExit -und 0 -acc F3

    menu $menu(exec)
    $menu(exec) add command -label "Execute"  -command doSql     -und 0 -acc F5
    $menu(exec) add command -label "Continue" -command fetchRows -und 1 \
                                              -state disabled
    $menu(exec) add command -label "Cancel"   -command doCancel  -und 0 -acc F3 \
                                              -state disabled
    $menu(exec) add sep
    $menu(exec) add command -label "Commit now" -underline 7     \
	    -command {adacommit $lda; setMsg "Committed" }
    $menu(exec) add command -label "Rollback now"   -underline 0 \
	    -command {adarollback $lda; setMsg "Rolled back" }

    menu $menu(out)
    $menu(out) add command -label "Clear" -command "clearoutput .o" -und 0
    $menu(out) add command -label "Configure..." -state disabled \
	    -command "mkPanel .panel .o"

    $menu(out) add separator
    $menu(out) add checkbutton -label "Protocol" -und 0 \
	    -var options(val:withProt) -command setProt
    $menu(out) add command -label "Save As..." -und 0 -command saveOut
    $menu(out) add command -label "Preview"    -und 3 -command preview
    $menu(out) add separator
    $menu(out) add command -label "Scroll Left"  -und 7 -acc F10 \
	    -command "scrollBox .o left"
    $menu(out) add command -label "Scroll Right" -und 7 -acc F11 \
	    -command "scrollBox .o right"

    menu $menu(op)
    $menu(op) add checkbutton -label "Autocommit" -underline 0 \
	    -variable options(val:autoCommit) -command setAutoCommit

    $menu(op) add command -label "Null Value..."   -und 0 -command setNull
    $menu(op) add command -label "Special Null..." -und 0 -command setSpecialNull
    $menu(op) add cascade -label "Date Format"     -und 0 -menu $menu(op).df
    menu $menu(op).df
    foreach format {ansi eur internal iso jis oracle usa} {
	$menu(op).df add radiobutton -variable options(val:dateFormat) \
		-value $format -label $format -command "setFormat"
    }
    $menu(op) add sep
    $menu(op) add checkbutton -label "Tool bar" -variable options(val:withToolBar) \
	    -command toggleToolBar
    $menu(op) add command -label "Font..." -und 0 -command "chooseFont .o.c"
    $menu(op) add checkbutton -label "Popup on error" -underline 9 \
	    -variable options(val:errorPopup)
    $menu(op) add cascade -label "Output Style" -und 0 -menu $menu(op).os
    menu $menu(op).os
    foreach style {plain html latex} {
	$menu(op).os add radiobutton -variable options(val:outputStyle) \
		-value $style -label $style -command setStyle
    }

    menu $menu(ob)
    $menu(ob) add command -label "Users"   -command "showTree users"   -und 0
    $menu(ob) add command -label "Tables"  -command "showTree tables"  -und 0
    $menu(ob) add command -label "Views"   -command "showTree views"   -und 0
    $menu(ob) add command -label "Indexes" -command "showTree indexes" -und 0

    menu $menu(help)
    $menu(help) add command -label "About..."          -underline 0 \
	    -command "aboutHelp TkQuery"
    $menu(help) add command -label "TkQuery..."        -underline 2 -acc F1\
	    -command "callHtmlHelp tkquery"
    $menu(help) add command -label "AdabasTcl..."      -underline 6 \
	    -command "callHtmlHelp adabastcl"
    $menu(help) add check   -label "With balloon help" -underline 5 \
	    -variable options(val:balloonHelp)
    $menu(help) add check   -label "Use Netscape"      -underline 4 \
	    -variable options(val:netscapeHelp)

    # create a message at the bottom (its important to pack it first,
    # since then it remains visible, even if window got very small...
    frame .msgBar
    label .msg -text "" -relief sunken -bd 1 -anchor w
    label .msgBar.user -width 12 -relief sunken -bd 1\
	    -textvariable connect(user)
    pack  .msg -in .msgBar -side left -padx 2 -expand yes -fill both
    pack  .msgBar.user -side left -padx 2
    pack  .msgBar -side bottom -fill x -pady 2
    if ![string compare $tcl_platform(platform) unix] {
	foreach lab {.msg .msgBar.user} {
	    if $tkCompat(fontCmd) {
		$lab configure -font "Helvetica 12 normal"
	    } else {
		$lab configure -font -*-Helvetica-Medium-R-Normal--*-120-*-*-*-*-*-*
	    }
	}
    }

    # create a button bar, if wanted.
    if $options(val:withToolBar) {
	toolBox .b
    }

    # Lets create the fonts for the sql and output windows
    setFont

    # create a frame listing sql code
    frame .s -relief raised -borderwidth 2

    label .s.l
    setHeader current "" w
    scrollbar .s.vert -relief sunken -command ".s.sql yview" \
	    -orient vertical -highlightthickness 0
    text .s.sql -relief sunken -height 8 -width 256 -yscroll ".s.vert set" \
	    -wrap word -highlightthickness 0 -bg white -foreground black
    if $tkCompat(fontCmd) {
	.s.sql configure -font fixedFont
    } else {
	catch {.s.sql configure -font -*-courier-*-r-*-*-14-*-*-*-*-*-*-*}
    }

    # bind some shortcut keys to the text window
    bind .s.sql <F5>    {if {$execCmd == "doSql"} $execCmd; break}
    bind .s.sql <F3>    {doCancel;             break}
    bind .s.sql <F1>    {callHtmlHelp tkquery; break}
    bind .s.sql <F2>    {doNew;                break}
    bind .s.sql <F9>    {prevSql -1;           break}
    bind .s.sql <F6>    {prevSql  1;           break}
    bind .s.sql <F4>    {doSwitch;             break}

    bind .  <FocusIn> {focus .s.sql}
    bind .s <FocusIn> {focus .s.sql}

    pack .s.l    -side top   -fill x
    pack .s.vert -side right -fill y
    pack .s.sql  -side left  -fill both -expand 1

    pack .s -side top -fill both

    pack [resizer .resizer -neighbor .s.sql] -side top -fill x

    pannedListboxes .o .s.sql

    focus .s.sql

    if [catch knlVersion connect(knlVersion)] {
	tk_messageBox -title "Fatal Error" -icon error -message \
		"Can't determine version of database: $connect(knlVersion)"
	exit 1
    }

    # Initialize the options...
    initOptions .o.c

    # Remember some information, that depends on the kernel version.
    if {$connect(knlVersion) >= 62} {
	set connect(querylib)        querylib
	set connect(querylibrary)    querylibrary
	set connect(query_own)       query_own
	set connect(query_call)      query_call
	set connect(ind_uses_col)    indcoluses
	set connect(sproc)           sproc
	set connect(objtype_qual)    ""
	set connect(refobjtype_qual) ""
	set connect(qlib_blankkeys)  ""
	set connect(qlib_blankvals)  ""
	set connect(qown_blankkeys)  ""
	set connect(qown_blankvals)  ""
    } else {
	set connect(querylib)        sysquerylib
	set connect(querylibrary)    sysquerylibrary
	set connect(query_own)       sysquery_own
	set connect(query_call)      sysquery_call
	set connect(ind_uses_col)    ind_uses_col
	set connect(sproc)           sys_proc
	set blank_key                blank_key
	set connect(objtype_qual)    "AND    objtype = 'QUERYCOMMAND'"
	set connect(refobjtype_qual) "AND refobjtype = 'QUERYCOMMAND'"
	foreach nr {4 5 6 7} {
	    append connect(qlib_blankkeys) ", $blank_key$nr"
	    append connect(qlib_blankvals) ", ''"
	}
	foreach nr {2 3 5 6 7 11 12 13 14} {
	    append connect(qown_blankkeys) ", $blank_key$nr"
	    append connect(qown_blankvals) ", ''"
	}
    }

    wm deiconify .
    setMsg
}

proc toggleToolBar {} {
    global options
    if $options(val:withToolBar) {
	toolBox .b -before .s
    } else {
	destroy .b
    }
}

# Now find out, if we are connected to a database server prior to 6.2.
# If here anything went wrong, we better cancel the excution completely.
# Since the wonderful table sysdd.versions exists only >= 6.2, we must
# use the something outdated SHOW VERSION...
proc knlVersion {} {
    global lda cur

    if {[catch {adasql $cur "SELECT kernel INTO :vers FROM domain.versions"}] &&
        [catch {adasql $cur "SELECT kernel INTO :vers FROM sysdd.version"}]} {
	adasql $cur "SHOW (version) VERSION"
	adasql $cur "SELECT kernel INTO :vers FROM version"
    }
    adarollback $lda
    regexp { ([0-9])\.([0-9])\.([0-9])} $vers match major minor cl
    return $major$minor.$cl
}

proc canvasConf {c} {
    global oldCols

    if [grid propagate $c.f] {grid propagate $c.f 0}
    $c.f configure -height [winfo height $c]
    if $oldCols($c) {updateListboxes $c $oldCols($c)}
}

proc bindOutYView {c args} {
    global oldCols

    if [catch "$c.f.l0 yview $args"] return
    set topLine [$c.f.l0 index @0,0]
    for {set ix 1} {$ix < $oldCols($c)} {incr ix} {
	$c.f.l$ix yview $topLine
    }
}

proc chooseFont {c} {
    global options

    if [winfo exist .font] {
	wm deiconify .font
	raise .font
	return
    }

    toplevel .font
    wm title .font "Font for results"
    wm group .font .

    label .font.fontFamily -text "Family"
    grid  .font.fontFamily -row [set row 0] -column 0
    foreach name $options(dom:fontFamily) {
	radiobutton .font.$name -variable options(val:fontFamily) \
		-value $name -text $name -command "setFont $c"
	grid .font.$name -row [incr row] -column 0 -sticky w
    }

    label .font.fontSize -text "Size"
    grid  .font.fontSize -row [set row 0] -column 1
    foreach size $options(dom:fontSize) {
	radiobutton .font.s$size -variable options(val:fontSize) \
		-value $size -text $size -command "setFont $c"
	grid .font.s$size -row [incr row] -column 1 -sticky w
    }
    
    label .font.fontStyle -text "Style"
    grid  .font.fontStyle -row [set row 0] -column 2
    foreach style $options(dom:fontStyle) {
	radiobutton .font.$style -variable options(val:fontStyle) \
		-value $style -text $style -command "setFont $c"
	grid .font.$style -row [incr row] -column 2 -sticky w
    }

    button .font.dismiss -text Dismiss -command "destroy .font"
    grid   .font.dismiss - - -column 0 -sticky ew

    bind .font <Return> ".font.dismiss invoke"
}

if $tkCompat(fontCmd) {
    proc setFont {{c ""}} {
	global options fontCorr

	switch $options(val:fontStyle) {
	    roman      {set weight normal; set slant roman}
	    bold       {set weight bold;   set slant roman}
	    italic     {set weight normal; set slant italic}
	    bolditalic {set weight bold;   set slant italic}
	}
	if [string length $c] {
	    set kind configure
	} else {
	    set kind create
	}
	font $kind outFont \
		-family $options(val:fontFamily) \
		-size   $options(val:fontSize)   \
		-weight $weight \
		-slant  $slant
	font $kind fixedFont \
		-family courier \
		-size   $options(val:fontSize)   \
		-weight $weight \
		-slant  $slant

	if [string length $c] {
	    canvasConf $c
	}
	if [font metrics outFont -fixed] {
	    set fontCorr 1.0
	} else {
	    set fontCorr 1.2
	}
	update idletask
    }
} else {
    proc setFont {{c ""}} {
	global options font fontCorr

	switch $options(val:fontStyle) {
	    roman      {set weight medium; set style r}
	    bold       {set weight bold;   set style r}
	    italic     {set weight medium; set style o}
	    bolditalic {set weight bold;   set style o}
	}
	# For times it is called italic instead of oblique...
	if {![string compare $options(val:fontFamily) times] \
		&& ![string compare $style o]} {
	    set style i
	}

	set font -*-$options(val:fontFamily)-$weight-$style-*-*-$options(val:fontSize)-*-*-*-*-*-*-*
	if [string length $c] {
	    if [catch {.o.pad configure -font $font}] return
	    set n 0
	    while 1 {
		if [catch {$c.f.h$n configure -font $font} msg] break
		if [catch {$c.f.l$n configure -font $font} msg] break
		incr n
	    }
	    canvasConf $c
	}
	option add *o.c*font $font interactive
	switch $options(val:fontFamily) {
	    courier {set fontCorr 1.0}
	    default {set fontCorr 1.2}
	}
	update idletask
    }
}

proc setStyle {{occasion onRadio}} {
    global options format protFilename oldStyle

    if ![catch {open $protFilename a} f] {
	if ![info exists oldStyle] {
	    set oldStyle $options(val:outputStyle)
	}
	puts  $f $format(filEnd,$oldStyle)
	close $f
    }
    set oldStyle $options(val:outputStyle)

    if ![string compare $occasion onRadio] {
	setProt
    }
}

proc setProt {} {
    global options format protFilename errorCode

    if $options(val:withProt) {
	switch $options(val:outputStyle) {
	    html  {set protFilename "tkquery.html"}
	    latex {set protFilename "tkquery.tex"}
	    plain {set protFilename "tkquery.prot"}
	}
	if [catch {open $protFilename w} f] {
	    setMsg "Error while open '$protFilename':\
[lindex $errorCode 2], no protocol"
	    set options(val:withProt) 0
	} else {
	    puts  $f $format(filSta,$options(val:outputStyle))
	    close $f
	}
    }
}

proc doCancel {} {
    global execCmd

    if [string compare $execCmd doSql] {
	uplevel #0 $execCmd
    } else {
	confirmExit
    }
}


proc setAutoCommit {{noMsg 0}} {
    global lda options menu

    set onOff [expr {$options(val:autoCommit) ? "on" : "off"}]
    adaautocom $lda $onOff;
    if !$noMsg {
	setMsg "Autocommit $onOff"
    }

    set state [expr {$options(val:autoCommit) ? "disabled" : "normal"}]
    foreach entry {"Commit now" "Rollback now*"} {
	confMenueEntry $menu(exec)  $entry -state $state
    }
}

proc setFormat {{noMsg 0}} {
    global options adamsg connect

    if [string compare $connect(sqlmode) oracle] {
	if [catch {callSql "SET FORMAT $options(val:dateFormat)" \
		-sqlmode adabas} error] {
	    if [info exists adamsg(errortxt)] {
		set msg "Error $adamsg(rc):$adamsg(errortxt) while setting format" 
	    } else {
		set msg "Error $error while setting format"
	    }
	} else {
	    set msg "Set format $options(val:dateFormat)"
	}
	if !$noMsg {
	    setMsg $msg
	}
    }
}

#
# setMsg
#
#   set the text for the label at bottom of results window
#

proc setMsg {args}  {
    global options defaultMsg

    set popup    0

    switch [llength $args] {
	0 {set msg [set defaultMsg ""]}
	1 {
	    if ![string compare -reset [lindex $args 0]] {
		catch {set defaultMsg} msg
	    } else {
		set msg   [set defaultMsg [lindex $args 0]]
		set popup [expr {[regexp ^Error $msg] && $options(val:errorPopup)}]
	    }
	}
	2 {
	    if ![string compare -help [lindex $args 0]] {
		set msg [lindex $args 1]
	    }
	}
    }
    if ![info exists msg] {return -code error "usage: setMsg ?-help? msg"}

    if [regexp ^Error $msg] {
	.msg configure -text $msg -foreground red
    } else {
	set defaultColor [lindex [.msg configure -foreground] 3]
	.msg configure -text $msg -foreground $defaultColor
    }
    update idletasks

    if $popup {
	tk_messageBox -title "SQL Error" -icon error -message $msg
    }
}

proc setHeader {name1 name2 op} {
    upvar #0 $name1 current

    switch $name2 query - file {} default return

    set textList ""
    if [string length $current(query)] {
	lappend textList "Query $current(query)"
    }
    if [string length $current(file)] {
	lappend textList "File $current(file)"
    }
    if [llength $textList] {
	set text [join $textList " / "]
    } else {
	set text noname
    }
    .s.l configure -text "SQL ($text)"
}

proc confirmExit {} {
  global lda cur

    switch [tk_messageBox -title "Exit?" -icon question \
	    -message "Really Exit?" -type yesno -default yes] {
	yes {
	    adaclose  $cur  ;# close the cursor,
	    adalogoff $lda  ;# disconnect from the database server,
	    setStyle onExit ;# finish the protocol,
	    exit            ;# and bye, bye...
	}
    }
}

#
# clearsql
#
#   clear the sql code window
#
proc clearsql {} {
  global current cmdIdx cmdLast

  set cmdIdx $cmdLast

  .s.sql delete 1.0 end
  set current(file)  ""
  set current(query) ""
  focus .s.sql
}

# clearoutput
#
#   clear the output canvas with all the listboxes
#
proc clearoutput {w {onlyReset 0}} {
    global menu

    if !$onlyReset {
	# Since there are no select columns, there is nothing to configure...
	confMenueEntry $menu(out)  "Configure..." -state disabled
	catch "destroy .panel"
    }

    confListbox $w 0 <stop>
    $w.c configure -scrollregion "0 0 0 0"
    $w.vert set 0 1

    focus .s.sql
}

proc topEntry {w title text varName {width 30}} {
    global _entryOk

    set _entryOk 0

    catch {destroy $w}
    toplevel $w -class Dialog
    makeTransient $w
    wm title $w $title

    frame $w.f1 -relief sunken -borderwidth 1

    label $w.f1.l -text $text
    entry $w.f1.e -width 18 -textvariable $varName -relief sunken \
	    -highlightthickness 0 -background white -foreground black

    pack $w.f1.l -side left
    pack $w.f1.e -side left -fill x -expand 1

    frame $w.f3 -relief sunken -borderwidth 1
    button $w.f3.can -text "Cancel" -command "destroy $w"
    button $w.f3.app -text "Set"    -command "set _entryOk 1; destroy $w"

    pack $w.f3.app $w.f3.can -side left -expand 1 -fill x
    pack $w.f1 $w.f3 -side top -padx 10 -pady 5 -fill both

    bind  $w.f1.e <F3>     "$w.f3.can invoke; break"
    bind  $w.f1.e <Return> "$w.f3.app invoke; break"

    grab  $w
    focus $w.f1.e
    tkwait window $w

    return $_entryOk
}

# ---------------------------------------------------------------------------
# selectQuery: pops up a toplevel with a tree widget, to let the user choose
#              a stored command by selecting its name.
# ---------------------------------------------------------------------------
proc selectQuery {title allQueries func} {
    global cur connect

    catch {destroy .queries}
    toplevel    .queries
    wm title    .queries $title
    wm withdraw .queries
    makeTransient .queries

    button .queries.b -text "Dismiss" -command "destroy .queries"
    pack   .queries.b -side bottom -fill x

    pack [tree create .queries.t] -side top -fill both -expand 1

    mkTreeImages .queries.t

    tree addtype .queries.t -type db      -image dbImg \
	    -opencommand "querySel_listUser $allQueries"
    tree addtype .queries.t -type userNo  -image userImg
    tree addtype .queries.t -type userYes -image userImg \
	    -opencommand querySel_listQueries
    tree addtype .queries.t -type query   -image queryImg -command $func

    if [catch {tree add .queries.t -text $connect(serverdb) -type db \
	    -open "$connect(serverdb) $connect(user)"}] {
	if !$allQueries {
	    setMsg "No stored query found"
	    destroy .queries
	    return
	}
    }

    wm deiconify .queries
    focus .queries.t
    bind  .queries <F3> "destroy .queries"
}

proc mkTreeImages {t} {
    global adabastcl_library tkCompat

    set TRANSPARENT_GIF_COLOR [$t cget -bg]
    if $tkCompat(imageViaData) {
	image create photo dbImg -format gif -data {
R0lGODlhCwAMAMIAAP///wAA/wD//wAAAAD/AAAAAAAAAAAAACH5BAEAAAAALAAAAAALAAwA
AAMnCLq88RCqIKoVYeKIwfjDhX0E8UGjOZhXsJZm5L5fs5KDDZuNB/YJAD
	}
	image create photo userImg -format gif -data {
R0lGODlhCwAMAMIAAP///wAAAP//AP8AAAAA/wAAAAAAAAAAACH5BAEAAAAALAAAAAALAAwA
AAMoCArR+0HI8FQTrsZdWctLMwQjZQ1oOjIqmbow6pZfvQZEnlMmrhOmBAA7
	}
	image create photo queryImg -format gif -data {
R0lGODlhDAAMAKEAALLA3AAAAP//8wAAACH5BAEAAAAALAAAAAAMAAwAAAIgRI4Ha+IfWHsO
rSASvJTGhnhcV3EJlo3kh53ltF5nAhQAO3B
	}
    } else {
	if [catch {image create photo queryImages -format gif \
		-file [file join $adabastcl_library query.gif]} msg] {
	    return -code error "Could not create stored query images: $msg"
	}
	[image create photo dbImg]    copy queryImages -from  0 0 10 12
	[image create photo queryImg] copy queryImages -from 11 0 22 12
	[image create photo userImg]  copy queryImages -from 23 0 34 12
	image delete queryImages
    }
}
proc querySel_listUser {allQueries c path} {
    global lda cur connect

    if $allQueries {
	if [catch {callSql "SELECT DISTINCT author
                              FROM SYSDBA.$connect(query_call)
                            UNION SELECT USER FROM SYSDBA.dual ORDER BY 1"}] {
	    return ""
	}
	adafetch $cur -command "lappend allUser @1"
    } else {
	set allUser $connect(user)
    }

    foreach owner $allUser {
	if [string compare $owner $connect(user)] {
	    adasql $cur "SELECT COUNT (*) INTO :cnt
                            FROM SYSDBA.$connect(query_call) \
		            WHERE author = '$owner'"
	} else {
	    adasql $cur "SELECT COUNT (*) INTO :cnt
                           FROM SYSDBA.$connect(query_own)"
	}
	if $cnt {
	    lappend ret [list userYes $owner]
	} else {
	    if !$allQueries {
		return ""
	    }
	    lappend ret [list userNo  $owner]
	}
    }
    return $ret
}

proc querySel_listQueries {c path} {
    global lda cur connect

    set owner [lindex $path end]
    if [string compare $owner $connect(user)] {
	if [catch {callSql "SELECT command_name \
                              FROM SYSDBA.$connect(query_call) \
                              WHERE author = '$owner' ORDER BY command_name" \
				-sqlmode adabas}] {
	    return ""
	}
    } else {
	if [catch {callSql "SELECT command_name \
                              FROM SYSDBA.$connect(query_own) \
                              ORDER BY command_name" -sqlmode adabas}] {
	    return ""
	}
    }
    adafetch $cur -command {
	lappend ret [list query @1]
    }
    return $ret
}

proc getQuery {} {
    selectQuery "Get query" 1 getThisQuery
}

proc getThisQuery {c path} {
    global cur current connect

    setMsg

    set queryName [lindex $path 1].[lindex $path 2]

    if [regexp {^([^.]*)\.(.*)$} $queryName match author queryName] {
	set queryLib SYSDBA($author).$connect(querylibrary)
	set author   '$author'
    } else {
	set queryLib SYSDBA.$connect(querylib)
	set author   USERGROUP
    }
    callSql "SELECT command
               FROM $queryLib
               WHERE author       = $author
                 AND command_name = '$queryName'" -sqlmode adabas
    set cmd [adareadlong $cur -descriptor [lindex [adafetch $cur] 0]]
    set cmdLines [join [adabas get $cmd editform] \n]

    clearsql
    clearoutput .o
    .s.sql insert 1.0 $cmdLines

    setMsg "Query $queryName loaded"
    set current(query) $queryName
}

proc putQuery {{kind replaced}} {
    global current cur lda connect adamsg

    if [catch {analyzeSqlWindow .s.sql 1.0}] return

    setMsg
    if ![string length $current(query)] {
	return [putQueryAs]
    }

    set  cmdString [.s.sql get 1.0 end]
    set  strLength [string length $cmdString]
    set  cmdList   [split $cmdString "\n"]
    incr strLength [llength $cmdList]

    set whereCond "author = USER \
                   $connect(objtype_qual) \
                   AND command_name = '$current(query)'"
    set select "UPDATE $connect(querylib)
                  SET updatedate = DATE,
                      updatetime = TIME,
                      strlength  = $strLength
                  WHERE $whereCond"
    if [catch {callSql $select -sqlmode adabas} msg] {
	if ![string compare $msg 100] {
	    return [putQueryAs]
	}
	setMsg "Error: $msg"
	return
    }

    adawritelong $cur -table $connect(querylib) -column command \
                      -where $whereCond -value [adabas put $cmdList editform]
    adacommit $lda

    adausage $lda on -objecttype QUERYCOM \
	    -parameters [list $current(query)]
    if [catch {adasql $cur -parse $cmdString}] {
	setMsg "Incorrect command $current(query) $kind: $adamsg(errortxt)"
	catch {adausage $lda off}
    } elseif {[catch {adausage $lda off} err]} {
	setMsg "Data dictionary not updated: $err"
    } else {
	setMsg "Command $current(query) $kind."
    }
}

proc putQueryAs {} {
    global current _queryName connect

    if [catch {analyzeSqlWindow .s.sql 1.0}] return

    set okay 0
    setMsg
    set _queryName $current(query)

    while {!$okay} {
	if {![topEntry .storeAs "Store Query As" "Query name" _queryName 18] \
		|| ![string length $_queryName]} {
	    unset _queryName
	    return
	}

	# The insert doesn't like it at all, if command_name is too long,
	# and xquery doesn't allow a dot or pattern chars...
	set _queryName [string range $_queryName 0 17]
	if [regexp {[.*?%_]} $_queryName] {
	    setMsg "Error: Special characters in query name not allowed"
	} else {
	    setMsg
	    set okay 1
	}
    }

    set insert "INSERT $connect(querylib) \
                  (command_name, codetype $connect(qlib_blankkeys)) \
                  VALUES ('$_queryName', 'ASCII' $connect(qlib_blankvals))"

    if [catch {callSql $insert -sqlmode adabas} ret] {

	if {$ret != 200} {return -code error $ret}

	switch [tk_messageBox -title "Replace query?" \
		-message "Query $_queryName already exists! Replace it?" \
		-icon question -default cancel -type okcancel] {
	    cancel {return}
	}

	set current(query) $_queryName
	return [putQuery]
    }

    set insert "INSERT $connect(query_own) \
                  (command_name $connect(qown_blankkeys)) \
                  VALUES ('$_queryName' $connect(qown_blankvals))"
    callSql $insert -sqlmode adabas

    set current(query) $_queryName
    unset _queryName
    putQuery inserted    
}


proc deleteQuery {} {
    selectQuery "Delete query" 0 deleteThisQuery
}

proc deleteThisQuery {c path} {
    global lda current connect

    setMsg
    set user      [lindex $path 1]
    set queryName [lindex $path 2]

    switch [tk_messageBox -title "Delete query?" \
	    -message "Really delete query $queryName?" \
	    -icon question -default ok -type okcancel] {
	cancel {return}
    }

    callSql "DELETE FROM SYSDBA.$connect(querylib) \
               WHERE author       = '$user'        \
                 $connect(objtype_qual)            \
                 AND command_name = '$queryName'" -sqlmode adabas
    callSql "DELETE FROM SYSDBA.$connect(query_own) \
               WHERE author       = '$user'         \
                 $connect(refobjtype_qual)          \
                 AND command_name = '$queryName'" -sqlmode adabas
    if {$connect(knlVersion) >= 62} {
	set query $queryName
    } else {
	set query [string toupper $queryName]
    }
    callSql "DBPROC DOMAIN.$connect(sproc).DELETE_OBJECT \
                  ('QUERYCOMMAND','$user','$query','','')" -sqlmode adabas
    adacommit $lda

    setMsg "Query $queryName deleted"
    if ![string compare $current(query) $queryName] {
	set current(query) ""
    }
    tree delete $c -path $path
}

#
# openSql
#
#    try to open the file selected by the user in a file selection box

proc openSql {} {
    global current cmdIdx cmdLast

    set fileName [tk_getOpenFile -title "TkQuery: Open SQL File" \
	    -filetypes {{"SQL command file" *.sql} {"All Files" *}}]
    if ![string length $fileName] return

    set cmdIdx $cmdLast

    if [file isfile $fileName] {
	clearsql
	clearoutput .o
	set current(file) [file tail $fileName]
	set result_lines [exec cat -s $fileName]
	.s.sql insert 1.0 "$result_lines"
	setMsg "$fileName loaded"
    } else {
	setMsg "$fileName not found"
    }
    focus .s.sql
}

#
# saveSqlAs
#
#    save the sql code
#

proc saveSqlAs {} {
    global current cmdIdx cmdLast errorCode

    set cmdIdx $cmdLast

    set fileName [tk_getSaveFile -title "TkQuery: Save SQL File" \
	    -filetypes {{"SQL command file" *.sql} {"All Files" *}}]
    if ![string length $fileName] return

    if [catch {open $fileName w} f] {
	setMsg "Error while open '$fileName': [lindex $errorCode 2], not saved"
	return
    }

    set current(file) $fileName

    puts $f [.s.sql get 1.0 end]
    close $f
    setMsg "SQL saved to $current(file)"
}

proc getRowCnt {c} {
    global oldCols

    if !$oldCols($c) {
	return 0
    } else {
	return [expr {int([$c.f.l0 index end])-2}]
    }
}

proc preview {} {
    global options format cmdIdx cmdLast errorCode oldCols \
	    tkCompat env tcl_platform

    set cmdIdx $cmdLast
    set w      .o
    set c      $w.c
    set sql    .s.sql
    set rows   [getRowCnt $c]

    if !$rows {
	setMsg "No output to preview"
	return
    }
    
    switch $options(val:outputStyle) {
	plain {
	    append results $format(filSta,plain)
	    append results [getSql     $sql plain]
	    append results [getResults $w   plain $rows]
	    append results $format(filEnd,plain)

	    if [catch {toplevel .preview}] {
		eval destroy [grid slaves .preview]
	    }
	    wm title .preview "TkQuery: Preview"
	    wm group .preview .

	    grid rowconfigure    .preview 0 -weight $tkCompat(gridWeightMax)
	    grid columnconfigure .preview 0 -weight $tkCompat(gridWeightMax)

	    button .preview.quit -text Dismiss -com "destroy .preview"
	    grid   .preview.quit -sticky ew -row 2 -columnspan 2

	    scrollbar .preview.xscr -command {.preview.text xview} -orient horiz
	    grid      .preview.xscr -row 1 -column 0 -sticky ew

	    scrollbar .preview.yscr -command {.preview.text yview}
	    grid      .preview.yscr -row 0 -column 1 -sticky ns

	    text .preview.text -wrap none \
		    -xscrollcommand {.preview.xscr set} \
		    -yscrollcommand {.preview.yscr set}
	    if $tkCompat(fontCmd) {
		.preview.text configure -font fixedFont
	    }
	    grid .preview.text -row 0 -column 0 -sticky nsew -ipady 10 -ipadx 10

	    bind .preview.text <Prior> ".preview.text yview scroll -1 pages; break"
	    bind .preview.text <Next>  ".preview.text yview scroll  1 pages; break"
	    bind .preview.text <Home>  ".preview.text yview moveto  0;       break"
	    bind .preview.text <End>   ".preview.text yview moveto  1;       break"
	    bind .preview.text <F10>   ".preview.text xview scroll -1 pages; break"
	    bind .preview.text <F11>   ".preview.text xview scroll  1 pages; break"
	    bind .preview.text <F3>    "destroy .preview;                    break"
	    bind .preview    <FocusIn> "focus .preview.text;                 break"

	    .preview.text insert 1.0 $results
	    .preview.text configure -state disabled
	}

	latex {
	    setMsg "Preview of latex not yet implemented"
	}

	html {
	    # First we have to determine the temp directory.
	    foreach var {TMPDIR TMP TEMP} {
		if [info exists env($var)] {
		    set tmpDir $env($var)
		    break
		}
	    }
	    if ![info exists tmpDir] {
		switch $tcl_platform(platform) {
		    windows {set tmpDir c:/tmp}
		    default {set tmpDir /tmp}
		}
	    }
	    set tmpFile [file join $tmpDir tkq[pid].html]
	    if [catch {open $tmpFile w} f] {
		setMsg "Error while open '$tmpFile': [lindex $errorCode 2]"
		return
	    }
	    setMsg "Writing file..."
	    puts $f $format(filSta,html)
	    puts $f [getSql     $sql html]
	    puts $f [getResults $w   html $rows]
	    puts $f $format(filEnd,html)
	    close $f

	    setMsg "Calling netscape..."
	    if [catch {callNetscape $tmpFile}] {
		tk_messageBox -title "No Netscape started" \
			-message "Since Netscape could not be started,\
the html document cannot be previewed" -icon error -type ok
	    }
	    setMsg
	    if $tkCompat(fileDelCmd) {
		set fileDel "file delete $tmpFile"
	    } else {
		switch $tcl_platform(platform) {
		    windows {set fileDel "exec del $tmpFile"}
		    default {set fileDel "exec rm $tmpFile >& /dev/null"}
		}
	    }
	    bind .tkquery <Destroy> $fileDel
	}
    }
}

#
# saveOut
#
#    save the sql results
#

proc saveOut {{fileName ""}} {
    global cmdIdx cmdLast oldCols errorCode options format

    set w      .o
    set c      $w.c
    set sql    .s.sql
    set cmdIdx $cmdLast
    set style  $options(val:outputStyle)
    set rows   [getRowCnt $c]

    if !$rows {
	setMsg "No output to save"
	return
    }

    if [string length $fileName] {
	set mode   a+
	set append 1
    } else {
	switch $style {
	    html    {set mask *.html}
	    latex   {set mask *.tex}
	    default {set mask *.prt}
	}
	set fileName [tk_getSaveFile -title "TkQuery: Save result" \
		-filetypes {{"Protocol Files" *.prt} {"All Files" *}}]
	if ![string length $fileName] return
	
	set mode   w
	set append 0
    }
    if [catch {open $fileName $mode} f] {
	setMsg "Error while open '$fileName': [lindex $errorCode 2], not saved"
	return
    }

    setMsg "Writing file..."
    if !$append {puts $f $format(filSta,$style)}
    puts $f [getSql     $sql $style]
    puts $f [getResults $w   $style $rows]
    if !$append {puts $f $format(filEnd,$style)}
    close $f

    if !$append {
	setMsg "Results saved to $fileName"
    } else {
	setMsg
    }
}

proc getSql {sql style} {
    global format

    append res "$format(vrbSta,$style)\n"
    append res [$sql get 1.0 end]
    append res "$format(vrbEnd,$style)\n"
    return $res
}

proc getResults {w style rows} {
    global fmt oldCols format options colNames boxConf adamsg

    set c $w.c
    append res "[$format(header,$style) fmt $oldCols($c)]\n"
    append res "$format(hdrSta,$style)"
    set currSep ""
    for {set j 0} {$j < $oldCols($c)} {incr j} {
	if $boxConf(visible,$w,$j) {
	    append res $currSep
	    set currSep $format(hdrSep,$style)
	    append res [$format(filter,$style) $fmt($j,head) \
		    [lindex $colNames $j]]
	}
    }
    append res "$format(hdrEnd,$style)\n"

    for {set i 1} {$i <= $rows} {incr i} {
	if {$i > 1} {append res "$format(rowSep,$style)\n"}

	set currSep ""
	for {set j 0} {$j < $oldCols($c)} {incr j} {
	    if $boxConf(visible,$w,$j) {
		append res $currSep
		set currSep $format(colSep,$style)
		set currVal [$c.f.l$j get $i.0 "$i.0 lineend"]
		if {![string compare $currVal $adamsg(nullvalue)] \
			&& [info exists format(null,$style)]} {
		    append res [$format(null,$style) $currVal]
		} else {
		    append res [$format(filter,$style) $fmt($j) $currVal]
		}
	    }
	}
	if {$i&&!($i%20)} {setMsg "Writing file ($i rows out of $rows so far)..."}
    }
    append res "$format(tailer,$style)"

    setMsg
    return $res
}

#
# saveSql
#
#    save the sql code to currentFile or use saveSqlAs
#

proc saveSql {} {
    global current cmdIdx cmdLast

    set cmdIdx $cmdLast

    if ![string length $current(file)] {
	saveSqlAs
	return
    }

    set f [open $current(file) w]
    puts $f [.s.sql get 1.0 end]
    close $f
    setMsg "saved to $current(file)"
}

# doNew
#   clear windows
proc doNew {} {
    global current cmdIdx cmdLast

    set cmdIdx $cmdLast

    clearsql
    setMsg
    focus .s.sql
}

# insSql
#   insert the current Sql into the cmdRing
proc insSql {} {
    global cmdRing cmdIdx cmdLast options

    set currentSql [.s.sql get 1.0 end]

    # don't save null buffers
    if ![string length [string trim $currentSql]] return

    set cmdRing($cmdLast) $currentSql
    set cmdIdx  $cmdLast

    if {[incr cmdLast] > $options(val:history)} {
	set cmdLast 0
    }
}

# prevSql
#   save current sql window, replace with previous (dir=-1) or next (dir=1) 
proc prevSql {dir} {
    global cmdRing cmdIdx cmdLast options

    set i 0
    set result_lines ""

    while {$i < $options(val:history) && ![string length $result_lines]} {
	incr cmdIdx $dir

	if {$cmdIdx < 0} {
	    set cmdIdx [expr {$options(val:history)-1}]
	} elseif {$cmdIdx >= $options(val:history)} {
	    set cmdIdx 0
	}
	set result_lines $cmdRing($cmdIdx)
	incr i
    }

    if [string length $result_lines] {
	.s.sql delete 1.0 end
	.s.sql insert 1.0 "$result_lines"
    }
}

proc setExecute {onOff} {
    global execCmd menu stopFlag options

    if $onOff {
	set execCmd "set contFlag 0; set stopFlag 0"
	confMenueEntry $menu(exec) "Execute"  -state disabled
	confMenueEntry $menu(exec) "Cancel"   -state normal
	confMenueEntry $menu(exec) "Continue" -state disabled
	if $options(val:withToolBar) {
	    confToolButton .b exec stop "Interrupt query" \
		    "Interrupt the current sql statement"
	}
    } else {
	set execCmd doSql
	confMenueEntry $menu(exec) "Execute" -state normal
	confMenueEntry $menu(exec) "Cancel"  -state disabled
	if !$stopFlag {
	    confMenueEntry $menu(exec) "Continue" -state normal
	}
	if $options(val:withToolBar) {
	    confToolButton .b exec exec "Execute SQL" \
		    "Execute the SQL statement"
	}
    }
}

# doSql: this is a wrapper proc around trySql,
#        that resets in every case the execution button.

proc doSql {} {
    setMsg
    set currPos 1.0
    set ret     0
    while {[string length $currPos] && !$ret} {
	set ret [catch "trySql $currPos" msg]
	if !$ret {set currPos $msg}
    }

    if $ret {
	global errorInfo

	setExecute 0
	return -code error -errorinfo $errorInfo $msg
    } else {
	return $msg
    }
}

proc analyzeSqlWindow {sql startPos} {
    global indexes

    getIndexesInSqlWindow $sql $startPos
    set sqlFilt [$sql get $indexes(sqlStart) $indexes(sqlEnd)]
    if ![string length [string trim $sqlFilt]] {
	if ![string compare $indexes(startPos) 1.0] {
	    setMsg "No SQL to execute"
	}
	return -code error ""
    }
    return $sqlFilt
}

proc trySql {startPos} {
    global cur adamsg contFlag stopFlag fmt oldCols menu indexes options \
	    colNames parameter boxConf MASSCNT ROW_NOT_FOUND

    set contFlag 1
    set stopFlag 1

    set sql .s.sql
    if [catch {analyzeSqlWindow $sql $startPos} sqlFilt] return

    if {[string length $indexes(layoutStart)] \
	    && [string length $indexes(layoutEnd)]} {
	set retMsg [analyzeLayout [$sql get $indexes(layoutStart)+1line \
					    $indexes(layoutEnd)]]
	if [string length $retMsg] {
	    setMsg $retMsg
	    return ""
	}
	if [info exists parameter] {
	    foreach parmNo [lsort -integer -decreasing [array names parameter]] {
		regsub -all "\[&%]$parmNo" $sqlFilt $parameter($parmNo) sqlFilt
	    }
	}
    }

    insSql
    setMsg "Running SQL"

    clearoutput .o 1

    set error [catch {callSql $sqlFilt} rc]
    if $error {
	if {[info exists adamsg(rc)] && [info exists adamsg(errortxt)]} {
	    setMsg "Error $adamsg(rc): $adamsg(errortxt)"
	} else {
	    setMsg "Error: $rc"
	}
	if [info exists adamsg(errorpos)] {
	    $sql mark set insert $indexes(sqlStart)+$adamsg(errorpos)chars-1chars
	    $sql see insert
	}
	return ""
    }
    if [info exists adamsg(intoVars)] {
	setMsg "Sorry, but SELECT INTOS not yet supported by TkQuery"
	set thatsAll 1
    } elseif $adamsg(rows) {
	setMsg "SQL finished, $adamsg(rows) rows affected"
	set thatsAll 1
    } elseif ![llength [set colNames [adacols $cur]]] {
	setMsg
	set thatsAll 1
    } else {
	setMsg "SQL finished, getting results"
	setExecute 1
	set thatsAll 0
    }
    if $thatsAll {
	return $indexes(multiSql)
    }

    catch {unset fmt}
    if {$adamsg(rc) != $ROW_NOT_FOUND} {
	confMenueEntry $menu(out) "Configure..." -state normal
	confListbox .o [set colNo 0] <start>
	foreach col $colNames {
	    setBoxConf  .o $colNo $col
	    confListbox .o $colNo -textvariable boxConf(title,.o,$colNo)
	    incr colNo
	}
	confListbox .o $colNo <stop>
	if [winfo exists .panel] {mkPanel .panel .o}

	set ret [catch {adafetch $cur -count $MASSCNT} rows]
    }
    if ![string length $rows] {
	set contFlag 0
    }

    if [string length $indexes(reportStart)] {
	set reportLines [$sql get "$indexes(reportStart)+1line linestart" \
				  "$indexes(reportEnd)"]
	foreach curLine [split $reportLines "\n"] {
	    if {[regexp -nocase "^\[ \t]*NAME\[ \t]*(\[0-9]*)(.*)$" \
		    $curLine match colNo colName]} {
		if {$colNo > 0 && $colNo <= $oldCols(.o.c)} {
		    set colName [string trim $colName]
		    if ![regexp "^'(.*)'$" $colName match colName] {
			set colName [string toupper $colName]
		    }
		    set boxConf(title,.o,[incr colNo -1]) $colName
		}
	    } elseif {[regexp -nocase \
		    "^\[ \t]*(EX|IN)CLUDE\[ \t]*(\[0-9]*)\[ \t]*$"\
		    $curLine match exIn colNo]} {
		if {$colNo > 0 && $colNo <= $oldCols(.o.c)} {
		    switchPanel .o [incr colNo -1] visible \
			    [regexp -nocase IN $exIn]
		}
	    }
	}
    }

    fetchRows $ret $rows

    return $indexes(multiSql)
}

proc getIndexesInSqlWindow {sql startPos} {
    global indexes

    catch {unset indexes}

    set selectionRanges [$sql tag ranges sel]
    if [llength $selectionRanges] {
	set beg [lindex $selectionRanges 0]
	set end [lindex $selectionRanges 1]
    } else {
	set beg $startPos
	set end end
    }
    set indexes(startPos) $beg

    set indexes(multiSql) [$sql search -regexp {^ *[/*]} $beg $end]
    if [string length $indexes(multiSql)] {
	set end $indexes(multiSql)
	append   indexes(multiSql) +1line
    }

    set indexes(layoutStart) \
	    [$sql search -nocase -regexp {^ *LAYOUT}    $beg $end]
    set indexes(layoutEnd) \
	    [$sql search -nocase -regexp {^ *ENDLAYOUT} $beg $end]
    set indexes(reportStart) \
	    [$sql search -nocase -regexp {^ *REPORT}    $beg $end]

    if [string length $indexes(layoutEnd)] {
	set indexes(sqlStart) $indexes(layoutEnd)+1line
    } else {
	set indexes(sqlStart) $beg
    }
    if [string length $indexes(reportStart)] {
	set indexes(sqlEnd)    $indexes(reportStart)
	set indexes(reportEnd) $end
    } else {
	set indexes(sqlEnd) $end
    }
}

proc fetchRows {{ret 0} {rows {}}} {
    global adamsg contFlag stopFlag boxConf fmt options cur indexes \
	    MASSCNT cnt protFilename

    set sql .s.sql
    set colNo 0

    if [llength $rows] {
	set cnt 0
    } else {
	setMsg "continuing after $cnt rows..."
	set ret [catch {adafetch $cur -count $MASSCNT} rows]
	set contFlag 1
	set stopFlag 1
	setExecute 1
    }

    while {!$adamsg(rc) && $contFlag}  {
	confListbox .o 0 <normal>
	set ix 0
	while {[string length [set row [lindex $rows $ix]]]} {
	    set colNo      0
	    set maxColRows 1
	    foreach col $row {
		switch -- $col $adamsg(nullvalue) {} $adamsg(specialnull) {} \
			default {
		    switch $boxConf(type,.o,$colNo) {
			data {
			    set col <BYTE>
			}
			long {
			    set col [subst -nocommands -novariables \
				    [adareadlong $cur -descriptor $col]]
			}
			numeric {
			    set col [format $fmt($colNo,number) $col]
			}
		    }
		}
		set colList [split $col \n]
		set colRows($colNo) [llength $colList]
		if !$colRows($colNo) {
		    set colRows($colNo) 1
		    insertListbox .o $colNo
		} else {
		    if {$colRows($colNo) > $maxColRows} {
			set maxColRows $colRows($colNo)
		    }
		    foreach colValue $colList {
			insertListbox .o $colNo $colValue
		    }
		}
		incr colNo
	    }
	    if {$maxColRows > 1} {
		for {set iy 0} {$iy < $colNo} {incr iy} {
		    for {set iz $colRows($iy)} {$iz < $maxColRows} {incr iz} {
			insertListbox .o $iy
		    }
		}
	    }
	    incr ix
	    incr cnt
	}

	confListbox .o 0 <disabled>
	updateListboxes .o.c $colNo
	setMsg "$cnt rows so far....(press F3 to cancel)"

	# Now allow cancel requests (via menu or stop button) to break through...
	update

	# It is important to continue fetching only,
	# if not all rows are fetched already.
	if {$ix == $MASSCNT} {
	    set ret [catch {adafetch $cur -count $MASSCNT} rows]
	} else {
	    set contFlag 0
	}
    }
    updateListboxes .o.c $colNo
    if $options(val:withProt) {
	saveOut $protFilename
    }

    set plural     [expr {$cnt != 1 ? "s" : ""}]
    set terminated [expr {$stopFlag ? "finished" : "interrupted"}]
    if $ret {
	if {[info exists adamsg(rc)] && [info exists adamsg(errortxt)]} {
	    setMsg "Error $adamsg(rc): $adamsg(errortxt)"
	} else {
	    setMsg "Error: $rc"
	}
    } elseif [string length $indexes(multiSql)] {
	setMsg "$cnt row$plural returned, hit any key to continue..."
	set old [bind $sql <Key>]
	bind $sql <Key> "set continueMultiSql 1;break"
	vwait continueMultiSql
	bind $sql <Key> $old
	setMsg
    } else {
	setMsg "SQL $terminated, $cnt row$plural returned"
    }
    setExecute 0
}

# ---------------------------------------------------------------------------
# callSql: This procedure is called anywhere in tkquery as wrapper around 
#          adasql for two reasons:
#          - To protect the environment against malicious select...into, since
#            here adasql is called in a separate namespace (without globals);
#          - if it fails due to a timeout, a reconnect is tried once.
#          The funny looking backslashes before the $sign are to prevent the
#          Tcl parser to substitute the variable; this way adasql gets the
#          cursor object instead of trhe cursor name...
# ---------------------------------------------------------------------------
proc callSql {sql args} {
    global cur

    if [llength $args] {
	set error [catch "adasql \$cur \$sql $args" rc]
    } else {
	set error [catch {adasql $cur $sql} rc]
    }

    if $error {
	switch -glob -- $rc {
	    "request failed: connection broken" -
	    "request failed: * timeout" {
		if ![adaReconnect] return
		set error [catch {adasql $cur $sql} rc]
	    }
	}
    }
    if $error {return -code error $rc}
    return $rc
}

#
# formatCols
#
#   return a format to use in column printing
#   names, types, and lengths are lists of equal size
#
proc formatCols {names types lengths} {

    while {[llength $names]} {
	set t   [lvarpop types]
	set len [lvarpop lengths]
	set n   [lvarpop names]

	switch $t {
	    fixed - float {set just ""}
	    default       {set just - }
	}

	# make sure length is as long as colunm name 
	set nameLen [string length $n]
	if {$len < $nameLen} {set len $nameLen}

	append fmt "%${just}${len}.${len}s "
    }
    return $fmt
}

proc analyzeLayout {layout} {
    global _layout parameter tkCompat

    if [winfo exist .layout] {
	wm deiconify .layout
	raise .layout
	return "SQL with layout already started"
    }

    catch {unset parameter}
    catch {unset _layout}

    toplevel .layout
    text .layout.t -setgrid 1
    button .layout.dis -text Cancel -command {set _layout "Execution cancelled"}
    button .layout.ok  -text Okay   -command {set _layout ""}
    pack .layout.t -fill both -expand 1
    pack .layout.dis .layout.ok -fill x -expand 1 -side left

    if $tkCompat(fontCmd) {
	font create layoutFont -family courier -size 14 -weight bold
	.layout.t configure -font layoutFont
    } else {
	if ![catch {.layout.t configure -font -*-courier-*-r-*-*-14-*-*-*-*-*}] {
	    option add *layout.t*font -*-courier-*-r-*-*-14-*-*-*-*-* widgetDefault
	}
    }

    set maxX          0
    set y             0
    set focusSet      0
    set lastEntryLine 0
    set extraHeight   0.0
    foreach row [split $layout \n] {
	incr y
	.layout.t insert end $row\n
        set wOffset 0
        set cOffset 0
	while {[regexp -indices {([&%][0-9]+)( *)} $row match parmInd blInd]} {
	    set parmStart  [lindex $parmInd 0]
	    set parmEnd    [lindex $parmInd 1]
	    set blankStart [lindex $blInd   0]
	    set blankEnd   [lindex $blInd   1]
	    set fieldEnd   [lindex $blInd   1]
	    if {$blankEnd >= $blankStart} {
		incr fieldEnd -1
	    }

	    # Is there a default value? (e.g. &1 := *)
	    set defFound [regexp -indices "^ *:= *('\[^']*'|\[^ \t]*)" \
		    [string range $row [expr $fieldEnd+1] end] match defInd]
	    if $defFound {
		set defStart  [expr [lindex $defInd 0]+$fieldEnd+1]
		incr fieldEnd [expr [lindex $defInd 1]+1]
		set defString [string range $row $defStart $fieldEnd]
		set defString [string trim $defString ']
	    }

	    set parmNo [string range $row [expr {$parmStart+1}] $parmEnd]
	    if [string length [string index $row [expr {$fieldEnd+1}]]] {
		set parmWidth [expr {$fieldEnd - $parmStart + 1}]
	    } else {
		set parmWidth [expr {80-$cOffset-$parmStart}]
	    }

	    entry .layout.t.p$parmNo -width $parmWidth \
		    -textvariable parameter($parmNo)   \
		    -foreground black -background white -relief sunken
	    if $tkCompat(fontCmd) {
		.layout.t.p$parmNo configure -font layoutFont
	    }
	    bind .layout.t.p$parmNo <F3>     ".layout.dis invoke"
	    bind .layout.t.p$parmNo <Return> ".layout.ok  invoke"
	    .layout.t delete $y.$parmStart+${wOffset}chars \
		             $y.$fieldEnd+${wOffset}chars+1char
	    .layout.t window create $y.$parmStart+${wOffset}chars \
		    -window .layout.t.p$parmNo

	    # Insert the default value, if one was found...
	    if $defFound {
		.layout.t.p$parmNo insert end $defString
	    }

	    if !$focusSet {set focusSet 1; focus .layout.t.p$parmNo}

	    incr wOffset $parmStart
	    incr cOffset $blankEnd
	    set row [string range $row $fieldEnd end]

	    if {$y != $lastEntryLine} {
		set lastEntryLine $y
		set extraHeight [expr {$extraHeight+0.33}]
	    }
	}
    }
    set height [expr {$y + ceil($extraHeight)}]
    .layout.t configure -width 82 -height $height -state disabled

    vwait   _layout
    destroy .layout
    if $tkCompat(fontCmd) {
	font delete layoutFont
    }

    return $_layout
}



proc showTree {{objKind ""}} {
    global connect

    catch {destroy .objects}
    toplevel    .objects
    wm title    .objects "DB Objects"
    wm iconname .objects "DB Objects"
    wm withdraw .objects
    makeTransient .objects

    button .objects.b -text "Dismiss" -command "destroy .objects"
    pack   .objects.b -side bottom -fill x

    pack [frame .objects.box] -side right -fill both -expand 1
    label .objects.title -text "" -bd 2 -relief raised
    frame .objects.f -bd 2 -relief raised
    pack  .objects.title -in .objects.box -side top -fill x
    pack  .objects.f     -in .objects.box -side top -fill both -expand 1
    pannedListboxes .objects.f.tab

    pack [resizer .objects.r -orientation vertical -neighbor .objects.t] \
	    -fill y -side right

    pack [tree create .objects.t] -side right -fill y
    mkTreeImages .objects.t

    tree addtype .objects.t -type db      -image dbImg \
	    -opencommand "showTree_listUsers"
    tree addtype .objects.t -type users   -image userImg \
	    -opencommand "showTree_listObjKinds"
    tree addtype .objects.t -type noTabs  -image queryImg
    tree addtype .objects.t -type tabs    -image queryImg \
	    -opencommand "showTree_listTables"
    tree addtype .objects.t -type noViews -image queryImg
    tree addtype .objects.t -type views   -image queryImg \
	    -opencommand "showTree_listViews"
    tree addtype .objects.t -type noInds  -image queryImg
    tree addtype .objects.t -type inds    -image queryImg \
	    -opencommand "showTree_listIndexes"
    tree addtype .objects.t -type table   -image queryImg \
	    -command "showTree_tabColumns 0"
    tree addtype .objects.t -type synonym -image queryImg \
	    -command "showTree_tabColumns 1"
    tree addtype .objects.t -type view    -image queryImg \
	    -command showTree_viewDef
    tree addtype .objects.t -type index   -image queryImg \
	    -command showTree_indColumns

    switch $objKind {
	users   {set start "$connect(db)"}
	tables  {set start "$connect(db) $connect(user) Tables"}
	views   {set start "$connect(db) $connect(user) Views"}
	indexes {set start "$connect(db) $connect(user) Indexes"}
	default {set start "$connect(db) $connect(user)"}
    }
    tree add .objects.t -text $connect(serverdb) -type db -open $start

    wm deiconify .objects
    focus .objects.t
    bind  .objects <F3> "destroy .objects"
}

proc showTree_listUsers {c path} {
    global lda cur connect

    if [catch {callSql "SELECT username FROM domain.users ORDER BY 1" \
	          -sqlmode adabas}] {
	return ""
    }
    adafetch $cur -command {lappend allUser [list users @1]}
    return $allUser
}

proc showTree_listObjKinds {c path} {
    global cur connect

    switch $connect(sqlmode) {
	oracle  {set catPrefix sys.all_}
	default {set catPrefix domain.}
    }
    set user [lindex $path 1]

    foreach {objCat typeList} {
	Tables  {tabs  noTabs}
	Views   {views noViews}
	Indexes {inds  noInds}
    } {
	set selectCmd "SELECT owner FROM $catPrefix$objCat WHERE owner = '$user'"
	set index     [catch {callSql $selectCmd}]
	lappend allObjs [list [lindex $typeList $index] $objCat]
    }
    return $allObjs
}

proc showTree_listTables {c path} {
    global cur connect

    switch $connect(sqlmode) {
	oracle {
	    set selectCmd "SELECT table_name, table_type FROM sys.all_catalog
                             WHERE table_type IN ('TABLE',   'VIEW',
                                                  'SYNONYM', 'SNAPSHOT')
                               AND owner = '[lindex $path 1]' ORDER BY 1"
	}
	default {
	    set selectCmd "SELECT tablename, type FROM domain.tables
                             WHERE owner = '[lindex $path 1]' ORDER BY 1"
	}
    }
    if [catch {callSql $selectCmd}] {return ""}
    adafetch $cur -command {
	switch @2 {
	    SYNONYM {set type synonym}
	    default {set type table}
	}
	lappend allTables [list $type @1]
    }
    return $allTables
}

proc showTree_listViews {c path} {
    global cur connect

    switch $connect(sqlmode) {
	oracle {
	    set selectCmd "SELECT view_name FROM sys.all_views
                             WHERE owner = '[lindex $path 1]' ORDER BY 1"
	}
	default {
	    set selectCmd "SELECT viewname FROM domain.views
                             WHERE owner = '[lindex $path 1]' ORDER BY 1"
	}
    }
    if [catch {callSql $selectCmd}] {return ""}
    adafetch $cur -command {lappend allViews [list view @1]}
    return $allViews
}

proc showTree_listIndexes {c path} {
    global cur connect

    switch $connect(sqlmode) {
	oracle {
	    set selectCmd "SELECT index_name, table_name FROM sys.all_indexes
                             WHERE owner = '[lindex $path 1]' ORDER BY 1"
	}
	default {
	    set selectCmd "SELECT indexname, tablename FROM domain.indexes
                             WHERE owner = '[lindex $path 1]' ORDER BY 1"
	}
    }
    if [catch {callSql $selectCmd}] {return ""}
    adafetch $cur -command {
	lappend allIndexes [list index [format "%s.%s" @2 @1]]
    }
    return $allIndexes
}

proc showTree_tabColumns {isSynonym c path} {
    global cur adamsg connect

    set owner [lindex $path 1]
    set tab   [lindex $path 3]
    if {![string length $owner] || ![string length $tab]} return

    if $isSynonym {
	switch $connect(sqlmode) {
	    oracle {
		set selectCmd "SELECT table_owner, table_name
                                 FROM sys.all_synonyms
		                 WHERE owner        = '[lindex $path 1]'
                                   AND synonym_name = '$tab'"
	    }
	    default {
		set selectCmd "SELECT tableowner, tablename
                                 FROM domain.synonyms
                                 WHERE owner       = '$owner'
                                   AND synonymname = '$tab'"
	    }
	}
	if [catch {callSql $selectCmd}] {
	    tk_messageBox -title "Missing privilege" -icon error -message \
		    "Missing privilege to see definition of $owner.$tab"
	    return
	}
	set row   [adafetch $cur]
	set owner [lindex $row 0]
	set tab   [lindex $row 1]

	set title "[lindex $path 3]: Synonym for $owner.$tab"
    } else {
	set title "Columns of $owner.$tab"
    }

    switch $connect(sqlmode) {
	oracle {
	    set selectCmd "SELECT column_name, data_type, data_length,
                                  data_precision, nullable
	                     FROM sys.all_tab_columns
	                     WHERE table_name = '$tab' AND owner = '$owner'
	                     ORDER BY column_id"
	}
	default {
	    set selectCmd "SELECT columnname, datatype, len, dec, mode
	                     FROM domain.columns
	                     WHERE tablename = '$tab' AND owner = '$owner'
	                     ORDER BY pos"
	}
    }
    callSql $selectCmd

    adacols  $cur
    adafetch $cur -sqlmode adabas -command {
	set name @1
	set t    @2
	set l    @3
	set d    @4
	set m    @5

	switch $t {
	    DATE - TIME - TIMESTAMP - BOOLEAN - INTEGER - SMALLINT {
		set type $t
	    }
	    FLOAT {
		set type ${t}($l,$d)
	    }
	    default {
		set type ${t}($l)
	    }
	}
	switch $connect(sqlmode) {
	    oracle {
		switch $m {
		    N {set mod "NOT NULL"}
		    Y {set mod ""}
		}
	    }
	    default {
		switch $m {
		    KEY {set mod KEY}
		    OPT {set mod ""}
		    MAN {set mod "NOT NULL"}
		}
	    }
	}
	lappend plist [list $name $type $mod]
    }

    if ![llength $plist] {
	setMsg "No fields in table $owner.$tab"
	return
    }

    pickList $title "Columnname Type Mod" $plist
}

proc showTree_viewDef {c path} {
    global cur connect

    set owner [lindex $path 1]
    set tab   [lindex $path 3]

    switch $connect(sqlmode) {
	oracle {
	    set selectCmd "SELECT text FROM sys.all_views
                             WHERE view_name = '$tab' AND owner = '$owner'"
	}
	default {
	    set selectCmd "SELECT definition FROM domain.viewdefs
                             WHERE viewname = '$tab' AND owner = '$owner'"
	}
    }
    if [catch {callSql $selectCmd}] {
	tk_messageBox -title "Missing privilege" -icon error -message \
		"Missing privilege to see definition of $owner.$tab"
	return
    }

    set longDesc [lindex [adafetch $cur -sqlmode adabas] 0]
    set def      [adareadlong $cur -descriptor $longDesc]
    adacols $cur

    # Try to wrap the view definition, so that it fits nicely into the window.

    set words [split $def " "]
    set line  ""
    set def   ""
    foreach word $words {
	set newLine [concat $line " " $word]
	if {[string length $newLine] >= 30} {
	    if [string length $line] {
		lappend def [list $line]
		set line $word
	    } else {
		lappend def [list $word]
	    }
	} else {
	    append line " " $word
	}
    }
    if [string length $line] {
	lappend def [list $line]
    }

    pickList "View $owner.$tab" Definition $def
}

proc showTree_indColumns {c path} {
    global cur connect

    set owner [lindex $path 1]
    regexp {([^.]*)\.(.*)} [lindex $path end] match tab obj

    switch $connect(sqlmode) {
	oracle {
	    set selectCmd "SELECT column_name, '' FROM sys.all_ind_columns
                             WHERE index_name  = '$obj'
                               AND index_owner = '$owner'
                               AND table_name  = '$tab'
                             ORDER BY column_position"
	}
	default {
	    set selectCmd "SELECT refcolumnname, type
                 FROM domain.$connect(ind_uses_col)
                 WHERE defindexname = '$obj'
                   AND defowner     = '$owner'
                   AND deftablename = '$tab'
                 ORDER BY pos"
	}
    }

    if [catch {callSql $selectCmd}] {
	setMsg "No index columns in index $owner.$tab"
	return
    }
    adacols  $cur
    adafetch $cur -command "lappend plist @1; set uniq @2" -sqlmode adabas

    pickList "Columns of [string tolower $uniq] index $obj" Column $plist
}

########################
#
# pickList
#
#   return a selection from a listbox by calling a proc
#

proc pickList {title heading plist} {
    .objects.title configure -text $title

    set win .objects.f.tab

    # clear old output
    confListbox $win 0 <stop>

    confListbox $win [set colNo 0] <start>
    foreach col $heading {
	setBoxConf  $win $colNo $col
	confListbox $win $colNo -text $col
	incr colNo
    }
    confListbox $win $colNo <stop>

    confListbox $win 0 <normal>
    foreach row $plist {
	set colNo 0
	foreach cell $row {
	    insertListbox $win $colNo $cell
	    incr colNo
	}
    }
    confListbox $win 0 <disabled>
}

proc setNull {} {
    global adamsg

    set oldVal $adamsg(nullvalue)
    if ![topEntry .snull "Set Null Value" Value adamsg(nullvalue)] {
	set adamsg(nullvalue) $oldVal
	return
    }
    set adamsg(nullvalue) [string trim $adamsg(nullvalue)]
}

proc setSpecialNull {} {
    global adamsg

    set oldVal $adamsg(specialnull)
    if ![topEntry .snull "Set Special Null" Value adamsg(specialnull)] {
	set adamsg(specialnull) $oldVal
	return
    }
    set adamsg(specialnull) [string trim $adamsg(specialnull)]
}

proc doSwitch {} {
    global lda cur switchReady switch

    if [winfo exist .sw] {
	wm deiconify .sw
	raise .sw
	return
    }

    toplevel .sw
    wm title .sw "TkQuery: Switch"
    makeTransient .sw

    frame .sw.ak  -relief raised -borderwidth 2
    frame .sw.bd  -relief raised -borderwidth 2
    frame .sw.kb  -relief raised -borderwidth 2
    frame .sw.ad  -relief raised -borderwidth 2
    frame .sw.but -relief raised -borderwidth 2

    label .sw.ak.akhead -text "AK debugging"
    label .sw.bd.bdhead -text "BD debugging"
    label .sw.kb.kbhead -text "KB debugging"
    label .sw.ad.adhead -text "Stored procedures"

    checkbutton .sw.ak.ak -text "AK  ak; not AI, AJ"  -var switch(ak) -an w
    checkbutton .sw.ak.ac -text "AC  ak_cache"        -var switch(ac) -an w
    checkbutton .sw.ak.ai -text "AI  ak_isolev"       -var switch(ai) -an w
    checkbutton .sw.ak.aj -text "AJ  ak_join"         -var switch(aj) -an w
    checkbutton .sw.ak.as -text "AS  ak_sem"          -var switch(as) -an w
    checkbutton .sw.ak.at -text "AT  ak_strat"        -var switch(at) -an w
    checkbutton .sw.ak.ay -text "AY  ak_syn"          -var switch(ay) -an w
    label       .sw.ak.e1 -text ""
    checkbutton .sw.ak.fh -text "FH  fs_ak"           -var switch(fh) -an w
    checkbutton .sw.ak.tp -text "TP  test_ak"         -var switch(tp) -an w
    label       .sw.ak.e2 -text ""
    label       .sw.ak.e3 -text ""
    checkbutton .sw.ak.tb -text "TB  test_bd"         -var switch(tb) -an w

    checkbutton .sw.bd.bd -text "BD  bd; not BI"      -var switch(bd) -an w
    checkbutton .sw.bd.bb -text "BB  bd_buf"          -var switch(bb) -an w
    checkbutton .sw.bd.by -text "BY  bd_byte"         -var switch(by) -an w
    checkbutton .sw.bd.bx -text "BX  bd_index"        -var switch(bx) -an w
    checkbutton .sw.bd.bv -text "BV  bd_inv"          -var switch(bv) -an w
    checkbutton .sw.bd.bo -text "BO  bd_io"           -var switch(bo) -an w
    checkbutton .sw.bd.bq -text "BQ  bd_ioqueue"      -var switch(bq) -an w
    checkbutton .sw.bd.bm -text "BM  bd_keym"         -var switch(bm) -an w
    checkbutton .sw.bd.bl -text "BL  bd_lock"         -var switch(bl) -an w
    checkbutton .sw.bd.bf -text "BF  bd_oflw"         -var switch(bf) -an w
    checkbutton .sw.bd.bp -text "BP  bd_psm"          -var switch(bp) -an w
    checkbutton .sw.bd.bt -text "BT  bd_tbuf"         -var switch(bt) -an w
    checkbutton .sw.bd.bi -text "BI  <bd_interface>"  -var switch(bi) -an w

    checkbutton .sw.kb.kb -text "KB  kb; not KE"      -var switch(kb) -an w
    checkbutton .sw.kb.kd -text "KD  kb_dist"         -var switch(kd) -an w
    checkbutton .sw.kb.kf -text "KF  kb_func"         -var switch(kf) -an w
    checkbutton .sw.kb.kl -text "KL  kb_lock"         -var switch(kl) -an w
    checkbutton .sw.kb.ke -text "KE  kb_lockentry"    -var switch(ke) -an w
    checkbutton .sw.kb.kg -text "KG  kb_log"          -var switch(kg) -an w
    checkbutton .sw.kb.kn -text "KN  kb_net, kb_dist" -var switch(kn) -an w
    checkbutton .sw.kb.kq -text "KQ  kb_qual"         -var switch(kq) -an w
    checkbutton .sw.kb.ks -text "KS  kb_sel"          -var switch(ks) -an w
    checkbutton .sw.kb.kv -text "KV  kb_save"         -var switch(kv) -an w
    label       .sw.kb.e1 -text ""
    checkbutton .sw.kb.fl -text "FL  fs_kb"           -var switch(fl) -an w
    checkbutton .sw.kb.tk -text "TK  test_kb"         -var switch(tk) -an w

    checkbutton .sw.ad.pr -text "PR"                  -var switch(tk) -an w
    checkbutton .sw.ad.pi -text "PI  sproc"           -var switch(pi) -an w
    checkbutton .sw.ad.pc -text "PC  sproc_call"      -var switch(pc) -an w
    checkbutton .sw.ad.ps -text "PS  sproc_sql"       -var switch(ps) -an w
    label       .sw.ad.e1 -text ""
    label       .sw.ad.h2 -text "Other functions"
    checkbutton .sw.ad.gg -text "GG"                  -var switch(gg) -an w
    checkbutton .sw.ad.st -text "ST  st_stack"        -var switch(st) -an w
    checkbutton .sw.ad.uc -text "UC  unicode"         -var switch(uc) -an w

    frame  .sw.but.default -relief sunken -bd 1
    button .sw.but.sw -text Switch -command {set switchReady 1; destroy .sw}
    button .sw.but.re -text Reset  -command {
	foreach currSw [array names switch] {
	    set switch($currSw) 0
	}
    }
    button .sw.but.vt -text Vtrace -command {set switchReady 2; destroy .sw}
    button .sw.but.ca -text Cancel -command {set switchReady 0; destroy .sw}
    bind .sw.but.sw <Return> ".sw.but.sw invoke"
    pack   .sw.but.default .sw.but.re .sw.but.vt .sw.but.ca \
	    -padx 10 -pady 10 -ipadx 10 -fill x
    pack   .sw.but.sw -in .sw.but.default -padx 1 -pady 1 -fill both -expand 1

    pack .sw.ak .sw.bd .sw.kb .sw.ad .sw.but -side left -fill both -expand 1

    pack .sw.ak.akhead .sw.ak.ak .sw.ak.ac .sw.ak.ai .sw.ak.aj .sw.ak.as \
	    .sw.ak.at .sw.ak.ay .sw.ak.e1 .sw.ak.fh .sw.ak.tp .sw.ak.e2  \
	    .sw.ak.e3 .sw.ak.tb -side top -anchor w -fill x
    pack .sw.bd.bdhead .sw.bd.bd .sw.bd.bb .sw.bd.by .sw.bd.bx .sw.bd.bv \
	    .sw.bd.bo .sw.bd.bq .sw.bd.bm .sw.bd.bl .sw.bd.bf .sw.bd.bp  \
	    .sw.bd.bt .sw.bd.bi -side top -anchor w -fill x
    pack .sw.kb.kbhead .sw.kb.kb .sw.kb.kd .sw.kb.kf .sw.kb.kl .sw.kb.ke \
	    .sw.kb.kg .sw.kb.kn .sw.kb.kq .sw.kb.ks .sw.kb.kv .sw.kb.e1  \
	    .sw.kb.fl .sw.kb.tk -side top -anchor w -fill x
    pack .sw.ad.adhead .sw.ad.pr .sw.ad.pi .sw.ad.pc .sw.ad.ps .sw.ad.e1 \
	    .sw.ad.h2 .sw.ad.gg .sw.ad.st .sw.ad.uc                      \
	    -side top -anchor w -fill x

    focus .sw.but.sw
    vwait switchReady
    switch $switchReady {
	1 {
	    set debug ""
	    foreach currSw [array names switch] {
		if $switch($currSw) {
		    lappend debug $currSw
		}
	    }
	    set layer AKBD
	    if [catch {adaspecial $lda switch $layer $debug} msg] {
		setMsg $msg
	    }
	}
	2 {
	    callSql VTRACE -sqlmode adabas
	}
    }
}

proc lvarpop {list} {
    upvar 1 $list x

    set first [lindex $x 0]
    set x [lreplace $x 0 0]
    return $first
}

proc max {a b} {
    if {$a > $b} {return $a} {return $b}
}

proc makeTransient {w} {
    wm transient $w .
    set xpos [expr [winfo rootx .]+30]
    set ypos [expr [winfo rooty .]+30]
    wm geom $w +${xpos}+$ypos
}

proc confMenueEntry {m index args} {
    set tearOffs [info commands .tearoff*]
    if [llength $tearOffs] {
        set m "$m $tearOffs"
    }
    foreach menu $m {
        catch {eval $menu entryconfigure [list $index] $args}
    }
}

proc resizer {w args} {
    global _resizer

    set arguments(-orientation) horizontal
    if [catch {array set arguments $args} msg] {
	return -code error $msg
    }
    switch $arguments(-orientation) {
	horizontal {
	    frame $w -height 2
	    set cursor sb_v_double_arrow
	}
	vertical {
	    frame $w -width 2
	    set cursor sb_h_double_arrow
	}
	default {
	    return -code error "Unknown orientation $arguments(-orientation)"
	}
    }
    if [info exists arguments(-neighbor)] {
	set _resizer($w,nb) $arguments(-neighbor)
    }

    bind  $w <Enter> "$w configure -cursor $cursor"
    bind  $w <Leave> "$w configure -cursor {}"
    set orient $arguments(-orientation)
    bind  $w <1>                   "resizeDown $w $orient %x %y"
    bind  $w <B1-Motion>           "resizeMove $w $orient %x %y"
    bind  $w <Any-ButtonRelease-1> "resizeUp   $w $orient %x %y"

    return $w
}

proc resizeDown {w orient relX relY} {
    global _resizer

    set _resizer($w,parent)  [winfo parent  $w]
    wm geometry $_resizer($w,parent) [wm geometry $_resizer($w,parent)]
    set siblings [lsort -command "cmpPos $orient" \
	    [[winfo manager $w] slaves $_resizer($w,parent)]]
    set wPos     [lsearch -exact $siblings $w]

    if ![info exists _resizer($w,nb)] {
	set _resizer($w,nb)  [lindex $siblings [expr $wPos-1]]
    }
    set _resizer($w,nb2) [lindex $siblings [expr $wPos+1]]
    set _resizer($w,xl)  [winfo x $_resizer($w,nb)]
    set _resizer($w,yl)  [winfo y $_resizer($w,nb)]
    set _resizer($w,xh)  [expr {[winfo x $_resizer($w,nb2)] + \
	    			[winfo width  $_resizer($w,nb2)]}]
    set _resizer($w,yh)  [expr {[winfo y $_resizer($w,nb2)] + \
	    			[winfo height $_resizer($w,nb2)]}]

    set _resizer($w,relYmin) [expr {$_resizer($w,yl)-[winfo y $w]+8}]
    set _resizer($w,relYmax) [expr {$_resizer($w,yh)-[winfo y $w]-8}]
    set _resizer($w,relXmin) [expr {$_resizer($w,xl)-[winfo x $w]+8}]
    set _resizer($w,relXmax) [expr {$_resizer($w,xh)-[winfo x $w]-8}]

    switch $orient {
	horizontal {
	    set width  [winfo width $_resizer($w,parent)]
	    set height 1
	    set x      $_resizer($w,xl)
	    set y      [expr {[winfo y $w]+$relY}]
	}
	vertical {
	    set width  1
	    set height [winfo height $_resizer($w,parent)]
	    set x      [expr {[winfo x $w]+$relX}]
	    set y      $_resizer($w,yl)
	}
    }
    if [string compare $_resizer($w,parent) .] {
	set _resizer($w,line) $_resizer($w,parent).line
    } else {
	set _resizer($w,line) .line
    }
    frame $_resizer($w,line) -width $width -height $height \
	    -background black -highlightthickness 0
    place $_resizer($w,line) -x $x -y $y
}

proc resizeMove {w orient relX relY} {
    global _resizer

    switch $orient {
	horizontal {
	    if {$relY < $_resizer($w,relYmin)} {
		set relY $_resizer($w,relYmin)
	    } elseif {$relY > $_resizer($w,relYmax)} {
		set relY $_resizer($w,relYmax)
	    }
	    set x $_resizer($w,xl)
	    set y [expr {[winfo y $w]+$relY}]
	}
	vertical {
	    if {$relX < $_resizer($w,relXmin)} {
		set relX $_resizer($w,relXmin)
	    } elseif {$relX > $_resizer($w,relXmax)} {
		set relX $_resizer($w,relXmax)
	    }
	    set x [expr {[winfo x $w]+$relX}]
	    set y $_resizer($w,yl)
	}
    }
    place $_resizer($w,line) -x $x -y $y
}

proc resizeUp {w orient relX relY} {
    global _resizer

    destroy $_resizer($w,line)
    catch {grid propagate $_resizer($w,nb) 0}
    catch {pack propagate $_resizer($w,nb) 0}

    switch $orient {
	horizontal {
	    if {$relY < $_resizer($w,relYmin)} {
		set relY $_resizer($w,relYmin)
	    } elseif {$relY > $_resizer($w,relYmax)} {
		set relY $_resizer($w,relYmax)
	    }
	    set newHeight    [expr {[winfo height $_resizer($w,nb)]+$relY-4}]
	    set realHeight   [expr {[winfo height $_resizer($w,nb)]-6}]
	    set widgetHeight [$_resizer($w,nb) cget -height]
	    if $widgetHeight {
		set newHeight [expr int(double($newHeight*$widgetHeight)/$realHeight)]
	    }
	    $_resizer($w,nb) configure -height $newHeight
	}
	vertical {
	    if {$relX < $_resizer($w,relXmin)} {
		set relX $_resizer($w,relXmin)
	    } elseif {$relX > $_resizer($w,relXmax)} {
		set relX $_resizer($w,relXmax)
	    }
	    set newWidth    [expr {[winfo width $_resizer($w,nb)]+$relX-4}]
	    set realWidth   [expr {[winfo width $_resizer($w,nb)]-6}]
	    set widgetWidth [$_resizer($w,nb) cget -width]
	    if $widgetWidth {
		set newWidth [expr int(double($newWidth*$widgetWidth)/$realWidth)]
	    }
	    $_resizer($w,nb) configure -width $newWidth
	}
    }
}

proc cmpPos {orient w1 w2} {
    switch $orient {
	horizontal {return [expr [winfo y $w1]-[winfo y $w2]]}
	vertical   {return [expr [winfo x $w1]-[winfo x $w2]]}
    }
}

# ---------------------------------------------------------------------
# 'adaConnect' will be called, when the user presses the Connect button
# in the connect dialog. It tries to connect with the given user/password
# combination to the database, and sets - if succesful - the global
# variables 'lda' (login handle) and 'cur' (cursor).
# ---------------------------------------------------------------------
proc adaConnect {} {
    global connect lda cur

    if ![tryConnect] {return 0}

    set lda $connect(db)
    set cur [adaopen $lda]

    return 1
}

proc adaReconnect {} {
    global cur lda options adamsg

    catch {adaclose  $cur}
    catch {adalogoff $lda}

    if ![adaConnect] {
	catch {unset adamsg(errortxt)}
	setMsg "Connection to database server broken"
	return 0
    }

    initOptions .o.c
    if !$options(val:autoCommit) {
	setMsg "Work rolled back (inactivity timeout)"
    }
    return 1
}

proc initOptions {c} {
    global adamsg options

    setAutoCommit 1
    setFormat     1
    setProt
    set adamsg(nullvalue)   $options(val:nullValue)
    set adamsg(specialnull) $options(val:specialNull)
}

# ---------------------------------------------------------------------
# Mainprogram
# ---------------------------------------------------------------------
wm withdraw .
loadAdabastcl
getDbnameUserPassword "Adabas D: TkQuery" adaConnect
createMain
