# ---------------------------------------------------------------------
# $Id: tkuser.tcl,v 1.24 1997/07/03 22:06:08 adabas Exp $
# ---------------------------------------------------------------------

# ---------------------------------------------------------------------
# If a xuser file exists, the user must give name and password of the
# DEFAULT entry. For this we play up a reduced version of the usual
# connect window (it has no serverdb and servernode entries)
# ---------------------------------------------------------------------
proc checkUserPw {} {
    global env bitmap connect tkCompat

    if {[catch {array set default [adabas xuser get]}]} return

    # Give the user name as a clue.
    set connect(user) $default(user)

    toplevel .connect
    
    image create bitmap logo -data $bitmap(sag-logo) -back "" -foreground blue
    image create bitmap sag  -data $bitmap(sag)      -back "" -foreground black
    frame .connect.sag
    label .connect.sag.logo -image logo
    label .connect.sag.sag  -image sag
    pack  .connect.sag -fill x -expand 1 -padx 30 -pady 10
    pack  .connect.sag.logo -side left -fill x -expand 1 -anchor e
    pack  .connect.sag.sag  -side left -fill x -expand 1 -anchor w

    message .connect.msg -text "Adabas D User" -aspect 1000
    if $tkCompat(fontCmd) {
	.connect.msg configure -font "Times 24 normal"
    } else {
	catch {.connect.msg configure -font -*-Times-Bold-R-*-*-24-*-*-*-*-*-*-*}
    }
    pack  .connect.msg -fill x -pady 10 -padx 10

    frame .connect.up -relief sunken -border 2
    pack  .connect.up -expand 1 -fill x -padx 20 -ipadx 5 -ipady 4

    frame .connect.user
    label .connect.user.text -text "Username:"
    entry .connect.user.entr -textvariable connect(user) \
	    -relief sunken -width 18 -highlightthickness 0 -bg white
    pack  .connect.user.text -side left
    pack  .connect.user.entr -side left -expand 1 -pady 3 -anchor e
    pack  .connect.user      -expand 1 -fill x -in .connect.up \
	    -padx 7 -anchor s

    frame .connect.passwd
    label .connect.passwd.text -text "Password:"
    entry .connect.passwd.entr -textvariable connect(passwd) -show * \
	    -relief sunken -width 18 -highlightthickness 0 -bg white
    pack  .connect.passwd.text -side left
    pack  .connect.passwd.entr -side left -expand 1 -pady 3 -anchor e
    pack  .connect.passwd      -expand 1 -fill x -in .connect.up \
	    -padx 7 -anchor n

    message .connect.err -textvariable connect(error) -aspect 10000
    pack    .connect.err -fill x  

    frame  .connect.but
    button .connect.but.ok  -text Connect \
	    -command [list compareUser .connect $default(user) $default(password)]
    button .connect.but.end -text "End (F3)" -command "exit"
    pack .connect.but.ok  -side left -fill x -expand 1
    pack .connect.but.end -side left -fill x -expand 1
    pack .connect.but     -fill x -expand 1

    focus .connect.passwd.entr

    bind .connect.user.entr   <Return> "focus .connect.passwd.entr"
    bind .connect.passwd.entr <Return> \
	    ".connect.passwd.entr icursor 0; .connect.but.ok invoke"

    bind .connect <F3> ".connect.but.end invoke"
    wm title .connect "Adabas D User"

    tkwait window .connect
    
    image delete logo
    image delete sag
}

# -------------------------------------------------------------------------
# "compareUser" will be called as an event handler, if the user presses the
# okay button in the connect dialog. It has to compare the entered values
# with the values read from the xuser file (which are given as parameters).
# -------------------------------------------------------------------------
proc compareUser {w user password} {
    global connect
    
    foreach entry {user passwd} {
	if {![string length $connect($entry)]} {
	    lappend missing $entry
	}
    }
    if {[info exists missing]} {
	set connect(error) "Please enter [join $missing " and "]"
	return 0
    }

    if {[string compare [string toupper $user] \
	                [string toupper $connect(user)]] || \
        [string compare [adabas crypt [string toupper $connect(passwd)]] \
                        $password]} {
	set connect(error) "Unknown user/password combination"
	return 0
    }
    
    destroy $w
}

# -----------------------------------------------------------------------
# The procedure, that creates all the elements in the toplevel, does the
# key binding, reads all the xuser entries and displays the first record.
# -----------------------------------------------------------------------
proc createMain {} {
    global adamsg list changeStatus currXUser currUser knlVersion optMenu tkCompat

    regexp {Adabas D ([^)]*)} $adamsg(version) match knlVersion

    button  .fontTest
    set font [.fontTest cget -font]
    destroy .fontTest

    option add *Listbox.foreground         black widgetDefault
    option add *Listbox.background         white widgetDefault
    option add *Entry.foreground           black widgetDefault
    option add *Entry.background           white widgetDefault
    option add *Entry.font                 $font widgetDefault
    option add *Button.takeFocus           0     widgetDefault
    option add *Entry.highlightThickness   0     widgetDefault
    option add *Listbox.highlightThickness 0     widgetDefault

    wm title    . "Adabas D Users"

    frame .list   -bd 2 -relief raised
    frame .entry  -bd 2 -relief raised
    frame .quit   -bd 2 -relief raised
    frame .butt   -bd 2 -relief raised
    frame .status -bd 2 -relief raised
    grid .list   -row 0 -column 0 -sticky nsew -ipadx 20 -ipady 10
    grid .entry  -row 0 -column 1 -sticky nsew -ipadx 10 -ipady 10
    grid .quit   -row 0 -column 2 -sticky nsew -rowspan 2
    grid .butt   -row 1 -column 0 -sticky nsew -columnspan 2 -ipady 10
    grid .status -row 3 -column 0 -sticky nsew -columnspan 3 -sticky ew
    grid rowconfigure    . 0 -weight $tkCompat(gridWeightMax)
    grid columnconfigure . 1 -weight $tkCompat(gridWeightMax)

    label      .list.h -text "User keys"
    grid       .list.h -row 0 -column 0 -sticky w

    set list .list.l
    listbox $list -height 16 -width 20
    grid    $list -row 1 -column 0
    bindtags $list "Listbox $list"
    bind $list <FocusIn>    {                   set currXUser [$list get active]}
    bind $list <Any-Button> {set statusLine ""; set currXUser [$list get anchor]}
    bind $list <Any-Key>    {set statusLine ""; set currXUser [$list get active]}
    bind $list <Tab>        {set statusLine ""; focus [tk_focusNext $list]}
    trace variable currXUser w showXUser

    label      .entry.h -text "Attributes"
    grid       .entry.h -row 0 -column 0 -sticky w

    if $tkCompat(fontCmd) {
	.list.h  configure -font "Times 18 bold"
	.entry.h configure -font "Times 18 bold"
    } else {
	catch {.list.h  configure -font -*-Times-Bold-R-Normal-*-18-*-*-*-*-*-*-*}
	catch {.entry.h configure -font -*-Times-Bold-R-Normal-*-18-*-*-*-*-*-*-*}
    }

    label .entry.lab:userkey -text "User Key"
    entry .entry.txt:userkey -textvar currUser(userkey)
    grid  .entry.lab:userkey -row 1 -column 0 -sticky w
    grid  .entry.txt:userkey -row 1 -column 1 -padx 10 -pady 4

    label .entry.lab:user -text "User Name"
    entry .entry.txt:user -textvar currUser(user)
    grid  .entry.lab:user -row 2 -column 0 -sticky w
    grid  .entry.txt:user -row 2 -column 1 -padx 10 -pady 4

    label .entry.lab:passwd -text "Password"
    entry .entry.txt:passwd -textvar currUser(passwd) -show *
    grid  .entry.lab:passwd -row 3 -column 0 -sticky w
    grid  .entry.txt:passwd -row 3 -column 1 -padx 10 -pady 4

    label .entry.lab:passwd2 -text "Confirm Password"
    entry .entry.txt:passwd2 -textvar currUser(passwd2) -show *
    grid  .entry.lab:passwd2 -row 4 -column 0 -sticky w
    grid  .entry.txt:passwd2 -row 4 -column 1 -padx 10 -pady 4

    label .entry.lab:serverdb -text "Server DB"
    entry .entry.txt:serverdb -textvar currUser(serverdb)
    grid  .entry.lab:serverdb -row 5 -column 0 -sticky w
    grid  .entry.txt:serverdb -row 5 -column 1 -padx 10 -pady 4

    label .entry.lab:servernode -text "Server Node"
    entry .entry.txt:servernode -textvar currUser(servernode)
    grid  .entry.lab:servernode -row 6 -column 0 -sticky w
    grid  .entry.txt:servernode -row 6 -column 1 -padx 10 -pady 4

    if {$knlVersion >= 6.2} {
	label .entry.lab:dblang -text "Locale"
	entry .entry.txt:dblang -textvar currUser(dblang)
	grid  .entry.lab:dblang -row 7 -column 0 -sticky w
	grid  .entry.txt:dblang -row 7 -column 1 -padx 10 -pady 4
    }

    label .entry.lab:sqlmode -text "SQL Mode"
    set optMenu(sqlmode) [tk_optionMenu .entry.txt:sqlmode \
	    currUser(sqlmode) Adabas Oracle Ansi DB2]
    .entry.txt:sqlmode configure -background white -anchor w \
	    -takefocus 1 -highlightthickness 0
    grid  .entry.lab:sqlmode -row 8 -column 0 -sticky w
    grid  .entry.txt:sqlmode -row 8 -column 1 -padx 10 -pady 1 -sticky ew

    if {$knlVersion < 6.2} {
	label .entry.lab:cachelimit -text "Cachelimit"
	entry .entry.txt:cachelimit -textvar currUser(cachelimit)
	grid  .entry.lab:cachelimit -row 9 -column 0 -sticky w
	grid  .entry.txt:cachelimit -row 9 -column 1 -padx 10 -pady 4
    }

    label .entry.lab:timeout -text "Timeout"
    entry .entry.txt:timeout -textvar currUser(timeout)
    grid  .entry.lab:timeout -row 10 -column 0 -sticky w
    grid  .entry.txt:timeout -row 10 -column 1 -padx 10 -pady 4

    label .entry.lab:isolation -text "Isolation Level"
    set optMenu(isolation) [tk_optionMenu .entry.txt:isolation \
	    currUser(isolation) -1 0 1 15 2 3]
    .entry.txt:isolation configure -background white -anchor w \
	    -takefocus 1 -highlightthickness 0
    grid  .entry.lab:isolation -row 11 -column 0 -sticky w
    grid  .entry.txt:isolation -row 11 -column 1 -padx 10 -pady 1 -sticky ew

    for {set i 0} {$i < 12} {incr i} {
	grid rowconfigure .entry $i -weight $tkCompat(gridWeightMax)
    }

    label  .quit.header
    button .quit.cancel   -width 11 -text "Cancel" -und 0 -com cancelXUser
    button .quit.ok       -width 11 -text "Ok"     -und 0 -command exitXUser
    label  .quit.footer
    grid   .quit.header   -row 0 -column 0
    grid   .quit.cancel   -row 1 -column 0 -pady 5 -padx 10
    grid   .quit.ok       -row 2 -column 0 -pady 5 -padx 10
    grid   .quit.footer   -row 3 -column 0 -sticky nesw
    grid rowconfigure .quit 3 -weight $tkCompat(gridWeightMax)

    button .butt.save      -width 11 -text "Save"       -und 0 -com saveXUser
    button .butt.clear     -width 11 -text "Clear"      -und 1 -com clearXUser
    button .butt.delete    -width 11 -text "Delete"     -und 0 -com deleteXUser
    button .butt.deleteAll -width 11 -text "Delete All" -und 7 -com delAllXUser

    set saveAsDefault [makeDefaultButton .butt.save]

    grid   $saveAsDefault  -row 0 -column 0 -padx 5
    grid   .butt.clear     -row 0 -column 1 -padx 5
    grid   .butt.delete    -row 0 -column 2 -padx 5
    grid   .butt.deleteAll -row 0 -column 3 -padx 5

    foreach w ". $list" {
	bind $w <Return> ".butt.save      invoke"
	bind $w <Alt-c>  ".quit.cancel    invoke"
	bind $w <Alt-o>  ".quit.ok        invoke"
	bind $w <Alt-s>  ".butt.save      invoke"
	bind $w <Alt-l>  ".butt.clear     invoke"
	bind $w <Alt-a>  ".butt.deleteAll invoke"
	bind $w <Alt-d>  ".butt.delete    invoke"
	bind $w <F3>     ".quit.cancel    invoke"
	bind $w <F5>     ".quit.ok        invoke"
    }

    label .status.msg  -textvariable statusLine -anchor w
    grid  .status.msg  -row 0 -column 1 -sticky ew
    grid  columnconfigure .status 1 -weight $tkCompat(gridWeightMax)

    set changeStatus unchanged

    readXUser

    # The update is an answer to the (buggy?) effect, that without it first
    # the small empty frame pops up.
    update
    wm deiconify .

    # If at least one user exists, highlight the first in
    # the listbox, else fill the entries with defaults.
    if {![$list size]} {
	clearXUser
	trace variable currUser w checkChanges
    } else {
	listActivateUser 0
    }
}

# -----------------------------------------------------------------------
# Procedure, that makes the given button look like the default button.
# Only needed due to backward compatibility with Tk4.x.
# -----------------------------------------------------------------------
proc makeDefaultButton {button} {
    global tkCompat

    if [info exists tkCompat(defaultButton)] {
	$button configure -default $tkCompat(defaultButton)
	return $button
    } else {
	regsub {\.[^.]*$} $button ".default" defButton
	frame $defButton -relief sunken -bd 1
	pack  $button -in $defButton -padx 1 -pady 1
	raise $button
	return $defButton
    }
}

# -----------------------------------------------------------------------
# Utility procedure, that displays the given text as error popup.
# -----------------------------------------------------------------------
proc errorMsg {text} {
    tk_messageBox -icon error -title "Error" -message $text
}

# -----------------------------------------------------------------------
# Utility procedure, that prompts the given text as a yes no question.
# -----------------------------------------------------------------------
proc question {head text} {
    return [string compare no \
	    [tk_messageBox -icon question -default yes -type yesno \
	    -title $head -message $text]]
}
	    
# -----------------------------------------------------------------------
# Utility procedure, that highlights the userkey with the given index
# in the listbox.
# -----------------------------------------------------------------------
proc listActivateUser {index} {
    global list

    $list activate $index
    $list selection clear 0 end
    $list selection set $index
    focus $list
}

# -----------------------------------------------------------------------
# Look for the specified userkey in the listbox and return its index.
# -----------------------------------------------------------------------
proc listGetIndex {key} {
    global list

    set userCnt  [$list size]
    for {set index 0} {$index < $userCnt} {incr index} {
	if {![string compare $key [$list get $index]]} {
	    return $index
	}
    }
    return -1
}

# -----------------------------------------------------------------------
# Execute the given adabas xuser command. Catch errors and put them into
# the status line.
# -----------------------------------------------------------------------
proc xuser {args} {
    global statusLine

    if {[catch {uplevel 1 adabas xuser $args} msg]} {
	errorMsg $msg
	return -code error $msg
    } else {
	set statusLine ""
	return $msg
    }
}

# -----------------------------------------------------------------------
# Procedure, that reads the complete xuser file and stores all user entries
# in al global array allUsers, indexed by userkey and property. This
# proc should be called only once at startup time, since it creates a backup
# copy of allUsers in backupValues, that can be used by the Cancel proc.
# -----------------------------------------------------------------------
proc readXUser {} {
    global allUsers backupValues backupKeys list

    $list delete 1 end

    if {[catch {xuser open}]} return
    set index 0
    while {![catch {array set user [adabas xuser index [incr index]]}]} {
	$list insert end $user(userkey)
	foreach entry [array names user] {
	    set allUsers($user(userkey),$entry) $user($entry)
	}
    }
    if {[catch {xuser close}]} return

    # backup allUsers into backupUsers
    catch {array set backupValues [array get allUsers]}
    set backupKeys [$list get 0 end]
}

# -----------------------------------------------------------------------
# This proc saves all the xuser entries shown in the listbox by clearing
# the xuser, and then adding every mentioned user; this is the only way
# to obtain the order of the user keys.
# -----------------------------------------------------------------------
proc writeXUser {} {
    global allUsers list

    set ret 1

    if {[catch {xuser clear}]} {return 0}
    if {[catch {xuser open }]} {return 0}
    for {set index 0} {$index < [$list size]} {incr index} {
	set userKey [$list get $index]
	if {[string length $userKey]} {
	    regsub -all " $userKey," " [array get allUsers $userKey,*]" " " values

	    # Note, that it is better to continue after an error while writing
	    # one entry, since all the other entries would be lost otherwise.
	    if {[catch "xuser put $values"]} {set ret 0}
	}
    }
    if {[catch {xuser close}]} {return 0}
    
    return $ret
}

# -----------------------------------------------------------------------
# "showUser" is the call back, if some user is selected in the listbox.
# All the entries will be updated to show the values of this user.
# -----------------------------------------------------------------------
proc showXUser {name blurb op} {
    global currUser allUsers changeStatus
    upvar #0 $name xuser

    switch $changeStatus {
	changed       {set newStatus unchanged}
	saved,changed {set newStatus saved}
	default       {set newStatus $changeStatus}
    }

    # Note: This will also unset the old trace of currUser.
    catch {unset currUser}
    set currUser(passwd)  ""
    set currUser(passwd2) ""

    foreach userEntry [array names allUsers $xuser,*] {
	set entry [string range $userEntry [string length $xuser,] end]
	set currUser($entry) $allUsers($userEntry)
    }

    switch $currUser(sqlmode) {
	oracle  {set currUser(sqlmode) Oracle}
	ansi    {set currUser(sqlmode) Ansi}
	db2     {set currUser(sqlmode) DB2}
	default {set currUser(sqlmode) Adabas}
    }

    trace variable currUser w checkChanges
    set changeStatus $newStatus
}

# -----------------------------------------------------------------------
# Trace procedure, that sets the global variable changeStatus; it will be
# called, if the user changes any entry (and so any element of currUser).
# -----------------------------------------------------------------------
proc checkChanges {name entry op} {
    global allUsers changeStatus statusLine optMenu
    upvar #0 $name currUser

    set statusLine   ""
    switch $changeStatus {
	unchanged {set changeStatus changed}
	saved     {set changeStatus saved,changed}
    }
    switch $currUser(sqlmode) {
	Adabas {
	    $optMenu(isolation) entryconfigure *15* -state normal
	}
	default {
	    $optMenu(isolation) entryconfigure *15* -state disabled
	}
    }
    switch -- $currUser(isolation) {
	15 {
	    foreach nonAdabasMode {Ansi Oracle DB2} {
		$optMenu(sqlmode) entryconfigure $nonAdabasMode -state disabled
	    }
	}
	default {
	    foreach nonAdabasMode {Ansi Oracle DB2} {
		$optMenu(sqlmode) entryconfigure $nonAdabasMode -state normal
	    }
	}
    }
}

# -----------------------------------------------------------------------
# Procedure to handle a press of the Cancel button: Restore the initial
# values into the allUsers array and write them onto disk, not after
# asking the user, if she really wants to cancel all her work.
# -----------------------------------------------------------------------
proc cancelXUser {} {
    global allUsers backupValues backupKeys changeStatus list

    if {$changeStatus != "unchanged"} {
	if {![question "Cancel" \
		"Do you really want to cancel all your changes?"]} {
	    return
	}
	if {[catch {array set allUsers [array get backupValues]}]} {
	    unset allUsers
	}
	$list delete 0 end
	eval $list insert end $backupKeys
	if {![writeXUser]} return
    }
    exit
}

# -----------------------------------------------------------------------
# Procedure to handle a press of the Ok button: Since with every press
# of the Save button the complete xuser file is written, here we only have 
# to check, if some editing of the current user is still to save.
# -----------------------------------------------------------------------
proc exitXUser {} {
    saveXUser askUser
    exit
}

# -----------------------------------------------------------------------
# Procedure to handle a press of the Delete All button: It deletes all
# users in the list and in the xuser file also.
# -----------------------------------------------------------------------
proc delAllXUser {} {
    global list statusLine currXUser changeStatus allUsers

    set userCnt [$list size]
    if {!$userCnt} {
	set statusLine "No user to delete"
	return
    }
    $list delete 0 end
    clearXUser
    catch {unset allUsers}
    set currXUser ""

    if {[writeXUser]} {
	set statusLine   "All user deleted"
	set changeStatus saved
    }
}

# -----------------------------------------------------------------------
# Utility procedure, that puts some reasonable values into the empty
# entries of the current user.
# -----------------------------------------------------------------------
proc resetCurrUser {} {
    global currUser knlVersion

    foreach entry [array names currUser] {
	set currUser($entry) ""
    }
    set currUser(sqlmode)   Adabas
    set currUser(isolation)  -1
    set currUser(timeout)    -1
    if {$knlVersion < 6.2} {
	set currUser(cachelimit) -1
    }
}

# -----------------------------------------------------------------------
# Procedure to handle a press of the Clear button: The current user will
# be reset to an (almost) empty state, so that new user data can be entered.
# A little logic sparcles here: There is no sense to call this function, if
# all there are already 16 users; and the userkey will be initailized as
# DEFAULT, if currently no userkey at all exists.
# -----------------------------------------------------------------------
proc clearXUser {} {
    global list currUser currXUser

    set userCnt [$list size]
    if {$userCnt >= 16} {
	errorMsg "No more entries possible"
	return
    }

    saveXUser askUser

    resetCurrUser

    $list selection clear 0 end

    if {!$userCnt} {
	set currUser(userkey) DEFAULT
	focus .entry.txt:user
    } else {
	focus .entry.txt:userkey
    }
}

# -----------------------------------------------------------------------
# Procedure to handle a press of the Delete button: The currently
# displayed user entry will be deleted (from screen and from disk).
# -----------------------------------------------------------------------
proc deleteXUser {} {
    global list statusLine currXUser changeStatus

    set userCnt [$list size]
    if {!$userCnt} {
	set statusLine "Nothing to delete"
	return
    }
    if {![string compare $currXUser DEFAULT]} {
	errorMsg "Not allowed to delete user entry DEFAULT"
	return
    }
    set index [listGetIndex $currXUser]
    $list delete $index
    set currXUser [$list get $index]

    if {[writeXUser]} {
	set statusLine   "Deleted"
	set changeStatus saved
    }

    listActivateUser $index
}

# -----------------------------------------------------------------------
# Procedure to handle a press of the Save button (and the Return key):
# The currently displayed user entry, which was changed by the user,
# will be saved into the xuser file. Before that, a lot of checks are to
# be done; and there are still more checks to come...
# -----------------------------------------------------------------------
proc saveXUser {{ask no}} {
    global currUser allUsers currXUser statusLine changeStatus list knlVersion

    switch $changeStatus {
	unchanged - saved {
	    if {![string compare $ask no]} {
		set statusLine "No changes need to be saved"
	    }
	    return
	}
    } 
    set key $currUser(userkey)
    if {![string length $key]} {
	if {[string compare $ask askUser]} {
	    errorMsg "Empty user key not allowed"
	    focus .entry.txt:userkey
	}
	return
    }

    if {![string length $currUser(user)]} {
	if {[string compare $ask askUser]} {
	    errorMsg "Empty user not allowed"
	    focus .entry.txt:user
	}
	return
    }
    set currUser(user) [string toupper $currUser(user)]

    if {![string compare $currUser(userkey) DEFAULT]
	  && (![info exists currUser(password)]
	      || ![string length $currUser(password)])
	  && ![string length $currUser(passwd)]} {
	errorMsg "For the default user must be a password specified"
	return
    }

    if {![string compare $ask askUser]} {
	if {![question "Save" "Save changes of current user?"]} {
	  return
	}
    }

    if {![$list size] && [string compare $currUser(userkey) DEFAULT]} {
	errorMsg "First userkey must be DEFAULT"
	return
    }

    if {[string compare $currUser(passwd) $currUser(passwd2)]} {
	errorMsg "Confirm the password correctly"
	set currUser(passwd)   ""
	set currUser(passwd2)  ""
	return
    }

    # Look for the given userkey in the list of allkeys.
    set index [listGetIndex $key]

    if {[info exists currXUser] \
	    && ![string compare $currXUser $currUser(userkey)]} {
	if {![question "Update" "Do you want to update the user record?"]} {
	    return
	}
    }

    if {![info exists currXUser] \
	    || [string compare $currXUser $currUser(userkey)]} {
	if {$index >= 0} {
	    errorMsg "Userkey already used, choose new userkey"
	    return
	}
	set userCnt [$list size]
	if {$userCnt >= 16} {
	    errorMsg "No more entries possible"
	    return
	}
	$list insert end $key
	listActivateUser end
    }

    if {[catch {expr int($currUser(timeout))}]               \
	    || int($currUser(timeout)) != $currUser(timeout) \
	    || $currUser(timeout) < -1                       \
	    || $currUser(timeout) > 65535} {
	errorMsg "Timeout is an invalid integer"
	return
    }

    if {$knlVersion < 6.2} {
	if {[catch {expr int($currUser(cachelimit))}]                  \
		|| int($currUser(cachelimit)) != $currUser(cachelimit) \
		|| $currUser(cachelimit) < -1                          \
		|| $currUser(cachelimit) > 2000000000} {
	    errorMsg "Cachelimit is an invalid integer"
	    return
	}
    }

    foreach entry [array names currUser] {
	set allUsers($key,$entry) $currUser($entry)
    }
    switch $currUser(sqlmode) {
	Oracle  {set allUsers($key,sqlmode) oracle}
	Ansi    {set allUsers($key,sqlmode) ansi}
	DB2     {set allUsers($key,sqlmode) db2}
	default {set allUsers($key,sqlmode) adabas}
    }

    # Be sure to write the crypted password only, if the user gives a new one.
    if {[string length $allUsers($key,passwd)]} {
	set allUsers($key,password) \
		[adabas crypt [string toupper $allUsers($key,passwd)]]
    } elseif {![info exists allUsers($key,password)]} {
	set allUsers($key,password) ""
    }
    unset allUsers($key,passwd) allUsers($key,passwd2)

    set currXUser   $key

    if {[writeXUser]} {
	set statusLine  "Saved"
	set changeStatus saved
    }
}

set bitmap(sag) {
#define sag_width 255
#define sag_height 45
static unsigned char sag_bits[] = {
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0xff,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xff,
0x1f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0xc0,0xff,0x1f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0xff,0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0xff,
0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0xff,0x3f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0x07,0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0x07,
0x3f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0xe0,0xff,0x3f,0x00,0x00,0x00,0xe0,0x07,0x3f,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xf8,0xff,0xff,0xe0,
0xff,0xff,0xe1,0xff,0xff,0xff,0x7f,0xc0,0xc7,0x1f,0xf8,0xff,
0x3f,0xc0,0xff,0x7f,0x80,0xff,0xff,0x3f,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0xfc,0xff,0xff,0xe1,0xff,0xff,0xe7,0xff,
0xff,0xff,0x7f,0xc0,0xc7,0x7f,0xfc,0xff,0xff,0xf0,0xff,0xff,
0xc1,0xff,0xff,0x3f,0x00,0xf8,0xff,0xff,0xc0,0xff,0xff,0x81,
0xfc,0xff,0xff,0xf1,0xff,0xff,0xe7,0xff,0xff,0xff,0x7f,0xc0,
0xc7,0x7f,0xfc,0xff,0xff,0xf0,0xff,0xff,0xc3,0xff,0xff,0x3f,
0x00,0xfc,0xff,0xff,0xe0,0xff,0xff,0x83,0xfc,0xff,0xff,0xf3,
0xff,0xff,0xef,0xff,0xff,0xff,0x7f,0xc0,0xc7,0xff,0xfc,0xff,
0xff,0xf8,0xff,0xff,0xe7,0xff,0xff,0x3f,0x00,0xfc,0xff,0xff,
0xf1,0xff,0xff,0x87,0xfc,0xff,0xff,0xf3,0xff,0xff,0xef,0xff,
0xff,0xff,0x7f,0xc0,0xc7,0xff,0xfe,0xff,0xff,0xf9,0xff,0xff,
0xe7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xf1,0xff,0xff,0x8f,
0xfc,0xff,0xff,0xfb,0xff,0xff,0xef,0xff,0xff,0xff,0x7f,0xc0,
0xc7,0xff,0xfe,0xff,0xff,0xf9,0xff,0xff,0xf7,0xff,0xff,0x3f,
0x00,0xfe,0xff,0xff,0xf9,0xff,0xff,0x8f,0xfc,0x00,0xe0,0xfb,
0xff,0xff,0xef,0xff,0xff,0xff,0x7e,0xc0,0xc7,0xff,0xfe,0x00,
0xf8,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,
0xfb,0xff,0xff,0x8f,0xfc,0x00,0xe0,0xfb,0x01,0xc0,0xef,0x07,
0x7f,0x00,0x7e,0xc0,0x07,0xfc,0x7e,0x00,0xf8,0xfd,0x01,0xe0,
0xf7,0x07,0x00,0x00,0x00,0xfe,0x00,0xf8,0xfb,0xff,0xff,0x8f,
0xfc,0xff,0x7f,0xf8,0x01,0xc0,0xef,0x03,0x3f,0x00,0x7e,0xc0,
0x07,0xf8,0x7e,0x00,0xf0,0xfd,0x00,0xe0,0xf7,0x03,0x00,0x00,
0x00,0x7e,0x00,0xf0,0xfb,0x01,0xc0,0x8f,0xfc,0xff,0xff,0xf8,
0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,
0xff,0xfd,0xff,0xff,0xf7,0x03,0x00,0x00,0x00,0x7e,0x00,0xf0,
0xfb,0x01,0x00,0x80,0xfc,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,
0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,
0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xfb,0x81,0xff,0x8f,
0xfc,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,
0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,
0x00,0xfe,0xff,0xff,0xfb,0xc1,0xff,0x8f,0xf8,0xff,0xff,0xfb,
0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,
0xff,0xfd,0xff,0xff,0xf7,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,
0xfb,0xc1,0xff,0x8f,0xf0,0xff,0xff,0xfb,0x01,0x80,0xef,0x03,
0x3e,0x00,0x7e,0xc0,0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,
0xf3,0xff,0xff,0x3f,0x00,0xfe,0xff,0xff,0xfb,0xe1,0xff,0x8f,
0x00,0x00,0xf8,0xfb,0x01,0x80,0xef,0x03,0x3e,0x00,0x7e,0xc0,
0x07,0xf8,0xfe,0xff,0xff,0xfd,0xff,0xff,0xf1,0xff,0xff,0x1f,
0x00,0xfe,0xff,0xff,0xfb,0xc1,0xff,0x8f,0xfc,0x00,0xf0,0xfb,
0x01,0x80,0xef,0x03,0x3e,0xf0,0x7f,0xc0,0x07,0xf8,0x7e,0x00,
0xf8,0xfd,0x01,0xf8,0xf1,0x03,0x00,0x00,0x00,0xfe,0xff,0xff,
0xfb,0x81,0xff,0x8f,0xfc,0x00,0xf0,0xfb,0x01,0xc0,0xef,0x03,
0x3e,0xf0,0x7f,0xe0,0x07,0xfc,0x7e,0x00,0xf0,0xfd,0x00,0xf8,
0xf1,0x03,0x00,0x00,0x00,0x7e,0x00,0xf8,0xfb,0x01,0xc0,0x8f,
0xfc,0xff,0xff,0xfb,0xff,0xff,0xef,0x03,0x7e,0xf8,0xff,0xf0,
0x0f,0xfe,0x7e,0x00,0xf0,0xfd,0x00,0xf8,0xf1,0x03,0x00,0x00,
0x00,0x7e,0x00,0xf0,0xfb,0x01,0xc0,0x8f,0xfc,0xff,0xff,0xf3,
0xff,0xff,0xef,0x03,0xfe,0xff,0xff,0xff,0xff,0xff,0x7e,0x00,
0xf0,0xfd,0x00,0xf0,0xf3,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xfb,0x03,0xe0,0x8f,0xfc,0xff,0xff,0xf3,0xff,0xff,0xef,0x03,
0xfe,0xff,0xff,0xff,0xff,0xff,0x7e,0x00,0xf0,0xfd,0x00,0xf0,
0xf3,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,0xfb,0xff,0xff,0x8f,
0xf8,0xff,0xff,0xf3,0xff,0xff,0xef,0x03,0xfe,0xff,0xfd,0xff,
0xff,0xff,0x7e,0x00,0xf0,0xfd,0x00,0xf0,0xe3,0xff,0xff,0x1f,
0x00,0x7e,0x00,0xf0,0xfb,0xff,0xff,0x8f,0xf8,0xff,0xff,0xf1,
0xff,0xff,0xe7,0x03,0xfe,0xff,0xfc,0xff,0xff,0x7f,0x7e,0x00,
0xf0,0xfd,0x00,0xe0,0xe7,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xf3,0xff,0xff,0x8f,0xf0,0xff,0xff,0xe1,0xff,0xff,0xe7,0x03,
0xfc,0xff,0xf8,0xff,0xff,0x7f,0x7e,0x00,0xf0,0xfd,0x00,0xe0,
0xe7,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,0xf3,0xff,0xff,0x87,
0xe0,0xff,0xff,0xc0,0xff,0xff,0xe3,0x03,0xf8,0x7f,0xf8,0xff,
0xfe,0x3f,0x7e,0x00,0xf0,0xfd,0x00,0xe0,0xc7,0xff,0xff,0x1f,
0x00,0x7e,0x00,0xf0,0xe3,0xff,0xff,0x87,0x00,0x00,0x00,0x00,
0x00,0x00,0xe0,0x03,0xe0,0x1f,0xc0,0x3f,0xf8,0x0f,0x3e,0x00,
0xf0,0xf9,0x00,0xc0,0x87,0xff,0xff,0x1f,0x00,0x7e,0x00,0xf0,
0xc3,0xff,0xff,0x83,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x7e,0x00,0xf0,0x83,0xff,0xff,0x81,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80};
}

set bitmap(sag-logo) {
#define sag-logo_width 37
#define sag-logo_height 61
static unsigned char sag-logo_bits[] = {
0x00,0x00,0x00,0x00,0xe0,0x00,0x00,0x00,0x00,0xe0,0x00,0x00,
0x00,0x00,0xe0,0x00,0x00,0x00,0x00,0xe0,0x00,0x80,0xff,0xff,
0xe3,0x00,0xf0,0xff,0xff,0xe3,0x00,0xf8,0xff,0xff,0xe3,0x00,
0xfc,0xff,0xff,0xe3,0x00,0xfe,0xff,0xff,0xe3,0x00,0xff,0xff,
0xff,0xe3,0x80,0xff,0xff,0xff,0xe3,0xc0,0xff,0xff,0xff,0xe3,
0xc0,0xff,0xff,0xff,0xe3,0xe0,0xff,0xff,0xff,0xe3,0xe0,0xff,
0xff,0xff,0xe3,0xf0,0xff,0xff,0xff,0xe3,0xf0,0xff,0xff,0xff,
0xe3,0xf0,0xff,0xff,0xff,0xe3,0xf8,0xff,0x07,0x00,0xe0,0xf8,
0xff,0x03,0x00,0xe0,0xf8,0xff,0x01,0x00,0xe0,0xf8,0xff,0x00,
0x00,0xe0,0xf8,0xff,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,
0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,
0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,
0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,
0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,
0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,
0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,0xe0,0xff,0xe3,0xfc,0x7f,
0xe0,0xff,0xe3,0xfc,0x7f,0xf0,0xff,0xe3,0x00,0x00,0xf0,0xff,
0xe1,0x00,0x00,0xf8,0xff,0xe1,0x00,0x00,0xfe,0xff,0xe1,0xfc,
0xff,0xff,0xff,0xe1,0xfc,0xff,0xff,0xff,0xe1,0xfc,0xff,0xff,
0xff,0xe1,0xfc,0xff,0xff,0xff,0xe0,0xfc,0xff,0xff,0xff,0xe0,
0xfc,0xff,0xff,0xff,0xe0,0xfc,0xff,0xff,0x7f,0xe0,0xfc,0xff,
0xff,0x7f,0xe0,0xfc,0xff,0xff,0x3f,0xe0,0xfc,0xff,0xff,0x3f,
0xe0,0xfc,0xff,0xff,0x1f,0xe0,0xfc,0xff,0xff,0x0f,0xe0,0xfc,
0xff,0xff,0x03,0xe0,0xfc,0xff,0xff,0x00,0xe0,0x00,0x00,0x00,
0x00,0xe0,0x00,0x00,0x00,0x00,0xe0,0x00,0x00,0x00,0x00,0xe0,
0x00,0x00,0x00,0x00,0xe0};
}

# ---------------------------------------------------------------------
# Mainprogram
# ---------------------------------------------------------------------
wm withdraw .
loadAdabastcl
checkUserPw
createMain
