# ---------------------------------------------------------------------
# Copyright (c) 1997 Christian Krone. All rights reserved.
# This program is free software; you can redistribute it and/or
# modify it under the same terms as Tcl itself.
# See also licence.terms
# $Id: tree.tcl,v 1.8 1997/07/03 22:03:19 adabas Exp $
# ----------------------------------------------------------------------------

proc tree {cmd name args} {
    global _tree_ tkCompat

    # First  check, if the given widget already exists.
    switch $cmd {
	create {
	    if [info exists _tree_($name,list)] {
		return -code error "tree widget $name already exists"
	    }
	}
	default {
	    if ![info exists _tree_($name,list)] {
		return -code error "tree widget $name doesn't exists"
	    }
	}
    }

    # Then parse the options.
    if [catch "array set arg [list $args]"] {
	puts "tree $cmd $name $args"
	return -code error "incorrect option/value pairs"
    }

    switch $cmd {
	create {
	    #
	    # First of all unset all old array elements.
	    #

	    foreach oldElem [array names _tree_ $name*] {
		unset _tree_($oldElem)
	    }

	    #
	    # Create a new tree widget...
	    #

	    set _tree_($name,rowHeight) 16
	    frame $name -relief sunken -borderwidth 2
	    canvas $name.c -yscrollcommand "$name.s set" \
		    -yscrollincrement $_tree_($name,rowHeight)
	    scrollbar $name.s -command "$name.c yview"
	    bind $name <Destroy> "tree destroy $name"
	    bind $name <Prior>   "$name.c yview scroll -1 pages; break"
	    bind $name <Next>    "$name.c yview scroll  1 pages; break"
	    bind $name <Up>      "$name.c yview scroll -1 unit;  break"
	    bind $name <Down>    "$name.c yview scroll  1 unit;  break"
	    bind $name <Home>    "$name.c yview moveto  0;       break"
	    bind $name <End>     "$name.c yview moveto  1;       break"

	    grid $name.c -row 0 -column 0 -sticky nsew
	    grid $name.s -row 0 -column 1 -sticky ns
	    grid rowconfigure    $name 0 -weight $tkCompat(gridWeightMax)
	    grid columnconfigure $name 0 -weight $tkCompat(gridWeightMax)

	    if ![info exists _tree_(bitmap,plus)] {
		_tree_bitmaps
	    }

	    set _tree_($name,list) {}

	    return $name
	}

	addtype {
	    if ![info exists arg(-type)] {
		return -code error "-type option must be given"
	    }
	    set _tree_($name,type,$arg(-type)) exist
	    if [info exists arg(-image)] {
		set _tree_($name,image,$arg(-type)) $arg(-image)
	    }
	    if [info exists arg(-opencommand)] {
		set _tree_($name,opencommand,$arg(-type)) $arg(-opencommand)
	    }
	    if [info exists arg(-command)] {
		set _tree_($name,command,$arg(-type)) $arg(-command)
	    }
	}

	listtype {
	    foreach entry [array names _tree_ $name,type,*] {
		regexp ^$name,type,(.*)$ $entry match type
		set row $type
		foreach attr {image opencommand command} {
		    if [info exists _tree_($name,$attr,$type)] {
			lappend row $_tree_($name,$attr,$type)
		    } else {
			lappend row {}
		    }
		}
		lappend res $row
	    }
	    return $res
	}

	add {
	    #
	    # Add a line to the tree widget at the top of its hierarchy...
	    #

	    if [info exists arg(-type)] {
		if ![info exists _tree_($name,type,$arg(-type))] {
		    return -code error "unknown type $arg(*-type)"
		}
		foreach attr {image opencommand command} {
		    if {[info exists _tree_($name,$attr,$arg(-type))] \
			    && ![info exists arg(-$attr)]} {
			set arg(-$attr) $_tree_($name,$attr,$arg(-type))
		    }
		}
	    }

	    if ![info exists arg(-text)] {
		return -code error "-text option must be specified"
	    }
	    if {[lsearch -exact $_tree_($name,list) $arg(-text)] >= 0} {
		return -code error "Node $arg(-text) already exists"
	    }
	    if ![info exists arg(-image)] {
		return -code error "Either -image or -type must be specified"
	    }
	    if [info exists arg(-father)] {
		return -code error "add command not allowed with -father option"
	    }
	    set fIndex [llength $_tree_($name,list)]
	    set _tree_($name,list) \
		    [linsert $_tree_($name,list) 0 $arg(-text)]

	    if ![info exists arg(-opencommand)] {
		set arg(-opencommand) ""
	    }
	    if ![info exists arg(-command)] {
		set arg(-command) ""
	    }
	    if [catch {_tree_add $name $fIndex 0 0 1 \
		    $arg(-image) $arg(-text) $arg(-text) "" \
		    $arg(-opencommand) $arg(-command)} msg] {
		return -code error $msg
	    }

	    if [info exists arg(-open)] {
		set openCount [llength $arg(-open)]
		for {set i 0} {$i < $openCount} {incr i} {
		    if [catch {tree open $name \
			    -father [lrange $arg(-open) 0 $i]}] {
			if {$i < $openCount-1} {
			    return -code error "Can't open $arg(-open)"
			}
		    }
		}
		set revPos  [lsearch -exact $_tree_($name,list) $arg(-open)]
		set lLength [llength $_tree_($name,list)]
		$name.c yview moveto [expr {1 - (double($revPos)+1)/$lLength}]
	    }
	}

	open {
	    #
	    # Add one (or more) lines to the widget relative to a father...
	    #

	    if ![info exists arg(-father)] {
		return -code error "open command must have a -father option"
	    }
	    set fIndex [lsearch -glob $_tree_($name,list) $arg(-father)]
	    if {$fIndex < 0} {
		return -code error "Node to open ($arg(-father)) doesn't exist"
	    }

	    $name.c itemconfigure $arg(-father):+ \
		    -image $_tree_(bitmap,minus)
	    $name.c bind $arg(-father) <1> \
		    "tree close $name -father [list $arg(-father)];break"

	    set depth [expr {$_tree_($name,depth,$arg(-father))+1}]
	    if {$fIndex > 0} {
		set next [lindex $_tree_($name,list) [expr {$fIndex-1}]]
		set nextDepth $_tree_($name,depth,$next)
	    } else {
		set nextDepth 0
	    }

	    if [catch {set _tree_($name,$arg(-father),opencommand)} callback] {
		return -code error "father $arg(-father) could not be opened"
	    }
	    set newRows  [eval $callback $name [list $arg(-father)]]
	    set rowCnt   [llength $newRows]

	    if $rowCnt {

		set moveDistance [expr {$rowCnt*$_tree_($name,rowHeight)}]
		for {set i 0} {$i < $fIndex} {incr i} {
		    $name.c dtag all moveIt
		    $name.c addtag moveIt withtag \
			    [lindex $_tree_($name,list) $i]
		    $name.c move moveIt 0 $moveDistance
		}

		set curr 0
		foreach row $newRows {
		    switch [llength $row] {
			4 {
			    set image       [lindex $row 0]
			    set text        [lindex $row 1]
			    set opencommand [lindex $row 2]
			    set command     [lindex $row 3]
			}
			2 {
			    set type [lindex $row 0]
			    set text [lindex $row 1]
			    if ![info exists _tree_($name,type,$type)] {
				return -code error "Unknown type $type"
			    }
			    foreach attr {image opencommand command} {
				if [info exists _tree_($name,$attr,$type)] {
				    set $attr $_tree_($name,$attr,$type)
				} else {
				    set $attr ""
				}
			    }
			}
			default {
			    return -code error "Invalid row (must have 2 or 4 elems): $row"
			}
		    }
		    if {![string length $image] || ![string length $text]} {
			return -code error "Both (image and text) must be specified"
		    }

		    set     path $arg(-father)
		    lappend path $text
		    set row   [expr {[llength $_tree_($name,list)]-$fIndex}]
		    set _tree_($name,list) \
			    [linsert $_tree_($name,list) $fIndex $path]

		    set isLast    [expr [incr curr]==$rowCnt]
		    if $depth {
			set vertMask $_tree_($name,vertMask,$arg(-father))[expr !$isLast]
		    } else {
			set vertMask ""
		    }
		    if [catch {_tree_add $name $row $depth $nextDepth $isLast \
			    $image $text $path $vertMask \
			    $opencommand $command} msg] {
			return -code error $msg
		    }
		}
	    }
	}

	close {
	    #
	    # Delete all lines belonging to a father from the tree widget...
	    #

	    if ![info exists arg(-father)] {
		return -code error "close command must have a -father option"
	    }
	    set fIndex [lsearch -exact $_tree_($name,list) $arg(-father)]
	    if {$fIndex < 0} {
		return -code error "Unknown -father $arg(-father)"
	    }

	    set fIndex [lsearch -glob $_tree_($name,list) "$arg(-father) *"]
	    set count 0
	    while {[string match "$arg(-father) *" \
		    [lindex $_tree_($name,list) [expr {$fIndex+$count}]]]} {
		incr count
	    }
	    if [catch {_tree_del $name $fIndex $count} del] {
		return -code error $msg
	    }

	    $name.c itemconfigure $arg(-father):+ -image $_tree_(bitmap,plus)
	    $name.c bind $arg(-father) <1> \
		    "tree open $name -father [list $arg(-father)]"
	    set lLength [llength $_tree_($name,list)]
	    $name.c configure -scrollregion "0 0 [$name.c cget -width] \
		    [expr $lLength*$_tree_($name,rowHeight)]"
	}

	delete {
	    #
	    # Delete a line from the tree widget at the top of its hierarchy...
	    #

	    if [info exists arg(-path)] {
		set arg(-index) [lsearch -exact $_tree_($name,list) $arg(-path)]
		if {$arg(-index) < 0} {
		    return -code error "unknow path $arg(-path)"
		}
	    }
	    if ![info exists arg(-index)] {
		return -code error "delete command needs -path or -index option"
	    }
	    if [catch {_tree_del $name $arg(-index) 1} msg] {
		return -code error $msg
	    }
	}

	default {
	    return -code error "Unknown command $cmd: must be create, addtype, listtype, add, open or delete"
	}

	destroy {
	    catch {destroy $name}
	    unset _tree_($name,list)
	}
    }
}

proc _tree_add {name row depth nextDepth isLast image text path vertMask \
	opencommand command} {
    global _tree_

    set x [expr {10*$depth}]
    set y [expr {$row*$_tree_($name,rowHeight)+5}]
    $name.c create image [expr $x+13] $y -image $image -anchor w \
	    -tags [list $path $path:x]
    $name.c create text  [expr $x+33] $y -text  $text  -anchor w \
	    -tags [list $path $path:x]

    for {set i 0} {$i < $nextDepth && $i < $depth-1} {incr i} {
	if [string index $vertMask $i] {
	    $name.c create image [expr {10*$i}] $y -image $_tree_(bitmap,vert) \
		    -anchor w -tags [list $path]
	}
    }
    if $isLast {
	$name.c create image [expr {10*($depth-1)}] $y \
		-anchor w -tags [list $path] -image $_tree_(bitmap,corn)
    } else {
	$name.c create image [expr {10*($depth-1)}] $y \
		-anchor w -tags [list $path] -image $_tree_(bitmap,tee)
    }

    $name.c configure -scrollregion "0 0 [$name.c cget -width] \
	    [expr [llength $_tree_($name,list)]*$_tree_($name,rowHeight)]"
    set _tree_($name,depth,$path)    $depth
    set _tree_($name,vertMask,$path) $vertMask

    if [string length $opencommand] {
	$name.c create image $x $y -tags [list $path $path:+] -anchor w \
		-image $_tree_(bitmap,plus)
	set _tree_($name,$path,opencommand) $opencommand
	$name.c bind $path <1> "tree open $name -father [list $path]"
    } else {
	$name.c create image $x $y -tags [list $path] -anchor w \
		-image $_tree_(bitmap,horz)
    }
    if [string length $command] {
	$name.c bind $path:x <1>     "$command $name [list $path];break"
	$name.c bind $path:x <Enter> "$name.c configure -cursor hand2"
	$name.c bind $path:x <Leave> "$name.c configure -cursor {}"
    }
}

proc _tree_del {name index count} {
    global _tree_

    for {set i 0} {$i < $index} {incr i} {
	$name.c dtag all moveIt
	$name.c addtag moveIt withtag \
		[lindex $_tree_($name,list) $i]
	$name.c move moveIt 0 [expr {-$count*$_tree_($name,rowHeight)}]
    }
    for {set i 0} {$i < $count} {incr i} {
	$name.c delete [lindex $_tree_($name,list) [expr {$index+$i}]]
    }
    set _tree_($name,list) \
	    [lreplace $_tree_($name,list) $index [expr {$index+$count-1}]]
}

proc _tree_bitmaps {} {
    global _tree_

    set _tree_(bitmap,plus)  [image create bitmap -data "
#define plus_width 10
#define plus_height 10
static unsigned char plus_bits[] = {
   0xfc, 0x00, 0x02, 0x01, 0x31, 0x02, 0x31, 0x02, 0xfd, 0x02, 0xfd, 0x02,
   0x31, 0x02, 0x31, 0x02, 0x02, 0x01, 0xfc, 0x00};
"]
    set _tree_(bitmap,minus) [image create bitmap -data "
#define minus_width 10
#define minus_height 10
static unsigned char minus_bits[] = {
   0xfc, 0x00, 0x02, 0x01, 0x01, 0x02, 0x01, 0x02, 0xfd, 0x02, 0xfd, 0x02,
   0x01, 0x02, 0x01, 0x02, 0x02, 0x01, 0xfc, 0x00};
"]
    set _tree_(bitmap,vert) [image create bitmap -data "
#define vert_width 10
#define vert_height 19
static unsigned char vert_bits[] = {
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00,
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00,
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00,
   0x00, 0x00};
"]
    set _tree_(bitmap,corn) [image create bitmap -data "
#define corn_width 10
#define corn_height 19
static unsigned char corn_bits[] = {
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00,
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xa0, 0x02, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00};
"]
    set _tree_(bitmap,tee) [image create bitmap -data "
#define tee_width 10
#define tee_height 19
static unsigned char tee_bits[] = {
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00,
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xa0, 0x02, 0x00, 0x00, 0x20, 0x00,
   0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00,
   0x00, 0x00};
"]
    set _tree_(bitmap,horz) [image create bitmap -data "
#define horz_width 10
#define horz_height 3
static unsigned char horz_bits[] = {
   0x00, 0x00, 0xaa, 0x02, 0x00, 0x00};
"]
}

