/*
 * AdabasApi.c --
 *
 *      This module contains procedures, that implement the functionalty
 *      of the adasql procedures (adalogon, adaopen, adasql, adafetch, ...).
 *      Despite the fact, that all procedures return Tcl results, it
 *      could be used by other extensions (perl, python, ...), too.
 *
 * Copyright (c) 1996-1997 Christian Krone. All rights reserved.
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Tcl itself.
 * See also licence.terms
 *
 *  $Id: AdabasApi.c,v 1.25 1997/06/30 21:43:34 adabas Exp $
 */

/*
 * Import interface (should move into AdabasTclInt.h someday)
 */

#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <tcl.h>

#include "adabas.h"
#include "AdabasPort.h"
#include "AdabasRte.h"
#include "AdabasPacket.h"
#include "AdabasUtil.h"
#include "AdabasSend.h"
#include "AdabasApi.h"
#include "AdabasFormat.h"
#include "AdabasLong.h"


/*
 * When inspecting a given SQL command, there is heavily use of regular
 * expressions including parenthesis. All the regexp and the corresponding
 * range numbers are defined here.
 */

#define CONNECT_RE        "^([^,]*),(.*)$"
#define CONNECT_USER_RG   1
#define CONNECT_PASSWD_RG 2

#define COMMAND_RE        "^[ \t\n]*([^ \t\n(]*)"
#define COMMAND_TOKEN_RG  1

#define INTO_RE           " INTO  *(:)"
#define INTO_VAR_RG       1

#define PARAM_RE          ":([a-zA-Z0-9_()]*)"
#define PARAM_NAME_RG     1

#define RESULT_RE         "^[ \t\n]*[a-zA-Z]*[ \t\n]*\\(([^)]*)\\)"
#define RESULT_NAME_RG    1

/*
 * The procedure examineSqlCommand inspects the given SQL command string
 * and returns one of the following enums to give a clue about the
 * kind of statement.
 */

typedef enum {
  noSelect,
  massSelect,
  intoSelect,
  showSelect,
  explainSelect
} SelectKind;

/*
 * Prototypes for procedures referenced only in this file.
 */

static int   selectInto _ANSI_ARGS_((Tcl_Interp *interp, CursorInfo *cursorVars,
		 char *selectInto));
static void  invalidateResultTable _ANSI_ARGS_((CursorInfo *cursorVars));
static int   executeWithParameter _ANSI_ARGS_((Tcl_Interp *interp,
		 CursorInfo  *cursorVars, Tcl_DString *parsId));
static void  examineSqlCommand _ANSI_ARGS_((Tcl_Interp *interp, char *cmdString,
		 int *withInfo, int *withResCount, SelectKind *selectKind,
		 char **selectInto));
static char *getNextData _ANSI_ARGS_((char *data, CursorInfo *cursorVars));
static int   appendOneValue _ANSI_ARGS_((Tcl_Interp *interp, AdabasInfo *vars,
		 CursorInfo *cursorVars, char *data, Tcl_Obj *destObj,
		 char *nullValue, char *specialNull, field_info *currSi,
		 int isElement));
static void  getNullValue _ANSI_ARGS_((Tcl_Interp *interp, char **nullValue,
		 char **specialNull));
static int   checkCursor _ANSI_ARGS_((Tcl_Interp *interp, CursorInfo *cursor));
static char *applKind2String _ANSI_ARGS_((ApplKind kind));
static char *TracefileHandler _ANSI_ARGS_((ClientData dummy,
		 Tcl_Interp *interp, char *name1, char *name2, int flags));

/*
 * This local variable is a hash table full with all created cursor handles.
 * Since we must be able to get this handle from the name of a cursor, it isn't
 * sufficient to have a list of all cursors associated with a logon handle.
 */

static Tcl_HashTable cursorHash;

/*
 * If this local variable is set, every call of any API function is logged
 * in the file. The variable will be set by a trace handler of adamsg(tracefile).
 */

static FILE *protFile = (FILE *) NULL;

/*
 *----------------------------------------------------------------------
 *
 * AdabasApiInit --
 *
 *      Initializes this module.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      All the API functions can be called afterwards.
 *
 *----------------------------------------------------------------------
 */

void
AdabasApiInit (interp)
     Tcl_Interp *interp;		/* Current interpreter. */
{
  char version[200];			/* Buffer for version string. */

  SET_VERSION (version);
  AdabasSetMsgString (interp, AdamsgVersion,     version);
  AdabasSetMsgString (interp, AdamsgNullvalue,   "");
  AdabasSetMsgString (interp, AdamsgSpecialnull, "***");

  /*
   * Initialize the only global variable, we have: the hash table of cursors.
   */

  Tcl_InitHashTable (&cursorHash, TCL_STRING_KEYS);

  Tcl_TraceVar2 (interp, "adamsg", "tracefile",
		 TCL_TRACE_WRITES | TCL_TRACE_UNSETS | TCL_GLOBAL_ONLY,
		 TracefileHandler, (ClientData) NULL);

  AdabasFormatInit (interp);
}

/*
 *----------------------------------------------------------------------
 *
 * getDataOfCursor --
 *
 *      Looks for the data part in the return segment of the given cursor.
 *      Note, that all cursors out of one logon share the same segment.
 *
 * Results:
 *      A pointer to the data part, or NULL, if no data part is found.
 *      If no data part is found and an interpreter is given, an error
 *      message will be stored in its object result.
 *
 * Side effects:
 *      Resets an internal counter (rowIx) of the given cursor, which will
 *      be used by subsequent calls of getNextData.
 *
 *----------------------------------------------------------------------
 */

char *
getDataOfCursor (interp, cursorVars, flags)
     Tcl_Interp *interp;		/* Current interpreter (or NULL). */
     CursorInfo *cursorVars;		/* Current cursor. */
     int         flags;			/* defined in AdabasPacket.h */
{
  char *data;				/* Pointer to data part. */

#if ADABAS_VERSION >= 62
  data = getData (&cursorVars->logonInfo->rcvPacket->segm,
		  &cursorVars->dataLength, &cursorVars->massRows, flags);

#else /* ADABAS_VERSION <= 61 */
  data = getData (&cursorVars->logonInfo->sqlPacket,
		  &cursorVars->dataLength, &cursorVars->massRows, flags);
#endif

  cursorVars->rowIx = 0;

  if (!data) {
    if (interp) {
      SetResult (interp, "Oops, no data part found???");
    }
    return (char *) NULL;
  }
  return data;
}

/* -------------------------------------------------------------- */

CursorInfo *
AdabasOpen (interp, vars)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Current logon. */
{
  CursorInfo    *cursorVars;		/* the newly created cursorHandle. */
  Tcl_HashEntry *newEntry;		/* Entry with cursor name and infoPrt. */
  int            added;			/* Hash entry successfully added? */
  int            cnt;			/* Count above cursor. */
  int            ret;			/* return value. */

  Tcl_ResetResult (interp);
  ret = AdabasCheckLogon (interp, vars);
  if (protFile) {
    fprintf (protFile, "OPEN     (logon=\"%s\")\n",
	     ret == TCL_OK ? vars->logonName : "###invalid###");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return (CursorInfo *) NULL;
  }
    
  if ((cursorVars = (CursorInfo *) ckalloc (sizeof (CursorInfo))) == NULL) {
    AppendResult (interp, "Allocation of cursor failed due to lack of memory");
    return (CursorInfo *) NULL;
  }
  cursorVars->logonInfo    = vars;
  cursorVars->cntParams    = 0;
  cursorVars->longColFound = 0;
  cursorVars->dataLength   = 0;
  cursorVars->dataEnd      = 0;
  cursorVars->shortInfos   = NULL;
  cursorVars->columnNames  = NULL;
  cursorVars->magic        = CURSOR_MAGIC;

  /*
   * Store the new cursorHandle into our hashtable with its name as key
   * Put the name of the cursor into the CursorInfo.
   */

  added = 0;
  cnt   = 0;
  while (!added) {
    sprintf (cursorVars->cursorName, "cursor%d", ++cnt);
    newEntry = Tcl_CreateHashEntry (&cursorHash, cursorVars->cursorName, &added);
  }
  Tcl_SetHashValue (newEntry, cursorVars);

  return cursorVars;
}

/* -------------------------------------------------------------- */

int
AdabasClose (interp, cursorVars)
     Tcl_Interp *interp;		/* Current interpreter. */
     CursorInfo *cursorVars;		/* Cursor to be freed. */
{
  Tcl_HashEntry *oldEntry;		/* hash table entry of cursorName. */
  int            ret;			/* return value. */

  ret = checkCursor (interp, cursorVars);
  if (protFile) {
    fprintf (protFile, "CLOSE    (cursor=\"%s\")\n",
	     ret == TCL_OK ? cursorVars->cursorName : "###invalid###");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (!(oldEntry = Tcl_FindHashEntry (&cursorHash, cursorVars->cursorName))) {
    SetResult (interp, "Oops, cursor not found in hash table???");
    return TCL_ERROR;
  }

  if (cursorVars->columnNames) {
    ckfree ((char *) cursorVars->columnNames);
  }
  if (cursorVars->shortInfos) {
    ckfree ((char *) cursorVars->shortInfos);
  }
  cursorVars->magic = 0;
  ckfree ((char *) cursorVars);

  /*
   * Delete the cursorHandle from our hashtable with its name as key.
   */

  Tcl_DeleteHashEntry (oldEntry);
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasSql --
 *
 *      Sends the given sql command to the database kernel.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      The sql command is executed. After a SELECT the cursor is ready for
 *      subsequent fetches. Some entries of the global adamsg array are
 *      modified.
 *
 *----------------------------------------------------------------------
 */

int
AdabasSql (interp, cursorVars, messType, withParameter,
	   cmd, resultTable, sqlMode)
     Tcl_Interp *interp;		/* Current interpreter. */
     CursorInfo *cursorVars;		/* Current Cursor. */
     int         messType;		/* Message type (DBS, PARSE, EXECUTE) */
     int         withParameter;		/* Execute with parameter? */
     char       *cmd;			/* Sql command to execute. */
     char       *resultTable;		/* Name of resulttable (or NULL). */
     char       *sqlMode;		/* Wanted SQL mode (or NULL). */
{
  int         ret;			/* return value. */
  int         orgMessType;		/* original message type. */
  AdabasInfo *logonHandle;
  int         withInfo;
  int         withResCount;
  char       *selInto;
  int         resCnt;
  int         oldSqlMode  = -1;		/* Original value in segment header. */
  int         errorOffset = 0;
  int         pos;
  SelectKind  selectKind;
  Tcl_DString cmdStr;
  Tcl_DString parsId;			/* ParsId returned, if only parsing. */
#if ADABAS_VERSION >= 62
  segment    *rcvSegment;		/* Segment returned by database. */
#else /* ADABAS_VERSION <= 61 */
  packet     *rcvSegment;		/* Packet returned by database. */
  Tcl_RegExp  regExp;			/* compiled regexp for cmd */
  char       *start;			/* Index into cmd returned by regexp */
  char       *end;			/* Index into cmd returned by regexp */
#endif
  int         describeNeeded;
  Tcl_DString describe;
  char       *p;
  int         sqlModeInt = -1;		/* Scanned value of sqlmode option. */

  Tcl_ResetResult (interp);
  ret = checkCursor (interp, cursorVars);
  if (protFile) {
    fprintf (protFile, "SQL      (cursor=\"%s\",cmd=\"%s\"",
	     ret == TCL_OK ? cursorVars->cursorName : "###invalid###",
	     cmd);
    if (resultTable) {
      fprintf (protFile, ",resultTable=\"%s\"", resultTable);
    }
    if (sqlMode) {
      fprintf (protFile, ",sqlmode=%s", sqlMode);
    }
    fprintf (protFile, ")\n");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  logonHandle = cursorVars->logonInfo;
  AdabasDeleteAllLongDescrOf (logonHandle);

  if (sqlMode && *sqlMode) {
    sqlModeInt = scanSqlMode (interp, sqlMode, 0);
#if ADABAS_VERSION >= 62
    if (sqlModeInt == SM_NIL) return TCL_ERROR;
#else /* ADABAS_VERSION <= 61 */
    if (sqlModeInt < 0) return TCL_ERROR;
#endif
  }

  cursorVars->longColFound = 0;
  cursorVars->dataLength   = 0;

  /*
   * Note, that adabas doesn't like Newlines in the command...
   */

  for (p = cmd; *p; p++) if (*p == '\n') *p = ' ';

  /*
   * Look at the command string and determine the kind of statement
   * (data manipulation, select, select into or the like)
   */

  examineSqlCommand (interp, cmd, &withInfo, &withResCount,
		     &selectKind, &selInto);

  if (selectKind == massSelect && resultTable && *resultTable) {
    Tcl_DStringInit   (&cmdStr);
    Tcl_DStringAppend (&cmdStr,  "DECLARE ",    -1);
    Tcl_DStringAppend (&cmdStr,   resultTable,  -1);
    Tcl_DStringAppend (&cmdStr, " CURSOR FOR ", -1);
    errorOffset = strlen (Tcl_DStringValue (&cmdStr));

    Tcl_DStringAppend (&cmdStr, cmd, -1);
    cmd = Tcl_DStringValue (&cmdStr);
  }

  if (sqlModeInt >= 0) {
    oldSqlMode = logonHandle->packetInfo.sqlMode;
    logonHandle->packetInfo.sqlMode = sqlModeInt;
  }

  orgMessType = logonHandle->packetInfo.messType;
  logonHandle->packetInfo.messType = messType;
  ret = AdabasSendCmdPacket (interp, logonHandle, withInfo, 1, cmd);
  logonHandle->packetInfo.messType = orgMessType;

#if ADABAS_VERSION >= 62
  rcvSegment = &logonHandle->rcvPacket->segm;
#else
  rcvSegment = &logonHandle->sqlPacket;
#endif

  if (ret == TCL_OK && messType == M_PARSE) {
    Tcl_DStringInit (&parsId);
    if (!getParsId (interp, rcvSegment, &parsId)) {
      AppendResult (interp, "No returning parsid???");
      return TCL_ERROR;
    }
    ret = TCL_OK;
  }

  /*
   * Set the resultcount, if wanted (update, delete or insert).
   */

  if (messType == M_PARSE) {
    AdabasUnsetMsg  (interp, AdamsgRows);
  } else {
    if (withResCount) {
      resCnt = getResultCount (rcvSegment);
    } else {
      resCnt = selectKind == intoSelect ? 1 : 0;
    }
    AdabasSetMsgInt (interp, AdamsgRows, resCnt);
  }
  AdabasUnsetMsg  (interp, AdamsgIntoVars);

  if (withInfo && ret == TCL_OK) {
#if ADABAS_VERSION >= 62
    if (!getResultTable (rcvSegment, cursorVars->resultTable)) {
      *cursorVars->resultTable = 0;
    }
#else /* ADABAS_VERSION <= 61 */
    if (!getResultTable (rcvSegment, messType, cursorVars->resultTable)) {
      if (selectKind == showSelect || selectKind == explainSelect) {
	regExp = Tcl_RegExpCompile (interp, RESULT_RE);
	if (Tcl_RegExpExec (interp, regExp, cmd, cmd)) {
	  Tcl_RegExpRange (regExp, RESULT_NAME_RG, &start, &end);
	  if (end-start > 40) {
	    end = start+40;
	  }
	  p2c (cursorVars->resultTable, start, end-start);
	  for (start = cursorVars->resultTable; *start; start++) {
	    *start = toupper (*start);
	  }
	} else {
	  strcpy (cursorVars->resultTable, "SHOW");
	}
      } else if (selectKind == massSelect && resultTable && *resultTable) {
	strncpy (cursorVars->resultTable, resultTable,
		 sizeof (cursorVars->resultTable));
      } else {
	cursorVars->resultTable[0] = 0;
      }
    }
#endif
  } else {
    cursorVars->cntParams = 0;
  }

#if ADABAS_VERSION >= 62
  describeNeeded = withInfo &&  messType == M_PARSE;
#else
  describeNeeded = withInfo && (messType == M_PARSE || sqlModeInt >= 0);
#endif
  if (describeNeeded && ret == TCL_OK) {
    Tcl_DStringInit (&describe);
    Tcl_DStringAppend (&describe, "DESCRIBE ", -1);
    Tcl_DStringAppend (&describe, cursorVars->resultTable, -1);
    ret = AdabasSendCmdPacket (interp, logonHandle, 0, 1,
			       Tcl_DStringValue (&describe));
    Tcl_DStringFree (&describe);

#if ADABAS_VERSION >= 62
    rcvSegment = &logonHandle->rcvPacket->segm;
#endif
  }

  if (withInfo && ret == TCL_OK) {
    cursorVars->cntParams = getShortInfos (rcvSegment,
					   &cursorVars->shortInfos,
					   &cursorVars->columnNames);
  }

  /*
   * First clean up the resources before leaving the procedure.
   */

  if (selectKind == massSelect && resultTable && *resultTable) {
    Tcl_DStringFree (&cmdStr);
  }
  if (sqlModeInt >= 0) {
    logonHandle->packetInfo.sqlMode = oldSqlMode;
  }

  /*
   * If we modified the sql command (put a DECLARE ... CURSOR FOR
   * in front of it), we have to adjust the error position.
   */

  if (ret == TCL_ERROR) {
    if (errorOffset
	&& AdabasGetMsgInt (interp, AdamsgErrorpos, &pos) == TCL_OK
	&& pos > errorOffset) {
      AdabasSetMsgInt (interp, AdamsgErrorpos, pos - errorOffset);
    }
    return TCL_ERROR;
  }

  if (messType == M_PARSE) {
    
    if (!withParameter) {
      if (ret == TCL_OK) {
	AppendResult (interp, Tcl_DStringValue (&parsId));
	Tcl_DStringFree (&parsId);
      }
      return ret;
    }

    ret = executeWithParameter (interp, cursorVars, &parsId);
    Tcl_DStringFree (&parsId);
    return ret;
  }

  if (selectKind == intoSelect) {
    return selectInto (interp, cursorVars, selInto);
  }

  if (selectKind == massSelect) {
    invalidateResultTable (cursorVars);
  }

#if ADABAS_VERSION <= 61
  if (sqlModeInt >= 0 && logonHandle->packetInfo.commitImmediately) {
    return AdabasSendCmdPacket (interp, logonHandle, 0, 1, "COMMIT WORK");
  }
#endif
  
  return TCL_OK;
}

/* -------------------------------------------------------------- */

static int
selectInto (interp, cursorVars, selectInto)
     Tcl_Interp *interp;		/* Current interpreter. */
     CursorInfo *cursorVars;		/* Current cursor. */
     char       *selectInto;		/* Pointing to first parameter. */
{
  Tcl_RegExp  regExp;			/* compiled regexp for selectInto */
  char       *start;			/* Index into selectInto ret. by regexp */
  char       *end;			/* Index into selectInto ret. by regexp */
  char       *data;			/* Pointer to data of current row. */
  char       *nullValue;		/* Representation of a NULL value. */
  char       *specialNull;		/* Representation of a SPECIAL NULL. */
  char       *colon  = selectInto;	/* Start of current parameter in cmd. */
  int         paramI = 0;		/* Index of current parameter. */
  int         ret    = TCL_OK;		/* return value. */
  field_info *currSi;			/* Shortinfo of current parameter. */
#ifdef HAS_TCL_OBJECTS
  Tcl_Obj    *varObj;			/* Object for INTO variable name. */
  Tcl_Obj    *resObj;			/* Result to store into INTO variable. */
#else
  Tcl_DString colName;			/* Columnname of INTO variable. */
#endif

  if (!(data = getDataOfCursor (interp, cursorVars, 0))) {
    return TCL_ERROR;
  }

  regExp = Tcl_RegExpCompile (interp, PARAM_RE);

  currSi = cursorVars->shortInfos;
  getNullValue (interp, &nullValue, &specialNull);

  while (colon && Tcl_RegExpExec (interp, regExp, colon, selectInto)) {
    Tcl_RegExpRange (regExp, PARAM_NAME_RG, &start, &end);

#ifdef HAS_TCL_OBJECTS

    varObj = Tcl_NewStringObj (start, end-start);
    resObj = Tcl_NewObj ();
    ret    = appendOneValue (interp, cursorVars->logonInfo, cursorVars, data,
			     resObj, nullValue, specialNull, currSi, 0);
    if (ret == TCL_OK) {
      Tcl_ObjSetVar2 (interp, varObj, (Tcl_Obj *) NULL, resObj,
		      TCL_PARSE_PART1);
      AdabasLAppendMsgString (interp, AdamsgIntoVars,
			      Tcl_GetStringFromObj (varObj, (int *) NULL));
    }

#else

    Tcl_DStringInit   (&colName);
    Tcl_DStringAppend (&colName, start, end-start);
    ret = appendOneValue (interp, cursorVars->logonInfo, cursorVars, data,
			  (Tcl_Obj *) NULL, nullValue, specialNull, currSi, 0);
    if (ret == TCL_OK) {
      Tcl_SetVar (interp, Tcl_DStringValue (&colName), interp->result, 0);
      Tcl_ResetResult (interp);
      AdabasLAppendMsgString (interp, AdamsgIntoVars,
			      Tcl_DStringValue (&colName));
    }
    Tcl_DStringFree  (&colName);

#endif

    if (ret == TCL_ERROR) {
      return TCL_ERROR;
    }

    /*
     * Scan the command for the next into parameter. If found, store the position
     * of the ':' into colon and set currSi to the corresponding shortinfo.
     */

    colon = (char *) NULL;
    if (++paramI < cursorVars->cntParams) {
      while (*end == ' ') end++;
      if (*end == ',') {
	end++; 
	while (*end == ' ') end++;
	if (*end == ':') {
	  colon = end;
	  currSi++;
	}
      }
    }
  }

  return TCL_OK;
}

/* -------------------------------------------------------------- */

static void
invalidateResultTable (cursorVars)
     CursorInfo *cursorVars;		/* Current cursor. */
{
  Tcl_HashEntry *hashEntry;
  Tcl_HashSearch hashSearch;
  CursorInfo    *cmpCursor;

  hashEntry = Tcl_FirstHashEntry (&cursorHash, &hashSearch);
  while (hashEntry) {
    cmpCursor = (CursorInfo *) Tcl_GetHashValue (hashEntry);

    /*
     * If we have another cursor for the same logon with the same
     * resulttablename, its cursor infos will be invalidated for future
     * fetches, since this information is now obsolete.
     */

    if (cursorVars != cmpCursor
	&& cursorVars->logonInfo == cmpCursor->logonInfo
	&& !strcmp (cursorVars->resultTable, cmpCursor->resultTable))
      cmpCursor->cntParams = 0;

    hashEntry = Tcl_NextHashEntry (&hashSearch);
  }
}

/* -------------------------------------------------------------- */

int
AdabasExecParsId (interp, cursorVars, parsId)
     Tcl_Interp *interp;		/* Current interpreter. */
     CursorInfo *cursorVars;		/* Current cursor. */
     char       *parsId;		/* ParsId to execute. */
{
  int ret;				/* return value. */
  int orgMessType;			/* original message type. */

  Tcl_ResetResult (interp);
  if (checkCursor (interp, cursorVars) == TCL_ERROR) {
    return TCL_ERROR;
  }

  orgMessType = cursorVars->logonInfo->packetInfo.messType;
  cursorVars->logonInfo->packetInfo.messType = M_EXECUTE;

  ret = AdabasSend2PartPacket (interp, cursorVars->logonInfo,
			       pkParsid, parsId, strlen (parsId),
			       pkNil, "", 0);

  cursorVars->logonInfo->packetInfo.messType = orgMessType;

  return ret;
}

/* -------------------------------------------------------------- */

static int
executeWithParameter (interp, cursorVars, parsId)
     Tcl_Interp  *interp;		/* Current interpreter. */
     CursorInfo  *cursorVars;		/* Current cursor. */
     Tcl_DString *parsId;		/* ParsId to execute. */
{
  int ret;				/* return value. */
  int orgMessType;			/* original message type. */

  orgMessType = cursorVars->logonInfo->packetInfo.messType;
  cursorVars->logonInfo->packetInfo.messType = M_EXECUTE;

  ret = AdabasSend2PartPacket (interp, cursorVars->logonInfo,
			       pkParsid, Tcl_DStringValue (parsId),
			       Tcl_DStringLength (parsId),
			       pkNil, "", 0);

  cursorVars->logonInfo->packetInfo.messType = orgMessType;
  if (ret == TCL_ERROR) return TCL_ERROR;

  return AdabasSend2PartPacket (interp, cursorVars->logonInfo,
				pkCommand, "DROP PARSEID", 0,
				pkParsid, Tcl_DStringValue (parsId),
				Tcl_DStringLength (parsId));
}

/* -------------------------------------------------------------- */

int
AdabasDropParsId (interp, cursorVars, parsId)
     Tcl_Interp *interp;		/* Current interpreter. */
     CursorInfo *cursorVars;		/* Current cursor. */
     char       *parsId;		/* ParsId to drop. */
{
  Tcl_ResetResult (interp);
  if (checkCursor (interp, cursorVars) == TCL_ERROR) {
    return TCL_ERROR;
  }

  return AdabasSend2PartPacket (interp, cursorVars->logonInfo,
				pkCommand, "DROP PARSEID", 0,
				pkParsid, parsId, strlen (parsId));
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasCancel --
 *
 *      Tells the database to cancel the currently running sql command.
 *      Also the current result table of the cursor will be invalidated.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursor
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      You can't fetch on the cursor after cancelling it.
 *
 *----------------------------------------------------------------------
 */

int
AdabasCancel (interp, cursorVars)
     Tcl_Interp *interp;		/* Current interpreter. */
     CursorInfo *cursorVars;		/* Current cursor. */
{
  int ret;				/* return value. */

  Tcl_ResetResult (interp);
  ret = checkCursor (interp, cursorVars);
  if (protFile) {
    fprintf (protFile, "CANCEL   (cursor=\"%s\")\n",
	     ret == TCL_OK ? cursorVars->cursorName : "###invalid###");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  AdabasDeleteAllLongDescrOf (cursorVars->logonInfo);

  cursorVars->cntParams    = 0;
  cursorVars->longColFound = 0;
  AdabasRteCancel (&cursorVars->logonInfo->rteInfo);
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasFetch --
 *
 *      The next line(s) are fetched from the current result table
 *      and returned as Tcl list (of lists).
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursor
 *      given or the database kernel reports any error while converting the
 *      data, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasFetch (interp, cursorVars, target, count, sqlMode, arraySet)
     Tcl_Interp *interp;		/* Current interpreter. */
     CursorInfo *cursorVars;		/* Current cursor. */
     char       *target;		/* Target to fetch (e.g. "first"). */
     int         count;			/* Count for mass fetches. */
     char       *sqlMode;		/* Sqlmode (or NULL). */
     int         arraySet;		/* Return results as column/value pairs? */
{
  char        cmdBuf[100];		/* Buffer for the FETCH command text. */
  char        posBuf[20];		/* Buffer for POS (n) traget. */
  int         pos;			/* Numerical value of target. */
  int         rc;			/* SQL returncode of fetch command. */
  char       *data;			/* Pointer to data of current row. */
  char       *nullValue;		/* Representation of a NULL value. */
  char       *specialNull;		/* Representation of a SPECIAL NULL. */
  int         ret;			/* return value. */
  int         oldSqlMode  = -1;		/* Original value in segment header. */
  int         ix;			/* Count on fetch parameters. */
  field_info *currSi;			/* Shortinfo of current parameter. */
  char       *currCol    = NULL;	/* Pointer to name of current parameter. */
  int         sqlModeInt = -1;		/* Scanned value of sqlmode option. */
  int         length;			/* Length (used for option parsing). */
  int         c;			/* Char   (used for option parsing). */
  Tcl_Obj    *resObj = NULL;		/* Object to put result into. */
#ifdef HAS_TCL_OBJECTS
  Tcl_Obj    *resList;			/* List object to put one row into. */
#else
  char        currName[IDENTIFIER+1];	/* Buffer for null terminated name. */
#endif

  Tcl_ResetResult (interp);
  ret = checkCursor (interp, cursorVars);
  if (protFile) {
    fprintf (protFile, "FETCH    (cursor=\"%s\"",
	     ret == TCL_OK ? cursorVars->cursorName : "###invalid###");
    if (*target) {
      fprintf (protFile, ",target=%s", target);
    }
    if (count > 1) {
      fprintf (protFile, ",count=%d", count);
    }
    if (sqlMode) {
      fprintf (protFile, ",sqlmode=%s", sqlMode);
    }
    if (arraySet) {
      fprintf (protFile, ",arraySet");
    }
    fprintf (protFile, ")\n");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  if (*target) {
    c      = target[0];
    length = strlen (target);
    if (c == 'f' && !strncmp (target, "first", length))
      target = "FIRST";
    else if (c == 'l' && !strncmp (target, "last",  length))
      target = "LAST";
    else if (c == 'n' && !strncmp (target, "next",  length))
      target = "NEXT";
    else if (c == 'p' && !strncmp (target, "prev",  length))
      target = "PREV";
    else if (Tcl_GetInt ((Tcl_Interp *) NULL, target, &pos) == TCL_OK) {
      sprintf (posBuf, "POS (%d)", pos);
      target = posBuf;
    } else {
      AppendResult (interp, "bad value for switch -position \"");
      AppendResult (interp, target);
      AppendResult (interp, "\": must be -first, -last, -prev, -next or a ");
      AppendResult (interp, "number");
      return TCL_ERROR;
    }
  }

  AdabasDeleteLongDescr (cursorVars->logonInfo, cursorVars);

  if (sqlMode && *sqlMode) {
    sqlModeInt = scanSqlMode (interp, sqlMode, 0);
#if ADABAS_VERSION >= 62
    if (sqlModeInt == SM_NIL) return TCL_ERROR;
#else /* ADABAS_VERSION <= 61 */
    if (sqlModeInt < 0) return TCL_ERROR;
#endif
  }

  if (sqlModeInt >= 0) {
    oldSqlMode = cursorVars->logonInfo->packetInfo.sqlMode;
    cursorVars->logonInfo->packetInfo.sqlMode = sqlModeInt;
  }

  /*
   * Send the fetch command to the database kernel.
   */

  sprintf (cmdBuf, "FETCH %s %s INTO ?", target, cursorVars->resultTable);
  ret = AdabasSendCmdPacket (interp, cursorVars->logonInfo,
			     0, count, cmdBuf);
  if (sqlModeInt >= 0) {
    cursorVars->logonInfo->packetInfo.sqlMode = oldSqlMode;
  }

  if (ret == TCL_ERROR) {

    /*
     * Check, if a positional fetch was called; then a better error
     * message then MISSING KEYWORD or UNKNOWN RESULT TABLE:LAST
     * should be given.
     */

    if (*target
	&& AdabasGetMsgInt (interp, AdamsgRc, &rc) == TCL_OK
	&& (rc == -4000 || rc == -3008 || rc == -5015)) {
      SetResult (interp, "Positional fetch only valid in sqlmode adabas");
      return TCL_ERROR;
    }
    if (AdabasGetMsgInt (interp, AdamsgRc, &rc) == TCL_OK
	&& rc != 100) {
      return TCL_ERROR;
    }
    Tcl_ResetResult (interp);
    return TCL_OK;
  }

  if (!(data = getDataOfCursor (interp, cursorVars,
				count > 1 ? MASS_CMD : 0))) {
    return TCL_ERROR;
  }


#ifdef HAS_TCL_OBJECTS
  resList = Tcl_NewListObj (0, (Tcl_Obj **) NULL);
  Tcl_SetObjResult (interp, resList);
#endif

  getNullValue (interp, &nullValue, &specialNull);
  while (data) {

    if (count > 1) {
#ifdef HAS_TCL_OBJECTS
      resList = Tcl_NewListObj (0, (Tcl_Obj **) NULL);
      Tcl_ListObjAppendElement (interp, Tcl_GetObjResult (interp), resList);
#else
      AppendResult (interp, "{");
#endif
    }

    currSi  = cursorVars->shortInfos;
    if (arraySet) {
      currCol = cursorVars->columnNames;
    }
    for (ix = 0; ix < cursorVars->cntParams && ret == TCL_OK; ix++) {
      if (arraySet) {
#ifdef HAS_TCL_OBJECTS
	Tcl_ListObjAppendElement (interp, resList,
				  Tcl_NewStringObj (currCol+1, *currCol));
#else
	sprintf (currName, "%.*s", *currCol, currCol+1);
	Tcl_AppendElement (interp, currName);
#endif
	currCol += *currCol + 1;
      }
#ifdef HAS_TCL_OBJECTS
      resObj = Tcl_NewObj ();
      Tcl_ListObjAppendElement (interp, resList, resObj);
#endif
      ret = appendOneValue (interp, cursorVars->logonInfo, cursorVars, data,
			    resObj, nullValue, specialNull, currSi++, 1);
    }
    if (count > 1 && ret == TCL_OK) {
#ifndef HAS_TCL_OBJECTS
      AppendResult (interp, "} ");
#endif
      data = getNextData (data, cursorVars);
    } else {
      data = NULL;
    }
  }

  return ret;
}

/* -------------------------------------------------------------- */

static int
appendOneValue (interp, vars, cursorVars, data, destObj, nullValue, specialNull,
		currSi, isElement)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Current logon Handle. */
     CursorInfo *cursorVars;		/* Current cursor (or NULL). */
     char       *data;			/* Value to append as string. */
     Tcl_Obj    *destObj;		/* Destination object to append element,
					 * or Null, if dest is interp->result. */
     char       *nullValue;		/* Representation of a NULL value. */
     char       *specialNull;		/* Representation of a SPECIAL NULL. */
     field_info *currSi;		/* Shortfield info of value. */
     int         isElement;		/* append result as element? */
{
  int ret;				/* return value. */

#if ADABAS_VERSION >= 62
  if (currSi->io_type == IO_INPUT)
    return TCL_OK; /* nothing to append. */
#endif    

  ret = AdabasAppendData (interp, vars, cursorVars, data+currSi->bufpos-1,
			  destObj, currSi->data_type, currSi->in_out_len,
			  currSi->length, currSi->frac,
			  nullValue, specialNull, isElement);

  if (cursorVars) {
#if ADABAS_VERSION >= 62
    if (currSi->data_type == DSTRA  ||
	currSi->data_type == DSTRE  ||
	currSi->data_type == DSTRB  ||
	currSi->data_type == DSTRDB ||
	currSi->data_type == DSTRUNI) {
      cursorVars->longColFound = 1;
    }
#else /* ADABAS_VERSION <= 61 */
    if (currSi->data_type == CSP_INFO_C_OLDLONG_CHAR  ||
	currSi->data_type == CSP_INFO_B_OLDLONG_BYTE  ||
	currSi->data_type == CSP_INFO_A_OLDLONG_ASCII_DBYTE) {
      cursorVars->longColFound = 1;
    }
#endif    
  }
  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * getNextData --
 *
 *      Looks for the next row in the given dataPart.
 *
 * Results:
 *      A pointer to the found next row, or NULL if there is no further row.
 *
 * Side effects:
 *      Increases some internal counter (rowIx and dataEnd) of the given cursor,
 *      which will be used by subsequent calls of getNextData.
 *
 *----------------------------------------------------------------------
 */

static char *
getNextData (data, cursorVars)
     char       *data;			/* Buffer of data part. */
     CursorInfo *cursorVars;		/* Current cursor. */
{
  field_info *lastSi;			/* Shortinfo of last column in row. */
  int4        nextOff;			/* Length of one row. */

  cursorVars->rowIx++;
  lastSi  = cursorVars->shortInfos+(cursorVars->cntParams-1);
  nextOff = lastSi->bufpos+lastSi->in_out_len-1;

  if (cursorVars->rowIx >= cursorVars->massRows ||
      cursorVars->dataEnd + nextOff >= cursorVars->dataLength) {

    /*
     * We reached the end of data, so better return a NULL.
     */

    return NULL;
  }

  /*
   * Increase the dataEnd marker in cursorVars and
   * return the adjusted data pointer.
   */

  cursorVars->dataEnd += nextOff;
  return data+nextOff;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasColumns --
 *
 *      A list of the names of the selected columns of the last sql command
 *      will be constructed.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursor
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message. In all other cases the adamsg array
 *      will contain lists, that describes data type, length and fraction
 *      of the selected columns.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasColumns (interp, cursorVars)
     Tcl_Interp *interp;		/* Current interpreter. */
     CursorInfo *cursorVars;		/* Current cursor. */
{
  int         ret;			/* return value. */
  char       *dataType;			/* Data type of current column. */
  long        length;			/* Length of current column. */
  long        frac;			/* Fraction of current column. */
  int         ix;			/* Count on fetch parameters. */
  field_info *currSi;			/* Shortinfo of current parameter. */
  char       *currCol;			/* Pointer to name of current parameter. */
#ifndef HAS_TCL_OBJECTS
  char        currName[IDENTIFIER+1];	/* Buffer for null terminated name. */
#endif

  Tcl_ResetResult (interp);
  ret = checkCursor (interp, cursorVars);
  if (protFile) {
    fprintf (protFile, "COLUMNS  (cursor=\"%s\")\n",
	     ret == TCL_OK ? cursorVars->cursorName : "###invalid###");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  AdabasUnsetMsg (interp, AdamsgColtypes);
  AdabasUnsetMsg (interp, AdamsgCollengths);
  AdabasUnsetMsg (interp, AdamsgColprecs);
  AdabasUnsetMsg (interp, AdamsgColscales);

  currSi  = cursorVars->shortInfos;
  currCol = cursorVars->columnNames;

  for (ix = 0; ix < cursorVars->cntParams; ix++) {
    dataType = dataType2String (currSi->data_type);
    length   = currSi->length;
    frac     = currSi->frac;

    AdabasLAppendMsgString (interp, AdamsgColtypes,   dataType);
    AdabasLAppendMsgInt    (interp, AdamsgCollengths, length);
    AdabasLAppendMsgInt    (interp, AdamsgColprecs,   length);
    AdabasLAppendMsgInt    (interp, AdamsgColscales,  frac);

#ifdef HAS_TCL_OBJECTS
    Tcl_ListObjAppendElement (interp, Tcl_GetObjResult (interp),
			      Tcl_NewStringObj (currCol+1, *currCol));
#else
    sprintf (currName, "%.*s", *currCol, currCol+1);
    Tcl_AppendElement (interp, currName);
#endif

    currSi++;
    currCol += *currCol + 1;
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasReadLong --
 *
 *       *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursor
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasReadLong (interp, cursorVars, longDescObj, tableName, columnName,
		whereCond, encoding, fileName)
     Tcl_Interp  *interp;		/* Current interpreter. */
     CursorInfo  *cursorVars;		/* Current cursor. */
     Tcl_Obj     *longDescObj;		/* Long descriptor (or NULL). */
     char        *tableName;		/* Table name (or NULL). */
     char        *columnName;		/* Column name (or NULL). */
     char        *whereCond;		/* Where condition (or NULL). */
     int          encoding;		/* Encoding kind. */
     char        *fileName;		/* Name of file to write in (or NULL). */
{
  int              ret;			/* return value. */
  char            *descrName;		/* Name of long descriptor. */
  long_descriptor  longDescBuf;		/* Long descriptor of column to read. */
  LongDescInfo    *longDescInfo;	/* long desc info of given object. */
  long_descriptor *longDesc;		/* Pointer to current long desc. */
  long             bytes;		/* Number of bytes already read. */
  int              cont;		/* Must we continue to read? */
  int              dataType;		/* Data type of long column. */
  char            *nullValue;		/* Representation of a NULL value. */
  char            *specialNull;		/* Representation of a SPECIAL NULL. */

  Tcl_ResetResult (interp);
  ret = checkCursor (interp, cursorVars);
  if (protFile) {
    fprintf (protFile, "READLONG (cursor=\"%s\"",
	     ret == TCL_OK ? cursorVars->cursorName : "###invalid###");
    if (longDescObj) {
      fprintf (protFile, ",descriptor=\"%s\"",
	       Tcl_GetStringFromObj (longDescObj, (int *) NULL));
    }
    if (tableName) {
      fprintf (protFile, ",tableName=\"%s\"", tableName);
    }
    if (columnName) {
      fprintf (protFile, ",columnName=\"%s\"", columnName);
    }
    if (whereCond) {
      fprintf (protFile, ",whereCond=\"%s\"", whereCond);
    }
    if (encoding != encNil) {
      fprintf (protFile, ",encoding=%d", encoding);
    }
    if (fileName) {
      fprintf (protFile, ",fileName=\"%s\"", fileName);
    }
    fprintf (protFile, ")\n");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  if (longDescObj) {

    /*
     * The variant with the long descriptor is given.
     * First check, if it is a nullvalue.
     */

    descrName = Tcl_GetStringFromObj (longDescObj, (int *) NULL);
    getNullValue (interp, &nullValue, &specialNull);
    if (!strcmp (descrName, nullValue)) {
      AppendResult (interp, nullValue);
      return TCL_OK;
    }
    if (AdabasGetLongDescFromObj (interp, longDescObj,
				  &longDescInfo) == TCL_ERROR) {
      return TCL_ERROR;
    }
    if (cursorVars != longDescInfo->cursor) {
      SetResult (interp, "Long handle is not valid for this cursor");
      return TCL_ERROR;
    }
    if (!cursorVars->longColFound) {
      SetResult (interp, "call of adareadlong must be preceded by adafetch");
      return TCL_ERROR;
    }
    longDesc = &longDescInfo->longDesc;
    dataType =  longDescInfo->dataType;
  } else {

    /*
     * The variant with table, column and condition is given.
     * So first get a new copy of the cursor handle, then start the SELECT.
     */

    AdabasDeleteAllLongDescrOf (cursorVars->logonInfo);
    if (!(cursorVars = AdabasOpen (interp, cursorVars->logonInfo))) {
      return TCL_ERROR;
    }
    longDesc = &longDescBuf;
    if (AdabasLongSelect (interp, cursorVars, tableName, columnName,
			  whereCond, longDesc) == TCL_ERROR) {
      (void) AdabasClose (interp, cursorVars);
      return TCL_ERROR;
    }
    dataType = cursorVars->shortInfos->data_type;
  }

  if (longDesc->valmode == VM_NODATA) {
    bytes = 0;
    cont  = 1;
  } else {
    if (AdabasLongReadFirst (interp, cursorVars, longDesc, dataType, encoding,
			     fileName, &bytes, &cont) == TCL_ERROR) {
      if (!longDescObj) {
	(void) AdabasClose (interp, cursorVars);
      }
      return TCL_ERROR;
    }
  }

  while (cont) {
    if (AdabasLongReadNext (interp, cursorVars, longDesc, dataType, encoding,
			    fileName, &bytes, &cont) == TCL_ERROR) {
      if (!longDescObj) {
	(void) AdabasClose (interp, cursorVars);
      }
      return TCL_ERROR;
    }
  }

  /*
   * If called with a filename to write in, the number of
   * bytes written are returned.
   */

  if (fileName) {
    SetIntResult (interp, bytes);
  }

  if (!longDescObj) {
    (void) AdabasClose (interp, cursorVars);
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasWriteLong --
 *
 *       *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursor
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasWriteLong (interp, cursorVars, tableName, columnName,
		 whereCond, encoding, fileName, longValue)
     Tcl_Interp  *interp;		/* Current interpreter. */
     CursorInfo  *cursorVars;		/* Current cursor. */
     char        *tableName;		/* Table name (or NULL). */
     char        *columnName;		/* Column name (or NULL). */
     char        *whereCond;		/* Where coindition (or NULL). */
     int          encoding;		/* Encoding kind. */
     char        *fileName;		/* Name of file to read from (or NULL). */
     char        *longValue;		/* Value to write (or NULL). */
{
  int             ret;			/* return value. */
  long_descriptor longDesc;		/* Long descriptor of column to write. */
  int             longLength;		/* Length of value to write. */
  int4            bytes;		/* Number of bytes already read. */
  int             cont;			/* Must we continue to read? */
  Tcl_DString     parsId;		/* Parsid of UPDATE command. */
  Tcl_DString     substVal;		/* encoded value of 'longValue'. */

  Tcl_ResetResult (interp);
  ret = checkCursor (interp, cursorVars);
  if (protFile) {
    fprintf (protFile, "WRITLONG (cursor=\"%s\"",
	     ret == TCL_OK ? cursorVars->cursorName : "###invalid###");
    if (tableName) {
      fprintf (protFile, ",tableName=\"%s\"", tableName);
    }
    if (columnName) {
      fprintf (protFile, ",columnName=\"%s\"", columnName);
    }
    if (whereCond) {
      fprintf (protFile, ",whereCond=\"%s\"", whereCond);
    }
    if (encoding != encNil) {
      fprintf (protFile, ",encoding=\"%d\"", encoding);
    }
    if (fileName) {
      fprintf (protFile, ",fileName=\"%s\"", fileName);
    }
    if (longValue) {
      fprintf (protFile, ",longValue[50]=\"%.50s\"", longValue);
    }
    fprintf (protFile, ")\n");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  AdabasDeleteAllLongDescrOf (cursorVars->logonInfo);
  if (!(cursorVars = AdabasOpen (interp, cursorVars->logonInfo))) {
    return TCL_ERROR;
  }

  /*
   * Send the constructed UPDATE sql command to the kernel for parsing.
   */

  if (AdabasLongUpdate (interp, cursorVars, tableName, columnName,
			whereCond, &parsId) == TCL_ERROR) {
    (void) AdabasClose (interp, cursorVars);
    return TCL_ERROR;
  }

  if (AdabasLongValLength (interp, fileName, longValue, encoding,
			   &substVal, &longLength) == TCL_ERROR) {
    (void) AdabasClose (interp, cursorVars);
    return TCL_ERROR;
  }
    
  /*
   * Now the parsid will be executed, the first part of the long value
   * (or all, if it fits into the packet) is given as data.
   */

  if (AdabasLongWriteFirst (interp, cursorVars, &longDesc, &parsId, fileName,
			    Tcl_DStringValue (&substVal), encoding, longLength,
			    &bytes, &cont) == TCL_ERROR) {
    Tcl_DStringFree (&parsId);
    if (longValue) {
      Tcl_DStringFree (&substVal);
    }
    (void) AdabasClose (interp, cursorVars);
    return TCL_ERROR;
  }

  if (cont) {

    /*
     * While not all of the long value is written,
     * continue with sending PUTVAL packets.
     */

    while (cont) {
      if (AdabasLongWriteNext (interp, cursorVars, &longDesc, fileName,
			       Tcl_DStringValue (&substVal), encoding,
			       longLength, &bytes, &cont) == TCL_ERROR) {
	Tcl_DStringFree (&parsId);
	if (longValue) Tcl_DStringFree (&substVal);
	(void) AdabasClose (interp, cursorVars);
	return TCL_ERROR;
      }
    }
  }

  /*
   * Finally throw away the generated parsid; ignore its return value,
   * since we are in the clean up phase now.
   */

  (void) AdabasSend2PartPacket (interp, cursorVars->logonInfo,
				pkCommand, "DROP PARSEID", 0,
				pkParsid, Tcl_DStringValue (&parsId),
				Tcl_DStringLength (&parsId));

#if ADABAS_VERSION <= 61
  if (cursorVars->logonInfo->packetInfo.commitImmediately) {
    (void) AdabasSendCmdPacket (interp, cursorVars->logonInfo, 0, 1,
				"COMMIT WORK");
  }
#endif

  Tcl_DStringFree (&parsId);
  if (longValue) Tcl_DStringFree (&substVal);
  return AdabasClose (interp, cursorVars);
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasAutoCommit --
 *
 *      Switches on/off the auto commit feature af adabas d.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasAutoCommit (interp, vars, onOff)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Logon handle. */
     int         onOff;			/* autocommit on? */
{
  int ret;				/* return value. */

  Tcl_ResetResult (interp);
  ret = AdabasCheckLogon (interp, vars);
  if (protFile) {
    fprintf (protFile, "AUTOCOM  (logon=\"%s\",%s)\n",
	     ret == TCL_OK ? vars->logonName : "###invalid###",
	     onOff ? "on" : "off");
    fflush (protFile);
  }

  if (ret == TCL_OK) {
    vars->packetInfo.commitImmediately = onOff;
  }
  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasCommit --
 *
 *      Sends a COMMIT on the given logon to the database kernel.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      All transactions of the given logon are commited.
 *
 *----------------------------------------------------------------------
 */

int
AdabasCommit (interp, vars)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Logon handle. */
{
  int ret;				/* return value. */

  Tcl_ResetResult (interp);
  ret = AdabasCheckLogon (interp, vars);
  if (protFile) {
    fprintf (protFile, "COMMIT   (logon=\"%s\")\n",
	     ret == TCL_OK ? vars->logonName : "###invalid###");
    fflush (protFile);
  }

  AdabasDeleteAllLongDescrOf (vars);
  if (ret == TCL_OK) {
    ret = AdabasSendCmdPacket (interp, vars, 0, 1, "COMMIT WORK");
  }
  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasRollback --
 *
 *      Sends a ROLLBACK on the given logon to the database kernel.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      All transactions of the given logon are rollbacked.
 *
 *----------------------------------------------------------------------
 */

int
AdabasRollback (interp, vars)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Logon handle. */
{
  int ret;				/* return value. */

  Tcl_ResetResult (interp);
  ret = AdabasCheckLogon (interp, vars);
  if (protFile) {
    fprintf (protFile, "ROLLBACK (logon=\"%s\")\n",
	     ret == TCL_OK ? vars->logonName : "###invalid###");
    fflush (protFile);
  }

  AdabasDeleteAllLongDescrOf (vars);
  if (ret == TCL_OK) {
    ret = AdabasSendCmdPacket (interp, vars, 0, 1, "ROLLBACK WORK");
  }
  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasUtil --
 *
 *      Sends the given utility command to the database kernel.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      Depends on the command send to the database.
 *
 *----------------------------------------------------------------------
 */

int
AdabasUtil (interp, vars, cmd)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Logon handle. */
     char       *cmd;			/* Utility command to execute. */
{
  char       *data;			/* Pointer to resulting data part. */
  int4        dataLength;		/* Length of resulting data. */
  int         argCount;			/* No. of lines in data part (always 1). */
  int         cntParams;		/* No. of columns in data part. */
  int         ret;			/* return value. */
  char       *nullValue;		/* Representation of a NULL value. */
  char       *specialNull;		/* Representation of a SPECIAL NULL. */
  char        dbName[200];		/* Null terminated dbname. */
  int         ix;			/* Count on fetch parameters. */
  field_info *currSi;			/* Shortinfo of current parameter. */
  field_info *shortInfos  = NULL;	/* Array of column descriptions. */
  char       *columnNames = NULL;	/* Array of column names. */
  Tcl_Obj    *resObj;			/* Object to put result into. */
#ifndef HAS_TCL_OBJECTS
  Tcl_DString resultStr;		/* Temporary result string. */
#endif

  Tcl_ResetResult (interp);
  ret = AdabasCheckLogon (interp, vars);
  if (protFile) {
    fprintf (protFile, "UTIL     (logon=\"%s\",cmd=\"%s\")\n",
	     ret == TCL_OK ? vars->logonName : "###invalid###",
	     cmd);
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  if (vars->packetInfo.messType != M_UTILITY) {
    SetResult (interp, "Logon handle must be utility service");
    return TCL_ERROR;
  }

  AdabasDeleteAllLongDescrOf (vars);

  if (AdabasSendCmdPacket (interp, vars, 0, 1, cmd) == TCL_ERROR) {
    return TCL_ERROR;
  }

#if ADABAS_VERSION >= 62
  data = getData (&vars->rcvPacket->segm, &dataLength, &argCount, UTILITY_CMD);
#else /* ADABAS_VERSION <= 61 */
  data = getData (&vars->sqlPacket, &dataLength, &argCount, UTILITY_CMD);
#endif
  if (data && dataLength) {
#if ADABAS_VERSION >= 62
    cntParams = getShortInfos (&vars->rcvPacket->segm, &shortInfos, &columnNames);
#else /* ADABAS_VERSION <= 61 */
    cntParams = 0;
#endif

    if (cntParams && shortInfos) {
      getNullValue (interp, &nullValue, &specialNull);
      currSi = shortInfos;
      ret    = TCL_OK;
#ifdef HAS_TCL_OBJECTS
      resObj = Tcl_GetObjResult (interp);
#else
      resObj = (Tcl_Obj *) NULL;
#endif
      for (ix = 0; ix < cntParams && ret == TCL_OK; ix++) {
	ret = appendOneValue (interp, vars, (CursorInfo *) NULL, data, resObj,
			      nullValue, specialNull, currSi++, 1);
      }

    } else 
      switch (*data) {
      case  2:
	p2c (dbName, data+1, dataLength-1);
	AppendResult (interp, "warm ");
	AppendResult (interp, dbName);
	break;
      case 0: case 1:
	AppendResult (interp, "cold");
	break;
      default:
#ifdef HAS_TCL_OBJECTS
	Tcl_SetStringObj (Tcl_GetObjResult (interp), data, dataLength);
#else
	Tcl_DStringInit   (&resultStr);
	Tcl_DStringAppend (&resultStr, data, dataLength);
	Tcl_DStringResult (interp, &resultStr);
#endif
    }
    if (shortInfos) {
      ckfree ((char *) shortInfos);
    }
    if (columnNames) {
      ckfree ((char *) columnNames);
    }
  }
  return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * AdabasUsage --
 *
 *      Sends usage command with the given kind to the database kernel.
 *
 * Results:
 *      The return value is normally TCL_OK.
 *
 * Side effects:
 *      Depends on the command send to the database.
 *
 *----------------------------------------------------------------------
 */
int
AdabasUsage (interp, vars, usageKind, objectType,
	     param1, param2, param3)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Current cursor. */
     int         usageKind;		/* 0=on, 1=add, 2=off. */
     char       *objectType;		/* Object type. */
     char       *param1;		/* First parameter (or NULL). */
     char       *param2;		/* Second parameter (or NULL). */
     char       *param3;		/* Third parameter (or NULL). */
{
  int         ret;			/* Return value of send request. */
  char       *command;			/* Command to send to the kernel. */
  Tcl_DString data;			/* Constructed data part. */
  char        param[IDENTIFIER+2];	/* Buffer for defbyte, param and \0. */

  Tcl_ResetResult (interp);
  ret = AdabasCheckLogon (interp, vars);
  if (protFile) {
    fprintf (protFile, "USAGE    %s (logon=\"%s\"",
	     ret == TCL_OK ? vars->logonName : "###invalid###",
	     usageKind == 0 ? "ON" : usageKind == 1 ? "ADD" : "OFF");
    if (objectType) {
      fprintf (protFile, ",objectType=\"%s\"", objectType);
    }
    if (param1) {
      fprintf (protFile, ",param1=\"%s\"", param1);
    }
    if (param2) {
      fprintf (protFile, ",param2=\"%s\"", param2);
    }
    if (param3) {
      fprintf (protFile, ",param3=\"%s\"", param3);
    }
    fprintf (protFile, ")\n");
    fflush (protFile);
  }
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  /*
   * Check, if the parameters are consistent.
   * Also determine the command to send dependend on the usage kind.
   */

  switch (usageKind) {
  case 0:  command = "USAGE ON";  break;
  case 1:  command = "USAGE ADD"; break;
  default: command = "USAGE OFF"; break;
  }

  /*
   * Construct the data part and send it to the kernel.
   */

  Tcl_DStringInit (&data);
  if (objectType) {
    sprintf (param, " %-*s", 8, objectType);
    Tcl_DStringAppend (&data, param, -1);
  }
  if (param1) {
    sprintf (param, " %-*s", IDENTIFIER, param1);
    Tcl_DStringAppend (&data, param, -1);
  }
  if (param2) {
    sprintf (param, " %-*s", IDENTIFIER, param2);
    Tcl_DStringAppend (&data, param, -1);
  }
  if (param3) {
    sprintf (param, " %-*s", IDENTIFIER, param3);
    Tcl_DStringAppend (&data, param, -1);
  }

  ret = AdabasSend2PartPacket (interp, vars,
			       pkCommand, command, 0,
			       pkData, Tcl_DStringValue (&data), 0);
  Tcl_DStringFree (&data);
  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasSpecial --
 *
 *      Sends a special command (like switch) to the database kernel.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasSpecial (interp, vars, messType, data)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Logon handle. */
     int         messType;		/* Special message type. */
     char       *data;			/* Data for special command (or ""). */
{
  int ret;				/* return value. */
  int orgMessType;			/* original message type. */

  Tcl_ResetResult (interp);
  ret = AdabasCheckLogon (interp, vars);
  if (protFile) {
    fprintf (protFile, "SPECIAL  (logon=\"%s\",messtype=\"%s\",data=\"%s\")\n",
	     ret == TCL_OK ? vars->logonName : "###invalid###",
	     messType2String (messType), data);
    fflush (protFile);
  }

  AdabasDeleteAllLongDescrOf (vars);

  if (ret == TCL_OK) {
    orgMessType = vars->packetInfo.messType;
    vars->packetInfo.messType = messType;
    ret = AdabasSend2PartPacket (interp, vars, pkData, data, 0, pkNil, "", 0);
    vars->packetInfo.messType = orgMessType;
  }
  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasLogon --
 *
 *      Opens a connection to the runtime system of an Adabas D datatbase server.
 *      If the parameter connectStr is not '-noconnect', a connection on the
 *      SQL level is made also.
 *
 * Results:
 *      The return value is normally the newly created logon handle; if there
 *      ar any problems while connecting, NULL will be returned.
 *
 * Side effects:
 *      If all was okay, a connection is established, that can be accessed
 *      via the returned logon handle.
 *
 *----------------------------------------------------------------------
 */

AdabasInfo *
AdabasLogon (interp, serverdb, connectStr, isolation, sqlMode, serv,
	     locale, applKind)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *serverdb;		/* Serverdb to connect to (or NULL) */
     char       *connectStr;		/* Connect string (e.g. user,passwd) */
     int         isolation;		/* Isolation level (or -1) */
     char       *sqlMode;		/* Wanted Sqlmode (or NULL) */
     int         serv;			/* SQL Service (user/utility) */
     char       *locale;		/* Character set to use (or NULL) */
     int         applKind;  	 	/* User, control or odbs service */
{
  Tcl_RegExp   regExp;			/* compiled regexp for connectStr */
  char        *start;			/* Index into string returned by regexp */
  char        *end;			/* Index into string returned by regexp */
  Tcl_DString  connect;			/* constructed sql CONNECT command */
  Tcl_DString  data;			/* data part of CONNECT command */
  Tcl_DString  cUser;			/* user name (out of connectStr) */
  Tcl_DString  cCryptPasswd;		/* crypted password (null terminated) */
  char         isoString[80];		/* isolation level of sql command */
  Tcl_DString  help;			/* help string */
  char         xCryptPasswd[24];	/* crypted password (pascal like) */
  char         xUser[sizeof(username)+1]; /* user name (pascal like) */
  char         xNode[sizeof(nodeid)+1]; /* servernode name (pascal like) */
  char         xDbName[sizeof(dbname)+1]; /* serverdb name (pascal like) */
  AdabasInfo  *vars;			/* Freshly opened logon handle */
  char        *xuserKey;		/* Given userkey (ala ,<userkey>) */
  int          ret;			/* Temporary return value */
  char        *user;			/* user name (null terminated) */
  xuser_record userParams;		/* xuser record of arg line */

  if (protFile) {
    fprintf (protFile, "LOGON    (connect=\"%s\"", connectStr);
    if (serverdb) {
      fprintf (protFile, ",serverdb=\"%s\"", serverdb);
    }
    if (sqlMode) {
      fprintf (protFile, ",sqlmode=\"%s\"", sqlMode);
    }
    if (isolation >= 0) {
      fprintf (protFile, ",isolation=\"%d\"", isolation);
    }
    if (serv != SQL_USER) {
      fprintf (protFile, ",service=\"%s\"", service2String (serv));
    }
    if (locale) {
      fprintf (protFile, ",locale=\"%s\"", locale);
    }
    if (applKind != UserAppl) {
      fprintf (protFile, ",applKind=\"%s\"", applKind2String (applKind));
    }
    fprintf (protFile, ")\n");
    fflush (protFile);
  }

  Tcl_ResetResult (interp);

  if (isolation >= 0
      && (isolation != 0 && isolation != 15 &&
	  isolation != 1 && isolation != 10 &&
	  isolation != 2 && isolation != 20 &&
	  isolation != 3 && isolation != 30)) {
    SetResult (interp, "isolation must be 0, 1, 2, 3, 10, 15, 20 or 30");
    return (AdabasInfo *) NULL;
  }
  if (sqlMode && scanSqlMode (interp, sqlMode, 1) < 0)
    return (AdabasInfo *) NULL;
    
  Tcl_DStringInit (&help);

  regExp = Tcl_RegExpCompile (interp, CONNECT_RE);
  if (!Tcl_RegExpExec (interp, regExp, connectStr, connectStr)) {

    /*
     * There is a pseudo user "-noconnect" to allow sessions without
     * a real connect (e.g. for vtrace or switch commands).
     */

    if (!strcmp (connectStr, "-noconnect")) {
      if (sqlMode) {
	SetResult (interp, "specification of sqlmode useless, if not connecting");
	return (AdabasInfo *) NULL;
      }
      user = (char *) NULL; /* special flag: no sql connection, please. */
    } else {
      AppendResult (interp, "Invalid connectString \"");
      AppendResult (interp, connectStr);
      AppendResult (interp, "\", must be \"user,passwd\", \",userKey\", \",\"");
      AppendResult (interp, " or \"-noconnect\"");
      return (AdabasInfo *) NULL;
    }
  } else {
    Tcl_DStringInit (&cUser);
    Tcl_DStringInit (&cCryptPasswd);

    Tcl_RegExpRange (regExp, CONNECT_USER_RG, &start, &end);
    if (start == end) {

      /*
       * There was no user given; perhaps the name behind
       * the comma is meant as an xuser key.
       */

      Tcl_RegExpRange (regExp, CONNECT_PASSWD_RG, &start, &end);
      if (start == end) {

	/*
	 * There was no xuser key given either.
	 * So let's look after the default user.
	 */

	xuserKey = NULL;
      } else {
	Tcl_DStringAppend (&help, start, end-start);
	xuserKey = Tcl_DStringValue (&help);
      }
      if (AdabasXUserGet (interp, xuserKey, &userParams) == TCL_ERROR) {
	Tcl_DStringFree (&help);
	return (AdabasInfo *) NULL;
      }

      p2c (xUser, userParams.xu_user,
	   sizeof (userParams.xu_user));
      p2c (xDbName, userParams.xu_serverdb,
	   sizeof (userParams.xu_serverdb));
      p2c (xNode, userParams.xu_servernode,
	   sizeof (userParams.xu_servernode));
      memcpy (xCryptPasswd, userParams.xu_password,
	      sizeof (userParams.xu_password));

      Tcl_DStringFree (&help);
      decodeSequence (interp, encEscape, (Tcl_Obj *) NULL, &cCryptPasswd,
		      xCryptPasswd, sizeof (xCryptPasswd));
      user = xUser;
      if (*xDbName && *xDbName != ' ') {
	if (*xNode && *xNode != ' ') {
	  Tcl_DStringAppend (&help, xNode, -1);
	  Tcl_DStringAppend (&help, ":",    1);
	}
	Tcl_DStringAppend (&help,  xDbName, -1);
	serverdb = Tcl_DStringValue (&help);
      }
    } else {

      /*
       * There was really a pair of user and password given.
       */

      Tcl_DStringAppend (&cUser, start, end-start); 
      user = Tcl_DStringValue (&cUser);

      Tcl_RegExpRange   (regExp, CONNECT_PASSWD_RG, &start, &end);
      Tcl_DStringAppend (&help, start, end-start);

      /*
       * Change passwd into uppercase before crypting it.
       */

      if (*(Tcl_DStringValue (&help))               != '"' &&
	  *(Tcl_DStringValue (&help)+(end-start-1)) != '"') {
	for (start = Tcl_DStringValue (&help); *start; start++) {
	  *start = toupper (*start);
	}
      }
      AdabasPasswdCrypt (interp, &cCryptPasswd, Tcl_DStringValue (&help));
    }
  }

  /*
   * Now open the connection to adabas with the given service and serverdb.
   */

  ret = AdabasOpenConnection (interp, serv, serverdb, "", &vars);
  Tcl_DStringFree (&help);
  if (ret == TCL_ERROR) return (AdabasInfo *) NULL;

  switch (applKind) {
  case ControlAppl:
    memcpy (vars->packetInfo.application, "CON", 3);
    break;
  case OdbcAppl:
    memcpy (vars->packetInfo.application, "ODB", 3);
    break;
  default:
    if (serv == SQL_UTILITY) {
      vars->packetInfo.messType = M_UTILITY;
    }
  }

  if (user) {

    /*
     * Now create a session with the given user, password (and sqlmode).
     */

    Tcl_DStringInit   (&connect);
    Tcl_DStringAppend (&connect, "CONNECT ", -1);
    Tcl_DStringAppend (&connect, user,       -1);
    Tcl_DStringAppend (&connect, " IDENTIFIED BY :x", -1);

    if (sqlMode) {
      Tcl_DStringAppend (&connect, " SQLMODE ", -1);
      Tcl_DStringAppend (&connect,   sqlMode,   -1);
    }

    if (isolation >= 0) {
      sprintf (isoString, " ISOLATION LEVEL %d", isolation);
      Tcl_DStringAppend (&connect, isoString, -1);
    }

#if ADABAS_VERSION >= 62
    if (locale) {
      Tcl_DStringAppend (&connect, " CHARACTER SET ", -1);
      Tcl_DStringAppend (&connect,   locale,          -1);
    }
#endif

    /*
     * Construct a data part, which consists of a defined byte,
     * the password and the terminal ID.
     */

    Tcl_DStringInit   (&data);
    decodeSequence    (interp, encEscape, (Tcl_Obj *) NULL, &data, "", 1);
    Tcl_DStringAppend (&data, Tcl_DStringValue (&cCryptPasswd), -1);
    Tcl_DStringAppend (&data, " ", 1);
    AdabasRteTermId   (Tcl_DStringValue (&help)); 
    Tcl_DStringAppend (&data, Tcl_DStringValue (&help), -1); 

    ret = AdabasSend2PartPacket (interp, vars,
				 pkCommand, Tcl_DStringValue (&connect), 0,
				 pkData,    Tcl_DStringValue (&data),    0);
  
    Tcl_DStringFree (&connect);
    Tcl_DStringFree (&data);
    Tcl_DStringFree (&help);
  }

  if (ret == TCL_ERROR) {
    (void) AdabasCloseConnection (interp, vars->logonName, vars);
    return (AdabasInfo *) NULL;
  }

  Tcl_ResetResult (interp);
  AppendResult (interp, vars->logonName);

  return vars;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasLogoff --
 *
 *      Terminates the connection of the given logon handle.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      The given logon handle couldn't be used further on.
 *
 *----------------------------------------------------------------------
 */

int
AdabasLogoff (interp, vars)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Current logon. */
{
  int ret;				/* return value. */

  Tcl_ResetResult (interp);
  ret = AdabasCheckLogon (interp, vars);
  if (protFile) {
    fprintf (protFile, "LOGOFF   (logon=\"%s\")\n",
	     ret == TCL_OK ? vars->logonName : "###invalid###");
    fflush (protFile);
  }

  if (ret == TCL_OK) {
    AdabasDeleteAllLongDescrOf (vars);
    (void) AdabasSendCmdPacket (interp, vars, 0, 1, "ROLLBACK WORK RELEASE");
    (void) AdabasCloseConnection (interp, vars->logonName, vars);
  }
  return ret;
}

/* -------------------------------------------------------------- */

static void
examineSqlCommand (interp, cmdString, withInfo, withResCount,
		   selectKind, selectInto)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *cmdString;		/* String containing a SQL command. */
     int        *withInfo;		/* SQL, that returns results? */
     int        *withResCount;		/* SQL, that returns rowcnt? */
     SelectKind *selectKind;		/* returned kind of sql statement. */
     char      **selectInto;		/* position of first parameter name
					 * in cmdString (for SELECT INTO). */
{
  Tcl_DString cmd;
  Tcl_RegExp  regExp;			/* compiled regexp for sqlCommand */
  char       *start;			/* Index into cmdString ret. by regexp */
  char       *end;			/* Index into cmdString ret. by regexp */

  *withInfo     = 0;
  *withResCount = 0;
  *selectKind   = noSelect;
  *selectInto   = (char *) NULL;

  regExp = Tcl_RegExpCompile (interp, COMMAND_RE);
  if (!Tcl_RegExpExec (interp, regExp, cmdString, cmdString)) return;

  Tcl_RegExpRange   (regExp, COMMAND_TOKEN_RG, &start, &end);
  Tcl_DStringInit   (&cmd);
  Tcl_DStringAppend (&cmd, start, end-start);

  for (start = Tcl_DStringValue (&cmd); *start; start++)
    *start = toupper (*start);

  if (!strcmp (Tcl_DStringValue (&cmd), "SELECT")) {
    *selectKind = massSelect;
    *withInfo   = 1;
    /*
     * Now that we found a SELECT, lets look if it is a SELECT INTO.
     */
    Tcl_DStringFree   (&cmd);
    Tcl_DStringAppend (&cmd, cmdString, -1);
    for (start = Tcl_DStringValue (&cmd); *start; start++)
      *start = toupper (*start);
    regExp = Tcl_RegExpCompile (interp, INTO_RE);
    if (Tcl_RegExpExec (interp, regExp,
			Tcl_DStringValue (&cmd), Tcl_DStringValue (&cmd))) {
      Tcl_RegExpRange (regExp, INTO_VAR_RG, &start, &end);
      *selectKind = intoSelect;
      *selectInto = cmdString + (start - Tcl_DStringValue (&cmd));
    }

  } else if (!strcmp (Tcl_DStringValue (&cmd), "EXPLAIN")) {
    *withInfo     = 1;
    *selectKind   = explainSelect;

  } else if (!strcmp (Tcl_DStringValue (&cmd), "SHOW")) {
    *withInfo     = 1;
    *selectKind   = showSelect;

  } else if (!strcmp (Tcl_DStringValue (&cmd), "UPDATE") ||
	     !strcmp (Tcl_DStringValue (&cmd), "INSERT") ||
	     !strcmp (Tcl_DStringValue (&cmd), "DELETE")) {
    *withResCount = 1;
  }

  Tcl_DStringFree (&cmd);
}

/* -------------------------------------------------------------- */

static char *
applKind2String (kind)
     ApplKind kind;				/* Application kind to convert. */
{
  switch (kind) {
  case UserAppl:    return "user";
  case ControlAppl: return "control";
  case OdbcAppl:    return "odbc";
  default:          return "???";
  }
}

/* -------------------------------------------------------------- */

static void
getNullValue (interp, nullValue, specialNull)
     Tcl_Interp *interp;		/* Current interpreter. */
     char      **nullValue;		/* Representation of a NULL value. */
     char      **specialNull;		/* Representation of a SPECIAL NULL. */
{
  *nullValue   = AdabasGetMsgString (interp, AdamsgNullvalue, (int *) NULL);
  if (!nullValue)   *nullValue   = "";
  *specialNull = AdabasGetMsgString (interp, AdamsgSpecialnull, (int *) NULL);
  if (!specialNull) *specialNull = "";
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasCursorHandle --
 *
 *      Looks up the given cursorName in the cursor hash table.
 *
 * Results:
 *      The return value is normally a pointer to the CursorInfo structure
 *      of the found hash entry. If no cursor with the given name exists,
 *      NULL is returned.
 *
 * Side effects:
 *      The global array element adamsg(handle) will contain the cursorName,
 *      if interp isn't NULL.
 *
 *----------------------------------------------------------------------
 */

CursorInfo *
AdabasCursorHandle (interp, cursorName)
     Tcl_Interp *interp;		/* Used for error reporting if not NULL. */
     char       *cursorName;		/* Name to look for. */
{
  Tcl_HashEntry *hashEntry;		/* Entry with cursor name and infoPrt. */
  CursorInfo    *cursorVars;		/* Found cursorHandle to return. */

  if (!(hashEntry = Tcl_FindHashEntry (&cursorHash, cursorName))) {
    if (interp) {
      AppendResult (interp, "No valid cursor handle: ");
      AppendResult (interp, cursorName);
    }
    return (CursorInfo *) NULL;
  }
  cursorVars = (CursorInfo *) Tcl_GetHashValue (hashEntry);
  if (checkCursor (interp, cursorVars) == TCL_ERROR) {
    return (CursorInfo *) NULL;
  }

  if (interp) {
    AdabasSetMsgString (interp, AdamsgHandle, cursorName);
  }
  return cursorVars;
}

/*
 *----------------------------------------------------------------------
 *
 * checkCursor --
 *
 *      Checks whether the given cursor is known and alive.
 *
 * Results:
 *      The return value is normally TCL_OK; if the cursor is invalid,
 *      TCL_ERRORNULL is returned and interp->result will point to the
 *      error message.
 *
 * Side effects:
 *      None
 *
 *----------------------------------------------------------------------
 */

static int
checkCursor (interp, cursor)
     Tcl_Interp *interp;		/* Used for error reporting if not NULL. */
     CursorInfo *cursor;		/* Cursor handle. */
{
  if (!cursor) {
    if (interp) {
      SetResult (interp, "Invalid cursorHandle (NULL)");
    }
    return TCL_ERROR;
  }
  if (cursor->magic != CURSOR_MAGIC) {
    if (interp) {
      AppendResult (interp, "No valid cursor handle: ");
      AppendResult (interp, cursor->cursorName);
    }
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TracefileHandler --
 *
 *      Procedure, that traces all assignments to adamsg(tracefile)
 *      and reacts accordingly (Create, truncate or close the tracefile).
 *
 * Results:
 *      The return value is normally NULL; if the tracefile couldn't be
 *      opened, the error message is returned.
 *
 * Side effects:
 *      The trace file is opened or closed, the global variable protFile
 *      is set accordingly (NULL: protFile closed; else: protFile opened).
 *
 *----------------------------------------------------------------------
 */

static char *
TracefileHandler (dummy, interp, name1, name2, flags)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *name1;			/* Name of the whole array. */
     char       *name2;			/* Name of the array index. */
     int         flags;			/* What happened to the array element? */
{
  char *fileName;			/* Current value of adamsg(tracefile). */

  if (TCL_TRACE_DESTROYED & flags && !(TCL_INTERP_DESTROYED & flags)) {
    Tcl_TraceVar2 (interp, "adamsg", "tracefile",
		   TCL_TRACE_WRITES | TCL_TRACE_UNSETS | TCL_GLOBAL_ONLY,
		   TracefileHandler, (ClientData) NULL);
  }

  if (TCL_TRACE_WRITES & flags) {
    if (protFile) {
      fclose (protFile);
    }

    fileName = AdabasGetMsgString (interp, AdamsgTracefile, (int *) NULL);
    if (!(protFile = fopen (fileName, "w"))) {
      AdabasUnsetMsg (interp, AdamsgTracefile);
      return Tcl_PosixError (interp);
    }
    return (char *) NULL;
  } else if (TCL_TRACE_UNSETS & flags) {
    if (protFile) {
      fclose (protFile);
      protFile = (FILE *) NULL;
    }
    return (char *) NULL;
  } else {
    return "Invalid flag for var trace of adamsg(tracefile)";
  }
}

void
AdabasTrace (kind, cursorName, dir)
     char *kind;
     char *cursorName;
     char *dir;
{
  if (protFile) {
    fprintf (protFile, "%s %s '%s'\n", dir, kind, cursorName);
    fflush (protFile);    
  }
}
