/*
 * AdabasFormat.c --
 *
 *      Functions to transform a value in the internal format of Adabas D
 *      into a c string, integer, float or the like and vice versa.
 *
 * Copyright (c) 1996-1997 Christian Krone. All rights reserved.
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Tcl itself.
 * See also licence.terms
 *
 * $Id: AdabasFormat.c,v 1.50 1997/06/21 17:28:09 adabas Exp $
 */

/*
 * Import interface (should move into AdabasTclInt.h someday)
 */

#include <stdlib.h>
#include <string.h>
#include <tcl.h>

#include "adabas.h"
#include "sqlrte.h"
#include "AdabasPort.h"
#include "AdabasUtil.h"
#include "AdabasRte.h"
#include "AdabasPacket.h"
#include "AdabasSend.h"
#include "AdabasApi.h"
#include "AdabasFormat.h"

/*
 * Type definitions
 */

#define DATA_TYPE_REGEXP_STRING "^([^(]*)\\(([0-9]*)(,(-?[0-9]*))?\\)$"
#define DATA_TYPE_RANGE 1
#define LENGTH_RANGE    2
#define FRAC_RANGE      4
#define VAL_POS        33
#define VAL_LEN        37

/*
 * Numeric conversions return 0, if okay or 1, if number was truncated.
 * Since truncation is considered okay, we define a macro here.
 */

#define VDN_OK(numError) ((numError) <= 1)

/*
 * This local variable is a hash table full with all created long desciptors.
 * Since we must be able to get a descriptor from its name, it isn't
 * sufficient to have a list of all descriptor associated with its cursor.
 */

static Tcl_HashTable longDescHash;

/*
 * When creating names for long descriptors, we simply add one to the
 * following counter, which will be resetted befor each adasql,
 * adareadlong or adawritelong.
 */
static int longDescCount;

/*
 * Prototypes for procedures referenced only in this file.
 */

static int checkGetParams _ANSI_ARGS_((Tcl_Interp *interp, int objc,
				       Tcl_Obj *CONST objv[],
				       data_type *dataType,
				       int *ioLen, int *len, int *frac,
				       int *startPos, char **nullValue,
				       char **specialNull));
static int putInt _ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *data, int ioLen,
			       int len, int startPos, char *nullValue,
			       char *specialNull));
static int getIoLen _ANSI_ARGS_((int dataType, int length));
static LongDescInfo *longDescr2Hash _ANSI_ARGS_((Tcl_Interp *interp, int dataType,
						 AdabasInfo *vars,
						 CursorInfo *cursorVars,
						 long_descriptor *longDesc));
static int getEditForm _ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *formObj));
static int putEditForm _ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *queryList));
static int hash2LongInfo _ANSI_ARGS_((Tcl_Interp *interp, char *inDescName,
				      LongDescInfo **longDescInfo));
#if ADABAS_VERSION >= 62
static int appendUnicode _ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *destObj,
				      char *data, int length, int isElement));
#endif

#ifdef HAS_TCL_OBJECTS
static int longDescSetFromAny _ANSI_ARGS_((Tcl_Interp *interp,
					   Tcl_Obj *objPtr));
static void longDescUpdateString _ANSI_ARGS_((Tcl_Obj *objPtr));
static void longDescDupInternalRep _ANSI_ARGS_((Tcl_Obj *srcPtr,
						Tcl_Obj *dupPtr));
static Tcl_ObjType LongDescObjType =
{"longdesc", NULL, &longDescDupInternalRep, &longDescUpdateString,
 &longDescSetFromAny};
#endif

void
AdabasFormatInit (interp)
     Tcl_Interp *interp;
{
  /*
   * Initialize the only global variable, we have: the hash table of
   * long descriptors.
   */

  Tcl_InitHashTable (&longDescHash, TCL_STRING_KEYS);

#ifdef HAS_TCL_OBJ

  /*
   * The long descriptor parameter will become its own Tcl_ObjType,
   * so that most (if not all) of the hash table lookups can be omitted.
   */

  Tcl_RegisterObjType (&LongDescObjType);
#endif
}

/* ----------------- Exported procedures ------------------------- */

int
AdabasCryptCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     Tcl_Obj    *CONST objv[]; 		/* Argument objects. */
{
  Tcl_DString result;			/* Crypted password to return. */

  if (objc < 2) {
    SetResult (interp, "wrong # args: should be \"adabas crypt clear_pw\"");
    return TCL_ERROR;      
  }
  
  Tcl_DStringInit   (&result);
  AdabasPasswdCrypt (interp, &result,
		     Tcl_GetStringFromObj (objv[1], (int *) NULL));
  SetResult (interp, Tcl_DStringValue (&result));
  return TCL_OK;
}

/* -------------------------------------------------------------- */

void
AdabasPasswdCrypt (interp, cryptPw, passwd)
     Tcl_Interp  *interp;		/* Current interpreter. */
     Tcl_DString *cryptPw;		/* resulting crypted password. */
     char        *passwd;		/* clear password to be cryptified. */
{
  name    clearname;
  cryptpw crypt_pw;

  c2p (clearname, passwd, sizeof (clearname));
  s02applencrypt (clearname, crypt_pw);
  decodeSequence (interp, encEscape, (Tcl_Obj *) NULL,
		  cryptPw, crypt_pw, sizeof (crypt_pw));
}

/* -------------------------------------------------------------- */

int
AdabasPutCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     Tcl_Obj    *CONST objv[]; 		/* Argument objects. */
{
  data_type dataType    = DCHA;		/* Data type of value. */
  int       ioLen       = 2;		/* IO-Length of value. */
  int       len         = 1;		/* Extern length of value. */
  int       startPos    = 1;		/* Startposition in data part. */
  int       frac        = 0;		/* Fraction, if value is a number. */
  char     *nullValue   = "";		/* Representation of a NULL value. */
  char     *specialNull = "";		/* Representation of a SPECIAL NULL. */

  if (checkGetParams (interp, objc, objv, &dataType, &ioLen, &len, &frac,
		      &startPos, &nullValue, &specialNull) != TCL_OK)
    return TCL_ERROR;

  switch (dataType) {
#if ADABAS_VERSION >= 62
  case DFIXED:
  case DSMALLINT:
  case DINTEGER:
#else /* ADABAS_VERSION <= 61 */
  case CSP_INFO_FIXED:
  case CSP_INFO_SMALLINT:
  case CSP_INFO_INTEGER:
#endif
    return putInt (interp, objv[1], ioLen, len, startPos,
		   nullValue, specialNull);

  /*
   * Datatype UNKNOWN is a special flag, that the data should be
   * regarded as edit form.
   */

  case DUNKNOWN:
    Tcl_ResetResult (interp);
    return putEditForm (interp, objv[1]);

  default:
    AppendResult (interp, " <");
    AppendResult (interp, dataType2String (dataType));
    AppendResult (interp, ">");
    return TCL_OK;
  }
}

/* -------------------------------------------------------------- */

int
AdabasGetCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     Tcl_Obj    *CONST objv[]; 		/* Argument objects. */
{
  data_type   dataType    = DCHA;	/* Data type of value. */
  int         ioLen       = 2;		/* IO-Length of value. */
  int         len         = 1;		/* Extern length of value. */
  int        startPos    = 1;		/* Startposition in data part. */
  int         frac        = 0;		/* Fraction, if value is a number. */
  char       *nullValue   = "";		/* Representation of a NULL value. */
  char       *specialNull = "";		/* Representation of a SPECIAL NULL. */
  Tcl_Obj    *resObj;			/* Object to put result into. */
  Tcl_DString substitute;		/* Encoded value. */
  int         ret;			/* return value. */

  if (checkGetParams (interp, objc, objv, &dataType, &ioLen, &len, &frac,
		      &startPos, &nullValue, &specialNull) != TCL_OK) {
    return TCL_ERROR;
  }

  /*
   * Datatype UNKNOWN is a special flag, that the data should be
   * regarded as query form.
   */

  if (dataType == DUNKNOWN) {
    return getEditForm (interp, objv[1]);
  }

  /*
   * Substitute the escaped characters of the given data and put a fixed
   * width string (with width 'ioLen') into 'substitute'.
   */

  Tcl_DStringInit (&substitute);
  ioLen = encodeSequence (interp, encEscape, "data", &substitute,
			  Tcl_GetStringFromObj (objv[1], (int *) NULL),
			  startPos, ioLen);
#ifdef HAS_TCL_OBJECTS
  resObj = Tcl_GetObjResult (interp);
#else
  resObj = (Tcl_Obj *) NULL;
#endif
  ret = AdabasAppendData (interp, (AdabasInfo *) NULL,  (CursorInfo *) NULL,
			  Tcl_DStringValue (&substitute),
			  resObj, dataType, ioLen, len, frac,
			  nullValue, specialNull, 0);

  /*
   * For clean up free the resources, then return the result.
   */

  Tcl_DStringFree (&substitute);
  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasAppendData --
 *
 *      Appends the given data (in adabas format) to the destination object
 *      (if with Tcl_Objs) or to the interpreter result (if without Tcl_Objs).
 *
 * Results:
 *      The return value is normally TCL_OK. If there is any error detected,
 *      TCL_ERROR will be returned; in this case interp->objResults will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasAppendData (interp, vars, cursorVars, data, destObj, dataType,
		  ioLen, len, frac, nullValue, specialNull, isElement)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Current logon handle (or NULL). */
     CursorInfo *cursorVars;		/* Current cursor handle (or NULL). */
     char       *data;			/* Value to append as string. */
     Tcl_Obj    *destObj;		/* Destination object to append Element,
					 * or Null, if without Tcl_Obj. */
     int         dataType;              /* Data type of value. */
     int         ioLen;			/* IO-Length of value. */
     int         len;			/* Extern length of value. */
     int         frac;			/* Fraction, if value is a number. */
     char       *nullValue;		/* Representation of a NULL value. */
     char       *specialNull;		/* Representation of a SPECIAL NULL. */
     int         isElement;		/* append result as element? */
{
  LongDescInfo   *descInfo;		/* symbolic long desc hash value. */
  long_descriptor longDesc;		/* long descriptor from database. */
  int4            intNumber;		/* converted integer value. */
  double          floatNumber;		/* converted float value. */
  num_error       numError;		/* return code of conversion procs. */
  char           *p;			/* Char pointer for indexing. */
  char            numBuf[40];		/* temp buffer for numeric conversions. */
#ifndef HAS_TCL_OBJECTS
  char            remC;			/* temp store of char, which will be 
					 * overwritten with terminating NUL. */
#endif
  
  /*
   * Check, if the value to put is the NULL value.
   * If so, print the given nullValue (or nothing as default).
   */

  if (*(unsigned char *) data == UNDEF_BYTE) {
#ifdef HAS_TCL_OBJECTS
    Tcl_SetStringObj (destObj, nullValue ? nullValue : "", -1);
#else
    if (isElement) {
      Tcl_AppendElement (interp, nullValue ? nullValue : "");
    } else {
      AppendResult  (interp, nullValue ? nullValue : "");
    }
#endif
  } else if (*(unsigned char *) data == OFLW_BYTE) {
#ifdef HAS_TCL_OBJECTS
    Tcl_SetStringObj (destObj, specialNull ? specialNull : "", -1);
#else
    if (isElement) {
      Tcl_AppendElement (interp, specialNull ? specialNull : "");
    } else {
      AppendResult  (interp, specialNull ? specialNull : "");
    }
#endif
  } else {

    /*
     * Okay, it is a NOT NULL value to put. So now depending on the data type
     * we convert the value to string and return.
     */

    switch (dataType) {

      /*
       * Convert a vdn number without fraction into the 
       * string representation of an integer.
       */

#if ADABAS_VERSION >= 62
    case DFIXED:
    case DSMALLINT:
    case DINTEGER:
#else /* ADABAS_VERSION <= 61 */
    case CSP_INFO_FIXED:
    case CSP_INFO_SMALLINT:
    case CSP_INFO_INTEGER:
#endif
      if (!frac) {
	s40glint (data, 2, len, &intNumber, &numError);
	if (!VDN_OK (numError)) {
	  sprintf (numBuf, "<error %d>", numError);
	}
#ifdef HAS_TCL_OBJECTS
	if (!VDN_OK (numError)) {
	  Tcl_SetStringObj (destObj, numBuf, -1);
	} else {
	  Tcl_SetIntObj (destObj, intNumber);
	}
#else
	if (VDN_OK (numError)) {
	  sprintf (numBuf, "%d", intNumber);
	}
	if (isElement) {
	  Tcl_AppendElement (interp, numBuf);
	} else {
	  AppendResult (interp, numBuf);
	}
#endif
	break;
      } /* else nobreak; */
      
      /*
       * Convert a vdn number into the string representation of a float.
       */

#if ADABAS_VERSION >= 62
    case DFLOAT:
    case DVFLOAT:
#else /* ADABAS_VERSION <= 61 */
    case CSP_INFO_FLOAT:
    case CSP_INFO_EXPRESSION:
#endif
      s40glrel (data, 2, ioLen-1, &floatNumber, &numError);
      if (!VDN_OK (numError)) {
	sprintf (numBuf, "<error %d>", numError);
      }
#ifdef HAS_TCL_OBJECTS
      if (!VDN_OK (numError)) {
	Tcl_SetStringObj (destObj, numBuf, -1);
      } else {
	Tcl_SetDoubleObj (destObj, floatNumber);
      }
#else
      if (VDN_OK (numError)) {
	sprintf (numBuf, "%.*f", frac, floatNumber); 
      }
      if (isElement) {
	Tcl_AppendElement (interp, numBuf);
      } else {
	AppendResult (interp, numBuf);
      }
#endif
      break;
    
      /*
       * Convert a fixed width char array into a right side trimmed string.
       */

#if ADABAS_VERSION >= 62
    case DCHA:
    case DCHE:
    case DCHB:
    case DDATE:
    case DTIME:
    case DTIMESTAMP:
    case DUNICODE:
    case DVARCHARA:
    case DVARCHARE:
    case DVARCHARB:
#else /* ADABAS_VERSION <= 61 */
    case CSP_INFO_CHAR:
    case CSP_INFO_BYTE:
    case CSP_INFO_VARCHAR:
    case CSP_INFO_DATE:
    case CSP_INFO_TIME:
    case CSP_INFO_TIMESTAMP:
#endif

#if ADABAS_VERSION >= 62
      if (*data == 1) {
	if (appendUnicode (interp, destObj, data+1, ioLen-1,
			   isElement) == TCL_ERROR) {
	  return TCL_ERROR;
	}
	break;
      }
#endif

      for (p = data + ioLen; p-1 > data && *(p-1) == ' '; p--);

#ifdef HAS_TCL_OBJECTS
      Tcl_SetStringObj (destObj, data+1, p-data-1);
#else

      /*
       * We found the last char in the string, which isn't a blank, if any.
       * Then put the terminating null behind it, but remember the
       * old content of this cell, since it can be the start of the
       * next data...
       */

      remC = *p;
      *p   = 0;
      if (isElement) {
	Tcl_AppendElement (interp, data+1);
      } else {
	AppendResult (interp, data+1);
      }
      *p   = remC;
#endif
      break;
    
      /*
       * Convert a boolean (0/1) into its string representation.
       */

#if ADABAS_VERSION >= 62
    case DBOOLEAN:  
#else /* ADABAS_VERSION <= 61 */
    case CSP_INFO_BOOLEAN:
#endif

#ifdef HAS_TCL_OBJECTS
      Tcl_SetBooleanObj (destObj, *(data+1));
#else
      if (isElement) {
	Tcl_AppendElement (interp, *(data+1) ? "1" : "0");
      } else {
	AppendResult (interp, *(data+1) ? "1" : "0");
      }
#endif
      break; 

#if ADABAS_VERSION >= 62
    case DSTRA:
    case DSTRB:
    case DSTRE:
    case DSTRUNI:
#else /* ADABAS_VERSION <= 61 */
    case CSP_INFO_C_OLDLONG_CHAR:
    case CSP_INFO_B_OLDLONG_BYTE:
    case CSP_INFO_A_OLDLONG_ASCII_DBYTE:
#endif

      /*
       * The data is a long descriptor.
       */

      if (!vars || !cursorVars) {
	SetResult (interp, "Reading of LONG column only with logon and cursor");
	return TCL_ERROR;
      }

      memcpy (&longDesc, data+1, sizeof (longDesc));
      descInfo = longDescr2Hash (interp, dataType, vars, cursorVars, &longDesc);
#ifdef HAS_TCL_OBJECTS
      Tcl_SetStringObj (destObj, descInfo->longDescName, -1);
      destObj->internalRep.otherValuePtr = (VOID *) descInfo;
      destObj->typePtr = &LongDescObjType;
#else
      if (isElement) {
	Tcl_AppendElement (interp, descInfo->longDescName);
      } else {
	AppendResult (interp, descInfo->longDescName);
      }
#endif
      break;

    default:
      AppendResult (interp, " <");
      AppendResult (interp, dataType2String (dataType));
      AppendResult (interp, ">");
      return TCL_ERROR;
    }
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * checkGetParams --
 *
 *      Inspects the given arguments (in objv) and sets the output parameter
 *      accordingly.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is any error detected,
 *      TCL_ERROR will be returned; in this case interp->objResults will
 *      be set to the error message.
 *      For every correct switch on the command line the corresponding
 *      variable will be set, e.g. if there is a "-ioLen 19" pair in
 *      objc/objv, the variable, where ioLen points to, will get the value 19.
 *      Note, that these variables should be set to their default values,
 *      before calling this procedure.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
checkGetParams (interp, objc, objv, dataType, ioLen, len, frac,
		startPos, nullValue, specialNull)
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     Tcl_Obj    *CONST objv[]; 		/* Argument objects. */
     data_type  *dataType;		/* Data type of value. */
     int        *ioLen;			/* IO-Length of value. */
     int        *len;			/* Extern length of value. */
     int        *frac;			/* Fraction, if value is a number. */
     int        *startPos;		/* Startposition in data part. */
     char      **nullValue;		/* Representation of a NULL value. */
     char      **specialNull;		/* Representation of a SPECIAL NULL. */
{
  int          currOption;		/* Index of current option. */
  Tcl_DString  dStr;			/* Help string for datatype parsing. */
  char        *typeStr = "";		/* string value of dataType object. */
  Tcl_RegExp   dataTypeRegexp;		/* compiled regexp for dataType */
  char        *start;			/* Index into cmd returned by regexp */
  char        *end;			/* Index into cmd returned by regexp */
  int          newVal;			/* help integer for type compatibility. */
  static char *options[] =		/* options of shortinfo description. */
  {"-ioLen", "-len", "-frac", "-bufPos", "-mode",
   "-nullValue", "-specialNull", "-ioType", (char *) NULL};

  if (objc < 2) {
    AppendResult (interp, "wrong # args: should be \"adabas ");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, " string ?data_type? ?option ...?\"");
    return TCL_ERROR;
  }

  if (objc >= 3) {
    typeStr = Tcl_GetStringFromObj (objv[2], (int *) NULL);
  }
  if (*typeStr != '-') {

    /*
     * First compile the regular expression for 'datatype(len,frac?)'.
     * Since tcl maintains a small cache of compiled regexp, there are
     * good chances, that no new compilation is neccassary...
     */

    dataTypeRegexp = Tcl_RegExpCompile (interp, DATA_TYPE_REGEXP_STRING);
    if (dataTypeRegexp == (Tcl_RegExp) NULL) {
      return TCL_ERROR;
    }

    switch (Tcl_RegExpExec (interp, dataTypeRegexp, typeStr, typeStr)) {
    case -1:
      return TCL_ERROR;

    case 1:
      Tcl_DStringInit (&dStr);

      Tcl_RegExpRange (dataTypeRegexp, DATA_TYPE_RANGE, &start, &end);
      Tcl_DStringAppend (&dStr, start, end-start);
      *dataType = scanDataType (interp, Tcl_DStringValue (&dStr));
      Tcl_DStringFree (&dStr);

      Tcl_RegExpRange (dataTypeRegexp, LENGTH_RANGE, &start, &end);
      Tcl_DStringAppend (&dStr, start, end-start);
      if (Tcl_GetInt (interp, Tcl_DStringValue (&dStr), &newVal) == TCL_ERROR) {
	return TCL_ERROR;
      }
      *len = newVal;
      Tcl_DStringFree (&dStr);

      Tcl_RegExpRange (dataTypeRegexp, FRAC_RANGE, &start, &end);
      if (start) {
	Tcl_DStringAppend (&dStr, start, end-start);
	if (Tcl_GetInt (interp, Tcl_DStringValue (&dStr), &newVal) == TCL_ERROR) {
	  return TCL_ERROR;
	}
	*frac = newVal;
	Tcl_DStringFree (&dStr);
      }
      break;

    case 0:
      *dataType = scanDataType (interp, typeStr);

      if (*dataType == DUNKNOWN) {
	if (!strncmp (typeStr, "editform", strlen (typeStr))) {
	  Tcl_ResetResult (interp);
	  if (objc > 3) {
	    AppendResult (interp, "wrong # args: should be \"adabas ");
	    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
	    AppendResult (interp, " string editform\"");
	    return TCL_ERROR;
	  }
	  return TCL_OK;
	}
	return TCL_ERROR;
      }
      break;
    }

    objc--; objv++;
  }

  /*
   * Now parse the trailing options, that resembles the shortinfo of the
   * transformation.
   */

  for (objv += 2, objc -= 2; objc > 1; objv += 2, objc -= 2) {
    if (Tcl_GetIndexFromObj (interp, objv[0], options,
			     "option", 0, &currOption) == TCL_ERROR) {
      return TCL_ERROR;
    }
    switch (currOption) {
    case 0: /* -ioLen */
      if (Tcl_GetIntFromObj (interp, objv[1], ioLen) == TCL_ERROR) {
	return TCL_ERROR;
      }
      break;
    case 1: /* -len */
      if (Tcl_GetIntFromObj (interp, objv[1], len) == TCL_ERROR) {
	return TCL_ERROR;
      }
      break;
    case 2: /* -frac */
      if (Tcl_GetIntFromObj (interp, objv[1], frac) == TCL_ERROR) {
	return TCL_ERROR;
      }
      break;
    case 3: /* -bufPos */
      if (Tcl_GetIntFromObj (interp, objv[1], startPos) == TCL_ERROR) {
	return TCL_ERROR;
      }
      break;
    case 4: /* -mode */
      break;
    case 5: /* -nullValue */
      *nullValue = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;
    case 6: /* -specialNull */
      *specialNull = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;
    case 7: /* -ioType */
      break;
    }
  }
  if (objc == 1) {
    AppendResult (interp, "value wanted for option \"");
    AppendResult (interp,  Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }

  /*
   * Some datatypes have default ioLens. So if they are not given
   * explicitly, here we can assign them.
   */

  if (*ioLen < 0) {
    *ioLen = getIoLen (*dataType, *len);
  }
  return TCL_OK;
}

/* -------------------------------------------------------------- */

static int
putInt (interp, data, ioLen, len, startPos, nullValue, specialNull)
     Tcl_Interp *interp;		/* Current interpreter. */
     Tcl_Obj    *data;			/* Object to append as string. */
     int         ioLen;			/* IO-Length of value. */
     int         len;			/* Extern length of value. */
     int         startPos;		/* Startposition in data part. */
     char       *nullValue;		/* Representation of a NULL value. */
     char       *specialNull;		/* Representation of a SPECIAL NULL. */
{
  Tcl_DString result;
  num_error   num_error;
  int         intNumber;
  char        vdnNumber[12];
  char        msg[80];

  if (Tcl_GetIntFromObj (interp, data, &intNumber) == TCL_ERROR)
    return TCL_ERROR;

  s41plint (vdnNumber, 1, len, 0, intNumber, &num_error);
  if (num_error) {
    sprintf (msg, "putInt conversion error %d", num_error);
    SetResult (interp, msg);
  } else {
    Tcl_DStringInit (&result);
    decodeSequence (interp, encEscape, (Tcl_Obj *) NULL, &result, "", 1);
    decodeSequence (interp, encEscape, (Tcl_Obj *) NULL,
		    &result, vdnNumber, ioLen-1);
    Tcl_DStringResult (interp, &result);
  }
  return TCL_OK;
}

/* -------------------------------------------------------------- */

static int
getIoLen (dataType, length)
     int dataType;			/* Data type to get IO length of. */
     int length;			/* Extern length of value. */
{
  switch (dataType) {

  case DSMALLINT:
    return 5;

  case DFIXED:
  case DINTEGER:
  case DFLOAT:
  case DVFLOAT:
    return 11;

  case DSTRA:
  case DSTRE:
  case DSTRUNI:
    return 41;

  case DBOOLEAN:
    return 2;

  case DDATE:
    return 11;

  case DTIME:
    return 11;

  case DTIMESTAMP:
    return 21;

  case DUNICODE:
    return 2*length+1;

  case DCHA:
  case DCHB:
  case DCHE:
  case DVARCHARA:
  case DVARCHARB:
  case DVARCHARE:
  default:
    return length+1;
  }
}

/* -------------------------------------------------------------- */

static LongDescInfo *
longDescr2Hash (interp, dataType, vars, cursorVars, longDesc)
     Tcl_Interp      *interp;		/* Current interpreter. */
     int              dataType;		/* of long column (ascii, unicode,...) */
     AdabasInfo      *vars;		/* Current logon handle. */
     CursorInfo      *cursorVars;	/* Current cursor handle. */
     long_descriptor *longDesc;		/* Long descriptor to convert. */
{
  LongDescInfo  *newInfo;		/* Freshly created long desc info. */
  Tcl_HashEntry *newEntry;		/* Hash entry with long desc. */
  int            added;			/* Hash entry successfully added? */

  if (!(newInfo = ((LongDescInfo *) ckalloc (sizeof (LongDescInfo))))) {
    SetResult (interp, "alloc of long desc failed???");
    return (LongDescInfo *) NULL;
  }

  sprintf (newInfo->longDescName, "longdesc%d", ++longDescCount);
  newInfo->longDesc  = *longDesc;
  newInfo->dataType  =  dataType;
  newInfo->logon     =  vars;
  newInfo->cursor    =  cursorVars;
  newInfo->longMagic =  LONG_MAGIC;

  newEntry = Tcl_CreateHashEntry (&longDescHash, newInfo->longDescName,
				  &added);
  if (!added) {
    SetResult (interp, "long desc already in hash table???");
    return (LongDescInfo *) NULL;
  }
  Tcl_SetHashValue (newEntry, newInfo);
  return newInfo;
}

/*
 *----------------------------------------------------------------------
 *
 * hash2LongInfo --
 *
 *      Looks up the given long descriptor info in the long hash table.
 *
 * Results:
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
hash2LongInfo (interp, inDescName, longDescInfo)
     Tcl_Interp    *interp;		/* Current interpreter. */
     char          *inDescName;		/* char representation of long desc. */
     LongDescInfo **longDescInfo;	/* Returned long desc info. */
{
  Tcl_HashEntry *hashEntry;		/* Entry with long desc info. */
  LongDescInfo  *foundInfo;		/* Found long desc info. */

  if (!(hashEntry = Tcl_FindHashEntry (&longDescHash, inDescName))) {
    if (interp) {
      AppendResult (interp, "No valid long desc handle: ");
      AppendResult (interp, inDescName);
    }
    return TCL_ERROR;
  }
  foundInfo = (LongDescInfo *) Tcl_GetHashValue (hashEntry);
  if (foundInfo->longMagic != LONG_MAGIC) {
    if (interp) {
      AppendResult (interp, "No valid long desc handle: ");
      AppendResult (interp, inDescName);
    }
    return TCL_ERROR;
  }
  if (foundInfo->longMagic != LONG_MAGIC) {
    if (interp) {
      AppendResult (interp, "No valid long desc handle: ");
      AppendResult (interp, inDescName);
    }
    return TCL_ERROR;
  }
  *longDescInfo = foundInfo;
  return TCL_OK;
}

/* -------------------------------------------------------------- */
#if ADABAS_VERSION >= 62

static int
appendUnicode (interp, destObj, data, length, isElement)
     Tcl_Interp *interp;		/* Current interpreter. */
     Tcl_Obj    *destObj;		/* Destination object to append Element,
					 * or Null, if dest is interp->result. */
     char       *data;			/* value to append (w/o defbyte). */
     int         length;		/* length of value. */
     int         isElement;		/* append result as element? */
{
  Tcl_DString uniBuf;
  int4        destLen = length;
  char       *p;
  char       *val;
#if ADABAS_VERSION >= 62
  int         trunc;
#endif
  Tcl_DStringInit (&uniBuf);
  Tcl_DStringSetLength (&uniBuf, length);

#if ADABAS_VERSION >= 62
  if (AdabasFromToUnicode (interp, data, length, Tcl_DStringValue (&uniBuf),
			   &destLen, &trunc, FROM_UNICODE) == TCL_ERROR) {
    return TCL_ERROR;
  }
#endif

  val = Tcl_DStringValue (&uniBuf);
  for (p = val + destLen; p-1 > val && *(p-1) == ' '; p--);
  Tcl_DStringSetLength (&uniBuf, p-val);

#ifdef HAS_TCL_OBJECTS
  Tcl_SetStringObj (destObj, Tcl_DStringValue (&uniBuf),
		    Tcl_DStringLength (&uniBuf));
#else
  if (isElement) {
    Tcl_AppendElement (interp, Tcl_DStringValue (&uniBuf));
  } else {
    AppendResult (interp, Tcl_DStringValue (&uniBuf));
  }
#endif

  Tcl_DStringFree (&uniBuf);
  return TCL_OK;
}
#endif /* ADABAS_VERSION >= 62 */

/*
 *----------------------------------------------------------------------
 *
 * getEditForm --
 *
 *      This procedure transforms a so called editform (the internal format
 *      for storing queries in a LONG BYTE column) into a valid tcl list
 *      (representing e.g. some lines of a text widget).
 *      The internal format is:
 *      - one byte: the length of the line (without surrounding blanks);
 *      - one byte: the number of leading blanks;
 *      - the trimmed line.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is any error detected,
 *      TCL_ERROR will be returned; in this case interp->objResults will
 *      be set to the error message. Else the result will be set to the
 *      list of transformed lines.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
getEditForm (interp, formObj)
     Tcl_Interp *interp;		/* Current interpreter. */
     Tcl_Obj    *formObj;		/* object with internal format. */
{
  char       *formStr;			/* String value of formObj. */
  int         length;			/* String length of formStr. */
  Tcl_DString substitute;		/* Encoded value of formStr. */
  int         ix;			/* Index over substitute. */
  int         ioLen;			/* Length of substitute. */
  int         lineLength;		/* Length of line (trimmed). */
  int         cntBlanks;		/* Number of leading blanks on a line. */
  char        blankBuf[256];		/* Array of blanks. */
  char       *str;			/* Start of current row. */
#ifdef HAS_TCL_OBJECTS
  Tcl_Obj    *listObj;			/* Resulting list element to append. */
#else
  Tcl_DString editForm;			/* String for constructing result. */
#endif

  formStr = Tcl_GetStringFromObj (formObj, &length);
  if ((ioLen = encodeSequence (interp, encEscape, "editForm", &substitute,
			       formStr, 1, length)) < 0) {
    return TCL_ERROR;
  }

  memset (blankBuf, ' ', sizeof (blankBuf));
#ifndef HAS_TCL_OBJECTS
  Tcl_DStringInit (&editForm);
#endif

  for (ix = 0; ix < ioLen; ix += lineLength) {
    str        = Tcl_DStringValue (&substitute) + ix;
    lineLength = * (unsigned char *) str++;
    cntBlanks  = * (unsigned char *) str++;

    if (ix + lineLength > ioLen) {
      sprintf (blankBuf, "Invalid query form at position %d", ix);
      SetResult (interp, blankBuf);
#ifndef HAS_TCL_OBJECTS
      Tcl_DStringFree (&editForm);
#endif
      Tcl_DStringFree (&substitute);
      return TCL_ERROR;
    }

#ifdef HAS_TCL_OBJECTS
    listObj = Tcl_NewStringObj (blankBuf, cntBlanks);
    Tcl_AppendToObj (listObj, str, lineLength - 2);
    Tcl_ListObjAppendElement (interp, Tcl_GetObjResult (interp), listObj);
#else
    Tcl_DStringStartSublist (&editForm);
    if (cntBlanks) {
      Tcl_DStringAppend (&editForm, blankBuf, cntBlanks);
    }
    Tcl_DStringAppend (&editForm, str, lineLength - 2);
    Tcl_DStringEndSublist (&editForm);
#endif
  }

#ifndef HAS_TCL_OBJECTS
  Tcl_DStringResult (interp, &editForm);
#endif

  Tcl_DStringFree (&substitute);
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * putEditForm --
 *
 *      This procedure transforms a valid tcl list (containing e.g. some
 *      lines from a text widget) into a so called editform, the internal
 *      format for storing queries into a LONG BYTE column.
 *      For a description of the internal format see getEditForm above.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is any error detected,
 *      TCL_ERROR will be returned; in this case interp->objResults will
 *      be set to the error message. Else the result will be set to the
 *      decoded string of the internal format.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
putEditForm (interp, queryList)
     Tcl_Interp *interp;		/* Current interpreter. */
     Tcl_Obj    *queryList;		/* list object with external format. */
{
  int           lineLength;		/* Length of line (trimmed). */
  int           cntBlanks;		/* Number of leading blanks on a line. */
  Tcl_DString   editForm;		/* resulting string. */
  int           objc;			/* Number of list elements in queryList. */
  Tcl_Obj     **objv;			/* Elements of queryList. */
  Tcl_Obj     **curr;			/* Current element of objv. */
  char         *str;			/* String value of curr. */
  unsigned char info[2];		/* Two byte prefix of every line. */
  int           ret;			/* return value. */

  if (Tcl_ListObjGetElements (interp, queryList, &objc, &objv) == TCL_ERROR) {
    return TCL_ERROR;
  }
  Tcl_DStringInit (&editForm);
  for (curr = objv; objc; curr++, objc--) {
    str = Tcl_GetStringFromObj (*curr, (int *) NULL);
    for (cntBlanks = 0; *str == ' '; str++, cntBlanks++);
    lineLength = strlen (str);
    while (lineLength && str[lineLength-1] == ' ') {
      lineLength--;
    }
    if (cntBlanks > 255 || lineLength > 255) {
      SetResult (interp, "Too long line in query form");
      ret = TCL_ERROR;
      goto exit;
    }
    info[0] = (unsigned char) lineLength + 2;
    info[1] = (unsigned char) cntBlanks;
    decodeSequence (interp, encEscape, (Tcl_Obj *) NULL, &editForm,
		    (char *) info, sizeof (info));
    Tcl_DStringAppend (&editForm, str, lineLength);
  }

  AppendResult (interp, Tcl_DStringValue (&editForm));
  ret = TCL_OK;

exit:
#ifndef HAS_TCL_OBJECTS
  ckfree ((char *) objv);
#endif
  Tcl_DStringFree (&editForm);
  return ret;
}

int
AdabasGetLongDescFromObj (interp, longDescObj, outLongInfo)
     Tcl_Interp    *interp;		/* Used for error reporting if not NULL. */
     Tcl_Obj       *longDescObj;	/* Object to get long desc info for. */
     LongDescInfo **outLongInfo;	/* Returned long desc info. */
{

#ifdef HAS_TCL_OBJECTS
  if (Tcl_ConvertToType (interp, longDescObj, &LongDescObjType) == TCL_ERROR) {
    return TCL_ERROR;
  } else {
    *outLongInfo = (LongDescInfo *) longDescObj->internalRep.otherValuePtr;
    if ((*outLongInfo)->longMagic == LONG_MAGIC) {
      return TCL_OK;
    }
  }
#endif
  return hash2LongInfo (interp,
			Tcl_GetStringFromObj (longDescObj, (int *) NULL),
			outLongInfo);
}

void
AdabasDeleteLongDescr (ofThisLogon, exceptForThisCursor)
     AdabasInfo *ofThisLogon;		/* Logon handle, which long descr
					 * must be deleted. */
     CursorInfo *exceptForThisCursor;	/* Cursor handle, which long descr
					 * must not be deleted (or NULL). */
{
  Tcl_HashSearch hashSearch;		/* Keeps track of descr scan. */
  Tcl_HashEntry *hashEntry;		/* Current hash entry. */
  LongDescInfo  *foundInfo;		/* Found long desc info. */

  if (!(hashEntry = Tcl_FirstHashEntry (&longDescHash, &hashSearch))) {
    longDescCount = 0;
  } else {
    while (hashEntry) {
      foundInfo = (LongDescInfo *) Tcl_GetHashValue (hashEntry);
      if (foundInfo->logon == ofThisLogon &&
	  (!exceptForThisCursor || exceptForThisCursor != foundInfo->cursor)) {
	AdabasTrace ("longdesc", foundInfo->longDescName, "To delete");
	foundInfo->longMagic = 0;
	ckfree ((char *) foundInfo);
	Tcl_DeleteHashEntry (hashEntry);
	hashEntry = Tcl_FirstHashEntry (&longDescHash, &hashSearch);
      } else {
	hashEntry = Tcl_NextHashEntry (&hashSearch);
      }
    }
  }
}

#ifdef HAS_TCL_OBJECTS

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ObjType procedures for longDescType --
 *
 *      The long descriptor parameter will become its own Tcl_ObjType,
 *      so that most (if not all) of the hash table lookups can be omitted.
 *
 *----------------------------------------------------------------------
 */

static int
longDescSetFromAny (interp, objPtr)
     Tcl_Interp *interp;		/* Used for error reporting if not NULL. */
     Tcl_Obj    *objPtr;		/* The object to convert. */
{
  char         *longDescName;		/* Name of long descr Handle. */
  LongDescInfo *longDescInfo;		/* descriptor identified by objPtr. */

  longDescName = Tcl_GetStringFromObj (objPtr, (int *) NULL);
  if (hash2LongInfo (interp, longDescName, &longDescInfo) == TCL_ERROR) {
    AdabasTrace ("longdesc", longDescName, "Invalid");
    return TCL_ERROR;
  }
  objPtr->internalRep.otherValuePtr = (VOID *) longDescInfo;
  objPtr->typePtr = &LongDescObjType;
  AdabasTrace ("longdesc", longDescInfo->longDescName, "Set string from");
  return TCL_OK;
}

static void
longDescUpdateString (objPtr)
     Tcl_Obj *objPtr;			/* Object whose string rep to update. */
{
  LongDescInfo *longDescInfo;		/* descriptor identified by objPtr. */

  longDescInfo   = (LongDescInfo *) objPtr->internalRep.otherValuePtr;
  objPtr->length = strlen (longDescInfo->longDescName);
  objPtr->bytes  = ckalloc((unsigned) objPtr->length + 1);
  strcpy (objPtr->bytes, longDescInfo->longDescName);
  AdabasTrace ("longdesc", longDescInfo->longDescName, "Update string of");
}

static void
longDescDupInternalRep (srcPtr, dupPtr)
     Tcl_Obj *srcPtr;			/* Object with internal rep to copy. */
     Tcl_Obj *dupPtr;			/* Object with internal rep to set. */
{
    dupPtr->internalRep.otherValuePtr = srcPtr->internalRep.otherValuePtr;
    dupPtr->typePtr = &LongDescObjType;
}

#endif
