/*
 * AdabasLong.c --
 *
 *      This module contains procedures, that implement the long column
 *      handling in AdabasTcl.
 *
 * Copyright (c) 1996-1997 Christian Krone. All rights reserved.
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Tcl itself.
 * See also licence.terms
 *
 *  $Id: AdabasLong.c,v 1.24 1997/06/14 16:13:58 adabas Exp $
 */

/*
 * Import interface (should move into AdabasTclInt.h someday)
 */

#include <string.h>
#include <fcntl.h>
#include <tcl.h>

#include "adabas.h"
#include "AdabasPort.h"
#include "AdabasUtil.h"
#include "AdabasRte.h"
#include "AdabasPacket.h"
#include "AdabasSend.h"
#include "AdabasApi.h"
#include "AdabasLong.h"

/*
 * Local constants
 */

#if ADABAS_VERSION >= 62
#  define LONG_VALPOS_FIRST  (1+sizeof(long_descriptor)+1)
#  define LONG_VALPOS_NEXT   (1+sizeof(long_descriptor)+1)
#  define LONG_HEADER        200
#  define LONG_SIZE_READ     MAXINT4
#  define LONG_SIZE_FIRST(i) ((i)->rteInfo.sqlPacketSize \
			      - LONG_HEADER - LONG_VALPOS_FIRST)
#  define LONG_SIZE_NEXT(i)  ((i)->rteInfo.sqlPacketSize \
			      - LONG_HEADER - LONG_VALPOS_NEXT)
#else /* ADABAS_VERSION <= 61 */
#  define LONG_VALPOS_FIRST  (1+sizeof(long_descriptor)+11)
#  define LONG_VALPOS_NEXT   (1+sizeof(long_descriptor)+13)
#  define LONG_SIZE_READ     MAXINT2
#  define LONG_SIZE_FIRST(i) (VARPART - LONG_VALPOS_FIRST - 1)
#  define LONG_SIZE_NEXT(i)  (VARPART - LONG_VALPOS_NEXT)
#endif 

/*
 * What kind of long operation we are doing?
 */

typedef enum {
  longSelect,
  longWriteFirst,
  longWriteNext,
  longReadNext
} LongAction;

/*
 * Prototypes for procedures referenced only in this file.
 */

static long fileLength _ANSI_ARGS_((Tcl_Interp *interp, char *fileName));
static int  readFromFile _ANSI_ARGS_((Tcl_Interp *interp, char *fileName,
	        char *data, long offset, long length));
static int  getLongDescriptor _ANSI_ARGS_((Tcl_Interp *interp, AdabasInfo *vars,
		long_descriptor *longDesc, int isError, LongAction action));

/* ----------------- Exported procedures ------------------------- */

int AdabasLongSelect (interp, cursorVars, tableName, columnName,
		      whereCond, longDesc)
     Tcl_Interp      *interp;		/* Current interpreter. */
     CursorInfo      *cursorVars;	/* Current cursor. */
     char            *tableName;	/* Name of table with long column. */
     char            *columnName;	/* Name of long column in table. */
     char            *whereCond;	/* Where condition (or NULL). */
     long_descriptor *longDesc;		/* Descriptor returned by a fetch. */
{
  Tcl_DString selCommand;
  int         ret;
  char        buffer[100];

  Tcl_DStringInit   (&selCommand);
  Tcl_DStringAppend (&selCommand, "SELECT ",  -1);
  Tcl_DStringAppend (&selCommand, columnName, -1);
  Tcl_DStringAppend (&selCommand, " FROM ",   -1);
  Tcl_DStringAppend (&selCommand, tableName,  -1);
  if (whereCond) {
    Tcl_DStringAppend (&selCommand, " WHERE ",  -1);
    Tcl_DStringAppend (&selCommand, whereCond,  -1);
  }

  ret = AdabasSendCmdPacket (interp, cursorVars->logonInfo, 1, 0,
			     Tcl_DStringValue (&selCommand));
  Tcl_DStringFree (&selCommand);
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

#if ADABAS_VERSION >= 62
  if (!getResultTable (&cursorVars->logonInfo->rcvPacket->segm,
		       cursorVars->resultTable)) {
    *cursorVars->resultTable = 0;
  }
#else /* ADABAS_VERSION <= 61 */
  if (!getResultTable (&cursorVars->logonInfo->sqlPacket, M_DBS,
		       cursorVars->resultTable)) {
    *cursorVars->resultTable = 0;
  }
#endif

  sprintf (buffer, "FETCH %s INTO ?", cursorVars->resultTable);
  if (AdabasSendCmdPacket (interp, cursorVars->logonInfo,
			   1, 0, buffer) == TCL_ERROR) {
    return TCL_ERROR;
  }
#if ADABAS_VERSION >= 62
  cursorVars->cntParams =
    getShortInfos (&cursorVars->logonInfo->rcvPacket->segm,
		   &cursorVars->shortInfos, &cursorVars->columnNames);
#else /* ADABAS_VERSION <= 61 */
  cursorVars->cntParams =
    getShortInfos (&cursorVars->logonInfo->sqlPacket,
		   &cursorVars->shortInfos, &cursorVars->columnNames);
#endif
  ret = cursorVars->cntParams == 1
    && datatypeIsLong (cursorVars->shortInfos->data_type);

  if (!ret) {
    SetResult (interp, "invalid select parameter for adareadlong");
    return TCL_ERROR;
  }

  return getLongDescriptor (interp, cursorVars->logonInfo, 
			    longDesc, 1, longSelect);
}

/* -------------------------------------------------------------- */

int AdabasLongReadFirst (interp, cursorVars, longDesc, dataType,
			 encoding, fileName, bytes, cont)
     Tcl_Interp      *interp;		/* Current interpreter. */
     CursorInfo      *cursorVars;	/* Current cursor. */
     long_descriptor *longDesc;		/* Descriptor returned by select. */
     int              encoding;		/* Long encoding, if read into string. */
     char            *fileName;		/* Filename to write long into. */
     long            *bytes;		/* Bytes read in this round. */
     int             *cont;		/* Is there more data to read? */
{
  char       *data;
  int         length;
  Tcl_DString longVal;
#if ADABAS_VERSION >= 62
  int         trunc;
#endif

  if (!(data = getDataOfCursor (interp, cursorVars, FOR_LONG_OP))) {
    return TCL_ERROR;
  }

  data  += longDesc->valpos - 1;
  length = longDesc->vallen;

#if ADABAS_VERSION >= 62
  if (dataType == DSTRUNI) {
    if (AdabasFromToUnicode (interp, data, length, data,
			     &length, &trunc, FROM_UNICODE) == TCL_ERROR){
      return TCL_ERROR;
    }
  }
#endif

  if (fileName) {
    if (AdabasLongToFile (interp, 1, fileName, length, data) == TCL_ERROR) {
      return TCL_ERROR;
    }
  } else {
#ifdef HAS_TCL_OBJECTS
    decodeSequence (interp, encoding, Tcl_GetObjResult (interp),
		    &longVal, data, length);
#else
    Tcl_DStringInit (&longVal);
    decodeSequence (interp, encoding, (Tcl_Obj *) NULL,
		    &longVal, data, length);
    Tcl_DStringResult (interp, &longVal);
#endif
  }

  *bytes = length;
  *cont  = longDesc->valmode == VM_DATAPART;

  return TCL_OK;
}

/* -------------------------------------------------------------- */

int AdabasLongReadNext (interp, cursorVars, longDesc, dataType,
			encoding, fileName, bytes, cont)
     Tcl_Interp      *interp;		/* Current interpreter. */
     CursorInfo      *cursorVars;	/* Current cursor. */
     long_descriptor *longDesc;		/* Descriptor returned by select. */
     int              dataType;		/* Data type of long column. */
     int              encoding;		/* Long encoding, if read into string. */
     char            *fileName;		/* Filename to write long into. */
     long            *bytes;		/* Bytes read in this round. */
     int             *cont;		/* Is there more data to read? */
{
  char            *data;
  int              length;
  Tcl_DString      longVal;
  int              partI;
  char             descBuf[100];
  AdabasInfo      *vars        = cursorVars->logonInfo;
  AdabasPacketInfo thisOptions = vars->packetInfo;
#if ADABAS_VERSION <= 61
  int2 argCnt = 1;
#endif

  longDesc->vallen = LONG_SIZE_READ;

  /*
   * Send a GETVAL order to the database server to get
   * the next portion of this long column.
   */
  for (partI = 0; partI < PART_KINDS; partI++) {
    vars->partInfos[ partI ].bufLength = 0;
  }
  
#if ADABAS_VERSION >= 62
  descBuf[0] = 0; /* defined byte */
  memcpy (descBuf+1, (char *) longDesc, 40);
  vars->partInfos[ pkLongdata ].buf       = descBuf;
  vars->partInfos[ pkLongdata ].bufLength = 41;
#else /* ADABAS_VERSION <= 61 */
  memcpy (descBuf,   &argCnt,  2);
  descBuf[2] = 0; /* defined byte */
  memcpy (descBuf+3, longDesc, 32);
  vars->partInfos[ pkLongdata ].buf       = descBuf;
  vars->partInfos[ pkLongdata ].bufLength = 35;
#endif

  thisOptions.messType = M_GETVAL;
  if (AdabasFillPacket (interp, vars, &thisOptions) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (AdabasSendPacket (interp, vars) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (getLongDescriptor (interp, vars, longDesc, 1, longReadNext) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (!(data = getDataOfCursor (interp, cursorVars, FOR_LONG_OP))) {
    return TCL_ERROR;
  }

  data  += longDesc->valpos - 1;
  length = longDesc->vallen;

  if (fileName) {
    if (AdabasLongToFile (interp, 2, fileName, length, data) == TCL_ERROR) {
      return TCL_ERROR;
    }
  } else {
#ifdef HAS_TCL_OBJECTS
    if (decodeSequenceAppend (interp, encoding, Tcl_GetObjResult (interp),
			      &longVal, data, *bytes, length) == TCL_ERROR) {
      return TCL_ERROR;
    }
#else
    Tcl_DStringInit (&longVal);
    if (decodeSequenceAppend (interp, encoding, (Tcl_Obj *) NULL,
			      &longVal, data, *bytes, length) == TCL_ERROR) {
      return TCL_ERROR;
    }
    AppendResult (interp, Tcl_DStringValue (&longVal));
    Tcl_DStringFree (&longVal);
#endif
  }

  *bytes += length;
  *cont   = longDesc->valmode == VM_DATAPART;

  return TCL_OK;
}

/* -------------------------------------------------------------- */

int AdabasLongUpdate (interp, cursorVars, tableName, columnName,
		      whereCond, parsId)
     Tcl_Interp  *interp;		/* Current interpreter. */
     CursorInfo  *cursorVars;		/* Current cursor. */
     char        *tableName;		/* Name of table with long column. */
     char        *columnName;		/* Name of long column in table. */
     char        *whereCond;		/* Where condition (or NULL). */
     Tcl_DString *parsId;		/* ParsId returned by this update. */
{
  AdabasInfo *vars = cursorVars->logonInfo;
  Tcl_DString updCommand;
  int         oldMessType;
  int         ret;

  /*
   *  Construct the 'UPDATE OF table SET column = :x WHERE condition' command
   *  and let it parse by the database kernel.
   */

  Tcl_DStringInit   (&updCommand);
  Tcl_DStringAppend (&updCommand, "UPDATE OF ", -1);
  Tcl_DStringAppend (&updCommand, tableName,    -1);
  Tcl_DStringAppend (&updCommand, " SET ",      -1);
  Tcl_DStringAppend (&updCommand, columnName,   -1);
  Tcl_DStringAppend (&updCommand, " = :x",      -1);
  if (whereCond) {
    Tcl_DStringAppend (&updCommand, " WHERE ", -1);
    Tcl_DStringAppend (&updCommand, whereCond, -1);
  }

  oldMessType               = vars->packetInfo.messType;
  vars->packetInfo.messType = M_PARSE;
  ret = AdabasSendCmdPacket (interp, vars, 1, 0,
			     Tcl_DStringValue (&updCommand));
  vars->packetInfo.messType = oldMessType;
  Tcl_DStringFree (&updCommand);
  if (ret == TCL_ERROR) {
    return TCL_ERROR;
  }

  Tcl_DStringInit (parsId);
#if ADABAS_VERSION >= 62
  cursorVars->cntParams =
    getShortInfos (&vars->rcvPacket->segm,
		   &cursorVars->shortInfos, &cursorVars->columnNames);
  ret = cursorVars->cntParams == 1
    && datatypeIsLong (cursorVars->shortInfos->data_type)
      && getParsId (interp, &vars->rcvPacket->segm, parsId);
#else /* ADABAS_VERSION <= 61 */
  cursorVars->cntParams =
    getShortInfos (&vars->sqlPacket,
		   &cursorVars->shortInfos, &cursorVars->columnNames);
  ret = cursorVars->cntParams == 1
    && datatypeIsLong (cursorVars->shortInfos->data_type)
      && getParsId (interp, &vars->sqlPacket, parsId);
#endif
  if (!ret) {
    SetResult (interp, "invalid update parameter for adawritelong");
    return TCL_ERROR;
  }

  return TCL_OK;
}

/* -------------------------------------------------------------- */

int AdabasLongWriteFirst (interp, cursorVars, longDesc, parsId, fileName,
			  longValue, encoding, length, bytes, cont)
     Tcl_Interp      *interp;		/* Current interpreter. */
     CursorInfo      *cursorVars;	/* Current cursor. */
     long_descriptor *longDesc;		/* Descriptor returned by update. */
     Tcl_DString     *parsId;		/* ParsId of previous update. */
     char            *fileName;		/* Filename to read long out of. */
     char            *longValue;	/* directly given long to write;
					 * note, that this is already encoded. */
     int              encoding;		/* Long encoding, if write into string. */
     int              length;		/* length of data in external encoding,
					   amount still to write. */
     int4            *bytes;		/* Bytes written in this round. */
     int             *cont;		/* Is there more data to write? */
{
  int              partI;
  char            *data;
  char            *longData;
  AdabasInfo      *vars        = cursorVars->logonInfo;
  AdabasPacketInfo thisOptions = vars->packetInfo;
  field_info      *longSi      = cursorVars->shortInfos;
#if ADABAS_VERSION >= 62
  int4             dc1, dc2;
  int              trunc;
#endif

  /*
   * Now write the parsid for execution into the packet, followed
   * by the first portion of the long value.
   */

  for (partI = 0; partI < PART_KINDS; partI++) {
    vars->partInfos[ partI ].bufLength = 0;
  }
  vars->partInfos[ pkParsid ].buf       = Tcl_DStringValue (parsId);
  vars->partInfos[ pkParsid ].bufLength = Tcl_DStringLength (parsId);
  vars->partInfos[ pkData   ].buf       = NULL; /* we fill up later... */
  vars->partInfos[ pkData   ].bufLength =
    longSi->bufpos + sizeof (long_descriptor)
    + (length > LONG_SIZE_FIRST (vars) ? LONG_SIZE_FIRST (vars) : length);
  thisOptions.messType = M_EXECUTE;
  if (AdabasFillPacket (interp, vars, &thisOptions) == TCL_ERROR) {
    return TCL_ERROR;
  }

#if ADABAS_VERSION >= 62
  data  = getData (&vars->sndPacket->segm, &dc1, &dc2, 0);
#else /* ADABAS_VERSION <= 61*/
  data  = vars->sqlPacket.var_part+vars->sqlPacket.part1_length;
#endif
  if (!data) {
    SetResult (interp, "No data part???");
    return TCL_ERROR;
  }

  longData  = data + 1 + sizeof (long_descriptor);
#if ADABAS_VERSION >= 62
  longData += longSi->bufpos - 1;
#endif

  if ((*cont = (length > LONG_SIZE_FIRST (vars)))) {
    length = LONG_SIZE_FIRST (vars);
  }
  *bytes = length;

  if (fileName) {
    if (readFromFile (interp, fileName, longData, 0, length) == TCL_ERROR) {
      return TCL_ERROR;
    }
  } else {
    memcpy (longData, longValue, length);
  }

#if ADABAS_VERSION >= 62
  if (longSi->data_type == DSTRUNI) {
    *bytes = LONG_SIZE_FIRST (vars);
    if (AdabasFromToUnicode (interp, longData, length, longData,
			     bytes, &trunc, TO_UNICODE) == TCL_ERROR){
      return TCL_ERROR;
    }
    if (trunc) {
      *cont = 1;
    }
  }
#endif

  /*
   * Filling up the long descriptor.
   */

  memset ((char *) longDesc, 0, sizeof (*longDesc));
  longDesc->valmode = *cont ? VM_DATAPART : VM_ALLDATA;
  longDesc->vallen  = *bytes;
#if ADABAS_VERSION >= 62
  longDesc->valpos  = longSi->bufpos + 1 + sizeof (long_descriptor);
#else /* ADABAS_VERSION <= 61 */
  longDesc->valpos  = PARSID + 1 + sizeof (long_descriptor) + 1;
#endif

  data[longSi->bufpos-1] = 0; /* defined byte. */
  memcpy (data + longSi->bufpos, (char *) longDesc, sizeof (*longDesc));

  if (AdabasSendPacket (interp, vars) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (getLongDescriptor (interp, vars, longDesc, *cont,
			 longWriteFirst) == TCL_ERROR) {
    return TCL_ERROR;
  }

  *bytes = length;
  return TCL_OK;
}

/* -------------------------------------------------------------- */

extern int AdabasLongWriteNext (interp, cursorVars, longDesc, fileName,
				longValue, encoding, length, bytes, cont)
     Tcl_Interp      *interp;		/* Current interpreter. */
     CursorInfo      *cursorVars;	/* Current cursor. */
     long_descriptor *longDesc;		/* Descriptor returned by update. */
     char            *fileName;		/* Filename to read long out of. */
     char            *longValue;	/* directly given long to write;
					 * note, that this is already encoded. */
     int              encoding;		/* Long encoding, if write into string. */
     int              length;		/* length, still to write. */
     int4            *bytes;		/* Bytes written in this round. */
     int             *cont;		/* Is there more data to write? */
{
  int              partI;
  int4             dc1, dc2;
  char            *data;
  AdabasInfo      *vars        = cursorVars->logonInfo;
  AdabasPacketInfo thisOptions = vars->packetInfo;
  char            *longDescPtr;
  char             msg[80];		/* for error message. */
#if ADABAS_VERSION <= 61
  int2             argCount = 1;
#endif

  length -= *bytes;
  if (length < 0) {
    sprintf (msg, "AdabasLongWriteNext: Invalid length %d???", length);
    AppendResult (interp, msg);
    return TCL_ERROR;
  }
  if ((*cont = length > LONG_SIZE_NEXT (vars))) {
    length = LONG_SIZE_NEXT (vars);
  }

  /*
   * Filling up the long descriptor.
   */

  longDesc->valmode = *cont ? VM_DATAPART : VM_LASTDATA;
  longDesc->vallen  = length;
  longDesc->valpos  = LONG_VALPOS_NEXT;

  /*
   * Now construct the packet of mess type PUTVAL, which consists
   * solely of a LONGDATA part.
   */

  for (partI = 0; partI < PART_KINDS; partI++) {
    vars->partInfos[ partI ].bufLength = 0;
  }
  vars->partInfos[ pkLongdata ].buf       = NULL; /* we fill up later... */
  vars->partInfos[ pkLongdata ].bufLength = 
    LONG_VALPOS_NEXT + longDesc->vallen - 1;

  thisOptions.messType = M_PUTVAL;
  if (AdabasFillPacket (interp, vars, &thisOptions) == TCL_ERROR) {
    return TCL_ERROR;
  }

#if ADABAS_VERSION >= 62
  data  = getData (&vars->sndPacket->segm, &dc1, &dc2, FOR_LONG_OP);
#else /* ADABAS_VERSION <= 61*/
  data  = getData (&vars->sqlPacket, &dc1, &dc2, FOR_LONG_OP);
#endif
  if (!data) {
    SetResult (interp, "No data part???");
    return TCL_ERROR;
  }

#if ADABAS_VERSION >= 62
  longDescPtr = data;
#else /* ADABAS_VERSION <= 61*/
  memcpy (data, (char *) &argCount, sizeof (int2));
  longDescPtr = data + sizeof (int2);
#endif

  *longDescPtr = 0; /* defined byte. */
  memcpy (longDescPtr+1, (char *) longDesc, sizeof (*longDesc));

  data += longDesc->valpos - 1;

  if (fileName) {
    if (readFromFile (interp, fileName, data, *bytes, length) == TCL_ERROR) {
      return TCL_ERROR;
    }
  } else {
    memcpy (data, longValue, length);
  }

  if (AdabasSendPacket (interp, vars) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (getLongDescriptor (interp, vars, longDesc, *cont,
			 longWriteNext) == TCL_ERROR) {
    return TCL_ERROR;
  }

  *bytes += length;
  return TCL_OK;
}

/* -------------------------------------------------------------- */

int AdabasLongToFile (interp, toFile, fileName, length, data)
     Tcl_Interp *interp;		/* Current interpreter. */
     int         toFile;		/* File mode: 1=create; 2=append. */
     char       *fileName;		/* Filename to write data into. */
     int         length;		/* Length to write. */
     char       *data;			/* Long data to write. */
{
  int file;
  int mode;

  switch (toFile) {
  case 2:  mode = O_WRONLY | O_RAW | O_APPEND;          break;
  case 1:  mode = O_WRONLY | O_RAW | O_TRUNC | O_CREAT; break;
  default: return TCL_ERROR;
  }

  if ((file = open (fileName, mode, 0666)) < 0) {
    AppendResult (interp, fileName);
    AppendResult (interp, ": ");
    AppendResult (interp, Tcl_PosixError (interp));
    return TCL_ERROR;
  }
  write (file, data, length);
  close (file);

  return TCL_OK;
}

/* -------------------------------------------------------------- */

static int getLongDescriptor (interp, vars, longDesc, isError, action)
     Tcl_Interp      *interp;		/* Current interpreter. */
     AdabasInfo      *vars;		/* Current logon handle. */
     long_descriptor *longDesc;		/* Returned long descriptor. */
     int              isError;		/* Is it an error to find no descr? */
     LongAction       action;		/* For whom we look for a descr. */
{
  char *longDescPtr;
  char *procName;

#if ADABAS_VERSION >= 62
  longDescPtr = getLongdesc (&vars->rcvPacket->segm);
#else /* ADABAS_VERSION <= 61 */
  longDescPtr = getLongdesc (&vars->sqlPacket, action != longSelect);
#endif

  if (longDescPtr) {
    memcpy (longDesc, longDescPtr, sizeof (*longDesc));
  } else if (isError) {
    switch (action) {
    case longSelect:     procName = "longSelect";     break;
    case longWriteFirst: procName = "longWriteFirst"; break;
    case longWriteNext:  procName = "longWriteNext";  break;
    case longReadNext:   procName = "longReadNext";   break;
    default:             procName = "unknown action"; break;
    }
    Tcl_ResetResult (interp);
    AppendResult (interp, procName);
    AppendResult (interp, ": No returning long descriptor???");
    return TCL_ERROR;
  }
  return TCL_OK;
}

/* -------------------------------------------------------------- */

int AdabasLongValLength (interp, fileName, longValue,
			 encoding, substVal, length)
     Tcl_Interp  *interp;		/* Current interpreter. */
     char        *fileName;		/* Filename to read long out of. */
     char        *longValue;		/* directly given long to read. */
     int          encoding;		/* Encoding, if read from longValue. */
     Tcl_DString *substVal;		/* Substituted value of longValue,
					 * if longValue is not NULL. */
     int4        *length;		/* returning length of file or val. */
{
  if (fileName) {
    *length = fileLength (interp, fileName);
    Tcl_DStringInit (substVal);
  } else {
    *length = encodeSequence (interp, encoding, "long value",
			      substVal, longValue, 1, strlen (longValue));
  }
  return *length < 0 ? TCL_ERROR : TCL_OK;
}

/* ----------------- Local procedures --------------------------- */

static long fileLength (interp, fileName)
     Tcl_Interp  *interp;		/* Current interpreter. */
     char        *fileName;		/* Filename to read long out of. */
{
  int  file;
  long length;

  if ((file = open (fileName, O_RDONLY | O_RAW)) < 0) {
    AppendResult (interp, fileName);
    AppendResult (interp, ": ");
    AppendResult (interp, Tcl_PosixError (interp));
    return -1;
  }

  length = lseek (file, 0, 2);
  close (file);

  return length;
}

/* -------------------------------------------------------------- */

static int readFromFile (interp, fileName, data, offset, length)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *fileName;		/* Filename to read long out of. */
     char       *data;			/* Buffer to read long data into. */
     long        offset;		/* Where to start the reading. */
     long        length;		/* Length to read. */
{
  int  file;
  long ret;
  char msg[80];				/* for error message. */

  if ((file = open (fileName, O_RDONLY | O_RAW)) < 0) {
    AppendResult (interp, fileName);
    AppendResult (interp, ": ");
    AppendResult (interp, Tcl_PosixError (interp));
    return TCL_ERROR;
  }

  if (offset) {
    ret = lseek (file, offset, 0);
    if (ret != offset) {
      sprintf (msg, "seek to position %ld failed", offset);
      AppendResult (interp, msg);
      return TCL_ERROR;
    }
  }
  
  ret = read (file, data, length);
  close (file);

  if (ret != length) {
    sprintf (msg, "%s: wanted %ld bytes starting at offset %ld, but got only %ld",
	     fileName, length, offset, ret);
    AppendResult (interp, msg);
    return TCL_ERROR;
  }

  return TCL_OK;
}



