/* -------------------------------------------------------------- */
/* $Id: AdabasPacket.c,v 1.46 1997/02/03 20:50:26 adabas Exp $ */
/* -------------------------------------------------------------- */
/* Copyright (c) 1996-1997 Christian Krone. All rights reserved.  */
/* This program is free software; you can redistribute it and/or  */
/* modify it under the same terms as Tcl itself.                  */
/* See also licence.terms                                         */
/* -------------------------------------------------------------- */

/* ----------------- Import interface --------------------------- */
#include <stdlib.h>
#include <string.h>
#include <tcl.h>

#include "adabas.h"
#include "sqlrte.h"
#include "AdabasPort.h"
#include "AdabasUtil.h"

/* ----------------- Type definitions --------------------------- */
#if ADABAS_VERSION >= 62
#define ERRORTEXT_LENGTH   140
#define RESULTTABLE_LENGTH 20
#else /* ADABAS_VERSION <= 61 */
#define ERRORTEXT_LENGTH   70
#define RESULTTABLE_LENGTH 18
#endif
#define COLUMNNAME_LENGTH  18
#define CONVTABLE_LENGTH   256

/* ----------------- Export interface --------------------------- */
#include "AdabasPacket.h"

/* ----------------- Local macro definition --------------------- */
#if ADABAS_VERSION >= 62
#define PART_LENGTH(part) ((((part)->part_header.buf_len+7)/8)*8 \
			   + sizeof (part_header))
#endif

/* ----------------- Local procedure headings ------------------- */
#if ADABAS_VERSION >= 62
static void putConv _ANSI_ARGS_((Tcl_Interp *interp, char *buf, int count));
#else /* ADABAS_VERSION <= 61 */
static void putUser _ANSI_ARGS_((Tcl_Interp *interp, char *buf, int2 *pos));
static void putConv _ANSI_ARGS_((Tcl_Interp *interp, char *buf, int2 *pos));
#endif
static void putData _ANSI_ARGS_((Tcl_Interp *interp, char *buf, int len));
static void putNames _ANSI_ARGS_((Tcl_Interp *interp, char *buf, int count));
static void putInfos _ANSI_ARGS_((Tcl_Interp *interp, char *buf, int count));

/* ----------------- Exported procedures ------------------------- */

void AdabasPacketInit (vars)
     AdabasPacketInfo *vars;		/* Current sql packet. */
{
  int4  swapIt  = 1;
  char *pointIt = (char *) &swapIt;

  /*
   * Initialize the configurable options in the info record with some
   * suitable default values.
   */
  vars->messType          = M_DBS;
#if ADABAS_VERSION >= 62
  vars->producer          = PR_USER_CMD;
  vars->sqlMode           = SM_SESSION_SQLMODE;
#else /* ADABAS_VERSION <= 61 */
  vars->sqlMode           = SQLM_NIL;
#endif
  vars->withInfo          = 0;
  vars->commitImmediately = 0;
  vars->massCmd           = 0;
  memcpy (vars->application, "XCI", 3);

  vars->swapKind = pointIt[0] ? SW_FULL_SWAPPED
                 : pointIt[3] ? SW_NORMAL
		              : SW_PART_SWAPPED;
}

/* -------------------------------------------------------------- */

int scanMessType (interp, arg)
     Tcl_Interp *interp;		/* Current interpreter (or NULL). */
     char       *arg;			/* messtype to scan. */
{
  int length = strlen (arg);

  switch (*arg) {

  case 'b':
    if (!strncmp (arg, "buflength", length)) {
      return M_BUFLENGTH;
    }
    break;
    
  case 'd':
    if (!strncmp (arg, "dbs", length)) {
      return M_DBS;
    }
    break;
    
  case 'e':
    if (!strncmp (arg, "execute", length)) {
      return M_EXECUTE;
    }
    break;

  case 'g':
    if (!strncmp (arg, "getexecute", length) && length >= 4) {
      return M_GETEXECUTE;
    } else if (!strncmp (arg, "getparse", length) && length >= 4) {
      return M_GETPARSE;
    } else if (!strncmp (arg, "getval", length) && length >= 4) {
      return M_GETVAL;
    }
    break;

  case 'h':
    if (!strncmp (arg, "hello", length)) {
      return M_HELLO;
    }
    break;

  case 'i':
    if (!strncmp (arg, "incopy", length)) {
      return M_INCOPY;
    }
    break;

  case 'l':
    if (!strncmp (arg, "load", length)) {
      return M_LOAD;
    }
    break;

  case 'm':
    if (!strncmp (arg, "maxbuf", length) && length >= 2) {
      return M_MAXBUF;
    } else if (!strncmp (arg, "minbuf", length) && length >= 2) {
      return M_MINBUF;
    }
    break;

  case 'o':
    if (!strncmp (arg, "outcopy", length)) {
      return M_OUTCOPY;
    }
    break;

  case 'p':
    if (!strncmp (arg, "parse", length) && length >= 2) {
      return M_PARSE;
    } else if (!strncmp (arg, "putval", length) && length >= 2) {
      return M_PUTVAL;
    }
    break;

  case 's':
    if (!strncmp (arg, "switch", length) && length >= 2) {
      return M_SWITCH;
    } else if (!strncmp (arg, "switchlimit", length) && length >= 7) {
      return M_SWITCHLIMIT;
    } else if (!strncmp (arg, "syntax", length) && length >= 2) {
      return M_SYNTAX;
    }
    break;

  case 'u':
    if (!strncmp (arg, "unload", length) && length >= 2) {
      return M_UNLOAD;
    } else if (!strncmp (arg, "utility", length) && length >= 2) {
      return M_UTILITY;
    }
    break;
  }

  if (interp) {
    AppendResult (interp, "unknown messType \"");
    AppendResult (interp, arg);
    AppendResult (interp, "\"");
  }
  return M_NIL;
}

/* -------------------------------------------------------------- */

int scanDataType (interp, arg)
     Tcl_Interp *interp;		/* Current interpreter (or NULL). */
     char       *arg;			/* datatype to scan. */
{
  int length = strlen (arg);

  switch (*arg) {

  case 'b':
    if (!strncmp (arg, "boolean",  length)) {
#if ADABAS_VERSION >= 62
      return DBOOLEAN;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_BOOLEAN;
#endif
    }
    break;

  case 'c':
    if (!strncmp (arg, "char_ascii",  length) && length >= 6) {
#if ADABAS_VERSION >= 62
      return DCHA;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_CHAR;
#endif
    } else if (!strncmp (arg, "char_byte",   length) && length >= 6) {
#if ADABAS_VERSION >= 62
      return DCHB;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_BYTE;
#endif
    } else if (!strncmp (arg, "char_ebcdic", length) && length >= 6) {
#if ADABAS_VERSION >= 62
      return DCHE;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_CHAR;
#endif
    }
    break;

  case 'd':
    if (!strncmp (arg, "date",         length) && length >= 2) {
#if ADABAS_VERSION >= 62
      return DDATE;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_DATE;
#endif
    } else if (!strncmp (arg, "duration",     length) && length >= 2) {
      return DDURATION;
    } else if (!strncmp (arg, "dbyte_ebcdic", length) && length >= 2) {
#if ADABAS_VERSION >= 62
      return DDBYTEEBCDIC;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_DBYTE_EBCDIC;
#endif
    }
    break;

  case 'f':
    if (!strncmp (arg, "fixed", length) && length >= 2) {
#if ADABAS_VERSION >= 62
      return DFIXED;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_FIXED;
#endif
    } else if (!strncmp (arg, "float", length) && length >= 2) {
#if ADABAS_VERSION >= 62
      return DFLOAT;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_FLOAT;
#endif
    }
    break;

  case 'i':
    if (!strncmp (arg, "integer", length)) {
#if ADABAS_VERSION >= 62
      return DINTEGER;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_INTEGER;
#endif
    }
    break;

  case 'l':
    if (!strncmp (arg, "long_ascii",   length) && length >= 6) {
#if ADABAS_VERSION >= 62
      return DSTRA;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_C_OLDLONG_CHAR;
#endif
    } else if (!strncmp (arg, "long_byte",    length) && length >= 6) {
#if ADABAS_VERSION >= 62
      return DSTRB;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_B_OLDLONG_BYTE;
#endif
    } else if (!strncmp (arg, "long_dbyte",   length) && length >= 6) {
#if ADABAS_VERSION >= 62
      return DSTRDB;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_A_OLDLONG_ASCII_DBYTE;
#endif
    } else if (!strncmp (arg, "long_ebcdic",  length) && length >= 6) {
#if ADABAS_VERSION >= 62
      return DSTRE;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_C_OLDLONG_CHAR;
#endif
    } else if (!strncmp (arg, "long_unicode", length) && length >= 6) {
      return DSTRUNI;
    }
    break;

  case 'n':
    if (!strncmp (arg, "new_long_ascii",  length) && length >= 10) {
#if ADABAS_VERSION >= 62
      return DLONGA;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_C_LONG_CHAR;
#endif
    } else if (!strncmp (arg, "new_long_byte",   length) && length >= 10) {
#if ADABAS_VERSION >= 62
      return DLONGB;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_B_LONG_BYTE;
#endif
    } else if (!strncmp (arg, "new_long_dbyte",  length) && length >= 10) {
#if ADABAS_VERSION >= 62
      return DLONGDB;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_A_LONG_ASCII_DBYTE;
#endif
    } else if (!strncmp (arg, "new_long_ebcdic", length) && length >= 10) {
      return DLONGE;
    }
    break;

  case 'r':
    if (!strncmp (arg, "rowid", length)) {
#if ADABAS_VERSION >= 62
      return DROWID;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_OTHERWISE;
#endif
    }
    break;

  case 's':
    if (!strncmp (arg, "smallint", length)) {
#if ADABAS_VERSION >= 62
      return DSMALLINT;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_SMALLINT;
#endif
    }
    break;

  case 't':
    if (!strncmp (arg, "time",      length) && length == 4) {
#if ADABAS_VERSION >= 62
      return DTIME;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_TIME;
#endif
    } else if (!strncmp (arg, "timestamp", length) && length >= 4) {
#if ADABAS_VERSION >= 62
      return DTIMESTAMP;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_TIMESTAMP;
#endif
    }
    break;

  case 'v':
    if (!strncmp (arg, "varchar_ascii",  length) && length >= 9) {
#if ADABAS_VERSION >= 62
      return DVARCHARA;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_VARCHAR;
#endif
    } else if (!strncmp (arg, "varchar_byte",   length) && length >= 9) {
#if ADABAS_VERSION >= 62
      return DVARCHARE;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_VARCHAR;
#endif
    } else if (!strncmp (arg, "varchar_ebcdic", length) && length >= 9) {
#if ADABAS_VERSION >= 62
      return DVARCHARB;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_VARCHAR;
#endif
    } else if (!strncmp (arg, "vfloat",         length) && length >= 2) {
#if ADABAS_VERSION >= 62
      return DVFLOAT;
#else /* ADABAS_VERSION <= 61 */
      return CSP_INFO_EXPRESSION;
#endif
    }
    break;

  case 'u':
    if (!strncmp (arg, "unicode", length)) {
      return DUNICODE;
    }
    break;

  }

  if (interp) {
    AppendResult (interp, "unknown dataType \"");
    AppendResult (interp, arg);
    AppendResult (interp, "\"");
  }
  return DUNKNOWN;
}

/* -------------------------------------------------------------- */

#if ADABAS_VERSION >= 62
int scanProducer (interp, arg)
     Tcl_Interp *interp;		/* Current interpreter (or null). */
     char       *arg;			/* producer to scan. */
{
  int length = strlen (arg);

  /*
   * Scan the given producer. Note, that there is no chance to give
   * kernel as producer; sometimes we better allow not anything...
   */

  switch (*arg) {

  case 'i':
    if (!strncmp (arg, "internal",     length) && length >= 3) {
      return PR_INTERNAL_CMD;
    } else if (!strncmp (arg, "installation", length) && length >= 3) {
      return PR_INSTALLATION;
    }
    break;

  case 'u':
    if (!strncmp (arg, "user", length)) {
      return PR_USER_CMD;
    }
    break;
  }

  if (interp) {
    AppendResult (interp, "unknown producer \"");
    AppendResult (interp, arg);
    AppendResult (interp, "\"");
  }
  return PR_NIL;
}
#endif

/* -------------------------------------------------------------- */

int scanSqlMode (interp, arg, external)
     Tcl_Interp *interp;		/* Current interpreter (or null). */
     char       *arg;			/* producer to scan. */
     int         external;		/* Should external sqlmode be used? */
{
  int length = strlen (arg);

  switch (*arg) {

  case 'a':
    if (!strncmp (arg, "adabas", length) && length >= 2) {
#if ADABAS_VERSION >= 62
      if (!external) {
	return SM_ADABAS;
      }
#endif
      return SQLM_ADABAS;
    }
    if (!strncmp (arg, "ansi",   length) && length >= 2) {
#if ADABAS_VERSION >= 62
      if (!external) {
	return SM_ANSI;
      }
#endif
      return SQLM_ANSI;
    }
    break;

  case 'd':
    if (!strncmp (arg, "db2", length)) {
#if ADABAS_VERSION >= 62
      if (!external) {
	return SM_DB2;
      }
#endif
      return SQLM_DB2;
    }
    break;

  case 'o':
    if (!strncmp (arg, "oracle", length)) {
#if ADABAS_VERSION >= 62
      if (!external) {
	return SM_ORACLE;
      }
#endif
      return SQLM_ORACLE;
    }
    break;

  case 's':
#if ADABAS_VERSION >= 62
    if (!strncmp (arg, "session", length)) {
      if (!external) {
	return SM_SESSION_SQLMODE;
      }
    }
#endif
    break;
  }

  if (interp) {
    AppendResult (interp, "unknown sqlmode \"");
    AppendResult (interp, arg);
    AppendResult (interp, "\"");
  }
#if ADABAS_VERSION >= 62
  if (!external) {
    return SM_NIL;
  }
#endif
  return -1;
}

/* -------------------------------------------------------------- */

/*
 * Here now comes the procedures that differences the most between
 * release 6.1 and 6.2, since it interprets the result packets...
 */
#if ADABAS_VERSION >= 62
void putResultPart (interp, ret_part, part_length)
     Tcl_Interp *interp;		/* Current interpreter. */
     part       *ret_part;		/* result part to put into interp. */
     int4       *part_length;		/* length of this part. */
{
  char      error_text[ERRORTEXT_LENGTH+1];
  char      result_table[RESULTTABLE_LENGTH+1];
  char      count_string[11+1];
  int4      result_count;
  num_error num_error;

  *part_length = PART_LENGTH (ret_part);

  switch (ret_part->part_header.part_kind) {
  case PK_APPL_PARAMETER_DESCRIPTION:
    AppendResult (interp, " -applParameterDescription");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_COLUMNNAMES:
    AppendResult (interp, " -columnnames {");
    putNames (interp, ret_part->buf, ret_part->part_header.arg_count);
    AppendResult (interp, "}");
    break;

  case PK_COMMAND:
    AppendResult (interp, " -command");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_CONV_TABLES_RETURNED:
    AppendResult (interp, " -convTablesReturned {");
    putConv (interp, ret_part->buf, ret_part->part_header.arg_count);
    AppendResult (interp, "}");
    break;

  case PK_DATA:
    AppendResult (interp, " -data");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_ERRORTEXT:
    AppendResult  (interp, " -errortext");
    p2c (error_text, ret_part->buf, ret_part->part_header.buf_len);
    Tcl_AppendElement (interp, error_text);
    break;

  case PK_GETINFO:
    AppendResult (interp, " -getinfo");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_MODULNAME:
    AppendResult (interp, " -modulname");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_PAGE:
    AppendResult (interp, " -page");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_PARSID:
    AppendResult (interp, " -parsid");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_PARSID_OF_SELECT:
    AppendResult (interp, " -parsidOfSelect");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_RESULTCOUNT:
    if (ret_part->buf[0])
      sprintf (count_string, " NULL");
    else {
      s40glint (ret_part->buf, 2, 10, &result_count, &num_error);
      if (num_error)
	sprintf (count_string, " ?");
      else
	sprintf (count_string, " %d", result_count);
    }
    AppendResult  (interp, " -resultcount");
    AppendResult  (interp, count_string);
    break;

  case PK_RESULTTABLENAME:
    if (ret_part->part_header.buf_len)
      p2c (result_table, ret_part->buf, ret_part->part_header.buf_len);
    else
      *result_table = '\0';
    AppendResult  (interp, " -resulttablename");
    Tcl_AppendElement (interp, result_table);
    break;

  case PK_SHORTINFO:
    AppendResult (interp, " -shortinfos {");
    putInfos (interp, ret_part->buf, ret_part->part_header.arg_count);
    AppendResult (interp, "}");
    break;

  case PK_USER_INFO_RETURNED:
    AppendResult (interp, " -userInfoReturned {");
    putNames (interp, ret_part->buf, 4);
    AppendResult (interp, "}");
    break;

  case PK_SURROGATE:
    AppendResult (interp, " -surrogate");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_BDINFO:
    AppendResult (interp, " -bdinfo");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;

  case PK_LONGDATA:
    AppendResult (interp, " -longdata");
    putData (interp, ret_part->buf, ret_part->part_header.buf_len);
    break;
  }
}
#else /* ADABAS_VERSION <= 61 */
void putResultPart (interp, partNo, currPos, retSegment, sndMessType)
     Tcl_Interp *interp;		/* Current interpreter. */
     int         partNo;		/* part1 or part2 of var_part? */
     int2       *currPos;		/* current position in part. */
     packet     *retSegment;		/* the whole return packet. */
     int         sndMessType;		/* messtype, that was send. */
{
  char      error_text[ERRORTEXT_LENGTH+1];
  char      result_table[RESULTTABLE_LENGTH+1];
  char      count_string[11+1];
  int4      result_count;
  num_error num_error;
  int2      argCount;
  char     *partBuf;
  int       part1IsInfo;

  switch (partNo) {
  case 1:
    if (retSegment->part1_length) {
      if (retSegment->return_code) {

	/*
	 * In case of an error in part1 there is the SP1PK_ERRORTEXT.
	 */

	AppendResult  (interp, " -errortext");
	p2c (error_text, retSegment->var_part, retSegment->part1_length);
	Tcl_AppendElement (interp, error_text);
	*currPos += retSegment->part1_length;

      } else if (!retSegment->part2_length) {
	if (retSegment->part1_length == 1093
	    || retSegment->part1_length == 1605) {

	  /*
	   * Let's hope, that the following statement is true: If we have a
	   * part1 with one of the given lengths, but no part2, it must be
	   * the SP1PK_CONV_TABLES_RETURNED and SP1PK_USER_INFO_RETURNED.
	   * The two different lengths depend on the messtype: the kernel
	   * returns two more tables for termsets.
	   */

	  AppendResult (interp, " -userInfoReturned {");
	  putUser (interp, retSegment->var_part, currPos);
	  AppendResult (interp, "}");

	  AppendResult (interp, " -convTablesReturned {");
	  putConv (interp, retSegment->var_part+66, currPos);
	  AppendResult (interp, "}");

	} else {
	  part1IsInfo = 0;
	  if (retSegment->part1_length >= 2) {
	    memcpy ((char *) &argCount, retSegment->var_part, 2);
	    part1IsInfo = 2+argCount*sizeof (field_info61)
	      == retSegment->part1_length;
	  }

	  if (part1IsInfo) {

	    /*
	     * So it seems, that we have found a shortinfo/columnnames part.
	     */

	    AppendResult (interp, " -shortinfos {");
	    putInfos (interp, retSegment->var_part + 2, argCount);
	    AppendResult (interp, "}");

	    AppendResult (interp, " -columnnames {");
	    putNames (interp, retSegment->var_part + 2, argCount);
	    AppendResult (interp, "}");

	    *currPos = retSegment->part1_length + 1;
	  } else {

	    /*
	     * If we have no part2 and a part1 with none of the above lengths,
	     * part1 is a special kind for some utility commands and the like.
	     */

	    AppendResult (interp, " -data");
	    putData (interp, retSegment->var_part, retSegment->part1_length);
	    *currPos = retSegment->part1_length + 1;
	  }
	}

      } else if (retSegment->part1_length == RESULTTABLE_LENGTH) {

	/*
	 * If part1 is exactly 18 bytes long, it is a SP1PK_RESULTTABLENAME.
	 */

	p2c (result_table, retSegment->var_part, retSegment->part1_length);
	AppendResult  (interp, " -resulttablename");
	Tcl_AppendElement (interp, result_table);
	*currPos += RESULTTABLE_LENGTH;

      } else {

	/*
	 * If all went well, part1 are the SP1PK_SHORTINFO and the
	 * SP1PK_COLUMNNAMES intermixed.
	 */

	AppendResult (interp, " -shortinfos {");
	memcpy ((char *) &argCount, retSegment->var_part, 2);
	putInfos (interp, retSegment->var_part + 2, argCount);
	AppendResult (interp, "}");

	AppendResult (interp, " -columnnames {");
	putNames (interp, retSegment->var_part + 2, argCount);
	AppendResult (interp, "}");

	*currPos += 2+argCount*sizeof (field_info61);
      }
    }
  break;
  case 2:
    if (retSegment->part2_length) {
      partBuf = retSegment->var_part
	+ retSegment->part1_length + *currPos -1;
      if (sndMessType == M_GETVAL) {

	/*
	 * For the reply to a getval request all the part 2 (except
	 * the first 2 bytes is a PK_LONGDATA part.
	 */

	partBuf += 2;
	AppendResult (interp, " -longdata");
	putData (interp, partBuf, retSegment->part2_length - 2);
	*currPos = retSegment->part2_length + 1;

      } else if (sndMessType == M_PARSE) {

	/*
	 * For the reply to a parse request the part2 consists of a
	 * parsid followed by a resulttablename.
	 */

	if (*currPos == 1) {
	  AppendResult (interp, " -parsid");
	  putData (interp, partBuf, 12);
	  *currPos += 12;
	} else {
	  AppendResult (interp, " -resulttablename");
	  putData (interp, partBuf, retSegment->part2_length - *currPos + 1);
	  *currPos += retSegment->part2_length + 1;
	}
      } else if (*currPos == 1 && *partBuf == 1) {

	/*
	 * At the start of part2 there is the SP1PK_RESULTCUNT.
	 */

	s40glint (partBuf, 2, 10,
		  &result_count, &num_error);
	if (num_error)
	  sprintf (count_string, " ?");
	else
	  sprintf (count_string, " %d", result_count);
	AppendResult (interp, " -resultcount");
	AppendResult (interp, count_string);
	*currPos += 11;

      } else { 

	/*
	 * The rest of part2 is filled by the SP1PK_DATA.
	 */

	AppendResult (interp, " -data");
	putData (interp, partBuf, retSegment->part2_length - *currPos + 1);
	*currPos = retSegment->part2_length + 1;
      }
    }
    break;
  }
}
#endif

/* -------------------------------------------------------------- */

#if ADABAS_VERSION >= 62
int getResultCount (retSegment)
     segment *retSegment;		/* Return Segment. */
{
  int4      resultCount;
  num_error numError;
  part     *currPart = &retSegment->part;
  int ix;

  for (ix = 0; ix < retSegment->segm_header.kind.cmd.no_of_parts; ix++) {
    if (currPart->part_header.part_kind == PK_RESULTCOUNT) {
      if (currPart->buf[0])
	return 0; /* NULL */
      s40glint (currPart->buf, 2, 10, &resultCount, &numError);
      return numError ? 0 : resultCount;
    }
    currPart = (part *) (((char *) currPart) + PART_LENGTH (currPart));
  }
  return 0; /* No resultcount part at all. */
}

#else /* ADABAS_VERSION <= 61 */
int getResultCount (retSegment)
     packet     *retSegment;		/* the whole return packet. */
{
  int4      resultCount;
  num_error numError;

  if (retSegment->var_part[retSegment->part1_length] != 1)
    return 0; /* NULL or no result count at all. */
  s40glint (retSegment->var_part + retSegment->part1_length,
	    2, 10, &resultCount, &numError);
  return numError ? 0 : resultCount;
}
#endif

/* -------------------------------------------------------------- */

#if ADABAS_VERSION >= 62
int getResultTable (retSegment, resultTable)
     segment *retSegment;		/* Return Segment. */
     char    *resultTable;		/* Name of result table returned. */
{
  part *currPart = &retSegment->part;
  int ix;

  for (ix = 0; ix < retSegment->segm_header.kind.cmd.no_of_parts; ix++) {
    if (currPart->part_header.part_kind == PK_RESULTTABLENAME) {
      if (currPart->part_header.buf_len)
	p2c (resultTable, currPart->buf, currPart->part_header.buf_len);
      else
	*resultTable = '\0';
      return 1;
    }
    currPart = (part *) (((char *) currPart) + PART_LENGTH (currPart));
  }
  return 0;
}

int
getParsId (interp, retSegment, parsIdDString)
     Tcl_Interp  *interp;		/* Current interpreter. */
     segment     *retSegment;		/* Return Segment. */
     Tcl_DString *parsIdDString;	/* ParsId returned. */
{
  part *currPart = &retSegment->part;
  int ix;

  for (ix = 0; ix < retSegment->segm_header.kind.cmd.no_of_parts; ix++) {
    if (currPart->part_header.part_kind == PK_PARSID) {
      decodeSequence (interp, encHex, (Tcl_Obj *) NULL, parsIdDString,
		      currPart->buf, currPart->part_header.buf_len);
      return 1;
    }
    currPart = (part *) (((char *) currPart) + PART_LENGTH (currPart));
  }
  return 0;
}

#else /* ADABAS_VERSION <= 61 */

int
getResultTable (retSegment, messType, resultTable)
     packet *retSegment;		/* the whole return packet. */
     int     messType;			/* message type, the packet was send. */
     char   *resultTable;		/* Name of result table returned. */
{
  if (messType == M_PARSE
      && retSegment->part2_length == PARSID + RESULTTABLE_LENGTH) {
    p2c (resultTable, retSegment->var_part + retSegment->part1_length + PARSID,
	 RESULTTABLE_LENGTH);
    return 1;
  } else if (messType == M_DBS
	     &&retSegment->part1_length == RESULTTABLE_LENGTH) {
    p2c (resultTable, retSegment->var_part, retSegment->part1_length);
    return 1;
  } else {
    return 0;
  }
}

int getParsId (interp, retSegment, parsIdDString)
     Tcl_Interp  *interp;		/* Current interpreter. */
     packet      *retSegment;		/* the whole return packet. */
     Tcl_DString *parsIdDString;	/* ParsId returned. */
{
  if (retSegment->part2_length >= PARSID) {
    decodeSequence (interp, encHex, (Tcl_Obj *) NULL, parsIdDString,
		    retSegment->var_part+retSegment->part1_length,
		    PARSID);
    return 1;
  } else {
    return 0;
  }
}
#endif

/* -------------------------------------------------------------- */

#if ADABAS_VERSION >= 62
int getShortInfos (retSegment, shortInfos, columnNames)
     segment     *retSegment;		/* Return Segment. */
     field_info **shortInfos;		/* Shortinfos returned. */
     char       **columnNames;		/* Columnnames returned. */
{
  part *currPart = &retSegment->part;
  int ix;
  int cntInfos = 0;
  int cntNames = 0;

  for (ix = 0; ix < retSegment->segm_header.kind.cmd.no_of_parts; ix++) {
    switch (currPart->part_header.part_kind) {
    case PK_SHORTINFO:
      if ((cntInfos = currPart->part_header.arg_count)) {
	if (*shortInfos) {
	  *shortInfos = (field_info *) ckrealloc ((char *) *shortInfos,
						  cntInfos * sizeof (field_info));
	} else {
	  *shortInfos = (field_info *) ckalloc (cntInfos * sizeof (field_info));
	}
	memcpy (*shortInfos, currPart->buf, cntInfos * sizeof (field_info));
      }
      break;
    case PK_COLUMNNAMES:
      if ((cntNames = currPart->part_header.arg_count)) {
	if (*columnNames) {
	  *columnNames = ckrealloc (*columnNames,
				    currPart->part_header.buf_len);
	} else {
	  *columnNames = ckalloc (currPart->part_header.buf_len);
	}
	memcpy (*columnNames, currPart->buf, currPart->part_header.buf_len);
      }
    }
    currPart = (part *) (((char *) currPart) + PART_LENGTH (currPart));
  }

  return cntInfos; /* Note, that cntNames can be 0. */
}

char *getData (retSegment, dataLength, argCount, flags)
     segment *retSegment;		/* Return Segment. */
     int4    *dataLength;		/* Length of returned data part. */
     int     *argCount;			/* Its argument count. */
     int      flags;			/* as defined in AdabasPacket.h. */
{
  part *currPart = &retSegment->part;
  int ix;

  for (ix = 0; ix < retSegment->segm_header.kind.cmd.no_of_parts; ix++) {
    switch (currPart->part_header.part_kind) {
    case PK_LONGDATA:
      if (!(FOR_LONG_OP & flags)) {
	break;
      }
    case PK_DATA:
      *dataLength = currPart->part_header.buf_len;
      *argCount   = currPart->part_header.arg_count;
      return currPart->buf;
    }
    currPart = (part *) (((char *) currPart) + PART_LENGTH (currPart));
  }
  /*
   * No data part found at all.
   */
  *dataLength = 0;
  *argCount   = 0;
  return (char *) NULL;
}

char *getLongdesc (retSegment)
     segment *retSegment;		/* Return Segment. */
{
  part *currPart = &retSegment->part;
  int ix;

  for (ix = 0; ix < retSegment->segm_header.kind.cmd.no_of_parts; ix++) {
    if (currPart->part_header.part_kind == PK_DATA ||
	currPart->part_header.part_kind == PK_LONGDATA) {
      return currPart->buf+1; /* skip defined byte. */
    }
    currPart = (part *) (((char *) currPart) + PART_LENGTH (currPart));
  }
  return (char *) NULL; /* No longdata part at all. */
}

extern int datatypeIsLong (dataType)
     int dataType;			/* Datatype to check. */
{
  switch (dataType) {
  case DSTRA:
  case DSTRE:
  case DSTRB:
  case DSTRDB:
  case DSTRUNI:
    return 1;
  default:
    return 0;
  }
}

#else /* ADABAS_VERSION <= 61 */
int getShortInfos (retSegm, shortInfos, columnNames)
     packet      *retSegm;		/* the whole return packet. */
     field_info **shortInfos;		/* Shortinfos returned. */
     char       **columnNames;		/* Columnnames returned. */
{
  int2        argCount;
  int         withColumnName;
  char       *srcPtr;
  field_info *destSi;
  int1        length1;
  char        columnName[COLUMNNAME_LENGTH+1];
  Tcl_DString colNames;
  int         curr;

  /* Here we have to check very carefully, if part1 is really containing
   * param infos, since it may be the name of the resulttable.
   * Further more we have to determine, if there are fieldinfos (including
   * column name) or shortfieldinfos (without columnname) in part1.
   */
  if (retSegm->return_code == 0 && retSegm->part1_length > 0) {

    memcpy ((char *) &argCount, retSegm->var_part, 2);
    if (argCount * sizeof (field_info61) + 2 == retSegm->part1_length) {
      withColumnName = 1;
    } else if (argCount * sizeof (sfield_info61) + 2 == retSegm->part1_length) {
      withColumnName = 0;
    } else {
      return 0; /* no infos in part1. */
    }
  } else {
    return 0; /* empty part1. */
  }
  if (*shortInfos) {
    *shortInfos = (field_info *) ckrealloc ((char *) *shortInfos,
					    argCount * sizeof (field_info));
  } else {
    *shortInfos = (field_info *) ckalloc (argCount * sizeof (field_info));
  }
  srcPtr  = retSegm->var_part+2;
  destSi  = *shortInfos;
  if (withColumnName) {
    Tcl_DStringInit (&colNames);
  }

  for (curr = 0; curr < argCount; curr++) {
    if (withColumnName) {
      p2c (columnName, srcPtr, COLUMNNAME_LENGTH);		/* fieldname */
      length1 = (int1) strlen (columnName);
      Tcl_DStringAppend (&colNames, (char *) &length1, 1);
      Tcl_DStringAppend (&colNames, columnName,       -1);
      srcPtr += COLUMNNAME_LENGTH;
    } else {
      destSi->mode    = *srcPtr; srcPtr++;			/* mode */
      destSi->io_type = *srcPtr; srcPtr++;			/* io_type */
    }

    destSi->data_type = *srcPtr; srcPtr++;			/* d_type */
    destSi->length    = *(unsigned char *) srcPtr; srcPtr++;	/* length */
    if (destSi->length) {
      destSi->frac       = *srcPtr; srcPtr++;			/* frac */
      destSi->in_out_len = *(unsigned char *) srcPtr; srcPtr++;	/* in_out_l */
    } else {
      destSi->in_out_len = *(int2 *) srcPtr; srcPtr += 2;	/* longiolen */
      destSi->length     = destSi->in_out_len - 1;
    }
    destSi->bufpos = *(int2 *) srcPtr; srcPtr += 2;		/* bufpos */

    destSi++;
  }

  if (withColumnName) {
    if (*columnNames) {
      *columnNames = ckrealloc (*columnNames, Tcl_DStringLength (&colNames));
    } else {
      *columnNames = ckalloc (Tcl_DStringLength (&colNames));
    }
    memcpy (*columnNames, Tcl_DStringValue (&colNames),
	    Tcl_DStringLength (&colNames));
    Tcl_DStringFree (&colNames);
  }
    
  return argCount;
}

char *getData (retSegm, dataLength, argCount, flags)
     packet  *retSegm;			/* the whole return packet. */
     int4    *dataLength;		/* Length of returned data part. */
     int     *argCount;			/* Its argument count. */
     int      flags;			/* as defined in AdabasPacket.h. */
{
  int  offset = FOR_LONG_OP & flags ? 0 : 11;
  int2 args;
  
  if (UTILITY_CMD & flags) {
    if (retSegm->part1_length) {
      *argCount   = 1;
      *dataLength = retSegm->part1_length;
      return retSegm->var_part;
    }

  } else if (retSegm->part2_length > offset) {
    if (MASS_CMD & flags) {
      memcpy ((char *) &args,
	      retSegm->var_part + retSegm->part1_length + offset, 2);
      offset += 2;
    }
    *argCount   = args;
    *dataLength = retSegm->part2_length - offset;
    if (FOR_LONG_OP & flags) {
	return retSegm->var_part;
    } else {
	return retSegm->var_part + retSegm->part1_length + offset;
    }
  }

  /*
   * No data part found at all.
   */

  *dataLength = 0;
  *argCount   = 0;
  return (char *) NULL;
}

char *getLongdesc (retSegm, withArgCount)
     packet  *retSegm;			/* the whole return packet. */
     int withArgCount;			/* part2 with leading arg count? */
{
  int    offset;

  if (retSegm->part2_length >= 44
      && retSegm->var_part[ retSegm->part1_length   ] == 1
      && retSegm->var_part[ retSegm->part1_length+1 ] != 0) {
    offset = withArgCount ? 13 : 11; /* Skip resultcount (+ argCount). */
  } else if (retSegm->part2_length >= 35
	   && retSegm->var_part[ retSegm->part1_length   ] == 1
	     && retSegm->var_part[ retSegm->part1_length+1 ] == 0) {
    offset = 2;
  } else {
    return (char *) NULL; /* No long descriptor at all. */
  }
  if (retSegm->var_part[ retSegm->part1_length + offset ]) {
    return (char *) NULL; /* undef byte before long descriptor. */
  }
  return retSegm->var_part + retSegm->part1_length + offset + 1;
}

extern int datatypeIsLong (dataType)
     int dataType;			/* Datatype to check. */
{
  switch (dataType) {
  case CSP_INFO_C_OLDLONG_CHAR:
  case CSP_INFO_B_OLDLONG_BYTE:
  case CSP_INFO_A_OLDLONG_ASCII_DBYTE:
    return 1;
  default:
    return 0;
  }
}

#endif

/* -------------------------------------------------------------- */

char *scanApplication (interp, arg)
     Tcl_Interp *interp;		/* Current interpreter (or null). */
     char       *arg;			/* application to scan. */
{
  int length = strlen (arg);

  switch (*arg) {

  case 'c':
    if (!strncmp (arg, "commandInterface", length) && length >= 3) {
      return "XCI";
    } else if (!strncmp (arg, "control", length) && length >= 3) {
      return "CON";
    }
    break;

  case 'l':
    if (!strncmp (arg, "load", length)) {
      return "LOA";
    }
    break;

  case 'u':
    if (!strncmp (arg, "utility", length)) {
      return "UTI";
    }
    break;
  }

  if (interp) {
    AppendResult (interp, "unknown application \"");
    AppendResult (interp, arg);
    AppendResult (interp, "\"");
  }
  return (char *) NULL;
}

service scanService (interp, arg)
     Tcl_Interp *interp;		/* Current interpreter (or null). */
     char       *arg;			/* service to scan. */
{
  int length = strlen (arg);

  if (length > 1 && *arg == 'u') {
    switch (arg[1]) {
    case 's':
      if (!strncmp (arg, "user", length)) {
	return SQL_USER;
      }
      break;

    case 't':
      if (!strncmp (arg, "utility", length)) {
	return SQL_UTILITY;
      }
      break;
    }
  }
  if (interp) {
    AppendResult (interp, "bad service \"");
    AppendResult (interp, arg);
    AppendResult (interp, "\": must be user or utility");
  }
  return -1;
}

/* -------------------------------------------------------------- */

char *dataType2String (dataType)
     int dataType;			/* Datatype to convert. */
{
  switch (dataType) {
#if ADABAS_VERSION >= 62
  case DFIXED:       return "fixed";
  case DFLOAT:       return "float";
  case DCHA:         return "char_ascii";
  case DCHE:         return "char_ebcdic";
  case DCHB:         return "char_byte";
  case DROWID:       return "rowid";
  case DSTRA:        return "long_ascii";
  case DSTRE:        return "long_ebcdic";
  case DSTRB:        return "long_byte";
  case DSTRDB:       return "long_dbyte";
  case DSTRUNI:      return "long_unicode";
  case DDATE:        return "date";
  case DTIME:        return "time";
  case DVFLOAT:      return "vfloat";
  case DTIMESTAMP:   return "timestamp";
  case DUNKNOWN:     return "unknown";
  case DNUMBER:      return "number";
  case DNONUMBER:    return "nonumber";
  case DDURATION:    return "duration";
  case DDBYTEEBCDIC: return "dbyte_ebcdic";
  case DLONGA:       return "new_long_ascii";
  case DLONGE:       return "new_long_ebcdic";
  case DLONGB:       return "new_long_byte";
  case DLONGDB:      return "new_long_dbyte";
  case DBOOLEAN:     return "boolean";
  case DUNICODE:     return "unicode";
  case DSMALLINT:    return "smallint";
  case DINTEGER:     return "integer";
  case DVARCHARA:    return "varchar_ascii";
  case DVARCHARE:    return "varchar_ebcdic";
  case DVARCHARB:    return "varchar_byte";
#else /* ADABAS_VERSION <= 61 */
  case CSP_INFO_FIXED:                 return "fixed";
  case CSP_INFO_FLOAT:                 return "float";
  case CSP_INFO_CHAR:                  return "char_ascii";
  case CSP_INFO_BYTE:                  return "char_byte";
  case CSP_INFO_OTHERWISE:             return "rowid";
  case CSP_INFO_C_OLDLONG_CHAR:        return "long_ascii";
  case CSP_INFO_B_OLDLONG_BYTE:        return "long_byte";
  case CSP_INFO_A_OLDLONG_ASCII_DBYTE: return "long_dbyte";
  case CSP_INFO_DATE:                  return "date";
  case CSP_INFO_TIME:                  return "time";
  case CSP_INFO_EXPRESSION:            return "vfloat";
  case CSP_INFO_TIMESTAMP:             return "timestamp";
  case CSP_INFO_DBYTE_EBCDIC:          return "dbyte_ebcdic";
  case CSP_INFO_C_LONG_CHAR:           return "new_long_ascii";
  case CSP_INFO_A_LONG_ASCII_DBYTE:    return "new_long_byte";
  case CSP_INFO_B_LONG_BYTE:           return "new_long_dbyte";
  case CSP_INFO_BOOLEAN:               return "boolean";
  case CSP_INFO_SMALLINT:              return "smallint";
  case CSP_INFO_INTEGER:               return "integer";
  case CSP_INFO_VARCHAR:               return "varchar_ascii";
#endif
  default: return "???";
  }  
}

/* -------------------------------------------------------------- */

char *messType2String (messType)
     int messType;			/* Messtype to convert. */
{
  switch (messType) {
  case M_BUFLENGTH:   return "buflength";
  case M_DBS:         return "dbs";
  case M_EXECUTE:     return "execute";
  case M_GETEXECUTE:  return "getexecute";
  case M_GETPARSE:    return "getparse";
  case M_GETVAL:      return "getval";
  case M_HELLO:       return "hello";
  case M_INCOPY:      return "incopy";
  case M_LOAD:        return "load";
  case M_MAXBUF:      return "maxbuf";
  case M_MINBUF:      return "minbuf";
  case M_OUTCOPY:     return "outcopy";
  case M_PARSE:       return "parse";
  case M_PUTVAL:      return "putval";
  case M_SWITCH:      return "switch";
  case M_SWITCHLIMIT: return "switchlimit";
  case M_SYNTAX:      return "syntax";
  case M_UNLOAD:      return "unload";
  case M_UTILITY:     return "utility";
  default: return "???";
  }
}

/* -------------------------------------------------------------- */

#if ADABAS_VERSION >= 62
char *producer2String (prod)
     int prod;				/* Producer to convert. */
{
  switch (prod) {
  case PR_INTERNAL_CMD: return "internal";
  case PR_INSTALLATION: return "installation";
  case PR_USER_CMD:     return "user";
  default:              return "???";
  }
}
#endif

/* -------------------------------------------------------------- */

char *sqlMode2String (sqlMode, external)
     int sqlMode;			/* Sqlmode to convert. */
     int external;			/* Should external sqlmode be used? */
{
#if ADABAS_VERSION >= 62
  if (!external) {
    switch (sqlMode) {
    case SM_SESSION_SQLMODE: return "session";
    case SM_ADABAS:          return "adabas";
    case SM_ANSI:            return "ansi";
    case SM_DB2:             return "db2";
    case SM_ORACLE:          return "oracle";
    default:                 return "???";
    }
  }
#endif

  switch (sqlMode) {
  case SQLM_NIL:    return "session";
  case SQLM_ADABAS: return "adabas";
  case SQLM_ANSI:   return "ansi";
  case SQLM_DB2:    return "db2";
  case SQLM_ORACLE: return "oracle";
  default: return "???";
  }
}

/* -------------------------------------------------------------- */

char *application2String (application)
     char *application;			/* Application to convert. */
{
  switch (*application) {
  case 'X': return "commandInterface";
  case 'C': return "control";
  case 'U': return "utility";
  case 'L': return "load";
  default:  return "???";
  }
}

/* -------------------------------------------------------------- */

char *service2String (serv)
     int serv;				/* Service to convert. */
{
  switch (serv) {
  case SQL_USER:    return "user";
  case SQL_UTILITY: return "utility";
  default:          return "???";
  }
}

/* ----------------- Local procedures --------------------------- */

#if ADABAS_VERSION <= 61
static void putUser (interp, buf, pos)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *buf;			/* String containing the user info. */
     int2       *pos;			/* Will be incremented by length. */
{
  char column_name[COLUMNNAME_LENGTH+1];

  p2c (column_name, buf+1, COLUMNNAME_LENGTH);
  Tcl_AppendElement (interp, column_name);
  p2c (column_name, buf+20, COLUMNNAME_LENGTH);
  Tcl_AppendElement (interp, column_name);
  p2c (column_name, buf+39, 8);
  Tcl_AppendElement (interp, column_name);
  p2c (column_name, buf+47, COLUMNNAME_LENGTH);
  Tcl_AppendElement (interp, column_name);
  *pos += 66;
}

/* -------------------------------------------------------------- */

static void putConv (interp, buf, currPos)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *buf;			/* String containing the conv table. */
     int2       *currPos;		/* Will be incremented by length. */
{

  Tcl_DString data_dstring;
  int count;

  for (count = 0; count < 2; count++) {
    Tcl_DStringInit (&data_dstring);
    decodeSequence (interp, encEscape, (Tcl_Obj *) NULL, &data_dstring, buf, 1);
    Tcl_AppendElement (interp, Tcl_DStringValue (&data_dstring));
    Tcl_DStringFree (&data_dstring);
    buf++;
    (*currPos)++;
  }

  for (count = 0; count < 4; count++) {
    Tcl_DStringInit (&data_dstring);
    decodeSequence (interp, encEscape, (Tcl_Obj *) NULL,
		    &data_dstring, buf, CONVTABLE_LENGTH);
    Tcl_AppendElement (interp, Tcl_DStringValue (&data_dstring));
    Tcl_DStringFree (&data_dstring);
    buf      += CONVTABLE_LENGTH;
    *currPos += CONVTABLE_LENGTH;
  }
}

/* -------------------------------------------------------------- */

#else /* ADABAS_VERSION >= 62 */

static void putConv (interp, buf, count)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *buf;			/* String containing the conv table. */
     int4        count;			/* Number of conv tables in buf. */
{
  Tcl_DString data_dstring;

  while (count-- > 0) {
    Tcl_DStringInit (&data_dstring);
    decodeSequence (interp, encEscape, (Tcl_Obj *) NULL,
		    &data_dstring, buf, CONVTABLE_LENGTH);
    Tcl_AppendElement (interp, Tcl_DStringValue (&data_dstring));
    Tcl_DStringFree (&data_dstring);
    buf += CONVTABLE_LENGTH;
  }
}
#endif

/* -------------------------------------------------------------- */

static void putData (interp, buf, len)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *buf;			/* String containing the data. */
     int4        len;			/* Length of data in buf. */
{
  Tcl_DString   data_dstring;
  
  Tcl_DStringInit (&data_dstring);
  decodeSequence (interp, encEscape, (Tcl_Obj *) NULL, &data_dstring, buf, len);
  Tcl_AppendElement (interp, Tcl_DStringValue (&data_dstring));
  Tcl_DStringFree (&data_dstring);
}

/* -------------------------------------------------------------- */

static void putNames (interp, buf, count)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *buf;			/* String containing the col names. */
     int         count;			/* Number of names in buf. */
{
  char column_name[COLUMNNAME_LENGTH+1];

#if ADABAS_VERSION <= 61
  field_info61 *pi = (field_info61 *) buf;

  while (count-- > 0) {
    p2c (column_name, pi->fieldname, COLUMNNAME_LENGTH);
    Tcl_AppendElement (interp, column_name);
    pi++;
  }      

#else /* ADABAS_VERSION >= 62 */
  while (count-- > 0) {
    p2c (column_name, buf+1, *buf);
    Tcl_AppendElement (interp, column_name);
    buf += *buf+1;
  }
#endif
}

/* -------------------------------------------------------------- */

static void putInfos (interp, buf, count)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *buf;			/* String containing short infos. */
     int         count;			/* Number of short infos in buf. */
{
  char        result[50];
  char       *data_type;
  char       *io_type;
  Tcl_DString result_dstring;
#if ADABAS_VERSION >= 62
  field_info *pi     = (field_info *) buf;
#else /* ADABAS_VERSION <= 61 */
  char dummy_mode    = CSP_INFO_MANDATORY;
  char dummy_io_type = CSP_INFO_OUTPUT;
  field_info61 *pi   = (field_info61 *) buf;
#endif

  /*
   * For every given shortinfo append a list to the result.
   */

  while (count-- > 0) {

    Tcl_DStringInit (&result_dstring);

    /*
     * First of all the datatype in the form of e.g. fixed(5,1)
     */

#if ADABAS_VERSION >= 62
    data_type = dataType2String (pi->data_type);
    if (pi->frac)
      sprintf (result, "%s(%d,%d)", data_type, pi->length, pi->frac);
    else
      sprintf (result, "%s(%d)", data_type, pi->length);
#else /* ADABAS_VERSION <= 61 */
    data_type = dataType2String (pi->d_type);
    if (pi->length)
      if (pi->var.s.frac)
	sprintf (result, "%s(%d,%d)", data_type,
		 (unsigned char) pi->length, pi->var.s.frac);
      else
	sprintf (result, "%s(%d)", data_type, (unsigned char) pi->length);
    else
      sprintf (result, "%s(%d)", data_type, pi->var.l.longiolen - 1);
#endif
    Tcl_DStringAppendElement (&result_dstring, result);

    /*
     * Append the modes (possibly a list!) to the result.
     */
    Tcl_DStringAppendElement (&result_dstring, "-mode");
#if ADABAS_VERSION >= 62
    Tcl_DStringStartSublist (&result_dstring);
    if (pi->mode & (1 << OT_MANDATORY))
      Tcl_DStringAppendElement (&result_dstring, "mandatory");
    if (pi->mode & (1 << OT_OPTIONAL))
      Tcl_DStringAppendElement (&result_dstring, "optional");
    if (pi->mode & (1 << OT_DEFAULT))
      Tcl_DStringAppendElement (&result_dstring, "default");
    if (pi->mode & (1 << OT_ESCAPE_CHAR))
      Tcl_DStringAppendElement (&result_dstring, "escape_char");
    Tcl_DStringEndSublist (&result_dstring);
#else /* ADABAS_VERSION <= 61 */
    switch (dummy_mode) {
    case CSP_INFO_MANDATORY:
      Tcl_DStringAppendElement (&result_dstring, "mandatory"); break;
    case CSP_INFO_OPTIONAL:
      Tcl_DStringAppendElement (&result_dstring, "optional");  break;
    case CSP_INFO_DEFAULT:
      Tcl_DStringAppendElement (&result_dstring, "default");   break;
    }
#endif
    /*
     * Append the ioType to the result.
     */
    Tcl_DStringAppendElement (&result_dstring, "-ioType");
#if ADABAS_VERSION >= 62
    switch (pi->io_type) {
    case IO_INPUT:  io_type = "input";  break;
    case IO_OUTPUT: io_type = "output"; break;
    case IO_INOUT:  io_type = "inout";  break;
#else /* ADABAS_VERSION <= 62 */
    switch (dummy_io_type) {
    case CSP_INFO_INPUT:  io_type = "input";  break;
    case CSP_INFO_OUTPUT: io_type = "output"; break;
    case CSP_INFO_INOUT:  io_type = "inout";  break;
#endif
    default:           io_type = "?";      break;
    }
    Tcl_DStringAppendElement (&result_dstring, io_type);

    /*
     * Append the io_length to the result.
     */
    Tcl_DStringAppendElement (&result_dstring, "-ioLen");
#if ADABAS_VERSION >= 62
    sprintf (result, "%d", pi->in_out_len);
#else /* ADABAS_VERSION <= 61 */
    sprintf (result, "%d", pi->length
	     ? pi->var.s.in_out_l
	     : pi->var.l.longiolen);
#endif
    Tcl_DStringAppendElement (&result_dstring, result);

    /*
     * And finally the buf_pos.
     */
    Tcl_DStringAppendElement (&result_dstring, "-bufPos");
#if ADABAS_VERSION >= 62
    sprintf (result, "%d", pi->bufpos);
#else /* ADABAS_VERSION <= 61 */
    sprintf (result, "%d", pi->var.s.bufpos);
#endif
    Tcl_DStringAppendElement (&result_dstring, result);

    /*
     * Finally copy the constructed shortinfo to the result as one list.
     */
    Tcl_AppendElement (interp, Tcl_DStringValue (&result_dstring));
    Tcl_DStringFree (&result_dstring);

    pi++;
  }      
}
