/*
 * AdabasRte.c --
 *
 *     This module is the layer above the runtime environment of
 *     Adabas D. Every used procedure of the rte (like sqlaconnect)
 *     has a correspondend procedure (e.g. AdabasRteConnect),
 *     which does some error checking and write its state into
 *     a structure of type AdabasRteInfo.
 *
 * Copyright (c) 1996-1997 Christian Krone. All rights reserved.
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Tcl itself.
 * See also licence.terms
 *
 * $Id: AdabasRte.c,v 1.37 1997/06/14 14:01:54 adabas Exp $
 */

#include <stdlib.h>
#include <string.h>
#include <tcl.h>

#include "adabas.h"
#include "sqlrte.h"
#include "AdabasPort.h"
#include "AdabasUtil.h"
#include "AdabasRte.h"

/*
 * On unix there exists a variable, which contains a version string.
 * Since this is missing in the Windows rte, it has to be emulated here.
 */

#ifdef __WIN32__
c40 SQLRTEVERSION;
#endif

/*
 * The cancel byte is the same for all connections to one rte, so it
 * can't be in the rte info, which is allocated for each connection.
 */

static boolean cancel;

/*
 * For a very short time (6.2.7) the user and dblang fields are
 * still in its old places, so here the are copied in/out.
 * They are not needed for 6.1.
 */

#if ADABAS_VERSION >= 62
static void SetOutXUserRecord _ANSI_ARGS_((xuser_record *userParams));
static void SetInXUserRecord  _ANSI_ARGS_((xuser_record *userParams));
#else /* ADABAS_VERSION <= 61 */
#define SetOutXUserRecord(userParams) {}
#define SetInXUserRecord(userParams) {}
#endif


/*
 *----------------------------------------------------------------------
 *
 * AdabasRteInit --
 *
 *      Initializes this module by calling some init procs of the
 *      Adabas D runtime environment. On windows systems it sets also
 *      the local variable rteversion.
 *
 * Results:
 *      The struct, the parameter vars points to, is initialized after
 *      a call of this function.
 *
 * Side effects:
 *      The Adabas D rte is initialized.
 *
 *----------------------------------------------------------------------
 */

void
AdabasRteInit ()
{
  c64     component;			/* Component name (Pascal like string). */
  int4    dontCare;			/* the name says all... */
  boolean ok;				/* output status parameter of rte calls. */

  /*
   * A call of 'sqlinit' to initialize the rte.
   */

  c2p (component,  "AdabasTcl", sizeof (component));
  cancel = 0;
  sqlinit (component, &cancel);

  /*
   * A call of 'sqlfinit' to be able to open a file for read.
   */

  sqlfinit (0, &dontCare, &ok);

#ifdef __WIN32__ /* This shouldn't be here... */
#if ADABAS_VERSION <= 61
  c2p (SQLRTEVERSION, "RTE 6.1.2", sizeof (SQLRTEVERSION));
#else /* ADABAS_VERSION >= 62 */
  c2p (SQLRTEVERSION, "RTE 6.2.8", sizeof (SQLRTEVERSION));
#endif
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasRteConnect --
 *
 *      Tries to connect to the database kernel.
 *
 * Results:
 *      The struct, the parameter vars points to, contains the information
 *      needed for further calls of functions from this module.
 *
 * Side effects:
 *      A connection to the Adabas D database kernel is established.
 *
 *----------------------------------------------------------------------
 */

int
AdabasRteConnect (interp, vars, serverdb, serv)
     Tcl_Interp    *interp;		/* Current interpreter. */
     AdabasRteInfo *vars;		/* Current RunTimeEnvironment. */
     char          *serverdb;		/* Serverdb to connect to. */
     int            serv;		/* Service to open (USER/UTILITY). */
{
  comm_error returncode;		/* Rte error enumeration. */
  errtext    errtext;			/* Error message (Pascal like string). */
  char       cErrtext[ERRTEXT+1];	/* Error message (null terminated). */

  /*
   * Open the connection to the given database with 'sqlaconnect'.
   */

  c2p (vars->dbnode, "",       sizeof (vars->dbnode));
  c2p (vars->dbname, serverdb, sizeof (vars->dbname));
  vars->reference = 0;
  vars->service   = serv;
  sqlaconnect (0, vars->dbnode, vars->dbname, vars->service,
#if ADABAS_VERSION >= 62
	       1, &vars->reference, &vars->sqlPacketSize, vars->sqlPacketList,
#else /* ADABAS_VERSION <= 61 */
	       &vars->reference, 
#endif
	       errtext, &returncode);
  if (returncode) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "connect to \"");
    AppendResult (interp, serverdb);
    AppendResult (interp, "\" failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasRteSend --
 *
 *      Sends the given packet to the database kernel and waits for
 *      an answer.
 *
 * Results:
 *      The packet, rcvPacket points to, will contain the answer
 *      from the database kernel.
 *
 * Side effects:
 *      depends on the command, that was send to the kernel.
 *
 *----------------------------------------------------------------------
 */

int
AdabasRteSend (interp, vars, sndPacket, rcvPacket)
     Tcl_Interp    *interp;		/* Current interpreter. */
     AdabasRteInfo *vars;		/* Current RunTimeEnvironment. */
     packet        *sndPacket;		/* Packet to send to the kernel. */
     packet       **rcvPacket;		/* Packet received from the kernel,
					 * this is identical to sndPacket
					 * for versions <= 6.1. */
{
#if ADABAS_VERSION >= 62
  int4       packetLength;		/* Length of packet to send. */
  int4       resPacketLength;		/* Length of packet received. */
#else /* ADABAS_VERSION <= 61 */
  int2       packetLength;		/* Length of packet to send. */
  int2       resPacketLength;		/* Length of packet received. */
#endif
  comm_error returncode;		/* Rte error enumeration. */
  errtext    errtext;			/* Error message (Pascal like string). */
  char       cErrtext[ERRTEXT+1];	/* Error message (null terminated). */

  /*
   * Send the constructed packet to the database with 'sqlarequest'.
   * but first of all evaluate the packet length.
   */

#if ADABAS_VERSION >= 62
  packetLength = sndPacket->header.varpart_len + sizeof (packet_header);
#else /* ADABAS_VERSION <= 61 */
  packetLength = PACKETHEAD
    + sndPacket->part1_length + sndPacket->part2_length;
#endif
  sqlarequest (vars->reference, sndPacket, packetLength,
	       errtext, &returncode);
  if (returncode) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "request failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }

  /*
   * Now wait for the answer of the database with 'sqlareceive'.
   */

  sqlareceive (vars->reference,
#if ADABAS_VERSION >= 62
	       (void **) rcvPacket,
#else /* ADABAS_VERSION <= 61 */
	       sndPacket,
#endif
	       &resPacketLength, errtext, &returncode);

#if ADABAS_VERSION <= 61
  *rcvPacket = sndPacket;
#endif

  if (returncode) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "receive failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasRteTermId --
 *
 *      Requests the terminal id.
 *
 * Results:
 *      The character array, result points to, will contain the term id.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
AdabasRteTermId (result)
     char *result;			/* Returning terminalId. */
{
  termid terminalid;

  sqltermid (terminalid);
  sprintf (result, "%-*.*s",
	   sizeof (terminalid), (int) sizeof (terminalid), terminalid);
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasRteVersion --
 *
 *      Requests the rte version.
 *
 * Results:
 *      The character array, result points to, will contain the version.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
AdabasRteVersion (rteVersion)
     char *rteVersion;			/* Returning version of RTE. */
{
  sprintf (rteVersion, "%-.*s", (int) sizeof (SQLRTEVERSION), SQLRTEVERSION);
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasRteCancel --
 *
 *      Cancels the current rte request.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The database kernel will (hopefully) terminate its current activity.
 *
 *----------------------------------------------------------------------
 */

void
AdabasRteCancel (vars)
     AdabasRteInfo *vars;		/* Current RunTimeEnvironment. */
{
  /*
   * Cancels the execution of the currently running sql statement
   * with 'sqlacancel'.
   */

  sqlacancel (vars->reference);
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasRteClose --
 *
 *      Closes the connection opened with AdabasRteConnect.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The connection to the database kernel is invalid.
 *
 *----------------------------------------------------------------------
 */

void
AdabasRteClose (vars)
     AdabasRteInfo *vars;		/* Current RunTimeEnvironment. */
{
  /*
   * Close the connection to the database with 'sqlarelease'.
   */

  sqlarelease (vars->reference);
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasXUserClear --
 *
 *      Clears the xuser file.
 *
 * Results:
 *      A valid Tcl result.
 *
 * Side effects:
 *      The xuser file is empty.
 *
 *----------------------------------------------------------------------
 */

int
AdabasXUserClear (interp)
     Tcl_Interp    *interp;		/* Current interpreter. */
{
  sqlclearuser ();
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasXUserOpen --
 *
 *      opens the xuser file for subsequent reading.
 *
 * Results:
 *      A valid Tcl result.
 *
 * Side effects:
 *      The xuser file can be read.
 *
 *----------------------------------------------------------------------
 */

int
AdabasXUserOpen (interp)
     Tcl_Interp    *interp;		/* Current interpreter. */
{
  boolean ok;				/* output status parameter of rte calls. */
  errtext errtext;			/* Error message (Pascal like string). */
  char    cErrtext[ERRTEXT+1];		/* Error message (null terminated). */

  sqlxuopenuser  (errtext, &ok);
  if (!ok) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "open of xuser file failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasXUserClose --
 *
 *      closes the (already opened) xuser file.
 *
 * Results:
 *      A valid Tcl result.
 *
 * Side effects:
 *      The xuser file cannot be read further.
 *
 *----------------------------------------------------------------------
 */

int
AdabasXUserClose (interp)
     Tcl_Interp    *interp;		/* Current interpreter. */
{
  boolean ok;				/* output status parameter of rte calls. */
  errtext errtext;			/* Error message (Pascal like string). */
  char    cErrtext[ERRTEXT+1];		/* Error message (null terminated). */

  sqlxucloseuser (errtext, &ok);
  if (!ok) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "close of xuser file failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasXUserIndex --
 *
 *      get an xuser entry at the given position.
 *
 * Results:
 *      A valid Tcl result. The struct, userParams points to, will contain
 *      the wanted xuser records.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasXUserIndex (interp, userIndex, userParams)
     Tcl_Interp   *interp;		/* Current interpreter. */
     int           userIndex;		/* index of user to get. */
     xuser_record *userParams;		/* returned user record. */
{
  boolean ok;				/* output status parameter of rte calls. */
  errtext errtext;			/* Error message (Pascal like string). */
  char    cErrtext[ERRTEXT+1];		/* Error message (null terminated). */

  memset (userParams, ' ', sizeof (*userParams));
  sqlindexuser ((int2) userIndex, userParams, errtext, &ok);
  if (!ok) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "index of xuser file failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }
  SetOutXUserRecord (userParams);
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasXUserPut --
 *
 *      add an xuser entry to the xuser file (with the specified user key).
 *
 * Results:
 *      A valid Tcl result.
 *
 * Side effects:
 *      The xuser file is changed accordingly.
 *
 *----------------------------------------------------------------------
 */

int
AdabasXUserPut (interp, userParams)
     Tcl_Interp   *interp;		/* Current interpreter. */
     xuser_record *userParams;		/* user record to write. */
{
  boolean ok;				/* output status parameter of rte calls. */
  errtext errtext;			/* Error message (Pascal like string). */
  char    cErrtext[ERRTEXT+1];		/* Error message (null terminated). */

  SetInXUserRecord (userParams);
  sqlputuser (userParams, errtext, &ok);
  if (!ok) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "writing of xuser file failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasXUserGet --
 *
 *      get an xuser entry with the given user key.
 *
 * Results:
 *      A valid Tcl result. The struct, userParams points to, will contain
 *      the wanted xuser records.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasXUserGet (interp, userKey, userParams)
     Tcl_Interp   *interp;		/* Current interpreter. */
     char         *userKey;		/* User key, to whom record belongs.
					 * If NULL, get the record of DEFAULT. */
     xuser_record *userParams;		/* returned user record. */
{
  boolean ok;				/* output status parameter of rte calls. */
  errtext errtext;			/* Error message (Pascal like string). */
  char    cErrtext[ERRTEXT+1];		/* Error message (null terminated). */

  /*
   * Get the user data of the given XUSER key.
   */

  memset (userParams, ' ', sizeof (*userParams));
  c2p (userParams->xu_key, userKey ? userKey : "DEFAULT",
       sizeof (userParams->xu_key));
  sqlgetuser (userParams, errtext, &ok);
  if (!ok) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "xuser of ");
    AppendResult (interp, userKey ? userKey : "default user");
    AppendResult (interp, " failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }
  SetOutXUserRecord (userParams);
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdabasXUserArg --
 *
 *      get an xuser entry depending on the command line arguments.
 *
 * Results:
 *      A valid Tcl result. The struct, userParams points to, will contain
 *      the wanted xuser records. If an explicit password was given on
 *      the command line, it is stored in the character array, givenPasswd
 *      points to.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
AdabasXUserArgs (interp, userParams, givenPasswd)
     Tcl_Interp   *interp;		/* Current interpreter. */
     xuser_record *userParams;		/* returned user record. */
     char         *givenPasswd;		/* password from command line.*/
{
  SQL_PASSWORD        password;		/* crypted password. */
  struct args_options options;		/* options record (with lots of stuff). */
  xuser_set           xusertype;	/* not used really. */
  boolean             ok;		/* output status parameter of rte calls. */
  errtext             errtext;		/* Error message (Pascal like string). */
  char                cErrtext[ERRTEXT+1]; /* Error message (null terminated). */
  int                 orgArgc;		/* original contents of argc. */
  char              **orgArgv;		/* original contents of argv. */
  int                 ix;		/* loop var for converting password. */
  char                evalString[80];	/* buffer to store the Tcl command. */

  /*
   * We must put the arguments into the global variables argc and argv
   * outselves. And we better store the original...
   */

  strcpy (evalString, "linsert $argv 0 $argv0");
  if (Tcl_GlobalEval (interp, evalString) == TCL_ERROR) {
    SetResult (interp, "Variable \"argv\" or \"argv0\" not set");
    return TCL_ERROR;
  }
  orgArgc = RTE_ARGC;
  orgArgv = RTE_ARGV;
  if (Tcl_SplitList (interp, interp->result,
		     &RTE_ARGC, &RTE_ARGV) == TCL_ERROR) {
    return TCL_ERROR;
  }
  Tcl_ResetResult (interp);

  memset (userParams, ' ', sizeof (*userParams));
  *givenPasswd = '\0';

  /*
   * Get the user data, that was specified on the command line,
   * in the environment variables or in the xuser default entry.
   */

  options.opt_component = SQL_QUERY;
  sqlarg3 (userParams, password, &options, xusertype, errtext, &ok);

  /*
   * Do anything unhappen on the global level.
   */

  ckfree ((char *) RTE_ARGV);
  RTE_ARGC = orgArgc;
  RTE_ARGV = orgArgv;

  if (!ok) {
    p2c (cErrtext, errtext, sizeof (errtext));
    AppendResult (interp, "xuser of program arguments failed: ");
    AppendResult (interp, cErrtext);
    return TCL_ERROR;
  }

  /*
   * If a password was given on the command line (with -u user,passwd)
   * sqlarg3 gives it via its parameter password and we have to give
   * it to the caller ourself.
   */

  for (ix = 0; ix < sizeof (password); ix++) {
    if (password[ix] != ' ') {
      p2c (givenPasswd, password, sizeof (password));
      ix = sizeof (password); /* terminate loop. */
    }
  }

  SetOutXUserRecord (userParams);
  return TCL_OK;
}

#if ADABAS_VERSION >= 62

/*
 *----------------------------------------------------------------------
 *
 * SetOutXUserRecord --
 *
 *      convert a xuser record from 6.2.7 into the normal form.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static void
SetOutXUserRecord (userParams)
     xuser_record *userParams;
{
  if (*userParams->xu_user == ' ' && *userParams->xu_user61 != ' ') {
    memset (userParams->xu_user,   ' ', sizeof (userParams->xu_user));
    memcpy (userParams->xu_user, userParams->xu_user61,
	    sizeof (userParams->xu_user61));
  }
  if (*userParams->xu_dblang == ' ' && *userParams->xu_dblang61 != ' ') {
    memset (userParams->xu_dblang, ' ', sizeof (userParams->xu_dblang));
    memcpy (userParams->xu_dblang, userParams->xu_dblang61,
	    sizeof (userParams->xu_dblang61));
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SetInXUserRecord --
 *
 *      convert a xuser record from normal form into 6.2.7.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static void
SetInXUserRecord (userParams)
     xuser_record *userParams;
{
  memcpy (userParams->xu_user61, userParams->xu_user,
	  sizeof (userParams->xu_user61));
  memcpy (userParams->xu_dblang61, userParams->xu_dblang,
	  sizeof (userParams->xu_dblang61));
}

#endif /* ADABAS_VERSION >= 62 */
