/*
 * AdabasSend.h --
 *
 *      This module contains procedures, which opens a connection (logon)
 *      to the database server, and sends commands to it.
 *
 * Copyright (c) 1996-1997 Christian Krone. All rights reserved.
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Tcl itself.
 * See also licence.terms
 *
 *  $Id: AdabasSend.c,v 1.15 1997/06/30 21:45:32 adabas Exp $
 */

/*
 * Import interface (should move into AdabasTclInt.h someday)
 */

#include <stdlib.h>
#include <string.h>
#include <tcl.h>

#include "adabas.h"
#include "sqlrte.h"
#include "AdabasPort.h"
#include "AdabasUtil.h"
#include "AdabasRte.h"
#include "AdabasPacket.h"
#include "AdabasSend.h"
#include "AdabasApi.h"
#include "AdabasFormat.h"
#include "AdabasSql.h"
#include "AdabasLong.h"

/* ----------------- Local variables ---------------------------- */
static partKind partKinds[] = {
  pkCommand,
  pkParsid,
  pkResultcount,
  pkData,
  pkLongdata,
  pkNil /* termination flag */
};

static Tcl_HashTable logonHash;

#ifdef HAS_TCL_OBJECTS
Tcl_Obj *AdabasMessagesObj;
Tcl_Obj *AdamsgVersionObj;
Tcl_Obj *AdamsgNullvalueObj;
Tcl_Obj *AdamsgSpecialnullObj;
Tcl_Obj *AdamsgTracefileObj;
Tcl_Obj *AdamsgHandleObj;
Tcl_Obj *AdamsgErrorposObj;
Tcl_Obj *AdamsgErrortxtObj;
Tcl_Obj *AdamsgRowsObj;
Tcl_Obj *AdamsgIntoVarsObj;
Tcl_Obj *AdamsgRcObj;
Tcl_Obj *AdamsgColtypesObj;
Tcl_Obj *AdamsgCollengthsObj;
Tcl_Obj *AdamsgColprecsObj;
Tcl_Obj *AdamsgColscalesObj;
#endif /* HAS_TCL_OBJECTS */

/* ----------------- Local procedure headings ------------------- */

/*
 * All the procedures, that are used to fill a packet, have the packet
 * pointer as parameter and are therefore version dependend...
 */

#if ADABAS_VERSION >= 62
static void fillPacketHeader _ANSI_ARGS_((packet_header *packetHead,
					  int size, int swapKind,
					  char *application));
static void fillSegmentHeader _ANSI_ARGS_((packet_header *packetHead,
					   segment_header *segmentHead,
					   AdabasPacketInfo *options));
static int fillPart _ANSI_ARGS_((Tcl_Interp *interp, packet_header *packetHead,
				 segment_header *segmentHead, part **part,
				 int kind, AdabasPartInfo *partInfo));

#else /* ADABAS_VERSION <= 61 */
static void fillPacketHeader _ANSI_ARGS_((packet *packetHead, int swapKind,
					  char *application));
static void fillSegmentHeader _ANSI_ARGS_((packet *packetHead,
					   AdabasPacketInfo *options));
static int fillPart _ANSI_ARGS_((Tcl_Interp *interp, packet *packetHead,
				 partKind kind, AdabasPartInfo *partInfo,
				 int massCmd));
#endif

#ifdef HAS_TCL_OBJECTS
static void initAdamsgObjs ();
#endif /* HAS_TCL_OBJECTS */

#ifdef HAS_TCL_OBJECTS
static int logonSetFromAny _ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *objPtr));
static void logonUpdateString _ANSI_ARGS_((Tcl_Obj *objPtr));
static void logonDupInternalRep _ANSI_ARGS_((Tcl_Obj *srcPtr, Tcl_Obj *dupPtr));
Tcl_ObjType LogonObjType;
#endif

/* ----------------- Exported procedures ------------------------- */

void
AdabasSendInit (interp)
     Tcl_Interp  *interp;		/* Current interpreter. */
{
  AdabasRteInit ();

  Tcl_InitHashTable (&logonHash, TCL_STRING_KEYS);
#ifdef HAS_TCL_OBJECTS
  initAdamsgObjs ();

  /*
   * The logon handle parameter will become its own Tcl_ObjType,
   * so that most (if not all) of the hash table lookups can be omitted.
   */

  LogonObjType.name             = "cursor";
  LogonObjType.freeIntRepProc   = NULL;
  LogonObjType.dupIntRepProc    = &logonDupInternalRep;
  LogonObjType.updateStringProc = &logonUpdateString;
  LogonObjType.setFromAnyProc   = &logonSetFromAny;
  Tcl_RegisterObjType (&LogonObjType);
#endif /* HAS_TCL_OBJECTS */
}

/* -------------------------------------------------------------- */

int
AdabasOpenConnection (interp, connService, serverdb, logonName, logonHandle)
     Tcl_Interp  *interp;		/* Current interpreter. */
     int          connService;		/* User or utility service? */
     char        *serverdb;		/* Name of serverdb to connect to. */
     char        *logonName;		/* Name of logon handle. */
     AdabasInfo **logonHandle;		/* Created logon handle. */
{
  AdabasInfo    *vars;
  Tcl_HashEntry *newEntry;
  int            added;
  int            c;

  if (!serverdb || !*serverdb) {
    serverdb = Tcl_GetVar2 (interp, "env", "SERVERDB", TCL_GLOBAL_ONLY);
  }
  if (!serverdb || !*serverdb) {
    serverdb = Tcl_GetVar2 (interp, "env", "DBNAME",   TCL_GLOBAL_ONLY);
  }
  if (!serverdb || !*serverdb) {
    SetResult (interp, "Neither -serverdb specified nor env(SERVERDB) set");
    return TCL_ERROR;
  }

  /*
   * Initialize the database runtime environment and allocate the client_data.
   */
  if (!(vars = (AdabasInfo *) ckalloc (sizeof (AdabasInfo)))) {
    SetResult (interp, "Initialization failed due to lack of memory");
    return TCL_ERROR;
  }
  *logonHandle = vars;

  AdabasPacketInit (&vars->packetInfo);
#if ADABAS_VERSION <= 61
  memset (vars->sqlPacket.rte_header, 0, sizeof (rte_header));
#endif
  vars->partInfos[ pkNil         ].subst = encNil;
  vars->partInfos[ pkNil         ].name  = "???";
  vars->partInfos[ pkCommand     ].subst = encNil;
  vars->partInfos[ pkCommand     ].name  = "command";
  vars->partInfos[ pkData        ].subst = encEscape;
  vars->partInfos[ pkData        ].name  = "data";
  vars->partInfos[ pkLongdata    ].subst = encNil;
  vars->partInfos[ pkLongdata    ].name  = "longdata";
  vars->partInfos[ pkResultcount ].subst = encNil;
  vars->partInfos[ pkResultcount ].name  = "resultcount";
  vars->partInfos[ pkParsid      ].subst = encHex;
  vars->partInfos[ pkParsid      ].name  = "parsid";

  /*
   * Now open the connection to the given database.
   */
  if (AdabasRteConnect (interp, &vars->rteInfo,
			serverdb, connService) == TCL_ERROR) {
    ckfree ((char *) vars);
    return TCL_ERROR;
  }

  /*
   * Evaluate the name of the command to create:
   * If a name is given, use it, else take the name of the serverdb.
   * Append ' #n' with n a small int, if another command with that name
   * already exists.
   */

  if (*logonName) {
    strcpy  (vars->logonName, logonName);
  } else {
    strncpy (vars->logonName, serverdb, DBNAME);
  }

  c     = 1;
  added = 0;
  while (!added) {
    newEntry = Tcl_CreateHashEntry (&logonHash, vars->logonName, &added);
    if (!added) {
      sprintf (vars->logonName, "%.*s #%d", DBNAME, serverdb, ++c);
    }
  }

  /*
   * Register the logon handle under the computed name.
   */

  Tcl_SetHashValue (newEntry, vars);

  /*
   * Don't forget to set the magic entry.
   */

  vars->magic = LOGON_MAGIC;
  return TCL_OK;
}

/* -------------------------------------------------------------- */

int
AdabasLogonHandle (interp, logonName, logon)
     Tcl_Interp  *interp;		/* Current interpreter. */
     char        *logonName;		/* Name to look for. */
     AdabasInfo **logon;		/* Returned Login. */
{
  Tcl_HashEntry *hashEntry;

  if (!(hashEntry = Tcl_FindHashEntry (&logonHash, logonName))) {
    AppendResult (interp, "Invalid logonHandle \"");
    AppendResult (interp, logonName);
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }
  *logon = (AdabasInfo *) Tcl_GetHashValue (hashEntry);

  return AdabasCheckLogon (interp, *logon);
}

/* -------------------------------------------------------------- */

int AdabasCheckLogon (interp, logon)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *logon;			/* Login handle. */
{
  if (!logon) {
    SetResult (interp, "Invalid logonHandle (NULL)");
    return TCL_ERROR;
  }
  if (logon->magic != LOGON_MAGIC) {
    if (interp) {
      SetResult (interp, "Invalid logonHandle (disconnected)");
    }
    return TCL_ERROR;
  }
  return TCL_OK;
}

/* ------------------------------------------------------------------*/

int
AdabasCloseConnection (interp, logonName, vars)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *logonName;		/* Name of logon handle to delete. */
     AdabasInfo *vars;			/* Logon handle to delete. */
{
  Tcl_HashEntry *hashEntry;

  /*
   * Close the connection to the database.
   */

  AdabasRteClose (&vars->rteInfo);
  vars->magic = 0;
  ckfree ((char *) vars);

  /*
   * Unregister this logon handle.
   */

  if ((hashEntry = Tcl_FindHashEntry (&logonHash, logonName))) {
    Tcl_DeleteHashEntry (hashEntry);
  }
  return TCL_OK;
}

/* ------------------------------------------------------------------*/

#if ADABAS_VERSION >=62
static void
fillPacketHeader (head, size, swapKind, application)
     packet_header *head;		/* Header of packet to be filled. */
     int            size;		/* Size of packet. */
     int            swapKind;		/* Swap kind (see adabas.h). */
     char          *application;	/* Application. */
{
  head->mess_code    = 0; /*is_ascii*/;
  head->mess_swap    = swapKind;
  head->varpart_size = size - sizeof (packet_header);
  head->varpart_len  = 0;
  head->no_of_segm   = 0;
  memcpy (head->appl_version, "62080",     5);
  memcpy (head->application,  application, 3);
}
#else /* ADABAS_VERSION <= 61 */
static void
fillPacketHeader (head, swapKind, application)
     packet *head;			/* Packet with header to be filled. */
     int     swapKind;			/* Swap kind (see adabas.h). */
     char   *application;		/* Application. */
{
  head->mess_code_type = 0; /*is_ascii*/;
  head->mess_swap_kind = swapKind;
  head->return_code    = 0;
  head->error_code     = 0;
  head->part1_length   = 0;
  head->part2_length   = 0;
  memcpy (head->senderid,   "61115",     5);
  memcpy (head->senderid+5, application, 3);
}
#endif

/* ------------------------------------------------------------------*/

#if ADABAS_VERSION >=62
static void
fillSegmentHeader (packetHead, segmentHead, options)
     packet_header    *packetHead;	/* Header of packet. */
     segment_header   *segmentHead;	/* Header of segment to be filled. */
     AdabasPacketInfo *options;		/* Current packet options. */
{
  segmentHead->kind.cmd.segm_len            = sizeof (segment_header);
  segmentHead->kind.cmd.segm_offset         = 0;
  segmentHead->kind.cmd.no_of_parts         = 0;
  segmentHead->kind.cmd.own_index           = 1;
  segmentHead->kind.cmd.segm_kind           = SK_CMD;
  segmentHead->kind.cmd.mess_type           = options->messType;
  segmentHead->kind.cmd.sqlmode             = options->sqlMode;
  segmentHead->kind.cmd.producer            = options->producer;
  segmentHead->kind.cmd.commit_immediately  = options->commitImmediately;
  segmentHead->kind.cmd.ignore_costwarning  = 0;
  segmentHead->kind.cmd.prepare             = 0;
  segmentHead->kind.cmd.with_info           = options->withInfo;
  segmentHead->kind.cmd.mass_cmd            = options->massCmd;
  segmentHead->kind.cmd.parsing_again       = 0;

  packetHead->varpart_len += sizeof (segment_header);
  packetHead->no_of_segm  += 1;
}
#else /* ADABAS_VERSION <= 61 */
static void
fillSegmentHeader (packetHead, options)
     packet           *packetHead;	/* Packet with segment to be filled. */
     AdabasPacketInfo *options;		/* Current packet options. */
{
  packetHead->mess_type = options->messType;
  if (options->commitImmediately && packetHead->mess_type == M_DBS) {
    packetHead->mess_type = M_DBSCOMMIT;
  }
  if (options->withInfo) {
    switch (packetHead->mess_type) {
    case M_DBS:
      packetHead->mess_type = M_DBSINFO;
      break;
    case M_DBSCOMMIT:
      packetHead->mess_type = M_DBSINFOCOMMIT;
      break;
    }
  }
  switch (options->sqlMode) {
  case SQLM_ADABAS:
    switch (packetHead->mess_type) {
    case M_DBS: case M_DBSINFO: case M_DBSCOMMIT: case M_DBSINFOCOMMIT:
      packetHead->mess_type = M_DBSADABAS;   break;
    case M_PARSE:
      packetHead->mess_type = M_PARSEADABAS; break;
    }
    break;
  case SQLM_ANSI:
    switch (packetHead->mess_type) {
    case M_DBS: case M_DBSINFO: case M_DBSCOMMIT: case M_DBSINFOCOMMIT:
      packetHead->mess_type = M_DBSANSI;   break;
    case M_PARSE:
      packetHead->mess_type = M_PARSEANSI; break;
    }
    break;
  case SQLM_DB2:
    switch (packetHead->mess_type) {
    case M_DBS: case M_DBSINFO: case M_DBSCOMMIT: case M_DBSINFOCOMMIT:
      packetHead->mess_type = M_DBSDB2;   break;
    case M_PARSE:
      packetHead->mess_type = M_PARSEDB2; break;
    }
    break;
  case SQLM_ORACLE:
    switch (packetHead->mess_type) {
    case M_DBS: case M_DBSINFO: case M_DBSCOMMIT: case M_DBSINFOCOMMIT:
      packetHead->mess_type = M_DBSORACLE;   break;
    case M_PARSE:
      packetHead->mess_type = M_PARSEORACLE; break;
    }
    break;
  }
}
#endif

/* ------------------------------------------------------------------*/

#if ADABAS_VERSION >=62
static int
fillPart (interp, packetHead, segmentHead, currPart, kind, partInfo)
     Tcl_Interp     *interp;		/* Current interpreter. */
     packet_header  *packetHead;	/* Header of packet. */
     segment_header *segmentHead;	/* Header of segment. */
     part          **currPart;		/* Part to be filled. */
     int             kind;		/* Part kind (see adabas.h). */
     AdabasPartInfo *partInfo;		/* Current part options. */
{
  part_header *partHead;
  int4         partLength;
  int4         available;
  Tcl_DString  substitute;
  num_error    numError;

  if (partInfo->subst != encNil && partInfo->buf) {
    if ((partInfo->bufLength = encodeSequence (interp, partInfo->subst,
					       partInfo->name, &substitute,
					       partInfo->buf, 1,
					       partInfo->bufLength)) < 0) {
      return TCL_ERROR;
    }
    partInfo->buf = Tcl_DStringValue (&substitute);
  }
  partHead   = &(*currPart)->part_header;
  available  = packetHead->varpart_size - segmentHead->kind.cmd.segm_len;
  partLength = ((partInfo->bufLength+7)/8)*8 + sizeof (part_header);

  if (partLength > available) {
    if (partInfo->subst != encNil) {
      Tcl_DStringFree (&substitute);
    }
    SetResult (interp, "too big packet");
    return TCL_ERROR;
  }

  partHead->part_kind   = kind;
  partHead->attributes  = 0;
  partHead->arg_count   = 1;
  partHead->segm_offset = 0;
  partHead->buf_len     = partInfo->bufLength;
  partHead->buf_size    = available - sizeof (part_header);

  if (kind == PK_RESULTCOUNT) {
    (*currPart)->buf[0] = 0;
    s41plint ((*currPart)->buf, 2, 18, 0, partInfo->bufLength, &numError);
    partHead->buf_len = sizeof (resnum);
    if (numError) {
      AppendResult (interp, "invalid resultcount \"");
      AppendResult (interp, partInfo->buf);
      AppendResult (interp, "\"");
      return TCL_ERROR;
    }
  } else if (partInfo->buf) {
    memcpy ((*currPart)->buf, partInfo->buf, partInfo->bufLength);
  }

  if (partInfo->subst != encNil && partInfo->buf) {
    Tcl_DStringFree (&substitute);
  }

  packetHead->varpart_len           += partLength;
  segmentHead->kind.cmd.segm_len    += partLength;
  segmentHead->kind.cmd.no_of_parts += 1;

  *currPart = (part *) (((char *) *currPart) + partLength);

  return TCL_OK;
}
#else /* ADABAS_VERSION <= 61 */
static int
fillPart (interp, packetHead, kind, partInfo, massCmd)
     Tcl_Interp     *interp;		/* Current interpreter. */
     packet         *packetHead;	/* Packet with part to be filled. */
     partKind        kind;		/* Part kind. */
     AdabasPartInfo *partInfo;		/* Current part options. */
     int             massCmd;		/* Is this a mass command? */
{
  int4        available;
  Tcl_DString substitute;
  int2        argCount;

  if (partInfo->subst != encNil && partInfo->buf) {
    if ((partInfo->bufLength = encodeSequence (interp, partInfo->subst,
					       partInfo->name, &substitute,
					       partInfo->buf, 1,
					       partInfo->bufLength)) < 0) {
      return TCL_ERROR;
    }
    partInfo->buf = Tcl_DStringValue (&substitute);
  }
  available  = sizeof (packetHead->var_part)
    - packetHead->part1_length - packetHead->part2_length;

  if (partInfo->bufLength > available) {
    if (partInfo->subst != encNil) Tcl_DStringFree (&substitute);
    SetResult (interp, "too big packet");
    return TCL_ERROR;
  }

  switch (kind) {
  case pkCommand:
    if (massCmd) {
      packetHead->var_part[0]  = 'M';
      packetHead->part1_length = 1;
    } else {
      packetHead->part1_length = 0;
    }
    if (partInfo->buf) {
      memcpy (packetHead->var_part + packetHead->part1_length,
	      partInfo->buf, partInfo->bufLength);
    }
    packetHead->part1_length += partInfo->bufLength;
    break;

  case pkData:
    if (partInfo->buf) {
      memcpy (packetHead->var_part
	      + packetHead->part1_length + packetHead->part2_length,
	      partInfo->buf, partInfo->bufLength);
    }
    packetHead->part2_length += partInfo->bufLength;
    break;

  case pkResultcount:
    argCount = (int2) partInfo->bufLength;
    if (partInfo->buf) {
      memcpy (packetHead->var_part + packetHead->part1_length,
	      (char *) &argCount, 2);
    }
    packetHead->part2_length = 2;
    break;

  case pkLongdata:
    if (partInfo->buf) {
      memcpy (packetHead->var_part, partInfo->buf, partInfo->bufLength);
    }
    packetHead->part1_length = 0;
    packetHead->part2_length = partInfo->bufLength;
    break;

  case pkParsid:
    if (partInfo->bufLength != 12) {
      SetResult (interp, "invalid parsid length ???");
      return TCL_ERROR;
    }
    if (partInfo->buf)
      memcpy (packetHead->var_part+packetHead->part1_length,
	      partInfo->buf, 12);
    if (packetHead->part1_length) {
      packetHead->part2_length = 12;
    } else {
      packetHead->part1_length = 12;
    }
    break;
  default:
    if (partInfo->subst != encNil) {
      Tcl_DStringFree (&substitute);
    }
    SetResult (interp, "unknown part kind ???");
    return TCL_ERROR;
  }

  if (partInfo->subst != encNil && partInfo->buf) {
    Tcl_DStringFree (&substitute);
  }
  return TCL_OK;
}
#endif

/* ------------------------------------------------------------------*/

int
AdabasSendPacket (interp, vars)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Current logon. */
{
  int      retc;
  int      errp;
  char     errt[256];
#if ADABAS_VERSION >= 62
  segment *retSegm;
#endif

  /*
   * Send the constructed packet to the database and receive a reply.
   */

#if ADABAS_VERSION >= 62
  if (AdabasRteSend (interp, &vars->rteInfo,
		     vars->sndPacket, &vars->rcvPacket) == TCL_ERROR) {
    AdabasUnsetMsg (interp, AdamsgRc);
    AdabasUnsetMsg (interp, AdamsgErrorpos);
    AdabasUnsetMsg (interp, AdamsgErrortxt);
    return TCL_ERROR;
  }
  retSegm = &vars->rcvPacket->segm;
  if ((retc = retSegm->segm_header.kind.ret.returncode)) {
    errp = retSegm->segm_header.kind.ret.errorpos;
    /*
     * In some strange situations the database kernel returns no part at all.
     * Then we better don't apply onto the error message in part1.
     */
    if (retSegm->segm_header.kind.cmd.no_of_parts) {
      p2c (errt, retSegm->part.buf, retSegm->part.part_header.buf_len);
    } else {
      sprintf (errt, "Error %d", retSegm->segm_header.kind.ret.returncode);
    }
  }
#else /* ADABAS_VERSION <= 61 */
  if (AdabasRteSend (interp, &vars->rteInfo,
		     &vars->sqlPacket, &vars->rcvPacket) == TCL_ERROR) {
    AdabasUnsetMsg (interp, AdamsgRc);
    AdabasUnsetMsg (interp, AdamsgErrorpos);
    AdabasUnsetMsg (interp, AdamsgErrortxt);
    return TCL_ERROR;
  }
  if ((retc = vars->sqlPacket.return_code)) {
    p2c (errt, vars->sqlPacket.var_part, vars->sqlPacket.part1_length);
    errp = vars->sqlPacket.error_code;
  }
#endif

  if (retc) {
    SetIntResult (interp, retc);
    AdabasSetMsgInt (interp, AdamsgRc, retc);
    AdabasSetMsgInt (interp, AdamsgErrorpos, errp);
    AdabasSetMsgString (interp, AdamsgErrortxt, errt);
    return TCL_ERROR;
  }

  AdabasSetMsgInt (interp, AdamsgRc, 0);
  AdabasUnsetMsg (interp, AdamsgErrorpos);
  AdabasUnsetMsg (interp, AdamsgErrortxt);
  return TCL_OK;
}

/* ------------------------------------------------------------------*/

int
AdabasFillPacket (interp, vars, thisOptions)
     Tcl_Interp       *interp;		/* Current interpreter. */
     AdabasInfo       *vars;		/* Current logon. */
     AdabasPacketInfo *thisOptions;	/* Packet options for this command. */
{
  int partKind;
  int partI;

#if ADABAS_VERSION >= 62
  segment  *cmdSegm;
  part     *currPart;
  part_kind kind62;

  /*
   * Initialize the packet header.
   */
  vars->sndPacket = *vars->rteInfo.sqlPacketList;
  fillPacketHeader (&vars->sndPacket->header,
		    vars->rteInfo.sqlPacketSize,
		    vars->packetInfo.swapKind,
		    thisOptions->application);

  /*
   * Initialize the first (and only) command segment.
   */
  cmdSegm = (segment *) &vars->sndPacket->segm;
  fillSegmentHeader (&vars->sndPacket->header,
		     &cmdSegm->segm_header, thisOptions);

  /*
   * Fill the command part with the given string.
   */
  currPart = (part *) &cmdSegm->part;
  for (partKind = partKinds[partI = 0]; partKind;
       partKind = partKinds[++partI]) {
    if (vars->partInfos[partKind].bufLength) {
      switch (partKind) {
      case pkCommand:     kind62 = PK_COMMAND;       break;
      case pkData:        kind62 = PK_DATA;          break;
      case pkResultcount: kind62 = PK_RESULTCOUNT;   break;
      case pkLongdata:    kind62 = PK_LONGDATA;      break;
      case pkParsid:      kind62 = PK_PARSID;        break;
      default: 
	AppendResult (interp, "invalid part kind???");
	return TCL_ERROR;
      }
      if (fillPart (interp, &vars->sndPacket->header,
		    &cmdSegm->segm_header, &currPart,
		    kind62, vars->partInfos+partKind) == TCL_ERROR) {
	return TCL_ERROR;
      }
    }
  }

#else /* ADABAS_VERSION <= 61 */

  /*
   * Initialize the packet header.
   */
  fillPacketHeader  (&vars->sqlPacket, vars->packetInfo.swapKind,
		     thisOptions->application);
  fillSegmentHeader (&vars->sqlPacket, thisOptions);

  /*
   * Fill all the given parts.
   */
  for (partKind = partKinds[partI = 0]; partKind;
       partKind = partKinds[++partI]) {
    if (vars->partInfos[partKind].bufLength) {
      if (fillPart (interp, &vars->sqlPacket, partKind,
		    vars->partInfos+partKind,
		    thisOptions->massCmd) == TCL_ERROR) {
	return TCL_ERROR;
      }
    }
  }
#endif

  return TCL_OK;
}

/* ------------------------------------------------------------------*/

int
AdabasSend2PartPacket (interp, vars, part1Kind, part1, part1Int,
		       part2Kind, part2, part2Int)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Current logon. */
     int         part1Kind;		/* Part1: Part kind. */
     char       *part1;			/*        Buffer... */
     int         part1Int;		/*        ... or Number. */
     int         part2Kind;		/* Part2: Part kind. */
     char       *part2;			/*        Buffer... */
     int         part2Int;		/*        ... or Number. */
{
  int partI;

  for (partI = 0; partI < PART_KINDS; partI++) {
    vars->partInfos[ partI ].bufLength = 0;
  }
  switch (part1Kind) {
  case pkNil:
    break;
  case pkResultcount:
    vars->partInfos[part1Kind].buf       = "";
    vars->partInfos[part1Kind].bufLength = part1Int;
    break;
  default:
    vars->partInfos[part1Kind].buf       = part1;
    vars->partInfos[part1Kind].bufLength = strlen (part1);
  }
  switch (part2Kind) {
  case pkNil:
    break;
  case pkResultcount:
    vars->partInfos[part2Kind].buf       = "";
    vars->partInfos[part2Kind].bufLength = part2Int;
    break;
  default:
    vars->partInfos[part2Kind].buf       = part2;
    vars->partInfos[part2Kind].bufLength = strlen (part2);
  }

  if (AdabasFillPacket (interp, vars, &vars->packetInfo) == TCL_ERROR) {
    return TCL_ERROR;
  }
  return AdabasSendPacket (interp, vars);
}

/* ------------------------------------------------------------------*/

int
AdabasSendCmdPacket (interp, vars, withInfo, massCnt, command)
     Tcl_Interp *interp;		/* Current interpreter. */
     AdabasInfo *vars;			/* Current logon. */
     int         withInfo;		/* SQL, that returns results? */
     int         massCnt;		/* Number of rows for mass fetch. */
     char       *command;		/* SQL command to be send. */
{
  int ret;
  int orgWithInfo = vars->packetInfo.withInfo;
  int orgMassCmd  = vars->packetInfo.massCmd;

  vars->packetInfo.withInfo = withInfo;
  vars->packetInfo.massCmd  = massCnt > 1;

  ret = AdabasSend2PartPacket (interp, vars, pkCommand, command, 0,
			       massCnt > 1 ? pkResultcount : pkNil,
			       "", massCnt);

  vars->packetInfo.withInfo = orgWithInfo;
  vars->packetInfo.massCmd  = orgMassCmd;
  return ret;
}

/* ------------------------------------------------------------------*/

#ifdef HAS_TCL_OBJECTS

static void
initAdamsgObjs ()
{
  AdabasMessagesObj    = Tcl_NewStringObj ("adamsg",      -1);
  AdamsgVersionObj     = Tcl_NewStringObj ("version",     -1);
  AdamsgNullvalueObj   = Tcl_NewStringObj ("nullvalue",   -1);
  AdamsgSpecialnullObj = Tcl_NewStringObj ("specialnull", -1);
  AdamsgTracefileObj   = Tcl_NewStringObj ("tracefile",   -1);
  AdamsgHandleObj      = Tcl_NewStringObj ("handle",      -1);
  AdamsgErrorposObj    = Tcl_NewStringObj ("errorpos",    -1);
  AdamsgErrortxtObj    = Tcl_NewStringObj ("errortxt",    -1);
  AdamsgRowsObj        = Tcl_NewStringObj ("rows",        -1);
  AdamsgIntoVarsObj    = Tcl_NewStringObj ("intovars",    -1);
  AdamsgRcObj          = Tcl_NewStringObj ("rc",          -1);
  AdamsgColtypesObj    = Tcl_NewStringObj ("coltypes",    -1);
  AdamsgCollengthsObj  = Tcl_NewStringObj ("collengths",  -1);
  AdamsgColprecsObj    = Tcl_NewStringObj ("colprecs",    -1);
  AdamsgColscalesObj   = Tcl_NewStringObj ("colscales",   -1);
}

#else /* no HAS_TCL_OBJECTS */

int
AdabasGetMsgInt (interp, index, intPtr)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *index;			/* Entry name in adamsg array. */
     int        *intPtr;		/* resulting integer value. */
{
  char *str = Tcl_GetVar2 (interp, AdabasMessages, index, TCL_GLOBAL_ONLY);
  if (!str || Tcl_GetInt (interp, str, intPtr)) {
    return TCL_ERROR;
  }
  return TCL_OK;
}  

#endif /* HAS_TCL_OBJECTS */

#ifdef HAS_TCL_OBJECTS

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ObjType procedures for logonType --
 *
 *      The logon handle parameter will become its own Tcl_ObjType,
 *      so that most (if not all) of the hash table lookups can be omitted.
 *
 *----------------------------------------------------------------------
 */

static int
logonSetFromAny (interp, objPtr)
     Tcl_Interp *interp;		/* Used for error reporting if not NULL. */
     Tcl_Obj    *objPtr;		/* The object to convert. */
{
  char       *logonName;		/* Name of logon Handle. */
  AdabasInfo *vars;			/* logonHandle identified by objPtr. */

  logonName = Tcl_GetStringFromObj (objPtr, (int *) NULL);
  if (AdabasLogonHandle (interp, logonName, &vars) == TCL_ERROR) {
    AdabasTrace ("logon", logonName, "Invalid");
    return TCL_ERROR;
  }
  objPtr->internalRep.otherValuePtr = (VOID *) vars;
  objPtr->typePtr = &LogonObjType;
  AdabasTrace ("logon", vars->logonName, "Set string from");
  return TCL_OK;
}

static void
logonUpdateString (objPtr)
     Tcl_Obj *objPtr;			/* Object whose string rep to update. */
{
  AdabasInfo *vars;			/* logonHandle identified by objPtr. */

  vars           = (AdabasInfo  *) objPtr->internalRep.otherValuePtr;
  objPtr->length = strlen (vars->logonName);
  objPtr->bytes  = ckalloc((unsigned) objPtr->length + 1);
  strcpy (objPtr->bytes, vars->logonName);
  AdabasTrace ("logon", vars->logonName, "Update string of");
}

static void
logonDupInternalRep (srcPtr, dupPtr)
     Tcl_Obj *srcPtr;			/* Object with internal rep to copy. */
     Tcl_Obj *dupPtr;			/* Object with internal rep to set. */
{
    dupPtr->internalRep.otherValuePtr = srcPtr->internalRep.otherValuePtr;
    dupPtr->typePtr = &LogonObjType;
}

#endif
