/*
 * AdabasSql.c --
 *
 * This file creates the tcl commands of the extension (e.g. adalogon
 * and adasql). Most of the procedures do only a little option checking
 * and then call the corrensponding procedure in AdabasApi.c (e.g.
 * the tcl command adafetch is implemented by AdaFetchCmd, which ends
 * in a call of AdabasFetch from AdabasApi.c
 *
 * Copyright (c) 1996-1997 Christian Krone. All rights reserved.
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Tcl itself.
 * See also licence.terms
 *
 * $Id: AdabasSql.c,v 1.75 1997/06/29 11:06:14 adabas Exp $
 */

/*
 * Import interface (should move into AdabasTclInt.h someday)
 */

#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <tcl.h>

#include "adabas.h"
#include "AdabasPort.h"
#include "AdabasRte.h"
#include "AdabasPacket.h"
#include "AdabasUtil.h"
#include "AdabasSend.h"
#include "AdabasSql.h"
#include "AdabasApi.h"
#include "AdabasFormat.h"
#include "AdabasLong.h"

/*
 * Prototypes for procedures referenced only in this file. Since they are
 * used in the following ProcTable, they are declared before the variable
 * declarations.
 */

static int AdaLogonCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaLogoffCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaOpenCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaCloseCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaSqlCmd _ANSI_ARGS_((ClientData clientData,
	      Tcl_Interp *interp,  int objc, ConstObjPtr objv[]));
static int AdaCancelCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaFetchCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaColsCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaReadLongCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaWriteLongCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaAutoCommitCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaCommitCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaRollbackCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaUtilCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaUsageCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int AdaSpecialCmd _ANSI_ARGS_((ClientData clientData,
	       Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int fetchAll _ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *cursorObj,
	       int count, Tcl_Obj *tclCommand));

/*
 * For every tcl command implemented by this file, there is an entry
 * in the following (null terminated) table, which combines the name
 *  of the tcl command with its command procedure.
 */

typedef struct {
  char           *name;
  Tcl_ObjCmdProc *proc;
} ObjProcTable;

static ObjProcTable adabasObjProcs[] = {
  { "adalogon",     AdaLogonCmd       },
  { "adalogoff",    AdaLogoffCmd      },
  { "adaopen",      AdaOpenCmd        },
  { "adaclose",     AdaCloseCmd       },
  { "adasql",       AdaSqlCmd         },
  { "adacancel",    AdaCancelCmd      },
  { "adafetch",     AdaFetchCmd       },
  { "adacols",      AdaColsCmd        },
  { "adaautocom",   AdaAutoCommitCmd  },
  { "adacommit",    AdaCommitCmd      },
  { "adarollback",  AdaRollbackCmd    },
  { "adareadlong",  AdaReadLongCmd    },
  { "adawritelong", AdaWriteLongCmd   },
  { "adautil",      AdaUtilCmd        },
  { "adausage",     AdaUsageCmd       },
  { "adaspecial",   AdaSpecialCmd     },
  { (char *) NULL, (Tcl_ObjCmdProc *) NULL }
};

#ifdef HAS_TCL_OBJECTS
static int         cursorSetFromAny _ANSI_ARGS_((Tcl_Interp *interp,
		       Tcl_Obj *objPtr));
static void        cursorUpdateString _ANSI_ARGS_((Tcl_Obj *objPtr));
static void        cursorDupInternalRep _ANSI_ARGS_((Tcl_Obj *srcPtr,
		       Tcl_Obj *dupPtr));
static Tcl_ObjType CursorObjType =
{"cursor", NULL, &cursorDupInternalRep, &cursorUpdateString, &cursorSetFromAny};
#endif

#ifdef HAS_TCL_OBJECTS
#define ConvertToCursor(interp,objPtr) ((CursorInfo *) \
  (Tcl_ConvertToType ((interp), (objPtr), &CursorObjType) == TCL_OK \
   ? (objPtr)->internalRep.otherValuePtr : NULL))
#else
#define ConvertToCursor AdabasCursorHandle
#endif

/*
 *----------------------------------------------------------------------
 *
 * AdabasSqlInit --
 *
 *      Initializes this module by registering all the command procedures.
 *
 * Results:
 *      The return value is normally TCL_OK. If there are problems by
 *      evaluating the startup script adasql.tcl, TCL_ERROR will be returned.
 *
 * Side effects:
 *      The tcl commands of the AdabasTcl extension are defined in the given
 *      interpreter.
 *
 *----------------------------------------------------------------------
 */

int
AdabasSqlInit (interp)
     Tcl_Interp *interp;		/* Current interpreter. */
{
  ObjProcTable *pObj;			/* index pointer in proc table. */
  Tcl_DString   fileName;		/* full path name of startup script. */
  int           erg;			/* tcl result to return. */
  char         *adabasTclDir;		/* environment variable ADABASTCL_LIB. */

  /*
   * Define all the tcl commands, that are mentioned in proc table.
   */

  for (pObj = adabasObjProcs; pObj->name; pObj++) {
    Tcl_CreateObjCommand (interp, pObj->name, pObj->proc,
			  (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  }

#ifdef HAS_TCL_OBJ
  /*
   * The cursor handle parameter will become its own Tcl_ObjType,
   * so that most (if not all) of the hash table lookups can be omitted.
   */

  Tcl_RegisterObjType (&CursorObjType);
#endif

  /*
   * Evaluate the startup script '$ADABASTCL_DIR/adasql.tcl' and
   * return the result of that evaluation. If something went wrong here,
   * we must set result (and not objResult) of the interpreter, since
   * this is not called by an ObjCommand but by an init procedure...
   */

  adabasTclDir = Tcl_GetVar (interp, "adabastcl_library", TCL_GLOBAL_ONLY);
  if (!adabasTclDir) {
    interp->result = "Fatal error: Neither DBROOT nor ADABASTCL_DIR set";
    return TCL_ERROR;
  }

  Tcl_DStringInit (&fileName);
  Tcl_DStringAppend (&fileName, adabasTclDir,  -1);
  Tcl_DStringAppend (&fileName, "/adasql.tcl", -1);
  erg = Tcl_EvalFile (interp, Tcl_DStringValue (&fileName));
  Tcl_DStringFree (&fileName);
  return erg;
}

/*
 *----------------------------------------------------------------------
 *
 * AdaOpenCmd --
 *
 *      Implements the tcl command adaopen with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK; in this case interp->results
 *      will be set to the name of the fresh created cursorHandle. If there
 *      are malformed parameters or any problems while createing the cursor,
 *      TCL_ERROR will be returned.
 *
 * Side effects:
 *      The hashtable of cursorHandle will have an additional line, which
 *      contains a structure with information about the newly created cursor.
 *
 *----------------------------------------------------------------------
 */

static int
AdaOpenCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  AdabasInfo *vars;			/* logonHandle identified by 1. param. */
  CursorInfo *cursorVars;		/* the newly created cursorHandle. */

  /*
   * Check, if the one and only parameter is a valid logonHandle. Therefore
   * it must be found and it must be created by adalogon with -service user
   * or -service control.
   */

  if (objc != 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "logonHandle");
    return TCL_ERROR;
  }
  if (ConvertToLogon (interp, objv[1], &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (vars->packetInfo.messType != M_DBS) {
    AppendResult (interp, "Logon handle \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[1], (int *) NULL));
    AppendResult (interp, "\" must be user or control service");
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which open the cursor and allocates
   * some information about it.
   */

  if (!(cursorVars = AdabasOpen (interp, vars))) {
    return TCL_ERROR;
  }

  AdabasSetMsgString (interp, AdamsgHandle, cursorVars->cursorName);
  Tcl_ResetResult (interp);
  AppendResult (interp, cursorVars->cursorName);
#ifdef HAS_TCL_OBJECTS
  cursorSetFromAny (interp, Tcl_GetObjResult (interp));
#endif
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AdaCloseCmd --
 *
 *      Implements the tcl command adaclose with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursorName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      The hashtable of cursorHandle will have one fewer line, which used to
 *      contain a structure with information about the destroyed cursor.
 *
 *----------------------------------------------------------------------
 */

static int
AdaCloseCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  CursorInfo *cursorVars;		/* the cursorHandle to destroy. */

  /*
   * Check, if the one and only parameter is a valid cursorHandle.
   */

  if (objc != 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle");
    return TCL_ERROR;
  }
  if (!(cursorVars = ConvertToCursor (interp, objv[1]))) {
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which closes the cursor and
   * frees the information about it.
   */

  return AdabasClose (interp, cursorVars);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaSqlCmd --
 *
 *      Implements the tcl command adasql with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursorName
 *      or any malformed combination of options given, TCL_ERROR will be
 *      returned; in this case interp->results will be set to the error message.
 *      The same goes, if the database kernel returned a returncode <> 0.
 *
 * Side effects:
 *      Some entries in the tcl array adamsg will bet set.
 *
 *----------------------------------------------------------------------
 */

static int
AdaSqlCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  Tcl_Obj     *cursorObj;		/* Object parameter of cursor Handle. */
  CursorInfo  *cursorVars;		/* cursorHandle given by 1. parameter. */
  int          withParameter = 0;	/* -parameter option given? */
  char        *sqlMode       = NULL;     /* value of -sqlmode option. */
  char        *resultTable   = NULL;	/* value of -resulttable option. */
  char        *parsId;			/* ParsId to execute. */
  int          thisMessType  = M_DBS;	/* messType of this sql request. */
  char        *cmd;			/* SQL command (2. or 3. parameter). */
  int          currOption;		/* Index of current option. */
  int          missingParams;		/* Too few parameters? */
  static char *messTypes[] = 		/* The command kinds of adasql. */
  {"-command", "-parameter", "-parse", "-execute", "-drop", (char *) NULL};
  static char *options[] = 		/* The options of adasql. */
  {"-resulttable", "-sqlmode", (char *) NULL};

  /*
   * Check, if there are at least the cursorName and a SQL statement.
   */

  missingParams = objc < 3;
  if (!missingParams) {
    cursorObj = objv[1];
    if (Tcl_GetIndexFromObj ((Tcl_Interp *) NULL, objv[2], messTypes,
			     "messType", 0, &currOption) == TCL_OK) {
      if (objc < 4) {
	missingParams = 1;
      } else {
	objc--; objv++;
      }
    } else {
      currOption = -1;
    }
  }
  if (missingParams) {
    Tcl_WrongNumArgs (interp, 1, objv,
		      "cursorHandle ?kind? sqlStatement ?options?");
    return TCL_ERROR;
  }
  cmd = Tcl_GetStringFromObj (objv[2], (int *) NULL);

  if (currOption < 0 && *cmd == '-') {
    Tcl_GetIndexFromObj (interp, objv[2], messTypes,
			 "messType", 0, &currOption);
    return TCL_ERROR;
  }

  /*
   * If the second prameter starts with an hyphen, it is treated as description
   * of the messType of the SQL request. If not specified, -command is default.
   *  option      messType           comment
   *  -command    M_DBS	             pars/execute in one step (default);
   *  -parameter  M_PARSE/M_EXECUTE  parsing and executing separately;
   *  -parse      M_PARSE            only parsing, return a parsId;
   *  -execute    M_EXECUTE          only executing of the given parsId.
   */

  switch (currOption) {

  case 0: /* -command */
    thisMessType = M_DBS;
    break;

  case 1: /* -parameter */
    thisMessType  = M_PARSE;
    withParameter = 1;
    break;

  case 2: /* -parse */
    thisMessType = M_PARSE;
    break;

  case 3: /* -execute */
    if (objc != 3) {
      Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle -execute parsId");
      return TCL_ERROR;
    }
    if (!(cursorVars = ConvertToCursor (interp, cursorObj))) {
      return TCL_ERROR;
    }
    parsId = Tcl_GetStringFromObj (objv[2], (int *) NULL);
    return AdabasExecParsId (interp, cursorVars, parsId);

  case 4: /* -drop */
    if (objc != 3) {
      Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle -drop parsId");
      return TCL_ERROR;
    }
    if (!(cursorVars = ConvertToCursor (interp, cursorObj))) {
      return TCL_ERROR;
    }
    parsId = Tcl_GetStringFromObj (objv[2], (int *) NULL);
    return AdabasDropParsId (interp, cursorVars, parsId);
  }

  /*
   * Check, if the given cursorName is the name of a valid cursorHandle,
   * and parse the trailing option switches (-resulttable or -sqlmode).
   */

  if (!(cursorVars = ConvertToCursor (interp, cursorObj))) {
    return TCL_ERROR;
  }
  for (objv+=3, objc-=3; objc > 1; objv += 2, objc -= 2) {
    if (Tcl_GetIndexFromObj (interp, objv[0], options,
			     "option", 0, &currOption) == TCL_ERROR) {
      return TCL_ERROR;
    }

    switch (currOption) {
    case 0: /* -resulttable */
      resultTable = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 1: /* -sqlmode */
      sqlMode = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;
    }
  }
  if (objc > 0) {
    AppendResult (interp, "missing argument for switch: \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which handles the SQL request.
   */

  return AdabasSql (interp, cursorVars, thisMessType, withParameter,
		    cmd, resultTable, sqlMode);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaCancelCmd --
 *
 *      Implements the tcl command adacancel with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursorName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      The currently active SQL request will be aborted. Cursor variables
 *      will be reset.
 *
 *----------------------------------------------------------------------
 */

static int
AdaCancelCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  CursorInfo *cursorVars;		/* cursorHandle given by 1. parameter. */

  /*
   * Check, if the one and only parameter is a valid cursorHandle.
   */

  if (objc != 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle");
    return TCL_ERROR;
  }
  if (!(cursorVars = ConvertToCursor (interp, objv[1]))) {
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which cancels the request and resets
   * all information of the cursor.
   */

  return AdabasCancel (interp, cursorVars);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaFetchCmd --
 *
 *      Implements the tcl command adafetch with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursorName
 *      or any malformed combination of options given, TCL_ERROR will be
 *      returned; in this case interp->results will be set to the error message.
 *      The same goes, if the database kernel returned a returncode <> (0, 100).
 *
 *      If the returnvalue is TCL_OK, interp->result will point to a valid
 *      tcl list, that is the represantation of the fetched row. If the
 *      option -count was given with a value > 1, interp->result will point
 *      to a list of lists, each represanting a fetched row.
 *
 * Side effects:
 *      The cursor is moved, so that the next positional fetch will return a
 *      different row.
 *
 *----------------------------------------------------------------------
 */

static int
AdaFetchCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  Tcl_Obj     *cursorObj;		/* Second parameter (needed by fetchAll) */
  CursorInfo  *cursorVars;		/* cursorHandle given by 1. parameter. */
  int          currOption;		/* Index of current option. */
  char        *target   = "";		/* modified value of -position option. */
  int          massCnt    = 0;		/* value of -count option. */
  int          arraySet   = 0;		/* value of -array option. */
  char        *sqlMode    = NULL;	/* value of -sqlmode option. */
  Tcl_Obj     *commandObj = NULL;	/* value of -command option. */
  char         msg[80];			/* Used for error message. */
  static char *options[] = 		/* The options of adafetch. */
  {"-array", "-command", "-count", "-position", "-sqlmode", (char *) NULL};

  /*
   * Check, if the given cursorName is the name of a valid cursorHandle,
   * and if it is currently open for a fetch (i.e. the last SQL request
   * was a successful mass select).
   */

  if (objc < 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle ?options?");
    return TCL_ERROR;
  }
  cursorObj = objv[1];
  if (!(cursorVars = ConvertToCursor (interp, cursorObj))) {
    return TCL_ERROR;
  }
  if (!cursorVars->cntParams) {
    AppendResult (interp, "\"");
    AppendResult (interp, cursorVars->cursorName);
    AppendResult (interp, "\" not opened for fetch");
    return TCL_ERROR;
  }
  cursorVars->dataEnd = 0;

  /*
   * Parse the trailing option switches.
   */

  for (objv+=2, objc-=2; objc > 1; objv += 2, objc -= 2) {
    if (Tcl_GetIndexFromObj (interp, objv[0], options,
			     "option", 0, &currOption) == TCL_ERROR) {
      return TCL_ERROR;
    }
    switch (currOption) {
    case 0: /* -array */
      if (Tcl_GetBooleanFromObj (interp, objv[1], &arraySet) == TCL_ERROR) {
	return TCL_ERROR;
      }
      break;

    case 1: /* -command */
      commandObj = objv[1];
      break;

    case 2: /* -count */
      if (Tcl_GetIntFromObj (interp, objv[1], &massCnt) == TCL_ERROR) {
	return TCL_ERROR;
      }
      if (massCnt < 1) {
	sprintf (msg, "invalid -count %d", massCnt);
	AppendResult (interp, msg);
	return TCL_ERROR;
      }
      break;

    case 3: /* -position */
      target = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 4: /* -sqlmode */
      sqlMode = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;
    }
  }
  if (objc > 0) {
    AppendResult (interp, "missing argument for switch: \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }
  if (*target && massCnt > 0) {
    SetResult (interp, "Invalid combination of -count and -position");
    return TCL_ERROR;
  }
  if (*target && commandObj) {
    SetResult (interp, "Invalid combination of -command and -position");
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi dependend of the value of massCnt,
   * which fetches one or many rows and returns them in interp->result.
   */

  if (commandObj) {
    if (massCnt < 1) massCnt = 20;
    return fetchAll (interp, cursorObj, massCnt, commandObj);
  }
  if (!massCnt) {
    massCnt = 1;
  }
  return AdabasFetch (interp, cursorVars, target, massCnt, sqlMode, arraySet);
}

/*
 *----------------------------------------------------------------------
 *
 * fetchAll --
 *
 *      Implements the variant of adafetch with -command option.
 *      This procedure is nothing more than a wrapper around the Tcl script
 *      procedure _adafetch_all (defined in adasql.tcl).
 *
 * Results:
 *      See adasql.tcl.
 *
 * Side effects:
 *      See adasql.tcl.
 *
 *----------------------------------------------------------------------
 */

static int
fetchAll (interp, cursorObj, count, tclCommand)
     Tcl_Interp *interp;		/* Current interpreter. */
     Tcl_Obj    *cursorObj;		/* Object of current cursor. */
     int         count;			/* Count for mass fetches. */
     Tcl_Obj    *tclCommand;		/* Command to execute for each row. */
{
  Tcl_Obj *objv[5];			/* Array of objects to hold Tcl script. */
  int      ret;				/* Return value. */
#ifdef HAS_TCL_OBJECTS
  Tcl_Obj *scriptObj;			/* Script object. */
#else
  char     countBuf[30];		/* String buffer for count. */
  char    *command;			/* pointer to merged tcl script. */
#endif

  Tcl_ResetResult (interp);

#ifdef HAS_TCL_OBJECTS

  objv[0]   = Tcl_NewStringObj ("_adafetch_all", -1);
  objv[1]   = cursorObj;
  objv[2]   = Tcl_NewIntObj (count);
  objv[3]   = tclCommand;
  scriptObj = Tcl_NewListObj (4, objv);
  ret = Tcl_EvalObj (interp, scriptObj);

#else /* !defined HAS_TCL_OBJECTS */

  sprintf (countBuf, "%d", count);

  argv[0] = "_adafetch_all";
  argv[1] = cursorObj;
  argv[2] = countBuf;
  argv[3] = tclCommand;
  argv[4] = (char *) NULL;
  command = Tcl_Merge (4, argv);
  ret     = Tcl_Eval (interp, command);
  ckfree (command);

#endif /* HAS_TCL_OBJECTS */

  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * AdaColsCmd --
 *
 *      Implements the tcl command adacols with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursorName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *      If the returnvalue is TCL_OK, interp->result will point to a valid
 *      tcl list, containing the columnnames of the last select of this cursor.
 *
 * Side effects:
 *      Some entries in the tcl array adamsg will bet set.
 *
 *----------------------------------------------------------------------
 */

static int
AdaColsCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  CursorInfo *cursorVars;		/* cursorHandle given by 1. parameter. */

  /*
   * Check, if the one and only parameter is a valid cursorHandle.
   */

  if (objc != 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle");
    return TCL_ERROR;
  }
  if (!(cursorVars = ConvertToCursor (interp, objv[1]))) {
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which constructs the list of columnNames.
   */

  return AdabasColumns (interp, cursorVars);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaReadLongCmd --
 *
 *      Implements the tcl command adareadlong with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursorName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *      If the returnvalue is TCL_OK and a -filename option was given,
 *      interp->result will point to the number of bytes written.
 *      If the returnvalue is TCL_OK and no -filename option was given,
 *      interp->result will point to the value of the long column.
 *
 * Side effects:
 *      If the returnvalue is TCL_OK and a -filename option was given,
 *      the mentioned file contains the value of the long column.
 *
 *----------------------------------------------------------------------
 */

static int
AdaReadLongCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  CursorInfo  *cursorVars;		/* cursorHandle given by 1. parameter. */
  int          currOption;		/* Index of current option. */
  char        *tableName   = NULL;	/* value of -tablename option. */
  char        *columnName  = NULL;	/* value of -columnname option. */
  Tcl_Obj     *longDescr   = NULL;	/* value of -descriptor option. */
  char        *fileName    = NULL;	/* value of -filename option. */
  char        *whereCond   = NULL;	/* value of -where option. */
  DataEncoding encoding    = encNil;	/* modified value of -encoding option. */
  static char *options[]   = 		/* The options of adareadlong. */
  {"-column", "-descriptor", "-filename", "-encoding",
   "-table", "-where", (char *) NULL};

  /*
   * Check, if the first parameter is the name of a valid cursorHandle.
   */

  if (objc <= 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle options...");
    return TCL_ERROR;
  }
  if (!(cursorVars = ConvertToCursor (interp, objv[1]))) {
    return TCL_ERROR;
  }

  /*
   * Parse the trailing option switches.
   */

  for (objv+=2, objc-=2; objc > 1; objv += 2, objc -= 2) {
    if (Tcl_GetIndexFromObj (interp, objv[0], options,
			     "switch", 0, &currOption) == TCL_ERROR) {
      return TCL_ERROR;
    }
    switch (currOption) {
    case 0: /* -column */
      columnName = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 1: /* -descriptor */
      longDescr = objv[1];
      break;

    case 2: /* -filename */
      fileName = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 3: /* -encoding */
      encoding = scanEncoding (interp,
			       Tcl_GetStringFromObj (objv[1], (int *) NULL));
      if (encoding == encNil) {
	return TCL_ERROR;
      }
      break;

    case 4: /* -table */
      tableName = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 5: /* -where */
      whereCond = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;
    }
  }
  if (objc > 0) {
    AppendResult (interp, "missing argument for switch: \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }
  if ((fileName && encoding != encNil) ||
      !(( longDescr && !tableName && !columnName && !whereCond) ||
	(!longDescr &&  tableName &&  columnName))) {
    SetResult (interp, "invalid combination of switches.");
    return TCL_ERROR;
  }
  if (encoding == encNil) {
    encoding = encEscape;
  }

  /*
   * Call the procedure from AdabasApi, which generates the SQL requests
   * to read the long column and write it into the given file (if fileName
   * is <> NULL) or let interp->result point to it otherwise.
   */

  return AdabasReadLong (interp, cursorVars, longDescr, tableName, columnName,
			 whereCond, encoding, fileName);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaWriteLongCmd --
 *
 *      Implements the tcl command adawritelong with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid cursorName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *      If the returnvalue is TCL_OK and a -filename option was given,
 *      interp->result will point to the number of bytes written.
 *      If the returnvalue is TCL_OK and no -filename option was given,
 *      interp->result will point to the value of the long column.
 *
 * Side effects:
 *      If the returnvalue is TCL_OK and a -filename option was given,
 *      the mentioned file contains the value of the long column.
 *
 *----------------------------------------------------------------------
 */

static int
AdaWriteLongCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  CursorInfo  *cursorVars;		/* cursorHandle given by 1. parameter. */
  int          currOption;		/* Index of current option. */
  char        *tableName  = NULL;	/* value of -tablename option. */
  char        *columnName = NULL;	/* value of -columnname option. */
  char        *fileName   = NULL;	/* value of -filename option. */
  char        *longValue  = NULL;	/* value of -longvalue option. */
  char        *whereCond  = NULL;	/* value of -where option. */
  DataEncoding encoding   = encNil;	/* modified value of -encoding option. */
  static char *options[] = 		/* The options of adawritelong. */
  {"-column", "-encoding", "-filename", "-table",
   "-value", "-where", (char *) NULL};

  /*
   * Check, if the first parameter is the name of a valid cursorHandle.
   */

  if (objc <= 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle options...");
    return TCL_ERROR;
  }
  if (!(cursorVars = ConvertToCursor (interp, objv[1]))) {
    return TCL_ERROR;
  }

  /*
   * Parse the trailing option switches.
   */

  for (objv+=2, objc-=2; objc > 1; objv += 2, objc -= 2) {
    if (Tcl_GetIndexFromObj (interp, objv[0], options,
			     "switch", 0, &currOption) == TCL_ERROR) {
      return TCL_ERROR;
    }
    switch (currOption) {
    case 0: /* -column */
      columnName = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 1: /* -encoding */
      encoding = scanEncoding (interp,
			       Tcl_GetStringFromObj (objv[1], (int *) NULL));
      if (encoding == encNil) {
	return TCL_ERROR;
      }
      break;

    case 2: /* -filename */
      fileName = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 3: /* -table */
      tableName = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 4: /* -value */
      longValue = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 5: /* -where */
      whereCond = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;
    }
  }
  if (objc > 0) {
    AppendResult (interp, "missing argument for switch: \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }
  if ((fileName && encoding != encNil) || !tableName || !columnName) {
    SetResult (interp, "invalid combination of switches.");
    return TCL_ERROR;
  }
  if (!longValue && !fileName) {
    SetResult (interp, "either -filename or -value switch must be given");
    return TCL_ERROR;
  }
  if (encoding == encNil) {
    encoding = encEscape;
  }

  /*
   * Call the procedure from AdabasApi, which generates the SQL requests
   * to write the long column from the given file (if fileName
   * is <> NULL) or from the given longValue.
   */

  return AdabasWriteLong (interp, cursorVars, tableName, columnName,
			  whereCond, encoding, fileName, longValue);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaAutoCommitCmd --
 *
 *      Implements the tcl command adaautocom with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      or boolean value given, TCL_ERROR will be returned; in this case
 *      interp->results will be set to the error message.
 *
 * Side effects:
 *      The autocommit flag in future SQL requests will be set accordingly.
 *
 *----------------------------------------------------------------------
 */

static int
AdaAutoCommitCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  AdabasInfo *vars;			/* logonHandle identified by 1. param. */
  int         onOff;			/* modified value of 2. param. */

  /*
   * Check, if the first parameter is the name of a valid logonHandle
   * and if the second parameter is a valid boolean value.
   */

  if (objc != 3) {
    Tcl_WrongNumArgs (interp, 1, objv, "logonHandle onOff");
    return TCL_ERROR;
  }
  if (ConvertToLogon (interp, objv[1], &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (Tcl_GetBooleanFromObj (interp, objv[2], &onOff) == TCL_ERROR) {
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which switches the
   * autocommit feature of the database kernel.
   */

  return AdabasAutoCommit (interp, vars, onOff);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaCommitCmd --
 *
 *      Implements the tcl command adacommit with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      All SQL statements of the current logon are commited.
 *
 *----------------------------------------------------------------------
 */

static int
AdaCommitCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  AdabasInfo *vars;			/* logonHandle identified by 1. param. */

  /*
   * Check, if the only parameter is the name of a valid logonHandle.
   */

  if (objc != 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "logonHandle");
    return TCL_ERROR;
  }
  if (ConvertToLogon (interp, objv[1], &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which sends a COMMIT request
   * to the database kernel.
   */

  return AdabasCommit (interp, vars);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaRollbackCmd --
 *
 *      Implements the tcl command adarollback with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      All SQL statements of the current logon are rollbacked.
 *
 *----------------------------------------------------------------------
 */

static int
AdaRollbackCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  AdabasInfo *vars;			/* logonHandle identified by 1. param. */

  /*
   * Check, if the only parameter is the name of a valid logonHandle.
   */

  if (objc != 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "logonHandle");
    return TCL_ERROR;
  }
  if (ConvertToLogon (interp, objv[1], &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which sends a ROLLBACK request
   * to the database kernel.
   */

  return AdabasRollback (interp, vars);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaUtilCmd --
 *
 *      Implements the tcl command adautil with the 8.0 object interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
AdaUtilCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  AdabasInfo *vars;			/* logonHandle identified by 1. param. */
  char       *cmd;			/* Utility command to send (2. param). */

  /*
   * Check, if the first parameter is the name of a valid logonHandle.
   */
 
  if (objc != 3) {
    Tcl_WrongNumArgs (interp, 1, objv, "logonHandle command");
    return TCL_ERROR;
  }
  if (ConvertToLogon (interp, objv[1], &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which sends the given string as
   * utility command to the database kernel.
   */

  cmd = Tcl_GetStringFromObj (objv[2], (int *) NULL);
  return AdabasUtil (interp, vars, cmd);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaUsageCmd --
 *
 *      Implements the tcl command adausage with the 8.0 object interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
AdaUsageCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  AdabasInfo  *vars;			/* logonHandle identified by 1. param. */
  int          usageKind;		/* parsed value of 2. parameter. */
  int          currOption;		/* Index of current option. */
  int          objTypeLength;		/* Length of -objecttype option. */
  char        *objectType = NULL;	/* Value of -objecttype option. */
  char        *param1     = NULL;	/* 1. element of -parameters option. */
  char        *param2     = NULL;	/* 2. element of -parameters option. */
  char        *param3     = NULL;	/* 3. element of -parameters option. */
  int          parc;
  Tcl_Obj    **parv;
  static char *usageKinds[] = 		/* The usage kinds of adausage. */
  {"on", "add", "off", (char *) NULL};
  static char *options[] = 		/* The options of adausage on/add. */
  {"-objecttype", "-parameters", (char *) NULL};

  /*
   * Check, if the first parameter is the name of a valid logonHandle.
   */

  if (objc < 3) {
    Tcl_WrongNumArgs (interp, 1, objv, "logonHandle usageKind ?options...?");
    return TCL_ERROR;
  }
  if (ConvertToLogon (interp, objv[1], &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }

  /*
   * Parse the usage kind.
   */

  if (Tcl_GetIndexFromObj (interp, objv[2], usageKinds,
			   "usageKind", 0, &usageKind) == TCL_ERROR) {
    return TCL_ERROR;
  }

  switch (usageKind) {
  case 0: /* on */
  case 1: /* add */

    /*
     * Parse the trailing option switches.
     */

    for (objv+=3, objc-=3; objc > 1; objv += 2, objc -= 2) {
      if (Tcl_GetIndexFromObj (interp, objv[0], options,
			       "switch", 0, &currOption) == TCL_ERROR) {
	return TCL_ERROR;
      }
      switch (currOption) {
      case 0: /* -objecttype */
	objectType = Tcl_GetStringFromObj (objv[1], &objTypeLength);
	if (objTypeLength > 8) {
	  AppendResult (interp, "Objecttype \"");
	  AppendResult (interp, objectType);
	  AppendResult (interp, "\" too long (must be up to 8 chars)");	  
	  return TCL_ERROR;
	}
	break;

      case 1: /* -parameters */
	if (Tcl_ListObjGetElements (interp, objv[1],
				    &parc, &parv) == TCL_ERROR) {
	  return TCL_ERROR;
	}
	if (parc < 1 || parc > 3) {
	  SetResult (interp, "Only 1 to 3 parameters allowed");
	  return TCL_ERROR;
	}
	param1 = Tcl_GetStringFromObj (parv[0], (int *) NULL);
	if (parc >= 2) {
	  param2 = Tcl_GetStringFromObj (parv[1], (int *) NULL);
	}
	if (parc >= 3) {
	  param3 = Tcl_GetStringFromObj (parv[2], (int *) NULL);
	}
#ifndef HAS_TCL_OBJECTS
	ckfree ((char *) parv);
#endif
	break;
      }
    }
    if (objc > 0) {
      AppendResult (interp, "missing argument for switch: \"");
      AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
      AppendResult (interp, "\"");
      return TCL_ERROR;
    }
    break;

  case 2: /* off */
    if (objc != 3) {
      Tcl_WrongNumArgs (interp, 1, objv, "cursorHandle off");
      return TCL_ERROR;
    }
    break;
  }

  /*
   * Call the procedure from AdabasApi, which sends the given string as
   * utility command to the database kernel.
   */

  return AdabasUsage (interp, vars, usageKind,
		      objectType, param1, param2, param3);
}

/*
 *----------------------------------------------------------------------
 *
 * AdaSpecialCmd --
 *
 *      Implements the tcl command adaspecial with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
AdaSpecialCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  AdabasInfo *vars;			/* logonHandle identified by 1. param. */
  char       *currOption;		/* Option string (used for parsing). */
  int         ret;			/* Return value of send request. */
  int         thisMessType;		/* MessType (depends on 2. param) */
  int         count;			/* Value of count for switchlimit. */
  int         limit;			/* Value of limit for buflength. */
  char        data[200];		/* Buffer for switch values. */
  int         rc;			/* Numeric value of adamsg(rc). */

  /*
   * Check, if the first parameter is the name of a valid logonHandle.
   */

  if (objc < 3) {
    Tcl_WrongNumArgs (interp, 1, objv, "logonHandle command ?params...?");
    return TCL_ERROR;
  }
  if (ConvertToLogon (interp, objv[1], &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }

  /*
   * Scan the second parameter as messType; dependend on this interpret
   * the following parameters.
   */

  currOption = Tcl_GetStringFromObj (objv[2], (int *) NULL);
  switch ((thisMessType = scanMessType (interp, currOption))) {
  case M_BUFLENGTH:
    if (objc != 4) {
      Tcl_WrongNumArgs (interp, 1, objv, "logonHandle buflength limit");
      return TCL_ERROR;
    }
    if (Tcl_GetIntFromObj (interp, objv[3], &limit) == TCL_ERROR) {
      return TCL_ERROR;
    }
    if (limit <= 0 || limit > MAXINT2) {
      sprintf (data, "limit must be between 1 and %d", MAXINT2);
      AppendResult (interp, data);
      return TCL_ERROR;
    }
    sprintf (data, "\\%03o\\%03o", limit/256, limit%256);
    break;

  case M_HELLO:
  case M_MINBUF:
  case M_MAXBUF:
    if (objc != 3) {
      SetResult (interp, "no parameter allowed");
      return TCL_ERROR;
    }
    strcpy (data, " ");
    break;

  case M_SWITCH:
    if (objc != 5) {
      Tcl_WrongNumArgs (interp, 1, objv, "logonHandle switch trace debug");
      return TCL_ERROR;
    }
    sprintf (data, "%-20.20s%-20.20s",
	     Tcl_GetStringFromObj (objv[3], (int *) NULL),
	     Tcl_GetStringFromObj (objv[4], (int *) NULL));
    break;

  case M_SWITCHLIMIT:
    if (objc != 10) {
      Tcl_WrongNumArgs (interp, 1, objv, "logonHandle switch trace debug start-layer start-proc stop-layer stop-proc count");
      return TCL_ERROR;
    }
    if (Tcl_GetIntFromObj (interp, objv[9], &count) == TCL_ERROR) {
      return TCL_ERROR;
    }
    if (count <= 0 || count > MAXINT2) {
      sprintf (data, "count must be between 1 and %d", MAXINT2);
      AppendResult (interp, data);
      return TCL_ERROR;
    }
    sprintf (data,
	     "%-20.20s%-20.20s%-5.5s %-10.10s%-5.5s %-10.10s\\%03o\\%03o",
	     Tcl_GetStringFromObj (objv[3], (int *) NULL),
	     Tcl_GetStringFromObj (objv[4], (int *) NULL),
	     Tcl_GetStringFromObj (objv[5], (int *) NULL),
	     Tcl_GetStringFromObj (objv[6], (int *) NULL),
	     Tcl_GetStringFromObj (objv[7], (int *) NULL),
	     Tcl_GetStringFromObj (objv[8], (int *) NULL),
	     count/256, count%256);
    break;

  default:
    AppendResult (interp, "unknown special command \"");
    AppendResult (interp, currOption);
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which sends the given string as
   * special command to the database kernel.
   */

  ret = AdabasSpecial (interp, vars, thisMessType, data);

  /*
   * Check, if the kernel returned -5 (invalid messType).
   * In this case it seems to be a quick or fast kernel (where
   * this commands are not compiled in).
   */
  if (ret == TCL_ERROR
      && AdabasGetMsgInt (interp, AdamsgRc, &rc) == TCL_OK
      && rc == -5) {
    Tcl_ResetResult  (interp);
    AppendResult (interp, "special command \"");
    AppendResult (interp, currOption);
    AppendResult (interp, "\" only available for a slow kernel");
  }
  return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * AdaLogonCmd --
 *
 *      Implements the tcl command adalogon with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there are invalid options or
 *      the attempt to connect to the database fails for any reason,
 *      TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      There is a new logonHandle with an open database connection accessible
 *      under the returned name.
 *
 *----------------------------------------------------------------------
 */

static int
AdaLogonCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[];		/* Argument Objects. */
{
  char        *connectStr;		/* String (like "krischan,geheim"). */
  char        *serverdb  = NULL;	/* Value of the -serverdb option. */
  char        *sqlMode   = NULL;	/* Value of the -sqlmode option. */
  char        *locale    = NULL;	/* Value of the -locale option. */
  service      service   = SQL_USER;	/* Value of the -service option. */
  ApplKind     applKind  = UserAppl;	/* Scanned value of -service option. */
  int          isolation = -1;		/* Value of -isolation option as int. */
  int          currOption;		/* Index of current option. */
  char        *currParameter;		/* Value  string (used for parsing). */
  static char *options[] =		/* The options of adalogon. */
  {"-isolationlevel", "-locale", "-service", "-serverdb",
   "-sqlmode", (char *) NULL};

  if (objc < 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "connectStr ?options?");
    return TCL_ERROR;
  }
  connectStr = Tcl_GetStringFromObj (objv[1], (int *) NULL);

  for (objv+=2, objc-=2; objc > 1; objv += 2, objc -= 2) {
    if (Tcl_GetIndexFromObj (interp, objv[0], options,
			     "switch", 0, &currOption) == TCL_ERROR) {
      return TCL_ERROR;
    }
    switch (currOption) {
    case 0: /* -isolationlevel */
      if (Tcl_GetIntFromObj (interp, objv[1], &isolation) == TCL_ERROR) {
	return TCL_ERROR;
      }
      break;

    case 1: /* -locale */
      locale = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 2: /* -service */
      currParameter = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      if (!strcmp (currParameter, "control")) {
	applKind = ControlAppl;
      } else if (!strcmp (currParameter, "odbc")) {
	applKind = OdbcAppl;
      } else if ((service = scanService (interp, currParameter)) < 0) {
	Tcl_ResetResult (interp);	
	AppendResult (interp, "Unknown service \"");
	AppendResult (interp, currParameter);
	AppendResult (interp, "\": must be user, utility, control or odbc");
	return TCL_ERROR;
      }
      break;

    case 3: /* -serverdb */
      serverdb = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 4: /* -sqlmode */
      sqlMode = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;
    }
  }

  if (objc > 0) {
    AppendResult (interp, "missing argument for switch: \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }

  if (service == SQL_UTILITY && sqlMode) {
    SetResult (interp, "invalid combination of switches.");
    return TCL_ERROR;
  }

  /*
   * Now open the connection to adabas with the given parameter.
   */

  return AdabasLogon (interp, serverdb, connectStr, isolation, sqlMode,
		      service, locale, applKind) ? TCL_OK : TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * AdaLogoffCmd --
 *
 *      Implements the tcl command adalogoff with the pre8.0 string interface.
 *
 * Results:
 *      The return value is normally TCL_OK. If there is an invalid logonName
 *      given, TCL_ERROR will be returned; in this case interp->results will
 *      be set to the error message.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
AdaLogoffCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[];		/* Argument Objects. */
{
  AdabasInfo *vars;			/* logonHandle identified by 1. param. */

  /*
   * Check, if the only parameter is the name of a valid logonHandle.
   */

  if (objc != 2) {
    Tcl_WrongNumArgs (interp, 1, objv, "logonHandle");
    return TCL_ERROR;
  }
  if (ConvertToLogon (interp, objv[1], &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }

  /*
   * Call the procedure from AdabasApi, which closes the connection.
   */

  return AdabasLogoff (interp, vars);
}

#ifdef HAS_TCL_OBJECTS

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ObjType procedures for cursorType --
 *
 *      The cursor handle parameter will become its own Tcl_ObjType,
 *      so that most (if not all) of the hash table lookups can be omitted.
 *
 *----------------------------------------------------------------------
 */

static int
cursorSetFromAny (interp, objPtr)
     Tcl_Interp *interp;		/* Used for error reporting if not NULL. */
     Tcl_Obj    *objPtr;		/* The object to convert. */
{
  char        *cursorName;		/* Name of cursor Handle. */
  CursorInfo  *cursorVars;		/* cursorHandle identified by objPtr. */

  cursorName = Tcl_GetStringFromObj (objPtr, (int *) NULL);
  if (!(cursorVars = AdabasCursorHandle (interp, cursorName))) {
    AdabasTrace ("cursor", cursorName, "Invalid");
    if (interp) {
      Tcl_ResetResult (interp);
      AppendResult (interp, "No valid cursor handle: ");
      AppendResult (interp, cursorName);
    }
    return TCL_ERROR;
  }
  objPtr->internalRep.otherValuePtr = (VOID *) cursorVars;
  objPtr->typePtr = &CursorObjType;
  AdabasTrace ("cursor", cursorVars->cursorName, "Set string from");
  return TCL_OK;
}

static void
cursorUpdateString (objPtr)
     Tcl_Obj *objPtr;			/* Object whose string rep to update. */
{
  CursorInfo  *cursorVars;		/* cursorHandle identified by objPtr. */

  cursorVars     = (CursorInfo  *) objPtr->internalRep.otherValuePtr;
  objPtr->length = strlen (cursorVars->cursorName);
  objPtr->bytes  = ckalloc((unsigned) objPtr->length + 1);
  strcpy (objPtr->bytes, cursorVars->cursorName);
  AdabasTrace ("cursor" ,cursorVars->cursorName, "Update string of");
}

static void
cursorDupInternalRep (srcPtr, dupPtr)
     Tcl_Obj *srcPtr;			/* Object with internal rep to copy. */
     Tcl_Obj *dupPtr;			/* Object with internal rep to set. */
{
    dupPtr->internalRep.otherValuePtr = srcPtr->internalRep.otherValuePtr;
    dupPtr->typePtr = &CursorObjType;
}

#endif
