/*
 * AdabasTcl.c --
 *
 *      This module contains the main procedures for the executables
 *      adabastclsh and adabaswish, and the entry procedure for the
 *      shared library Adabastcl.
 *      It also defines the adabas command (mostly obsolete now).
 *
 * Copyright (c) 1996-1997 Christian Krone. All rights reserved.
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Tcl itself.
 * See also licence.terms
 *
 * $Id: AdabasTcl.c,v 1.94 1997/07/04 21:22:03 adabas Exp $
 */


/*
 * Import interface (should move into AdabasTclInt.h someday)
 */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <tcl.h>
#ifdef ADABAS_WISH
#include <tk.h>
#endif

#include "adabas.h"
#include "sqlrte.h"
#include "AdabasPort.h"
#include "AdabasUtil.h"
#include "AdabasRte.h"
#include "AdabasPacket.h"
#include "AdabasSend.h"
#include "AdabasApi.h"
#include "AdabasFormat.h"
#include "AdabasSql.h"
#include "AdabasLong.h"

/*
 * The following hack is stolen from ftp://ftp.smli.com/.../example.tar.gz
 * and is needed to get the shared library constructed under Windows.
 */

#if defined(__WIN32__)
#   define WIN32_LEAN_AND_MEAN
#   include "windows.h"
#   undef WIN32_LEAN_AND_MEAN

/*
 * VC++ has an alternate entry point called DllMain, so we need to rename
 * our entry point.
 */

#   if defined(_MSC_VER)
#       define EXPORT(a,b) __declspec(dllexport) a b
#       define DllEntryPoint DllMain
#   else
#       if defined(__BORLANDC__)
#           define EXPORT(a,b) a _export b
#       else
#           define EXPORT(a,b) a b
#       endif
#   endif
#else
#   define EXPORT(a,b) a b
#endif

/*
 * Exported procedure headings (exported to the environment, that is).
 */

#ifdef ADABAS_MODULE
EXTERN EXPORT(int,Adabastcl_Init)     _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN EXPORT(int,Adabastcl_SafeInit) _ANSI_ARGS_((Tcl_Interp *interp));
#else
extern int main _ANSI_ARGS_((int argc, char **argv));
#endif

/*
 * Local procedure headings.
 */

#ifndef ADABAS_MODULE
static int  Adabastcl_Init     _ANSI_ARGS_((Tcl_Interp *interp));
static int  Adabastcl_SafeInit _ANSI_ARGS_((Tcl_Interp *interp));
#endif

static int  AdabasCmd _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
		int objc, ConstObjPtr objv[]));
static int  AdabasOpenCmd _ANSI_ARGS_((ClientData dummy, Tcl_Interp *interp,
		int objc, ConstObjPtr objv[]));
static int  AdabasCloseCmd _ANSI_ARGS_((Tcl_Interp *interp, char *cmdName,
		AdabasInfo *vars));
static int  AdabasConnCmd _ANSI_ARGS_((ClientData clientData,
		Tcl_Interp *interp, int objc, ConstObjPtr objv[]));
static int  AdabasVersion _ANSI_ARGS_((ClientData dummy, Tcl_Interp *interp,
		int objc, ConstObjPtr objv[]));
static int  AdabasXUser _ANSI_ARGS_((ClientData dummy, Tcl_Interp *interp,
		int objc, ConstObjPtr objv[]));
static int  args2xuser _ANSI_ARGS_((Tcl_Interp *interp, int objc,
		ConstObjPtr objv[], xuser_record *userParams));
static void xuser2result _ANSI_ARGS_((Tcl_Interp *interp,
		xuser_record *userParams, char *givenPasswd));
static int  AdabasSendConfigure _ANSI_ARGS_((Tcl_Interp *interp,
		AdabasPacketInfo *vars, Tcl_Obj *optionName));
static int  scanPacketConfOpts _ANSI_ARGS_((Tcl_Interp *interp, char *expected,
		Tcl_Obj *optionName, int objc, ConstObjPtr objv[],
		AdabasPacketInfo *vars));

/*
 * All the procedures, that are used to fill a packet, have the packet
 * pointer as parameter and are therefore version dependend...
 */

#if ADABAS_VERSION >= 62
static void setResultSegment _ANSI_ARGS_((Tcl_Interp *interp,
		segment *retSegment));

#else /* ADABAS_VERSION <= 61 */
static void setResultSegment _ANSI_ARGS_((Tcl_Interp *interp,
		packet *retSegment, int sndMessType));
#endif

/* --------------------------------------------------------------- */

#ifdef ADABAS_TCLSH
int
main (argc, argv)
     int argc;				/* Number of arguments. */
     char **argv;			/* Argument strings. */
{
  Tcl_Main (argc, argv, Adabastcl_Init);
  return 0;
}
#endif

#ifdef ADABAS_WISH
int
main (argc, argv)
     int argc;				/* Number of arguments. */
     char **argv;			/* Argument strings. */
{
  Tk_Main (argc, argv, Adabastcl_Init);
  return 0;
}
#endif

#ifdef ADABAS_MODULE
EXPORT(int,Adabastcl_Init)(interp)
#else
static int Adabastcl_Init (interp)
#endif
     Tcl_Interp *interp;		/* Current interpreter. */
{
  char *adabasTclDir = getenv ("ADABASTCL_DIR");
  char *dbroot       = getenv ("DBROOT");
  Tcl_DString dbRootTcl;
  char  versionString[100];
  int   versionNumber;
  Tcl_RegExp regExp;
  char *start, *end;

#if defined ADABAS_TCLSH || defined ADABAS_WISH
  if (Tcl_Init(interp) == TCL_ERROR) {
    return TCL_ERROR;
  }
#endif

#ifdef ADABAS_WISH
  if (Tk_Init(interp) == TCL_ERROR) {
    return TCL_ERROR;
  }
#endif

#if 0 /* Endless loop, to terminate in debugger. */
  while (!i);
#endif

  /*
   * Experience has pointed out, that it is very important to talk to
   * a database server of the correct version; so here we check, if
   * ADABAS_VERSION is configured correctly...
   */

  AdabasRteVersion (versionString);
  regExp = Tcl_RegExpCompile (interp, "RTE *(([0-9])\\.([0-9])[0-9.]*)");
  if (Tcl_RegExpExec (interp, regExp, versionString, versionString) <= 0) {
    versionNumber = 0;
  } else {
    Tcl_RegExpRange (regExp, 2, &start, &end);
    versionNumber  = (*start-'0')*10;
    Tcl_RegExpRange (regExp, 3, &start, &end);
    versionNumber += (*start-'0');
  }
  if ((ADABAS_VERSION <= 61 && versionNumber > 61) ||
      (ADABAS_VERSION >= 62 && versionNumber < 62)) {
#if ADABAS_VERSION <= 61
    char *wanted = "upto version 6.1.1";
#else /* ADABAS_VERSION >= 62 */
    char *wanted = "of version 6.2 or above";
#endif
    char found[80];

    if (versionNumber) {
      Tcl_RegExpRange (regExp, 1, &start, &end);
      sprintf (found, "%.*s", end-start, start);
    } else {
      strcpy  (found, "unknown");
    }
    
    /*
     * It is importannt to write this error message into interp->result,
     * (and not into interp->objResult), since tcl expects it there...
     */

    Tcl_AppendResult (interp, "AdabasTcl was configured for a database ",
		      wanted, ", but found a server of version ",
		      found,  (char *) NULL);
    return TCL_ERROR;
  }

  AdabasSendInit (interp);

  /*
   * Here now comes the core of this procedure:
   * First create the adabas command, which is the workhorse of this extension.
   */

  Tcl_CreateObjCommand (interp, "adabas",
			AdabasCmd, (ClientData) NULL,
			(Tcl_CmdDeleteProc *) NULL);

  /*
   * Define the place, where the AdabasTcl library goes; there are the
   * sourcable .tcl files - not the shared library, which resides in
   * $DBROOT/lib, since this place you must know to execute this code).
   */

  Tcl_DStringInit (&dbRootTcl);
  if (!adabasTclDir || !*adabasTclDir) {
    if (dbroot && *dbroot) {
      Tcl_DStringAppend (&dbRootTcl, dbroot, -1);
      Tcl_DStringAppend (&dbRootTcl, "/tcl", -1);
      adabasTclDir = Tcl_DStringValue (&dbRootTcl);
    }
  }
  if (adabasTclDir) {
    Tcl_SetVar (interp, "adabastcl_library", adabasTclDir, TCL_GLOBAL_ONLY);
    Tcl_DStringFree (&dbRootTcl);
  }

  /*
   * And here are the API procedures adaopen, adasql, adafetch,...
   * (that are defined in tcl in the first version).
   */

  if (AdabasSqlInit (interp) != TCL_OK) {
    return TCL_ERROR;
  }
  AdabasApiInit (interp);

#ifdef ADABAS_TCLSH
  Tcl_SetVar(interp, "tcl_rcFileName", "~/.adabastclshrc", TCL_GLOBAL_ONLY);
#endif

#ifdef ADABAS_WISH
  Tcl_SetVar(interp, "tcl_rcFileName", "~/.adabaswishrc",  TCL_GLOBAL_ONLY);
#endif

  sprintf (versionString, "%.1f", ADABAS_TCL_VERSION);
  if (Tcl_PkgProvide (interp, "Adabastcl", versionString) == TCL_ERROR) {
    return TCL_ERROR;
  }
#ifndef ADABAS_MODULE
  Tcl_StaticPackage (interp, "Adabastcl", Adabastcl_Init, Adabastcl_SafeInit);
#endif
#ifdef ADABAS_WISH
  Tcl_StaticPackage (interp, "Tk", Tk_Init, Tk_SafeInit);
#endif

  return TCL_OK;
}

#ifdef ADABAS_MODULE
EXPORT(int,Adabastcl_SafeInit)(interp)
#else
static int Adabastcl_SafeInit (interp)
#endif
     Tcl_Interp *interp;		/* Current interpreter. */
{
  if (Adabastcl_Init (interp) == TCL_ERROR) {
    return TCL_ERROR;
  }

  /* Here we MUST disable the adabas command before distribute this code! */
  return TCL_OK;
}

/*
 * This is some special hack just for our Unixware installation.
 */

#ifdef ADABAS_MODULE
int   sys_nerr;
char *sys_errlist[1];
#endif

/*
 * This is a very special hack for windows...
 */

#ifdef __WIN32__
BOOL APIENTRY
DllEntryPoint(hInst,reason,reserved)
    HINSTANCE hInst;          /* Library instance handle. */
    DWORD     reason;         /* Reason this function is beeing called. */
    LPVOID    reserved;       /* Not used. */
{
    return TRUE;
}
#endif

/* ------------------------------------------------------------------*/

static int
AdabasCmd (clientData, interp, objc, objv)
     ClientData  clientData;		/* Reached through to subcommands. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  int          currOption;		/* Index of current option. */
  static char *options[] =		/* Options of adabas. */
  {"open", "crypt", "get", "put", "termId", "version", "xuser", (char *) NULL};

  if (objc < 2) {
    AppendResult (interp, "wrong # args: should be \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, " option ?arg ...?\"");
    return TCL_ERROR;
  }

  if (Tcl_GetIndexFromObj (interp, objv[1], options,
			   "option", 0, &currOption) == TCL_ERROR) {
    return TCL_ERROR;
  }
  switch (currOption) {
  case 0: /* open */
    return AdabasOpenCmd (clientData, interp, objc-1, objv+1); 

  case 1: /* crypt */
    return AdabasCryptCmd (clientData, interp, objc-1, objv+1); 

  case 2: /* opt */
    return AdabasGetCmd (clientData, interp, objc-1, objv+1); 

  case 3: /* put */
    return AdabasPutCmd (clientData, interp, objc-1, objv+1); 

  case 4: /* termId */
    AdabasRteTermId (interp->result); 
    return TCL_OK;

  case 5: /* version */
    return AdabasVersion (clientData, interp, objc-1, objv+1); 

  case 6: /* xuser */
    return AdabasXUser (clientData, interp, objc-1, objv+1);

  default:
    AppendResult (interp, "Oops, funny adabas option???");
    return TCL_ERROR;
  } 
}

/* ------------------------------------------------------------------*/

static int
AdabasOpenCmd (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  int          currOption;		/* Index of current option. */
  int          service  = SQL_USER;
  char        *serverdb = (char *) NULL;
  char        *cmdName  = "";
  AdabasInfo  *vars;
  static char *options[] =		/* Options of adabas open. */
  {"-name", "-service", "-serverdb", (char *) NULL};

  /*
   * Processing of the options.
   */

  for (objv+=1, objc-=1; objc > 1; objv += 2, objc -= 2) {
    if (Tcl_GetIndexFromObj (interp, objv[1], options,
			     "switch", 0, &currOption) == TCL_ERROR) {
      return TCL_ERROR;
    }
    switch (currOption) {
    case 0: /* -name */
      cmdName = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;

    case 1: /* -service */
      service = scanService (interp, Tcl_GetStringFromObj (objv[1], (int *) NULL));
      if (service < 0) {
	return TCL_ERROR;
      }
      break;

    case 2: /* -serverdb */
      serverdb = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      break;
    }
  }
  if (objc > 0) {
    AppendResult (interp, "missing argument for switch: \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }

  /*
   * Initialize the database runtime environment and allocate the info var.
   */

  if (AdabasOpenConnection (interp, service, serverdb,
			    cmdName, &vars) == TCL_ERROR) {
    return TCL_ERROR;
  }
    
  /*
   * Now create the command with the evaluated name.
   */

  Tcl_CreateObjCommand (interp, vars->logonName,
			AdabasConnCmd, (ClientData) vars,
			(Tcl_CmdDeleteProc *) NULL);
  AppendResult (interp, vars->logonName);

  return TCL_OK;
}

/* ------------------------------------------------------------------*/

static int
AdabasConnCmd (clientData, interp, objc, objv)
     ClientData  clientData;		/* Logon handle (generic type). */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  AdabasInfo      *vars        = (AdabasInfo *) clientData;
  AdabasPacketInfo thisOptions = vars->packetInfo;
  int              partKind;
  int              currOption;		/* Index of current option. */
  int              objLength;		/* Length of string object. */
  int              partI;
  char             serverdb[DBNAME+1];
  static char     *options[] =		/* Options of connection. */
  {"send", "close", "cancel", "reopen", "configure", (char *) NULL};
  static char     *partKinds[] =	/* Part kind to send. */
  {"-command", "-data", "-longdata", "-parsid", "-resultcount", (char *) NULL};

  if (objc < 2) {
    AppendResult (interp, "wrong # args: should be \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, " option ?arg ...?\"");
    return TCL_ERROR;
  }

  if (Tcl_GetIndexFromObj (interp, objv[1], options,
			   "option", 0, &currOption) == TCL_ERROR) {
    return TCL_ERROR;
  }
  switch (currOption) {

    /*
     * <command> send: communicate with the database kernel by
     *                 - constructing the order packet,
     *                 - sending it to the kernel and
     *                 - waiting for the answer.
     */

  case 0: /* send */
    for (partI = 0; partI < PART_KINDS; partI++) {
      vars->partInfos[ partI ].bufLength = 0;
    }

#if ADABAS_VERSION <= 61
    /*
     * Since in pre6.2 there is no possibility to set autocommit
     * or withInfo, when switching the sqlmode, we must forbid this.
     */

    if (thisOptions.sqlMode != SQLM_NIL) {
      if (thisOptions.commitImmediately || thisOptions.withInfo) {
	interp->result = "No autocommit or withInfo, if sqlmode changed.";
	return TCL_ERROR;
      }
    }
#endif

    /*
     * Processing of the options.
     */

    if (objc < 3) {
      SetResult (interp, "missing specification, what to send");
      return TCL_ERROR;      
    }
    for (objv+=2, objc-=2; objc > 1; objv += 2, objc -= 2) {
      partKind = pkNil;

      if (Tcl_GetIndexFromObj ((Tcl_Interp *) NULL, objv[0], partKinds,
			       "partKind", 0, &currOption) == TCL_OK) {
	switch (currOption) {
	case 0: /* command */
	  partKind = pkCommand;
	  break;
	case 1: /* data */
	  partKind = pkData;
	  break;
	case 2: /* longdata */
	  partKind = pkLongdata;
	  break;
	case 3: /* parsid */
	  partKind = pkParsid;
	  break;
	case 4: /* resultcount */
	  partKind = pkResultcount;
	  break;
	}
      } else {
	if (scanPacketConfOpts (interp,
				"-command, -data, -resultcount, -longdata, ", 
				objv[0], objc-1, objv+1,
				&thisOptions) == TCL_ERROR) {
	  return TCL_ERROR;
	}
      }

      if (partKind != pkNil) {
	if (vars->partInfos[ partKind ].bufLength) {
	  AppendResult (interp, "\"");
	  AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
	  AppendResult (interp, "\" is given twice");
	  return TCL_ERROR;
	}
	vars->partInfos[partKind].buf = Tcl_GetStringFromObj (objv[1], &objLength);
	vars->partInfos[partKind].bufLength = objLength;
	if (partKind == pkResultcount &&
	    Tcl_GetIntFromObj (interp, objv[1],
			       &vars->partInfos[partKind].bufLength) != TCL_OK) {
	  return TCL_ERROR;
	}
      }
    }

    if (objc > 0) {
      AppendResult (interp, "missing argument for switch: \"");
      AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
      AppendResult (interp, "\"");
      return TCL_ERROR;
    }

    if (AdabasFillPacket (interp, vars, &thisOptions) == TCL_ERROR) {
      return TCL_ERROR;
    }
    if (AdabasSendPacket (interp, vars) == TCL_ERROR) {
      return TCL_ERROR;
    }

    /*
     * Transform the received packet to a valid tcl result.
     */

#if ADABAS_VERSION >= 62
    setResultSegment (interp, &vars->rcvPacket->segm);
#else /* ADABAS_VERSION <= 61 */
    setResultSegment (interp, &vars->sqlPacket, thisOptions.messType);
#endif
    break;


    /*
     * <command> close: Close a connection to the database and delete
     *                  this tcl command.
     */

  case 1: /* close */
    return AdabasCloseCmd (interp,
			   Tcl_GetStringFromObj (objv[0], (int *) NULL),
			   vars);

    /*
     * <command> cancel: Cancels the currently running sql statement.
     */

  case 2: /* cancel */
    AdabasRteCancel (&vars->rteInfo);
    break;

    /*
     * <command> reopen: Close and then open the connection to the database.
     */

  case 3: /* reopen */
    AdabasRteClose (&vars->rteInfo);
    p2c (serverdb, vars->rteInfo.dbname, DBNAME);
    return AdabasRteConnect (interp, &vars->rteInfo,
			     serverdb, vars->rteInfo.service);

    /*
     * <command> configure: Query or set the configuration.
     */

  case 4: /* configure */
    switch (objc) {
    case 2:
      return AdabasSendConfigure (interp, &vars->packetInfo, (Tcl_Obj *) NULL);
    case 3:
      return AdabasSendConfigure (interp, &vars->packetInfo, objv[2]);
    default:
      return scanPacketConfOpts (interp, "", objv[2], objc-3, objv+3,
				 &vars->packetInfo);
    }
  }
  return TCL_OK;
}

/* ------------------------------------------------------------------*/

static int
AdabasVersion (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  char version[200];			/* Buffer for version string. */

  if (objc != 1) {
    AppendResult (interp, "wrong # args: should be adabas version");
    return TCL_ERROR;
  }
  SET_VERSION (version);
  SetResult (interp, version);
  return TCL_OK;
}

/* ------------------------------------------------------------------*/

static int
AdabasXUser (dummy, interp, objc, objv)
     ClientData  dummy;			/* Not used. */
     Tcl_Interp *interp;		/* Current interpreter. */
     int         objc;			/* Number of arguments. */
     ConstObjPtr objv[]; 		/* Argument objects. */
{
  xuser_record userParams;
  int          index;
  char        *userName;		/* userkey to get the data of. */
  char         givenPasswd[100];	/* more then enough. */
  int          currOption;		/* Index of current option. */
  static char *options[] =		/* options of adabas xuser. */
  {"args", "clear", "close", "get", "index", "open", "put", (char *) NULL};

  if (objc == 1) {
    AppendResult (interp, "wrong # args: should be adabas xuser option ?...?");
    return TCL_ERROR;
  }

  if (Tcl_GetIndexFromObj (interp, objv[1], options,
			   "option", 0, &currOption) == TCL_ERROR) {
    return TCL_ERROR;
  }
  switch (currOption) {
  case 0: /* args */
    if (objc != 2) {
      AppendResult (interp, "wrong # args: should be adabas xuser args");
      return TCL_ERROR;
    }
    if (AdabasXUserArgs (interp, &userParams, givenPasswd) == TCL_ERROR) {
      return TCL_ERROR;
    }
    xuser2result (interp, &userParams, givenPasswd);
    return TCL_OK;

  case 1: /* clear */
    if (objc != 2) {
      AppendResult (interp, "wrong # args: should be adabas xuser clear");
      return TCL_ERROR;
    }
    return AdabasXUserClear (interp);

  case 2: /* close */
    if (objc != 2) {
      AppendResult (interp, "wrong # args: should be adabas xuser close");
      return TCL_ERROR;
    }
    return AdabasXUserClose (interp);

  case 3: /* get */
    if (objc > 3) {
      AppendResult (interp, "wrong # args: should be adabas xuser get ?userkey?");
      return TCL_ERROR;
    } else if (objc == 3) {
      userName = Tcl_GetStringFromObj (objv[2], (int *) NULL);
    } else {
      userName = NULL;
    }
    if (AdabasXUserGet (interp, userName, &userParams) == TCL_ERROR) {
      return TCL_ERROR;
    }
    xuser2result (interp, &userParams, (char *) NULL);
    return TCL_OK;

  case 4: /* index */
    if (objc != 3) {
      AppendResult (interp, "wrong # args: should be adabas xuser index no");
      return TCL_ERROR;
    }
    if (Tcl_GetIntFromObj (interp, objv[2], &index) == TCL_ERROR) {
      return TCL_ERROR;
    }
    if (AdabasXUserIndex (interp, index, &userParams) == TCL_ERROR) {
      return TCL_ERROR;
    }
    xuser2result (interp, &userParams, (char *) NULL);
    return TCL_OK;

  case 5: /* open */
    if (objc != 2) {
      AppendResult (interp, "wrong # args: should be adabas xuser open");
      return TCL_ERROR;
    }
    return AdabasXUserOpen (interp);

  case 6: /* put */
    if (objc == 2) {
      AppendResult (interp,
		    "wrong # args: adabas xuser put entry value ?entry value?...");
      return TCL_ERROR;
    }
    if (args2xuser (interp, objc-2, objv+2, &userParams) == TCL_ERROR) {
      return TCL_ERROR;
    }
    return AdabasXUserPut (interp, &userParams);

  default:
    AppendResult (interp, "Oops, funny xuser option???");
    return TCL_ERROR;
  }
}

/* ------------------------------------------------------------------*/

static void
xuser2result (interp, userParams, givenPasswd)
     Tcl_Interp   *interp;		/* Current interpreter. */
     xuser_record *userParams;		/* Incoming xuser record. */
     char         *givenPasswd;		/* Clear passwd from command line. */
{
  char        buffer[256];
  Tcl_DString pw;

  Tcl_ResetResult (interp);

  LAppendStringResult (interp, "userkey");
  p2c (buffer, userParams->xu_key,
       sizeof (userParams->xu_key));
  LAppendStringResult (interp, buffer);

  LAppendStringResult (interp, "servernode");
  p2c (buffer, userParams->xu_servernode,
       sizeof (userParams->xu_servernode));
  LAppendStringResult (interp, buffer);

  LAppendStringResult (interp, "serverdb");
  p2c (buffer, userParams->xu_serverdb,
       sizeof (userParams->xu_serverdb));
  LAppendStringResult (interp, buffer);

  LAppendStringResult (interp, "user");
  p2c (buffer, userParams->xu_user,
       sizeof (userParams->xu_user));
  LAppendStringResult (interp, buffer);

  if (givenPasswd && *givenPasswd) {
    LAppendStringResult (interp, "clearpw");
    LAppendStringResult (interp, givenPasswd);
  } else {
    LAppendStringResult (interp, "password");
    memcpy (buffer, userParams->xu_password,
	    sizeof (userParams->xu_password));
    Tcl_DStringInit (&pw);
    decodeSequence (interp, encEscape, (Tcl_Obj *) NULL, &pw, buffer, 24);
    LAppendStringResult (interp, Tcl_DStringValue (&pw));
    Tcl_DStringFree (&pw);
  }

  LAppendStringResult (interp, "sqlmode");
  p2c (buffer, userParams->xu_sqlmode,
       sizeof (userParams->xu_sqlmode));
  LAppendStringResult (interp, buffer);

#if ADABAS_VERSION <= 61
  LAppendStringResult (interp, "cachelimit");
  LAppendIntResult (interp, userParams->xu_cachelimit);
#endif

  LAppendStringResult (interp, "timeout");
  LAppendIntResult (interp, userParams->xu_timeout);

  LAppendStringResult (interp, "isolation");
  LAppendIntResult (interp, userParams->xu_isolation);

#if ADABAS_VERSION >= 62
  LAppendStringResult (interp, "dblang");
  p2c (buffer, userParams->xu_dblang,
       sizeof (userParams->xu_dblang));
  LAppendStringResult (interp, buffer);
#endif
}

/* ------------------------------------------------------------------*/

static int
args2xuser (interp, objc, objv, userParams)
     Tcl_Interp   *interp;		/* Current interpreter. */
     int           objc;		/* Number of arguments. */
     ConstObjPtr   objv[];		/* Argument objects. */
     xuser_record *userParams;		/* Returning xuser record. */
{
  int          currOption;		/* Index of current option. */
  int          isBlank;			/* sqlmode completely blank? */
  Tcl_DString  pw;
  int          value;
  int          keyFound = 0;
  char        *p;
  char        *sqlMode;			/* String value of sqlmode option. */
  static char *args[] =			/* Arguments of adabas xuser put. */
  {"cachelimit", "isolation", "servernode", "serverdb", "sqlmode",
   "password", "timeout", "user", "userkey",
#if ADABAS_VERSION >= 62
   "dblang",
#endif
   (char *) NULL};

  /*
   * First initialize all char fields to blank and numeric fields to -1.
   */

  memset (userParams, ' ', sizeof (*userParams));
  c2p (userParams->xu_sqlmode, sqlMode2String (SQLM_ADABAS, 1),
       sizeof (userParams->xu_sqlmode));
  userParams->xu_cachelimit = -1;
  userParams->xu_timeout    = -1;
  userParams->xu_isolation  = -1;

  /*
   * Now scan in a loop all given parameter specs.
   */

  for (; objc >= 2; objc -=2, objv +=2) {
    if (Tcl_GetIndexFromObj (interp, objv[0], args,
			     "xuser entry", 0, &currOption) == TCL_ERROR) {
      return TCL_ERROR;
    }
    switch (currOption) {
    case 0: /* cachelimit */
      if (Tcl_GetIntFromObj (interp, objv[1], &value) == TCL_ERROR) {
	return TCL_ERROR;
      }
      userParams->xu_cachelimit = (int4) value;
      break;

    case 1: /* isolation */
      if (Tcl_GetIntFromObj (interp, objv[1], &value) == TCL_ERROR) {
	return TCL_ERROR;
      }
      userParams->xu_isolation = (int2) value;
      break;

    case 2: /* servernode */
      c2p (userParams->xu_servernode,
	   Tcl_GetStringFromObj (objv[1], (int *) NULL),
	   sizeof (userParams->xu_servernode));
      break;

    case 3: /* serverdb */
      c2p (userParams->xu_serverdb,
	   Tcl_GetStringFromObj (objv[1], (int *) NULL),
	   sizeof (userParams->xu_serverdb));
      break;

    case 4: /* sqlmode */
      sqlMode = Tcl_GetStringFromObj (objv[1], (int *) NULL);
      if (strlen (sqlMode) &&
	  scanSqlMode ((Tcl_Interp *) NULL, sqlMode, 1) < 0) {
	isBlank = 1;
	for (p = sqlMode; *p; p++) {
	  *p = tolower (*p);
	  if (!isBlank && *p != ' ') {
	    isBlank = 0;
	  }
	}
	if (!isBlank && strcmp (sqlMode, "sapr3")) {
	  AppendResult (interp, "Invalid sqlmode \"");
	  AppendResult (interp, sqlMode);
	  AppendResult (interp, "\"");
	  return TCL_ERROR;
	}
      }
      c2p (userParams->xu_sqlmode, sqlMode,
	   sizeof (userParams->xu_sqlmode));
      break;

    case 5: /* password */
      Tcl_DStringInit (&pw);
      (void) encodeSequence (interp, encEscape, "passwd", &pw,
			     Tcl_GetStringFromObj (objv[1], (int *) NULL), 1, 0);
      memcpy (userParams->xu_password, Tcl_DStringValue (&pw),
	      sizeof (userParams->xu_password));
      Tcl_DStringFree (&pw);
      break;

    case 6: /* timeout */
      if (Tcl_GetIntFromObj (interp, objv[1], &value) == TCL_ERROR) {
	return TCL_ERROR;
      }
      userParams->xu_cachelimit = (int2) value;
      break;

    case 7: /* user */
      c2p (userParams->xu_user,
	   Tcl_GetStringFromObj (objv[1], (int *) NULL),
	   sizeof (userParams->xu_user));
      break;

    case 8: /* userkey */
      c2p (userParams->xu_key,
	   Tcl_GetStringFromObj (objv[1], (int *) NULL),
	   sizeof (userParams->xu_key));
      keyFound = 1;
      break;

#if ADABAS_VERSION >= 62
    case 9: /* dblang */
      c2p (userParams->xu_dblang,
	   Tcl_GetStringFromObj (objv[1], (int *) NULL),
	   sizeof (userParams->xu_dblang));
      break;
#endif
    }
  }

  /*
   * Is there a dangling parameter on the command line?
   */

  if (objc == 1) {
    AppendResult (interp, "Expected value for entry \"");
    AppendResult (interp, Tcl_GetStringFromObj (objv[0], (int *) NULL));
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }

  /*
   * One of the specifications must be the userkey itself.
   */

  if (!keyFound) {
    SetResult (interp, "No userkey specified");
    return TCL_ERROR;
  }
  return TCL_OK;
}

/* ------------------------------------------------------------------*/

static int
AdabasCloseCmd (interp, cmdName, vars)
     Tcl_Interp *interp;		/* Current interpreter. */
     char       *cmdName;		/* Name of logon handle to delete. */
     AdabasInfo *vars;			/* Logon handle to delete. */
{
  /*
   * Delete this tcl command.
   */

  if (Tcl_DeleteCommand (interp, cmdName) == TCL_ERROR) {
    AppendResult (interp, "can't delete command \"");
    AppendResult (interp, cmdName);
    AppendResult (interp, "\"");
    return TCL_ERROR;
  }

  return AdabasCloseConnection (interp, cmdName, vars);
}

/* ------------------------------------------------------------------*/

static int
AdabasSendConfigure (interp, vars, optionName)
     Tcl_Interp       *interp;		/* Current interpreter. */
     AdabasPacketInfo *vars;		/* Packet handle. */
     Tcl_Obj          *optionName;	/* Option to examine (or NULL). */
{
  char       **o;			/* Option to examine, if no optionName. */
  int          currOption = 0;		/* Index of current option. */
  Tcl_DString  oneValue;
  char        *defaultValue;
  char        *currentValue;
  static char *options[] =		/* Options of <conn> configure. */
  {"-application", "-messType", "-producer", "-sqlMode", "-withInfo",
   "-massCmd", "-commitImmediately", (char *) NULL};

  if (optionName
      && Tcl_GetIndexFromObj (interp, optionName, options,
			      "switch", 0, &currOption) == TCL_ERROR) {
    return TCL_ERROR;
  }

  Tcl_DStringInit (&oneValue);

  for (o = options; *o; o++, currOption++) {
    switch (currOption) {
    case 0: /* -application */
      defaultValue = application2String ("XCI");
      currentValue = application2String (vars->application);
      break;

    case 1: /* -messType */
      defaultValue = messType2String (M_DBS);
      currentValue = messType2String (vars->messType);
      break;

    case 2: /* -producer */
#if ADABAS_VERSION >= 62
      defaultValue = producer2String (PR_USER_CMD);
      currentValue = producer2String (vars->producer);
#else /* ADABAS_VERSION <= 61 */
      defaultValue = currentValue = "user";
#endif
      break;

    case 3: /* -sqlMode */
#if ADABAS_VERSION >= 62
      defaultValue = sqlMode2String (SM_SESSION_SQLMODE, 0);
#else /* ADABAS_VERSION <= 61 */
      defaultValue = sqlMode2String (SQLM_NIL, 0);
#endif
      currentValue = sqlMode2String (vars->sqlMode, 0);
      break;

    case 4: /* -withInfo */
      defaultValue = "false";
      currentValue = vars->withInfo ? "true" : "false";
      break;

    case 5: /* -massCmd */
      defaultValue = "false";
      currentValue = vars->massCmd ? "true" : "false";
      break;

    case 6: /* commitImmediately */
      defaultValue = "false";
      currentValue = vars->commitImmediately ? "true" : "false";
      break;

    default:
      SetResult (interp, "Oops, invalid configure option?!?");
      return TCL_ERROR;
    }

    Tcl_DStringAppendElement (&oneValue, *o);
    Tcl_DStringAppendElement (&oneValue, defaultValue);
    Tcl_DStringAppendElement (&oneValue, currentValue);

    if (optionName) {
      Tcl_DStringResult (interp, &oneValue);
      return TCL_OK;
    }

    Tcl_AppendElement (interp, Tcl_DStringValue (&oneValue));
    Tcl_DStringFree (&oneValue);
  }
  return TCL_OK;
}

/* ------------------------------------------------------------------*/

static int
scanPacketConfOpts (interp, expected, optionName, objc, objv, vars)
     Tcl_Interp       *interp;		/* Current interpreter. */
     char             *expected;	/* List of options (for errortext). */
     Tcl_Obj          *optionName;	/* Name of option to configure. */
     int               objc;		/* Number of arguments. */
     ConstObjPtr       objv[];		/* Argument objects. */
     AdabasPacketInfo *vars;		/* Packet handle. */
{
  int   currOption;			/* Index of current option. */
  char *currParameter;			/* String value of parameter object. */
  int   newMessType;
  char *newApplication;
  int   newSqlMode;
#if ADABAS_VERSION >= 62
  int   newProducer;
#endif
  static char *options[] =		/* Options of <conn> configure. */
  {"-application", "-messType", "-producer", "-sqlMode", "-withInfo",
   "-massCmd", "-commitImmediately", (char *) NULL};

  if (Tcl_GetIndexFromObj ((Tcl_Interp *) NULL, optionName, options,
			   "switch", 0, &currOption) == TCL_OK) {
    switch (currOption) {

      /*
       * <cmd> configure -application <commandInterface|load|utility...>
       * Kind of application, sending the request to the kernel.
       */

    case 0: /* -application */
      currParameter = Tcl_GetStringFromObj (objv[0], (int *) NULL);
      if (!(newApplication = scanApplication (interp, currParameter))) {
	return TCL_ERROR;
      }
      memcpy (vars->application, newApplication, 3);
      return TCL_OK;

      /*
       * <cmd> configure -messType <dbs|parse|execute|utility|...>
       * Message type of packet, that will be send to the kernel.
       */

    case 1: /* -messType */
      currParameter = Tcl_GetStringFromObj (objv[0], (int *) NULL);
      newMessType = scanMessType (interp, currParameter);
      if (newMessType == M_NIL) {
	return TCL_ERROR;
      }
      vars->messType = newMessType;
      return TCL_OK;

      /*
       * <cmd> configure -producer <user|internal|installation>
       * Who is the producer of the request?
       */

#if ADABAS_VERSION >= 62
    case 2: /* -producer */
      currParameter = Tcl_GetStringFromObj (objv[0], (int *) NULL);
      if ((newProducer = scanProducer (interp, currParameter)) == PR_NIL) {
	return TCL_ERROR;
      }
      vars->producer = newProducer;
      return TCL_OK;
#endif

      /*
       * <cmd> configure -sqlMode <session|adabas|ansi|db2|oracle>
       * In what mode should the request be interpreted?
       */

    case 3: /* -sqlmode */
      currParameter = Tcl_GetStringFromObj (objv[0], (int *) NULL);
#if ADABAS_VERSION >= 62
      if ((newSqlMode = scanSqlMode (interp, currParameter, 0)) == SM_NIL) {
	return TCL_ERROR;
      }
#else /* ADABAS_VERSION <= 61 */
      if ((newSqlMode = scanSqlMode (interp, currParameter, 0)) < 0) {
	return TCL_ERROR;
      }
#endif
      vars->sqlMode = newSqlMode;
      return TCL_OK;

      /*
       * <cmd> configure -withInfo <bool>.
       * Should the kernel respond with the addition info of columnnames
       * and shortinfos for a select?
       */

    case 4: /* -withInfo */
      return Tcl_GetBooleanFromObj (interp, objv[0], &vars->withInfo);

      /*
       * <cmd> configure -massCmd <bool>.
       * Should the given sqlStatement be executed as mass command?
       */

    case 5: /* -massCmd */
      return Tcl_GetBooleanFromObj (interp, objv[0], &vars->massCmd);

      /*
       * <cmd> configure -commitImmediately <bool>.
       * Should the kernel execute a commit after any ddl statement?
       */

    case 6: /* -commitImmediately */
      return Tcl_GetBooleanFromObj (interp, objv[0], &vars->commitImmediately);

    default:
      AppendResult (interp, "Oops, funny xuser option???");
      return TCL_ERROR;
    }
  } else {

    /*
     * No known option was given, if control gets to this point.
     */

    AppendResult (interp, "bad switch \"");
    AppendResult (interp, Tcl_GetStringFromObj (optionName, (int *) NULL));
    AppendResult (interp, "\": must be ");
    AppendResult (interp, expected);
    AppendResult (interp, "-application, -messType, -producer,");
    AppendResult (interp, " -sqlMode, -masscmd, -commitImmediately or -withInfo");
    return TCL_ERROR;
  }
}

/* ------------------------------------------------------------------*/

#if ADABAS_VERSION >= 62
static void setResultSegment (interp, retSegment)
     Tcl_Interp *interp;		/* Current interpreter. */
     segment    *retSegment;		/* Return segment to be converted. */
{
  part *currPart = &retSegment->part;
  int4  partLength;
  char  result[80];
  int   ix;

  sprintf (result, "-returncode %d ",
	   retSegment->segm_header.kind.ret.returncode);
  AppendResult (interp, result);
  sprintf (result, "-errorpos %d ",
	   retSegment->segm_header.kind.ret.errorpos);
  AppendResult (interp, result);
  sprintf (result, "-sqlstate %5.5s ",
	   retSegment->segm_header.kind.ret.sqlstate);
  AppendResult (interp, result);

  for (ix = 0; ix < retSegment->segm_header.kind.cmd.no_of_parts; ix++) {
    if (ix > 0) {
      AppendResult (interp, " ");
    }
    putResultPart (interp, currPart, &partLength);
    currPart = (part *) (((char *) currPart) + partLength);
  }
}
#else /* ADABAS_VERSION <= 61 */
static void setResultSegment (interp, retSegment, sndMessType)
     Tcl_Interp *interp;		/* Current interpreter. */
     packet     *retSegment;		/* Return segment to be converted. */
     int         sndMessType;		/* Mess type of sended packet. */
{
  char result[80];
  int2 currPos;

  sprintf (result, "-returncode %d ",  retSegment->return_code);
  AppendResult (interp, result);
  if (retSegment->return_code) {
    sprintf (result, "-errorpos %d ",   retSegment->error_code);
    AppendResult (interp, result);
  } else {
    /* Warning set not yet... */
  }
  currPos = 1;
  while (currPos <= retSegment->part1_length) {
    putResultPart (interp, 1, &currPos, retSegment, sndMessType);
  }
  currPos = 1;
  while (currPos <= retSegment->part2_length) {
    putResultPart (interp, 2, &currPos, retSegment, sndMessType);
  }
}
#endif
