
# ---------------------------------------------------------------------
# $Id: sqlsh.tcl,v 1.15 1997/01/03 19:34:03 adabas Exp adabas $
# ---------------------------------------------------------------------
# Copyright (c) 1996-1997 Christian Krone. All rights reserved.
# This program is free software; you can redistribute it and/or
# modify it under the same terms as Tcl itself.
# See also licence.terms
# ----------------------------------------------------------------------------

# Make sure, that this file is only evaluated once...
if [string length [info commands sqlCommand]] return

trace variable adamsg(prot) w _sql_openProtFile
trace variable adamsg(prot) u _sql_closeProtFile

proc _sql_openProtFile {name1 name2} {
    upvar $name adamsg

    if [info exists _sql_var(protFile)] {
	close $_sql_var(protFile)
    }
    if [catch {open $adamsg($name2) w} ret] {
	return -code error $ret
    } else {
	set _sql_var(protFile) $ret
    }
}

proc _sql_closeProtFile {name1 name2} {
    upvar $name adamsg

    if [info exists _sql_var(protFile)] {
	close $_sql_var(protFile)
	unset _sql_var(protFile)
    }
}

proc sqlCommand {args} {
    global adamsg _sql_var

    connect

    if [info exists _sql_var(protFile)] {
	puts $_sql_var(protFile) $args
    }
    set firstToken [string tolower [lindex $args 0]]
    switch $firstToken {
    	commit - rollback {
	    _sql_release $firstToken $args
	}
	default {
	    if [catch {uplevel 1 adasql $_sql_var(currCursor) [list $args]} rc] {
		_sql_putError $rc
	    }

	    switch $firstToken {
		select - show - explain {
		    if ![info exists adamsg(intoVars)] {
			_sql_showResults
		    }
		}
		insert - update - delete {
		    puts "Resultcount: $adamsg(rows)"
		}
	    }
	}
    }
}

proc _sql_putError {rc} {
    global adamsg env

    if [info exists adamsg(errortxt)] {
	if [info exists adamsg(errorpos)] {
	    set pos " (pos $adamsg(errorpos))"
	} else {
	    set pos ""
	}
	set notAvail "MESSAGE NOT AVAILABLE"
	if [string match "$notAvail*" $adamsg(errortxt)] {
	    if ![catch {exec grep -e " $adamsg(rc) " \
		    $env(DBROOT)/env/SQLM.eng} err] {
		regsub $notAvail $adamsg(errortxt) \
			[string range $err 11 end] err
		error "Error $adamsg(rc)$pos: $err"
	    }
	}
	error "Error $adamsg(rc)$pos: $adamsg(errortxt)"
    }
    error "Error $rc"
}

proc _sql_release {firstToken command} {
    global _sql_var

    set ret [catch {ada$firstToken $_sql_var(currLogon)} msg]

    if {[string first release [string tolower $command]] >= 0} {
	catch {adaclose  $_sql_var(currCursor)}
	catch {adalogoff $_sql_var(currLogon)}
	catch {unset _sql_var(currLogon)}
	catch {unset _sql_var(currCursor)}
    }

    if $ret {return -code error $msg}
    return $msg
}

proc _sql_showResults {} {
    global adamsg _sql_var

    set _sql_var(rowCnt)     0
    set _sql_var(noHold)     0
    set _sql_var(lineLength) 0
    set _sql_var(colNames)   [adacols $_sql_var(currCursor)]

    set colNo  0
    foreach col $_sql_var(colNames) {
	incr _sql_var(lineLength) [expr {$colNo > 0}]

	set headLength [string length $col]
	switch -glob [lindex $adamsg(coltypes) $colNo] {
	    long_byte* {
		if {$headLength < 11} {set length 11}
		set _sql_var(fmt,$colNo) "<LONG BYTE>"
	    }
	    long_* {
		set length 100
		set _sql_var(fmt,$colNo) "<LONG CHAR>"
	    }
	    fixed* - float* {
		set length [lindex $adamsg(collengths) $colNo]
		if {$headLength > $length} {
		    set length $headLength
		}
		set _sql_var(fmt,$colNo) "%${length}s"
	    }
	    default {
		set length [lindex $adamsg(collengths) $colNo]
		if {$headLength > $length} {
		    set length $headLength
		}
		set _sql_var(fmt,$colNo) "%-${length}s"
	    }
	}
	incr _sql_var(lineLength) $length
	incr colNo
    }

    set _sql_var(headerWanted) [expr {$_sql_var(lineLength) < 80}]

    adafetch $_sql_var(currCursor) -command "_sql_showRow @0"

    puts "Resultcount: $_sql_var(rowCnt)"
}

proc _sql_showRow {row} {
    global _sql_var adamsg

    if {$_sql_var(rowCnt) && !$_sql_var(noHold) \
	    && ($_sql_var(lineLength) >= 80 || !($_sql_var(rowCnt)%22))} {
	puts "HOLDING..."
	if {[catch {gets stdin} answer] || $answer == "q" || $answer == "Q"} {
	    return -code return
	}
	if {$answer == "n"} {
	    set _sql_var(noHold) 1
	} else {
	    set _sql_var(headerWanted) [expr {$_sql_var(lineLength) < 80}]
	}
    }

    if $_sql_var(headerWanted) {
	set colNo 0
	set sep ""
	foreach col $_sql_var(colNames) {
	    puts -nonewline "$sep[format $_sql_var(fmt,$colNo) $col]"
	    incr colNo
	    set sep " "
	}
	puts ""
	set _sql_var(headerWanted) 0
    }

    set colNo 0
    set sep   ""
    foreach col $row {
	if {$_sql_var(lineLength) < 80} {
	    puts -nonewline $sep
	    set sep " "
	} else {
	    set colName [lindex $_sql_var(colNames) $colNo]
	    puts -nonewline "[format "%-18s" $colName] : "
	}

	if ![string compare $_sql_var(fmt,$colNo) "<LONG CHAR>"] {
	    puts -nonewline [adareadlong $_sql_var(currCursor) -descr $col]
	} else {
	    if {$_sql_var(lineLength) >= 80 \
		    && [string match %-* $_sql_var(fmt,$colNo)]} {
		puts -nonewline $col
	    } else {
		puts -nonewline [format $_sql_var(fmt,$colNo) $col]
	    }
	}

	incr colNo
	if {$_sql_var(lineLength) >= 80} {puts ""}
    }
    if {$_sql_var(lineLength) < 80} {puts ""}

    incr _sql_var(rowCnt)
}

proc utility {{user ""} {passwd ""}} {
    global _sql_var

    while {![info exists _sql_var(currUtility)]} {
	while {![string length $user]} {
	    puts -nonewline "Controluser: "
	    flush stdout
	    if {[gets stdin user] < 0} {error "\nNo connection yet"}
	}
	while {![string length $passwd]} {
	    puts -nonewline "Password   : "
	    flush stdout
	    if {[gets stdin passwd] < 0} {error "\nNo connection yet"}
	}
	if [catch {adalogon $user,$passwd -service utility} \
					_sql_var(currUtility)] {
	    if {[info exists adamsg(rc)] && [info exists adamsg(errortxt)]} {
		set msg "Error $adamsg(rc): $adamsg(errortxt)"
	    } else {
		set msg "Error: $_sql_var(currUtility)"
		unset _sql_var(currUtility)
		error $msg
	    }
	    unset _sql_var(currUtility)
	    set user   ""
	    set passwd ""
	}
    }
    return ""
}

proc util {args} {
    global _sql_var

    if ![info exists _sql_var(currUtility)] {
	utility
    }
    set ret [adautil $_sql_var(currUtility) $args]

    switch -regexp [string tolower $args] {
	release - restart - shutdown {
	    adalogoff $_sql_var(currUtility)
	    unset _sql_var(currUtility)
	}
    }
    return $ret
}

proc connect {args} {
    global adamsg _sql_var

    if {[llength $args] > 2} {
	if {[llength $args] < 4} {error "Invalid end of connect statement"}

	if {[string compare [string toupper [lindex $args 1]] IDENTIFIED] \
		|| [string compare [string toupper [lindex $args 2]] BY]} {
	    error "syntax error"
	}
	set user [lindex $args 0]
	set clearPW [string toupper [lindex $args 3]]

	set opts    [lrange $args 4 end]
	set options ""
	while {[llength $opts]} {
	    switch [string toupper [lindex $opts 0]] {
		SQLMODE {
		    if {[llength $opts] < 2} {error "invalid end"}
		    append options " -sqlmode [lindex $opts 1]"
		}
		ISOLATION {
		    if {[llength $opts] < 3} {error "invalid end"}
		    if [string compare [string toupper \
			    [lindex $opts 1]] LEVEL] {
			error "LEVEL missing"
		    }
		    append options " -isolationlevel [lindex $opts 2]"
		}
		default {
		    error "invalid option"
		}
	    }
	    set opts [lrange $opts 2 end]
	}
	set _sql_var(currLogon)  [eval adalogon $user,$clearPW $options]
	set _sql_var(currCursor) [adaopen $_sql_var(currLogon)]
	return
    }

    set user   ""
    set passwd ""
    if {[llength $args] >= 1} {set user   [lindex $args 0]}
    if {[llength $args] >= 2} {set passwd [lindex $args 1]}
    while {![info exists _sql_var(currLogon)]} {
	while {![string length $user]} {
	    puts -nonewline "Username: "
	    flush stdout
	    if {[gets stdin user] < 0} {error "\nNo connection yet"}
	}
	while {![string length $passwd]} {
	    puts -nonewline "Password: "
	    flush stdout
	    if {[gets stdin passwd] < 0} {error "\nNo connection yet"}
	}
	if [catch {adalogon $user,$passwd} _sql_var(currLogon)] {
	    if {[info exists adamsg(rc)] && [info exists adamsg(errortxt)]} {
		set msg "Error $adamsg(rc): $adamsg(errortxt)"
	    } else {
		set msg "Error: $_sql_var(currLogon)"
		unset _sql_var(currLogon)
		error $msg
	    }
	    unset _sql_var(currLogon)
	    set user   ""
	    set passwd ""
	}
    }
    if ![info exists _sql_var(currCursor)] {
	set _sql_var(currCursor) [adaopen $_sql_var(currLogon)]
    }
    return [expr {[llength $args] ? "" : $_sql_var(currCursor)}]
}

proc vtrace {} {
    global _sql_var

    if [info exists _sql_var(currCursor)] {
	adasql $_sql_var(currCursor) VTRACE -sqlmode adabas
    } else {
	set currLogon  [adalogon -noconnect]
	set currCursor [adaopen $currLogon]
	adasql $currCursor VTRACE
	catch {adaclose  $currCursor}
	catch {adalogoff $currLogon}
    }
}

if ![string length [info commands tcl_switch]] {
    rename switch tcl_switch
}

proc switch {args} {
    global _sql_var

    if [llength $args]>1 {return [uplevel 1 tcl_switch $args]}
    if [llength $args] {
	regexp {^([^/]*)/(.*)} [lindex $args 0] match layer debug
    } else {
	puts -nonewline "Layer: "
	flush stdout
	if {[gets stdin layer] < 0} {error "\n"}
	puts -nonewline "Debug: "
	flush stdout
	if {[gets stdin debug] < 0} {error "\n"}
    }

    set connected [info exists _sql_var(currLogon)]
    if [info exists _sql_var(currLogon)] {
	adaspecial $_sql_var(currLogon) switch $layer $debug
    } else {
	set currLogon [adalogon -noconnect]
	adaspecial $currLogon switch $layer $debug
	catch {adalogoff $currLogon}
    }
}

if {![string length [info commands tcl_update]] \
	&& [string length [info commands update]]} {
    rename update tk_update
}

proc update {args} {
    if {![llength $args] || ![string first $args idletasks]} {
	return [uplevel 1 tk_update $args]
    }
    uplevel 1 sqlCommand update $args
}

# Here it is important to rename rename at last...
foreach sql {
    alter  clear    commit comment create  delete  drop
    exists explain  grant  insert  monitor refresh
    revoke rollback select show

    rename
} {
    if {![string compare [info commands $sql] $sql] \
	    && ![string length [info commands tcl_$sql]]} {
	rename $sql tcl_$sql
    }
    foreach cmd [list $sql [string toupper $sql]] {
	proc $cmd {args} "uplevel 1 sqlCommand $cmd \$args"
    }
}
unset sql
