
# ---------------------------------------------------------------------------
# Copyright (c) 1997 Christian Krone. All rights reserved.
# This program is free software; you can redistribute it and/or
# modify it under the same terms as Tcl itself.
# See also licence.terms
# ---------------------------------------------------------------------------
# $Id: makeVC.tcl,v 1.6 1997/07/03 22:04:49 adabas Exp $
# ---------------------------------------------------------------------------

set default {
# ---------------------------------------------------------------------------
# Copyright (c) 1997 Christian Krone. All rights reserved.
#.This program is free software; you can redistribute it and/or
#.modify it under the same terms as Tcl itself.
#.See also licence.terms
# ---------------------------------------------------------------------------
#.Don't edit this file with an editor, better use 'tclsh mkMakefileVC.tcl'...
# ---------------------------------------------------------------------------

# +------------------------------------------------------------------
# Some paths to adjust: the C++ development system, Tcl, Adabas D,...
# -------------------------------------------------------------------
TOOLS    = C:\msdev
TCL      = C:\Program Files\TCL
ADABAS   = D:\adabas95

# +------------------------------------------------------------------
# The current version of Tcl/Tk (as two digit decimal)
# -------------------------------------------------------------------
TCL_VERS = 80
TK_VERS  = 80

# +------------------------------------------------------------------
# Version of Adabas D: 61 for systems <= 6.1, 62 for systems >= 6.2
# -------------------------------------------------------------------
VERSION  = 61

# +------------------------------------------------------------------
# The name of some operating system programs.
# -------------------------------------------------------------------
MAKE     = nmake
RM       = del
MKDIR    = md
INSTALL  = copy
CAT      = type

#.-------------------------------------------------------------------
#. values to compute dependend from options above.
#.-------------------------------------------------------------------
SQLLIBS  = ###
}

# ---------------------------------------------------------------------------
# mkFrame: the init procedure to create a toplevel window, which allows the
#          user to change the values of the configuration variables.
# ---------------------------------------------------------------------------
proc mkFrame {} {
    global value tkCompat lines target

    catch {console hide}
    set lines [readAdabas95mak]
    readValues $lines

    set target install

    mkMenu    .m
    mkEntries .e
    mkMsgBar  .mb
    mkButtons .b
    wm title . "AdabasTcl: Create Makefiles for VC++"
    focus -force .
}

# ---------------------------------------------------------------------------
# readAdabas95mak: read the contents of the status file 'adabas95.mak' and
#                  return a list of all lines. If the file doesn't exist,
#                  return a default list.
# ---------------------------------------------------------------------------
proc readAdabas95mak {} {
    global default

    if [catch {open adabas95.mak} f] {
	set lines $default
    } else {
	set lines [read $f]
	close $f
    }
    return [split $lines "\n"]
}

# ---------------------------------------------------------------------------
# writeAdabas95mak: write the given list into the status file 'adabas95.mak'.
# ---------------------------------------------------------------------------
proc writeAdabas95mak {lines} {
    global msg

    if [catch {open adabas95.mak w} out] {
	set msg $out
	exit 1
    }
    puts -nonewline $out [join $lines "\n"]
    close $out
}

# ---------------------------------------------------------------------------
# readValues: scan the given list of variable assignments (VAR = value) or
#             layout specs. Put the results in global arrays value and sorted.
# ---------------------------------------------------------------------------
proc readValues {lines}  {
    global value sorted

    set sorted {}
    foreach line $lines {
	if [regexp {^([a-zA-Z0-9_]*) *= *([^#]*)###$} $line match entry val] {
	    set value($entry) $val
	} elseif [regexp {^([a-zA-Z0-9_]*) *= *(.*)$} $line match entry val] {
	    set value($entry) $val
	    lappend sorted $entry
	} elseif [regexp {^# \+} $line] {
	    set name sep[llength $sorted]
	    set value($name) +++
	    lappend sorted $name
	} elseif [regexp {^# ([^-].*$)} $line match text] {
	    set value($text) ---
	    lappend sorted $text
	}
    }
}

# ---------------------------------------------------------------------------
# substAdabas95mak: replace all variable assignments in the given list with
#                   the values correspondend to the global array value.
# ---------------------------------------------------------------------------
proc substAdabas95mak {lines}  {
    global value

    if {$value(VERSION) <= 61} {
	set value(SQLLIBS) {$(ADABAS)\lib\sqlpcr.lib $(ADABAS)\lib\ptocnt.lib ###}
    } else {
	set value(SQLLIBS) {$(ADABAS)\lib\pcr62smt.lib ###}
    }

    foreach line $lines {
	foreach entry [array names value] {
	    if [string compare $value($entry) ---] {
		if [regexp -indices "^$entry *= *" $line indexes] {
		    set    line [string range $line 0 [lindex $indexes 1]]
		    append line $value($entry)
		    break
		}
	    }
	}
	lappend resLines $line
    }
    return $resLines
}

# ---------------------------------------------------------------------------
# writeMakefiles: copy all makefiles of this director hierarchy from
#                 makefile.vc, if they don't exist already.
# ---------------------------------------------------------------------------
proc writeMakefiles {args} {
    global msg

    foreach dir $args {
	set makefile [file join $dir makefile]
	if ![file exists $makefile] {

            # Note, that it is essantial here to copy the files by reading
            # and writing them, and not by using the Tcl command file copy,
            # since this way the newlines are transformed into crlf by puts
            # automatically.

            if [catch {open $makefile w} out] {
                set msg $out
            } elseif [catch {open $makefile.vc} in] {
                set msg $in
            } else {
                puts -nonewline $out [read $in]
                close $in
                close $out
	    }
	}
    }
}

# ---------------------------------------------------------------------------
# ---------------------------------------------------------------------------
proc editValues {} {
}

# ---------------------------------------------------------------------------
# mkMenu: Create the menubar of the dialog window.
# ---------------------------------------------------------------------------
proc mkMenu {w} {
    global tk_version

    if {$tk_version >= 8.0} {
	. configure -menu [menu $w]
    	set menu(file)   $w.file
    	set menu(target) $w.target
	$w add cascade -label "File"   -menu $menu(file)   -underline 0
	$w add cascade -label "Target" -menu $menu(target) -underline 0
    } else {
	frame $w -relief raised -borderwidth 2
	pack  $w  -side top -fill x

	set menu(file)   $w.file.m
	set menu(target) $w.target.m
	menubutton $w.file   -text "File"   -menu $menu(file)   -underline 0
	menubutton $w.target -text "Target" -menu $menu(target) -underline 0
        pack $w
    }
    menu $menu(file)
    $menu(file) add command -label "About" -accelerator F1 -command about
    $menu(file) add command -label "Save"                  -command "action 0"
    $menu(file) add command -label "Make"                  -command "action 1"
    $menu(file) add command -label "Exit"  -accelerator F3 -command exit

    menu $menu(target)
    $menu(target) add radio -variable target -value install -label Install
    $menu(target) add radio -variable target -value all     -label All
    $menu(target) add radio -variable target -value clean   -label Clean
}

# ---------------------------------------------------------------------------
# mkEntries: Create a frame with the entries  of the dialog window.
# ---------------------------------------------------------------------------
proc mkEntries {w} {
    global value sorted

    pack [frame $w] -fill x -expand 1
    set row 0
    foreach entry $sorted {
	switch -- $value($entry) {
	    +++ {
		frame $w.lab$row -height 2 -relief sunken -borderwidth 1
		#label $w.lab$row -text ---
		grid $w.lab$row -row $row -column 0 -columnspan 2 -sticky ew \
			-pady 5
	    }
	    --- {
		label $w.lab$row -text $entry
		grid $w.lab$row -row $row -column 0 -columnspan 2 -sticky w
	    }
	    default {
		label $w.lab$entry -text $entry
		entry $w.ent$entry -textvariable value($entry)
		grid  $w.lab$entry -row $row -column 0 -sticky w
		grid  $w.ent$entry -row $row -column 1 -sticky ew -padx 10 -pady 1
		grid  columnconfigure $w 1 -weight 100
	    }
	}
	incr row
    }

    return $w
}

# ---------------------------------------------------------------------------
# mkButtons: Create the buttons at the bottom of the dialog window.
# ---------------------------------------------------------------------------
proc mkButtons {w} {
    pack [frame $w] -side bottom
    button $w.sav  -text "Save" -command "action 0"
    button $w.mak  -text "Make" -command "action 1"
    button $w.can -text "Exit" -command exit
    pack $w.sav [makeDefaultButton $w.mak] $w.can \
	    -side left -padx 5 -pady 5 -fill both

    bind . <Return> "$w.mak invoke"
    bind . <F3>     "$w.can invoke"

    return $w
}

# ---------------------------------------------------------------------------
# Procedure, that makes the given button look like the default button.
# Only needed due to backward compatibility with Tk4.x.
# ---------------------------------------------------------------------------
proc makeDefaultButton {button} {
    global tk_version tkCompat

    if [info exists tkCompat(defaultButton)] {
	$button configure -default $tkCompat(defaultButton)
	return $button
    } else {
	regsub {\.[^.]*$} $button ".default" defButton
	frame $defButton -relief sunken -bd 1
	pack  $button -in $defButton -padx 1 -pady 1
	raise $button
	return $defButton
    }
}

proc mkMsgBar {mb} {
    frame $mb
    label .msg -textvariable msg -relief sunken -bd 1 -anchor w
    label $mb.foo -width 8 -relief sunken -bd 1
    pack  .msg -in $mb -side left -padx 2 -expand yes -fill both
    pack  $mb.foo -side left -padx 2
    pack  $mb -side bottom -fill x -pady 2
}

# ---------------------------------------------------------------------------
# The event handler for the Safe or Make button
# ---------------------------------------------------------------------------
proc action {{withMake 0}} {
    global lines msg target

    set msg ""
    set lines [substAdabas95mak $lines]
    writeAdabas95mak $lines
    writeMakefiles . extension applications manuals
    checkVClibs

    if $withMake {
	if [catch {exec cmd.exe /c start "Make $target" cmd.exe /c "nmake $target && pause && exit"} f] {
	    set msg $f
	}
	set msg "Target $target made"
    } else {
	set msg "Configuration saved"
    }
}

# ---------------------------------------------------------------------------
# Since the name and location of the vc libs changed nearly every release,
# this proc checks, if they exist and tells the user, where they are expected.
# ---------------------------------------------------------------------------
proc checkVClibs {} {
    global value

    set missing ""
    foreach lib [list \
	    [file join $value(TCL) lib tcl$value(TCL_VERS)vc.lib] \
	    [file join $value(TCL) lib tk$value(TK_VERS)vc.lib]] {
	if ![file exists $lib] {
	    lappend missing $lib
	}
    }
    if [string length $missing] {
	tk_messageBox -title "Missing libraries" -icon warning \
		-message "The libraries [join $missing " and "] are missing."
    }
}

# ---------------------------------------------------------------------------
# about: Display a few words about Krischan.
# ---------------------------------------------------------------------------
proc about {} {
    tk_messageBox -title "About AdabasTcl" -message "AdabasTcl
Version 1.0\n\nJune 1997\n\nChristian Krone
krischan@cs.tu-berlin.de" -type ok
}

# ----------------------------------------------------------------------
# Now we have to do some limbo to get TclTk's version clear.
# 1. Do we have a Tk, where buttons understand the -default option?
# ----------------------------------------------------------------------
switch -glob $tk_patchLevel {
    tk8.0a? - tk8.0b1 {set tkCompat(defaultButton) 1}
    tk[89]*           {set tkCompat(defaultButton) normal}
}

# ---------------------------------------------------------------------------
# Finally call the "init" procedure...
# ---------------------------------------------------------------------------
mkFrame
