# ---------------------------------------------------------------------
#  $Id: adabastcl.tcl,v 1.18 1997/07/06 22:03:57 adabas Exp $
# ---------------------------------------------------------------------
# Copyright (c) 1996-1997 Christian Krone. All rights reserved.
# This program is free software; you can redistribute it and/or
# modify it under the same terms as Tcl itself.
# See also licence.terms
# ---------------------------------------------------------------------

html {

title "AdabasTcl - Adabas D Database Server access commands for Tcl"

#   ==============
sec "Introduction"
#   ==============

p "AdabasTcl is a Tcl extension providing access to an Adabas D Database Server.
   It is loadable as package and consists of a
   collection of Tcl commands and a Tcl global array,
   Each AdabasTcl command generally invokes several 
   Adabas D library functions."

p "The first of the following four sections will cover all procedures of the
   [emphasis "AdabasTcl call interface"]. The second section give hints, how to
   use the built in Tcl commands to load AdabasTcl into the Tcl interpreter.
   After that comes a description of the [func adabas] command, which provides
   a lower level interface to the database and a section about [emphasis sqlsh],
   a set of commands, that are automatically defined, when AdabasTcl is loaded
   in an interactive session."

#   ==========================
sec "AdabasTcl call interface"
#   ==========================

#    --------
ssec adalogon
#    --------

p [keyboard "adalogon [cgi_variable connect-str] ?[cgi_variable option] ...?"]

p "Connect to an Adabas D server using [cgi_variable connect-str].
   The connect string should be a string in one of the following forms:"

bullet_list {
  li [cgi_variable name][keyboard ,][cgi_variable password]
  li [keyboard ,][cgi_variable userkey]
  li [keyboard ,]
  li [keyboard -noconnect]
}

p "If name and password are given (comma separated), both are translated
   into upper case before they are used to connect to the database."

p "If a [cgi_variable userkey] is given (with a leading comma), the data for
   name and password are extracted from the xuser record.
   If only a comma is given, it stands for the DEFAULT xuser entry."

p "The special connect string [keyboard -noconnect] signals, that only a
   low level connection to the database server is established, but no connection
   at the user level is made. This connection can be used with the
   [keyboard adaspecial] command."

p "A logon handle is returned and should be used for all other AdabasTcl
   commands using this connection that require a logon handle.  
   Multiple connections to the same or different servers are allowed.
   Adalogon raises a Tcl error if the connection 
   is not made for any reason (login or password incorrect, 
   network unavailable, etc.)."

p "[keyboard Option] may be any of the following."

bullet_list {
  li "[keyboard -serverdb]        [cgi_variable serverdb]"
  li "[keyboard -sqlmode]         [cgi_variable sqlmode]"
  li "[keyboard -isolationlevel]  [cgi_variable isolation]"
  li "[keyboard -service]         [cgi_variable service]"
}

p "If a [cgi_variable serverdb] is given, the connection is made to this serverdb.
   Else the value of the environment variable [path SERVERDB], which resides
   in the global Tcl variable [path env(SERVERDB)], is used as the name of
   the server.
   If SERVERDB isn't set, the environment variable DBNAME is tested also."

p "The name of a serverdb may be prefixed by a colon separated hostname.
   An example call of adalogon would be:"
preformatted {
   adalogon demo,adabas -serverdb mycomp:mydb
}

p "If a [cgi_variable sqlmode] is given, the connection is made in this sqlmode.
   [cgi_variable Sqlmode] can be any of [keyboard adabas], [keyboard ansi],
   [keyboard db2] or [keyboard oracle]. Default is sqlmode [keyboard adabas]."

p "If an [cgi_variable isolation] is given, the connection is made with this
   isolation level. [cgi_variable Isolation] can be any of 0, 1, 2, 3 10, 15 or 20."

p "If a [cgi_variable service] is given, the connection is made
   for the given service. [cgi_variable Service] may be any of the following:"
bullet_list {
  li "[keyboard user]: The connection is made for a normal user session;
      this is the default."
  li "[keyboard control]: The connection is made as a user session, but with
      the privilege of the control user. Therefore the [cgi_variable connect-str]
      has to describe the control user."
  li "[keyboard utility]: The connection is made as utility session;
      Therefore the [cgi_variable connect-str] has to describe the control user.
      The returned logon handle can only be given as parameter of
      [keyboard adautil], the other commands with a [cgi_variable logon-handle]
      parameter expect a handle for a user session.[br]
      For a utility service it is not possible to specify an isolation or
      a sqlmode."
}

p "You can create multiple connections to the database server by
   repeatingly calling adalogon. The limit is set by the configuration of
   the database or by the amount of available memory, not by AdabasTcl.
   Since you can create multiple cursors out of one logon handle, it only
   makes sence to have multiple logons with a different user on each
   connection."

p "The returned logon handle for the first connection is the name of
   the serverdb, for the next connections a string in the form
   [keyboard "[cgi_variable serverdb] [hash][cgi_variable n]"] with [cgi_variable n]
   replaced by an increasing number. No programs should depend on this;
   instead you should store the returned logon handle into a variable:"
preformatted {
   set logon [adalogon demo,adabas]
}

#    ---------
ssec adalogoff
#    ---------

p [keyboard "adalogoff [cgi_variable logon-handle]"]

p "Logoff from the Adabas D server connection associated with
   [cgi_variable logon-handle]. [cgi_variable Logon-handle] must be a valid
   handle previously opened with adalogon.
   Adalogoff returns a null string.
   Adalogoff raises a Tcl error if the logon handle specified is not open."

#    -------
ssec adaopen
#    -------

p [keyboard "adaopen [cgi_variable logon-handle]"]

p "Open an SQL cursor to the server. Adaopen returns a cursor to be used
   on subsequent AdabasTcl commands that require a cursor handle.
   [cgi_variable Logon-handle] must be a valid handle previously opened with
   adalogon.
   Multiple cursors can be opened through the same or different logon
   handles.
   Adaopen raises a Tcl error if the logon handle specified is not open."

p "The returned cursor handle is a string in the form
   [keyboard cursor[cgi_variable n]], with [cgi_variable n] replaced by an
   increasing number. No programs should depend on this;
   instead you should store the returned cursor handle into a variable:"
preformatted {
   set cursor [adaopen $logon]
}

#    --------
ssec adaclose
#    --------

p [keyboard "adaclose [cgi_variable cursor-handle]"]

p "Closes the cursor associated with [cgi_variable cursor-handle].
   Adaclose raises a Tcl error if the cursor handle specified is not open."

#    ------
ssec adasql
#    ------

p [keyboard "adasql [cgi_variable cursor-handle] ?[keyboard -command]?
             [cgi_variable sql-statement]        ?[cgi_variable option] ...?"]

p "Send the Adabas D SQL statement [cgi_variable sql-statement] to the server.
   [cgi_variable Cursor-handle] must be a valid handle previously opened with
   adaopen. The argument [keyboard -command] can be omitted.
   Adasql will return the numeric return code 0 on successful
   execution of the  sql statement. The [path adamsg] array index [path rc]
   is set with the return code; the [path rows] index is set to the number
   of rows affected by the SQL statement in the case of [func insert],
   [func update], or [func delete]."

p "Only a single SQL statement may be specified in [cgi_variable sql-statement].  
   Adafetch allows retrieval of return rows generated."

p "[keyboard Option] may be any of the following."
bullet_list {
  li "[keyboard -sqlmode]     [cgi_variable sqlmode]"
  li "[keyboard -resulttable] [cgi_variable resulttable]"
}

p "If [cgi_variable sqlmode] is specified, then the [cgi_variable sql-statement] will
   be parsed and executed in this sqlmode, and not in the session wide sqlmode
   determined by adalogon. Note, that it may be neccesary to give the same
   [keyboard -sqlmode] option, when calling adafetch."

p "If [cgi_variable resulttable] is specified, this will become the name of the 
   resulttable of the [keyboard SELECT] statement. For any other statement
   kind (e.g. UPDATE or CREATE) the [keyboard -resulttable] option will be
   ignored."

p "You have to give an explicit resulttable name, if you want to fetch from
   more than one cursor at once. The resulttable names can be arbitrary
   (max. 18 characters long), but should be distinct between all cursors of
   one logon handle, you want to fetch from."

p "Adasql performs an implicit [func adacancel], if any results are still
   pending from the last execution of adasql.
   Adasql raises a Tcl error if the cursor handle specified is not open, if the
   SQL statement is syntactically incorrect or if no data was found
   for a SELECT or no row was affected by an UPDATE or DELETE command."

p "If the given [cgi_variable sql-statement] denotes a [emphasis "select into"],
   the selected values are stored directly into the mentioned parameters.
   The parameter names may denote ordinary variables or arrays. E.g."
preformatted {
adasql $cursor {SELECT DATE, TIME INTO :res(date), :res(time) FROM dual}   
}
p "After the above call of adasql the current date and time is stored
   into the array variable [keyboard res] with the indexes [keyboard date]
   and [keyboard time]."

p "There exist some alternative forms of the adasql command, where the second
   argument is a specification of the kind, the statement is handled:"

p [keyboard "adasql [cgi_variable cursor-handle] [keyboard -parameter]
             [cgi_variable sql-statement]        ?[cgi_variable option] ...?"]

p "In this case the [cgi_variable sql-statement] is executed in three steps:
   First it is parsed by the database server and a so called
   [cgi_variable parsId] is returned. This [cgi_variable parsId] is then immediately
   given for execution together with the values of all mentioned parameters.
   At last the [cgi_variable parsId] is dropped from the database catalogue.
   Note that this way there are three communication steps instead of one."

p "The good side of this is, that the [cgi_variable sql-statement] can contain
   parameter specifications."
preformatted {
   adasql $cursor -parameter "SELECT * FROM tab
                                WHERE numb >= :limit
                                  AND name BETWEEN :lower AND :upper
}
p "There are no string delimiter around [keyboard :lower] or [keyboard :upper],
   since the substitution with the current values and its conversion are done
   by the database server and not by the Tcl interpreter."

p "There exist options to do each of the three above mentioned steps separate:"
p [keyboard "adasql [cgi_variable cursor-handle] [keyboard -parse]
             [cgi_variable sql-statement]        ?[cgi_variable option] ...?"]
p [keyboard "adasql [cgi_variable cursor-handle] [keyboard -execute]
                    [cgi_variable parsId]"]
p [keyboard "adasql [cgi_variable cursor-handle] [keyboard -drop]
                    [cgi_variable parsId]"]

#    --------
ssec adafetch
#    --------

p [keyboard "adafetch [cgi_variable cursor-handle] ?option...?"]

p "[keyboard option] may be any of the following."
bullet_list {
  li "[keyboard -position]   [keyboard first]"
  li "[keyboard -position]   [keyboard last]"
  li "[keyboard -position]   [keyboard next]"
  li "[keyboard -position]   [keyboard prev]"
  li "[keyboard -position]   [cgi_variable number]"
  li "[keyboard -count]      [cgi_variable mass-count]"
  li "[keyboard -command]    [cgi_variable commands]"
  li "[keyboard -array]      [cgi_variable onOff]"
  li "[keyboard -sqlmode]    [cgi_variable sqlmode]"
}

p "Return the next row from the last SQL statement executed with adasql as
   a Tcl list. [cgi_variable Cursor-handle] must be a valid handle previously
   opened with adaopen.
   Adafetch raises a Tcl error if the cursor handle specified is not open.
   All returned columns are converted to character strings. (This is not
   completely true for Tcl starting at version 8, since there any numeric
   values (FIXED or FLOAT) is returned as number object. But you shouldn't
   notice any difference from the script level.)"

p "An empty list is returned if there are no more rows in the current set
   of results. The Tcl list that is returned by adafetch contains the values
   of the selected columns in the order specified by SELECT."

p "If the option [keyboard -array] specifies a true value (e.g. [keyboard 1]
   or [keyboard on]), the resulting list will be in a format, that can be
   given as last argument to an array set command.
   An example session follows."
preformatted {
    % adasql $c {SELECT USER "my_name" FROM dual}
    % array set x [adafetch $c -array 1]
    % set x(my_name)
    krischan
}

p "The cursor can be moved in any direction by means of the [func -position]
   option. Its associated value can have any of the forms [keyboard first],
   [keyboard last], [keyboard next], [keyboard prev] or it can be a
   [cgi_variable number]. The textual variants specify the direction, in which
   the cursor should be moved. A direction of e.g. [path first] rewinds
   the cursor to the start of the result set. A [cgi_variable number]
   denotes the rownum of the result row to fetch; the first row has the
   rownum [keyboard 1]. Default position is [keyboard next]."

p "It may be nessecary, to give the same [keyboard -sqlmode] option, as you
   give to adasql, since some mode dependend computations (e.g. date
   format) are delayed to the call of adafetch."

p "By means of the optional [cgi_variable -command] argument adafetch can
   repeatedly fetch rows and execute [cgi_variable commands] as a tcl script
   for each row.
   Substitutions are made on [cgi_variable commands] before passing it to
   [func Tcl[us]Eval()] for each row.  Adafetch interprets [keyboard @n]
   in commands as a result column specification.  For example,
   [keyboard "@1, @2, @3"] refer to the first, second, and third columns
   in the result. [keyboard @0] refers to the entire result row, as a Tcl
   list. Substitution columns may appear in any order, or more than once
   in the same command.
   Substituted columns are inserted into the commands string as
   proper list elements, i.e., one space will be added before and after the
   substitution and column values with embedded spaces are enclosed by
   braces if needed."

p "A Tcl error is raised if a column substitution number is greater than the
   number of columns in the results.  If the commands execute [path break],
   adafetch execution is interrupted and returns with [path TCL[us]OK].
   Remaining rows may be fetched with a subsequent adafetch command.  
   If the commands execute [path return] or [path continue], the remaining
   commands are skipped and adafetch execution continues with the next row.
   Adafetch will raise a Tcl error if the [emphasis commands] return an error.
   Commands should be enclosed in double quotes or braces."

p "Adatcl performs conversions for all data types.
   For long data only a long descriptor is returned, which can be used as
   argument for a subsequent call of [func adareadlong] with its
   [keyboard -descriptor] option."

p "The [path adamsg] array index [path rc] is set with the return code
   of the fetch.  0 indicates the row was fetched successfully;
   100 indicates the end of data was reached."

p "The [path adamsg] array index [path nullvalue] can be set to specify
   the value returned when a column is null. The default is an empty
   string for values of all datatypes."

p "The [path adamsg] array index [path specialnull] can be set to specify
   the value returned when a column is the special null value. The default
   is the string [keyboard ???] for values of all datatypes."

#    -------
ssec adacols
#    -------

p [keyboard "adacols [cgi_variable cursor-handle]"]

p "Return the names of the columns from the last adasql or adafetch
   command  as a Tcl list."

p "As a side effect of this command the gloabl array [path adamsg] is
   updated with some additional information about the selected columns.
   The [path adamsg] array index [path collengths] is set to a Tcl list
   corresponding to the lengths of the columns; index [path coltypes] is
   set to a Tcl list corresponding to the types of the columns;
   index [path colprecs] is set to a Tcl list corresponding to the precision
   of the numeric columns, other corresponding non-numeric columns got its
   length as precision; index [path colscales] is set to a Tcl list
   corresponding to the scale of the numeric columns,
   other corresponding non-numeric columns are 0;
   Adacols raises a Tcl error if the cursor handle specified is not open."

#    ---------
ssec adacancel
#    ---------

p [keyboard "adacancel [cgi_variable cursor-handle]"]

p "Cancels any pending results from a prior adasql command that use a
   cursor opened through the connection specified by
   [cgi_variable cursor-handle]. [cgi_variable Cursor-handle] must be a valid
   handle previously opened with [func adaopen].
   Adacancel raises a Tcl error if the cursor handle specified is not open."

p "Note, that this command doesn't cancel any long running sql statemant
   (though it should)."

#    ---------
ssec adacommit
#    ---------

p [keyboard "adacommit [cgi_variable logon-handle]"]

p "Commit any pending transactions from prior adasql commands that use a
   cursor opened through the connection specified by [cgi_variable logon-handle].  
   [cgi_variable Logon-handle] must be a valid handle previously opened with
   [func adalogon].
   Adacommit raises a Tcl error if the logon handle specified is not open."

#    -----------
ssec adarollback
#    -----------

p [keyboard "adarollback [cgi_variable logon-handle]"]

p "Rollback any pending transactions from prior adasql commands that use a
   cursor opened through the connection specified by [cgi_variable logon-handle].
   [cgi_variable Logon-handle] must be a valid handle previously opened with
   adalogon.
   Adarollback raises a Tcl error if the logon handle specified is not open."

#    ----------
ssec adaautocom
#    ----------

p [keyboard "adaautocom [cgi_variable logon-handle] [cgi_variable on-off]"]

p "Enables or disables automatic commit of SQL data manipulation 
   statements using a cursor opened through the connection specified
   by logon handle. [cgi_variable Logon-handle] must be a valid handle
   previously opened with adalogon.
   [cgi_variable on-off] must be a valid Tcl boolean value ([path 0], [path 1],
   [path false], [path true], [path no], [path yes], [path on],
   [path off] or abbreviations thereof).
   Adaautocom raises a Tcl error if the logon handle specified is not open."

#    -----------
ssec adareadlong
#    -----------

p [keyboard "adareadlong [cgi_variable cursor-handle] ?option ...?"]

p "[keyboard Option] may be any of the following."
bullet_list {
  li "[keyboard -table]      [cgi_variable table-name]"
  li "[keyboard -column]     [cgi_variable column-name]"
  li "[keyboard -where]      [cgi_variable where-condition]"
  li "[keyboard -descriptor] [cgi_variable long-descriptor]"
  li "[keyboard -filename]   [cgi_variable filename]"
  li "[keyboard -encoding]   [cgi_variable encoding]"
}

p "Read the contents of a LONG column and returns the result, or writes it
   into a file, if the optional parameter [cgi_variable filename] is specified.
   [cgi_variable Cursor-handle] must be a valid handle previously opened with
   adaopen."

p "The switches can be given in any order, but there must be present either
   the [keyboard -descriptor] switch or all of [keyboard -table],
   [keyboard -column] and (most probably) [keyboard -where] switches."

p "With [keyboard -table], [keyboard -column] and [keyboard -where] the
   column to be read can be specified, as if a select in the following
   form was specified:"

p "[keyboard SELECT] [cgi_variable column-name]
   [keyboard FROM]   [cgi_variable table-name]
   [keyboard WHERE]  [cgi_variable where-condition]"

p "The given column must be of data type [path LONG] and the
   [cgi_variable where-condition] must limit the resultcount of the
   above select statement to 1."

p "[cgi_variable Long-descriptor] is the Adabas long descriptor of a
   recently fetched row, as returned by adafetch."

p "Here are two examples, how to use the two variants of this command:"
preformatted {
adasql $cursor "SELECT longval FROM tab WHERE keyval=1"
set row [adafetch $cursor]
set val1 [adareadlong $cursor -descriptor [lindex $row 0]]

set val2 [adareadlong $cursor -table tab -column longval -where "keyval=1"]
}

p "[cgi_variable Filename] is the name of a file in which to write the LONG data."

p "If called with optional parameter [cgi_variable filename], adareadlong returns
   upon successful completion the number of bytes read from the LONG column."

p "The resulting string will be decoded, since it may contain characters,
   that are not printable, or even null characters. You can specify the
   encoding kind with the [keyboard -encoding] option. [cgi_variable Encoding]
   must be one out of the following list."
bullet_list {
  li "[keyboard escape]: This is the default encoding. All characters,
      that are not printable according to [keyboard isprint()], are printed
      in octal notion with a leading backslash."
  li "[keyboard hex]: All characters (even printable) are printed as a pair
      of two hexadecimal numbers."
  li "[keyboard base64]: An encoding format, that will be understood by the
      [keyboard image] command from Tk8.0 on. This way there is no need to
      store an image temporarily onto disk to display it.
      You can use a statement like the following instead:"
preformatted {
image create photo -data \ 
    [adareadlong $cursor -table people -column picture \ 
                -where "name='Krischan'" -encoding base64]
}
}
   
p "Adareadlong raises a Tcl error if the cursor handle specified is not open,
   [keyboard -descriptor] is given and either the long descriptor specified
   is invalid or no call of adafetch precedes the call of adareadlong,
   or if the [cgi_variable where-condition] doesn't limit to a single result."

#    ------------
ssec adawritelong
#    ------------

p [keyboard "adawritelong [cgi_variable cursor-handle] ?option ...?"]

p "[keyboard Option] may be any of the following."
bullet_list {
  li "[keyboard -table]    [cgi_variable table-name]"
  li "[keyboard -column]   [cgi_variable column-name]"
  li "[keyboard -where]    [cgi_variable where-condition]"
  li "[keyboard -value]    [cgi_variable long-value]"
  li "[keyboard -filename] [cgi_variable filename]"
  li "[keyboard -encoding] [cgi_variable encoding]"
}

p "Read the contents of a file or the given string and store into a LONG column.
   [cgi_variable Cursor-handle] must be a valid handle previously opened with
   adaopen."

p "The switches can be given in any order, but there must be present either
   the [keyboard -value] or [keyboard -filename] switch and all of
   [keyboard -table], [keyboard -column] and [keyboard -where] switches."

p "With [keyboard -table], [keyboard -column] and [keyboard -where] the
   column to be read must be specified, as if an update in the following
   form was specified:"

p "[keyboard UPDATE] [cgi_variable table-name]
   [keyboard SET]    [cgi_variable column-name] = '[cgi_variable value-to-be-inserted]'
   [keyboard WHERE]  [cgi_variable where-condition]"

p "The given column must be of data type [path LONG] and the
   [cgi_variable where-condition] must limit the resultcount of the
   above update statement to 1."

p "[cgi_variable Filename] is the name of a file out of which to read the
   LONG data. [cgi_variable Long-value] is the LONG data itself."

p "The string to insert will be decoded, since only this way it is possible
   to insert LONG columns containing characters, that are not printable, or
   even null characters. You can specify the encoding kind with the
   [keyboard -encoding] option. [cgi_variable Encoding] must be one out of the
   list, you can find above by the adareadlong command."

p "Adawritelong raises a Tcl error if the cursor handle specified is not open
   or if the [cgi_variable where-condition] doesn't limit to a single result."

#    ----------
ssec adaspecial
#    ----------

p [keyboard "adaspecial [cgi_variable logon-handle] [cgi_variable command]
                        ?[cgi_variable params]...?"]

p "[cgi_variable Logon-handle] must be a handle returned by a previous call of
   adalogon. Adaspecial is the only command, where logon handles created with
   the [keyboard -noconnect] option can be given."

p "[cgi_variable Command] and [cgi_variable params] may be any of the following."
bullet_list {
  li "[keyboard hello]"
  li "[keyboard switch]      [cgi_variable layer] [cgi_variable debug]"
  li "[keyboard switchlimit] [cgi_variable layer] [cgi_variable debug]
                             [cgi_variable start-layer] [cgi_variable start-proc]
                             [cgi_variable end-layer]   [cgi_variable end-proc]
                             [cgi_variable count]"
  li "[keyboard minbuf]"
  li "[keyboard maxbuf]"
  li "[keyboard buflimit]    [cgi_variable length]"
}

p "All these commands except [keyboard hello] are only available, if a slow
   database kernel is started, and are mainly fotr debugging of the
   database server. The [keyboard hello] command can be useed to avoid a timeout
   fot the given connection."

#    --------
ssec adausage
#    --------

p [keyboard "adausage [cgi_variable logon-handle] [cgi_variable usage-kind]
                      ?[cgi_variable option] ...?"]

p "[cgi_variable Usage-kind] must be [keyboard on], [keyboard off] or
   [keyboard add]. A call of adausage with usage kind [keyboard on] will
   inform the database kernel, that subsequent calls of adasql with the
   [keyboard -parse] option should be analyzed for usage relations. A call of
   adausage with usage kind [keyboard off] will fire the ddl triggers of
   Adabas D, which will update the data dictionary."

p "[cgi_variable option] can be [keyboard -objecttype] (maximal eight character)
   or [keyboard -parameters] (a list of up to 3 identifiers)."

p "As example here are the calls of adausage and adasql to store a
   SQL statement as stored query command:"
preformatted {
adausage $logon on -objecttype QUERYCOM -parameters [list $name]
adasql   $cursor -parse $sqlStatement
adausage $logon off
}
#    -------
ssec adautil
#    -------

p [keyboard "adautil [cgi_variable logon-handle] [cgi_variable utility-command]"]

p "Send the Adabas D utility command [cgi_variable utility-command] to the server.
   [cgi_variable Logon-handle] must be a valid handle previously opened with
   adalogon with the session specified as [keyboard utility].
   The return value depends heavily on the specified command."

#    --------------------------------------
ssec "Server message and error information"
#    --------------------------------------

p "AdabasTcl creates and maintains a Tcl global array to provide feedback of
   Adabas D server messages, named [path adamsg]. [path Adamsg] is also used
   to communicate with the AdabasTcl interface routines to
   specify null and specialnull return values.
   In all cases except for [path nullvalue], [path specialnull] and [path version],
   the contents of each element may be changed
   upon invocation of any AdabasTcl command, and any 
   element affected by the command is set. The [path adamsg] array is shared 
   among all open AdabasTcl handles.  [path Adamsg] should be defined with the
   global statement in any Tcl procedure needing access to [path adamsg].
   The following list defines all indexes of [path adamsg]."

definition_list {
  term [path nullvalue]
  term_definition "can be set by the programmer to indicate the string value
                   returned for any null result. Setting
                   [path adamsg(nullvalue)] to an empty string or unsetting
                   it will return an empty string for all null values.
                   [path Nullvalue] is initially set to an empty string."
  term [path specialnull]
  term_definition "can be set by the programmer to indicate the string value
                   returned for any result, that has the special null value.
                   Setting [path adamsg(specialnull)] to an empty string or
                   unsetting it will return an empty string for all
                   special null values.
                   [path Specialnull] is initially set to the string
                   [keyboard ***]."
  term [path tracefile]
  term_definition "can be set by the programmer to indicate that a trace of
                   every call of an API functions should be written.
                   A file with the name of this variable will be created; if
                   it already exists, its contents will be deleted."
  term [path handle]
  term_definition "indicates the handle of the last AdabasTcl command.
                   [path Handle] is set on every AdabasTcl command (execpt
                   where an invalid handle is used.)"
  term [path rc]
  term_definition "indicates the results of the last SQL command and
                   subsequent adafetch processing. [path Rc] is set by
                   [func adasql], [func adafetch], and is the numeric
                   return code from the last library function called by
                   an AdabasTcl command.  Refer to Adabas D Error Messages
                   and Codes manual for detailed information."

                p "Typical values are:"
                   bullet_list {
                     li "0:   Function completed normally, without error."
                     li "100: End of data was reached on an adafetch command,
                              no data was found for a select on a adasql
                              command or no row was affected by an update
                              or delete command on a adasql command."
                     li "all other returncodes unequal to 0 or 100:
                              invalid SQL statement, invalid sql statements,
                              missing keywords, invalid column names, no sql
                              statement, logon denied, insufficient
                              privileges, etc. The returncode corresponds
                              to the number in the messages manual of
                              Adabas D."
                   }
  term [path errortxt]
  term_definition "the message text associated with [path rc]."
  term [path collengths]
  term_definition "is a Tcl list of the lengths of the columns returned by
                   adacols. Collengths is only set by adacols."
  term [path coltypes]
  term_definition "is a Tcl list of the types of the columns returned by
                   [func adacols]. [path Coltypes] is only set by adacols.
                   Possible types returned are:
                   [path fixed], [path float], [path char[us]ascii],
                   [path char[us]ebcdic], [path char[us]byte], [path rowid],
                   [path long[us]ascii], [path long[us]ebcdic], [path long[us]byte],
                   [path long[us]dbyte], [path long[us]unicode], [path date],
                   [path time], [path vfloat], [path timestamp],
                   [path duration], [path dbyte[us]ebcdic], [path boolean],
                   [path unicode], [path smallint], [path integer],
                   [path varchar[us]ascii], [path varchar[us]ebcdic],
                   [path varchar[us]byte] or [path unknown]"
  term [path colprecs]
  term_definition "is a Tcl list of the precision of the numeric columns
                   returned by adacols. Colprecs is only set by adacols.
                   For non-numeric columns, the list entry is a zero."
  term [path colscales]
  term_definition "is a Tcl list of the scale of the numeric columns returned
                   by adacols. Colprecs is only set by adacols.
                   For non-numeric columns, the list entry is a zero."
  term [path rows]
  term_definition "the number of rows affected by an [func insert],
                   [func update], or [func delete] in an adasql command."
  term [path version]
  term_definition "a string containing the version of AdabasTcl and the
                   version of the Adabas D server, it is compiled for, in
                   the form of e.g.
                   [screen "AdabasTcl 1.0 (for Adabas D 6.1)"]."
}

#   =============================
sec "Using the AdabasTcl package"
#   =============================

p "As the introduction already said, AdabasTcl is an extension to Tcl.
   It is available as package with the name [path Adabastcl],
   following the Tcl conventions with capitalized package names."

sssec "Interpreters"

p "On Unix systems with AdabasTcl installed there exist at least the following
   four interpreters:"
number_list {
  li "[path tclsh] (perhaps with a version number tagged at the end):
      The [emphasis pure] interpreter containing just Tcl."
  li "[path wish] (perhaps with a version number tagged at the end):
      The Tcl interpreter with the Tk package included."
  li "[path adabastclsh]: The Tcl interpreter with the Adabastcl package included."
  li "[path adabaswish]: The Tcl interpreter with the Adabastcl and the Tk package
      included."
}
p "On a Windows system there are no interpreters with the AdabasTcl package
   included, so there exist only tclsh and wish."

sssec "Libraries"

p "In every case there is a shared library on Unix systems (with the extension
   [path .so] or [path .sl]) or a dynamic link library on Windows systems (with
   the extension [path .dll]) waiting in the [path lib] subdirectory of
   [path [dollar]DBROOT].
   You can load it into your current interpreter with a platform independent
   command like this:"
preformatted {
load [file join $env(DBROOT) lib Adabastcl[info sharedlibextension]]
}
p "If you have an interpreter program with AdabasTcl already included (e.g.
   [path adabaswish]), you can call it with an empty first argument and the
   package name as additional second argument, like this:"
preformatted {
load {} Adabastcl
}
p "Although this seems to be useless on the first look, it is useful, if you are
   working with multiple interpreter."

sssec Packages

p "The most elegant way to get the AdabasTcl commands known by the interpreter is
   not to use the [path load] command at all, but the [path package] command.
   It works the same, if AdabasTcl is built in or not: First you append
   the subdirectory lib of [path [dollar]DBROOT] to your [path auto_path] and then
   simply call [path "package require"] like this:"
preformatted {
lappend auto_path [file join $env(DBROOT) lib]
package require AdabasTcl
}

sssec "Where to look"

p "All the Tk applications in [dollar]DBROOT/bin (e.g. tkquery) starts with a
   procedure [path loadAdabastcl], which is responsible to make the 
   AdabasTcl extension available in the current interpreter. You are invited
   to look and improve."

#   ====================
sec "The adabas command"
#   ====================

p "The [func adabas] command provides a low level interface to the
   Adabas D database server. In deed, it is so low level, that you must
   know some internals about the order interface between applications and
   kernel, to get the most of the command variants functional."

p "In theory, it is possible to rebuild all the above given commands of
   the AdabasTcl call interface as purely [prog Tcl] commands by means of
   this command. But its performance is very bad, since all data at this
   level is binary, and [prog Tcl] isn't very good in handling strings
   with null characters in it."

p "Note, that this command is likely to deminish in future versions of
   AdabasTcl. Only the [keyboard "adabas xuser"] will be available further on
   with a different name (like [keyboard adaxuser])."

ssec "adabas version"

ssec "adabas crypt"
ssec "adabas termId"

ssec "adabas get"
ssec "adabas put"

ssec "adabas fget"
ssec "adabas fapp"
ssec "adabas sget"

ssec "adabas xuser"
bullet_list {
  li "[keyboard args]"
  li "[keyboard clear]"
  li "[keyboard close]"
  li "[keyboard get]"
  li "[keyboard index]"
  li "[keyboard open]"
  li "[keyboard put]"
}

ssec "adabas open"

#   =====================================
sec "A SQL shell for interactive queries"
#   =====================================

p "If [prog adabastclsh] was called without a filename to source, and
   therefore [keyboard tcl[us]interactive] is set to 1, it reads at startup
   time a file called [path ~/.adabastclshrc]. In this procedure you
   can e.g. set your prompt."

p "If AdabasTcl is loaded into an interactive Tcl interpreter (as static package
   or via the [keyboard load] command), a bunch of convinience commands, that have
   the names of all the Adabas SQL commands, are defined."

bullet_list {
  li alter
  li clear
  li commit
  li comment
  li connect
  li create
  li delete
  li drop
  li exists
  li explain
  li grant
  li insert
  li monitor
  li refresh
  li rename
  li revoke
  li rollback
  li select
  li show
  li switch
  li update
  li vtrace
}

p "The Tcl commands [func rename] and [func update] are accessible as
   [func tcl[us]rename] and [func tcl[us]update]. The [func switch] statement
   does its best to determine, if the Adabas SWITCH statement or the Tcl
   switch procedure is meant."

p "The [func connect] opens a connection to the database, that keeps
   established, until a session timeout occured or a [func commit] or
   [func rollback] with the [keyboard release] option was given."

p "After a [func select] statement, that returns no error and contains no
   [keyboard into] clause, all results are fetched in portions of 25 lines.
   At the prompt the user can quit the fetch by typing [keyboard q], can
   scroll the remaining lines without further prompting by typing
   [keyboard n] or can see the next lines by typing any other key."

p "All other commands are just send to the database kernel and the
   resulting errormessage, if any, is returned as error."
 
p "So an example session could look like the following."

preformatted {
  connect krischan geheim
  select date, time into :x, :y from dual
  select * from order where order_date = '$x'
  delete from account where account_date <> '$x'
  commit work release
}

p "Beside the above given SQL commands there are two more commands defined:
   [func utility] to connect as control user on the utility service and
   [func util], to perform some commands with this connection.
   Again an example:"

preformatted {
  utility control control
  util state
  util diagnose tabid krischan.adresse
  util commit release
}

#   =======================
sec "Environment variables"
#   =======================

definition_list {
  term [path SERVERDB]
  term_definition "The default Adabas D server name. If not set, the
                   variable [path DBNAME] is inspected also."
  term [path ADABASTCL[us]DIR]
  term_definition "The directory to look for the startup scripts;
                   also the init value for the global Tcl variable
                   [path adabastcl[us]library]."
}

#   =======
sec "Files"
#   =======

definition_list {
  term [path [dollar]HOME/.XUSER]
  term_definition or
  term [path [dollar]HOME/.XUSER.62]
  term_definition "The xuser file to look for connect parameters."
}
  
#   =======
sec "Notes"
#   =======

bullet_list {
  li "The environment variable [path ADABASTCL[us]DIR] should be ignored. The path
      of the library directory should be [path [dollar]DBROOT/tcl] only."
}

#   ======
sec "Bugs"
#   ======

bullet_list {
  li "The [keyboard -parameter] option of [keyboard adasql] isn't implemented
      completely. Instead it will yield an SQL system error from the database
      kernel, if there are host variables mentioned in the sql statement."
}

#   =========
sec "Credits"
#   =========

p "AdabasTcl is deeply inspired by [path orasql] (SQL interface to Oracle
    server) by Tom Poindexter"


#   ========
sec "Author"
#   ========

p "Christian Krone, Berlin. Version 1.0, July 1997."
p [path krischan@cs.tu-berlin.de]

}
