#  LAST EDIT: Sat Jul 20 14:18:38 1996 by Christian Krone (gutamil.v12-berlin.de!krischan) 
##################################################
#
# cgi.tcl - routines for writing CGI scripts in Tcl
# Author: Don Libes <libes@nist.gov>, January '95
#
# These routines implement the code described in the paper
# "Writing CGI scripts in Tcl" which appeared in the Tcl '96 conference.
# Please read the paper before using this code.  The paper is:
# http://www.cme.nist.gov/msid/pubs/libes96c.ps
#
##################################################

##################################################
# http header support
##################################################

proc cgi_http_head {args} {
    global _cgi env errorInfo

    if [info exists _cgi(http_head_done)] return

    set _cgi(http_head_in_progress) 1

    if {0 == [llength $args]} {
	cgi_content_type
    } else {
	if [catch {uplevel [lindex $args 0]} errMsg] {
	    set savedInfo $errorInfo
	    cgi_content_type
	}
    }
    puts ""

    unset _cgi(http_head_in_progress)
    set _cgi(http_head_done) 1

    if [info exists savedInfo] {
	error $errMsg $savedInfo
    }
}

# avoid generating http head if not in CGI environment
# to allow generation of pure HTML files
proc cgi_http_head_implicit {} {
    global env

    if [info exists env(REQUEST_METHOD)] cgi_http_head
}

# If these are called manually, they automatically generate the extra newline

proc cgi_content_type {args} {
    global _cgi

    if 0==[llength $args] {
	set t text/html
    } else {
	set t [lindex $args 0]
    }

    if [info exists _cgi(http_head_in_progress)] {
	puts "Content-type: $t"
    } else {
	cgi_http_head "cgi_content_type $t"
    }
}

proc cgi_location {t} {
    global _cgi

    if [info exists _cgi(http_head_in_progress)] {
	puts "Location: $t"
    } else {
	cgi_http_head "cgi_location $t"
    }
}

proc cgi_target {t} {
    if ![info exists _cgi(http_head_in_progress)] {
	error "cgi_target must be set from within cgi_http_head."
    }
    puts "Window-target: $t"
}

# Make client retrieve url in this many seconds ("client pull").
# With no 2nd arg, current url is retrieved.
proc cgi_refresh {seconds {url ""}} {
    if ![info exists _cgi(http_head_in_progress)] {
	error "cgi_refresh must be set from within cgi_http_head.  Try using cgi_http_equiv instead."
    }
    puts "Refresh: $seconds"

    if {0==[string compare $url ""]} {
	puts "; $url"
    }
}

# Example: cgi_pragma no-cache
proc cgi_pragma {arg} {
    if ![info exists _cgi(http_head_in_progress)] {
	error "cgi_pragma must be set from within cgi_http_head."
    }
    puts "Pragma: $arg"
}

##################################################
# support for debugging or other crucial things we need immediately
##################################################

proc cgi_comment	{args}	{}	;# need this asap

proc cgi_html_comment	{args}	{
    regsub -all {>} $args {\&gt;} args
    puts "<!--[cgi_list_to_string $args] -->"
}

proc cgi_debug {args} {
    global _cgi

    set arg [lindex $args 0]
    if {$arg == "-on"} {
	set _cgi(debug) 1
	set args [lrange $args 1 end]
    } elseif {$arg == "-off"} {
	set _cgi(debug) 0
	set args [lrange $args 1 end]
    } elseif {[regexp "^-t" $arg]} {
	set old $_cgi(debug)
	set _cgi(debug) 1
	set args [lrange $args 1 end]
    } elseif {[regexp "^-noprint$" $arg]} {
	set noprint 1
	set args [lrange $args 1 end]
    }

    set arg [lindex $args 0]
    if {$arg == "--"} {
	set args [lrange $args 1 end]
    }

    if {[llength $args]} {
	if $_cgi(debug) {

	    cgi_close_tag
	    # force http head and open html, head, body
	    catch {
		if [info exists noprint] {
		    uplevel [lindex $args 0]
		} else {
		    cgi_html {
			cgi_head {
			    cgi_title "debugging before complete HTML head"
			}
			# force body open and leave open
			cgi_body_start
			uplevel [lindex $args 0]
			# bop back out to catch, so we don't close body
			error "ignore"
		    }
		}
	    }
	}
    }

    if [info exists old] {
	set _cgi(debug) $old
    }
}

proc cgi_uid_check {user} {
    global _cgi env

    if [regexp "^-off$" $user] {
	set _cgi(ignore_uid) 1
	return
    }

    if [info exists _cgi(checked_uid)] return
    set _cgi(checked_uid) 1

    if {![info exists _cgi(ignore_uid)] && ![info exists env(REQUEST_METHOD)]} {
	if {0==[catch {exec whoami} whoami]} {
	    if {$whoami != "$user"} {

		error \
"Warning: This CGI script expects to run with uid \"$user\".  However,
this script is running as \"$whoami\".  This may allow your scripts to
work during development but fail during production.  To avoid this
trap, su to nobody before running CGI scripts by hand.  To ignore this
advice and suppress this message call cgi_uid_ignore immediately after
loading this Tcl-CGI support package."
	    }
	}
    }
} 

# print out elements of an array
# like Tcl's parray, but formatted for browser
proc cgi_parray {a {pattern *}} {
    upvar 1 $a array
    if ![array exists array] {
	error "\"$a\" isn't an array"
    }
    puts "<xmp>"
    set maxl 0
    foreach name [lsort [array names array $pattern]] {
	if {[string length $name] > $maxl} {
	    set maxl [string length $name]
	}
    }
    set maxl [expr {$maxl + [string length $a] + 2}]
    foreach name [lsort [array names array $pattern]] {
	set nameString [format %s(%s) $a $name]
	puts [format "%-*s = %s" $maxl $nameString $array($name)]
    }
    puts "</xmp>"
}

proc cgi_eval {cmd} {
    global env _cgi

    # put cmd somewhere that uplevel can find it
    set _cgi(body) $cmd

    uplevel #0 {
	if 1==[catch $_cgi(body)] {
	    # error occurred, handle it

	    set _cgi(errorInfo) $errorInfo

	    # the following code is all to force browsers into a state
	    # such that diagnostics can be reliably shown

	    # close irrelevant things
	    cgi_close_procs
	    # force http head and open html, head, body
	    cgi_html {
		cgi_body {
		    cgi_h3 "An internal error was detected in the service\
			    software.  The diagnostics are being emailed to\
			    the service system administrator."

		    if $_cgi(debug) {
			puts "Heck, since you're debugging, I'll show you the\
				errors right here:"
			# suppress formatting
			puts "<xmp>$_cgi(errorInfo)</xmp>"
		    } else {
			cgi_mail_start $_cgi(admin_email)
			cgi_mail_add "Subject: [cgi_name] CGI problem"
			cgi_mail_add
			if {[info exists env(REQUEST_METHOD)]} {
			    cgi_mail_add "CGI environment:"
			    cgi_mail_add "REQUEST_METHOD: $env(REQUEST_METHOD)"
			    cgi_mail_add "SCRIPT_NAME: $env(SCRIPT_NAME)"
			    # this next few things probably don't need
			    # a catch but I'm not positive
			    catch {cgi_mail_add "HTTP_USER_AGENT: $env(HTTP_USER_AGENT)"}
			    catch {cgi_mail_add "REMOTE_ADDR: $env(REMOTE_ADDR)"}
			    catch {cgi_mail_add "REMOTE_HOST: $env(REMOTE_HOST)"}
			}
			cgi_mail_add "input:"
			catch {cgi_mail_add $_cgi(input)}
			cgi_mail_add "cookie:"
			catch {cgi_mail_add $env(HTTP_COOKIE)}
			cgi_mail_add "errorInfo:"
			cgi_mail_add "$_cgi(errorInfo)"
			cgi_mail_end
		    }
		} ;# end cgi_body
	    } ;# end cgi_html
	} ;# end catch
    } ;# end uplevel
}

##################################################
# CGI URL creation
##################################################

# declare location of root of CGI files
# this allows all CGI references to be relative in the source
# making it easy to move everything in the future
# If you have multiple roots, just don't call this.
proc cgi_root {root} {
    global _cgi

    set _cgi(root) $root
}

# make a URL for a CGI script
proc cgi_cgi {x} {
    global _cgi

    return \"$_cgi(root)$x.cgi\"
}

##################################################
# URL dictionary support
##################################################

proc cgi_link {args} {
    global _cgi_link

    set tag [lindex $args 0]
    if {[llength $args] >= 3} {
	set _cgi_link($tag) [eval cgi_url [lrange $args 1 end]]
    }
    return $_cgi_link($tag)
}

# same as above but for images
# note: uses different namespace
proc cgi_imglink {args} {
    global _cgi_link

    set tag [lindex $args 0]
    if {[llength $args] >= 3} {
	set _cgi_imglink($tag) [eval cgi_img [lrange $args 1 end]]
    }
    return $_cgi_imglink($tag)
}

##################################################
# hyperlink support
##################################################

# construct a hyperlink labeled "display"
# last arg is the link destination
# any other args are passed through into <a> display
proc cgi_url {display args} {
    set buf "<a href=\"[lindex $args 0]\""
    foreach a [lrange $args 1 end] {
	if {[regexp "^target=(.*)" $a dummy str]} {
	    append buf " target=\"$str\""
	} else {
	    append buf " $a"
	}
    }
    return "$buf>$display</a>"
}

# generate an image reference (<img ...>)
# last arg is image url
# other args are passed through into <img> tag
proc cgi_img {args} {
    set buf "<img src=\"[lindex $args 0]\""
    foreach a [lrange $args 1 end] {
	if {[regexp "^alt=(.*)" $a dummy str]} {
	    append buf " alt=[cgi_dquote_html $str]"
	} elseif {[regexp "^width=(.*)" $a dummy str]} {
	    append buf " width=[cgi_dquote_html $str]"
	} elseif {[regexp "^height=(.*)" $a dummy str]} {
	    append buf " height=[cgi_dquote_html $str]"
	} elseif {[regexp "^lowsrc=(.*)" $a dummy str]} {
	    append buf " lowsrc=[cgi_dquote_html $str]"
	} elseif {[regexp "^usemap=(.*)" $a dummy str]} {
	    append buf " usemap=[cgi_dquote_html $str]"
	} else {
	    append buf " $a"
	}
    }
    return "$buf>"
}

# names an anchor so that it can be linked to
proc cgi_anchor_name {name} {
    puts "<a name=\"$name\">"
}

proc cgi_base {args} {
    puts "<base"
    foreach a $args {
	if {[regexp "^href=(.*)" $a dummy str]} {
	    cgi_put " href=[cgi_dquote_html $str]"
	} elseif {[regexp "^target=(.*)" $a dummy str]} {
	    cgi_put " target=\"$str\""
	} else {
	    cgi_put " $a"
	}
    }
    puts ">"
}

##################################################
# quoting support
##################################################

proc cgi_unquote_input {buf} {
    # rewrite "+" back to space
    regsub -all {\+} $buf { } buf
    # protect \ from quoting another \ and throwing off other things (first!)
    # protect $ from doing variable expansion
    # protect [ from doing evaluation
    # protect " from terminating string
    regsub -all {([\\["$])} $buf {\\\1} buf

    # replace line delimiters with newlines
    regsub -all -nocase "%0d%0a" $buf "\n" buf
    # Mosaic sends just %0A.  This is handled in the next command.

    # prepare to process all %-escapes 
    regsub -all -nocase {%([a-f0-9][a-f0-9])} $buf {[format %c 0x\1]} buf
    # process %-escapes and undo all protection
    eval return \"$buf\"
}

# return string but with html-special characters escaped,
# necessary if you want to send unknown text to an html-formatted page.
proc cgi_quote_html {s} {
    regsub -all {&}	$s {\&amp;}	s	;# must be first!
    regsub -all {"}	$s {\&quot;}	s
    regsub -all {<}	$s {\&lt;}	s
    regsub -all {>}	$s {\&gt;}	s
    return $s
}

proc cgi_dquote_html {s} {
    return \"[cgi_quote_html $s]\"
}

# return string quoted appropriately to appear in a url
proc cgi_quote_url {in} {
    regsub -all " " $in "+" in
    regsub -all "%" $in "%25" in
    return $in
}

##################################################
# block (paragraph) support
##################################################


proc cgi_division     {s} {puts <div>$s</div>}
proc cgi_blockquote   {s} {puts <blockquote>$s</blockquote>}
proc cgi_address      {s} {puts <address>$s</address>}
proc cgi_unbreakable  {s} {puts "<nobr>$s</nobr>"}

# HTML 2.0 spec recommends avoiding <xmp> but I can't figure out why
proc cgi_preformatted {s} {puts <pre>$s</pre>}
proc cgi_code	      {s} {puts <xmp>$s</xmp>}

# generate cgi_h1 and others
for {set i 1} {$i<8} {incr i} {
    proc cgi_h$i {{args}} "eval cgi_h $i \$args"
}
proc cgi_h {num args} {
    cgi_put "<h$num"
    if {[llength $args] > 1} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
	set args [lrange $args end end]
    }
    puts ">[lindex $args 0]</h$num>"
}

proc cgi_p {args} {
    cgi_put "<p"
    if {[llength $args] > 1} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
	set args [lrange $args end end]
    }
    puts ">[lindex $args 0]</p>"
}

proc cgi_br {args} {
    set ret "<br"
    if [llength $args] {
	append ret [cgi_list_to_string $args]
    }
    append ret ">"
    return $ret
}


##################################################
# list support
##################################################

proc cgi_li {args} {
    cgi_put <li
    if {[llength $args] > 1} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
    }
    puts ">[lindex $args end]"
}

proc cgi_number_list {args} {
    cgi_put "<ol"
    cgi_close_proc_push "puts </ol>"

    if {[llength $args] > 1} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
    }
    puts ">"
    uplevel [lindex $args end]

    cgi_close_proc
}

proc cgi_bullet_list {args} {
    cgi_put "<ul"
    cgi_close_proc_push "puts </ul>"

    if {[llength $args] > 1} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
    }
    puts ">"
    uplevel [lindex $args end]

    cgi_close_proc
}

# Following two are normally used from within definition lists
# but are actually paragraph types on their own.
proc cgi_term            {s} {puts <dt>$s}
proc cgi_term_definition {s} {puts <dd>$s}

proc cgi_definition_list {cmd} {
    puts "<dl>"
    cgi_close_proc_push "puts </dl>"

    uplevel $cmd
    cgi_close_proc
}

proc cgi_menu_list {cmd} {
    puts "<menu>"
    cgi_close_proc_push "puts </menu>"

    uplevel $cmd
    cgi_close_proc
}
proc cgi_directory_list {cmd} {
    puts "<dir>"
    cgi_close_proc_push "puts </dir>"

    uplevel $cmd
    cgi_close_proc
}

##################################################
# text support
##################################################

proc cgi_put	    {s} {puts -nonewline $s}

# some common special characters
proc cgi_lt	    {}  {return "&lt;"}
proc cgi_gt	    {}  {return "&gt;"}
proc cgi_amp	    {}  {return "&amp;"}
proc cgi_dollar	    {}  {return "$"}
proc cgi_us	    {}  {return "_"}
proc cgi_hash	    {}  {return "#"}
proc cgi_pct	    {}  {return "%"}
proc cgi_quote	    {}  {return "&quot;"}
proc cgi_enspace    {}  {return "&ensp;"}
proc cgi_emspace    {}  {return "&emsp;"}
proc cgi_nbspace    {}  {return "&nbsp;"} ;# nonbreaking space
proc cgi_tm	    {}  {return "&#174;"} ;# registered trademark
proc cgi_copyright  {}  {return "&#169;"}
proc cgi_isochar    {n} {return "&#$n;"}
proc cgi_breakable  {}  {return "<wbr>"}

proc cgi_bold	    {s} {return "<b>$s</b>"}
proc cgi_italic     {s} {return "<i>$s</i>"}
proc cgi_underline  {s} {return "<u>$s</u>"}
proc cgi_strikeout  {s} {return "<s>$s</s>"}
proc cgi_subscript  {s} {return "<sub>$s</sub>"}
proc cgi_superscript {s} {return "<sup>$s</sup>"}
proc cgi_typewriter {s} {return "<tt>$s</tt>"}
proc cgi_blink	    {s} {return "<blink>$s</blink>"}
proc cgi_emphasis   {s} {return "<em>$s</em>"}
proc cgi_strong	    {s} {return "<strong>$s</strong>"}
proc cgi_cite	    {s} {return "<cite>$s</cite>"}
proc cgi_sample     {s} {return "<samp>$s</samp>"}
proc cgi_keyboard   {s} {return "<kbd>$s</kbd>"}
proc cgi_variable   {s} {return "<var>$s</var>"}
proc cgi_definition {s} {return "<dfn>$s</dfn>"}
proc cgi_big	    {s} {return "<big>$s</big>"}
proc cgi_small	    {s} {return "<small>$s</small>"}

proc cgi_basefont   {size} {puts "<basefont size=$size>"}

proc cgi_font {args} {
    set buf "<font"
    foreach a [lrange $args 0 [expr [llength $args]-2]] {
	if {[regexp "^color=(.*)" $a dummy str]} {
	    append buf " color=\"$str\""
	} else {
	    append buf " $a"
	}
    }
    return "$buf>[lindex $args end]</font>"
}

#proc cgi_font       {size text} {return "<font size=$s>$text</font>"}
#proc cgi_fontcolor  {color text} {return "<font color=$color>$text</font>"}

##################################################
# html and tags that can appear in html top-level
##################################################

proc cgi_html {html} {
    cgi_html_start
    uplevel $html
    cgi_html_end
}

proc cgi_html_start {} {
    global _cgi
    
    if [info exists _cgi(html_in_progress)] return
    cgi_http_head_implicit

    set _cgi(html_in_progress) 1
    puts "<html>"
}

proc cgi_html_end {} {
    global _cgi
    unset _cgi(html_in_progress)
    puts "</html>"
}

##################################################
# head support
##################################################

proc cgi_head {{head {}}} {
    global _cgi

    if [info exists _cgi(head_done)] {
	return
    }

    # allow us to be recalled so that we can display errors
    if ![info exists _cgi(head_in_progress)] {
	cgi_http_head_implicit
	set _cgi(head_in_progress) 1
	puts "<head>"
    }

    # prevent cgi_html (during error handling) from generating html tags
    set _cgi(html_in_progress) 1
    # don't actually generate html tags since there's nothing to clean
    # them up

    if {0 == [string length $head]} {
	if {[catch {cgi_title}]} {
	    set head "cgi_title untitled"
	}
    }
    uplevel $head
    if ![info exists _cgi(head_suppress_tag)] {
	puts "</head>"
    } else {
	unset _cgi(head_suppress_tag)
    }

    set _cgi(head_done) 1

    # debugging can unset this in the uplevel above
    catch {unset _cgi(head_in_progress)}
}

# with one arg: set, print, and return title
# with no args: return title
proc cgi_title {args} {
    global _cgi

    cgi_http_head_implicit

    # we could just generate <head></head> tags, but  head-level commands
    # might follow so just suppress the head tags entirely
    if ![info exists _cgi(head_in_progress)] {
	set _cgi(head_in_progress) 1
	set _cgi(head_suppress_tag) 1
    }

    set title [lindex $args 0]

    if {[llength $args]} {
	set _cgi(title) $title
	puts "<title>$title</title>"
    }
    return $_cgi(title)
}

# This tag can only be called from with cgi_head.
# example: cgi_http_equiv Refresh 1
# There's really no reason to call this since it can be done directly
# from cgi_http_head.
proc cgi_http_equiv {type contents} {
    cgi_http_head_implicit
    puts "<meta http-equiv=\"$type\" content=[cgi_dquote_html $cont]>"
}

# Do whatever you want with meta tags.
# Example: <meta name="author" content="Don Libes">
proc cgi_meta {args} {
    cgi_put "<meta"
    foreach a $args {
	if {[regexp "^name=(.*)" $a dummy str]} {
	    cgi_put " name=[cgi_dquote_html $str]"
	} elseif {[regexp "^content=(.*)" $a dummy str]} {
	    cgi_put " name=[cgi_dquote_html $str]"
	} elseif {[regexp "^http-equiv=(.*)" $a dummy str]} {
	    cgi_put " http-equiv=[cgi_dquote_html $str]"
	} else {
	    cgi_put " $a"
	}
    }
    puts ">"
}

proc cgi_relationship {rel href args} {
    puts "<link rel=$rel $href=\"$href\""
    foreach a $args {
	if {[regexp "^title=(.*)" $a dummy str]} {
	    cgi_put " title=[cgi_dquote_html $str]"
	} else {
	    cgi_put " $a"
	}
    }
}

proc cgi_name {args} {
    global _cgi

    if [llength $args] {
	set _cgi(name) [lindex $args 0]
    }
    return $_cgi(name)
}

##################################################
# body and other top-level support
##################################################

proc cgi_body {args} {
    global errorInfo _cgi

    # allow user to "return" from the body without missing cgi_body_end
    if 1==[catch {
	eval cgi_body_start [lrange $args 0 [expr [llength $args]-2]]
	uplevel [lindex $args end]
    } errMsg] {
	set savedInfo $errorInfo
	error $errMsg $savedInfo
    }
    cgi_body_end
}

proc cgi_body_start {args} {
    global _cgi
    if [info exists _cgi(body_in_progress)] return

    cgi_head

    set _cgi(body_in_progress) 1

    cgi_put "<body"
    foreach a $args {
	if {[regexp "^background=(.*)" $a dummy str]} {
	    cgi_put " background=\"$str\""
	} elseif {[regexp "^bgcolor=(.*)" $a dummy str]} {
	    cgi_put " bgcolor=\"$str\""
	} elseif {[regexp "^text=(.*)" $a dummy str]} {
	    cgi_put " text=\"$str\""
	} elseif {[regexp "^link=(.*)" $a dummy str]} {
	    cgi_put " link=\"$str\""
	} elseif {[regexp "^vlink=(.*)" $a dummy str]} {
	    cgi_put " vlink=\"$str\""
	} elseif {[regexp "^alink=(.*)" $a dummy str]} {
	    cgi_put " alink=\"$str\""
	} else {
	    cgi_put " $a"
	}
    }
    puts ">"

    cgi_uid_check nobody
    cgi_debug {
	global env
	catch {puts "Input: <pre>$_cgi(input)</pre>"}
	catch {puts "Cookie: <pre>$env(HTTP_COOKIE)</pre>"}
    }

    if ![info exists _cgi(errorInfo)] {
	uplevel 2 app_body_start
    }
}

proc cgi_body_end {} {
    global _cgi
    if ![info exists _cgi(errorInfo)] {
	uplevel 2 app_body_end
    }
    unset _cgi(body_in_progress)
    puts "</body>"
}

proc cgi_script {args} {
    puts "<script[cgi_lrange $args 0 [expr [llength $args]-2]]>"
    cgi_close_proc_push "puts </script>"

    uplevel [lindex $args end]

    cgi_close_proc
}

proc cgi_applet {args} {
    puts "<applet[cgi_lrange $args 0 [expr [llength $args]-2]]>"
    cgi_close_proc_push "puts "</applet">

    uplevel [lindex $args end]
    cgi_close_proc
}

proc cgi_param {nameval} {
    regexp "(\[^=]*)(=?)(.*)" $nameval dummy name q value

    if {$q != "="} {
	set value ""
    }
    puts "<param name=\"$name\" value=[cgi_dquote_html $value]>"
}

# record any proc's that must be called prior to displaying an error
proc cgi_close_proc_push {p} {
    global _cgi
    append _cgi(close_proc) ";$p"
}

proc cgi_close_proc_pop {} {
    global _cgi
    regexp "(.*);(.*)" $_cgi(close_proc) dummy _cgi(close_proc) lastproc
    return $lastproc
}

# generic proc to close whatever is on the top of the stack
proc cgi_close_proc {} {
    eval [cgi_close_proc_pop]
}

proc cgi_close_procs {} {
    global _cgi

    cgi_close_tag
    if [info exists _cgi(close_proc)] {
	uplevel #0 $_cgi(close_proc)
    }
}

proc cgi_close_tag {} {
    global _cgi

    if [info exists _cgi(tag_in_progress)] {
	puts ">"
	unset _cgi(tag_in_progress)
    }
}

##################################################
# hr support
##################################################

proc cgi_hr {args} {
    global _cgi

    cgi_put "<hr"
    if [llength $args] {
	cgi_put "[cgi_list_to_string $args]"
    }
    puts ">"
}

##################################################
# form & isindex
##################################################

proc cgi_form {action args} {
    global _cgi

    cgi_form_multiple_check
    set _cgi(form_in_progress) 1

    cgi_close_proc_push cgi_form_end
    cgi_put "<form action="
    if [regexp "^http:" $action] {
	cgi_put "\"$action\""
    } else {
	cgi_put "[cgi_cgi $action]"
    }
    set method "method=post"
    foreach a [lrange $args 0 [expr [llength $args]-2]] {
	if {[regexp "^method=" $a]} {
	    set method $a
	} elseif {[regexp "^target=(.*)" $a dummy str]} {
	    cgi_put " target=\"$str\""
	} elseif {[regexp "^enctype=(.*)" $a dummy str]} {
	    cgi_put " enctype=\"$str\""
	} else {
	    cgi_put " $a"
	}
    }
    puts " $method>"
    uplevel [lindex $args end]
    cgi_form_end
}

proc cgi_form_end {} {
    global _cgi
    unset _cgi(form_in_progress)
    cgi_close_proc_pop
    puts "</form>"
}

proc cgi_form_multiple_check {} {
    global _cgi
    if [info exists _cgi(form_in_progress)] {
	error "Cannot create form (or isindex) with form already in progress."
    }
}

proc cgi_isindex {args} {
    cgi_form_multiple_check

    cgi_put "<isindex"
    foreach a $args {
	if {[regexp "^href=(.*)" $a dummy str]} {
	    cgi_put " href=\"$str\""
	} elseif {[regexp "^prompt=(.*)" $a dummy str]} {
	    cgi_put " prompt=[cgi_dquote_html $str]"
	} else {
	    cgi_put " $a"
	}
    }
    puts ">"
}

##################################################
# argument handling
##################################################

proc cgi_input {{fakeinput {}} {fakecookie {}}} {
    global env _cgi _cgi_uservar _cgi_cookie

    if {[info exists env(CONTENT_TYPE)] && [regexp ^multipart/form-data $env(CONTENT_TYPE)]} {
	if {![info exists env(REQUEST_METHOD)]} {
	    # running by hand
	    set fid [open $fakeinput]
	} else {
	    set fid stdin
	}
	cgi_input_multipart $fid
    } else {
	if {![info exists env(REQUEST_METHOD)]} {
	    set input $fakeinput
	    set env(HTTP_COOKIE) $fakecookie
	} elseif { $env(REQUEST_METHOD) == "GET" } {
	    set input $env(QUERY_STRING)
	} else {
	    set input [read stdin $env(CONTENT_LENGTH)]
	}
	# if script blows up later, enable access to the original input.
	cgi_debug -noprint {set _cgi(input) $input}

	set pairs [split $input &]
	foreach pair $pairs {
	    if {0 == [regexp (.*)=(.*) $pair dummy varname val]} {
		# if no match, unquote and leave it at that
		# this is typical of <isindex>-style queries
		set varname anonymous
		set val $pair
	    }

	    set val [cgi_unquote_input $val]
	    # handle lists of values correctly
	    if [regexp List$ $varname] {
		lappend _cgi_uservar($varname) $val
	    } else {
		set _cgi_uservar($varname) $val
	    }
	}
    }

    if ![info exists env(HTTP_COOKIE)] return
    foreach pair [split $env(HTTP_COOKIE) ";"] {
	# pairs are actually split by "; ", sigh
	set pair [string trimleft $pair " "]
	regexp (.*)=(.*) $pair dummy varname val

	set val [cgi_unquote_input $val]

	set _cgi_cookie($varname) $val
    }
}

proc cgi_input_multipart {fin} {
    global env _cgi _cgi_uservar

    cgi_debug -noprint {
	# save file for debugging purposes
	set foutname /tmp/CGIdbg.[pid]
	set fout [open $foutname w]
	set _cgi(input) $foutname
    }

    # figure out boundary
    regexp boundary=(.*) $env(CONTENT_TYPE) dummy boundary
    set boundary --$boundary

    # get first boundary line
    gets $fin buf
    if [info exists fout] {puts $fout $buf}

    set filecount 0
    while 1 {
	# process Content-Disposition:
	if {-1 == [gets $fin buf]} break
	if [info exists fout] {puts $fout $buf}
	catch {unset filename}
	foreach b $buf {
	    regexp {^name="(.*)"} $b dummy varname
	    regexp {^filename="(.*)"} $b dummy filename
	}

	# Skip remaining headers until blank line.
	# Content-Type: can appear here.  Ignore it.
	while 1 {
	    if {-1 == [gets $fin buf]} break
	    if [info exists fout] {puts $fout $buf}
	    if {0==[string compare $buf "\r"]} break
	}

	if {[info exists filename]} {
	    # read the part into a file
	    if [string length $filename] {
		set partfoutname /tmp/CGI[pid].[incr filecount]
		set partfout [open $partfoutname w]
		set _cgi_uservar($varname) "$partfoutname $filename"
	    }
	    while 1 {
		if {-1 == [gets $fin buf]} break
		if [info exists fout] {puts $fout $buf}
		if {[regexp ^[set boundary](--)?\r$ $buf]} break
		if [info exists partfout] {
		    puts $partfout $buf
		}
	    }
	    if [info exists partfout] {
		close $partfout
		unset partfout
	    }
	} else {
	    # read the part into a variable
	    set val ""
	    while 1 {
		if {-1 == [gets $fin buf]} break
		if [info exists fout] {puts $fout $buf}
		if {[regexp ^[set boundary](--)?$ $buf]} break

		append val $buf\n
	    }
	    # handle lists of values correctly
	    if [regexp List$ $varname] {
		lappend _cgi_uservar($varname) $val
	    } else {
		set _cgi_uservar($varname) $val
	    }
	}	    
    }
    if [info exists fout] {close $fout}
}

# export named variable
proc cgi_export {nameval} {
    regexp "(\[^=]*)(=?)(.*)" $nameval dummy name q value

    if {$q != "="} {
	set value [uplevel set [list $name]]
    }

    cgi_put "<input type=hidden name=\"$name\" value=[cgi_dquote_html $value]>"
}

proc cgi_export_cookie {name args} {
    upvar $name x
    eval cgi_cookie_set [list $name=$x] $args
}

# return list of variables available for import
proc cgi_import_list {} {
    global _cgi_uservar

    array names _cgi_uservar
}

# import named variable
proc cgi_import {name} {
    upvar $name var
    upvar #0 _cgi_uservar($name) val

    set var $val
}

proc cgi_import_as {name tclvar} {
    upvar $tclvar var
    upvar #0 _cgi_uservar($name) val

    set var $val
}

# like cgi_import but if not available, try cookie
proc cgi_import_cookie {name} {
    upvar $name var

    if {0==[catch {set var $_cgi_uservar($name)}]} return
    set var [cgi_cookie_get $name]
}

# like cgi_import but if not available, try cookie
proc cgi_import_cookie_as {name tclvar} {
    upvar $tclvar var

    if {0==[catch {set var $_cgi_uservar($name)}]} return
    set var [cgi_cookie_get $name]
}

##################################################
# button support
##################################################

proc cgi_submit_button {{nameval {=Submit Query}}} {
    regexp "(\[^=]*)=(.*)" $nameval dummy name value
    cgi_put "<input type=submit"
    if {0!=[string compare "" $name]} {
	cgi_put " name=\"$name\""
    }
    puts " value=[cgi_dquote_html $value]>"
}


proc cgi_reset_button {{value Reset}} {
    puts "<input type=reset value=[cgi_dquote_html $value]>"
}

proc cgi_radio_button {nameval args} {
    regexp "(\[^=]*)=(.*)" $nameval dummy name value

    puts "<input type=radio name=\"$name\" value=[cgi_dquote_html $value][cgi_list_to_string $args]>"
}

proc cgi_image_button {nameval} {
    regexp "(\[^=]*)=(.*)" $nameval dummy name value
    cgi_put "<input type=image"
    if {0!=[string compare "" $name]} {
	cgi_put " name=\"$name\""
    }
    puts " src=\"$value\">"
}

# map/area implement client-side image maps
proc cgi_map {name cmd} {
    cgi_put "<map name=\"$name\">"
    cgi_close_proc_push "puts </map>"

    uplevel [lindex $args end]
    cgi_close_proc
}

proc cgi_area {args} {
    cgi_put "<area"
    foreach a $args {
	if {[regexp "^coords=(.*)" $a dummy str]} {
	    cgi_put " coords=\"$str\""
	} elseif {[regexp "^shape=(.*)" $a dummy str]} {
	    cgi_put " shape=\"$str\""
	} elseif {[regexp "^href=(.*)" $a dummy str]} {
	    cgi_put " href=\"$str\""
	} elseif {[regexp "^target=(.*)" $a dummy str]} {
	    cgi_put " target=\"$str\""
	} else {
	    cgi_put " $a"
	}
    }
    puts ">"
}

##################################################
# checkbox support
##################################################

proc cgi_checkbox {nameval args} {
    regexp "(\[^=]*)(=?)(.*)" $nameval dummy name q value
    cgi_put "<input type=checkbox name=\"$name\""

    if {0!=[string compare "" $value]} {
	cgi_put " value=[cgi_dquote_html $value]"
    }
    puts "[cgi_list_to_string $args]>"
}

##################################################
# textentry support
##################################################

proc cgi_text {nameval args} {
    regexp "(\[^=]*)(=?)(.*)" $nameval dummy name q value

    cgi_put "<input name=\"$name\""

    if {$q != "="} {
	set value [uplevel set [list $name]]
    }
    puts " value=[cgi_dquote_html $value][cgi_list_to_string $args]>"
}

##################################################
# textarea support
##################################################

proc cgi_textarea {nameval args} {
    regexp "(\[^=]*)(=?)(.*)" $nameval dummy name q value

    puts "<textarea name=\"$name\"[cgi_list_to_string $args]>"

    if {$q != "="} {
	set value [uplevel set [list $name]]
    }
    puts [cgi_quote_html $value]

    puts "</textarea>"
}

##################################################
# file upload support
##################################################

# for this to work, pass enctype=multipart/form-data to cgi_form
proc cgi_file_button {name} {
    puts "<input type=file name=\"$name\">"
}

##################################################
# select support
##################################################

proc cgi_select {name args} {
    cgi_put "<select name=\"$name\""
    cgi_close_proc_push "puts </select>"
    foreach arg [lrange $args 0 [expr [llength $args]-2]] {
    	if 0==[string compare multiple $arg] {
	    ;# sanity check
	    if ![regexp "List$" $name] {
		puts ">" ;# prevent error from being absorbed
		error "When selecting multiple options, select variable\
			must end in \"List\" to allow the value to be\
			recognized as a list when it is processed later."
	    }
	}
	cgi_put " $arg"
    }
    puts ">"
    uplevel [lindex $args end]
    cgi_close_proc
}

proc cgi_option {o args} {
    cgi_put "<option"
    foreach arg $args {
	if [regexp "^selected_if_equal=(.*)" $arg dummy default] {
	    if 0==[string compare $default $o] {
		cgi_put " selected"
	    }
	} else {
	    cgi_put " $arg"
	}
    }
    puts ">$o"
}

##################################################
# plug-in support
##################################################

proc cgi_embed {src wh args} {
    regexp (.*)x(.*) $wh dummy width height
    cgi_put "<embed src=[cgi_dquote_html $src] width=\"$width\" height=\"$height\""
    foreach a $args {
	if {[regexp "^palette=(.*)" $a dummy str]} {
	    cgi_put " palette=\"$str\""
	} elseif {[regexp -- "-quote" $a]} {
	    set quote 1
	} else {
	    if [info exists quote] {
		regexp "(\[^=]*)=(.*)" $a dummy var val
		cgi_put " var=[cgi_dquote_html $var]"
	    } else {
		cgi_put " $a"
	    }
	}
    }
    puts ">"
}

##################################################
# mail support
##################################################

# mail to/from the service itself
proc cgi_mail_addr {args} {
    global _cgi

    if [llength $args] {
	set _cgi(email) [lindex $args 0]
    }
    return $_cgi(email)
}

proc cgi_mail_start {to} {
    global _cgi

    set _cgi(mailfile) /tmp/cgimail.[pid]
    set _cgi(mailfid) [open $_cgi(mailfile) w]

    # mail is actually sent by "nobody".  To force bounce messages
    # back to us, override the default return-path.
    cgi_mail_add "Return-Path: <$_cgi(email)>"
    cgi_mail_add "From: [cgi_name] <$_cgi(email)>"
    cgi_mail_add "To: $to"
}

# add another line to outgoing mail
# if no arg, add a blank line
proc cgi_mail_add {{arg {}}} {
    global _cgi

    puts $_cgi(mailfid) $arg
}	

# end the outgoing mail and send it
proc cgi_mail_end {} {
    global _cgi

    close $_cgi(mailfid)

    exec /usr/lib/sendmail -t -odb < $_cgi(mailfile)
    # Explanation:
    # -t   means: pick up recipient from body
    # -odb means: deliver in background
    # note: bogus local address cause sendmail to fail immediately

    exec /bin/rm -f $_cgi(mailfile)
}

##################################################
# cookie support
##################################################

# calls to cookie_set look like this:
#   cgi_cookie_set user=don domain=nist.gov expires=never
#   cgi_cookie_set user=don domain=nist.gov expires=now
#   cgi_cookie_set user=don domain=nist.gov expires=...actual date...

proc cgi_cookie_set {nameval args} {
    global _cgi

    if ![info exists _cgi(http_head_in_progress)] {
	error "Cookies must be set from within cgi_http_head."
    }
    puts -nonewline "Set-Cookie: [cgi_cookie_encode $nameval];"

    foreach arg $args {
	if [regexp "^expires=(.*)" $arg dummy expiration] {
	    if {0==[string compare $expiration "never"]} {
		set expiration "Friday, 31-Dec-99 23:59:59 GMT"
	    } elseif {0==[string compare $expiration "now"]} {
		set expiration "Friday, 31-Dec-90 23:59:59 GMT"
	    }
	    puts -nonewline " expires=[cgi_cookie_encode $expiration];"
	} elseif [regexp "^domain=(.*)" $arg dummy domain] {
	    puts -nonewline " domain=[cgi_cookie_encode $domain];"
	} elseif [regexp "^path=(.*)" $arg dummy path] {
	    puts -nonewline " path=[cgi_cookie_encode $path];"
	} elseif [regexp "^secure$" $arg] {
	    puts -nonewline " secure;"
	}
    }
    puts ""
}

# return list of cookies available for import
proc cgi_cookie_list {} {
    global _cgi_cookie

    array names _cgi_cookie
}

proc cgi_cookie_get {name} {
    global _cgi_cookie
    return $_cgi_cookie($name)
}

proc cgi_cookie_encode {in} {
    regsub -all " " $in "+" in
    regsub -all ";" $in "%3B" in
    regsub -all "," $in "%2C" in
    regsub -all "%" $in "%25" in   ;# must be last!
    return $in
}

##################################################
# table support
##################################################

proc cgi_table {args} {
    cgi_put "<table"
    cgi_close_proc_push "puts </table>"

    if {[llength $args]} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
    }
    puts ">"
    uplevel [lindex $args end]
    cgi_close_proc
}

proc cgi_caption {args} {
    cgi_put "<caption"
    if {[llength $args]} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
    }
    puts ">[lindex $args end]</caption>"
}

proc cgi_table_row args {
    cgi_put "<tr"
    cgi_close_proc_push "puts </tr>"
    if {[llength $args]} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
    }
    puts ">"
    uplevel [lindex $args end]
    cgi_close_proc
}

proc cgi_table_head args {
    cgi_put "<th"
    cgi_close_proc_push "puts </th>"

    if {[llength $args]} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
    }
    puts ">"
    uplevel [lindex $args end]
    cgi_close_proc
}

proc cgi_table_data args {
    cgi_put "<td"
    cgi_close_proc_push "puts </td>"

    if {[llength $args]} {
	cgi_put "[cgi_lrange $args 0 [expr [llength $args]-2]]"
    }
    puts ">"
    uplevel [lindex $args end]
    cgi_close_proc
}

##################################################
# frames
##################################################

proc cgi_frameset {args} {
    cgi_head ;# force it out, just in case none

    cgi_put "<frameset"
    cgi_close_proc_push "puts </frameset>"

    foreach a [lrange $args 0 [expr [llength $args]-2]] {
	if {[regexp "^rows=(.*)" $a dummy str]} {
	    cgi_put " rows=\"$str\""
	} elseif {[regexp "^cols=(.*)" $a dummy str]} {
	    cgi_put " cols=\"$str\""
	} else {
	    cgi_put " $a"
	}
    }
    puts ">"
    uplevel [lindex $args end]

    cgi_close_proc
}

proc cgi_frame {namesrc args} {
    cgi_put "<frame"

    regexp "(\[^=]*)(=?)(.*)" $namesrc dummy name q src

    if {$name != ""} {
	cgi_put " name=\"$name\""
    }

    if {$src != ""} {
	cgi_put " src=\"$src\""
    }

    foreach a $args {
	if {[regexp "^marginwidth=(.*)" $a dummy str]} {
	    cgi_put " marginwidth=\"$str\""
	} elseif {[regexp "^marginheight=(.*)" $a dummy str]} {
	    cgi_put " marginheight=\"$str\""
	} elseif {[regexp "^scrolling=(.*)" $a dummy str]} {
	    cgi_put " scrolling=\"$str\""
	} else {
	    cgi_put " $a"
	}
    }
    puts ">"
}

proc cgi_noframes {args} {
    puts "<noframes>"
    cgi_close_proc_push "puts </noframes>"
    uplevel [lindex $args end]
    cgi_close_proc
}

##################################################
# admin support
##################################################

# mail address of the administrator
proc cgi_admin_mail_addr {args} {
    global _cgi

    if [llength $args] {
	set _cgi(admin_email) [lindex $args 0]
    }
    return $_cgi(admin_email)
}

##################################################
# todo - the following are still unfinished
##################################################
# N2.0 <area shape="rect" coords=",,," href="url"|nohref>

##################################################
# if possible, make each cmd available without cgi_ prefix
##################################################
foreach p [info procs cgi_*] {
    regexp "cgi_(.*)" $p dummy name
    if [llength [info commands $name]] continue
    proc $name {args} "upvar _cgi_local x; set x \$args; uplevel \"$p \$x\""
}

##################################################
# --Deprecated
##################################################

# A recent Netscape invention and officially in 3.2 documentation, but
# everyone agrees that 3.0 alignment attributes are a better solution.
proc cgi_center	{cmd}	{
    puts <center>
    cgi_close_proc_push "puts </center>"
    uplevel $cmd
    cgi_close_proc
}

##################################################
# internal utilities
##################################################

# undo Tcl's quoting due to list protection
# This leaves a space at the beginning if the string is non-null
# but this is always desirable in the HTML context in which it is called
# and the resulting HTML looks more readable.
# (Alas, it makes the Tcl callers a little less readable - however, there
#  aren't more than a handful and they're all right here, so we'll live
#  with it.)
proc cgi_list_to_string {list} {
    set string ""
    foreach l $list {
	append string " $l"
    }
    # remove first space if possible
    # regexp "^ ?(.*)" $string dummy string
    return $string
}

# do lrange but return as string
# needed for stuff like: puts "[cgi_lrange $args ...
# Like cgi_list_to_string, also returns string with initial blank if non-null
proc cgi_lrange {list i1 i2} {
    cgi_list_to_string [lrange $list $i1 $i2]
}

##################################################
# user-defined procedures
##################################################

# User-defined procedure called immediately after <body>
# Good mechanism for controlling things such as if all of your pages
# start with the same graphic or other boilerplate.
proc app_body_start {} {}

# User-defined procedure called just before </body>
# Good place to generate signature lines, last-updated-by, etc.
proc app_body_end {} {}

##################################################
# do some initialization
##################################################

cgi_debug -off
cgi_name ""
cgi_root ""

# email addr of person responsible for this service
cgi_admin_mail_addr "root"	;# you should override this!

# most services won't have an actual email addr
cgi_mail_addr "CGI script - do not reply"
