/*
 * Adabas.xs --
 *
 *      This file contains a perl interface to the Adabas D database server.
 *
 * Copyright (c) 1997 Christian Krone. All rights reserved.
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Tcl itself.
 * See also licence.terms
 *
 * $Id: Adabas.xs,v 1.6 1997/06/28 17:14:51 adabas Exp $
 */

#ifdef __cplusplus
extern "C" {
#endif
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "tcl.h"
#include "../../extension/adabas.h"
#include "../../extension/AdabasPort.h"
#include "../../extension/AdabasApi.h"
#ifdef __cplusplus
}
#endif

#ifdef HAS_TCL_OBJECTS
#define IncrRefCount  Tcl_IncrRefCount
#define NewStringObj  Tcl_NewStringObj
#define ObjGetVar2    Tcl_ObjGetVar2

#define ResString Tcl_Obj *
#define ResStringInit(obj)  ((obj) = Tcl_NewStringObj ("", 0), Tcl_IncrRefCount (obj))
#define ResStringFree(obj)  (Tcl_SetStringObj (obj, "", 0))
#define ResStringValue(obj) (Tcl_GetStringFromObj ((obj), (int *) NULL))
#define ResStringGetResult(interp,obj) \
     (Tcl_SetStringObj ((obj), \
	   Tcl_GetStringFromObj (Tcl_GetObjResult (interp), (int *) NULL), -1))
#define ResStringSet(obj,str,len) (Tcl_SetStringObj ((obj), (str), (len)))

#define GetIntFromResult(errInterp,resInterp,longPtr) \
     (Tcl_GetIntFromObj ((errInterp), Tcl_GetObjResult (resInterp), (longPtr)))

#define GetListFromResult(interp,objc,objv) \
    (Tcl_ListObjGetElements (interp, Tcl_GetObjResult (interp), (objc), (objv)))
#define FreeListObjv(objv)
#else
#define IncrRefCount(obj)
#define ObjGetVar2 Tcl_GetVar2

#define NewStringObj(str,len) (str)

#define ResString Tcl_DString

#define ResStringInit(obj)  (Tcl_DStringInit(&(obj)))
#define ResStringFree(obj)  (Tcl_DStringFree(&(obj)))
#define ResStringValue(obj) (Tcl_DStringValue(&(obj)))
#define ResStringGetResult(interp,obj) (Tcl_DStringGetResult ((interp),&(obj)))
#define ResStringSet(obj,str,len) (Tcl_DStringAppend (&(obj), (str), (len)))

#define GetIntFromResult(errInterp,resInterp,intPtr) \
     (Tcl_GetInt ((errInterp), (resInterp)->result, (intPtr)))

#define GetListFromResult(interp,objc,objv) \
    (Tcl_SplitList (interp, (interp)->result, (objc), (objv)))
#define FreeListObjv(objv) ckfree(objv)
#endif

/*
 * This procedure isn't mentioned in AdabasApi.h
 */

extern int Adabastcl_Init _ANSI_ARGS_((Tcl_Interp *interp));

/*
 * And here are the global variables of the extension.
 */

Tcl_Interp *AdabasTclInterp;
ResString   AdabasErrorMsg;
int         AdabasRC       = 0;
int         AdabasErrorPos = 0;

Tcl_Obj *Adamsg;
Tcl_Obj *Errortxt;
Tcl_Obj *Errorpos;

MODULE = Adabas		PACKAGE = Adabas

PROTOTYPES: ENABLE

BOOT:
{
  /*
   * First create the Tcl interpreter, where all the Adabas stuff will
   * be evaluated.
   */
  AdabasTclInterp = Tcl_CreateInterp();

  /*
   * The following lines seem funny, since I would expect, that the Tcl_*
   * procedures are already present, as I just invoked Tcl_CreateInterp.
   * But these procedures may be loaded statically (and already resolved)
   * into Adabas.so and thus not applicable. And if they are already loaded,
   * the following command does almost nothing.
   * Don't ask the returncode here, since it may be perfectly alright to fail,
   * if tcl is loaded statically and there is no shared library available.
   */
  Tcl_VarEval (AdabasTclInterp, "load ", TCLLIB, "[info sharedlibextension]",
	       (char *) NULL);

  /*
   * adasql.tcl asks about it, so we better define it here.
   * We set it to false, since all the sqlsh commands are not used here.
   */
  Tcl_SetVar (AdabasTclInterp, "tcl_interactive", "0", TCL_GLOBAL_ONLY);

  if (Adabastcl_Init (AdabasTclInterp) != TCL_OK)
    fprintf (stderr, "%s\n", AdabasTclInterp->result);

  /*
   * Initialize the global variables.
   */
  ResStringInit (AdabasErrorMsg);

  Adamsg   = NewStringObj ("adamsg",   -1); IncrRefCount (Adamsg);
  Errortxt = NewStringObj ("errortxt", -1); IncrRefCount (Errortxt);
  Errorpos = NewStringObj ("errorpos", -1); IncrRefCount (Errorpos);
}

char *
version ()
CODE:
{
  ResStringFree (AdabasErrorMsg);
  Tcl_Eval (AdabasTclInterp, "adabas version");
  RETVAL = AdabasTclInterp->result;
}
OUTPUT:
RETVAL

char *
errortxt ()
CODE:
{
  RETVAL = ResStringValue (AdabasErrorMsg);
}
OUTPUT:
RETVAL

int
errorpos ()
CODE:
{
  RETVAL = AdabasErrorPos;
}
OUTPUT:
RETVAL

int
rc ()
CODE:
{
  RETVAL = AdabasRC;
}
OUTPUT:
RETVAL

AdabasInfo *
logon (connectString, serverdb="", sqlMode="adabas")
	char *connectString
	char *serverdb;
	char *sqlMode
CODE:
{
  int         isolation   = -1;
  service     serv        = SQL_USER;
  int         controlAppl = 0;
  char       *locale      = NULL;
  AdabasInfo *logonHandle;

  ResStringFree (AdabasErrorMsg);

  logonHandle = AdabasLogon (AdabasTclInterp, serverdb, connectString,
			     isolation, sqlMode, serv, locale, controlAppl);

  if (!logonHandle)
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);

  RETVAL = logonHandle;
}
OUTPUT:
RETVAL

void
logoff (logonHandle)
	AdabasInfo *logonHandle
CODE:
{
  ResStringFree (AdabasErrorMsg);
  AdabasLogoff (AdabasTclInterp, logonHandle);
}

CursorInfo *
open (logonHandle)
	AdabasInfo *logonHandle
CODE:
{
  CursorInfo *cursorHandle;

  ResStringFree (AdabasErrorMsg);
  cursorHandle = AdabasOpen (AdabasTclInterp, logonHandle);

  if (!cursorHandle)
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);

  RETVAL = cursorHandle;
}
OUTPUT:
RETVAL

void
close (cursorHandle)
	CursorInfo *cursorHandle
CODE:
{
  ResStringFree (AdabasErrorMsg);
  AdabasClose (AdabasTclInterp, cursorHandle);
}

int
sql (cursorHandle, sqlStatement, sqlMode="", resultTable="")
	CursorInfo *cursorHandle
	char       *sqlStatement
	char       *sqlMode
	char       *resultTable
CODE:
{
  int      messType       = M_DBS;
  int      withParameters = 0;
  Tcl_Obj *errorPosObj;
  int      length;

  if (AdabasSql (AdabasTclInterp,  cursorHandle, messType, withParameters,
		 sqlStatement, resultTable, sqlMode) == TCL_OK) {
    ResStringFree (AdabasErrorMsg);
    AdabasRC       = 0;
    AdabasErrorPos = 0;
  } else {
    if (GetIntFromResult ((Tcl_Interp *) NULL, AdabasTclInterp, &AdabasRC)
	!= TCL_OK) {
      ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);
      AdabasRC = -1;
    } else {
      Tcl_Obj *obj = ObjGetVar2 (AdabasTclInterp,
			Adamsg, Errortxt,
			TCL_GLOBAL_ONLY);
      char *str = Tcl_GetStringFromObj (obj, &length);
      ResStringSet (AdabasErrorMsg, str, length);
      /*ResStringSet (AdabasErrorMsg,
		    Tcl_GetStringFromObj (ObjGetVar2 (AdabasTclInterp,
						      Adamsg, Errortxt,
						      TCL_GLOBAL_ONLY), &length),
		    length);*/
      if ((errorPosObj = ObjGetVar2 (AdabasTclInterp, Adamsg, Errorpos,
				     TCL_GLOBAL_ONLY)) == NULL
	  || Tcl_GetIntFromObj (AdabasTclInterp, errorPosObj,
				&AdabasErrorPos) != TCL_OK) {
	AdabasErrorPos = 1;
      }
    }
  }
  RETVAL = AdabasRC;
}
OUTPUT:
RETVAL

void
fetch (cursorHandle, position="", sqlMode="")
	CursorInfo *cursorHandle
	char       *position
	char       *sqlMode
PPCODE:
{
  int       objc;
  Tcl_Obj **objv;
  Tcl_Obj **currObj;
  char     *currResult;
  int       count    = 1;
  int       arraySet = 0;
  int       length;

  if (AdabasFetch (AdabasTclInterp, cursorHandle, position,
		   count, sqlMode, arraySet) != TCL_OK) {
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);
  } else if (GetListFromResult (AdabasTclInterp, &objc, &objv) == TCL_ERROR) {
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);
    EXTEND (sp, 1);
    PUSHs (sv_newmortal ());
  } else {
    ResStringFree (AdabasErrorMsg);
    EXTEND (sp, objc);
    currObj = objv;
    while (objc--) {
      currResult = Tcl_GetStringFromObj (*currObj, &length);
      PUSHs (sv_2mortal (newSVpv (currResult, length)));
      currObj++;
    }
    FreeListObjv (objv);
  }
}

void
fetchHash (cursorHandle, position="", sqlMode="")
	CursorInfo *cursorHandle
	char       *position
	char       *sqlMode
PPCODE:
{
  int       objc;
  Tcl_Obj **objv;
  Tcl_Obj **currObj;
  char     *currResult;
  int       count    = 1;
  int       arraySet = 1;
  int       length;

  if (AdabasFetch (AdabasTclInterp, cursorHandle, position,
		   count, sqlMode, arraySet) != TCL_OK) {
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);
  } else if (GetListFromResult (AdabasTclInterp, &objc, &objv) == TCL_ERROR) {
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);
  } else {
    ResStringFree (AdabasErrorMsg);
    
    EXTEND (sp, objc);
    currObj = objv;
    while (objc--) {
      currResult = Tcl_GetStringFromObj (*currObj, &length);
      PUSHs (sv_2mortal (newSVpv (currResult, length)));
      currObj++;
    }
    FreeListObjv (objv);
  }
}

int
commit (logonHandle)
	AdabasInfo *logonHandle
CODE:
{
  ResStringFree (AdabasErrorMsg);
  if (AdabasCommit (AdabasTclInterp, logonHandle) != TCL_OK) {
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);
    RETVAL = 1;
  } else
    RETVAL = 0;
}
OUTPUT:
RETVAL

int
rollback (logonHandle)
	AdabasInfo *logonHandle
CODE:
{
  ResStringFree (AdabasErrorMsg);
  if (AdabasRollback (AdabasTclInterp, logonHandle) != TCL_OK) {
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);
    RETVAL = 1;
  } else
    RETVAL = 0;
}
OUTPUT:
RETVAL

int
autocommit (logonHandle, on)
	AdabasInfo *logonHandle
	int         on
CODE:
{
  ResStringFree (AdabasErrorMsg);
  if (AdabasAutoCommit (AdabasTclInterp, logonHandle, on) != TCL_OK) {
    ResStringGetResult (AdabasTclInterp, AdabasErrorMsg);
    RETVAL = 1;
  } else
    RETVAL = 0;
}
OUTPUT:
RETVAL
