#!./exwish -f
#
# Julia set calculator
#
# Demonstrates Tcl threads use in a Tk application.  Variables relevant
# to the computation are shared so they are available to all threads.
# Multiple threads can be calculating, so a mutex guards the progress of
# the computation.  Threads are tracked so they can be added and removed
# at will.
#
# Copyright (c) 1995 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.

wm title . "MT Julia Set"

set visual default
if {[winfo depth .] != 24}  {
    set visuals [winfo visualsavailable .]
    set truecolor [lsearch $visuals "truecolor 24"]
    if {$truecolor != -1}  {
	set visual [lindex $visuals $truecolor]
    }
}

share p q xmin xmax ymin ymax maxiter
share np nq npix npiy deltax deltay
share julia_lock done

set p -0.74356
set q 0.11135
set xmin -2.0
set xmax 2.0
set ymin -2.0
set ymax 2.0
set maxiter 256

set np 0
set nq 0
set npix 100
set npiy 100

set deltax [expr ($xmax-$xmin)/($npix-1)]
set deltay [expr ($ymax-$ymin)/($npiy-1)]

set thread_count 0
set threads {}
set julia_lock [mutex create]
set done 0

set p2 $p
set q2 $q

frame .p
label .p.label -text p
entry .p.entry -textvariable p
frame .q
label .q.label -text q
entry .q.entry -textvariable q

frame .threads
label .threads.label -text Threads
entry .threads.entry -state disabled -width 3 -textvariable thread_count
button .threads.add -text Add -command add_thread
button .threads.remove -state disabled -text Remove -command remove_thread

frame .julia -visual $visual
canvas .julia.c -width 150 -height 150

pack .p.label .p.entry -side left
pack .q.label .q.entry -side left
pack .threads.label .threads.entry .threads.add .threads.remove -side left
pack .julia.c
pack .p .q .threads .julia

image create photo julia-pic -height $npiy -width $npix
julia-pic blank
.julia.c create image 75 75 -image julia-pic

set tclprecision 17

if {[info command julia] != "julia"}  {
proc julia {x y p q maxiter} {
    set k 0
    set r 0
    while {$r<=$maxiter && $k<=$maxiter}  {
	set xkp1 [expr ($x+$y)*($x-$y)+$p]
	set ya [expr $x*$y]
	set ykp1 [expr $ya+$ya+$q]
	set r [expr $xkp1*$xkp1+$ykp1*$ykp1]
	incr k
	if {$r>$maxiter} {return $k}
	if {$k>=$maxiter} {return max}
	set x $xkp1
	set y $ykp1
    }
}
}

proc julia_next {x_var y_var y_pix}  {
    global np nq xmin ymin deltax deltay npix npiy julia_lock done
    upvar $x_var x $y_var y $y_pix yp

    mutex lock $julia_lock
    if {$done} {mutex unlock $julia_lock ; return done}
    set yp $nq
    set x $xmin
    set y [expr $ymin + $nq * $deltay]
    incr nq
    if {$nq > $npiy} {set nq 0 ; set done 1}
    mutex unlock $julia_lock
    return ""
}

proc julia_calc {}  {
    share -global p q np nq xmin xmax ymin ymax maxiter
    share -global deltax deltay npix npiy julia_lock done

    for {set r 0} {$r<16} {incr r}  {
	set color($r) [format "#%x%x%x" $r 0 [expr 15-$r]]
    }
    while {[catch {thread nextmsg 0}]}  {
	if {[julia_next x y ypix] == "done"} {break}
	set data {}
	for {set xpix 0} {$xpix<$npix} {incr xpix} {
	    set k [julia $x $y $p $q $maxiter]
	    if {$k=="max"}  {
		lappend data #000
	    } else {
		lappend data $color([expr $k%16])
	    }
	    set x [expr $x+$deltax]
	}
	thread post thread1 "julia-pic put [list [list $data]] -to 0 $ypix [expr $npix-1] [expr $ypix+1]"
    }
    thread post thread1 "thread_return [thread self]"
}

proc add_thread {} {
    global threads thread_count
    if {$thread_count == 0} {julia-pic blank}
    lappend threads [thread create -bound julia_calc]
    incr thread_count
    .threads.remove configure -state normal
}

proc remove_thread {} {
    global threads
    thread post [lindex $threads 0] return
}

proc thread_return {tid}  {
    global threads thread_count done
    thread join $tid
    set index [lsearch -exact $threads $tid]
    if {$index != -1} {
	set threads [lreplace $threads $index $index]
	incr thread_count -1
	if {$thread_count == 0}  {
            .threads.remove configure -state disabled
	}
	if {$done && $thread_count == 0} {
	    set done 0
	}
    }
}

