/* 
 * socket.c --
 *
 *      Tcl commands for manipulating network sockets and handling
 *	binary data.
 *
 *  Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <thread.h>

#include "tcl.h"

Tcl_CmdProc SocketCmd;
Tcl_CmdProc BlobCmd;

typedef struct {
    void *data;
    int size;
} Blob;
static Tcl_HashTable blob_hash;
static mutex_t blob_lock;
static int blob_once= 1;

int
Socket_tcl_Init( Tcl_Interp *interp)
{
    if ( blob_once)  {
	Tcl_InitHashTable( &blob_hash, TCL_STRING_KEYS);
	blob_once= 0;
    }
    Tcl_CreateCommand( interp, "socket", SocketCmd, 0, 0);
    Tcl_CreateCommand( interp, "blob", BlobCmd, 0, 0);
    return TCL_OK;
}


int
SocketServer(
     ClientData clientData,
     Tcl_Interp *interp,
     int argc,
     char **argv
)
{
    struct hostent *hostent;
    struct sockaddr_in server_addr;
    int sock, port, rc;
    int one= 1;
    FILE *file;

    if ( argc != 3)  {
	interp->result= "wrong # args; socket server port";
	return TCL_ERROR;
    }

    if ( Tcl_GetInt( interp, argv[2], &port) == TCL_ERROR)
	return TCL_ERROR;

    sock= socket( AF_INET, SOCK_STREAM, 0);
    if ( sock < 0)  {
	interp->result= strerror( errno);
	return TCL_ERROR;
    }

    (void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
		(char *)&one, sizeof(int));

    memset( &server_addr, 0, sizeof ( server_addr));
    server_addr.sin_family= AF_INET;
    server_addr.sin_port= htons(port);
    server_addr.sin_addr.s_addr= htonl( INADDR_ANY);

    rc= bind( sock, (struct sockaddr *) &server_addr, sizeof( server_addr));
    if ( rc < 0)  {
	interp->result= strerror( errno);
	close( sock);
	return TCL_ERROR;
    }

    listen( sock, 5);

    file= fdopen( sock, "r+");
    setvbuf( file, NULL, _IOLBF, BUFSIZ);
    Tcl_EnterFile( interp, file, TCL_FILE_READABLE|TCL_FILE_WRITABLE);
    return TCL_OK;
}

int
SocketAccept(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    struct sockaddr_in client_addr;
    int addr_len= sizeof ( client_addr);
    int client_sock, server_sock;
    FILE *file;

    if ( argc != 3)  {
	interp->result= "wrong # args; socket accept sockId";
	return TCL_ERROR;
    }

    if (Tcl_GetOpenFile (interp, argv[2], 0, 0, &file) != TCL_OK)
        return TCL_ERROR;

    server_sock= fileno( file);
    client_sock= accept( server_sock, (struct sockaddr *) &client_addr,
								&addr_len);
    if ( client_sock < 0)  {
	interp->result= strerror( errno);
	return TCL_ERROR;
    }
    
    file= fdopen( client_sock, "r+");
/*    setvbuf( file, NULL, _IOLBF, BUFSIZ); */
    Tcl_EnterFile( interp, file, TCL_FILE_READABLE|TCL_FILE_WRITABLE);
    return TCL_OK;
}

int
SocketConnect(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    struct sockaddr_in server_addr;
    struct hostent *hostent;
    int addr_len= sizeof ( server_addr);
    int client_sock, rc, port;
    FILE *file;

    if ( argc != 4)  {
	interp->result= "wrong # args; socket connect host port";
	return TCL_ERROR;
    }

    if ( Tcl_GetInt( interp, argv[3], &port) == TCL_ERROR)
	return TCL_ERROR;
 
    hostent= gethostbyname( argv[2]);
    if ( hostent == NULL)  {
	interp->result= "host not found";
	return TCL_ERROR;
    }

    memset( &server_addr, 0, sizeof ( server_addr));
    server_addr.sin_family= AF_INET;
    server_addr.sin_addr.s_addr= *((unsigned long *)hostent->h_addr_list[0]);
    server_addr.sin_port= port;

    client_sock= socket( AF_INET, SOCK_STREAM, 0);
    if (client_sock < 0)  {
	interp->result= strerror( errno);
	return TCL_ERROR;
    }
 
    rc= connect( client_sock, (struct sockaddr *) &server_addr,
					sizeof( server_addr));
    if (rc < 0)  {
	interp->result= strerror( errno);
	close( client_sock);
	return TCL_ERROR;
    }

    file= fdopen( client_sock, "r+");
/*    setvbuf( file, NULL, _IOLBF, BUFSIZ); */
    Tcl_EnterFile( interp, file, TCL_FILE_READABLE|TCL_FILE_WRITABLE);
    return TCL_OK;
}

int
SocketShutdown(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    FILE *file;

    if ( argc != 3)  {
	interp->result= "wrong # args; socket shutdown sockId";
	return TCL_ERROR;
    }

    if (Tcl_GetOpenFile (interp, argv[2], 0, 0, &file) != TCL_OK)
        return TCL_ERROR;

    if ( shutdown( fileno( file), 2) == -1)  {
	interp->result= strerror( errno);
	return TCL_ERROR;
    }
    return TCL_OK;
}

int
SocketCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    if ( argc < 2)  {
	interp->result= "wrong # args; socket option ?arg ...?";
	return TCL_ERROR;
    }
    if ( strcmp( "server", argv[1]) == 0)
	return SocketServer( clientData, interp, argc, argv);
    else if ( strcmp( "accept", argv[1]) == 0)
	return SocketAccept( clientData, interp, argc, argv);
    else if ( strcmp( "connect", argv[1]) == 0)
	return SocketConnect( clientData, interp, argc, argv);
    else if ( strcmp( "shutdown", argv[1]) == 0)
	return SocketShutdown( clientData, interp, argc, argv);

    interp->result= "bad option; should be accept, connect, server, or shutdown";
    return TCL_ERROR;
}

void
print_blobs( char *msg)
{
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    printf( "BLOBS --- %s\n", msg);
    entry= Tcl_FirstHashEntry( &blob_hash, &search);
    while ( entry != NULL)  {
	printf( "key %s\n", Tcl_GetHashKey( &blob_hash, entry));
	entry= Tcl_NextHashEntry( &search);
    }
}

int
BlobRead(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    FILE *file;
    int fd, created;
    off_t start, end;
    Blob *blob;
    Tcl_HashEntry *entry;

    if ( argc != 4)  {
	interp->result= "wrong # args; blob read fileId name";
	return TCL_ERROR;
    }

    if ( Tcl_GetOpenFile( interp, argv[2], 0, 1, &file) == TCL_ERROR)
	return TCL_ERROR;

    mutex_lock( &blob_lock);
    entry= Tcl_CreateHashEntry( &blob_hash, argv[3], &created);
    if ( ! created)
	free( Tcl_GetHashValue( entry));
    mutex_unlock( &blob_lock);

    fflush(file);
    fd= fileno(file);
    end= lseek( fd, 0, SEEK_END);
    lseek( fd, 0, SEEK_SET);

    blob= (Blob *) malloc( sizeof(Blob));
    blob->data= (void *) malloc( end);
    blob->size= end;
    sprintf( interp->result, "%d", read( fd, blob->data, blob->size));

    mutex_lock( &blob_lock);
    Tcl_SetHashValue( entry, (ClientData) blob);
    mutex_unlock( &blob_lock);

    return TCL_OK;
}

int
BlobWrite(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    FILE *file;
    int fd;
    Tcl_HashEntry *entry;
    Blob *blob;

    if ( argc != 4)  {
	interp->result= "wrong # args; blob write fileId name";
	return TCL_ERROR;
    }

    if ( Tcl_GetOpenFile( interp, argv[2], 1, 1, &file) == TCL_ERROR)
	return TCL_ERROR;

    mutex_lock( &blob_lock);
    entry= Tcl_FindHashEntry( &blob_hash, argv[3]);
    if ( entry == NULL)  {
	mutex_unlock( &blob_lock);
	sprintf( interp->result, "no blob named %s", argv[3]);
	return TCL_ERROR;
    }
    blob= (Blob *) Tcl_GetHashValue( entry);
    mutex_unlock( &blob_lock);

    fflush(file);
    fd= fileno(file);
    write( fd, blob->data, blob->size);

    mutex_lock( &blob_lock);
    free( blob->data);
    free( blob);
    Tcl_DeleteHashEntry( entry);
    mutex_unlock( &blob_lock);
       
    return TCL_OK;
}

int
BlobCmd(
    ClientData clientData,
    Tcl_Interp *interp,
    int argc,
    char **argv
)
{
    if ( argc < 2)  {
	interp->result= "wrong # args; blob option ?arg ...?";
	return TCL_ERROR;
    }
    if ( strcmp( "read", argv[1]) == 0)
	return BlobRead( clientData, interp, argc, argv);
    else if ( strcmp( "write", argv[1]) == 0)
	return BlobWrite( clientData, interp, argc, argv);

    interp->result= "bad option; should be read or write";
    return TCL_ERROR;
}


