/* 
 *  proc.c --
 *
 *	Provides Tcl procedure shareing for Tcl threads.
 *
 * Copyright (c) 1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */
 
#include <string.h>

#include "tcl_thread.h"
#include "tcl_threadP.h"


#ifndef DEBUG
#define DEBUG 0
#endif

int InitSharedProcTable( Tcl_Thread *first_thread);
void InitProcSharing( Tcl_Thread *tcl_thread);

typedef struct {
    char *proc;
    char *proc_name;
    int ref;
} ShareProc;

typedef struct {
    Tcl_CmdInfo info;
    Tcl_Thread *tcl_thread;
    ShareProc *share_proc;
} ShareProcInfo;

static Tcl_HashTable share_procs;
static mutex_t proc_lock;

static Tcl_CmdProc MTtcl_ProcCmd;
static Tcl_CmdProc MTtcl_UnknownCmd;
static Tcl_CmdProc MTtcl_RenameCmd;
static Tcl_CmdProc MTtcl_InfoCmd;
static Tcl_CmdProc MTtcl_InterpProc;
static Tcl_CmdDeleteProc MTtcl_ProcDeleteProc;

static char * GetProc( Tcl_Interp *interp, char *proc_name);
static void ReferenceProc( Tcl_Thread *tcl_thread, char *proc_name);
static int EnterProc( Tcl_Thread *tcl_thread, char *proc_name,
	ShareProc **share_proc, Tcl_CmdInfo *opt_info);

static char *default_tcl_unknown= NULL;


/*
 * InitSharedProcTable
 *
 * Initialize MTtcl proc sharing for a new process.  All the existing
 * procedures in the interpreter are copied and entered into the shared
 * proc table.  The unknown command, if it exists, is saved to be used
 * by all Tcl threads.
 */
int
InitSharedProcTable( Tcl_Thread *first_thread)
{
    Tcl_Interp *interp= first_thread->interp;
    int nprocs, p, created;
    char **procs, *proc_list;

    Tcl_InitHashTable( &share_procs, TCL_STRING_KEYS);

    Tcl_GlobalEval(interp, "info procs"); 
    proc_list= strdup( interp->result);
    Tcl_SplitList( interp, proc_list, &nprocs, &procs);
    free( proc_list);

    for ( p= 0; p < nprocs; p++)  {
	ShareProc *share_proc= NULL;
	if ( strcmp( procs[p], "unknown") == 0)
	    default_tcl_unknown= GetProc( interp, "unknown");
	else if ( EnterProc( first_thread, procs[p], &share_proc, 0) == 
								TCL_ERROR)  {
	    free( procs);
	    return TCL_ERROR;
	}
    }
    free( procs);
    return TCL_OK;
}

/*
 * InitProcSharing
 *
 * Initialize proc sharing for a new Tcl thread.  Save the real proc
 * command in the Tcl thread.  If the interpreter doesn't have a proc
 * command, then don't make our own proc or unknown commands.  Otherwise
 * change the interpreter to use the MTtcl proc and unknown commands.
 * If the interpreter does not have an "unknown" command, one is created
 * for it from the default unknown command.
 */
void
InitProcSharing( Tcl_Thread *tcl_thread)
{
    Tcl_Interp *interp= tcl_thread->interp;
    Tcl_CmdInfo cmdinfo;

    if ( Tcl_GetCommandInfo( interp, "proc", &tcl_thread->proc_info) == 0)  {
	tcl_thread->proc_info.proc= 0;
	return;
    }

    cmdinfo.proc= MTtcl_ProcCmd;
    cmdinfo.clientData= (ClientData) tcl_thread;
    cmdinfo.deleteProc= NULL;
    cmdinfo.deleteData= NULL;
    Tcl_SetCommandInfo( interp, "proc", &cmdinfo);

    if ( Tcl_GlobalEval(interp, "rename unknown __tcl_unknown") == TCL_ERROR &&
		default_tcl_unknown != NULL) {
	char **pargv;
	int pargc;
	Tcl_SplitList( interp, default_tcl_unknown, &pargc, &pargv);
	pargv[1]= "__tcl_unknown";
	(tcl_thread->proc_info.proc)( tcl_thread->proc_info.clientData,
		interp, pargc, pargv);
	free(pargv);
    }

    Tcl_CreateCommand( interp, "unknown", MTtcl_UnknownCmd,
	(ClientData) tcl_thread, NULL);

    if ( Tcl_GetCommandInfo(interp, "rename", &tcl_thread->rename_info) == 0)  {
	tcl_thread->rename_info.proc= 0;
	return;
    }

    cmdinfo.proc= MTtcl_RenameCmd;
    cmdinfo.clientData= (ClientData) tcl_thread;
    cmdinfo.deleteProc= NULL;
    cmdinfo.deleteData= NULL;
    Tcl_SetCommandInfo( interp, "rename", &cmdinfo);

    if ( Tcl_GetCommandInfo(interp, "info", &tcl_thread->info_info) == 0)  {
	tcl_thread->info_info.proc= 0;
	return;
    }

    cmdinfo.proc= MTtcl_InfoCmd;
    cmdinfo.clientData= (ClientData) tcl_thread;
    cmdinfo.deleteProc= NULL;
    cmdinfo.deleteData= NULL;
    Tcl_SetCommandInfo( interp, "info", &cmdinfo);

    Tcl_ResetResult( interp);
}

/*
 * MTtcl_ProcCmd
 *
 * Command procedure for the "proc" command of Tcl threads.  Make the
 * new command available to all Tcl threads.  The real Tcl proc command
 * is used to create the command in the current thread.  If this succeeds
 * the command is made available to all Tcl threads.
 * If the command being created is "unknown", rename it to "__tcl_unknown".
 * This thread will have its own version of the "unknown" command which is
 * not shared with other threads.
 */
int
MTtcl_ProcCmd(
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    Tcl_HashEntry *entry;
    Tcl_Thread *tcl_thread= (Tcl_Thread *) clientData;
    Tcl_CmdInfo info;
    ShareProc *share_proc;
    char *proc;
    int rc;

    if ( tcl_thread->proc_info.proc == 0)  {
	interp->result= "invalid command name \"proc\"";
	return TCL_ERROR;
    }

if (DEBUG)
printf( "(%d) PROC; %s\n", thr_self(), argv[1]);

    if ( argc > 1 && strcmp( argv[1], "unknown") == 0)  {
	/* change name to __tcl_unknown, but don't put in shared table
	 */
	argv[1]= "__tcl_unknown";
	return (tcl_thread->proc_info.proc)( tcl_thread->proc_info.clientData,
		interp, argc, argv);
    }

    /* Call the real proc command first.
     */

    rc= (tcl_thread->proc_info.proc)( tcl_thread->proc_info.clientData,
	interp, argc, argv);
    if ( rc == TCL_ERROR)
	return TCL_ERROR;

    mutex_lock( &proc_lock);
    share_proc= NULL;
    rc= EnterProc( tcl_thread, argv[1], &share_proc, NULL);
    mutex_unlock( &proc_lock);

    if ( rc == TCL_OK)
	Tcl_ResetResult( interp);
    return rc;
}

/*
 * MTtcl_UnknownCmd
 *
 * Command procedure for the "unknown" command of Tcl threads.  Load
 * command from shared proc table if already defined by another thread.
 * Call real unknown proc (renamed to __tcl_unknown) if the command
 * doesn't exist in the shared proc table.
 */
int
MTtcl_UnknownCmd(
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    Tcl_HashEntry *entry;
    Tcl_CmdInfo info;
    Tcl_Thread *tcl_thread= (Tcl_Thread *) clientData;
    ShareProc *share_proc;
    char *proc, **pargv, *cmd;
    int pargc, rc;

    /* Look for the procedure in the shared proc hash table.  If not found
     * invoke real unknown command.  If procedure found, call the real proc
     * command to enter the procedure, then call the original command.
     */

if (DEBUG)
printf( "(%d) UNKNOWN; %s\n", thr_self(), argv[1]);

    mutex_lock( &proc_lock);
    entry= Tcl_FindHashEntry( &share_procs, argv[1]);
    if ( entry == NULL)  {
	mutex_unlock( &proc_lock);
	
	if ( Tcl_GetCommandInfo( interp, "__tcl_unknown", &info) == 0)  {
	    Tcl_AppendResult( interp, "invalid command name \"", argv[1], "\"");
	    return TCL_ERROR;
	}

	return (info.proc)( info.clientData, interp, argc, argv);
    }

    if ( tcl_thread->proc_info.proc == 0)  {
        mutex_unlock( &proc_lock);
	interp->result= "invalid command name \"proc\"";
	return TCL_ERROR;
    }

    share_proc= ( ShareProc *) Tcl_GetHashValue( entry);
    Tcl_SplitList( interp, share_proc->proc, &pargc, &pargv);
    (tcl_thread->proc_info.proc)( tcl_thread->proc_info.clientData,
		interp, pargc, pargv);
    free( pargv);

    rc= EnterProc( tcl_thread, argv[1], &share_proc, &info);

    mutex_unlock( &proc_lock);

    if ( rc == TCL_ERROR)
	return TCL_ERROR;

    return (info.proc)( info.clientData, interp, argc-1, argv+1);
}

/*
 * MTtcl_RenameCmd
 *
 *
 */
int
MTtcl_RenameCmd(
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    Tcl_HashEntry *entry;
    Tcl_CmdInfo info;
    Tcl_Thread *tcl_thread= (Tcl_Thread *) clientData;
    ShareProc *share_proc, *old_proc;
    int rc;

    if ( argc == 3)  {
	if ( strcmp(argv[1], "unknown") == 0)
	    argv[1]= "__tcl_unknown";
	if ( strcmp(argv[2], "unknown") == 0)
	    argv[2]= "__tcl_unknown";

	mutex_lock( &proc_lock);
	if ( argv[1][0] != '\0')
	    ReferenceProc( tcl_thread, argv[1]);
	if ( argv[2][0] != '\0')
	    ReferenceProc( tcl_thread, argv[2]);
	mutex_unlock( &proc_lock);
    }
    
    rc= (tcl_thread->rename_info.proc)( tcl_thread->rename_info.clientData,
	interp, argc, argv);
    if ( rc == TCL_ERROR)
	return TCL_ERROR;

    if ( argv[2][0] == '\0')
	return TCL_OK;

    mutex_lock( &proc_lock);
    Tcl_GetCommandInfo( interp, argv[2], &info);
    if ( info.proc != MTtcl_InterpProc)  {
	mutex_unlock( &proc_lock);
	return TCL_OK;
    }
    if ( strcmp( argv[2], "__tcl_unknown") == 0)  {
	ShareProcInfo *share_proc_info;
	share_proc_info= (ShareProcInfo *) info.clientData;
	Tcl_SetCommandInfo( interp, "__tcl_unknown", &share_proc_info->info);
	share_proc_info->share_proc->ref--;
	if ( share_proc_info->share_proc->ref == 0)  {
	    entry= Tcl_FindHashEntry( &share_procs, argv[1]);
	    if ( entry != NULL)
		Tcl_DeleteHashEntry( entry);
	    if ( share_proc_info->share_proc->proc != NULL)
		free( share_proc_info->share_proc->proc);
	    free( share_proc_info->share_proc->proc_name);
	    free( share_proc_info->share_proc);
	}
	free( share_proc_info);
    }
    else  {
	int pargc;
	char **pargv;
	entry= Tcl_FindHashEntry( &share_procs, argv[1]);
	if ( entry == NULL)  {
	    mutex_unlock( &proc_lock);
	    return TCL_OK;
	}
	old_proc= (ShareProc *) Tcl_GetHashValue( entry);
	share_proc= (ShareProc *) malloc( sizeof ( ShareProc));

	Tcl_SplitList( interp, old_proc->proc, &pargc, &pargv);
	pargv[1]= argv[2];
	share_proc->proc= Tcl_Merge( pargc, pargv);
	free( pargv);
	free( old_proc->proc);
	old_proc->proc= NULL;

	share_proc->proc_name= strdup( argv[2]);
	share_proc->ref= 0;
	Tcl_DeleteHashEntry( entry);
	EnterProc( tcl_thread, argv[2], &share_proc, NULL);
    }
    mutex_unlock( &proc_lock);
    return TCL_OK;
}

/*
 *
 * Must be called with proc_lock held.
 */
void
ReferenceProc( Tcl_Thread *tcl_thread, char *proc_name)
{
    Tcl_Interp *interp= tcl_thread->interp;
    Tcl_HashEntry *entry;
    Tcl_CmdInfo info;
    ShareProc *share_proc;

    if ( Tcl_GetCommandInfo( interp, proc_name, &info) == 0)  {
	entry= Tcl_FindHashEntry( &share_procs, proc_name);
	if ( entry != NULL)  {
	    share_proc= (ShareProc *) Tcl_GetHashValue( entry);
	    if ( share_proc->proc != NULL)  {
		int pargc;
		char **pargv;
		Tcl_SplitList( interp, share_proc->proc, &pargc, &pargv);
		(tcl_thread->proc_info.proc)(
		    tcl_thread->proc_info.clientData, interp, pargc, pargv);
		free( pargv);
		EnterProc( tcl_thread, proc_name, &share_proc, &info);
	    }
	}
    }
}


/*
 * EnterProc
 *
 * Record current Tcl thread's use of a shared proc.  A per thread, per
 * command structure is allocated to aid in command deletion and the
 * Tcl info for the command is rewritten to use MTtcl_InterpProc.
 * If the command does not exist in the share proc table, an entry is
 * created for it.  Special handling is performed to clean up a race
 * condition when two threads create the same proc simultaneously.
 * The losing thread is reset to use the winning threads definition
 * of the procedure.
 *
 * Must be called with proc_lock held.
 */
int
EnterProc(
    Tcl_Thread *tcl_thread,
    char *proc_name,
    ShareProc **share_proc,
    Tcl_CmdInfo *opt_info
)
{
    Tcl_Interp *interp= tcl_thread->interp;
    Tcl_CmdInfo info;
    Tcl_HashEntry *entry;
    ShareProcInfo *share_proc_info;
    int created;
    char *proc;

if (DEBUG)
printf( "(%d) ENTERPROC1; %s share_proc=%08x\n", thr_self(),
	proc_name, *share_proc);

    if ( *share_proc == NULL)  {
	entry= Tcl_FindHashEntry( &share_procs, proc_name);
	if ( entry != NULL)  {
	    ShareProc *old_proc;
	    old_proc= (ShareProc *) Tcl_GetHashValue(entry);
	    free( old_proc->proc);
	    old_proc->proc= NULL;
	    Tcl_DeleteHashEntry(entry);
	}
	if ( tcl_thread->info_info.proc != NULL)
	    Tcl_SetCommandInfo( interp, "info", & tcl_thread->info_info);
	proc= GetProc( interp, proc_name);
	if ( tcl_thread->info_info.proc != NULL)  {
	    Tcl_CmdInfo info;
	    info.proc= MTtcl_InfoCmd;
	    info.clientData= tcl_thread;
	    info.deleteProc= NULL;
	    info.deleteData= NULL;
	    Tcl_SetCommandInfo( interp, "info", & info);
	}

	if ( proc == NULL)
	    return TCL_ERROR;

	*share_proc= (ShareProc *) malloc( sizeof ( ShareProc));
	(*share_proc)->proc= proc;
	(*share_proc)->proc_name= strdup( proc_name);
	(*share_proc)->ref= 0;
    }

if ( DEBUG)
printf( "(%d) ENTERPROC2; %s share_proc=%08x ref=%d\n", thr_self(),
	proc_name, *share_proc, (*share_proc)->ref);

if ( DEBUG && strlen( (*share_proc)->proc) < (size_t)80)
printf( "(%d) ENTERPROC3; %s\n", thr_self(), (*share_proc)->proc);

    if ( Tcl_GetCommandInfo( interp, proc_name, &info) == 0)
	return TCL_ERROR;

    /* Create thread specific proc info and reset command to use MTtcl
     * procedures.
     */

    if ( info.proc != MTtcl_InterpProc)  {
	share_proc_info= (ShareProcInfo *) malloc( sizeof ( ShareProcInfo));
	share_proc_info->info= info;
	share_proc_info->tcl_thread= tcl_thread;
	share_proc_info->share_proc= *share_proc;
    
	info.proc= MTtcl_InterpProc;
	info.clientData= share_proc_info;
	info.deleteProc= MTtcl_ProcDeleteProc;
	info.deleteData= share_proc_info;
	Tcl_SetCommandInfo( interp, proc_name, &info);
    }
    else  {
	share_proc_info= (ShareProcInfo *) info.clientData;
	share_proc_info->share_proc= *share_proc;
    }

    if ( opt_info != NULL)
	*opt_info= info;

    (*share_proc)->ref++;
    if ( (*share_proc)->ref == 1)  {
	entry= Tcl_CreateHashEntry( &share_procs, proc_name, &created);
        Tcl_SetHashValue( entry, (ClientData) *share_proc);
    }
    return TCL_OK;
}

/*
 * MTtcl_InterpProc
 *
 * Command procedure for all MTtcl shared procs.  If command exists in
 * shared proc table, call the real command procedure.  Otherwise delete
 * command from this Tcl thread.
 */
int
MTtcl_InterpProc(
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    ShareProcInfo *share_proc_info= (ShareProcInfo *) clientData;
    Tcl_CmdInfo info;
    char *proc;
    char **uargv;
    int uargc, rc;

    mutex_lock( &proc_lock);
    proc= share_proc_info->share_proc->proc;
    mutex_unlock( &proc_lock);

    if ( proc != NULL)
	return (share_proc_info->info.proc)(share_proc_info->info.clientData,
		interp, argc, argv);

    Tcl_DeleteCommand( interp, argv[0]);

    uargc= argc+1;
    uargv= (char **) malloc( sizeof(char *) * (uargc+1));
    memcpy( uargv+1, argv, sizeof(char *) * uargc);
    uargv[0]= "unknown";
    rc= MTtcl_UnknownCmd( share_proc_info->tcl_thread, interp,
	uargc, uargv);
    free( uargv);
    return rc;
}

/*
 * MTtcl_ProcDeleteProc
 *
 * Delete proc registered with all shared procs in each interpreter.
 * Resources used by the proc in MTtcl and Tcl are released.  Reference
 * counting is used to ensure that MTtcl structures used by all threads
 * aren't released until the last thread deletes the command.  There is
 * a special case when a proc is being deleted as a result of Tcl thread
 * exit.  In this case, the deletion is not propagated to all threads.
 * In all cases, the original deleteProc is called to free Tcl allocated
 * resources.
 */
void
MTtcl_ProcDeleteProc( ClientData clientData)
{
    ShareProcInfo *info= (ShareProcInfo *) clientData;

if ( DEBUG)
printf( "MTtcl_ProcDeleteProc %s; result=%d ref=%d\n",
	info->share_proc->proc_name,
	(int) info->tcl_thread->result, info->share_proc->ref);

    /* tcl_thread->result is not set until the thread completes evaluation
     * of its script.  This indicates whether the command is being deleted
     * as a result of thread exit.
     * The proc field of ShareProc is NULL if the command has been globally
     * deleted.
     */

    mutex_lock( &proc_lock);
    if ( (info->tcl_thread->result == NULL || tcl_thread_count == 0) &&
		 info->share_proc->proc != NULL)  {
	Tcl_HashEntry *entry;
	free( info->share_proc->proc);
	info->share_proc->proc= NULL;
	entry= Tcl_FindHashEntry( &share_procs, info->share_proc->proc_name);
        if ( entry != NULL)
	    Tcl_DeleteHashEntry( entry);
    }
    mutex_unlock( &proc_lock);

    if ( info->info.deleteProc != NULL)
	(info->info.deleteProc)( info->info.deleteData);

    mutex_lock( &proc_lock);
    info->share_proc->ref--;
    if ( info->share_proc->ref == 0 && tcl_thread_count == 0)  {
	free( info->share_proc->proc_name);
	free( info->share_proc);
    }
    free( info);
    mutex_unlock( &proc_lock);
}

/*
 * GetProc
 *
 * Return a string which when evaluated will recreate the named procedure
 * from the given interpreter.
 */
char *
GetProc( Tcl_Interp *interp, char *proc_name)
{
    Tcl_DString proc_cmd;
    char cmd[128], *result, **args;
    int nargs, i;

    Tcl_DStringInit( &proc_cmd);
    Tcl_DStringAppendElement( &proc_cmd, "proc");
    Tcl_DStringAppendElement( &proc_cmd, proc_name);
    Tcl_DStringStartSublist( &proc_cmd);

    sprintf( cmd, "info args %s", proc_name);
    if ( Tcl_GlobalEval( interp, cmd) == TCL_ERROR)
	return NULL;
    result= strdup( interp->result);
    Tcl_SplitList( interp, result, &nargs, &args);
    free( result);
    for ( i= 0; i < nargs; i++)  {
	sprintf( cmd, "info default %s %s __arg_default_", proc_name, args[i]); 
	Tcl_GlobalEval( interp, cmd);
	if ( *interp->result == '1')  {
	    char *arg_default;
	    arg_default= Tcl_GetVar( interp, "__arg_default_", TCL_GLOBAL_ONLY);
	    Tcl_DStringStartSublist( &proc_cmd);
	    Tcl_DStringAppendElement( &proc_cmd, args[i]);
	    Tcl_DStringAppendElement( &proc_cmd, arg_default);
	    Tcl_DStringEndSublist( &proc_cmd);
	}
	else  {
	    Tcl_DStringAppendElement( &proc_cmd, args[i]);
	}
    }
    free( args);
    Tcl_UnsetVar( interp, "__arg_default_", TCL_GLOBAL_ONLY);
    Tcl_DStringEndSublist( &proc_cmd);

    sprintf( cmd, "info body %s", proc_name);
    Tcl_GlobalEval( interp, cmd);
    Tcl_DStringAppendElement( &proc_cmd, interp->result);

    result= strdup( Tcl_DStringValue( &proc_cmd));
    Tcl_DStringFree( &proc_cmd);
    return result;
}

/*
Proc *
MTtclFindProc( Tcl_Thread *tcl_thread, char *proc_name)
{
    Tcl_CmdInfo info;
    Tcl_HashEntry *entry;
    ShareProc *share_proc;
    ShareProcInfo *share_proc_info;

    mutex_lock( &proc_lock);
    entry= Tcl_FindHashEntry( &share_procs, proc_name);
    if ( entry == NULL)  {
	mutex_unlock( &proc_lock);
	return NULL;
    }
    share_proc= (ShareProc *) Tcl_GetHashValue( entry);
    if ( share_proc->proc == NULL)  {
	mutex_unlock( &proc_lock);
	return NULL;
    }

    if ( Tcl_GetCommandInfo( tcl_thread->interp, proc_name, &info) == 0)  {
	ReferenceProc( tcl_thread, proc_name);
	mutex_unlock( &proc_lock);
	if ( Tcl_GetCommandInfo( tcl_thread->interp, proc_name, &info) == 0)
	    return NULL;
    }
    mutex_unlock( &proc_lock);
    share_proc_info= (ShareProcInfo *) (info.clientData);
    return (Proc *) share_proc_info->info.clientData;
}
*/

int
MTtcl_InfoCmd( 
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    Tcl_Thread *tcl_thread= (Tcl_Thread *) clientData;
    size_t length;
    int c, rc;
    ShareProc *share_proc;
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);

	/* The proc is entered into the interpreter if it has not been
	 * referenced yet.  But, the proc will look like a command, not
	 * a procedure.  So we modify the CommandInfo for the duration
	 * of the call to the real "info" command.
 	 */

    if ( (c == 'a' && strncmp(argv[1], "args", length) == 0) ||
		(c == 'b' && strncmp(argv[1], "body", length) == 0) ||
		(c == 'd' && strncmp(argv[1], "default", length) == 0) ) {
	Tcl_CmdInfo info;
	ClientData saved;
	if (c == 'a' && argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " args procname\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (c == 'b' && argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " body procname\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (c == 'd' && argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " default procname arg varname\"",
		    (char *) NULL);
	    return TCL_ERROR;
	}

	mutex_lock( &proc_lock);
	ReferenceProc( tcl_thread, argv[2]);
	mutex_unlock( &proc_lock);

	if ( Tcl_GetCommandInfo( tcl_thread->interp, argv[2], &info) == 0) {
	    infoNoSuchProc:
	    Tcl_AppendResult(interp, "\"", argv[2],
		    "\" isn't a procedure", (char *) NULL);
	    return TCL_ERROR;
	}
	if ( info.proc != MTtcl_InterpProc)
	    goto infoNoSuchProc;

	info.proc= ((ShareProcInfo *)info.clientData)->info.proc;
	saved= info.clientData;
	info.clientData= ((ShareProcInfo *)info.clientData)->info.clientData;
	if ( Tcl_SetCommandInfo( tcl_thread->interp, argv[2], &info) == 0) {
	    /* should never happen */
	    return TCL_ERROR;
	}

	rc= (tcl_thread->info_info.proc)(tcl_thread->info_info.clientData,
		tcl_thread->interp, argc, argv);
	info.proc= MTtcl_InterpProc;
	info.clientData= saved;
	if ( Tcl_SetCommandInfo( tcl_thread->interp, argv[2], &info) == 0) {
	    /* should never happen */
	    return TCL_ERROR;
	}
	return rc;
    } else if ((c == 'p') && (strncmp(argv[1], "procs", length) == 0) &&
	(length >= 2)) {
	if (argc > 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " procs [pattern]\"", (char *) NULL);
	    return TCL_ERROR;
	}
	for (entry = Tcl_FirstHashEntry(&share_procs, &search);
		entry != NULL; entry = Tcl_NextHashEntry(&search)) {
	    char *name = Tcl_GetHashKey(&share_procs, entry);

	    if ((argc == 3) && !Tcl_StringMatch(name, argv[2])) {
		continue;
	    }
	    Tcl_AppendElement(interp, name);
	}
	if ( tcl_thread->info_info.proc != NULL)  {
	    int iargc= argc;
	    char *iargv[4];
	    iargv[0]= argv[0];
	    iargv[1]= argv[1];
	    iargv[2]= argv[2];
	    if ( iargc == 3)
		iargv[3]= argv[3];
	    (tcl_thread->info_info.proc)(tcl_thread->info_info.clientData,
		tcl_thread->interp, iargc, iargv);
	}
	return TCL_OK;
    }

    if ( tcl_thread->info_info.proc != NULL)
	return (tcl_thread->info_info.proc)(tcl_thread->info_info.clientData,
		tcl_thread->interp, argc, argv);

    interp->result= "invalid command \"info\"";
    return TCL_ERROR;
}

