/*
 * synch.c --
 *
 *	Provides Tcl thread synchronization primitives.
 *
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <synch.h>
#include <errno.h>
#include "tcl_thread.h"

int CondTimedWait( Tcl_Interp *interp, cond_t *cond, mutex_t *mutex,
    char *sec, char *nsec, int absolute);


static char mutex_help[] = {
     "mutex create"
"\n" "      destroy mutex-id"
"\n" "      lock mutex-id"
"\n" "      trylock mutex-id"
"\n" "      unlock mutex-id"
};

int
MTtcl_MutexCmd (
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    static char *options= "create, destroy, lock, trylock, unlock";
    mutex_t *mutex;
    int rc;

    if ( argc == 1)  {
mutexWrongArgs:
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" option mutex-id\"", NULL);
	return TCL_ERROR;
    }
    if ( argc == 2 && strcmp( argv[1], "help") == 0)  {
	puts( mutex_help);
	return TCL_OK;
    }
    if ( argc == 2 && strncmp( argv[1], "opt", 3) == 0)  {
	Tcl_AppendResult( interp, argv[0], " options are: ", options, NULL);
	return TCL_OK;
    }

    if ( argv[1][0] == 'c' && strcmp( argv[1], "create") == 0)  {
	int type= USYNC_THREAD;
	if ( argc == 3 && 0)  {
	    if ( strcmp( argv[2], "-process") == 0)
	        type= USYNC_PROCESS;
	    else if ( strcmp( argv[2], "-thread") == 0)
	        type= USYNC_THREAD;
	    else  {
	        Tcl_AppendResult( interp, "bad mutex type \"", argv[2],
			"\": should be -process or -thread", NULL);
	        return TCL_ERROR;
	    }
	}
	else if ( argc != 2)
	    goto mutexWrongArgs;

        mutex= (mutex_t *) malloc( sizeof( mutex_t));
        if ( (rc= mutex_init( mutex, type, 0)) != 0)  {
	    errno= rc;
            Tcl_AppendResult( interp, "couldn't create mutex: ",
		Tcl_PosixError( interp), NULL);
            return TCL_ERROR;
        }
	sprintf( interp->result, "mutex%u", (unsigned int) mutex);
	return TCL_OK;
    }

    if ( argc < 3)
	goto mutexWrongArgs;

    if ( argv[2][0] != 'm' && argv[2][1] != 'u')  {
	Tcl_AppendResult( interp, "\"", argv[2], "\" is not a mutex", NULL);
	return TCL_ERROR;
    }
    mutex= (mutex_t *) atoi(argv[2]+5);

    if ( argv[1][0] == 'l' && strcmp( argv[1], "lock") == 0)  {
        rc= mutex_lock( mutex);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't lock mutex: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'u' && strcmp( argv[1], "unlock") == 0)  {
        rc= mutex_unlock( mutex);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't unlock mutex: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 't' && strcmp( argv[1], "trylock") == 0)  {
	rc= mutex_trylock( mutex);
	if ( rc == EBUSY)  {
	    interp->result= "lock busy";
	    return TCL_ERROR;
	}
	else if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't trylock mutex: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'd' && strcmp( argv[1], "destroy") == 0)  {
	rc= mutex_destroy( mutex);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't destroy mutex: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
	free( mutex);
    }
    else  {
	Tcl_AppendResult( interp, "bad option \"", argv[1], "\": should be ",
		options, NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}


static char sema_help[] = {
     "sema create count"
"\n" "     destroy sema-id"
"\n" "     post sema-id"
"\n" "     trywait sema-id"
"\n" "     wait sema-id"
};

int
MTtcl_SemaCmd (
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    static char *options= "create, destroy, post, trywait, wait";
    sema_t *sema;
    int rc;

    if ( argc == 1)  {
semaWrongArgs:
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" option sema-id\"", NULL);
	return TCL_ERROR;
    }
    if ( argc == 2 && strcmp( argv[1], "help") == 0)  {
	puts( sema_help);
	return TCL_OK;
    }
    if ( argc == 2 && strncmp( argv[1], "opt", 3) == 0)  {
	Tcl_AppendResult( interp, argv[0], " options are: ", options, NULL);
	return TCL_OK;
    }

    if ( argv[1][0] == 'c' && strcmp( argv[1], "create") == 0)  {
	int type= USYNC_THREAD, count;
	if ( argc == 4 && 0)  {
	    if ( strcmp( argv[2], "-process") == 0)
	        type= USYNC_PROCESS;
	    else if ( strcmp( argv[2], "-thread") == 0)
	        type= USYNC_THREAD;
	    else  {
	        Tcl_AppendResult( interp, "bad semaphore type \"", argv[2],
			"\": should be -process or -thread", NULL);
	        return TCL_ERROR;
	    }
	}
	else if ( argc != 3)
	    goto semaWrongArgs;

	if ( Tcl_GetInt( interp, argv[argc-1], &count) != TCL_OK)
	    return TCL_ERROR;

	sema= (sema_t *) malloc( sizeof( sema_t));
        if ( (rc= sema_init( sema, count, type, 0)) != 0)  {
	    errno= rc;
            Tcl_AppendResult( interp, "couldn't create semaphore: ",
		Tcl_PosixError( interp), NULL);
	    return TCL_ERROR;
        }

	sprintf( interp->result, "sema%u", (unsigned int) sema);
	return TCL_OK;
    }

    if ( argc < 3)
	goto semaWrongArgs;

    if ( argv[2][0] != 's' && argv[2][1] != 'e')  {
	Tcl_AppendResult( interp, "\"", argv[2], "\" is not a semaphore", NULL);
	return TCL_ERROR;
    }
    sema= (sema_t *) atoi( argv[2]+4);

    if ( argv[1][0] == 'w' && strcmp( argv[1], "wait") == 0)  {
	rc= sema_wait( sema);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't wait on semaphore: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'p' && strcmp( argv[1], "post") == 0)  {
	rc= sema_post( sema);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't post on semaphore: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 't' && strcmp( argv[1], "trywait") == 0)  {
	rc= sema_trywait( sema);
	if ( rc == EBUSY)  {
	    interp->result= "semaphore has zero count";
	    return TCL_ERROR;
	}
	else if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't trywait on semaphore: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'd' && strcmp( argv[1], "destroy") == 0)  {
	rc= sema_destroy( sema);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't destroy semaphore: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
	free( sema);
    }
    else  {
	Tcl_AppendResult( interp, "bad option \"", argv[1], "\": should be ",
		options, NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}


static char rwlock_help[] = {
     "rwlock create"
"\n" "       destroy rwlock-id"
"\n" "       read rwlock-id"
"\n" "       tryread rwlock-id"
"\n" "       trywrite rwlock-id"
"\n" "       unlock rwlock-id"
"\n" "       write rwlock-id"
};

int
MTtcl_RwlockCmd (
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    static char *options= "create, destroy, read, tryread, trywrite, unlock, write";
    rwlock_t *rwlock;
    int rc;

    if ( argc == 1)  {
rwlockWrongArgs:
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" option rwlock-id\"", NULL);
	return TCL_ERROR;
    }
    if ( argc == 2 && strcmp( argv[1], "help") == 0)  {
	puts( rwlock_help);
	return TCL_OK;
    }
    if ( argc == 2 && strncmp( argv[1], "opt", 3) == 0)  {
	Tcl_AppendResult( interp, argv[0], " options are: ", options, NULL);
	return TCL_OK;
    }

    if ( argv[1][0] == 'c' && strcmp( argv[1], "create") == 0)  {
	int type= USYNC_THREAD;
	if ( argc == 3 && 0)  {
	    if ( strcmp( argv[2], "-process") == 0)
	        type= USYNC_PROCESS;
	    else if ( strcmp( argv[2], "-thread") == 0)
	        type= USYNC_THREAD;
	    else  {
	        Tcl_AppendResult( interp, "bad rwlock type \"", argv[2],
			"\": should be -process or -thread", NULL);
	        return TCL_ERROR;
	    }
	}
	else if ( argc != 2)
	    goto rwlockWrongArgs;

        rwlock= (rwlock_t *) malloc( sizeof( rwlock_t));
        if ( (rc= rwlock_init( rwlock, type, 0)) != 0)  {
	    errno= rc;
            Tcl_AppendResult( interp, "couldn't create rwlock: ",
		Tcl_PosixError( interp), NULL);
            return TCL_ERROR;
        }
	sprintf( interp->result, "rwlock%u", (unsigned int) rwlock);
	return TCL_OK;
    }

    if ( argc < 3)
	goto rwlockWrongArgs;

    if ( argv[2][0] != 'r' && argv[2][1] != 'w')  {
	Tcl_AppendResult( interp, "\"", argv[2], "\" is not a rwlock", NULL);
	return TCL_ERROR;
    }
    rwlock= (rwlock_t *) atoi(argv[2]+6);

    if ( argv[1][0] == 'r' && strcmp( argv[1], "read") == 0)  {
        rc= rw_rdlock( rwlock);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't get read lock: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'w' && strcmp( argv[1], "write") == 0)  {
        rc= rw_wrlock( rwlock);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't get write lock: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'u' && strcmp( argv[1], "unlock") == 0)  {
        rc= rw_unlock( rwlock);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't unlock rwlock: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'd' && strcmp( argv[1], "destroy") == 0)  {
	rc= rwlock_destroy( rwlock);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't destroy rwlock: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
	free( rwlock);
    }
    else if ( strcmp( argv[1], "tryread") == 0)  {
	rc= rw_tryrdlock( rwlock);
	if ( rc == EBUSY)  {
	    interp->result= "lock busy";
	    return TCL_ERROR;
	}
	else if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't tryread rwlock: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( strcmp( argv[1], "trywrite") == 0)  {
	rc= rw_trywrlock( rwlock);
	if ( rc == EBUSY)  {
	    interp->result= "lock busy";
	    return TCL_ERROR;
	}
	else if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't trywrite rwlock: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else  {
	Tcl_AppendResult( interp, "bad option \"", argv[1], "\": should be ",
		options, NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}


static char cond_help[] = {
     "cond create"
"\n" "     broadcast cond-id"
"\n" "     destroy cond-id"
"\n" "     signal cond-id"
"\n" "     timedwait ?-absolute? cond-id mutex-id seconds ?nano-seconds?"
"\n" "     wait cond-id mutex-id"
};

int
MTtcl_CondCmd (
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    static char *options= "broadcast, create, destroy, signal, timedwait, wait";
    cond_t *cond;
    int rc;

    if ( argc == 1)  {
condWrongArgs:
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" option cond-id\"", NULL);
	return TCL_ERROR;
    }
    if ( argc == 2 && strcmp( argv[1], "help") == 0)  {
	puts( cond_help);
	return TCL_OK;
    }
    if ( argc == 2 && strncmp( argv[1], "opt", 3) == 0)  {
	Tcl_AppendResult( interp, argv[0], " options are: ", options, NULL);
	return TCL_OK;
    }

    if ( argv[1][0] == 'c' && strcmp( argv[1], "create") == 0)  {
	int type= USYNC_THREAD;
	if ( argc == 3 && 0)  {
	    if ( strcmp( argv[2], "-process") == 0)
	        type= USYNC_PROCESS;
	    else if ( strcmp( argv[2], "-thread") == 0)
	        type= USYNC_THREAD;
	    else  {
	        Tcl_AppendResult( interp, "bad mutex type \"", argv[2],
			"\": should be -process or -thread", NULL);
	        return TCL_ERROR;
	    }
	}
	else if ( argc != 2)
	    goto condWrongArgs;

        cond= (cond_t *) malloc( sizeof( cond_t));
        if ( (rc= cond_init( cond, type, 0)) != 0)  {
	    errno= rc;
            Tcl_AppendResult( interp, "couldn't create condition: ",
		Tcl_PosixError( interp), NULL);
            return TCL_ERROR;
        }
	sprintf( interp->result, "cond%u", (unsigned int) cond);
	return TCL_OK;
    }

    if ( argc < 3)
	goto condWrongArgs;

    if ( argv[1][0] == 't' && strcmp( argv[1], "timedwait") == 0)  {
	int absolute= 0, arg;
	mutex_t *mutex;
	timestruc_t abstime;
	if ( argc < 5)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
	    " timedwait ?-absolute? cond-id mutex-id seconds ?nano-seconds?\"",
		NULL);
	    return TCL_ERROR;
	}
	arg= 2;
	if ( argv[arg][0] == '-')  {
	    if ( strcmp( argv[arg], "-absolute") == 0)
		absolute++;
	    else  {
		Tcl_AppendResult( interp, "bad flag \"", argv[arg], "\"", NULL);
		return TCL_ERROR;
	    }
	    arg++;
	}
	cond= (cond_t *) atoi(argv[arg++]+4);
	mutex= (mutex_t *) atoi(argv[arg++]+5);
	rc= CondTimedWait( interp, cond, mutex, argv[arg++],
		( arg < argc) ? argv[arg++] : "0", absolute);
        if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't wait on condition: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
        }
    }

    if ( argv[2][0] != 'c' && argv[2][1] != 'o')  {
	Tcl_AppendResult( interp, "\"", argv[2], "\" is not a condition", NULL);
	return TCL_ERROR;
    }
    cond= (cond_t *) atoi(argv[2]+4);

    if ( argv[1][0] == 'w' && strcmp( argv[1], "wait") == 0)  {
	mutex_t *mutex;
	if ( argc != 4)  {
	    Tcl_AppendResult( interp,
		"wrong # args: should be \"", argv[0],
		" wait cond-id mutex-id\"", NULL);
	    return TCL_ERROR;
	}
        if ( argv[3][0] != 'm' && argv[3][1] != 'u')  {
	    Tcl_AppendResult( interp, "\"", argv[3], "\" is not a mutex", NULL);
	    return TCL_ERROR;
        }
	mutex= (mutex_t *) atoi( argv[3]+5);
        rc= cond_wait( cond, mutex);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't wait on condition: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 's' && strcmp( argv[1], "signal") == 0)  {
        rc= cond_signal( cond);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't signal condition: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'b' && strcmp( argv[1], "broadcast") == 0)  {
	rc= cond_broadcast( cond);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't broadcast on condition: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
    }
    else if ( argv[1][0] == 'd' && strcmp( argv[1], "destroy") == 0)  {
	rc= cond_destroy( cond);
	if ( rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't destroy condition: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
	free( cond);
    }
    else  {
	Tcl_AppendResult( interp, "bad option \"", argv[1], "\": should be ",
		options, NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}

int
CondTimedWait(
    Tcl_Interp *interp,
    cond_t *cond,
    mutex_t *mutex,
    char *sec,
    char *nsec,
    int absolute
)
{
    timestruc_t abstime;
    int rc;

    abstime.tv_nsec= 0;
    if ( Tcl_GetInt( interp, sec, (int *)&abstime.tv_sec) != TCL_OK)
	return TCL_ERROR;
    if ( Tcl_GetInt(interp, nsec, (int *)&abstime.tv_nsec) != TCL_OK)
	return TCL_ERROR;

    if ( ! absolute)  {
	struct timeval curtime;
	gettimeofday( &curtime, NULL);
	abstime.tv_sec+= abstime.tv_nsec / 1000000000U;
	abstime.tv_nsec= abstime.tv_nsec % 1000000000U;
	abstime.tv_sec+= curtime.tv_sec;
	abstime.tv_nsec+= curtime.tv_usec * 1000;
	abstime.tv_sec+= abstime.tv_nsec / 1000000000U;
	abstime.tv_nsec= abstime.tv_nsec % 1000000000U;
    }
    return cond_timedwait( cond, mutex, &abstime);

}
