/* 
 * tcl_thread.c --
 *
 *	Tcl threads, a threads extension for Solaris 2.
 *
 *  Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <thread.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <sys/stat.h>

#include "tcl_thread.h"
#include "tcl_threadP.h"

typedef struct {
    unsigned int array;
    union {
	Tcl_HashTable *array;
	char *value;
	void *void_ptr;
    } u;
    unsigned int ref;
} ShareVar;

static Tcl_HashTable share_vars;
static mutex_t share_lock;

/*
 * For each linked variable there is a data structure of the following
 * type, which describes the link and is the clientData for the trace
 * set on the Tcl variable.
 */
typedef struct {
    Tcl_Interp *interp;		/* Interpreter containing Tcl variable. */
    ShareVar *share_var;	/* Location of C variable. */
} ShareLink;

extern Tcl_CmdProc MTtcl_MutexCmd;
extern Tcl_CmdProc MTtcl_SemaCmd;
extern Tcl_CmdProc MTtcl_RwlockCmd;
extern Tcl_CmdProc MTtcl_CondCmd;
extern int CondTimedWait( Tcl_Interp *interp, cond_t *cond, mutex_t *mutex,
    char *sec, char *nsec, int absolute);

static Tcl_CmdProc MTtcl_ThreadCmd;
static Tcl_CmdProc MTtcl_ShareCmd;

static int ThreadCreate( Tcl_Interp *interp, int argc, char **argv);
static int ThreadJoin( Tcl_Interp *interp, int argc, char **argv);
static int ThreadList( Tcl_Interp *interp, int argc, char **argv);

static int NextMessageCmd( Tcl_Thread *tcl_thread, int argc, char *argv[]);
static int NextMessage( Tcl_Thread *tcl_thread, char *seconds,
	char *nano_seconds, char **message);

static void MTtcl_ShareUnlinkVar( Tcl_Interp *interp, char *varName);
static int MTtcl_ShareLinkVar( Tcl_Interp *interp, char *varName,
	ShareVar *var, int new_array, unsigned int flags);
static char * ShareLinkTraceProc (ClientData clientData,
	Tcl_Interp *interp, char *name1, char *name2, int flags);
static ShareVar *createShareVar( Tcl_Interp *interp, char *var_name,
	Tcl_HashEntry *entry);
static void cleanupShareVar( ShareVar *share_var);
static void releaseShareVar( ShareVar *share_var, char *name);
static void printShareTable( char *prefix1, char *prefix2);

static int GetThread( Tcl_Interp *interp, char *tcl_thread_id,
	Tcl_Thread **tcl_thread);

static void * Tcl_thread( void *arg);

static Tcl_HashTable threads;
static mutex_t threads_lock;
static thread_key_t tcl_thread_key;
static mutex_t thread_key_lock;

int tcl_thread_count= 0;

static char *DEFAULT_ERROR_PROC= "proc threaderror {err} {\n"
"    global errorInfo\n"
"    puts \"ERROR [thread self] $err\n$errorInfo\"\n"
"}";

static char *default_error_proc= NULL;

/* Called just before a thread is started for an interpretter
 */
static int (*ThrInterp_init)( Tcl_Interp *) = NULL;

#define THREAD_ID_MAXLEN	24

/*
 * Tcl_thread_Init
 *
 * Initialize Tcl threads extension.  Also the entry point for the
 * Shells dynamic loader.
 */
int
Tcl_thread_Init( Tcl_Interp *interp)
{
    static int once= 0;
    char tcl_thread_id[THREAD_ID_MAXLEN];
    Tcl_Thread *tcl_thread= NULL;

    sprintf( tcl_thread_id, "thread%u", thr_self());
    mutex_lock( &thread_key_lock);

    if ( once == 0)  {
	thread_t init_tid;
	Tcl_HashEntry *entry;
        int rc, created;

	if ( (rc= thr_keycreate( &tcl_thread_key, NULL)) != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't create thread key: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}

	Tcl_InitHashTable( &threads, TCL_STRING_KEYS);
	Tcl_InitHashTable( &share_vars, TCL_STRING_KEYS);

	tcl_thread= (Tcl_Thread *) calloc( 1, sizeof ( Tcl_Thread));
	tcl_thread->interp= interp;
	tcl_thread->tid= thr_self();
        tcl_thread->script= NULL;
	tcl_thread->thr_name= NULL;
	tcl_thread->eval_rc= -1;
	tcl_thread->evaluating= 1;
	tcl_thread->status= "running";
	tcl_thread->result= NULL;
	tcl_thread->messages= NULL;
	tcl_thread->msg_array_max= 0;
	tcl_thread->msg_array_len= 0;
	tcl_thread->msg_array_next= 0;
	cond_init( &tcl_thread->msg_cond, USYNC_THREAD, 0);
	tcl_thread->msg_notifier_pipe[0]= -1;
	tcl_thread->msg_notifier_pipe[1]= -1;
	tcl_thread->flags= 0;
	entry= Tcl_CreateHashEntry( &threads, tcl_thread_id, &created);
	if ( ! created)  {
	    tcl_thread->status= "collision";
	    mutex_unlock( &thread_key_lock);
	    interp->result= "couldn't init Tcl threads: thread-id collision";
	    return TCL_ERROR;
	}
	Tcl_SetHashValue( entry, (ClientData) tcl_thread);
	tcl_thread_count++;
	(void) thr_setspecific( tcl_thread_key, tcl_thread);
	default_error_proc= DEFAULT_ERROR_PROC;
	Tcl_GlobalEval( interp, default_error_proc);

	if ( InitSharedProcTable( tcl_thread) == TCL_ERROR)  {
	    mutex_unlock( &thread_key_lock);
	    return TCL_ERROR;
	}
	InitProcSharing( tcl_thread);
	once++;
    }

    mutex_unlock( &thread_key_lock);

    Tcl_CreateCommand( interp, "thread", MTtcl_ThreadCmd, tcl_thread, 0);
    Tcl_CreateCommand( interp, "share", MTtcl_ShareCmd, tcl_thread, 0);
    Tcl_CreateCommand( interp, "mutex", MTtcl_MutexCmd, tcl_thread, 0);
    Tcl_CreateCommand( interp, "sema", MTtcl_SemaCmd, tcl_thread, 0);
    Tcl_CreateCommand( interp, "rwlock", MTtcl_RwlockCmd, tcl_thread, 0);
    Tcl_CreateCommand( interp, "cond", MTtcl_CondCmd, tcl_thread, 0);

    return TCL_OK;
}

/*
 * Set function which will be invoked just before Tcl thread is
 * created.  This is a handy hook for interpretters built with
 * multiple extensions.  Each extension should be initialized
 * in the thread init function.
 */
void
MTtcl_SetThreadInitFunc( int (*init_func)( Tcl_Interp *))
{
    ThrInterp_init= init_func;
}


static char thread_help[] = {
"thread create ?-suspended? ?-detached? ?-bound? ?-daemon? ?-new_lwp? ?-name name? script ?arg arg ...?"
"\n" "       continue thread-id"
"\n" "       errorCode thread-id"
"\n" "       errorInfo thread-id"
"\n" "       exit ?result?"
"\n" "       getconcurrency"
"\n" "       getprio thread-id"
"\n" "       join ?-result variable? ?thread-id?"
"\n" "       kill thread-id signo"
"\n" "       list"
"\n" "       name thread-id ?name?"
"\n" "       nextmsg ?seconds? ?nano-seconds? ?variable?"
"\n" "       post thread-id message"
"\n" "       result thread-id"
"\n" "       script thread-id"
"\n" "       self"
"\n" "       setconcurrency level"
"\n" "       setprio thread-id priority"
"\n" "       sleep seconds ?nano-seconds?"
"\n" "       status thread-id"
"\n" "       suspend thread-id"
"\n" "       yield"
};


int
MTtcl_ThreadCmd (
    ClientData       clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    static char *options= "continue, create, errorCode, errorInfo, exit, getconcurrency, getprio, join, kill, list, name, nextmsg, post, result, script, self, setconcurrency, setprio, sleep, status, suspend, yield";
    int rc;
    char *cmd= argv[1];
    Tcl_Thread *tcl_thread= (Tcl_Thread *) clientData;
    Tcl_Thread *target_thread= NULL;

    if ( argc == 1)  {
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" option ?arg ...?\"", NULL);
	return TCL_ERROR;
    }

    if ( argc == 2 && strcmp( argv[1], "help") == 0)  {
	puts( thread_help);
	return TCL_OK;
    }
    if ( argc == 2 && strncmp( argv[1], "opt", 3) == 0)  {
	Tcl_AppendResult( interp, argv[0], " options are: \"", options, "\"",
		NULL);
	return TCL_OK;
    }

    if ( *cmd == 'c' && strcmp( cmd, "create") == 0)  {
	return ThreadCreate( interp, argc, argv);
    }
    else if ( *cmd == 'n' && strcmp( cmd, "nextmsg") == 0)  {
	return NextMessageCmd( tcl_thread, argc, argv);
    }
    else if ( *cmd == 'j' && strcmp( cmd, "join") == 0)  {
	return ThreadJoin( interp, argc, argv);
    }
    else if ( *cmd == 'l' && strcmp( cmd, "list") == 0)  {
	return ThreadList( interp, argc, argv);
    }
    else if ( *cmd == 's' && strcmp( cmd, "sleep") == 0)  {
	cond_t never;
	mutex_t say;
	if ( argc < 3 || argc > 4)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" sleep seconds ?nano-seconds?\"", NULL);
	    return TCL_ERROR;
	}
	cond_init( &never, USYNC_THREAD, 0);
	mutex_init( &say, USYNC_THREAD, 0);
	rc= CondTimedWait( interp, &never, &say, argv[2],
		( argc == 4) ? argv[3] : "0", 0);
	cond_destroy( &never);
	mutex_destroy( &say);
	if ( rc != ETIME && rc != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't sleep: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
	return TCL_OK;
    }
    else if ( *cmd == 'h' && strcmp( cmd, "hrtime") == 0)  {
	static hrtime_t last= 0;
	hrtime_t now= gethrtime();
	sprintf( interp->result, "%u", (unsigned int) (now-last));
	last= now;
	return TCL_OK;
    }
    else if ( *cmd == 'e' && strcmp( cmd, "exit") == 0)  {
	if ( tcl_thread != NULL)  {
	    mutex_lock( &tcl_thread->lock);
	    if ( argc == 2)
	        tcl_thread->result= strdup( "");
	    else
	        tcl_thread->result= strdup( argv[2]);
	    tcl_thread->status= "exit";
	    mutex_unlock( &tcl_thread->lock);
	}
	thr_exit( tcl_thread);
	/* not reached */
    }
    else if ( *cmd == 's' && strcmp( cmd, "self") == 0)  {
	if ( tcl_thread == NULL)  {
	    if ( GetThread(interp, "self", &tcl_thread) != 0)
		return TCL_ERROR;
	}
	else
	    mutex_lock( &tcl_thread->lock);
	sprintf( interp->result, "thread%u", tcl_thread->tid);
	mutex_unlock( &tcl_thread->lock);
	return TCL_OK;
    }
    else if ( *cmd == 's' && strcmp( cmd, "setconcurrency") == 0)  {
	int new_level;
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" setconcurrency level\"", NULL);
	    return TCL_ERROR;
	}
	if ( Tcl_GetInt( interp, argv[2], &new_level) == TCL_ERROR)
	    return TCL_ERROR;
	if ( (rc= thr_setconcurrency( new_level)) != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't setconcurrency: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
	return TCL_OK;
    }
    else if ( *cmd == 'g' && strcmp( cmd, "getconcurrency") == 0)  {
	if ( argc != 2)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" getconcurrency\"", NULL);
	    return TCL_ERROR;
	}
	sprintf( interp->result, "%d", thr_getconcurrency());
	return TCL_OK;
    }
    else if ( *cmd == 'y' && strcmp( cmd, "yield") == 0)  {
	if ( argc != 2)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" yield\"", NULL);
	    return TCL_ERROR;
	}
	thr_yield();
	return TCL_OK;
    }

    if ( argc > 2)  {
	if ( GetThread( interp, argv[2], &target_thread) != 0)
	    return TCL_ERROR;
    }

    /* Options which take a thread-id argument
     */

    do  {

    rc= TCL_OK;

    if ( *cmd == 'n' && strcmp( cmd, "name") == 0)  {
	if ( argc < 3 || argc > 4)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" name thread-id ?name?\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( argc == 4)  {
	    if ( target_thread->thr_name != NULL)	
	        free( target_thread->thr_name);
	    target_thread->thr_name= strdup( argv[3]);
	}
	else
	    Tcl_SetResult( interp, target_thread->thr_name, TCL_VOLATILE);
    }
    else if ( *cmd == 'p' && strcmp( cmd, "post") == 0)  {
	if ( argc != 4)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" post thread-id message\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	rc= MessagePost( interp, target_thread, argv[3]);
    }
    else if ( *cmd == 's' && strcmp( cmd, "status") == 0)  {
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" status thread-id\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	Tcl_SetResult( interp, target_thread->status, TCL_VOLATILE);
    }
    else if ( *cmd == 'r' && strcmp( cmd, "result") == 0)  {
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" result thread-id\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( target_thread->result == NULL)  {
	    interp->result= 
		"couldn't get result: not available until thread exits";
	    rc= TCL_ERROR;
	    break;
	}
	else
	    Tcl_SetResult( interp, target_thread->result, TCL_VOLATILE);
    }
    else if ( *cmd == 's' && strcmp( cmd, "script") == 0)  {
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" script thread-id\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( target_thread->script == NULL)
	    interp->result= "";
	else
	    Tcl_SetResult( interp, target_thread->script, TCL_VOLATILE);
    }
    else if ( *cmd == 's' && strcmp( cmd, "setprio") == 0)  {
	int priority;
	if ( argc != 4)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" setprio thread-id priority\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( Tcl_GetInt( interp, argv[3], &priority) == TCL_ERROR)  {
	    rc= TCL_ERROR;
	    break;
	}
	if ( (rc= thr_setprio( target_thread->tid, priority)) != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't set priority: ",
		Tcl_PosixError(interp), NULL);
	    rc= TCL_ERROR;
	    break;
	}
    }
    else if ( *cmd == 'g' && strcmp( cmd, "getprio") == 0)  {
	int priority;
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" getprio thread-id\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( (rc= thr_getprio( target_thread->tid, &priority)) != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't get priority: ",
		Tcl_PosixError(interp), NULL);
	    rc= TCL_ERROR;
	    break;
	}
	sprintf( interp->result, "%d", priority);
    }
    else if ( *cmd == 's' && strcmp( cmd, "suspend") == 0)  {
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" suspend thread-id\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	mutex_unlock( &target_thread->lock);
	if ( (rc= thr_suspend( target_thread->tid)) != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't suspend: ",
		Tcl_PosixError(interp), NULL);
	    return TCL_ERROR;
	}
	return TCL_OK;
    }
    else if ( *cmd == 'c' && strcmp( cmd, "continue") == 0)  {
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" continue thread-id\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( (rc= thr_continue( target_thread->tid)) != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't continue: ",
		Tcl_PosixError(interp), NULL);
	    rc= TCL_ERROR;
	    break;
	}
    }
    else if ( *cmd == 'k' && strcmp( cmd, "kill") == 0)  {
	int signo;
	if ( argc != 4)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" kill thread-id signo\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( Tcl_GetInt( interp, argv[3], &signo) == TCL_ERROR)  {
	    rc= TCL_ERROR;
	    break;
	}
	if ( (rc= thr_kill( target_thread->tid, signo)) != 0)  {
	    errno= rc;
	    Tcl_AppendResult( interp, "couldn't kill: ",
		Tcl_PosixError(interp), NULL);
	    rc= TCL_ERROR;
	    break;
	}
    }
    else if ( *cmd == 'e' && strcmp( cmd, "errorInfo") == 0)  {
	char *result;
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" errorInfo thread-id\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( target_thread->evaluating)  {
	    interp->result= "couldn't get errorInfo: not available while thread is running";
	    rc= TCL_ERROR;
	    break;
	}
	result= Tcl_GetVar( target_thread->interp, "errorInfo",
		TCL_GLOBAL_ONLY|TCL_LEAVE_ERR_MSG);
	if ( result == NULL)  {
	    rc= TCL_ERROR;
	    break;
	}
	Tcl_SetResult( interp, result, TCL_VOLATILE);
    }
    else if ( *cmd == 'e' && strcmp( cmd, "errorCode") == 0)  {
	char *result;
	if ( argc != 3)  {
	    Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" errorCode thread-id\"", NULL);
	    rc= TCL_ERROR;
	    break;
	}
	if ( strcmp( target_thread->status, "running") == 0)  {
	    interp->result=
		"couldn't get errorCode: not available until thread exits";
	    rc= TCL_ERROR;
	    break;
	}
	result= Tcl_GetVar( target_thread->interp, "errorCode",
		TCL_GLOBAL_ONLY|TCL_LEAVE_ERR_MSG);
	if ( result == NULL)  {
	    rc= TCL_ERROR;
	    break;
	}
	Tcl_SetResult( interp, result, TCL_VOLATILE);
    }
    else  {
        Tcl_AppendResult( interp, "bad option \"", cmd, "\": should be ",
		options, NULL);
	rc= TCL_ERROR;
    }

    } while (0);

    mutex_unlock( &target_thread->lock);
    return rc;
}

int
ThreadCreate(
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    static char *flags= "-bound -daemon -detached -new_lwp -suspended -name";
    static char *syntax= "create ?flags ...? script ?arg arg ...?";
    Tcl_Thread *tcl_thread;
    unsigned int thr_argc, thr_flags, arg;
    Tcl_DString thr_argv;
    char argc_str[16], tcl_thread_id[THREAD_ID_MAXLEN];
    char *thr_name, *var_value;
    Tcl_HashEntry *entry;
    sigset_t orig_mask, no_sigs;
    int created, rc;

    thr_flags= 0;
    thr_name= "";
    for ( arg= 2; arg < argc; arg++)  {
	if ( argv[arg][0] != '-')
	    break;
	if ( strcmp( "-bound", argv[arg]) == 0)
	    thr_flags|= THR_BOUND;
	else if ( strcmp( "-suspended", argv[arg]) == 0)
	    thr_flags|= THR_SUSPENDED;
	else if ( strcmp( "-detached", argv[arg]) == 0)
	    thr_flags|= THR_DETACHED;
	else if ( strcmp( "-daemon", argv[arg]) == 0)
	    thr_flags|= THR_DAEMON;
	else if ( strcmp( "-new_lwp", argv[arg]) == 0)
	    thr_flags|= THR_NEW_LWP;
	else if ( strcmp( "-name", argv[arg]) == 0)  {
	    arg++;
	    if ( arg >= argc)  {
		Tcl_AppendResult( interp, "value for \"-name\" missing",
			NULL);
		return TCL_ERROR;
	    }
	    thr_name= argv[arg];
	}
	else  {
	    Tcl_AppendResult( interp, "bad flag \"", argv[arg],
		"\": flags are ", flags, NULL);
	    return TCL_ERROR;
	}
    }

    if ( arg >= argc)  {
	Tcl_AppendResult( interp, "wrong # args; should be \"", argv[0], " ",
		syntax, "\"", NULL);
	return TCL_ERROR;
    }

    tcl_thread= (Tcl_Thread *) calloc( 1, sizeof( Tcl_Thread));
    tcl_thread->script= strdup( argv[arg]);
    tcl_thread->thr_name= strdup( thr_name);
    tcl_thread->eval_rc= -1;
    tcl_thread->evaluating= 0;
    tcl_thread->status= "suspended";
    tcl_thread->result= NULL;
    tcl_thread->flags= thr_flags;
    tcl_thread->interp= Tcl_CreateInterp();
    tcl_thread->messages= NULL;
    tcl_thread->msg_array_max= 0;
    tcl_thread->msg_array_len= 0;
    tcl_thread->msg_array_next= 0;
    cond_init( &tcl_thread->msg_cond, USYNC_THREAD, 0);
    tcl_thread->msg_notifier_pipe[0]= -1;
    tcl_thread->msg_notifier_pipe[1]= -1;

    Tcl_CreateCommand( tcl_thread->interp, "thread", MTtcl_ThreadCmd,
	tcl_thread, 0);
    Tcl_CreateCommand( tcl_thread->interp, "share", MTtcl_ShareCmd,
	tcl_thread, 0);
    Tcl_CreateCommand( tcl_thread->interp, "mutex", MTtcl_MutexCmd,
	tcl_thread, 0);
    Tcl_CreateCommand( tcl_thread->interp, "sema", MTtcl_SemaCmd,
	tcl_thread, 0);
    Tcl_CreateCommand( tcl_thread->interp, "rwlock", MTtcl_RwlockCmd,
	tcl_thread, 0);
    Tcl_CreateCommand( tcl_thread->interp, "cond", MTtcl_CondCmd,
	tcl_thread, 0);

    Tcl_DStringInit( &thr_argv);
    for ( thr_argc= 0, arg++; arg < argc; thr_argc++, arg++)
	Tcl_DStringAppendElement( &thr_argv, argv[arg]);
    sprintf( argc_str, "%u", thr_argc);
    Tcl_SetVar( tcl_thread->interp, "argc", argc_str, TCL_GLOBAL_ONLY);
    Tcl_SetVar( tcl_thread->interp, "argv", Tcl_DStringValue(&thr_argv),
	TCL_GLOBAL_ONLY);
    Tcl_DStringFree( &thr_argv);
    Tcl_SetVar( tcl_thread->interp, "tcl_interactive", "0", TCL_GLOBAL_ONLY);
    var_value= Tcl_GetVar( interp, "argv0", TCL_GLOBAL_ONLY);
    if ( var_value != NULL)
	Tcl_SetVar( tcl_thread->interp, "argv0", var_value, TCL_GLOBAL_ONLY);
    var_value= Tcl_GetVar( interp, "auto_path", TCL_GLOBAL_ONLY);
    if ( var_value != NULL)
        Tcl_SetVar( tcl_thread->interp, "auto_path",var_value, TCL_GLOBAL_ONLY);

    InitProcSharing( tcl_thread);

    if ( ThrInterp_init != NULL)
        ThrInterp_init( tcl_thread->interp);

    mutex_lock( & tcl_thread->lock);
    sigfillset(&no_sigs);
    thr_sigsetmask(SIG_SETMASK, &no_sigs, &orig_mask);

    rc= thr_create( NULL, 0, Tcl_thread, tcl_thread,
	tcl_thread->flags, &(tcl_thread->tid));
    thr_sigsetmask(SIG_SETMASK, &orig_mask, NULL);

    if ( rc != 0)  {
	errno= rc;
	Tcl_AppendResult( interp, "couldn't create thread: ",
		Tcl_PosixError(interp), NULL);
	return TCL_ERROR;
    }

    sprintf( tcl_thread_id, "thread%u", tcl_thread->tid);
    mutex_lock( &threads_lock);
    entry= Tcl_CreateHashEntry( &threads, tcl_thread_id, &created);
    if ( ! created)  {
	tcl_thread->status= "collision";
	mutex_unlock( &threads_lock);
        mutex_unlock( & tcl_thread->lock);
	interp->result= "couldn't create thread: thread-id collision";
	return TCL_ERROR;
    }
    Tcl_SetHashValue( entry, (ClientData) tcl_thread);
    tcl_thread_count++;
    mutex_unlock( &threads_lock);

    mutex_unlock( & tcl_thread->lock);

    sprintf( interp->result, "thread%u", tcl_thread->tid);
    return TCL_OK;
}

/*
 * Start function for a Tcl thread.  Evaluates the thread's script
 * and sets status when evaluation is complete.
 */

void *
Tcl_thread( void *arg)
{
    Tcl_Thread *tcl_thread= (Tcl_Thread *) arg;
 
    mutex_lock( & tcl_thread->lock);
    mutex_unlock( & tcl_thread->lock);

    if ( tcl_thread->status && strcmp( tcl_thread->status, "collision") == 0)  {
	tcl_thread->result= strdup( "collision");
	return tcl_thread;
    }

    tcl_thread->evaluating= 1;
    (void) thr_setspecific( tcl_thread_key, arg);
    tcl_thread->status= "running";

    tcl_thread->eval_rc= Tcl_GlobalEval( tcl_thread->interp,
							tcl_thread->script);

    tcl_thread->result= strdup( tcl_thread->interp->result);
    switch ( tcl_thread->eval_rc)  {
	case TCL_OK:		tcl_thread->status= "done"; break;
	case TCL_ERROR:		tcl_thread->status= "error"; break;
	case TCL_RETURN:	tcl_thread->status= "return"; break;
	case TCL_BREAK:		tcl_thread->status= "break"; break;
	case TCL_CONTINUE:	tcl_thread->status= "continue"; break;
	default:		tcl_thread->status= "unknown"; break;
    }

    if (tcl_thread->eval_rc == TCL_ERROR) {
	char *argv[2], *command, *errorInfo;
	errorInfo= Tcl_GetVar(tcl_thread->interp, "errorInfo", TCL_GLOBAL_ONLY);
	if ( errorInfo)
	    errorInfo= strdup( errorInfo);
	argv[0]= "threaderror";
	argv[1]= tcl_thread->result;
	command= Tcl_Merge( 2, argv);
	if ( Tcl_GlobalEval( tcl_thread->interp, command) == TCL_ERROR) {
	    Tcl_CmdInfo info;
	    if ( Tcl_GetCommandInfo(tcl_thread->interp, "threaderror", &info)) {
		fprintf( stderr, "threaderror failed to handle error\n");
		fprintf( stderr, "    Original error: %s\n",
			errorInfo ? errorInfo : "NONE");
		fprintf( stderr, "    Error in threaderror: %s\n",
			tcl_thread->interp->result);
	    }
	    else
		fprintf( stderr, "%s\n", errorInfo);
	}
	free( command);
	if ( errorInfo)
	    free( errorInfo);
    }
    tcl_thread->evaluating= 0;

    if ( tcl_thread->flags & THR_DETACHED)  {
	Tcl_HashEntry *entry;
	char thread_id[24];
	sprintf( thread_id, "thread%d", tcl_thread->tid);

	mutex_lock( &threads_lock);
	entry= Tcl_FindHashEntry( &threads, thread_id);
	Tcl_DeleteHashEntry( entry);
	mutex_unlock( &threads_lock);

	ThreadCleanup( tcl_thread);
	return NULL;
    }

    return tcl_thread;
}

int
MTtcl_MessageNotifier(
    Tcl_Interp *interp,
    char *tcl_thread_id,
    int fd[2]
)
{
    Tcl_Thread *tcl_thread;
    if ( GetThread( interp, tcl_thread_id, &tcl_thread) != 0)
	return TCL_ERROR;
    if ( tcl_thread != NULL)  {
	tcl_thread->msg_notifier_pipe[0]= fd[0];
	tcl_thread->msg_notifier_pipe[1]= fd[1];
	mutex_unlock( &tcl_thread->lock);
    }
    return TCL_OK;
}

int
MTtcl_MessageCount()
{
    Tcl_Thread *tcl_thread;
    int rc;
    if ( GetThread( NULL, "self", &tcl_thread) != 0)
	return -1;
    rc= tcl_thread->msg_array_len - tcl_thread->msg_array_next;
    mutex_unlock( &tcl_thread->lock);
    return rc;
}

int
MTtcl_NextMessage(
    Tcl_Interp *interp,
    char **message
)
{
    Tcl_Thread *tcl_thread;
    if ( GetThread( interp, "self", &tcl_thread) != 0)
	return TCL_ERROR;
    return NextMessage( tcl_thread, NULL, NULL, message);
}

int
MTtcl_PostMessage(
    Tcl_Interp *interp,
    char *tcl_thread_id,
    char *message
)
{
    Tcl_Thread *tcl_thread;
    int rc= 0;

    if ( GetThread( interp, tcl_thread_id, &tcl_thread) != 0)
	return TCL_ERROR;

    (void) MessagePost( interp, tcl_thread, message);
    mutex_unlock( &tcl_thread->lock);

    return 0;
}

int
MessagePost(
    Tcl_Interp *interp,
    Tcl_Thread *tcl_thread,
    char *message
)
{
    if ( tcl_thread->messages == NULL)  {
	tcl_thread->messages= ( char **) calloc( 1, sizeof(char*) * 20);
	tcl_thread->msg_array_max= 20;
    }
    else if ( tcl_thread->msg_array_len >= tcl_thread->msg_array_max)  {
	tcl_thread->messages= ( char **) realloc( tcl_thread->messages,
		(tcl_thread->msg_array_max + 20)*sizeof(char*));
	memset( tcl_thread->messages+tcl_thread->msg_array_max, 0,
		20*sizeof(char*));
	tcl_thread->msg_array_max+= 20;
    }

    tcl_thread->messages[tcl_thread->msg_array_len]= strdup( message);
    tcl_thread->msg_array_len++;

    cond_signal( &tcl_thread->msg_cond);

    if ( tcl_thread->msg_notifier_pipe[1] != -1)  {
	struct stat pipe_info;
	fstat( tcl_thread->msg_notifier_pipe[0], &pipe_info);
	if ( pipe_info.st_size == 0)
	    write( tcl_thread->msg_notifier_pipe[1], "B", 1);
    }
    return TCL_OK;
}

int
NextMessageCmd(
    Tcl_Thread *tcl_thread,
    int argc,
    char *argv[]
)
{
    char *seconds= NULL, *nano_seconds;
    char *message_var= NULL;
    char *message;
    int rc= 0;

    if ( tcl_thread == NULL)  {
	tcl_thread->interp->result= "could not identify thread";
	return TCL_ERROR;
    }

    if ( argc < 2 || argc > 5)  {
	Tcl_AppendResult( tcl_thread->interp, "wrong # args: should be \"",
	    argv[0], " nextmsg ?seconds? ?nano-seconds? ?variable?\"", NULL);
	return TCL_ERROR;
    }

    if ( argc >= 3)  {
	seconds= argv[2];
	nano_seconds= "0";
    }
    if ( argc >= 4)
	nano_seconds= argv[3];
    if ( argc == 5)
	message_var= argv[4];

    rc= NextMessage( tcl_thread, seconds, nano_seconds, &message);
    if ( rc == TCL_ERROR)
	return TCL_ERROR;

    if ( message == NULL)  {
	if ( message_var == NULL)  {
	    tcl_thread->interp->result= "time";
	    return TCL_ERROR;
	}
	else  {
	    tcl_thread->interp->result= "0";
	    return TCL_OK;
	}
    }

    if ( message_var != NULL)  {
	Tcl_SetVar( tcl_thread->interp, message_var, message,
		TCL_LEAVE_ERR_MSG);
	tcl_thread->interp->result= "1";
    }
    else
	Tcl_SetResult( tcl_thread->interp, message, TCL_DYNAMIC);

    return TCL_OK;
}

int
NextMessage(
    Tcl_Thread *tcl_thread,
    char *seconds,
    char *nano_seconds,
    char **message
)
{
    int rc= 0;
    *message= NULL;

    while ( tcl_thread->msg_array_next == tcl_thread->msg_array_len &&
		rc == 0)  {
	if ( seconds == NULL)
	    rc= cond_wait( &tcl_thread->msg_cond, &tcl_thread->lock);
	else
	    rc= CondTimedWait( tcl_thread->interp, &tcl_thread->msg_cond,
			&tcl_thread->lock, seconds, nano_seconds, 0);
    }

    if ( rc == ETIME)  {
	mutex_unlock( &tcl_thread->lock);
	return TCL_OK;
    }
    if ( rc != 0)  {
	mutex_unlock( &tcl_thread->lock);
	errno= rc;
	Tcl_AppendResult( tcl_thread->interp, "couldn't wait for message: ",
		Tcl_PosixError(tcl_thread->interp), NULL);
	return TCL_ERROR;
    }

    *message= tcl_thread->messages[tcl_thread->msg_array_next];
    tcl_thread->messages[tcl_thread->msg_array_next]= NULL;
    tcl_thread->msg_array_next++;

    if ( tcl_thread->msg_notifier_pipe[0] != -1)  {
	int foo;
	if ( tcl_thread->msg_array_next == tcl_thread->msg_array_len)
	    read( tcl_thread->msg_notifier_pipe[0], &foo, 1);
    }
    if ( tcl_thread->msg_array_next > 1000)  {
	unsigned int new_size= tcl_thread->msg_array_len -
					tcl_thread->msg_array_next;
	char **new_messages= (char **) calloc( 1, sizeof(char*)*(new_size+40));
	char **old_messages= tcl_thread->messages;
	memcpy( new_messages, tcl_thread->messages+tcl_thread->msg_array_next, 
		sizeof( char *) * new_size);
	tcl_thread->messages= new_messages;
	tcl_thread->msg_array_max= new_size+40;
        tcl_thread->msg_array_len= new_size;
	tcl_thread->msg_array_next= 0;
	free( old_messages);
    }
    mutex_unlock( &tcl_thread->lock);
    return TCL_OK;
}

int
ThreadJoin(
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    char *result_var= NULL;
    void *status;
    Tcl_Thread *join_thr;
    Tcl_Thread *joined_thr;
    thread_t join_tid= 0;
    thread_t joined_tid;
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    int rc;

    if ( argc >= 3 && argc <= 5)  {
	if ( argc > 3)  {
	    if ( strcmp( argv[2], "-result") != 0 )  {
	        Tcl_AppendResult( interp, "bad flag \"", argv[2],
			"\": should be -result", NULL);
	        return TCL_ERROR;
	    }
	    result_var= argv[3];
	}
	if ( argc != 4)  {
	    if ( sscanf( argv[argc-1], "thread%u", &join_tid) != 1)  {
		Tcl_AppendResult( interp, argv[argc-1], " is not a thread",
			NULL);
	        return TCL_ERROR;
	    }
	}
    }
    else if ( argc != 2) {
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" join ?-result variable? ?thread-id?\"", NULL);
	return TCL_ERROR;
    }

    rc= thr_join( join_tid, &joined_tid, &status);
    if ( rc != 0)  {
	errno= rc;
	Tcl_AppendResult( interp, "couldn't join: ", Tcl_PosixError(interp),
		NULL);
	return TCL_ERROR;
    }

    mutex_lock( &threads_lock);

    sprintf( interp->result, "thread%u", joined_tid);
    entry= Tcl_FindHashEntry( &threads, interp->result);

    /* Joined with a non-Tcl thread.  Set result variable to
     * thread return value.
     */
    if ( entry == NULL)  {
	mutex_unlock( &threads_lock);
	if ( result_var != NULL)  {
	    char result_str[30];
	    sprintf( result_str, "%u", (unsigned int) status);
	    Tcl_SetVar( interp, result_var, result_str, 0);
	}
	sprintf( interp->result, "thread%u", joined_tid);
	return TCL_OK;
    }
    Tcl_DeleteHashEntry( entry);
    tcl_thread_count--;
    mutex_unlock( &threads_lock);
 
    joined_thr= (Tcl_Thread *) status;

    if ( result_var != NULL)
	if ( joined_thr->result != NULL)
	    Tcl_SetVar( interp, result_var, joined_thr->result, 0);
	else
	    Tcl_SetVar( interp, result_var, "", 0);

    ThreadCleanup( joined_thr);

    return TCL_OK;
}

int
ThreadCleanup( Tcl_Thread *tcl_thread)
{
    if ( tcl_thread->script)       free( tcl_thread->script);
    if ( tcl_thread->thr_name)     free( tcl_thread->thr_name);
    if ( tcl_thread->result)       free( tcl_thread->result);
    if ( tcl_thread->messages)  {
	int m;
	for ( m= 0; m < tcl_thread->msg_array_max; m++)
	    if ( tcl_thread->messages[m])	free( tcl_thread->messages[m]);
	free( tcl_thread->messages);
    }

    Tcl_DeleteInterp( tcl_thread->interp);
    free( tcl_thread);
    return 0;
}


int
ThreadList(
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    Tcl_DString string;
    char scratch[256];

    if ( argc != 2)  {
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0],
		" list\"", NULL);
	return TCL_ERROR;
    }

/* ** thread list ?<thread-id> ...?
*/
    mutex_lock( &threads_lock);

    Tcl_DStringInit( &string);
    entry= Tcl_FirstHashEntry( &threads, &search);
    while ( entry != NULL)  {
	Tcl_Thread *tcl_thread= (Tcl_Thread *) Tcl_GetHashValue( entry);

	mutex_lock( &tcl_thread->lock);
	Tcl_DStringStartSublist( &string);
	sprintf( scratch, "thread%u", tcl_thread->tid);
	Tcl_DStringAppendElement( &string, scratch);
	if ( tcl_thread->thr_name != NULL)
	    Tcl_DStringAppendElement( &string, tcl_thread->thr_name);
	else
	    Tcl_DStringAppendElement( &string, "");
	Tcl_DStringAppendElement( &string, tcl_thread->status);
	if ( tcl_thread->result != NULL)
	    Tcl_DStringAppendElement( &string, tcl_thread->result);
	else
	    Tcl_DStringAppendElement( &string, "");
	Tcl_DStringEndSublist( &string);
	mutex_unlock( &tcl_thread->lock);
	entry= Tcl_NextHashEntry( &search);
    }

    mutex_unlock( &threads_lock);
    Tcl_DStringResult( interp, &string);
    return TCL_OK;
}

int
MTtcl_ShareCmd (
    ClientData  clientData,
    Tcl_Interp *interp,
    int         argc,
    char      **argv
)
{
    static char *options= "?-global? ?var var ...?";
    Tcl_HashEntry *entry;
    int arg, created, rc, global_rc, global= 0, new_array= 0;
    char *var_value;
    ShareVar *share_var;

    arg= 1;
    if ( argc == 1)  {
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0], " ",
		options, "\"", NULL);
	return TCL_ERROR;
    }

    if ( argv[1][0] == '-')  {
	if ( strcmp( "-global", argv[1])  == 0)
	    global++;
	else  {
	    Tcl_AppendResult( interp, "bad flag \"", argv[1],
		"\": should be -global", NULL);
	    return TCL_ERROR;
	}
	arg= 2;
    }

    if ( argc <= arg)  {
	Tcl_AppendResult( interp, "wrong # args: should be \"", argv[0], " ",
		options, "\"", NULL);
	return TCL_ERROR;
    }

    mutex_lock( &share_lock);
    for ( ; arg < argc; arg++)  {
	share_var= (ShareVar *) Tcl_VarTraceInfo( interp, argv[arg], 0,
		ShareLinkTraceProc, NULL);
	if ( share_var != NULL)
	    continue;
	entry= Tcl_CreateHashEntry( &share_vars, argv[arg], &created);
	if ( created)  {
	    share_var= createShareVar( interp, argv[arg], entry);
	    if ( share_var->array)
		new_array= 1;
	    Tcl_ResetResult( interp);
	}
	else
	    share_var= (ShareVar *) Tcl_GetHashValue( entry);

	rc= MTtcl_ShareLinkVar(interp, argv[arg], share_var, new_array, 0);
	if ( rc == TCL_OK && global)
	    global_rc= MTtcl_ShareLinkVar(interp, argv[arg], share_var,
				new_array, TCL_GLOBAL_ONLY);
	if ( rc == TCL_ERROR || (global && global_rc == TCL_ERROR))  {
	    if ( rc == TCL_OK)
		MTtcl_ShareUnlinkVar( interp, argv[arg]);
	    if ( created)  {
		Tcl_DeleteHashEntry( entry);
		cleanupShareVar( share_var);
	    }
	    mutex_unlock( &share_lock);
	    Tcl_AppendResult( interp, "failed to share \"", argv[arg], "\"",
		NULL);
	    return TCL_ERROR;
	}
	share_var->ref += 1 + global;
    }
    mutex_unlock( &share_lock);
    return TCL_OK;
}

static ShareVar *
createShareVar( Tcl_Interp *interp, char *var_name, Tcl_HashEntry *entry)
{
    ShareVar *share_var;
    char *var_value;

    share_var= (ShareVar *) calloc( 1, sizeof( ShareVar));
    share_var->ref= 0;
    Tcl_SetHashValue( entry, share_var);

    var_value= Tcl_GetVar( interp, var_name, 0);
    if ( var_value == NULL)  {
	char *vargv[4], *command;
	int vargc= 3;
	vargv[0]= "array";
	vargv[1]= "exists";
	vargv[2]= var_name;
	vargv[3]= NULL;
	command= Tcl_Merge( 3, vargv);
	if ( Tcl_GlobalEval( interp, command) != TCL_OK ||
		strcmp( interp->result, "1") != 0)  {
	    free( command);
	    return share_var;
	}
	free( command);
	vargv[1]= "names";
	command= Tcl_Merge( 3, vargv);
	if ( Tcl_GlobalEval( interp, command) == TCL_OK)  {
	    char **array_argv, *list= strdup( interp->result), *value;
	    int array_argc, arg, created;
	    Tcl_HashEntry *entry;
	    share_var->u.array= (Tcl_HashTable *) malloc(sizeof(Tcl_HashTable));
	    share_var->array= 1;
	    Tcl_InitHashTable( share_var->u.array, TCL_STRING_KEYS);
	    Tcl_SplitList(interp, list, &array_argc, &array_argv);
	    for ( arg= 0; arg < array_argc; arg++)  {
		value= Tcl_GetVar2( interp, var_name, array_argv[arg], 0);
		if ( value == NULL)
		    continue;
		entry= Tcl_CreateHashEntry( share_var->u.array, array_argv[arg],
			&created);
		Tcl_SetHashValue( entry, strdup( value));
	    }
	}
	free( command);
    }
    else
	share_var->u.value= strdup( var_value);

    return share_var;
}

static void
cleanupShareVar( ShareVar *share_var)
{
    if ( share_var->array && share_var->u.array != NULL)  {
	Tcl_HashSearch search;
	Tcl_HashEntry *entry;
	entry= Tcl_FirstHashEntry( share_var->u.array, &search);
	while ( entry != NULL)  {
	    free( Tcl_GetHashValue( entry));
	    entry= Tcl_NextHashEntry( &search);
	}
	Tcl_DeleteHashTable( share_var->u.array);
    }
    if ( share_var->u.void_ptr)
	free( share_var->u.void_ptr);
}

static void
releaseShareVar( ShareVar *share_var, char *name)
{
    share_var->ref--;
    if ( share_var->ref == 0)  {
	Tcl_HashEntry *entry;
	entry= Tcl_FindHashEntry( &share_vars, name);
	if ( entry != NULL)
	    Tcl_DeleteHashEntry( entry);
	cleanupShareVar( share_var);
    }
}

int
MTtcl_ShareLinkVar(
    Tcl_Interp *interp,	
    char *name,
    ShareVar *share_var,
    int new_array,
    unsigned int flags		/* TCL_GLOBAL_ONLY or 0 */
)
{
    if ( share_var->array && ! new_array)  {
	Tcl_HashEntry *entry;
	Tcl_HashSearch search;

	entry= Tcl_FirstHashEntry( share_var->u.array, &search);
	while ( entry != NULL)  {
	    char *key= Tcl_GetHashKey( share_var->u.array, entry);
	    Tcl_SetVar2( interp, name, key, "", flags);
	    entry= Tcl_NextHashEntry( &search);
	}
    }
    return Tcl_TraceVar(interp, name, TCL_TRACE_READS|
	    TCL_TRACE_WRITES|TCL_TRACE_UNSETS|flags, ShareLinkTraceProc,
	    (ClientData) share_var);
}

void
MTtcl_ShareUnlinkVar(
    Tcl_Interp *interp,
    char *name
)
{
    ShareVar *share_var;

    share_var = (ShareVar *) Tcl_VarTraceInfo(interp, name, 0,
	    ShareLinkTraceProc, (ClientData) NULL);
    if (share_var == NULL)
	return;

    Tcl_UntraceVar(interp, name,
	    TCL_TRACE_READS|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
	    ShareLinkTraceProc, (ClientData) share_var);
}

static char *
ShareLinkTraceProc(
    ClientData clientData,	/* Contains information about the link. */
    Tcl_Interp *interp,		/* Interpreter containing Tcl variable. */
    char *name1,		/* First part of variable name. */
    char *name2,		/* Second part of variable name. */
    int flags			/* Miscellaneous additional information. */
)
{
    ShareVar *share_var = (ShareVar *) clientData;
    Tcl_HashEntry *entry;
    char *new_value;

    if (flags & TCL_TRACE_UNSETS) {
	if (flags & TCL_TRACE_DESTROYED) {
	    mutex_lock( &share_lock);
	    releaseShareVar( share_var, name1);
	    mutex_unlock( &share_lock);
	}
	if (flags & TCL_INTERP_DESTROYED) {
	}
	return NULL;
    }

    if (flags & TCL_TRACE_READS) {
	if ( share_var->u.void_ptr == NULL || (share_var->array &&
		name2 == NULL)) 
	    return "no such variable";
	if ( name2 != NULL && ! share_var->array)
	    return "variable isn't array";
	if ( name2 == NULL)
	    Tcl_SetVar2(interp, name1, name2, share_var->u.value, 0);
	else  {
	    entry= Tcl_FindHashEntry( share_var->u.array, name2);
	    if ( entry == NULL)
		return "no such element in array";
	    Tcl_SetVar2(interp, name1, name2,
		(char *) Tcl_GetHashValue( entry), 0);
	}
	return NULL;
    }

    new_value = Tcl_GetVar2(interp, name1, name2,
		flags & (TCL_GLOBAL_ONLY|TCL_LEAVE_ERR_MSG));
    if (new_value == NULL) {
	/*
	 * This shouldn't ever happen.
	 */
	return "internal error: linked variable couldn't be read";
    }
    if ( share_var->u.void_ptr == NULL && name2 != NULL)  {
	share_var->array= 1;
	share_var->u.array= (Tcl_HashTable *)malloc( sizeof(Tcl_HashTable));
	Tcl_InitHashTable( share_var->u.array, TCL_STRING_KEYS);
    }
    entry= NULL;
    if ( share_var->u.void_ptr != NULL)  {
	if ( name2 == NULL && share_var->array)
	    return "no such variable";
	else if ( name2 != NULL && ! share_var->array)
	    return "variable isn't array";

	if ( name2 == NULL)
	    free( share_var->u.value);
	else  {
	    entry= Tcl_FindHashEntry( share_var->u.array, name2);
	    if ( entry)
		free( (void *) Tcl_GetHashValue( entry));
	}
    }
    if ( name2 == NULL)
	share_var->u.value= strdup( new_value);
    else  {
	int created;
	if ( entry == NULL)
	    entry= Tcl_CreateHashEntry( share_var->u.array, name2, &created);
	Tcl_SetHashValue( entry, strdup( new_value));
    }
    return NULL;
}

static void
printShareTable( char *prefix1, char *prefix2)
{
    Tcl_HashEntry *entry;
    Tcl_HashSearch search;
    entry= Tcl_FirstHashEntry( &share_vars, &search);
    while ( entry != NULL)  {
	ShareVar *share_var= (ShareVar *) Tcl_GetHashValue( entry);
	fprintf( stderr, "%s%s %12s %2d %s\n", prefix1, prefix2,
		Tcl_GetHashKey( &share_vars, entry), share_var->ref,
		share_var->u.value);
	entry= Tcl_NextHashEntry( &search);
    }
}

/* Return Tcl_Thread structure for a thread-id.  The thread-id is
 * "thread%u" where %u is the thread ID assigned by thr_create().
 * The thread-id may also be "self" to refer to the calling thread.
 * Upon successful completion, a pointer to the matching Tcl_Thread
 * is returned and threads_lock is locked.  Upon failure, NULL is
 * returned and threads_lock is not locked.
 */
static int
GetThread( Tcl_Interp *interp, char *tcl_thread_id, Tcl_Thread **tcl_thread)
{
    Tcl_HashEntry *entry;

    if ( tcl_thread_id[0] == 's' && strcmp( "self", tcl_thread_id) == 0)  {
	thr_getspecific( tcl_thread_key, (void **)tcl_thread);
	if ( *tcl_thread == NULL)  {
	    if ( interp != NULL)
		interp->result= "operation not available on main thread";
	    return MTTCL_INVALID_THREAD;
	}
	mutex_lock( &((*tcl_thread)->lock));
	return 0;
    }

    mutex_lock( &threads_lock);
    entry= Tcl_FindHashEntry( &threads, tcl_thread_id);
    if ( entry == NULL)  {
	mutex_unlock( &threads_lock);
	if ( interp != NULL)
	    Tcl_AppendResult( interp, tcl_thread_id, " is not a thread", NULL);
	return MTTCL_UNKNOWN_THREAD;
    }
    *tcl_thread= Tcl_GetHashValue( entry);
    mutex_lock( &((*tcl_thread)->lock));
    mutex_unlock( &threads_lock);
    return 0;
}

