# TclProp.tcl --
#
#	This file contains the functions needed to define formulas on
#       Tcl traceable objects, and a simple formula propagation engine.
#
# Copyright (c) 1994-1996 University of Minnesota.
#
# RCS:  $Id: tclProp.tcl,v 1.53 1996/12/02 22:12:53 safonov Exp safonov $


# TP_formulaID is the number corresponding to the highest-numbered formula
# A formula ID is returned by TclProp_formula and can be used to relax,
# enforce, or delete the formula later.

global TP_formulaID
set TP_formulaID 0

# TP_tracedVars is the array of variables that have already been traced

global TP_tracedVars

# TP_propTab is the array (table) of propagations, indexed by source variable
# each entry is a list of triplets -- formulaID (negative for inactive),
# destination variable, and code.

global TP_propTab

# TP_oldValues is an array of prior values for stopping loops

global TP_oldValues

# TP_curProps is an array of variables already changed in a given propagation
# it is also used for stopping loops

global TP_curProps 

# TP_readCmds is an array of read code for makeVar variables
# TP_readCmdNo is the index of the current element of that array

global TP_readCmds TP_readCmdNo
set TP_readCmdNo 0

global TP_dirtySrcList 
set TP_dirtySrcList {}
global TP_evalMode 
set TP_evalMode "eager"; # "deferred"

#  TP_formula -- takes a destination variable, a list of source
#  variables, and a formula expression, and it implements a formula to
#  automatically update the destination whenever any source changes.
#  Note:  this is also called from TP_trigger to handle triggers (the
#  des is empty in that case).  Note that the formula is activated once
#  when defined to "make things consistent" but if the formula cannot execute
#  (e.g., if some variables are not yet defined) the errors are ignored.

#  For example, a typical formula would be:
#      TP_formula a {b c} {expr b + c}

proc TP_formula {des srcs code {activate 1} } {

  global TP_formulaID
  global TP_tracedVars
  global TP_propTab
  global TP_oldValues
  global TP_curProps
  global TP_traceableType
  
  #  assign the formula ID
  incr TP_formulaID

  # Parse traceable reference $des into tag, object type, and 
  # type-specific parameters.
  if {[catch {set desRef [TP_parseTracedRef $des]} err]} {
    puts "TP_formula: $err"
    return
  }
  set desTag      [lindex $desRef 0]
  set desObjType  [lindex $desRef 1]
  set desTypeArgs [lindex $desRef 2]
  # remove tag from $des if it was there
  set des [list $desObjType $desTypeArgs]

  if {[string compare $des ""] != 0} {
    # set up variable doit for regexp substitution 
    set doit {uplevel \#0 {SETATTR}}

    # substitute SETATTR with attribute write code for destination object type
    regsub -all SETATTR $doit \
      [eval $TP_traceableType($desObjType,write) $desTypeArgs {\[CODE\]}] doit
  } else {
    # $des is empty (i.e., TP_formula is called from TP_trigger)
    set doit {uplevel \#0 {CODE}}
  }

  # substitute the actual code to execute that returns the value
  regsub CODE $doit $code doit

  # this does not make sense: $des on the r.h.s. will have the new value,
  # not the old one.
#   if {[string compare $desTag ""] != 0} {
#     # substitute destination tag with actual attribute read code
#     regsub -all $desTag $code \
#       \[[eval $TP_traceableType($desObjType,read) $desTypeArgs]\] code
#   }

  # the list srcList will store, for each source traceable reference,
  # triples of the form <srcTag, objType, typeArgs>
  set srcList {}
  
  #   Foreach source reference: substitute its tag if present, 
  # set trace if needed; formula table is modified later.
  #   Extra loop is needed so that all the tags are properly
  # substituted in $code before $code is added to the formula
  # table.
  foreach src $srcs {
    # Parse traceable reference $src into tag, object type, and 
    # type-specific parameters.

    if {[catch {set srcRef [TP_parseTracedRef $src]} err]} {
      puts "TP_formula: $err"
      return
    }

    set tag      [lindex $srcRef 0]
    set objType  [lindex $srcRef 1]
    set typeArgs [lindex $srcRef 2]

    # check that trace function is defined for this type
    if {! [info exist TP_traceableType($objType,traceCreate)] || \
	  [string compare TP_traceableType($objType,traceCreate) ""] == 0} {
      puts "TP_formula: trace function not defined for $src of type $objType"
      # skip the current source traceable reference
      continue
    }

    if {[string compare $tag ""] != 0} {
      # substitute attribute read code from table for the current source tag
      # $code is put in the formula table, and $doit is evaluated 
      # in TP_formula to set things up.

      # Create regexp for parsing the tag: dollar sign, tagname, the rest
      # (rest is parenthesised).
      set tagrexp {[$]$tag([^0-9A-Za-z_]*)}

      # In `tagrexp', substitute string `$tag' with the actual value 
      # of variable `tag'.
      regsub {[$]tag} $tagrexp "$tag" tagrexp

      # In variable `doit', substitute all occurrences of `tagrexp' value
      # with the Tcl code that reads the object attribute value. This Tcl
      # code is returned by the type-specific procedure whose name is stored
      # in `TP_traceableType($objType,read)'.
      set rc [regsub -all $tagrexp $doit \
              "\[[eval $TP_traceableType($objType,read) $typeArgs]\]\\1" doit]
      if {!$rc} {
	puts "*** ERROR replacing tag $tag with actual attribute read code"
	puts "*** contact the developer"
      }

      # In variable `code', substitute all occurrences of `tagrexp' value
      # with the Tcl code that reads the object attribute value. This Tcl
      # code is returned by the type-specific procedure whose name is stored
      # in `TP_traceableType($objType,read)'
      set rc [regsub -all $tagrexp $code \
              "\[[eval $TP_traceableType($objType,read) $typeArgs]\]\\1" code]
      if {!$rc} {
	puts "*** ERROR replacing tag $tag with actual attribute read code"
	puts "*** contact the developer"
      }

      # TP_traceableType($objType,read) contains the NAME of the
      # type-specific read function. When evaluated with the appropriate
      # parameters, this function returns the actual Tcl code that
      # reads and returns the value of the object's specified attribute.
      set readCode [eval $TP_traceableType($objType,read) $typeArgs]
      regsub "READCODE" {[READCODE]\1} $readCode readrexp
    }

    # save source type-specific arguments; 
    # TP_propTab is indexed by those.
    lappend srcList $typeArgs

    #  If trace was not set up yet on the current source object/attribute,
    #  do it now.
    if {! [info exist TP_tracedVars($typeArgs) ] } {
      
      ### Save the type of the source object/attribute
      global TP_objTypeTab
      # Quotes are needed since `typeArgs' can be a list
      set TP_objTypeTab($typeArgs) $objType

      set TP_tracedVars($typeArgs) 1

      if {[info exist TP_traceableType($objType,global)] && \
	    [string compare $TP_traceableType($objType,global) ""] != 0} {
	# make object global
	eval [eval $TP_traceableType($objType,global) $typeArgs]
      }

      # the inner eval splits the $typeArgs list into individual items 
      # for passing them to the type-specific access procedure, while 
      # the outer eval runs the access procedure whose name is stored in 
      # $TP_traceableType($objType,*). This latter returns the actual Tcl
      # code to check existence or read the object's attribute.
      set objExist [eval [eval $TP_traceableType($objType,exist) $typeArgs]]

      if {$objExist} {
	set TP_oldValues($typeArgs) \
	  [eval [eval $TP_traceableType($objType,read) $typeArgs]]
      } else {
	set TP_oldValues($typeArgs) TP_dummyValue
      }
      
      set TP_curProps($typeArgs) 0

      # set up the trace
      eval $TP_traceableType($objType,traceCreate) \
	$typeArgs w "\[list TP_genPropChange $objType\]"
    }
  }

  # create entries in the formula table
  foreach typeArgs $srcList {
    # Destination format in formula entry: 
    # list of (object type, list of (type-specific parameters)).
    set formula_entry [list $TP_formulaID $des $code]
    lappend TP_propTab($typeArgs) $formula_entry
  }

  # evaluate the formula once, to get things set up
  if {$activate} {
    if {[catch {eval $doit} errmsg]} {
      puts $errmsg
    }
  }

  #  return the formula ID
  return $TP_formulaID
}

# Parse $taggedTracedRef into tag, object type, and type-specific parameters.
# 
# $taggedTracedRef is a traceable reference in the form
#     tagged object attribute reference
# {tag {objType nameSpace1 .. nameSpaceN objName attrName}}
#     or untagged object attribute reference
# {objType nameSpace1 .. nameSpaceN objName attrName}
#     or tagged Tcl variable
# {tag varName}
#     or untagged Tcl variable
# varName
#     or
# "" (empty)
#
# Returns the list, consisting of 3 elements:
# 1. reference tag. May be empty.
# 2. object type. Never empty; for variables, type "variable" is returned.
# 3. list of type specific parameters. For widgets and canvas items, it
#    consists of 3 elements: namespace, object name, and attribute name.
#    For widgets, namespace is empty, object name is the widget name, and
#    attribute name is the widget attribute name. For canvas items, the 
#    namespace is the name of the canvas widget, and the object name is 
#    the traced item's tag or id.
proc TP_parseTracedRef {taggedTracedRef} {
    
  set taggedTracedRefLen [llength $taggedTracedRef]
  if {$taggedTracedRefLen > 1} {
    # $taggedTracedRef is a tagged traceable reference in the form
    # {tag {objType nameSpace1 .. nameSpaceN objName attrName}}
    #   or
    #      {objType nameSpace1 .. nameSpaceN objName attrName}
    
    set tracedRef [lindex $taggedTracedRef 1]
    if {[llength $tracedRef] > 1} {
      # parse the tag
      set tag [lindex $taggedTracedRef 0]
    } else {
      # no tag
      set tag ""
      set tracedRef $taggedTracedRef
    }
    
    # $tracedRef now contains the list
    # {objType nameSpace1 .. nameSpaceN objName attrName},
    # where nameSpaces are optional.
    # Parse objType (car) and typeArgs (cdr) from the list.
    if {[llength $tracedRef] < 3} {
      error "$tracedRef is not a valid traceable reference"
    }
    set objType  [lindex $tracedRef 0]
    set typeArgs [lrange $tracedRef 1 end]
    
  } elseif {$taggedTracedRefLen == 1} {
    # simple variable
    set tag ""
    set objType "variable"
    set typeArgs $taggedTracedRef
  } else {
    # empty
    set tag ""
    set objType ""
    set typeArgs ""
  }

  return [list $tag $objType $typeArgs]
}

#  TP_varname -- strips off array notation, leaving scalar variables alone
#  but returning the variable name for array variables

proc TP_varname {name} {
   scan $name {%[^(]} result
   return $result
}

#  TP_genPropChange -- the actual trace callback that handles propagation of 
#  changes to object attributes and variables. It is called whenever a source
#  object attribute, or source variable is changed. It then checks whether:  
#  a) the variable/object attribute value really changed, and 
#  b) we are in a loop.  
#  In either case, it returns, otherwise it evaluates each formula 
#  in which that variable/object attribute is a source.

#  Note:  This version does not print error messages -- it just ignores 
#  errors in formula code and goes on its merry way!  That should be fixed
#  in a later version if it can be handled efficiently.

proc TP_genPropChange {objType args} {
  global TP_oldValues TP_propTab TP_curProps
  global TP_traceableType

  # get operation op as the last elements of args
  set op [lindex $args end]
  set typeArgs [lrange $args 0 [expr [llength $args]-2]]

  # this is a hack, needed only because we are using the 
  # same C trace function for all objects (for now).
  # Canvases are "namespaces", while widgets have an empty ({})
  # namespace. This empty namespace is what gets discarded below.
  if {[string compare [lindex $typeArgs 0] ""] == 0} {
    set typeArgs [lrange $typeArgs 1 end]
  }

  # Special handling for traced Tcl variables and array elements:
  # generate proper variable or array element name.
  if {[string compare $objType "variable"] == 0} {
    set index [lindex $typeArgs 1]
    if {$index != ""} {
      set typeArgs [append [lindex $typeArgs 0] "(" $index ")"]
    } else {
      set typeArgs [lindex $typeArgs 0]
    }
  }

  global TP_dirtySrcList TP_evalMode 
  if {![string compare $TP_evalMode "deferred"]} {
    # puts "deferring $args"
    if {[lsearch $TP_dirtySrcList $args] == -1} {
      lappend TP_dirtySrcList $args
    }
    return
  }

  if {$TP_curProps($typeArgs)} {
    return
  } else {
    set TP_curProps($typeArgs) 1
  }

  # make object global
  if {[info exist TP_traceableType($objType,global)] && \
	[string compare $TP_traceableType($objType,global) ""] != 0} {
    eval [eval $TP_traceableType($objType,global) $typeArgs]
  }

  # read the (new) attribute value
  set newVal [eval [eval $TP_traceableType($objType,read) $typeArgs]]

  if {$TP_oldValues($typeArgs) != $newVal} {

    set TP_oldValues($typeArgs) $newVal

    foreach prop $TP_propTab($typeArgs) {
      set formID [TP_getPropId $prop]
      set des [TP_getPropObjDes $prop]
      set code [TP_getPropCode $prop]
      if {$formID > 0} {
	# parse destination into object type and type-specific arguments
	set desObjType [TP_getPropId $des]
	eval set desTypeArgs [lrange $des 1 end]

	if {$des != ""} {
	  # make destination traced object or variable global
	  if {[string compare $TP_traceableType($desObjType,global) ""] != 0} {
	    eval [eval $TP_traceableType($desObjType,global) $desTypeArgs]
	  }
	  # run the trace code and set the traced object attribute to the new value
	  if {[catch {set desVal [uplevel \#0 $code]}] == 0} {
	    eval [eval $TP_traceableType($desObjType,write) $desTypeArgs $desVal]
	  }
	} else {
	  # this is a trigger (no destination); run the trace code
	  catch {uplevel \#0 $code}
	}
      }
    }
  }
  set TP_curProps($typeArgs) 0
}


#  TP_trigger is a trigger interface (similar to trace) that simply indicates
#  that certain code should be executed whenever a variable changes values.
#  The syntax is simple:
#	TP_trigger a {puts "A changed!"}
#  Note that the trigger executes once right away when defined (unless it
#  cannot due to errors).

proc TP_trigger {srcs code} {
    TP_formula "" $srcs $code
}

#  TP_makeVar -- this function creates a synthetic variable that acts as an 
#  interface between the other formula functions and a non-traceable
#  attribute.  TP_makeVar takes a variable name, and optional -read and -write
#  code and and optional read frequency (-rf) in milliseconds.  For example:
#
#      TP_makeVar a-morethan-b -read {expr $a > $b} -rf 500
#
#  would set up a global variable a-morethan-b which polls every 1/2 second 
#  to update itself to the current condition of whether a is greater than 
#  b.  (Not a useful example.)  Writes are purely triggers, whereas reads are
#  implemented as after loops.

proc TP_makeVar {var args} {
  set read_freq 1000
  foreach arg $args {
    switch -- $arg {
      -read  { set next read }
      -write { set next write }
      -rf    { set next rf }
      default {
	switch $next {
	  read  { set read_action $arg }
	  write { TP_trigger $var $arg }
	  rf    { set read_freq $arg }
	}
      }
    }
  }
  if {[info exists read_action]} {
    global TP_readCmds TP_readCmdNo
    incr TP_readCmdNo
    set cmdString {uplevel \#0 {set VAR [ RA ] ; after RF $TP_readCmds(NUM)}}
    regsub VAR $cmdString $var cmdString
    regsub RA $cmdString $read_action cmdString
    regsub RF $cmdString $read_freq cmdString
    regsub NUM $cmdString $TP_readCmdNo cmdString
    set TP_readCmds($TP_readCmdNo) $cmdString
    eval $cmdString
  }
}

#  This is the quote procedure that makes it easier to write if statements
#  that return expression values.

proc quote {x} {return $x}

#  TP_enableFormulasById -- enables all disabled TclProp formulas with
#  the specified formula ID.
#  Formulas are disabled by making formulaID element negative.

proc TP_enableFormulasById {id} {

    global TP_propTab

    if {$id <= 0} {
	return
    }

    foreach srcObjDesc [array names TP_propTab] {
	set propNum 0
	foreach prop $TP_propTab($srcObjDesc) {
	    set formID [TP_getPropId $prop]
	    if {$id == -$formID} {
		set TP_propTab($srcObjDesc) \
		    [lreplace $TP_propTab($srcObjDesc) $propNum $propNum \
			 [list \
			    [expr -$formID] \
			    [TP_getPropObjDesc $prop] \
			    [TP_getPropCode $prop] ] ]
	    }
	    incr propNum
	}
    }
}

#  TP_disableFormulasByDest -- disables all TclProp formulas where the 
#  specified variable template matches the destination variable. 
#  Formulas are disabled by making formulaID element negative.
#  Variable names are compared using glob-style matching.

proc TP_disableFormulasByDest {des} {

    global TP_propTab

    foreach srcObjDesc [array names TP_propTab] {
	set propNum 0
	foreach prop $TP_propTab($srcObjDesc) {
	    set curdes [TP_getPropObjDes $prop]
	    if {[string match $des $curdes]} {
		set formID [TP_getPropId $prop]
		if {$formID > 0} {
		    set TP_propTab($srcObjDesc) \
			[lreplace $TP_propTab($srcObjDesc) $propNum $propNum \
			     [list -$formID $curdes [TP_getPropCode $prop] ] ]
		}
	    }
	    incr propNum
	}
    }
}

#  TP_enableFormulasByDest -- enables all disabled TclProp formulas 
#  where the specified variable template matches the destination 
#  variable. Formulas are enabled by making formulaID element positive 
#  again. A formula being enabled is forcibly evaluated once to bring 
#  things back in sync.
#  Variable names are compared using glob-style matching.

proc TP_enableFormulasByDest {des} {

    global TP_propTab
   
    foreach srcObjDesc [array names TP_propTab] {
	set propNum 0
	foreach prop $TP_propTab($srcObjDesc) {
	    set curdes [TP_getPropObjDes $prop]
	    if {[string match $des $curdes]} {
		set formID [TP_getPropId $prop]
		if {$formID < 0} {
		    set TP_propTab($srcObjDesc) \
			[lreplace $TP_propTab($srcObjDesc) $propNum $propNum \
			     [list [expr -$formID] $curdes [TP_getPropCode $prop] ] ]
		    # evaluate code once, to get things set up again
		    set code [TP_getPropCode $prop]
		    global [TP_varname $curdes]
		    catch {set $curdes [uplevel \#0 $code]}
		    catch {eval $doit}
		}
	    }
	    incr propNum
	}
    }
}

#  TP_deleteFormulasById -- deletes all TclProp formulas where the 
#  specified formula ID matches.
#  If all formulas for a given source are deleted, the element in array
#  `TP_propTab' corresponding to this source is also unset.

proc TP_deleteFormulasById {id} {

    global TP_propTab

    foreach srcObjDesc [array names TP_propTab] {
	# initialize new formula list for current source variable
	set newPropList ""
	foreach prop $TP_propTab($srcObjDesc) {
	    if {$id !=  [TP_getPropId $prop]} {
		lappend newPropList $prop
	    }
	}

	if {[llength $newPropList] == 0} {
	    # all formulas for this source deleted.
            TP_destroySrcProps $srcObjDesc
	} else {
	    set TP_propTab($srcObjDesc) $newPropList
	}
    }
}

#  TP_deleteFormulasByDest -- deletes all TclProp formulas where the 
#  specified template matches the formula destination. 
#  If all formulas for a given source are deleted, the element array
#  `TP_propTab' corresponding to this source is also unset.
#  Destination names are compared using glob-style matching.

proc TP_deleteFormulasByDest {des} {

    global TP_propTab

    foreach srcObjDesc [array names TP_propTab] {
	# initialize new formula list for current source
	set newPropList ""
	foreach prop $TP_propTab($srcObjDesc) {
  	    if {![string match $des [TP_getPropObjName $prop]} {
		lappend newPropList $prop
	    }
	}

	if {[llength $newPropList] == 0} {
	    # all formulas for this source deleted;
            TP_destroySrcProps $srcObjDesc
	} else {
	    set TP_propTab($srcObjDesc) $newPropList
	}
    }
}

#  TP_deleteFormulasBySrc -- deletes all TclProp formulas where the 
#  specified source template matches ANY ONE OF THE the formula sources. 
#  Source names are compared using glob-style matching.

proc TP_deleteFormulasBySrc {src} {
    global TP_propTab

    foreach srcObjDesc [array names TP_propTab] {
	if {[string match $src $srcObjDesc]} {
	    TP_destroySrcProps $$srcObjDesc
	}
    }
}

#  TP_destroySrcProps -- deletes all TclProp state information associated
#  with the specified source object/attribute.
#  This procedure is called when the last formula that has `srcObjDesc'
#  as source gets deleted.
#
# INTERNAL TclProp

proc TP_destroySrcProps {srcObjDesc} {

    global TP_propTab TP_tracedVars TP_oldValues TP_curProps
    global TP_dirtySrcList
    global TP_objTypeTab;  # type table for traced object/attribute pairs
    global TP_traceableType

    # clear the `srcObjDesc' slot in various state arrays.
    unset TP_propTab($srcObjDesc)
    unset TP_tracedVars($srcObjDesc)
    unset TP_oldValues($srcObjDesc)
    catch {unset TP_curProps($srcObjDesc)}
    catch {unset TP_dirtySrcList($srcObjDesc)}

    # set up an alias for srcObjDesc, for consistency with the
    # rest of the code that uses `typeArgs' as the array index name.
    set typeArgs $srcObjDesc

    # Retrieve the type of the source object/attribute
    set objType $TP_objTypeTab($typeArgs) 

    # Make object global.
    if {[info exist TP_traceableType($objType,global)] && \
	[string compare $TP_traceableType($objType,global) ""] != 0} {
      eval [eval $TP_traceableType($objType,global) $typeArgs]
    }

    # delete the trace on object attribute
    eval $TP_traceableType($objType,traceDelete) \
      $typeArgs w "\[list TP_genPropChange $objType\]"
}

###########################################################################
#   Get various fields and subfields of the a TclProp prop "object".
# These functions rely on the internal representation of formulas
# in TclProp and can change in the future, though the API should remain
# the same.
#
# INTERNAL TclProp
###########################################################################

proc TP_getPropId {prop} {

  if {[llength $prop] != 3} {
    puts "TclProp: malformed prop: $prop"
    puts "*** contact the developer"
    return 0
  }

  return [lindex $prop 0]
}

proc TP_getPropObjDesc {prop} {

  if {[llength $prop] != 3} {
    puts "TclProp: malformed prop: $prop"
    puts "*** contact the developer"
    return ""
  }

  set objDesc [lindex $prop 1]
  if {[llength $objDesc] != 2} {
    puts "TclProp: malformed destination object description: $objDesc"
    puts "*** contact the developer"
    return ""
  }

  return $objDesc 
}

proc TP_getPropObjType {prop} {

  set objDesc [TP_getPropObjDesc $prop]
  if {$objDesc == ""} {
    return ""
  }

  return [lindex $objDesc 0]
}

proc TP_getPropObjName {prop} {

  set objDesc [TP_getPropObjDesc $prop]
  if {$objDesc == ""} {
    return ""
  }

  return [lindex $objDesc 1]
}

proc TP_getPropCode {prop} {

  if {[llength $prop] != 3} {
    puts "TclProp: malformed prop: $prop"
    puts "*** contact the developer"
    return 0
  }
  
  return [lindex $prop 2]
}


###########################################################################
# TclProp Initialization
###########################################################################

catch {unset TP_propTab TP_tracedVars TP_curProps TP_oldValues TP_traceableType TP_objTypeTab}

# Initialize slots in the TP_traceableType array for each traceable type


# Tcl variables 

set TP_traceableType(variable,traceCreate)  "trace variable"
set TP_traceableType(variable,traceInfo)    "trace vinfo"
set TP_traceableType(variable,traceDelete)  "trace vdelete"

set TP_traceableType(variable,exist)        "varExistFunc"
set TP_traceableType(variable,global)       "varGlobalFunc"
set TP_traceableType(variable,read)         "varReadFunc"
set TP_traceableType(variable,write)        "varWriteFunc"

proc varExistFunc {varName} {
  quote "string compare \"\[info global $varName\]\" \"\""
}

proc varGlobalFunc {varName} {
  uplevel 1 global [TP_varname $varName]
}

proc varReadFunc {varName} {
  quote "set $varName"
}

proc varWriteFunc {varName value} {
  quote "set $varName $value"
}

#set TP_traceableType(variable,propFunc)     "TP_varPropChange"

# Tk widgets

set TP_traceableType(widget,traceCreate)    "trace widget"
set TP_traceableType(widget,traceInfo)      "trace winfo"
set TP_traceableType(widget,traceDelete)    "trace wdelete"

set TP_traceableType(widget,exist)          "widgExistFunc"
set TP_traceableType(widget,global)         ""
set TP_traceableType(widget,read)           "widgReadFunc"
set TP_traceableType(widget,write)          "widgWriteFunc"

proc widgExistFunc {widgName attrName} {
  quote "winfo exist $widgName"
}

proc widgReadFunc {widgName attrName} {
  quote "$widgName cget $attrName"
}

proc widgWriteFunc {widgName attrName value} {
  quote "$widgName config $attrName $value"
}

#set TP_traceableType(widget,propFunc)       "TP_objPropChange"

# Tk canvas items

set TP_traceableType(citem,traceCreate)     "trace citem"
set TP_traceableType(citem,traceInfo)       "trace cinfo"
set TP_traceableType(citem,traceDelete)     "trace cdelete"

set TP_traceableType(citem,exist)           "citemExistFunc"
set TP_traceableType(citem,global)          ""
set TP_traceableType(citem,read)            "citemReadFunc"
set TP_traceableType(citem,write)           "citemWriteFunc"

proc citemExistFunc {canvName tagOrId attrName} {
  quote "llength \[$canvName find withtag $tagOrId\]"
}

proc citemReadFunc {canvName tagOrId attrName} {
  if {[string compare $attrName "coords"]} {
    quote "$canvName itemcget $tagOrId $attrName"
  } else {
    quote "$canvName coords $tagOrId"
  }
}

proc citemWriteFunc {canvName tagOrId attrName value} {
  if {[string compare $attrName "coords"]} {
    quote "$canvName itemconf $tagOrId $attrName $value"
  } else {
    # $value must be a list of appropriate length
    quote "$canvName coords $tagOrId $value"
  }
}

#set TP_traceableType(citem,propFunc)        "TP_citemPropChange"

# CMT objects

set TP_traceableType(cmobj,traceCreate)     "trace cmobj"
set TP_traceableType(cmobj,traceInfo)       "trace cminfo"
set TP_traceableType(cmobj,traceDelete)     "trace cmdelete"

set TP_traceableType(cmobj,exist)           "cmobjExistFunc"
set TP_traceableType(cmobj,global)          ""
set TP_traceableType(cmobj,read)            "cmobjReadFunc"
set TP_traceableType(cmobj,write)           "cmobjWriteFunc"

proc cmobjExistFunc {varName attrName} {
    quote "string compare \"\[info command $varName\]\" \"\""
}

#proc cmobjGlobalFunc {varName attrName} {
#
#}

proc cmobjReadFunc {varName attrName} {
    quote "lindex \[$varName config $attrName\] 2"
}


proc cmobjWriteFunc {varName attrName value} {
    quote "$varName config $attrName $value"
}

#set TP_traceableType(cmobj,propFunc)        "TP_objPropChange"
