## **********************************************************************
## **********************************************************************
##
##  File "manager.tcl" contains the data management facilities which
##  correspond to the Video Poker application, TkVP.  The relationships
##  relationships between the various data values and widget configurations
##  are specified through the use of "TP_trigger", "TP_formula", and 
##  "TP_makeVar" provided by the TclProp system.  
##
## **********************************************************************
## **********************************************************************


## ********************************************************************
## 
##  The following code uses "TP_trigger" to create a trigger on the
##  value of the player's bet (bet_amt).  When the amount bet is equal
##  to the maximum bet allowed (max_bet), then the mode of the game
##  should be set to "draw".  When the amount bet is 0, and the current
##  mode is "pay", then a new game is established by setting the mode to
##  "bet".
##
## ********************************************************************

TP_trigger bet_amt  { 

        if { ( [string compare $mode pay] == 0 )  &&  \
             ( $bet_amt > 0 )  }  {
                set mode bet
        }

        if { $bet_amt == $max_bet }  {
                set mode draw
        }

}
        
## **********************************************************************
##
##  The following code uses "TP_makeVar" to create a variable
##  associated with the configuration (in particular, the state) of the
##  "Bet One" button.  This variable is created to be used in a formula 
##  which will establish the relationship between the "Bet One" button
##  configuration, the player's credit value (credits), and the mode of 
##  the game.
##
## **********************************************************************

TP_makeVar betOneState      \
        -write {  if  $betOneState { 
                      .poker.bet.betone config -state normal
                  } else {
                      .poker.bet.betone config -state disabled
                  }
               } 
        
## **********************************************************************
##
##  The following formula establishes the dependency of the configuration
##  of the "Bet One" button on the values of "credits" and "mode".  If
##  the current mode of the game is "draw", or the player's credit value
##  is less than or equal to 0, then the player should not be allowed to
##  bet, and the "Bet One" button should be disabled.
##
## **********************************************************************

TP_formula betOneState { credits mode }  {

        expr ([string compare $mode draw]) && ($credits > 0)
}

## **********************************************************************
##
##  The following code uses "TP_makeVar" to create a variable
##  associated with the configuration (in particular, the state) of the
##  "Bet Max" button.  This variable is created to be used in a formula 
##  which will establish the relationship between the "Bet Max" button
##  configuration, the player's credit value (credits), and the mode of 
##  the game.
##
## **********************************************************************

TP_makeVar betMaxState    \
        -write { if  $betMaxState {
                     .poker.bet.betmax config -state normal
                  } else {
                     .poker.bet.betmax config -state disabled
                  }
               }

## **********************************************************************
##
##  The following formula establishes the dependency of the configuration
##  of the "Bet Max" button on the values of "credits" and "mode".  If
##  the current mode of the game is "draw", or the player's credit value
##  is less than the maximum bet amount, then the "Bet Max" button should
##  be disabled.  
##
## **********************************************************************

TP_formula betMaxState { credits mode } {

        expr ( [string compare $mode draw] ) && ( $credits >= $max_bet )
}

## **********************************************************************
##
##  The following code uses "TP_makeVar" to create a variable
##  associated with the configuration (in particular, the state) of the
##  "Deal" button.  This variable is created to be used in a formula
##  which will establish the relationship between the "Deal" button
##  configuration, the amount of the bet, and the mode of the game.
##
## **********************************************************************

TP_makeVar dealBtnState    \
        -write {  if  $dealBtnState {
                      .poker.game.deal config -state normal
                  } else {
                      .poker.game.deal config -state disabled
                  }
               } 

        
## **********************************************************************
##
##  The following formula establishes the dependency of the configuration
##  of the "Deal" button on the values of "bet_amt" and "mode".  A player
##  may initiate the dealing of the cards only when the current game 
##  mode is "bet", and the amount of the bet (bet_amt) is non-zero.
##
## **********************************************************************

TP_formula dealBtnState { bet_amt mode }  {

        expr ([string compare $mode bet] == 0)  && $bet_amt
}

## **********************************************************************
##
##  The following code uses "TP_makeVar" to create a variable
##  associated with the configuration (in particular, the state) of the
##  "Stand" button.  This variable is created to be used in a formula
##  which will establish the relationship between the "Stand" button
##  configuration and the mode of the game.
##
## **********************************************************************

TP_makeVar standBtnState   \
        -write {  if $standBtnState {
                     .poker.game.stand config -state normal
                  } else {
                     .poker.game.stand config -state disabled
                  }
               }


## **********************************************************************
##
##  The following formula establishes the dependency of the configuration
##  of the "Stand" button on the current game mode ("mode").  If
##  the current mode of the game is not "draw", then the "Stand" button
##  should be disabled.
##
## **********************************************************************

TP_formula standBtnState  { mode }  {
        expr [string compare $mode draw] == 0
}

## **********************************************************************
##
##  The following code uses "TP_makeVar" to create a variable
##  associated with the configuration (in particular, the state) of the
##  "Cancel" button.  This variable is created to be used in a formula
##  which will establish the relationship between the "Cancel" button
##  configuration, the "hold" status of the cards, and the mode of the 
##  game.
##
## **********************************************************************

TP_makeVar cancelBtnState   \
        -write { if  $cancelBtnState {
                     .poker.game.cancel config -state normal
                  } else {
                     .poker.game.cancel config -state disabled
                  }
               }  


## **********************************************************************
##
##  The following formula establishes the dependency of the configuration
##  of the "Cancel" button on the values of the "hold" array and the 
##  current game mode.  If any of the cards in the player's hand are
##  currently held, and the game is in "draw" mode,  then the "Cancel" 
##  button should be enabled.  However, if no cards are held, or the game
##  is not in "draw" mode, then the "Cancel" button should be disabled.
##
## **********************************************************************

TP_formula cancelBtnState  \
        { hold(1) hold(2) hold(3) hold(4) hold(5) mode }  {

         expr (![string compare $mode draw] && \
               ($hold(1) || $hold(2) || $hold(3) || $hold(4) || $hold(5)))
}

## **********************************************************************
##
##  The following code uses "TP_makeVar" to create a variable
##  associated with the configuration (in particular, the state) of the
##  "Draw" button.  This variable is created to be used in a formula
##  which will establish the relationship between the "Draw" button
##  configuration and the mode of the game.
##
## **********************************************************************

TP_makeVar drawBtnState  \
        -write {  if $drawBtnState {
                     .poker.game.draw config -state normal
                  } else {
                     .poker.game.draw config -state disabled
                  }
               }


## **********************************************************************
##
##  The following formula establishes the dependency of the configuration
##  of the "Draw" button on the current game mode.  If the current mode
##  of the game is not "draw", then the "Draw" button should be disabled.
##
## **********************************************************************

TP_formula drawBtnState { mode } {
        expr [string compare $mode draw] == 0 
}
        
## **********************************************************************
##
##  The following section of code establishes the formulas which
##  correspond to the "Hold" button for each card in the hand.  The
##  formulas are used to change the "text" and "state" configure options
##  of the buttons.  The configuration of the "Hold" buttons are  
##  dependent upon the variable, "mode", and the array, "hold".  When
##  the current mode is not "draw", the "Hold" buttons are disabled 
##  and have no label.  If the current mode is "draw", and the value 
##  of an entry in the "hold" array is true (1), the button corresponding 
##  to that array entry is dimmed, and the label of that button is 
##  "held".  Otherwise, if the current mode is "draw" and the value 
##  of an entry of the "hold" array is false (0), the button is 
##  active and has the label, "Hold".  Thus, when in "draw" mode, the 
##  state of the button is determined by the status of the "hold" array 
##  entry corresponding to that button. 
##
## **********************************************************************

## Hold button for card 1

TP_formula btn1 { mode hold(1) }  {
        if { [string compare $mode draw] }  {
             quote "" 

        } elseif  { $hold(1) }  {
             quote held
        
        } else {
             quote Hold 
        }

}

## Hold button for card 2

TP_formula btn2 { hold(2) mode  }  {

        if { [string compare $mode draw] }  {
             quote "" 

        } elseif  { $hold(2) }  {
             quote held
        
        } else {
             quote Hold 
        }

}


## Hold button for card 3

TP_formula btn3 { hold(3) mode }  {

        if { [string compare $mode draw] }  {
             quote "" 

        } elseif  { $hold(3) }  {
             quote held
        
        } else {
             quote Hold 
        }

}
            

## Hold button for card 4

TP_formula btn4 { hold(4) mode }  {

        if { [string compare $mode draw] }  {
             quote "" 

        } elseif  { $hold(4) }  {
             quote held
        
        } else {
             quote Hold 
        }

}
            

## Hold button for card 5

TP_formula btn5 { hold(5) mode }  {

        if { [string compare $mode draw] }  {
             quote "" 

        } elseif  { $hold(5) }  {
             quote held
        
        } else {
             quote Hold 
        }

}
            
## **********************************************************************
##
##  The following code uses "TP_makeVar" to create a variable
##  associated with the configuration (in particular, the state) of the
##  "HOLD" buttons corresponding to each card.  This variable is created
##  to be used in a formula which will establish the relationship between
##  the configuration of the "HOLD" button and the corresponding value 
##  of the "hold" array.
##
## **********************************************************************

## Card One
  
TP_makeVar card1btnState   \
        -write { if  $card1btnState  {
                     .poker.hand.deck1.btn config -state normal
                 } else {  
                     .poker.hand.deck1.btn config -state disabled
                 }
               } 


TP_formula card1btnState { btn1 }  {
        expr [string compare $btn1 Hold] == 0
}



## Card Two

TP_makeVar card2btnState   \
        -write { if  $card2btnState  {
                     .poker.hand.deck2.btn config -state normal
                 } else {  
                     .poker.hand.deck2.btn config -state disabled
                 }
               }  


TP_formula card2btnState { btn2 }  {
        expr [string compare $btn2 Hold] == 0
}


## Card Three

TP_makeVar card3btnState   \
        -write { if  $card3btnState  {
                     .poker.hand.deck3.btn config -state normal
                 } else {  
                     .poker.hand.deck3.btn config -state disabled
                 }
               } 


TP_formula card3btnState { btn3 }  {
        expr [string compare $btn3 Hold] == 0
}


## Card Four

TP_makeVar card4btnState   \
        -write { if  $card4btnState  {
                     .poker.hand.deck4.btn config -state normal
                 } else {  
                     .poker.hand.deck4.btn config -state disabled
                 }
               }  

TP_formula card4btnState { btn4 }  {
        expr [string compare $btn4 Hold] == 0
}


## Card Five

TP_makeVar card5btnState   \
        -write { if  $card5btnState  {
                     .poker.hand.deck5.btn config -state normal
                 } else {  
                     .poker.hand.deck5.btn config -state disabled
                 }
               }  


TP_formula card5btnState { btn5 }  {
        expr [string compare $btn5 Hold] == 0
}

## **********************************************************************
##
##  The following trigger executes the necessary actions corresponding
##  to changes in the mode of the game.  In a single game, the
##  three possible modes are:  "bet", "draw", and "pay".  When the 
##  mode becomes "bet" mode, a new game is beginning.  The deck is
##  shuffled and a hand of facedown cards is dealt.  Any holds on cards
##  from the previous round are clear.  In "draw" mode, the hand is
##  revealed, and the user decides which actions he/she would like to
##  take (i.e. stand, hold cards, draw).  Finally, in "pay" mode, the
##  user's hand is evaluated, any winnings are determined, and the 
##  user's credit is updated as necessary.  The definitions which
##  correspond to the procedures used in this trigger are defined in
##  file "support.tcl".
##
## **********************************************************************

TP_trigger mode {

        switch $mode  {
             bet      { bet_mode_setup }
             draw     { draw_mode_setup }
             pay      { pay_mode_setup }
        }
}
