
## ---------------------------------------------------------------
##  File "support.tcl" contains various procedures used during the
##  execution of the TkVP (i.e. procedures to determine winnings,
##  to deal cards, to shuffle the deck, to get the rank and/or 
##  suit of a card, etc).
## ---------------------------------------------------------------

## ---------------------------------------------------------------

catch {unset deck}
catch {unset allRanks}
catch {unset allSuits}

set allRanks         "1 2 3 4 5 6 7 8 9 10 11 12 13"
set allSuits         "clubs diamonds hearts spades"
set winning_hands  { {royal_flush 500 "Royal Flush"}   
		     {straight_flush 100 "Straight Flush"}
                     {four_of_a_kind 25 "Four of a Kind"} 
		     {full_house 9 "Full House"}
                     {flush 6 "Flush"}           
		     {straight 4 "A Straight"} 
                     {three_of_a_kind 3 "3 of a Kind"} 
		     {two_pair 2 "2 Pair"}
                     {jacks_or_better 1 "Jacks or Better"} }


## ---------------------------------------------------------------
##  Procedure:		get_rank
##  Input:		A card
##  Process: 		Returns the rank of a given card.
##  Output:		The rank of the given card.
## ---------------------------------------------------------------

proc get_rank { card } {
     return [lindex $card 0]
}


## ---------------------------------------------------------------
##  Procedure:		get_suit
##  Input:		A card
##  Process: 		Returns the suit of a given card.
##  Output:		The suit of the given card.
## ---------------------------------------------------------------

proc get_suit { card } {
     return [lindex $card 1]
}


## ---------------------------------------------------------------
##  Procedure:		make_full_deck
##  Input:		None
##  Process: 		Creates a deck of 52 cards.
##  Output:		None
## ---------------------------------------------------------------

proc make_full_deck {}  {
     global full_deck
     global allRanks
     global allSuits

     catch { unset full_deck }
     foreach rank $allRanks  {

        foreach suit $allSuits  {
            set card [list $rank $suit]
            lappend full_deck $card
        }

    }
}

## ---------------------------------------------------------------
##  Procedure:		shuffle_deck
##  Input:		None
##  Process: 		Shuffles a deck of cards.
##  Output:		None
## ---------------------------------------------------------------

proc shuffle_deck {}  {
     global deck full_deck

     set deck ""

     while { [llength $deck] < 10 }  {
             set index [randomRange 52]
             set card [lindex $full_deck $index]
             if {[lsearch -exact $deck $card] == -1} {
                 lappend deck $card
             }
     }
}


## ---------------------------------------------------------------
##  Procedure:		deal_card
##  Input:		A card widget which will display the newly
##                      dealt card, and a variable indicating 
##                      whether the new card is dealt face up or 
##                      face down.  A card is dealt face down by 
##                      default.
##  Process: 		A new card is "dealt" from the global 
##                      variable, "deck", which maintains a deck
##                      of cards.  The bitmap of the card widget
##                      passed to the procedure as the parameter,
##                      "cardWidget", is configured according to
##                      the state of the parameter, "cardState".
##  Output:		A new card
## ---------------------------------------------------------------

proc deal_card { cardWidget { cardState facedown }  }  {
     global bitmapdir
     global deck
 
     set newCard [lindex $deck 0]
     set deck    [lreplace $deck 0 0]

     if { [string compare $cardState facedown] == 0 }  {
          $cardWidget config -bitmap @$bitmapdir/myback   \
                             -foreground Black
     } else {
          set rank [get_rank $newCard]
          set suit [get_suit $newCard]
          $cardWidget config -bitmap @$bitmapdir/$suit$rank

          if { ( [string compare $suit hearts] == 0 )  ||
               ( [string compare $suit diamonds] == 0 )  }  {
              $cardWidget config -foreground Red
          }    
     }

     return $newCard
}

## ---------------------------------------------------------------
##  Procedure:		deal_hand
##  Input:		None
##  Process: 		Deals a hand of 5 cards by calling procedure
##                      "deal_card" 5 times.
##  Output:		None
## ---------------------------------------------------------------

proc deal_hand {} {
     global hand
     global cardstats

     for {set index 1} {$index <= 5}  {incr index}  {
        set hand($index) [deal_card $cardstats$index.card]
     }
}

## ---------------------------------------------------------------
##  Procedure:		flip
##  Input:		A card represented as a list composed of the
##                      suit and rank of a card, a card widget which
##                      displays the card being "flipped", and a
##                      string variable indicating whether the card
##                      is to be flipped face up or face down.  A
##                      card is flipped face down by default.
##  Process: 		If the parameter, "cardface", indicates that
##                      a card is to be flipped face up, then the
##                      suit and rank values from the parameter, 
##                      "card", are used to generate the appropriate
##                      bitmap to be displayed.  The parameter, 
##                      "cardWidget", is configured to display the
##                      appropriate bitmap.
##  Output:		None
## ---------------------------------------------------------------

proc flip { card cardWidget { cardface facedown } }  {
     global bitmapdir

     if { [string compare $cardface facedown] == 0 }  {
          $cardWidget config -bitmap @$bitmapdir/myback
     } else {
          set rank [get_rank $card]
          set suit [get_suit $card]
          $cardWidget config -bitmap @$bitmapdir/$suit$rank

          if { ( [string compare $suit hearts] == 0 )  ||
               ( [string compare $suit diamonds] == 0 )  }  {
              $cardWidget config -foreground Red
          }
     }
}


## ---------------------------------------------------------------
##  Procedure:		bet_mode_setup
##  Input:		None
##  Process: 		Procedure "bet_mode_setup" is executed
##                      every time the mode of the game is set to
##                      "bet".  In "bet" mode, a full deck is 
##                      created and shuffled, the array, "hold",
##                      which indicates which cards in a player's
##                      hand are held, is cleared, and a new hand
##                      is dealt.  The "status" message is set to
##                      indicate that a new game is to begin and
##                      bets should be placed.  
##  Output:		None
## ---------------------------------------------------------------

proc bet_mode_setup {} {
     shuffle_deck

     clear_holds
     deal_hand

     set status "Place Your Bets!!"
}

## ---------------------------------------------------------------
##  Procedure:		draw_mode_setup
##  Input:		None
##  Process: 		Procedure "draw_mode_setup" is executed
##                      every time the mode of the game is set to
##                      "draw".  In "draw" mode, the cards in the
##                      player's hand are shown, and the status
##                      message is cleared.
##  Output:		None
## ---------------------------------------------------------------

proc draw_mode_setup {} {
     global hand
     global cardstats
     global status

     foreach index [array names hand] {
        flip $hand($index) $cardstats$index.card faceup
     }

     set status ""
}

## ---------------------------------------------------------------
##  Procedure:		pay_mode_setup
##  Input:		None
##  Process: 		Procedure "pay_mode_setup" is executed
##                      every time the mode of the game is set to
##                      "pay".  In "pay" mode, new cards are dealt
##                      for any cards in the player's hand which are,
##                      not being held.  Then, the hand is evaluated,
##                      the results output, and credits updated (if
##                      necessary).
##  Output:		None
## ---------------------------------------------------------------

proc pay_mode_setup {} {
     global bet_amt
     global cardstats
     global credits
     global hold
     global hand
     global status

     foreach index [array names hold]  {
        
        if { ! $hold($index) }  {
              set hand($index) [deal_card $cardstats$index.card faceup]
        }
     }

     sort_hand
     set winnings [winning_hand]
     if { [lindex $winnings 0] != 0 }  {
          incr credits [expr [lindex $winnings 1]*$bet_amt]
          set status "[lindex $winnings 2]! -- bet of $bet_amt pays\
                      out at [lindex $winnings 1] to 1 for [expr $bet_amt * \
                      [lindex $winnings 1]] !!!"
     } else {
          set status "Better Luck Next Time...."
     }

     set bet_amt 0

}

## ---------------------------------------------------------------
##  Procedure:		sort_hand
##  Input:		None
##  Process: 		Procedure "sort_hand" sorts the cards in
##                      the player's hand in accending order 
##                      according to the rank.  This procedure is
##                      used to simplify the process of evaluating
##                      a player's hand.
##  Output:		None
## ---------------------------------------------------------------

proc sort_hand {} {
     global hand

     for {set i 2}  {$i <= [array size hand] }  {incr i}  {

         for {set j [array size hand] } {$j >= $i} {incr j -1} {

              set rank1  [get_rank $hand($j)]
              set rank2  [get_rank $hand([expr $j-1])]

              if { $rank1 < $rank2 }  {
                   set temp $hand($j)
                   set hand($j) $hand([expr $j-1])
                   set hand([expr $j-1]) $temp
              }
         }
     }
}

## ---------------------------------------------------------------
##  Procedure:		winning_hand
##  Input:		None
##  Process: 		Procedure "winning_hand" determines the
##                      winnings of the player's hand.  This procedure
##                      evaluates the hand by calling each procedure
##                      associated with each possible winning hand
##                      (i.e. royal flush, straight flush, etc) until
##                      either a winning hand is determined, or all
##                      possible winning hands have been tested.
##  Output:		The amount won (possibly $0) is returned.
## ---------------------------------------------------------------

proc winning_hand {}  {
     global winning_hands

     set winnings 0
     foreach hand_type $winning_hands  {
             
             if { [[lindex $hand_type 0]] }  {
                  set winnings $hand_type
                  break
             }
     }  

     return $winnings
}

## ---------------------------------------------------------------
##  Procedure:		jacks_or_better
##  Input:		None
##  Process: 		The player's hand is tested for a pair
##                      of cards which are jacks or a higher rank.
##                      This procedure operates on the invariant
##                      that the hand is in sorted order (ascending)
##                      according to the rank of the cards.
##  Output:	        A flag value indicating whether or not the
##                      hand contains jacks or better.
## ---------------------------------------------------------------

proc jacks_or_better {} {
     global hand

     set found 0
     for {set index 1} {$index < [array size hand] }  {incr index}  {
          set current_rank [get_rank $hand($index)]
      
          if { ( $current_rank == [get_rank $hand([expr $index+1])] )  &&
               ( ($current_rank > 10 )  ||  ($current_rank == 1)  )  }  {
                  set found 1
                  break
          }
      }

      return $found
}

## ---------------------------------------------------------------
##  Procedure:		two_pair
##  Input:		None
##  Process: 		The player's hand is tested for two pairs
##                      of cards which are of any rank.  This
##                      procedure operates on the invariant that
##                      the hand is in sorted order (ascending)
##                      according to the rank of the cards.
##  Output:	        A flag value indicating whether or not the
##                      hand contains two pairs.
## ---------------------------------------------------------------

proc two_pair {} {
     global hand

     set pairs 0
     for {set index 1}  {$index < [array size hand]}  {incr index}  {
          
          if { [get_rank $hand($index)] == [get_rank $hand([expr $index+1])]} {
               incr pairs
          }
     }

     return  [expr $pairs > 1]
}

## ---------------------------------------------------------------
##  Procedure:		three_of_a_kind
##  Input:		None
##  Process: 		The player's hand is tested for three
##                      cards which are of the same rank.  This
##                      procedure operates on the invariant that
##                      the hand is in sorted order (ascending)
##                      according to the rank of the cards.
##  Output:	        A flag value indicating whether or not the
##                      hand contains three of a kind.
## ---------------------------------------------------------------

proc three_of_a_kind {} {
     global hand

     expr  [get_rank $hand(1)] == [get_rank $hand(3)] ||  \
           [get_rank $hand(2)] == [get_rank $hand(4)] ||  \
           [get_rank $hand(3)] == [get_rank $hand(5) ]
}

## ---------------------------------------------------------------
##  Procedure:		straight
##  Input:		None
##  Process: 		The player's hand is tested for a straight:
##                      the ranks of the cards form a sequence of
##                      five consecutive numbers.  In a straight,
##                      an ace can be considered as either high or
##                      low (i.e. ace, 2, 3, 4, 5 is a legal
##                      straight).  This procedure operates on the
##                      invariant that the hand is in sorted
##                      order (ascending) according to the rank of
##                      the cards.
##  Output:	        A flag value indicating whether or not the
##                      hand contains a straight.
## ---------------------------------------------------------------

proc straight {}  {
     global hand

     expr  [get_rank $hand(3)] == [expr [get_rank $hand(2)]+1]  &&  \
           [get_rank $hand(4)] == [expr [get_rank $hand(3)]+1]  &&  \
           [get_rank $hand(5)] == [expr [get_rank $hand(4)]+1]  &&  \
         ( [get_rank $hand(2)] == [expr [get_rank $hand(1)]+1]  ||  \
         ( [get_rank $hand(1)] == 1  && [get_rank $hand(5)] == 13 ) ) 

}

     
## ---------------------------------------------------------------
##  Procedure:		flush
##  Input:		None
##  Process: 		The player's hand is tested for a flush:
##                      all of the cards in the player's hand are
##                      of the same suit.   
##  Output:	        A flag value indicating whether or not the
##                      hand contains a flush.
## ---------------------------------------------------------------

proc flush {}  {
     global hand

     set same_suit 1
     for {set index 1}  {$index < [array size hand]}  {incr index}  {
         
          if { [get_suit $hand($index)] != 
               [get_suit $hand([expr $index+1])] }  {
               set same_suit 0
               break
          }
     }

     return $same_suit
}
   
     
## ---------------------------------------------------------------
##  Procedure:		full_house
##  Input:		None
##  Process: 		The player's hand is tested for a full house:
##                      three of the cards are of the same rank
##                      (three of a kind) and the remaining two 
##                      cards form a pair.  This procedure operates
##                      on the invariant that the player's hand is
##                      in sorted order (ascending) according to 
##                      the ranks of the cards.
##  Output:	        A flag value indicating whether or not the
##                      hand contains a full house.
## ---------------------------------------------------------------

proc full_house {}  {
     global hand

     expr ( [get_rank $hand(1)] == [get_rank $hand(3)] &&      \
            [get_rank $hand(4)] == [get_rank $hand(5)] )  ||   \
          ( [get_rank $hand(1)] == [get_rank $hand(2)] &&      \
            [get_rank $hand(3)] == [get_rank $hand(5)] )
}

## ---------------------------------------------------------------
##  Procedure:		four_of_a_kind
##  Input:		None
##  Process: 		The player's hand is tested for four cards
##                      all of the same rank.  This procedure
##                      operates on the invariant that the player's
##                      hand is in sorted order (ascending) according 
##                      to the ranks of the cards.
##  Output:	        A flag value indicating whether or not the
##                      hand contains four of a kind.
## ---------------------------------------------------------------

proc four_of_a_kind {}  {
     global hand

     expr ( [get_rank $hand(1)] == [get_rank $hand(4)] ) ||  \
          ( [get_rank $hand(2)] == [get_rank $hand(5)] )
         
}


## ---------------------------------------------------------------
##  Procedure:		straight_flush
##  Input:		None
##  Process: 		The player's hand is tested for a straight
##                      flush:  the hand forms a straight in which
##                      all of the cards are also of the same suit.
##                      This procedure determines whether or not
##                      the hand contains a straight flush by
##                      calling procedures "straight" and "flush".
##  Output:	        A flag value indicating whether or not the
##                      hand contains a straight flush.
## ---------------------------------------------------------------

proc straight_flush {}  {
     global hand
  
    expr [straight] && [flush]
}

## ---------------------------------------------------------------
##  Procedure:		royal_flush
##  Input:		None
##  Process: 		The player's hand is tested for a royal
##                      flush:  the hand forms a straight flush
##                      in which the ranks of the cards are ten,
##                      Jack, Queen, King, and Ace.  This 
##                      procedure determines whether or not
##                      the hand contains a royal flush by
##                      calling procedure "straight_flush" and 
##                      testing the ranks of the first and last
##                      cards in the hand, assuming that the hand
##                      is in ascending order according to the rank
##                      of the cards.
##  Output:	        A flag value indicating whether or not the
##                      hand contains a royal flush.
## ---------------------------------------------------------------

proc royal_flush {}  {
     global hand

     expr  [straight_flush]             &&   \
         ( [get_rank $hand(1)] ==  1 )  &&   \
         ( [get_rank $hand(5)] == 13 )
}


## -------------------------------------------------------------------
##
##  The following  procedures are needed to implement a random number
##  generator.  These procedures are taken from the Brent Welch book,
##  Practical Programming in Tcl and Tk, found in chapter 4 on page 46.
##
## -------------------------------------------------------------------

## -------------------------------------------------------------------
##
##  Procedure "randomInit" is used to seed the random number generator
##
##  Input:	   The value used to seed the random number generator.   
##                 A good value to use is the process id number found by
##                 executing the command, "pid".
##
##  Output:        None
##
##  Side Effects:  The element in the global "rand" array indexed by 
##                 "seed" is modified to contain the value input.
##
## -------------------------------------------------------------------

proc randomInit { seed }  {
     global rand

     set rand(ia)      9301     ;##  Multiplier
     set rand(ic)     49297     ;##  Constant
     set rand(im)    233280     ;##  Divisor
     set rand(seed)  $seed      ;##  Last result
}


## -------------------------------------------------------------------
##
##  Procedure "random" generates a random number between 0 and 1.
##
##  Input:	   None  
##
##  Output:        A random number.
##
##  Side Effects:  The element in the global "rand" array indexed by 
##                 "seed" is modified to be used in calculating the 
##                 random number. 
##
## -------------------------------------------------------------------

proc random { }  {
     global rand

     set rand(seed)  \
         [expr ($rand(seed)*$rand(ia) + $rand(ic)) % $rand(im) ]
     return [expr $rand(seed)/double($rand(im))]
}

## -------------------------------------------------------------------
##
##  Procedure "randomRange" generates a random number between 0 and 
##  the value given as the upper bound.
##
##  Input:	   A value which specifies the upper bound on the random
##                 number generated.  
##
##  Output:        A random number between 0 and the value given.
##
##  Side Effects:  None
##
## -------------------------------------------------------------------

proc randomRange { range }  {
     expr int([random]*$range)
}


randomInit [pid]       ;## Seed random number generator.

## initialize the full deck

make_full_deck
