#  Copyright statment

# TP_formulaID is the number corresponding to the highest-numbered formula
# A formula ID is returned by TclProp_formula and can be used to relax,
# enforce, or delete the formula later.

global TP_formulaID
set TP_formulaID 0

# TP_tracedVars is the array of variables that have already been traced

global TP_tracedVars

# TP_propTab is the array (table) of propagations, indexed by source variable
# each entry is a list of triplets -- formulaID (negative for inactive),
# destination variable, and code.

global TP_propTab

# TP_oldValues is an array of prior values for stopping loops

global TP_oldValues

# TP_curProps is an array of variables already changed in a given propagation
# it is also used for stopping loops

global TP_curProps 

# TP_readCmds is an array of read code for makeVar variables
# TP_readCmdNo is the index of the current element of that array

global TP_readCmds TP_readCmdNo
set TP_readCmdNo 0

#  TP_formula -- takes a destination variable, a list of source
#  variables, and a formula expression, and it implements a formula to
#  automatically update the destination whenever any source changes.
#  Note:  this is also called from TP_trigger to handle triggers (the
#  des is empty in that case).  Note that the formula is activated once
#  when defined to "make things consistent" but if the formula cannot execute
#  (e.g., if some variables are not yet defined) the errors are ignored.

#  For example, a typical formula would be:
#      TP_formula a {b c} {expr b + c}

proc TP_formula {des srcs code} {

   global TP_formulaID
   global TP_tracedVars
   global TP_propTab
   global TP_oldValues
   global TP_curProps
   
   #  assign the formula ID
   incr TP_formulaID

   #  foreach variable:  set trace if needed and enter in table

   foreach src $srcs {

     #  varname is variable name (i.e., x for x, a for a(5))
     set varname [TP_varname $src]

     #  handle trace if needed
     if {! [info exists TP_tracedVars($src)]} {
        set TP_tracedVars($src) 1
        global $varname

        if {[info exists $src]} {
           set TP_oldValues($src) [set $src]
        } else {
           set TP_oldValues($src) TP_dummyValue
        }

        set TP_curProps($src) 0

        trace variable $src w TP_propchange
     }

     set formula_entry [list $TP_formulaID $des $code]
     lappend TP_propTab($src) $formula_entry
   }

   # evaluate the formula once, to get things set up

   set doit {uplevel #0 {set DES [CODE]}}
   regsub DES $doit $des doit
   regsub CODE $doit $code doit
   
   catch {eval $doit}

   #  return the formula ID
   return $TP_formulaID
}

#  TP_varname -- strips off array notation, leaving scalar variables alone
#  but returning the variable name for array variables

proc TP_varname {name} {
   scan $name {%[^(]} result
   return $result
}

#  TP_propchange -- the actual trace callback that handles propagation of 
#  changes.  It is called whenever a source variable is changed.  It then
#  checks whether:  a)  the variable value really changed, and b) we are in 
#  a loop.  In either case, it returns, otherwise it evaluates each formula 
#  in which that variable is a source.

#  Note:  This version does not print error messages -- it just ignores 
#  errors in formula code and goes on its merry way!  That should be fixed
#  in a later version if it can be handled efficiently.

proc TP_propchange {name index op} {
   global $name
   global TP_oldValues TP_propTab TP_curProps
   if {$index != ""} {append name "(" $index ")"}
   if {$TP_curProps($name)} {
      return
   } else {
      set TP_curProps($name) 1
   }
   if {$TP_oldValues($name) != [set $name]} {
      set TP_oldValues($name) [set $name]
      foreach prop $TP_propTab($name) {
	 set formID [lindex $prop 0]
	 set des [lindex $prop 1]
	 set code [lindex $prop 2]
	 if {$formID > 0} {
	    if {$des != ""} {
	       global [TP_varname $des]
	       catch {set $des [uplevel #0 $code]}
	    } else {
	       catch {uplevel #0 $code}
	    }
	 }
      }
   }
   set TP_curProps($name) 0
}

#  TP_trigger is a trigger interface (similar to trace) that simply indicates
#  that certain code should be executed whenever a variable changes values.
#  The syntax is simple:
#	TP_trigger a {puts "A changed!"}
#  Note that the trigger executes once right away when defined (unless it
#  cannot due to errors).

proc TP_trigger {srcs code} {
    TP_formula "" $srcs $code
}

#  TP_makeVar -- this function creates a synthetic variable that acts as an 
#  interface between the other formula functions and a non-traceable
#  attribute.  TP_makeVar takes a variable name, and optional -read and -write
#  code and and optional read frequency (-rf) in milliseconds.  For example:
#
#      TP_makeVar a-morethan-b -read {expr $a > $b} -rf 500
#
#  would set up a global variable a-morethan-b which polls every 1/2 second 
#  to update itself to the current condition of whether a is greater than 
#  b.  (Not a useful example.)  Writes are purely triggers, whereas reads are
#  implemented as after loops.

proc TP_makeVar {var args} {
    set read_freq 1000
    foreach arg $args {
        switch -- $arg {
           -read  { set next read }
           -write { set next write }
           -rf    { set next rf }
           default {
              switch $next {
                 read  { set read_action $arg }
                 write { TP_trigger $var $arg }
                 rf    { set read_freq $arg }
              }
           }
        }
    }
    if {[info exists read_action]} {
        global TP_readCmds TP_readCmdNo
        incr TP_readCmdNo
        set cmdString {uplevel #0 {set VAR [ RA ] ; after RF $TP_readCmds(NUM)}}
        regsub VAR $cmdString $var cmdString
	regsub RA $cmdString $read_action cmdString
	regsub RF $cmdString $read_freq cmdString
	regsub NUM $cmdString $TP_readCmdNo cmdString
        set TP_readCmds($TP_readCmdNo) $cmdString
        eval $cmdString
    }
}

#  This is the quote procedure that makes it easier to write if statements
#  that return expression values.

proc quote {x} {return $x}
