/* 
 * oat.c --
 *
 *	This file contains the functions needed to support traces on 
 *	arbitrary Tcl and extension objects.
 *
 * Copyright (c) 1987-1993 The Regents of the University of California.
 * Copyright (c) 1994-1996 Sun Microsystems, Inc.
 * Copyright (c) 1996 University of Minnesota.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS:  $Id: oat.c,v 1.37 1996/11/30 23:40:25 safonov Exp safonov $ 
 */

#include "oat.h"
#include "oatInt.h"
#include "tclInt.h"  
/* tclInt.h is needed for the declarations of:
   - Tcl_TraceCmd();
   - struct Interp. Interp is used in OatTraceProc() to save
     and restore the interpreter state. Anything better than this kludge? 
 */


EXTERN EXPORT(int,Oat_Init) _ANSI_ARGS_((Tcl_Interp *interp));

/*
 *----------------------------------------------------------------------
 *
 * DllEntryPoint --
 *
 *	This wrapper function is used by Windows to invoke the
 *	initialization code for the DLL.  If we are compiling
 *	with Visual C++, this routine will be renamed to DllMain.
 *	routine.
 *
 * Results:
 *	Returns TRUE;
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

#ifdef __WIN32__
BOOL APIENTRY
DllEntryPoint(hInst, reason, reserved)
    HINSTANCE hInst;		/* Library instance handle. */
    DWORD reason;		/* Reason this function is being called. */
    LPVOID reserved;		/* Not used. */
{
    return TRUE;
}
#endif

/* 
 * 
 */

Oat_TraceableObjType Oat_TraceableVariableType = {
    "variable",			   /* nameKWord */
    "vinfo",			   /* infoKWord */
    "vdelete",			   /* deleteKWord */
    Tcl_TraceCmd,                  /* traceCmdProc */
    (Oat_TraceableObjType *) NULL  /* nextPtr */
};

/*
 * The following variable points to the first in a list of all known
 * image types.
 */

static Oat_TraceableObjType* traceableObjTypeList = NULL;

/*
 *----------------------------------------------------------------------
 *
 * Oat_Init --
 *
 *	This procedure initializes the extended trace command.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

EXPORT(int,Oat_Init)(interp)
    Tcl_Interp *interp;
{
    Tcl_DeleteCommand(interp, "trace");

    Tcl_CreateCommand(interp, "oatrace", Oat_TraceCmd, 
		      (ClientData)NULL, /*deleteProc*/NULL);

    Tcl_InitHashTable(&oatHash, TCL_STRING_KEYS);


    /* 
     * Initialize built-in traceable object type: variable
     */

    Oat_CreateTraceableObjType(&Oat_TraceableVariableType);

    return Tcl_PkgProvide(interp, "oat", OAT_VERSION);
}


/*
 * Hash table used to provide access to object traces from C code.
 */
Tcl_HashTable oatHash;

/*
 * first active object attribute trace
 */
OatActiveTrace* oatActiveTracePtr = NULL;

int OatDebugFlag = 1;

/*********************************************************************
 * Forward declarations for procedures defined in this file.
 *********************************************************************/

static char*            OatMakeHashKey _ANSI_ARGS_((char *objPtr,
						    char *attrName));
static char*            OatMakeAttrListHashKey _ANSI_ARGS_((char *objPtr));

/* 
 * Allocate trace header structure - it is hashed by the 
 * object address/attribute name pair and points to the head
 * of the list of trace info structures for this object & attribute.
 */

static OatTraceHeader* OatAllocTraceHeader _ANSI_ARGS_((char *nameSpace, 
							char *objName, 
							char *attrName));
static char*           OatGetTracedAttr    _ANSI_ARGS_((char *objPtr, 
							int isFirst));
static char* ckstrdup _ANSI_ARGS_((char* src));


/*
 *----------------------------------------------------------------------
 *
 * Oat_CreateTraceableObjType --
 *
 *	This procedure is invoked by the traceable object manager to tell Tcl
 *      about a new kind of traceable object and the procedures that manage 
 *      the new type.
 *	The procedure is typically invoked during Tcl_AppInit.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The new traceable object type is entered into a table used in the 
 *      "trace" command.
 *
 *----------------------------------------------------------------------
 */

EXPORT(void, Oat_CreateTraceableObjType) (typePtr)
    Oat_TraceableObjType *typePtr;  /* Structure describing the type.  All of
				     * the fields except "nextPtr" must be 
				     * filled in by caller. Must not have been
				     * passed to Oat_CreateTraceableObjType 
				     * previously. */
{
    Oat_TraceableObjType *typePtr2;

    typePtr2 = (Oat_TraceableObjType *) ckalloc(sizeof(Oat_TraceableObjType));
    *typePtr2 = *typePtr;

    typePtr2->nameKWord   = (char *)ckstrdup(typePtr->nameKWord);
    typePtr2->infoKWord   = (char *)ckstrdup(typePtr->infoKWord);
    typePtr2->deleteKWord = (char *)ckstrdup(typePtr->deleteKWord);

    typePtr2->nextPtr = traceableObjTypeList;
    traceableObjTypeList = typePtr2;

#ifdef OAT_DEBUG
    printf("Oat_CreateTraceableObjType(): added traceable object type %s %s %s\n", 
	   typePtr2->nameKWord, typePtr2->infoKWord, typePtr2->deleteKWord);
#endif /* OAT_DEBUG */
}

/*
 *----------------------------------------------------------------------
 *
 * Oat_TraceCmd --
 *
 *	This procedure is invoked to process the "trace" Tcl command.
 *      It attempts to match the keyword specified after the "trace"
 *      with a keyword for any of the known traceable types. If matched,
 *      the trace command procedure for that traceable type is invoked.
 *
 *      Variable is always known as a traceable type - this preserves
 *      the original variable trace mechanism.
 *
 * Results:
 *	If no keyword was matched, TCL_ERROR. Otherwise, the standard 
 *      Tcl result that is returned by trace command procedure for
 *      a specific traceable type.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Oat_TraceCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int c;
    size_t length;
    Oat_TraceableObjType* typePtr;


    if (argc < 2) {
      Tcl_AppendResult(interp, "too few args: should be \"",
		       argv[0], " option [arg arg ...]\"", (char *) NULL);
      return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);

    /*
     * Match the second word of the command line with one of the
     * {create,info,delete} keywords for known traceable object
     * types.
     */

    for (typePtr = traceableObjTypeList; 
	 typePtr != NULL;
	 typePtr = typePtr->nextPtr) {
      if (((c == typePtr->nameKWord[0]) && 
	   (strcmp(argv[1], typePtr->nameKWord) == 0)) ||
	  ((c == typePtr->infoKWord[0]) && 
	   (strcmp(argv[1], typePtr->infoKWord) == 0)) ||
	  ((c == typePtr->deleteKWord[0]) && 
	   (strcmp(argv[1], typePtr->deleteKWord) == 0))) {
	break;
      }
    }
    if (typePtr == NULL) {
      Tcl_AppendResult(interp, "bad option \"", argv[1],
		       "\": should be ", (char *) NULL);
      for (typePtr = traceableObjTypeList; 
	   typePtr != NULL;
	   typePtr = typePtr->nextPtr) {
	Tcl_AppendResult(interp, 
		 typePtr->nameKWord,   ", ",
		 typePtr->infoKWord,   (typePtr->nextPtr ? ", " : ", or "),
		 typePtr->deleteKWord, (typePtr->nextPtr ? ", " : ""),
			 (char *) NULL);
      }	  
      return TCL_ERROR;
    }

#ifdef OAT_DEBUG
    printf("Oat_TraceCmd(): found traceable object type %s\n", typePtr->nameKWord);
#endif 

    /* 
     * Call the type-specific C function for processing the trace Tcl
     * command.
     */

    return (typePtr->traceCmdProc)(dummy, interp, argc, argv);
}

/*
 *----------------------------------------------------------------------
 *
 * Oat_CreateTrace --
 *
 *	Arrange for reads and/or writes to an object attribute to cause a
 *	procedure to be invoked, which can monitor the operations
 *	and/or change their actions.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	A trace is set up on the attribute attrName of object pointed to by
 *      objPtrin the namespace nameSpace, such that future references to the 
 *      object attribute will be intermediated by proc. See the manual entry 
 *      for complete details on the calling sequence for proc.
 *
 *----------------------------------------------------------------------
 */

int
Oat_CreateTrace(interp, objPtr, nameSpace, objName, attrName, flags, proc, clientData)
    Tcl_Interp *interp;		/* Interpreter in which object attribute 
				   is to be traced. */
    char *objPtr;               /* pointer to the object record in memory */
    char *nameSpace;            /* name space where the traced object lives,
				 * such as the canvas name for a canvas item object */
    char *objName;		/* Name of traced object. */
    char *attrName;		/* Name of object attribute */
    int flags;			/* OR-ed collection of bits, including any
				 * of TCL_TRACE_READS, TCL_TRACE_WRITES,
				 * TCL_TRACE_UNSETS. */
    Oat_TraceProc *proc;	/* Procedure to call when specified ops are
				 * invoked upon objName. */
    ClientData clientData;	/* Arbitrary argument to pass to proc. */
{
    Tcl_HashEntry *entry;
    char* hashKey;
    int hnew;
    register OatTrace *tracePtr;
    OatTraceHeader* traceHeaderPtr;
    OatTracedAttr* tracedAttrPtr;

    /*
     * Set up trace information.
     */

    tracePtr = (OatTrace *) ckalloc(sizeof(OatTrace));
    tracePtr->traceProc = proc;
    tracePtr->clientData = clientData;
    tracePtr->flags = flags &
	    (TCL_TRACE_READS|TCL_TRACE_WRITES|TCL_TRACE_UNSETS);
    tracePtr->nextPtr = NULL;

    /* get or create an entry in the object trace table */
    hashKey = OatMakeHashKey(objPtr, attrName);
    entry = Tcl_FindHashEntry(&oatHash, hashKey);
    if( entry == NULL ) {
      /* trace header for this object and attribute is not
       * set up yet. Do it now. */
      entry = Tcl_CreateHashEntry(&oatHash, hashKey, &hnew);
      traceHeaderPtr = OatAllocTraceHeader(nameSpace, objName, attrName);
      traceHeaderPtr->firstTracePtr = tracePtr;
      Tcl_SetHashValue(entry, traceHeaderPtr);
    } else {
      /* trace header for this object and attribute already set up.
       * Add current trace as the first element in its list. */
      traceHeaderPtr = 
	(OatTraceHeader*)Tcl_GetHashValue(entry);
      tracePtr->nextPtr = traceHeaderPtr->firstTracePtr;
      traceHeaderPtr->firstTracePtr = tracePtr;
    }

    /*
     * For each object with traces on its attributes, a list with
     * names of traced attributes is maintained. This is needed for
     * cleanup of traces when objects are deleted. The first entry
     * in the list is accessed from the hash table using object
     * pointer as a key.
     */

    /* get or create an entry for traced attributes of this object *
     * in the object trace table */
    hashKey = OatMakeAttrListHashKey(objPtr);
    entry = Tcl_FindHashEntry(&oatHash, hashKey);
    if( entry == NULL ) {
      /* list of traced attributes for this object is not set up yet. *
       * Do it now. */
      tracedAttrPtr = 
	(OatTracedAttr *)ckalloc(sizeof(OatTracedAttr));

      tracedAttrPtr->attrName = ckstrdup(attrName);
      tracedAttrPtr->nextPtr = NULL;

      entry = Tcl_CreateHashEntry(&oatHash, hashKey, &hnew);
      Tcl_SetHashValue(entry, tracedAttrPtr);
    } else {
      int attrFound = 0;
      OatTracedAttr *curPtr, *newPtr;

      tracedAttrPtr = (OatTracedAttr *)Tcl_GetHashValue(entry);
      for (curPtr = tracedAttrPtr;
	   curPtr != NULL;
	   curPtr = curPtr->nextPtr) {
	if (strcmp(tracedAttrPtr->attrName, attrName) == 0) {
	  attrFound = 1;
	  break;
	}
      }

      if (!attrFound) {
	/* add the new traced attribute to the list */
	newPtr = (OatTracedAttr *)ckalloc(sizeof(OatTracedAttr));
	newPtr->attrName = ckstrdup(attrName);
	newPtr->nextPtr = tracedAttrPtr;
	Tcl_SetHashValue(entry, newPtr);
      }
    }
    return TCL_OK;
}
/*
 *----------------------------------------------------------------------
 *
 * Oat_QueryTrace --
 *
 *	Return the clientData value associated with a trace on a
 *	object attribute.  This procedure can also be used to step through
 *	all of the traces on a particular attribute of a specified object 
 *      that have the same trace procedure.
 *
 * Results:
 *	The return value is the clientData value associated with
 *	a trace on the given object attribute.  Information will only be
 *	returned for a trace with proc as trace procedure.  If
 *	the clientData argument is NULL then the first such trace is
 *	returned;  otherwise, the next relevant one after the one
 *	given by clientData will be returned.  If there are no (more) traces 
 *      for the specified attribute, then NULL is returned.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
ClientData
Oat_QueryTrace(interp, objPtr, nameSpace, objName, attrName, proc, prevClientData)
    Tcl_Interp *interp;		/* Interpreter containing object. */
    char *objPtr;               /* pointer to the object record in memory */
    char *nameSpace;            /* name space where the traced object lives,
				 * such as the canvas name for a canvas item */
    char *objName;		/* Name of object. */
    char *attrName;		/* Name of object attribute */
    Oat_TraceProc *proc;	/* Procedure assocated with trace. */
    ClientData prevClientData;	/* If non-NULL, gives last value returned
				 * by this procedure, so this call will
				 * return the next trace after that one.
				 * If NULL, this call will return the
				 * first trace. */
{
    Tcl_HashEntry *entry;
    register OatTrace *tracePtr;
    OatTraceHeader* traceHeaderPtr;


    entry = Tcl_FindHashEntry(&oatHash, 
			      OatMakeHashKey(objPtr, attrName));
    if( entry == NULL ) {
      return NULL;
    } 

    /*
     * Find the relevant trace, if any, and return its clientData.
     */
    traceHeaderPtr = (OatTraceHeader*)Tcl_GetHashValue(entry);
    tracePtr = traceHeaderPtr->firstTracePtr;
    if (prevClientData != NULL) {
      for ( ; 
	    tracePtr != NULL; 
	    tracePtr = tracePtr->nextPtr) {
	if ((tracePtr->clientData == prevClientData) && 
	    (tracePtr->traceProc == proc)) {
	  tracePtr = tracePtr->nextPtr;
	  break;
	}
      }
    }

    for ( ; 
	  tracePtr != NULL; 
	  tracePtr = tracePtr->nextPtr) {
      if (tracePtr->traceProc == proc) {
	return tracePtr->clientData;
      }
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Oat_DeleteTrace --
 *
 *      Delete the trace on attribute `attrName' of object pointed to `objPtr'.
 *      `proc' and `clientData' of a trace must match the specified ones for 
 *      the trace to be deleted. If several traces match on these two fields, 
 *      only the first one is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If there exists a trace for the object and attribute given by
 *      objName and attrName with the given flags, proc, and clientData, 
 *      then that trace is removed.
 *
 *----------------------------------------------------------------------
 */

void
Oat_DeleteTrace(interp, objPtr, nameSpace, objName, attrName, flags, proc, clientData)
    Tcl_Interp *interp;		/* Interpreter containing object. */
    char *objPtr;               /* pointer to the object record in memory */
    char *nameSpace;            /* name space where the traced object lives,
				 * such as the canvas name for a canvas item */
    char *objName;		/* Name of object. */
    char *attrName;		/* Name of object attribute */
    int flags;			/* OR-ed collection of bits describing
				 * current trace, including any of
				 * TCL_TRACE_READS, TCL_TRACE_WRITES,
				 * TCL_TRACE_UNSETS, and TCL_GLOBAL_ONLY */
    Oat_TraceProc *proc;	/* Procedure assocated with trace. */
    ClientData clientData;	/* Arbitrary argument to pass to proc. */
{
    Tcl_HashEntry *entry;
    register OatTrace *tracePtr;
    OatTrace *prevPtr;
    OatTraceHeader* traceHeaderPtr;
    OatActiveTrace *activePtr;


    entry = Tcl_FindHashEntry(&oatHash, 
			      OatMakeHashKey(objPtr, attrName));
    if( entry == NULL ) {
      return;
    } 

    traceHeaderPtr = (OatTraceHeader*)Tcl_GetHashValue(entry);

    /* Clear all flags except READS, WRITES and UNSETS */
    flags &= (TCL_TRACE_READS | TCL_TRACE_WRITES | TCL_TRACE_UNSETS);

    /* Search thru the list of traces for this object-attribute pair */
    for (tracePtr = traceHeaderPtr->firstTracePtr, prevPtr = NULL; 
	 ;
	 prevPtr = tracePtr, tracePtr = tracePtr->nextPtr) {
      if (tracePtr == NULL) {
	return;
      }
      if ((tracePtr->traceProc == proc) && 
	  (tracePtr->flags == flags) && 
	  (clientData == NULL || tracePtr->clientData == clientData)) {
	break;
      }
    }

    /*
     * The code below makes it possible to delete traces while traces
     * are active:  it makes sure that the deleted trace won't be
     * processed by CallTraces.
     */
    for (activePtr = oatActiveTracePtr; 
	 activePtr != NULL;
	 activePtr = activePtr->nextPtr) {
      if (activePtr->nextTracePtr == tracePtr) {
	activePtr->nextTracePtr = tracePtr->nextPtr;
      }
    }

    if (prevPtr == NULL) {
	traceHeaderPtr->firstTracePtr = tracePtr->nextPtr;
    } else {
	prevPtr->nextPtr = tracePtr->nextPtr;
    }
    ckfree((char *) tracePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Oat_DeleteAllTraces --
 *
 *	Remove all traces for the the specified object and attribute.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If there exists a trace for the object and attribute given by
 *      objPtr and attrName, then that trace is removed.
 *
 *----------------------------------------------------------------------
 */

void
Oat_DeleteObjAttrTraces(interp, objPtr, attrName)
    Tcl_Interp *interp;		/* Interpreter containing object. */
    char *objPtr;               /* pointer to the object record in memory */
    char *attrName;		/* Name of object attribute;  NULL means
				 * trace applies to object as-a-whole. */
{
  Tcl_HashEntry *entry;
  register OatTrace *tracePtr;
  OatTrace *nextPtr;
  OatTraceHeader* traceHeaderPtr;
  OatActiveTrace *activePtr;
  
  
  entry = Tcl_FindHashEntry(&oatHash, 
			    OatMakeHashKey(objPtr, attrName));
  if( entry == NULL ) {
    return;
  } 
  
  traceHeaderPtr = (OatTraceHeader*)Tcl_GetHashValue(entry);
  
  /* Delete all traces for this object-attribute pair */
  for (tracePtr = traceHeaderPtr->firstTracePtr; 
       tracePtr != NULL;
       tracePtr = nextPtr) {
    /*
     * The code below makes it possible to delete traces while traces
     * are active:  it makes sure that the deleted trace won't be
     * processed by CallTraces.
       */
    for (activePtr = oatActiveTracePtr; 
	 activePtr != NULL;
	 activePtr = activePtr->nextPtr) {
      if (activePtr->nextTracePtr == tracePtr) {
	activePtr->nextTracePtr = tracePtr->nextPtr;
      }
    }
    nextPtr = tracePtr->nextPtr;
    ckfree((char *) tracePtr);
  }
}

/*
 *----------------------------------------------------------------------
 *
 * Oat_CallTraces --
 *
 *	This procedure is invoked to find and invoke relevant
 *	trace procedures associated with a particular operation on
 *	a object attribute.
 *
 * Results:
 *	The return value is NULL if no trace procedures were invoked, or
 *	if all the invoked trace procedures returned successfully.
 *	The return value is non-zero if a trace procedure returned an
 *	error (in this case no more trace procedures were invoked after
 *	the error was returned).  In this case the return value is a
 *	pointer to a static string describing the error.
 *
 * Side effects:
 *	Almost anything can happen, depending on trace;  this procedure
 *	itself doesn't have any side effects.
 *
 *----------------------------------------------------------------------
 */

char*
Oat_CallTraces(interp, objPtr, attrName, flags)
    Tcl_Interp *interp;			/* Interpreter containing object. */
    char* objPtr;                       /* Pointer to object record */
    char* attrName;                     /* Attribute name */
    int flags;				/* Flags to pass to trace procedures:
					 * indicates what's happening to
					 * object, plus other stuff like
					 * TCL_GLOBAL_ONLY and
					 * TCL_INTERP_DESTROYED. */
{
    Tcl_HashEntry*      entry;
    OatTraceHeader* traceHeaderPtr; /* Header of the list of
					 * OatTrace items. */
    OatTrace*       tracePtr;       /* Item in the trace list for
					 * specified object/attribute */
    OatActiveTrace  active;
    char *result;


    /* get an entry in the widget trace table; it may be null */
    entry = Tcl_FindHashEntry(&oatHash, 
			      OatMakeHashKey(objPtr, attrName));
    if (entry == NULL) 
      return NULL;

    traceHeaderPtr = (OatTraceHeader*)Tcl_GetHashValue(entry);

    /*
     * If there are already trace procedures active for the
     * object and attribute, don't call them again.
     */
    if (traceHeaderPtr->traceActive) 
      return NULL;

    traceHeaderPtr->traceActive = 1;
    traceHeaderPtr->refCount++;

    result = NULL;
    active.nextPtr = oatActiveTracePtr;
    oatActiveTracePtr = &active;

    active.traceHeaderPtr = traceHeaderPtr;
    for (tracePtr = traceHeaderPtr->firstTracePtr;
	 tracePtr != NULL;
	 tracePtr = tracePtr->nextPtr) {
      if (!(tracePtr->flags & flags)) {
	continue;
      }

      active.nextTracePtr = tracePtr->nextPtr;

      /* call the traceProc specified in the OatTrace item
       * with the first clientData argument specified in the item */
      result = (*tracePtr->traceProc)(tracePtr->clientData,
				      interp, 
				      traceHeaderPtr->nameSpace, 
				      traceHeaderPtr->objName, 
				      traceHeaderPtr->attrName, 
				      flags);

      if (result != NULL) {
#ifdef OAT_DEBUG
	fprintf(stderr, "Oat_CallTraces: traceProc result: %s\n", result);
#endif
	if (flags & TCL_TRACE_UNSETS) {
	  result = NULL;
	} else {
	  goto done;
	}
      }
    }

    /*
     * Restore the object trace's flags, remove the record of our active
     * traces, and then return.
     */

    done:

    traceHeaderPtr->traceActive = 0;
    traceHeaderPtr->refCount--;
    oatActiveTracePtr = active.nextPtr;
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Oat_CmdNameToPtr --
 *
 *	This procedure uses the Tcl command hash table to lookup
 *      up the object information by its name, and returns the 
 *      pointer to the object record.
 *
 * Results:
 *	Normally returns the pointer to the object record. If the
 *      object with the specified name does not exist, returns NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
char*
Oat_CmdNameToPtr(interp, objName)
Tcl_Interp *interp;		/* Interpreter in which object name 
				   is to be looked up. */
char *objName;		        /* Name of object. */
{
  Tcl_CmdInfo cmdInfo;

  if (! Tcl_GetCommandInfo(interp, objName, &cmdInfo)) {
    return NULL;
  }
  
  return (char *)(cmdInfo.clientData);
}


/*
 *----------------------------------------------------------------------
 *
 * OatMakeHashKey --
 *
 *	This procedure creates a string key for the trace hash table from
 *      the address of the object record and the name of the
 *      attribute.
 *
 * Results:
 *	Normally returns the pointer to the statically allocated string
 *      key.
 *
 * Side effects:
 *	Statically allocated string key buffer is overwritten.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
char*
OatMakeHashKey(objPtr, attrName)
char *objPtr;		        /* Pointer to the object record (NOT name) */
char *attrName;		        /* Name of attribute. */
{
  static char hashKeyStr[100]; /* 100 bytes is enough for 64-bit hex addresses */

  sprintf(hashKeyStr, "%#.8lx.%s", (unsigned long)objPtr, attrName);
  
  return hashKeyStr;
}

/*
 *----------------------------------------------------------------------
 *
 * OatMakeAttrListHashKey --
 *
 *	This procedure creates a string key for the trace hash table from
 *      the address of the object record and the name of the
 *      attribute.
 *
 * Results:
 *	Normally returns the pointer to the statically allocated string
 *      key.
 *
 * Side effects:
 *	Statically allocated string key buffer is overwritten.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
char *
OatMakeAttrListHashKey(objPtr)
char *objPtr;		/* Pointer to the object record */
{
  static char hashKeyStr[100];

  sprintf(hashKeyStr, "%#.8lx_%s", (unsigned long)objPtr, "attrs");
  
  return hashKeyStr;
}


/*
 *----------------------------------------------------------------------
 *
 * OatTraceProc --
 *
 *	This procedure is called to handle object attribute operations
 *      that have been traced using the "trace" command. This function 
 *      expects the `clientData' to point to a valid Oat_TraceInfo structure.
 *
 * Results:
 *	Normally returns NULL.  If the trace command returns an error,
 *	then this procedure returns an error string.
 *
 * Side effects:
 *	Depends on the command associated with the trace.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
char*
OatTraceProc(clientData, interp, nameSpace, objName, attrName, flags)
    ClientData clientData;	/* Information about the object attribute trace. */
    Tcl_Interp *interp;		/* Interpreter containing object. */
    char *nameSpace;
    char *objName;		/* Name of object. */
    char *attrName;		/* Name of object attribute;  NULL means ? */
    int flags;			/* OR-ed bits giving operation and other
				 * information. */
{
    Oat_TraceInfo *trInfoPtr = (Oat_TraceInfo *)clientData;
    char *result;
    int code;
    Interp dummy;
    Tcl_DString cmd;

    result = NULL;
    if (trInfoPtr->errMsg != NULL) {
	ckfree(trInfoPtr->errMsg);
	trInfoPtr->errMsg = NULL;
    }
    if ((trInfoPtr->flags & flags) && !(flags & TCL_INTERP_DESTROYED)) {

	/*
	 * Generate a command to execute by appending list elements
	 * for the object and attribute names and the operation.  The five
	 * extra characters are for three space, the opcode character,
	 * and the terminating null.
	 */

	if (attrName == NULL) {
	    attrName = "";
	}
	Tcl_DStringInit(&cmd);
	Tcl_DStringAppend(&cmd, trInfoPtr->command, trInfoPtr->length);
	Tcl_DStringAppendElement(&cmd, nameSpace);
	Tcl_DStringAppendElement(&cmd, objName);
	Tcl_DStringAppendElement(&cmd, attrName);
	if (flags & TCL_TRACE_READS) {
	    Tcl_DStringAppend(&cmd, " r", 2);
	} else if (flags & TCL_TRACE_WRITES) {
	    Tcl_DStringAppend(&cmd, " w", 2);
	} else if (flags & TCL_TRACE_UNSETS) {
	    Tcl_DStringAppend(&cmd, " u", 2);
	}

	if (OatDebugFlag) {
	  fprintf(stderr, "Oat_CallTraces cmd: %s\n", Tcl_DStringValue(&cmd));
	}

	/*
	 * Execute the command.  Be careful to save and restore the
	 * result from the interpreter used for the command.
	 */

	if (interp->freeProc == 0) {
	    dummy.freeProc = (Tcl_FreeProc *) 0;
	    dummy.result = "";
	    Tcl_SetResult((Tcl_Interp *) &dummy, interp->result, TCL_VOLATILE);
	} else {
	    dummy.freeProc = interp->freeProc;
	    dummy.result = interp->result;
	    interp->freeProc = (Tcl_FreeProc *) 0;
	}
	code = Tcl_Eval(interp, Tcl_DStringValue(&cmd));
	Tcl_DStringFree(&cmd);
	if (code != TCL_OK) {
	    trInfoPtr->errMsg = (char*)ckstrdup(interp->result);
	    result = trInfoPtr->errMsg;
	    Tcl_ResetResult(interp);		/* Must clear error state. */
	}
	Tcl_SetResult(interp, dummy.result,
		(dummy.freeProc == 0) ? TCL_VOLATILE : dummy.freeProc);
    }
    if (flags & TCL_TRACE_DESTROYED) {
	result = NULL;
	if (trInfoPtr->errMsg != NULL) {
	    ckfree(trInfoPtr->errMsg);
	}
	ckfree((char *) trInfoPtr);
    }
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * OatGetTracedAttr --
 *
 *	If some of the object's attributes are being traced, then 
 *      returns either the name of the first traced attribute (if isFirst
 *      is True), or the next name or NULL (if isFirst is False).
 *      Otherwise, i.e., no object attributes are being traced,
 *      returns NULL.
 *
 * Results:
 *	The return value is a name of the first/next traced attribute
 *      for the specified object, or NULL.
 *
 * Side effects:
 *	Local static variable that remembers the last returned name
 *      in the attribute list is modified and set to the next name
 *      in the list (or to NULL).
 *
 *--------------------------------------------------------------
 */

static
char*
OatGetTracedAttr(objPtr, isFirst)
    char *objPtr;		/* Pointer to the object record */
    int isFirst;
{
  Tcl_HashEntry *entry;
  static OatTracedAttr* tracedAttrPtr = NULL;

  if (isFirst) {
    /* get an entry in the widget trace table; it may be null */
    entry = Tcl_FindHashEntry(&oatHash, 
			      OatMakeAttrListHashKey(objPtr));

    if (entry == NULL) {
      /* this object is not being traced */
      return NULL;
    }

    tracedAttrPtr = (OatTracedAttr *)Tcl_GetHashValue(entry);
  } else {
    if (tracedAttrPtr != NULL)
      tracedAttrPtr = tracedAttrPtr->nextPtr;
  }

  return (tracedAttrPtr != NULL ? tracedAttrPtr->attrName : NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * OatAllocTraceHeader --
 *
 *	Allocate and initialize the header of the list of OatTrace
 *      structures.
 *
 * Results:
 *	A pointer to allocated structure.
 *
 * Side effects:
 *	Space is allocated.
 *
 *----------------------------------------------------------------------
 */
OatTraceHeader* 
OatAllocTraceHeader(nameSpace, objName, attrName)
    char *nameSpace;            /* name space where the traced object lives,
				 * such as the canvas name for a canvas item */
    char *objName;		/* Name of traced object. */
    char *attrName;		/* Name of object attribute;  NULL means
				 * trace applies to object as-a-whole. */
{
  OatTraceHeader* traceHeaderPtr;


  traceHeaderPtr = 
    (OatTraceHeader *)ckalloc(sizeof(OatTraceHeader));

  traceHeaderPtr->nameSpace = ckstrdup(nameSpace);
  traceHeaderPtr->objName   = ckstrdup(objName);
  traceHeaderPtr->attrName  = ckstrdup(attrName);
  
  traceHeaderPtr->traceActive = 0;
  traceHeaderPtr->refCount = 0;
  
  return traceHeaderPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Oat_AllocTraceInfo --
 *
 *	Allocate and initialize the object trace command structure.
 *
 * Results:
 *	A pointer to allocated structure.
 *
 * Side effects:
 *	Space is allocated.
 *
 *----------------------------------------------------------------------
 */
Oat_TraceInfo* 
Oat_AllocTraceInfo(flags, cmd) 
  int flags;			/* OR-ed collection of bits, including any
				 * of TCL_TRACE_READS, TCL_TRACE_WRITES,
				 * TCL_TRACE_UNSETS. */
  char* cmd;                    /* command string */
{
  Oat_TraceInfo* trInfoPtr;
  int cmdLength;
  unsigned neededBytes;

  cmdLength = strlen(cmd);
  neededBytes = sizeof(Oat_TraceInfo) - sizeof(trInfoPtr->command)
                + cmdLength + 1;
  trInfoPtr = (Oat_TraceInfo *) ckalloc(neededBytes);

  trInfoPtr->flags = flags;
  trInfoPtr->errMsg = NULL;
  trInfoPtr->length = cmdLength;
  flags |= TCL_TRACE_UNSETS;
  strcpy(trInfoPtr->command, cmd);

  return trInfoPtr;
}


char* Oat_FirstTracedAttr(objPtr)
    char *objPtr;		/* Pointer to the object record */
{
  return OatGetTracedAttr(objPtr, 1);
}

char* Oat_NextTracedAttr(objPtr)
    char *objPtr;		/* Pointer to the object record */
{
  return OatGetTracedAttr(objPtr, 0);
}


int Oat_OpStringToFlags(opStr) 
char* opStr;
{
  int flags = 0;
  char *p;

  for (p = opStr; *p != 0; p++) {
    if (*p == 'r') {
      flags |= TCL_TRACE_READS;
    } else if (*p == 'w') {
      flags |= TCL_TRACE_WRITES;
    } else if (*p == 'u') {
      flags |= TCL_TRACE_UNSETS;
    } else {
      return 0;
    }
  }
  return flags;
}

char* Oat_OpFlagsToString(flags) 
int flags;
{
  static char ops[4];
  char *p;
	
  p = ops;
  if (flags & TCL_TRACE_READS)  *p++ = 'r';
  if (flags & TCL_TRACE_WRITES) *p++ = 'w';
  if (flags & TCL_TRACE_UNSETS) *p++ = 'u';
  *p = '\0';

  return ops;
}

static 
char* 
ckstrdup(src)
char *src;
{
  char* dest;

  dest = ckalloc(strlen(src)+1);
  strcpy(dest, src);
  return dest;
}
