/*
 * oat.h --
 *
 *	External declarations for OAT - Object Attribute Trace
 *      extension to Tcl.
 *
 *
 * Copyright (c) 1996 University of Minnesota.
 *
 * RCS: $Id: oat.h,v 1.9 1996/11/03 02:11:31 safonov Exp safonov $
 */

#ifndef _OAT_H
#define _OAT_H

#define OAT_MAJOR_VERSION   1
#define OAT_MINOR_VERSION   0

#define OAT_PKGNAME         "oat"
#define OAT_VERSION         "1.0"
#define OAT_PATCH_LEVEL     "1.0"


#ifndef _TCL
#include <tcl.h>
#endif

#if defined(__WIN32__)
#   define WIN32_LEAN_AND_MEAN
#   include <windows.h>
#   undef WIN32_LEAN_AND_MEAN

/*
 * VC++ has an alternate entry point called DllMain, so we need to rename
 * our entry point.
 */

#   if defined(_MSC_VER)
#	define EXPORT(a,b) __declspec(dllexport) a b
#	define DllEntryPoint DllMain
#   else
#	if defined(__BORLANDC__)
#	    define EXPORT(a,b) a _export b
#	else
#	    define EXPORT(a,b) a b
#	endif
#   endif
#else
#   define EXPORT(a,b) a b
#endif

/*------------------------------------------------------------------*
 * Mechanism to define new traceable object types.
 *------------------------------------------------------------------*/


/* 
 * Type definition for function called to process the "trace" Tcl command 
 * for a specific traceable object type. Tcl_TraceCmd() is of this type
 * and is used to process the "trace" Tcl command for Tcl variables.
 */

typedef int (Oat_CmdProc) _ANSI_ARGS_((ClientData dummy, 
				      Tcl_Interp* interp, 
				      int argc, char* argv[]));

/* 
 * One such structure needs to be defined and passed to function
 * Oat_CreateTraceableObjType() for every new traceable object type
 * being created.
 */

typedef struct Oat_TraceableObjType {
  char* nameKWord;
  char* infoKWord;
  char* deleteKWord;
  Oat_CmdProc* traceCmdProc;    /* e.g., Tcl_TraceCmd() for Tcl variables */
  struct Oat_TraceableObjType* nextPtr;
} Oat_TraceableObjType;

/* 
 * This procedure is invoked by the traceable object manager to tell Tcl
 * about a new kind of traceable object and the procedures that manage 
 * the new type.
 * The procedure is typically invoked during Tcl_AppInit.
 */

EXTERN void		Oat_CreateTraceableObjType _ANSI_ARGS_((
			    Oat_TraceableObjType *typePtr));


/*------------------------------------------------------------------*/

/* Type definition for C function to call when specified ops are
 * invoked upon object attribute specified by nameSpace, objName and attrName.
 * See typedef Tcl_VarTraceProc in tcl.h.
 */

typedef char* (Oat_TraceProc) _ANSI_ARGS_((ClientData clientData,
        Tcl_Interp *interp, char *nameSpace, char *objName, char *attrName, 
        int flags));

/*
 * Structure used to hold information about object traces:
 * passed to procedure stored in OatTrace
 */

typedef struct Oat_TraceInfo {
    int flags;			/* Operations for which Tcl command is
				 * to be invoked. */
    char *errMsg;		/* Error message returned from Tcl command,
				 * or NULL.  Malloc'ed. */
    int length;			/* Number of non-NULL chars. in command. */
    char command[4];		/* Space for Tcl command to invoke.  Actual
				 * size will be as large as necessary to
				 * hold command.  This field must be the
				 * last in the structure, so that it can
				 * be larger than 4 bytes. */
} Oat_TraceInfo;


/*
 * Object trace function prototypes
 */

/* 
 * Process the extended "trace" command, call the Oat_CmdProc function
 * for the appropriate traceable object type.
 */

EXTERN int	        Oat_TraceCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char **argv));

EXTERN Oat_TraceInfo*   Oat_AllocTraceInfo _ANSI_ARGS_((int flags, char* cmd));

/* 
 * Arrange for C function `proc' to be called with argument `clientData'
 * when operations specified in `flags' are invoked on attribute `attrName'
 * of object pointed to by `objPtr'.
 */

EXTERN int              Oat_CreateTrace _ANSI_ARGS_((Tcl_Interp *interp, 
                            char *objPtr,
			    char *nameSpace, char *objName, char *attrName, 
			    int flags, 
			    Oat_TraceProc *proc, ClientData clientData));

/* 
 * Return the `clientData' field of the trace on an attribute of an object 
 * with the matching values of clientData and flags. If `clientData' passed
 * to the function is NULL, the first trace matching on value of `flags'
 * is returned.
 */

EXTERN ClientData       Oat_QueryTrace _ANSI_ARGS_((Tcl_Interp *interp,
                            char *objPtr, 
 			    char *nameSpace, char *objName, char *attrName, 
			    Oat_TraceProc *proc, 
			    ClientData prevClientData));

/* 
 * Delete the trace on attribute `attrName' of object pointed to `objPtr'.
 * `proc' and `clientData' fields of a trace must match the specified ones 
 * for the trace to be deleted. If several traces match on these two fields, 
 * only the first one is deleted.
 */

EXTERN void             Oat_DeleteTrace _ANSI_ARGS_((Tcl_Interp *interp,
                            char *objPtr, 
 			    char *nameSpace, char *objName, char *attrName, 
			    int flags, Oat_TraceProc *proc, 
                            ClientData clientData));

/* 
 * This function is set up by extended "trace" Tcl command to be called
 * whenever one of the operations in `flags' is performed to attribute
 * `attrName' of object `objName' in namespace `nameSpace'. It expects 
 * `clientData' to point to a valid Oat_TraceInfo structure.
 */

EXTERN char*            OatTraceProc _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, char *nameSpace, 
                            char *objName, char *attrName,
			    int flags));

/* 
 * The following two functions allow to iterate through the list
 * of all traces on attributes of an object. Used when cleaning
 * up traces when an object is destroyed.
 */

EXTERN char*            Oat_FirstTracedAttr _ANSI_ARGS_((char* objName));
EXTERN char*            Oat_NextTracedAttr  _ANSI_ARGS_((char* objName));

/* 
 * This function deletes all traces for the specified object
 * and attribute. 
 */

EXTERN void             Oat_DeleteObjAttrTraces _ANSI_ARGS_((
			    Tcl_Interp *interp, char *objPtr, char *attrName));

/* 
 * Oat_CallTraces calls the C function stored in each trace on
 * the specified object and attribute.
 */

EXTERN char*            Oat_CallTraces _ANSI_ARGS_((Tcl_Interp *interp, 
                            char* objPtr, char* attrName, int flags));

/* 
 * Converts a Tcl command name (such as a widget name) 
 * into a pointer to the structure storing this command's information.
 */

EXTERN char*            Oat_CmdNameToPtr _ANSI_ARGS_((Tcl_Interp *interp,
			    char *objName));

/* 
 * Convert back and forth between string and OR-ed bit representations
 * of trace operations. Bit representations are TCL_TRACE_WRITES etc.
 */

EXTERN int              Oat_OpStringToFlags _ANSI_ARGS_((char* opStr));
EXTERN char*            Oat_OpFlagsToString _ANSI_ARGS_((int flags));


#endif   /* _OAT_H */
