/* 
 * tkoat.c --
 *
 *	This file contains the functions needed to support traces on 
 *	Tk widget and canvas item attributes.
 *
 * Copyright (c) 1987-1993 The Regents of the University of California.
 * Copyright (c) 1994-1996 Sun Microsystems, Inc.
 * Copyright (c) 1996 University of Minnesota.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS:  $Id: tkoat.c,v 1.5 1996/11/03 02:24:08 safonov Exp safonov $ 
 */

#ifndef _TKOAT_H
#define _TKOAT_H

#define TKOAT_MAJOR_VERSION   1
#define TKOAT_MINOR_VERSION   0

#define TKOAT_VERSION         "1.0"
#define TKOAT_PATCH_LEVEL     "1.0"


#include <tcl.h>
#include <oat.h>
#include <tk.h>
#include <tkCanvas.h>


EXTERN EXPORT(int,Tkoat_Init) _ANSI_ARGS_((Tcl_Interp *interp));


/*
 *----------------------------------------------------------------------
 *
 * DllEntryPoint --
 *
 *	This wrapper function is used by Windows to invoke the
 *	initialization code for the DLL.  If we are compiling
 *	with Visual C++, this routine will be renamed to DllMain.
 *	routine.
 *
 * Results:
 *	Returns TRUE;
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

#ifdef __WIN32__
BOOL APIENTRY
DllEntryPoint(hInst, reason, reserved)
    HINSTANCE hInst;		/* Library instance handle. */
    DWORD reason;		/* Reason this function is being called. */
    LPVOID reserved;		/* Not used. */
{
    return TRUE;
}
#endif

Oat_CmdProc Tk_WidgetTraceCmd;

Oat_TraceableObjType Oat_TraceableWidgetType = {
    "widget",			   /* nameKWord */
    "winfo",			   /* infoKWord */
    "wdelete",			   /* deleteKWord */
    Tk_WidgetTraceCmd,             /* traceCmdProc */
    (Oat_TraceableObjType *) NULL  /* nextPtr */
};

Oat_CmdProc Tk_CitemTraceCmd;

Oat_TraceableObjType Oat_TraceableCitemType = {
    "citem",			   /* nameKWord */
    "cinfo",			   /* infoKWord */
    "cdelete",			   /* deleteKWord */
    Tk_CitemTraceCmd,              /* traceCmdProc */
    (Oat_TraceableObjType *) NULL  /* nextPtr */
};

/*
 *----------------------------------------------------------------------
 *
 * Tkoat_Init --
 *
 *	This procedure creates new traceable object types, widget and
 *      canvas item.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

EXPORT(int,Tkoat_Init)(interp)
    Tcl_Interp *interp;
{
#if 0
    if (Tcl_PkgRequire(interp, OAT_PKGNAME, OAT_VERSION, /*exact=*/0) == 0)
	return TCL_ERROR;
#endif

    Oat_CreateTraceableObjType(&Oat_TraceableWidgetType);
    Oat_CreateTraceableObjType(&Oat_TraceableCitemType);

    return Tcl_PkgProvide(interp, "tkoat", TKOAT_VERSION);
}


/*
 *----------------------------------------------------------------------
 *
 * Tk_WidgetTraceCmd --
 *
 *	This procedure is invoked to process the "trace" Tcl command
 *	invoked on widget objects.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Tk_WidgetTraceCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int c;
    size_t length;
    char* nameSpace;
    char* objName;
    char* objPtr;
    char* attrName;
    char* opStr;
    int   flags;
    char* cmd;
    int   cmdLength;
    ClientData clientData;
    Oat_TraceInfo* trInfoPtr;

    length = strlen(argv[1]);
    c = argv[1][2];
    objName = argv[2];
    attrName = argv[3];
    
    if ((c == 'd') && 
	(strncmp(argv[1], "widget", length) == 0) && 
	(length >= 3)) {

      if (argc != 6) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " widget name attr ops command\"", 
			 (char *) NULL);
	return TCL_ERROR;
      }

      /* 
       * get address of the widget record from the command information 
       */

      objPtr = Oat_CmdNameToPtr(interp, objName);
      if (objPtr == NULL) {
	Tcl_AppendResult(interp, "trace widget: widget \"",
			 objName, "\" does not exist", (char *) NULL);
	return TCL_ERROR;
      }
      
      opStr = argv[4];
      cmd = argv[5];
      
      flags = Oat_OpStringToFlags(opStr);
      if (flags == 0) {
	goto badOps;
      }
      
      /* 
       * create Oat_TraceInfo record to be passed to OatTraceProc() 
       */

      trInfoPtr = Oat_AllocTraceInfo(flags, cmd);
      
      /* 
       * set up for OatTraceProc() to be called with trInfoPtr
       * when ops specified in flags are applied to objName.attrName 
       */

      if (Oat_CreateTrace(interp, objPtr, 
			  "", objName, attrName, flags, OatTraceProc,
			  (ClientData) trInfoPtr) != TCL_OK) {
	ckfree((char *) trInfoPtr);
	return TCL_ERROR;
      }
    } else if ((c == 'n') && 
	       (strncmp(argv[1], "winfo", length) == 0) && 
	       (length >= 3)) {
      
      char *prefix = "{";
      
      if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " winfo name attr\"", (char *) NULL);
	return TCL_ERROR;
      }
      
      /* get address of the widget record from the command information */
      objPtr = Oat_CmdNameToPtr(interp, objName);
      if (objPtr == NULL) {
	/* widget does not exist - no traces */
	return TCL_OK;
      }
      
      clientData = 0;
      while ((clientData = Oat_QueryTrace(interp, objPtr,
						"", objName, attrName,
						OatTraceProc, clientData)) != 0) {
	trInfoPtr = (Oat_TraceInfo *)clientData;
	
	Tcl_AppendResult(interp, prefix, (char *) NULL);
	Tcl_AppendElement(interp, Oat_OpFlagsToString(trInfoPtr->flags));
	Tcl_AppendElement(interp, trInfoPtr->command);
	Tcl_AppendResult(interp, "}", (char *) NULL);
	prefix = " {";
      }
    } else if ((c == 'e') && 
	       (strncmp(argv[1], "wdelete", length) == 0) && 
	       (length >= 3)) {
      
      if (argc != 6) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " wdelete name attr ops command\"", 
			 (char *) NULL);
	return TCL_ERROR;
      }
      
      /* get address of the widget record from the command information */
      objPtr = Oat_CmdNameToPtr(interp, objName);
      if (objPtr == NULL) {
	/* widget does not exist - no traces */
	return TCL_OK;
      }
      
      opStr = argv[4];
      cmd = argv[5];
      
      flags = Oat_OpStringToFlags(opStr);
      if (flags == 0) {
	goto badOps;
      }
      /*
       * Search through all of our traces on this widget attribute to
       * see if there's one with the given command.  If so, then
       * delete the first one that matches.
       */
      
      cmdLength = strlen(cmd);
      clientData = 0;
      while ((clientData = 
	      Oat_QueryTrace(interp, objPtr, "", objName, attrName,
				      OatTraceProc, clientData)) != 0) {
	trInfoPtr = (Oat_TraceInfo *)clientData;
	
	if ((trInfoPtr->length == cmdLength) && (trInfoPtr->flags == flags) && 
	    (strncmp(cmd, trInfoPtr->command, (size_t) cmdLength) == 0)) {
	  Oat_DeleteTrace(interp, objPtr, "", objName, attrName,
			     flags /* | TCL_TRACE_UNSETS*/,
			     OatTraceProc, clientData);
	  if (trInfoPtr->errMsg != NULL) {
	    ckfree(trInfoPtr->errMsg);
	  }
	  ckfree((char *) trInfoPtr);
	  break;
	}
      }
    } else {
      Tcl_AppendResult(interp, "bad option \"", argv[1],
		       "\": should be widget, winfo, or wdelete",
		       (char *) NULL);
      return TCL_ERROR;
    }

    return TCL_OK;

    badOps:
    Tcl_AppendResult(interp, "bad operations \"", opStr,
	    "\": should be one or more of rwu", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_CitemTraceCmd --
 *
 *	This procedure is invoked to process the "trace" Tcl command
 *	invoked on canvas item objects.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Tk_CitemTraceCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int c;
    size_t length;
    char* canvName;
    TkCanvas *canvPtr;
    char* objName;
    char* objPtr;
    TagSearch tagSearch;
    Tk_Item *itemPtr = NULL;
    char* attrName;
    char* opStr;
    int   flags;
    char* cmd;
    int   cmdLength;
    ClientData clientData;
    Oat_TraceInfo* trInfoPtr;


    length = strlen(argv[1]);
    c = argv[1][2];
    canvName = argv[2];
    objName = argv[3];
    attrName = argv[4];
    
    if ((c == 't') && 
	(strncmp(argv[1], "citem", length) == 0) && 
	(length >= 3)) {

      if (argc != 7) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " citem canvname tagOrId attr ops command\"", 
			 (char *) NULL);
	return TCL_ERROR;
      }
      
      /* get address of the canvas record from the command information */
      canvPtr = (TkCanvas*)Oat_CmdNameToPtr(interp, canvName);
      if (canvPtr == NULL) {
	Tcl_AppendResult(interp, "trace citem: canvas \"",
			 canvName, "\" does not exist", (char *) NULL);
	return TCL_ERROR;
      }
      
      opStr = argv[5];
      cmd = argv[6];
      
      flags = Oat_OpStringToFlags(opStr);
      if (flags == 0) {
	goto badOps;
      }
      
      for (itemPtr = StartTagSearch(canvPtr, objName, &tagSearch);
	   itemPtr != NULL; 
	   itemPtr = NextItem(&tagSearch)) {
	
	trInfoPtr = Oat_AllocTraceInfo(flags, cmd);
	
	if (Oat_CreateTrace(interp, (char*)itemPtr, 
			    canvName, objName, attrName, 
			    flags, OatTraceProc,
			    (ClientData) trInfoPtr) != TCL_OK) {
	  ckfree((char *) trInfoPtr);
	  return TCL_ERROR;
	}
      }
    } else if ((c == 'n') && 
	       (strncmp(argv[1], "cinfo", length) == 0) && 
	       (length >= 3)) {
      
      char *prefix = "{";
      
      if (argc != 5) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], argv[1], " canvName tagOrId attr\"", (char *) NULL);
	return TCL_ERROR;
      }
      
      /* get address of the canvas record from the command information */
      canvPtr = (TkCanvas*)Oat_CmdNameToPtr(interp, canvName);
      if (canvPtr == NULL) {
	/* canvas does not exist - no traces */
	return TCL_OK;
      }
      
      for (itemPtr = StartTagSearch(canvPtr, objName, &tagSearch);
	   itemPtr != NULL; 
	   itemPtr = NextItem(&tagSearch)) {
	
	clientData = 0;
	while ((clientData = 
		Oat_QueryTrace(interp, (char*)itemPtr,
				     canvName, objName, attrName,
				     OatTraceProc, clientData)) != 0) {
	  trInfoPtr = (Oat_TraceInfo *)clientData;
	  
	  Tcl_AppendResult(interp, prefix, (char *) NULL);
	  Tcl_AppendElement(interp, Oat_OpFlagsToString(trInfoPtr->flags));
	  Tcl_AppendElement(interp, trInfoPtr->command);
	  Tcl_AppendResult(interp, "}", (char *) NULL);
	  prefix = " {";
	} 
      }
    } else if ((c == 'e') && 
	       (strncmp(argv[1], "cdelete", length) == 0) && 
	       (length >= 3)) {
      
      if (argc != 7) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], argv[1], " canvname tagOrId attr ops command\"", 
			 (char *) NULL);
	return TCL_ERROR;
      }
      
      /* get address of the canvas record from the command information */
      canvPtr = (TkCanvas*)Oat_CmdNameToPtr(interp, canvName);
      if (canvPtr == NULL) {
	/* canvas does not exist - no traces */
	return TCL_OK;
      }
      
      opStr = argv[5];
      cmd = argv[6];
      
      flags = Oat_OpStringToFlags(opStr);
      if (flags == 0) {
	goto badOps;
      }
      
      /*
       * Search through all of our traces on this canvas item attribute to
       * see if there's one with the given command.  If so, then
       * delete the first one that matches.
       */
      
      cmdLength = strlen(cmd);
      for (itemPtr = StartTagSearch(canvPtr, objName, &tagSearch);
	   itemPtr != NULL; 
	   itemPtr = NextItem(&tagSearch)) {
	
	clientData = 0;
	while ((clientData = 
		Oat_QueryTrace(interp, (char*)itemPtr, 
				     canvName, objName, attrName,
				     OatTraceProc, clientData)) != 0) {
	  trInfoPtr = (Oat_TraceInfo *)clientData;
	  
	  if ((trInfoPtr->length == cmdLength) && 
	      (trInfoPtr->flags == flags) && 
	      (strncmp(cmd, trInfoPtr->command, (size_t)cmdLength) == 0)) {
	    Oat_DeleteTrace(interp, (char*)itemPtr, 
				  canvName, objName, attrName,
				  flags /*| TCL_TRACE_UNSETS*/,
				  OatTraceProc, clientData);
	    if (trInfoPtr->errMsg != NULL) {
	      ckfree(trInfoPtr->errMsg);
	    }
	    ckfree((char *) trInfoPtr);
	    break;
	  }
	}
      }
    } else {
      Tcl_AppendResult(interp, "bad option \"", argv[1],
		       "\": should be citem, cinfo, or cdelete",
		       (char *) NULL);
      return TCL_ERROR;
    }

    return TCL_OK;

    badOps:
    Tcl_AppendResult(interp, "bad operations \"", opStr,
	    "\": should be one or more of rwu", (char *) NULL);
    return TCL_ERROR;
}

#endif /* _TKOAT_H */
