# apptalk v1.0b1
#
# AppTalk - Tk-based inter-application communication interface
#
# Copyright (c) 1996 by Healthcare Communications, Inc. (HCI).
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, is permitted provided that the following conditions
# are met:
# 
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. All advertising materials mentioning features or use of this software
#    must display the following acknowledgement:
# 
#    This product includes software developed by Healthcare Communications, 
#    Inc.
# 
# 4. The name of HCI may not be used to endorse or promote 
#    products derived from this software without specific prior written
#    permission.
# 
# Disclaimer:
# 
# THIS SOFTWARE IS PROVIDED BY Healthcare Communications, Inc. (HCI)
# ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL HCI BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
# BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
# OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
# IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

######################################################################
# apptalk - interapplication communication interface
#
# let's define a couple of terms so we all understand what we are
# talking about.
#
# SERVER refers to an application that services apptalk requests.
#
# CLIENT refers to an application that requests services from an
#        apptalk enabled application (ie: a SERVER)
#
# TARGET  an identifier for a class of application. Typically the
#        TARGET is the name of the script which defines a SERVER
#        application. It is *not* the internal id used by tk's 
#        send command, and doesn't necessarily have to be the name
#        of the script.
#
# usage, server side:
#        apptalk init <target>
#        apptalk proc name documentation body
#
# usage, client side:
#        apptalk connect <target> ?command to exec?
#        @<target> <command>
#
# "apptalk connect" will create a proc named @<target>. For example, 
# "apptalk connect textedit" will create a command named
# @textedit. This command can be used to send commands to the
# server. For example, "@textedit commands" will request that the
# server return a list of commands to the client.
#
# this code uses one private global array, and two public global variables:
#    _apptalkData()   contains misc. data and code used by the server. 
#    apptalkServer    name of the current server
#    apptalkClient    name of the client which sent a command to the
#                     server
#
#
# a random note: apptalk commands are implemented as data stored
# in _apptalkData(code:?command?). Apptalk server commands are stored as 
# data indexed by code:?server?:?command?. For example, the code to
# implement the apptalk command "connect" is stored in 
# _apptalkData(code:connect). A server command such as "popup" for the 
# server named "foo" would be stored in _apptalkData(code:foo:popup).
#
# why is code stored as data? It keeps the namespace clutter to a 
# minumum, it's easy to generate a list of commands for a server using
# "array names _apptalkData(code:?server?:*)", and it's faster than
# using procs.

# The help information for each command is stored in _apptalkData, with
# the index info:?server?:?command?. 
#
# Internal commands used to implement the apptalk infrastructure are 
# normal procs named apptalk:?command? (eg: apptalk:popNotify).


package provide apptalk 1.0

######################################################################
# default resources used by these procs
######################################################################
#
# apptalk policy specifies how apptalk finds an app to talk to. 
#    ask    - presents a list of suitable apps for the user to pick 
#             from. If the list is exactly one item long, it will
#             use that app instead of asking.
#    first  - picks the first suitable app it finds (sorta random)
#    new    - always starts up a new instance of a program
#
option add "*apptalkPolicy" ask 0

######################################################################
# this is kinda confusing, but this single command works both on
# the client side and server side. So, pay attention! It's really
# quite simple -- it just looks to see if a command is defined and
# evals the appropriate code.
#
#
# the following is for tclX package support:
#
# PACKAGE-EXPORT: apptalk

proc apptalk {action args} {
    global _apptalkData
    global apptalkServer
    global apptalkClient

    if { [info exists _apptalkData(code:$action)] } {
	# ahhh.... life is gooooood. :-)
	eval $_apptalkData(code:$action)

    } else {
	error "unknown apptalk command $action"
    }
}

#---------------------------------------------------------------------
# init - this initializes the application which can receive commands
#        (ie: the server)
#
set _apptalkData(code:init) {
    set server  [lindex $args 0]
    set command [lindex $args 1]
    set args [lrange $args 2 end]
    
    # set a variable the calling program can use
    set apptalkServer $server

    set _apptalkData($apptalkServer:ident) $server

    # define the code for our server proc. It doesn't do much in
    # and of itself; it's main duty in life is to eval code we've
    # stuck in an array. Notice how similar it is to the "apptalk"
    # command. This was no accident.
    proc $server {action args} {
	global _apptalkData
	global apptalkClient
	global apptalkServer

	if { [info exists _apptalkData(code:$apptalkServer:$action)] } {
	    eval $_apptalkData(code:$apptalkServer:$action)

	} else {
	    # punt!
	    set message "Unknown command \"$action\"."
	    append message "\nKnown commands are "
	    append message "[apptalk:commands $apptalkServer]"
	    error $message
	}
    }

    #-----------------------------------------------------------------
    # the info command
    #-----------------------------------------------------------------
    apptalk proc info {
	usage: %N info ?command?
	Requests %N to display usage information on the named command.
	If no command is given, information about all known commands
	is returned.
    } {
	if {[llength $args] == 0} {
	    set commands [array names _apptalkData "code:$apptalkServer:*"]
	    regsub -all "code:$apptalkServer:" $commands {} commands
	} else {
	    if {![info exists _apptalkData(code:$apptalkServer:$args)]} {
		set message "$args is not a valid command for $apptalkServer."
		append message "\nKnown commands are "
		append message "[apptalk:commands $apptalkServer]"
		error $message
	    }
	    set commands $args
	}

	# for each command, try to find the documentation string;
	# if not found use *undocumented* instead.
	foreach command $commands {
	    set foobar($command) {}
	    if {[info exists _apptalkData(info:$apptalkServer:$command)]} {
		# this nonsense is to trim all the undesireable
		# whitespace and do some substitutions
		set tmp [string trim \
			     $_apptalkData(info:$apptalkServer:$command)]
		regsub -all "\n\[ \t\]*" $tmp "\n" tmp
		regsub -all "%N" $tmp "@$apptalkServer" foobar($command)
	    } else {
		set foobar($command) "*undocumented*"
	    }
	}

	# ok, for each thing we found, print the command on one line
	# and the definition following but indented four spaces
	set result {}
	foreach item [lsort [array names foobar]] {
	    append result "$item:\n"
	    foreach line [split $foobar($item) "\n"] {
		append result "    $line\n"
	    }
	    append result "\n"
	}
	return [string trimright $result]
    }

    #-----------------------------------------------------------------
    # the "commands" command
    #-----------------------------------------------------------------
    apptalk proc commands {
	usage: %N commands
	returns a list of apptalk commands known by %N
    } {
	set commands  [lsort [array names _apptalkData \
				  "code:$apptalkServer:*"]]
	regsub -all "code:$apptalkServer:" $commands {} commands

	return $commands
    }

    #-----------------------------------------------------------------
    # the "ident" command
    #-----------------------------------------------------------------
#     apptalk proc ident {
# 	usage: %N ident ?name?
# 	with an argument sets the ident to ?name?. Without an argument
# 	it returns the name previously set
#     } {
# 	if {[llength $args] > 0} {
# 	    set _apptalkData($apptalkServer:ident) [lindex $args 0]
# 	    # my orginal plan was to query all other interpreters to
# 	    # see if *they* think they have the same value for ident..
# 	    # It would make life cool if only one app could be a 
# 	    # particular ident at any one time. Maybe later :-(
# 	} else {
# 	    if {![info exists _apptalkData($apptalkServer:ident)]} {
# 		return {gee, I don't know!}
# 	    } else {
# 		return [set _apptalkData($apptalkServer:ident)]
# 	    }
# 	}
#     }

    #-----------------------------------------------------------------
    # the standard "popup" command. For servers with unique window 
    # arrangements, this can be redefined.
    #-----------------------------------------------------------------
    apptalk proc popup {
	usage: %N popup
	causes the main window of %N to pop up
    } {
	wm deiconify .
	raise .
    }

    #-----------------------------------------------------------------
    # the standard "exit" command. For servers with unique requirements
    # this can be redefined.
    #-----------------------------------------------------------------
    apptalk proc exit {
        usage: %N exit
    } {
        exit 0
    }

    #-----------------------------------------------------------------
    # the standard "popdown" command. For servers with unique window 
    # arrangements, this can be redefined.
    #-----------------------------------------------------------------
    apptalk proc popdown {
	usage: %N popdown
	causes the main window of %N to be minimized
    } {
	wm iconify .
    }

    #-----------------------------------------------------------------
    # ok, now notify all potential apptalk apps that we exist. 
    #-----------------------------------------------------------------
    set oldBinding [bind . <Visibility>]
    bind . <Visibility> "apptalk:notify;$oldBinding"

    # we are done!
    return 1
}

#---------------------------------------------------------------------
#  connect - get the id of an application to send to
#
set _apptalkData(code:connect) {

    set runningApp {}
    set target [lindex $args 0]
    set args [lrange $args 1 end]

    set apptalkPolicy [option get . apptalkPolicy ""]

    # generate a list of running interpreters; we'll need it in just
    # a moment
    set interps [winfo interps]

    # if we have previously found an app, see if it's still running.
    # If so, use it. If not, punt.
    if {[info exists _apptalkData($target:tkid)]} {
	if {[lsearch $interps $_apptalkData($target:tkid)] != -1} {
	    # ok, an interpreter exists with the same id; let's 
	    # check it's pid. If it's the same, cool. If not, bummer!
	    set remotePid [apptalk:send $_apptalkData($target:tkid) pid]
	    if {[string compare $remotePid $_apptalkData($target:pid)] == 0} {
		return $_apptalkData($target:tkid)
	    }
	}
    } 

    # ok, we're going to have to go hunting for an app to talk to...
    set cancel 0
    switch $apptalkPolicy {
	ask {
	    set candidates {}
	    foreach app [lsort $interps] {
		# *sigh* we have to catch this because of the send 
		# command -- it'll return an error if we try to send
		# a command to a pre-tk4 app. Feh.
		catch {
		    if {[apptalk:send $app info exists apptalkServer]} {
			set id [apptalk:send $app set apptalkServer]
			
			if {[string compare $id $target] == 0} {
			    lappend candidates $app
			}
		    }
		}
	    }

	    if {[llength $candidates] <= 0} {
		# do nothing; eventually this will force a new one
		# to spawn
	    } else {
		if {[catch {apptalk:ask $target $candidates} result]} {
		    return {}

		} else {
		    set runningApp $result
		}
	    }
	}

	new {
	    set runningApp {}
	} 

	first {
	    foreach app [lsort $interps] {

		# query the app to see if it has the magic command... if so,
		# it's what we want.

		if {[apptalk:send $app info commands $target] == $target} {
		    set runningApp $app
		    break;
		}
	    }
	}
    }

    if {[string compare $runningApp {}] != 0} {
	# cool... we're using an already running app
	set _apptalkData($target:tkid) $runningApp
	set _apptalkData($target:pid) [apptalk:send $runningApp pid]
	set retval $runningApp

    } else {
	# if runningApp is null, we couldn't find a running app so we
	# must start one up
	
	# if the args list is of zero length, we'll run a command by
	# the same name as the target. Otherwise, args contains a 
	# list of commands to pass to exec to start the program
	# spawn the sucker... 
	if {[llength $args] == 0} {
	    # ok, we don't know how to start this program. Let's
	    # see if at some point in our past we were told how
	    # to start it.
	    if {[info exists _apptalkData($target:exec)]} {
		# Yes? Cool.
		set args $_apptalkData($target:exec)
	    } else {
		# No? Punt.
		set args [list $target]
	    }
	}
	set _apptalkData($target:exec) $args
	set foo [eval apptalk:exec $args]
	set pid  [lindex $foo 0]
	set tkid [lindex $foo 1]


	if {$tkid != {}} {
	    # success!
	    set _apptalkData($target:tkid) $_apptalkData(notify)
	    set _apptalkData($target:pid) $pid
	    set retval $_apptalkData($target:tkid)

	} else {
	    # failure :-(
	    set retval {}
	}

    }
    if {[string compare $retval {}] != 0} {
	# create a proc named '@$target' that is an alias
	# to "apptalk $target send..."
	proc @$target {args} "apptalk send $target \$args"
    }	
    return $retval
}

#---------------------------------------------------------------------
# proc - create an apptalk server command. 
#
# Usage: apptalk proc name docstring body
#
set _apptalkData(code:proc) {
    set command [lindex $args 0]
    set _apptalkData(info:$apptalkServer:$command) [lindex $args 1]
    set _apptalkData(code:$apptalkServer:$command) [lindex $args 2]
}

#---------------------------------------------------------------------
# send - send a command to a server
#
set _apptalkData(code:send) {
    set app  [lindex $args 0]
    set cmd  [lindex $args 1]
    set args [lrange $args 2 end]

    # first, establish a connection to a running application
    # (starting one up if it's not running...)
    set application [apptalk connect $app]

    # note: in the server application there will be a command named
    # $app... it is that command we want to execute on behalf of
    # the user. $application, OTOH, is the internal name we must 
    # use for tk to be able to send data to it.

    # we actually want three commands to execute. The first is to
    # set the global variable apptalkClient, so the server knows
    # who is making the request. The second is to set apptalkServer,
    # so the server knows who the client is expecting to talk to, and
    # third, the actual command.
    set command "set apptalkClient {[tk appname]};"
    append command "set apptalkServer {$app};"
    append command "$app $cmd $args"

    apptalk:send $application $command
}

#---------------------------------------------------------------------
# this proc centers a dialog, pops it up, and issues a grab.
proc apptalk:popupdialog {w} {
    # Withdraw the window, update all the geometry information
    # so we know how big it wants to be, then center it in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w
}

#---------------------------------------------------------------------
# this proc sends notification to all other running interpreters
# that this proc exists. This is done asynchronously so we don't hang.
#
proc apptalk:notify {} {
    # first, strip the apptalk binding from the <Visibility> event
    # so this isn't called more than once
    set binding [bind . <Visibility>]
    regsub {apptalk:notify;} [bind . <Visibility>] {} binding
    set binding [string trim $binding]
    if {[string length $binding] > 0} {
	bind . <Visibility> $binding
    }

    # set up a command to run in the remote interpreters
    regsub {MY_NAME} {
	if {[string compare [info commands apptalk] {}] != 0} {
	    set _apptalkData(notify) {MY_NAME}
	}
    } [tk appname] remoteCommand

    # now, notify all other interps that we exist. This is 
    # problematic -- we want to use async so we don't hang, but
    # that means we don't know if there's some problem with
    # the send command itself. That sucks.
    foreach interp [winfo interps] { 
#	apptalk:send -async $interp $remoteCommand
	apptalk:send $interp $remoteCommand
    } 
}

#---------------------------------------------------------------------
# this code, to be run by the server only, returns a nice comma 
# separated list of known commands. It's a separate proc since we
# use it in more than one place.
proc apptalk:commands {server} {
    global apptalkServer
    global _apptalkData

    # this gives us a list of array indicies that just happen to 
    # correspond to our known commands.
    set legalOptions [join [lsort [array names _apptalkData \
				       "code:$apptalkServer:*"]] ", "]

    # this cleans up the command names by removing the server from
    # the array index
    regsub -all "code:$apptalkServer:" $legalOptions {} legalOptions

    # this bit of nonsense is to get the list to come out as
    # "foo, bar and baz" instead of "foo, bar, baz". *hack* *cough*
    regsub {, ([^,]*)$} $legalOptions { and \1} legalOptions


    return $legalOptions
}

#---------------------------------------------------------------------
# this proc pops up the window that says "starting foo..."
proc apptalk:popNotify {message} {
    global _apptalkData

    if {[string compare $message {}] == 0} {
	if {[winfo exists .apptalkDialog]} {
	    wm withdraw .apptalkDialog
	}
	return
    }

    if {![winfo exists .apptalkDialog]} {
	set _apptalkData(notifyDialog) .apptalkDialog
	toplevel .apptalkDialog
	wm withdraw .apptalkDialog
	wm title .apptalkDialog "Apptalk Dialog"

	wm group .apptalkDialog .
	wm transient .apptalkDialog .

	label .apptalkDialog.label -text {}
	pack .apptalkDialog.label \
	    -side top \
	    -fill both \
	    -expand y \
	    -padx 25 \
	    -pady 25
	button .apptalkDialog.ok \
	    -text "Dismiss" \
	    -command {wm withdraw .apptalkDialog}
	pack .apptalkDialog.ok -side bottom -anchor s -expand y

    }
    .apptalkDialog.label configure -text $message
    apptalk:popupdialog .apptalkDialog
    update idletasks

}

#---------------------------------------------------------------------
# apptalk:ask - pop up a dialog to let the user choose which interpreter
#               to use
proc apptalk:ask {target candidates} {
    global _apptalkData

    # ok, create a dialog...

    set dialog .apptalkAsk
    toplevel $dialog
    wm withdraw $dialog
    wm title $dialog "Apptalk Dialog"

    wm group $dialog .
    wm transient $dialog .

    frame $dialog.top -bd 1 -relief raised
    frame $dialog.bottom -bd 1 -relief raised
    frame $dialog.buttons

    pack $dialog.top -side top -fill both -expand y
    pack $dialog.bottom -side bottom -fill x 
    pack $dialog.buttons -in $dialog.bottom -side right -fill y

    label $dialog.image -bitmap "questhead"
    pack $dialog.image -in $dialog.top -side left -fill y -padx 10

    # create a kinda gross, verbose label...
    set N [llength $candidates]
    set message "You have requested to use the program $target. "
    if {$N > 1} {append message "There are $N running applications "}
    if {$N <= 1} {append message "There is $N running appllication "}
    append message "that could be used "
    append message "to provide the functionality of $target.\n\n"
    append message "Pick one of the running applications from the "
    append message "list or press 'New' to run a new instance "
    append message "of $target.\n\n"
    append message "Pick one of the items on the list and press "
    append message "'Show Me' to see which window that item represents."

    message $dialog.label -text $message -aspect 300
    pack $dialog.label -in $dialog.top -side top -fill x -expand y

    frame $dialog.listbox -bd 0
    scrollbar $dialog.listbox.vsb \
	-orient vertical \
	-command [list $dialog.listbox.listbox yview]
    listbox $dialog.listbox.listbox -height 5 \
	-yscrollcommand [list $dialog.listbox.vsb set]
    pack $dialog.listbox.listbox -side left -fill both -expand y
    pack $dialog.listbox.vsb -side right -fill y -expand n

    pack $dialog.listbox -in $dialog.bottom -side left -fill both -expand y
    set listbox $dialog.listbox.listbox
    set _apptalkData(ask:listbox) $listbox

    set index 0
    foreach tkid $candidates {
	set title [apptalk:send $tkid "wm title ."]
	$listbox insert end $title
	set _apptalkData(index2id:$index) $tkid
	incr index
    }

    button $dialog.new \
	-text "New" \
	-command "apptalk:ask.button new"

    button $dialog.showme \
	-state disabled \
	-text "Show Me" \
	-command "apptalk:ask.button showme"

    button $dialog.use    \
	-state disabled \
	-text "Use This One" \
	-command "apptalk:ask.button select"

    button $dialog.cancel \
	-text "Cancel" \
	-command "apptalk:ask.button cancel"

    pack $dialog.showme $dialog.use $dialog.new \
	-fill x \
	-in $dialog.buttons \
	-side top \
	-expand n

    pack $dialog.cancel \
	-fill x \
	-in $dialog.buttons \
	-side bottom \
	-expand n

    bind $listbox <Any-1> "$dialog.showme configure -state normal; \
                           $dialog.use configure -state normal"
	
    set oldFocus [focus]

    apptalk:popupdialog $dialog
    if {![winfo ismapped $dialog]} {
	tkwait visibility $dialog
    }
    grab $dialog

    # failsafe... after so long we'll simulate a cancel
    after 30000 apptalk:ask.button cancel
    tkwait variable _apptalkData(selectedApp)
    after cancel apptalk:ask.cancel

    # boom!
    destroy $dialog

    if {$_apptalkData(selectedApp) == "__New__"} {
	return {}

    } elseif {$_apptalkData(selectedApp) == {}} {
	error "user cancelled"

    } else {
	return $_apptalkData(selectedApp)
    }
}

#---------------------------------------------------------------------
# this is used by the ask dialog
proc apptalk:ask.button {button} {
    global _apptalkData

    switch $button {
	new {
	    set _apptalkData(selectedApp) __New__
	}
	cancel {
	    set _apptalkData(selectedApp) {}
	}
	select {
	    set listbox $_apptalkData(ask:listbox)
	    set index [$listbox curselection]
	    set target $_apptalkData(index2id:$index)
	    set _apptalkData(selectedApp) $target
	}

	showme {
	    set listbox $_apptalkData(ask:listbox)
	    set index [$listbox curselection]
	    set target $_apptalkData(index2id:$index) 
	    if {$target == {}} {
		bell
	    } else {
		apptalk:send $target {wm iconify . ; wm deiconify .; raise .}
	    }
	}
    }
}

#---------------------------------------------------------------------
# this is an abstraction of Tk's send command. In future versions of
# apptalk we'll support additional communciation methods in addition
# to send
proc apptalk:send {args} {
    eval send $args
}

#---------------------------------------------------------------------
# this is an abstraction of the exec command. In future versions of
# apptalk we may support other commands beside exec (eg: blt's bgexec,
# ferinstance)
#
# this proc simply passes it's args to exec. It returns a list of
# two values: the pid of the spawned process, and the internal tk
# id of the program that was spawned.
proc apptalk:exec {args} {
    global _apptalkData

    set target [lindex $args 0]

    # schedule a dialog to appear if things don't happen
    # fast enough
    set afterCmd "apptalk:popNotify {starting $target... please wait.}"
    after 1000 $afterCmd

    # exec the program...
    if {[catch {eval exec $args &} result]} {
	set pid {} 
	set tkid {}

    } else {

	# wait for notification. Of course, if the spawned program isn't
	# apptalk aware, we're hosed. We'll end up waiting forever. 
	# That sucks.

	vwait _apptalkData(notify)

	set pid $result
	set tkid $_apptalkData(notify)
    }

    # cancel the dialog; cancel the after in case it *didn't* appear,
    # and hide the dialog if it did.
    catch {after cancel $afterCmd}
    apptalk:popNotify {}

    # if things went in the dumper, tell the user
    if {$pid == {}} {
	tk_dialog .apptalkFailed "Apptalk Error" \
	    "There were problems starting $target:\n\n$result" \
	    error 0  "Drat!"
    }
    return [list $pid $tkid]
}
